;;; -*- Mode: LISP; Syntax: Common-lisp; Package: MVL; Base: 10 -*-

(in-package "MVL")

;; constructs bilattice for first-order ATMS in disjunctive normal form

;; the bilattice is constructed from a lattice as usual; the trick is
;; that the members of a particular context are not just sentences, but
;; dotted pairs (p . b) where p is a proposition number and b is a
;; binding list.

;; maximal and minimal elements are just like ATMS.

(defvar dmin-j)
(defvar dmax-j)

;; one context is contained in another if every sentence in it is
;; subsumed by a sentence in the other.  We want (p . ((?a . fred)))
;; to be contained in (p . nil) but not the other way around, since the
;; second one contains more elements of the assumption "schema"
;; corresponding to p.

(defun contained (x y) (subsetp x y :test #'just-subsumed))

;; recall from the above that (p . ((?a . fred))) is supposed to be
;; just-subsumed by (p . nil), so that one sentence is subsumed by
;; another if they are the same proposition and the first bindings are
;; le the second

(defun just-subsumed (a b)
  (and (eq (car a) (car b)) (binding-le (cdr a) (cdr b))))

;; glb and lub are just like ATMS lattice, except instead of just
;; deleting duplicates, we have to remove *any* subsumed elements.

(defun j-or (j1 j2)
  (atms-or j1 j2 #'jl-simplify))

(defun j-and (j1 j2)
  (atms-and j1 j2 #'jl-simplify #'(lambda (x y)
				    (j-simplify (append x (copy-list y))))))

(defun j-simplify (l)
  (delete-subsumed-entries l #'just-subsumed))

(defun jl-simplify (l)
  (delete-subsumed-entries l #'(lambda (x y) (contained y x))))

;; special function for comparing two values -- same as ATMS, but
;; elements aren't just equal; they have to be the same proposition and
;; binding lists.

(defun j-eq (j1 j2)
  (set-equal j1 j2 #'j-same))

(defun j-same (x y)
  (set-equal x y #'just-eq))

(defun just-eq (x y)
  (and (eq (car x) (car y)) (equal-binding (cdr x) (cdr y))))

(defun j-le (j1 j2)
  (atms-le j1 j2 #'contained))

;; dot with star is just like ATMS

(defun j-dws (j1 j2)
  (atms-dws j1 j2 #'contained))

;; stash uses empty binding list

(defun j-stash (p) `(((,(index-with-negation p) . nil))))

;; finding the variables in a truth value is a little subtle.  First,
;; we arrange to do it a term at a time ...

(defun j-vars (value)
  (when value (reduce #'union (mapcar #'jv-1 value))))

(defun jv-1 (context)
  (when context (reduce #'union (mapcar #'jv-2 context))))

;; The variables in a (prop . bdg) are those in the proposition that
;; *aren't* bound by the binding, together with any variables in the
;; cdr's of the binding.

(defun jv-2 (term &aux (ans (set-difference (get (car term) 'vars-in)
					    (mapcar #'car (cdr term)))))
  (dolist (item (cdr term) ans)
    (setq ans (vars-in (cdr item) ans))))

;; plugging is a little harder.  We could just do it with sublis, except
;; for * variables and such.  So we have to ground out in plug, eventually.

(defun j-plug (value bdgs)
  (jl-simplify (mapcar #'(lambda (x) (jp-1 x bdgs)) value)))

;; ... to plug into a context, we plug into each term and simplify ...

(defun jp-1 (context bdgs)
  (j-simplify (mapcar #'(lambda (x) (jp-2 x bdgs)) context)))

;; ... to plug into a term (p . bdg), we have to get the new binding
;; list, which is obtained by combining the two binding lists and then
;; taking the meaninful bindings from what you get.

(defun jp-2 (term bdgs)
  (cons (car term)
	(meaningful-bdgs (append-binding-lists (cdr term) bdgs)
			 (get (car term) 'vars-in))))

(defparameter *first-order-atms-lattice*
	      (make-lattice :max dmax-j
			    :min dmin-j
			    :stash-val #'j-stash
			    :and #'j-and
			    :or #'j-or
			    :eq #'j-eq
			    :leq #'j-le
			    :dws #'j-dws
			    :vars #'j-vars
			    :plug #'j-plug
			    :long-desc "First-order ATMS."
			    :short-desc "First-order ATMS"
			    :char #\o))

(defparameter *first-order-atms-bilattice* 
    (lattice-to-bilattice *first-order-atms-lattice*))

(bilattice *first-order-atms-bilattice*)
