;;; -*- Mode: LISP; Syntax: Common-lisp; Package: MVL; Base: 10 -*-

(in-package "MVL")

;; There are nine files that contain the code used by the theorem
;; prover.  The basic reason for this is that the theorem prover works by
;; invoking a first-order theorem prover and then accumulating the
;; results as described in various theoretical papers.  Because of this,
;; there are two "levels" to the prover:

;; 1.  The top level controls the invocations of the theorem prover.
;; This level includes the files bc.lisp, values.lisp, bc-user.lisp and
;; bc-interface.lisp.

;; 2.  The lower level is the prover itself.  This is a fairly standard
;; first-order theorem prover but has been modified to do goal-goal
;; resolutions and to delete any proof paths that are subsumed by other
;; portions of the search space.  This level contains the files
;; first-order.lisp, first-order-user-interface.lisp and
;; first-order-mvl-interface.lisp.

;; It is also important that we not use answers returned by the
;; first-order prover that have no bearing on the original query.  Note
;; that we don't mind *computing* these answers -- all proof paths that
;; don't hit a modal operator or a default sentence are assumed to be
;; short.  But if we *use* them, we're going to generate all sorts of
;; useless tasks, and the truth values being moved around are going to be
;; much more complex than usual.  So the top level has to be selective
;; about what it uses from the bottom level.  Code to deal with this is
;; in the file relevance.lisp.

;; Finally, at any given point in the process of responding to some
;; query, we will need to pick a particular invocation of the first-order
;; prover to work on.  This choice is the responsibility of the code in
;; control.lisp.

;; Because some of what follows is dependent on the interaction between
;; the top level prover and the individual proof tasks, let us begin by
;; describing it.  The basic question that we need to answer is the
;; following: What impact would a particular answer for this or other
;; proofs have on the final answer?  This is in general fairly expensive
;; to compute; the reason is that it is typically fairly expensive
;; (relative to other operations) to compute the truth value associated
;; to a task.  Every task therefore attempts to cache information of the
;; form, "if the children of this task have values v1, ..., vn, the task
;; itself has value v."

;; Additional information is contained in the pending-answers field of
;; the invocations of the prover, which list answers that have been
;; computed but not yet used.

;; Additional documentation on these features is in relevance.lisp, but
;; since the structures corresponding to proof attempts and to tasks
;; need to be defined previously, we have explained this here to make it
;; clear what the slots in these structures are.

;; The code is broken down as follows:

;; 1.  First-order theorem prover.
;;   1a.  The code for the theorem prover itself is in first-order.lisp.
;;   1b.  The code that describes the interface between the first-order
;;   	  theorem prover and the MVL functions that call the theorem
;;   	  prover is in first-order-mvl-interface.lisp.
;;   1c.  The code that describes the user interface to the first-order
;;   	  theorem prover (debugging and trace facilities) is in
;;   	  first-order-user-interface.lisp.
;; 2.  Basic MVL theorem prover.
;;   2a.  The code that does the work is in bc.lisp.
;;   2b.  The code that manipulates the truth values under consideration is
;;        in values.lisp.
;;   2c.  The code the defines the top-level user functions is in bc-user.lisp.
;;   2d.  The code that describes the user interface (debugging and
;;   	  trace facilities) is in bc-interface.lisp.
;; 3.  The code that handles the relevance computations described above
;;     is in relevance.lisp, and the control computations appear in
;;     control.lisp.

;; This file is first-order.lisp and contains the guts of the
;; first-order throem prover only.

;; Each proof effort has two sorts of information: that pointing up to
;; deal with the rest of the MVL system, and that pointing down to deal
;; with the first-order proof alone.  From the system's point of view,
;; each proof effort is described by an object of type proof.  This
;; object has the following fields:
;;   prop		The proposition being proven
;;   vars		variables in the query
;;   fotp		Either nil if the proof is complete or a function
;;   			of no variables that will return a list of new
;;   			solutions
;;   answers		the answers found so far but not processed
;;   pending-answers    answers that have yet to be used
;;   task		The task that spawned this invocation of the
;;   			first-order prover
;;   slaves		A list of (task . bdg) for other tasks using this
;;			invocation of the prover.
;;   relevance		A list of the current relevances for this proof
;;   			given the answer true.  If T, you have to recompute
;;   			this.

;; At the lower level, a proof is described by an object of type fotp.
;; This has the following fields:
;;   prop		proposition being proved (used for display only)
;;   active-nodes	nodes that still need to be expanded
;;   answers		the answers found (sometimes several are discovered
;;   			simultaneously and we need a place to put them)
;;   indexp		a crude index of the positive subgoals appearing in
;;			this proof tree
;;   indexn		a crude index of the negative subgoals appearing in
;;			this proof tree
;;   depth-limit	How deep to search before giving up

;; In order to avoid expanding duplicate portions of the first-order
;; search space, nodes that cannot contribute new values to the proof
;; effort are pruned.  To implement this idea, the prover retains a list
;; of all of the nodes that have been examined in the proof effort thus
;; far, and indexp and indexn are used to access particular nodes
;; quickly.

;; The nature of the subsumption check is described at the point that
;; the associated function (bc-subsumes) is defined; it is fairly
;; straightforward.

(defvar *depth-limit*)

(defstruct (proof (:constructor make-proof (prop fotp))
	    (:print-function print-proof))
  prop (vars (vars-in prop)) fotp
  (answers nil) (pending-answers nil)
  task (slaves nil) (relevance nil))

(defstruct (fotp (:constructor make-fotp (prop)) (:print-function print-fotp))
  prop (active-nodes nil) (answers nil)
  (indexp nil) (indexn nil) (depth-limit *depth-limit*))

;; To make things a bit simpler, the "literal" in any node has had
;; any leading negations stripped off of it, so that it is a pure
;; literal.  The "inverted" slot is T if the actual term being
;; considered is the negation of the given literal, and is NIL if not.

;; each active node looks like a <goal>
;;   literal		the literal being worked on
;;   inverted		do we really care about the negation of literal?
;;   vars		variables in the literal
;;   clause 		the list of literals or conjunction to prove
;;   binding		the binding list for this clause
;;   value		truth value
;;   just		partial justification
;;   opaque		is this literal referentially opaque?	
;;   parent             parent of this node
;;   children           a list of the children of this node
;;   depth		how deep in the search tree is this node?

;; Most of the fields have to do with this "level" of the proof tree
;; only, so that:

;; clause is a list of nodes that must succeed to get back to the next 
;; higher level of the proof tree, not a list of everything you have 
;; to do to solve the original problem.

;; binding is just the variables that have been instantiated at this 
;; level of the proof tree.  When moving back a level, we remove any 
;; irrelevant bindings (this makes the subsumption check more 
;; efficient, among other things).

;; value is the truth value that has been accumulated at this level 
;; only.

(defstruct (bcanswer (:include answer) (:print-function print-bcanswer)
	    (:constructor make-bcanswer (binding value just)))
  just)

(defstruct (goal (:print-function print-goal) (:include bcanswer)
	    (:constructor make-goal (binding value just clause parent)))
  literal inverted vars clause (opaque nil) parent (children nil)
  (depth (if parent (1+ (goal-depth parent)) 0)))

;; here is a function that computes the term being considered by the
;; goal node, inverting it if necessary.

(defun goal-term (goal)
  (if (goal-inverted goal)
      (list 'not (goal-literal goal))
    (goal-literal goal)))

(defvar fotp)

;; add an unexpanded node to the current proof effort; the parent of the
;; node should already have been established.  Set up the associated
;; literal, inversion flag and list of variables.  Then if it's at a
;; legal depth, display it and push it onto its parent's list of children
;; and onto the list of active nodes.

(defvar *save-nodes*)

(defun prover-addnode (node &aux (parent (goal-parent node)))
  (multiple-value-bind (literal inverted)
      (strip-nots (plug (car (goal-clause node)) (goal-binding node)))
    (setf (goal-literal node) literal
	  (goal-inverted node) inverted
	  (goal-vars node) (vars-in literal)))
  (when (and parent (legal-depth node))
    (bc-display-info node "creating" *save-nodes*)
    (push node (goal-children parent))
    (push node (fotp-active-nodes fotp))))

(defun legal-depth (n &aux (d (fotp-depth-limit fotp)))
  (or (null d) (<= (goal-depth n) d)))

;; Main entry point for the backward chainer.
;;   1.  Call bc-recursion to see if the node can be eliminated
;;   2.  Put the node into the goal index
;;   3.  Call bc-lookup and bc-rules on the node
;;   4.  Check for goal-goal resolutions

(defparameter *allow-goal-goal-resolution* t)

(defun bc-call (node)
  (bc-display-info node "working on")
  (unless (bc-recursion node)
    (bc-index node)
    (bc-lookup node)
    (bc-rules node)
    (when *allow-goal-goal-resolution* (bc-goalgoal node))))

;; Here we check to see if the given node can be eliminated.  We do this
;; by working through all nodes in the current proof effort that have
;; the same predicate and parity as this one does.
;;  a.  If this node can be subsumed, we should do that.  There is no
;;      point in checking to see if it subsumes another node, since any 
;;      such subsumed node would have been subsumed already.  We return
;;      t in this case.
;;  b.  Any other node that can be subsumed should be.

;; The check is done by bc-subsumes.

(defun bc-recursion (node &aux (friends (cdr (bc-index-find node))))
  (dolist (friend friends)
    (when (bc-subsumes friend node)
      (bc-display-info friend "... subsumed by")
      (return-from bc-recursion t)))
  (dolist (friend friends)
    (when (bc-subsumes node friend)
      (bc-display-info friend "... subsumes")
      (bc-delnode friend))))

;; Here we determine whether g1 subsumes g2.  This can only happen if g2
;; is an instance of g1, for a start.  In addition, we should do a samep
;; instead of an instp if the expression is referentially opaque.  (This
;; is true if the goal involves a modal operator, for example.)

;; If it is an instance, we need to see if it cannot return any new
;; answers to the given problem.  The "given problem" can be found by
;; finding the common ancestor of g1 and g2, and g2 will provide no new
;; answers provided that (a) the pending conjuncts for g1 are a subset
;; of those for g2 (i.e., no additional things need to be proven by g1),
;; (b) the relative bindings for g1 are a subset of those for g2, and
;; (c) the justification for g1 is a subset of the justification for g2
;; (i.e., no additional assumptions are needed by g1).  If all of this
;; happens, bc-subsumes returns t.

;; In the code that follows, b1/c1/j1 and b2/c2/j2 are constructed by
;; working from the node g1 or g2 back to the common ancestor,
;; accumulating the bindings/conjuncts left/justification along the
;; path.  These values are then compared to see if g1 subsumes g2.

(defun bc-subsumes (g1 g2 &aux (binding (check-generalizes g1 g2)))
  (when binding
    (let ((ancestor (bc-ancestor g1 g2)))
      (multiple-value-bind (b1 c1 j1) 
	  (subsumption-acc g1 ancestor (car binding))
	(multiple-value-bind (b2 c2 j2)
	    (subsumption-acc g2 ancestor nil)
	  (and (binding-le (meaningful-bdgs b2 (goal-vars ancestor))
			   (meaningful-bdgs b1 (goal-vars ancestor)))
	       (conj-subsumes (plug c1 b1) (plug c2 b2))
	       (subsetp (plug j1 b1) (plug j2 b2) :test #'eq-just)))))))

;; we often have to accumulate parts of an answer from a node back to an
;; ancestor; we don't want to accumulate what we don't need, so this code
;; defines some macros to define suitable functions.  A typical call is
;;   (make-proof-accumulator subsumption-acc :bdg t :conj aux :just aux)
;; indicating that the binding is a variable passed to the function and
;; accumulated; the conjunction and justification are assembled from NIL.

;; First, we need to produce the argument lists.  mpf-args gets the
;; arguments (the ones for which the keyword is T); mpf-aux tests the
;; keyword to see if it's aux, and mpf-vals test to see if it's there at
;; all (that's what you return in the end).

;; Since mpf-args (and so on) will be used later in this file, they
;; can't be defuns; we need them *now*.  So they have to be macros.  In
;; order to keep things even vaguely simple, we set them up so that they
;; expand into things like (bdg conj) -- which we can't evaluate because
;; they correspond to bad function calls -- and then the macro that makes
;; the accumulator functions uses macroexpand to "invoke" the macros.

(defmacro mpf-gen (name test)
  `(defmacro ,name (&rest args)
     (delete nil (mapcar #'(lambda (x y) (when ,test y))
			 args '(bdg conj just value)))))

(mpf-gen mpf-args (eql x t))
(mpf-gen mpf-aux (eql x 'aux))
(mpf-gen mpf-vals x)

;; here is the accumulator function.  Where we need an argument list, we
;; splice in the result of macroexpanding mpf-args or what have you.

(defmacro make-proof-accumulator (name &key bdg conj just value)
  `(defun ,name (node ancestor
		 ,@(macroexpand `(mpf-args ,bdg ,conj ,just ,value))
		 &aux ,@(macroexpand `(mpf-aux ,bdg ,conj ,just ,value)))
     (do ()
	 ((eq node ancestor)
	  (values ,@(macroexpand `(mpf-vals ,bdg ,conj ,just ,value))))
       (setq
	   ,@(when bdg '(bdg (append-binding-lists bdg (goal-binding node))))
	   ,@(when conj '(conj (append conj (cdr (goal-clause node)))))
	   ,@(when just '(just (append just (goal-just node))))
	   ,@(when value '(value (mvl-and (goal-value node) value)))
	   node (goal-parent node)))))

(make-proof-accumulator subsumption-acc :bdg t :conj aux :just aux)

(defun check-generalizes (g1 g2)
  (if (goal-opaque g2)
      (samep (goal-literal g1) (goal-literal g2) t)
    (instp (goal-literal g1) (goal-literal g2))))

;; Checking if one justification is equal to another -- if the car's are
;; "truth value", that means that the justification is from a procedural
;; attachment or some such, and we need to check that the cdr's (the
;; truth values produced) are mvl-eq.  In other cases, we have to make
;; sure that the propositions themselves are the same.

(defun eq-just (x y)
  (or (eq x y)
      (and (eq (car x) (car y))
	   (funcall (if (eq (car x) 'truth-value) #'mvl-eq #'samep)
		    (cdr x) (cdr y)))))

;; Find the common ancestor of two nodes.  The way it works is that we
;; first figure out which node is deeper than the other, then back that
;; up until they are the same depth.  Once that happens, we can back
;; them up in parallel.  This way, we don't have to keep comparing
;; whole lists of nodes.

(defun bc-ancestor (x y &aux (diff (- (goal-depth x) (goal-depth y))))
  (cond ((plusp diff) (dotimes (i diff) (setq x (goal-parent x))))
	((minusp diff) (dotimes (i (- diff)) (setq y (goal-parent y)))))
  (do nil ((eq x y) x)
    (setq x (goal-parent x) y (goal-parent y))))

;; Routine to determine if one conjunction is subsumed by another.  We
;; make sure that every clause in the first conjunction is a
;; generalization of some clause in the second conjunction.  As we go
;; along the first conjunction, we keep accumulating the bindings
;; obtained by unifying with some clause in the second one.

(defun conj-subsumes (c1 c2) (cj-1 c1 c2 (list nil)))

(defun cj-1 (c1 c2 bdgs &aux b)
  (if c1
      (some #'(lambda (y) (when (setq b (inst-1 (car c1) y bdgs nil))
			    (cj-1 (cdr c1) c2 b)))
	    c2)
    t))

;; Remove a node and all of its descendents from the active list.  We do
;; this by running through the list of descendants and removing them from
;; the goal index and active list.

(defun bc-delnode (node)
  (bc-unindex node)
  (popf (fotp-active-nodes fotp) node :count 1)
  (mapc #'bc-delnode (goal-children node)))

;; Check to see if an answer for this goal is in the database, and call
;; bc-propagate on each answer found.  If the answer was found in a
;; referentially opaque way, mark the node as opaque.  (This will be
;; true if the second value returned by lookups is not NIL.)

(defun bc-lookup (node &aux (prop (goal-term node)))
  (multiple-value-bind (ansl x) (lookups prop :succeed-on-modal-ops t)
    (bc-display-lookup node ansl)
    (mapc #'(lambda (ans)
	      (bc-propagate (make-bcanswer (answer-binding ans)
					   (answer-value ans)
					   (bc-just prop x ans))
			    node))
	  ansl)
    (when x (setf (goal-opaque node) t))))

;; Find rules that apply to a given literal.  For each answer, make a
;; new node, one deeper than the current one.  We have to be careful to
;; pull "clause" out of the binding list before deleting it from ans!

(defun bc-rules (node &aux (starvar '#:?*) 
			   (prop `(<= ,(goal-term node) ,starvar)))
  (multiple-value-bind (ansl x) (lookups prop)
    (mapc #'(lambda (ans &aux (clause (get-bdg starvar (answer-binding ans))))
	      (prover-addnode
	       (make-goal (delete-bdg starvar (answer-binding ans))
			  (answer-value ans)
			  (bc-just prop x ans)
			  clause node)))
	  ansl)))

;; bc-propagate takes an answer and trickles it back up the proof tree.
;; The way it works is that we can stop when either there are still
;; unanalyzed conjuncts for this node, or when the node has no parent.
;; In other cases, we dump any meaningless bindings from the answer
;; before proceeding.

;; If there are more conjuncts for this node, we absorb the answer and
;; pop the conjunct list.  Otherwise, we adjoin this answer to the
;; binding list for the current node, conjoin the value and update the
;; justification list, then go look at the parent node.  If the parent
;; node is nil, we can actually return an answer to the whole problem!

(defun bc-propagate (ans node)
  (do (new-val new-bdg new-just)
      ((null node) (push ans (fotp-answers fotp)))
    (setf (answer-binding ans)
      (meaningful-bdgs (answer-binding ans) (goal-vars node)))
    (setq new-val (mvl-and (answer-value ans)
			   (mvl-plug (goal-value node) (answer-binding ans))))
    (when (mvl-unk new-val) (return))
    (setq new-bdg
      (append-binding-lists (answer-binding ans) (goal-binding node))
      new-just 
      (append (just-plug (goal-just node) (answer-binding ans))
	      (bcanswer-just ans)))
    (when (cdr (goal-clause node))
      (prover-addnode (make-goal new-bdg new-val new-just
				 (cdr (goal-clause node)) (goal-parent node)))
      (return))
    (setq ans (make-bcanswer new-bdg new-val new-just)
	  node (goal-parent node))))

;; construct the partial justification for an answer given by lookup.  If
;; proposition is modal use ('modal p).  And if for some reason you
;; don't need to consider the negation but only want to retain the truth
;; value, push (truth-value . <truth value>) instead of the proposition.
;; Don't push it onto the list if it's true, either (since there's no
;; point in trying to prove its negation).

(defun bc-just (prop source answer &aux (value (answer-value answer)))
  (when (or (not (mvl-eq value true)) (eq source 'modal))
    (list (case source
	    (modal (list 'modal (plug prop (answer-binding answer))))
	    (attach (cons 'truth-value value))
	    (t (if (monotonic-value value) `(truth-value . ,value) 
		 (plug prop (answer-binding answer))))))))

;; plug a binding list into a justification.  The only trick is that if
;; an element on the justification is of the form (truth-value x), then
;; use mvl-plug on x instead of just plug.

(defun just-plug (just bdg)
  (if bdg (mapcar #'(lambda (x) (jp x bdg)) just) just))

(defun jp (clause bdg)
  (if (eq (car clause) 'truth-value)
      (reuse-cons (car clause) (mvl-plug (cdr clause) bdg) clause)
    (plug clause bdg)))

;; This routine does the equivalent of goal-goal resolutions for a
;; natural deduction theorem prover.  It takes a goal node and examines
;; all other goal nodes in the proof tree that might match the negation
;; of this goal.  When it finds one, it finds the common ancestor of the
;; two nodes and checks to see that the variable bindings along the
;; paths to the ancestor are consistent.  If so, it collects all the
;; conjuncts, justifications, and truth values along the two paths, and
;; builds a new subgoal of the ancestor node corresponding to the
;; ancestry filtered deduction.

(defun bc-goalgoal (node &aux (stripped (goal-literal node)) binding ancestor)
  (dolist (enemy (cdr (bc-index-find node t)))
    (when (and (setq binding (unifyp stripped (goal-literal enemy)))
	       (setq ancestor (bc-ancestor node enemy))
	       (setq binding
		 (dot-binding (car binding) (bc-ancestorbdgs enemy ancestor)))
	       (setq binding
		 (dot-binding (car binding) (bc-ancestorbdgs node ancestor))))
      (do-goal-goal node enemy (car binding) ancestor))))

;; the goal-goal "resolution" is done here.  We begin by assembling the
;; conjuncts left to solve, justification and truth value of the
;; resultant node.  Then if there is nothing left to do, it's easy -- we
;; just propagate a new answer for the ancestor.

(defun do-goal-goal (g1 g2 binding ancestor)
  (multiple-value-bind (conjuncts just value) (dgg-1 g1 ancestor nil nil true)
    (multiple-value-setq (conjuncts just value) 
      (dgg-1 g2 ancestor conjuncts just value))
    (setq value (mvl-plug value binding)
	  conjuncts (delete-duplicates (plug conjuncts binding) :test #'equal)
	  just (just-plug (delete-duplicates just :test #'eq-just) binding))
    (if conjuncts
	(prover-addnode (make-goal binding value just conjuncts ancestor))
      (bc-propagate (make-bcanswer binding value just) ancestor))))

;; Finds out the bindings that apply to the ancestor node.

(make-proof-accumulator bc-ancestorbdgs :bdg aux)

;; accumulate conjunction, justification and value back to ancestor node.

(make-proof-accumulator dgg-1 :conj t :just t :value t)

;; These routines have to do with maintenance of the goal index in the
;; proof structure. 

;; Put a new goal into the subgoal index for this proof tree.

(defun bc-index (goal &aux (temp (bc-index-find goal)))
  (if temp (push goal (cdr temp))
    (let ((new (list (car (goal-literal goal)) goal)))
      (if (goal-inverted goal) (push new (fotp-indexn fotp))
	(push new (fotp-indexp fotp))))))

;; Remove a subgoal from the goal index for this proof tree.

(defun bc-unindex (goal &aux (place (bc-index-find goal)))
  (when place (popf (cdr place) goal :count 1)))

(defun bc-index-find (goal &optional inv)
  (assoc (car (goal-literal goal))
	 (if (eql (goal-inverted goal) inv)
	     (fotp-indexp fotp) (fotp-indexn fotp))))
