;;; -*- Mode: LISP; Syntax: Common-lisp; Package: MVL; Base: 10 -*-

(in-package "MVL")

;; First-order theorem prover; interface and diagnostic stuff.
;; Functions defined:
;; 
;; init-prover (p)
;;   Start a new task to prove p.
;; 
;; cont-prover (proof)
;;   Reinvoke the proof task associated with a given bit of auxiliary
;;   information.

;; The interface between the first-order theorem prover and the rest of
;; the MVL system is handled by a single function:
;;
;; start-fotp (p)
;;   Accepts a proposition as an argument and returns a function of no
;;   variables.  Invoking the function should return either a list of
;;   bcanswers, each of which corresponds to a first-order proof of p, or
;;   NIL if no such proofs can be found.

;; The search control is done using the parameter *depth-limit*, which
;; is the maximum depth to which the space should be searched (or NIL to
;; simply search depth-first).

(defparameter *depth-limit* nil)

(defvar true)

;; Initialize the prover.  Creates and returns the instance of proof
;; that will be used to maintain all of the information about this proof
;; effort.  Also sets *watched-nodes* to NIL.

(defvar *watched-nodes*)

;; The only tricky thing here is if you are asked to prove a disjunction.
;; In order to make sure that the proof tree is singly-rooted, we create
;; a dummy node called "user-supplied-query" and put all the disjuncts
;; under that (the proposition also includes copies of the original
;; sentence, so that the variables are included).  Prover-addnode is
;; defined in first-order.lisp and adds a given node to the current proof
;; tree.

(defun init-prover (p)
  (make-proof p (start-fotp p)))

(defun start-fotp
    (p &aux (clauses (dnf p)) (fotp (make-fotp p)) (z fotp)
	    (dummy-root (make-goal nil true nil `((user-supplied-query ,p))
				   nil)))
  (setq *watched-nodes* nil)
  (prover-addnode dummy-root)
  (dolist (clause clauses)
    (prover-addnode (make-goal nil true nil clause dummy-root)))
  #'(lambda () (continue-fotp z)))

;; Invoke the prover.  proof is globally bound by the call, so all we
;; need to do is process the unpruned nodes until an answer is returned.

(defun cont-prover (proof &aux (task (proof-task proof)) ans)
  (declare (special *prop*))
  (task-diagnostic task "Prover for" *prop* "invoked")
  (cond ((and (proof-fotp proof) (setq ans (funcall (proof-fotp proof))))
	 (setf (proof-answers proof) ans)
	 (when (traced-task task)
	   (mapc #'(lambda (ans) (prover-show-return (answer-binding ans)
						     (bcanswer-just ans)))
		 (proof-answers proof)))
	 (proof-answers proof))
	(t (task-diagnostic task "Prover for" *prop* "terminated")
	   (setf (proof-fotp proof) nil))))

(defun continue-fotp (fotp)
  (do ()
      ((fotp-answers fotp) (prog1 (fotp-answers fotp)
			     (setf (fotp-answers fotp) nil)))
    (if (fotp-active-nodes fotp)
	(bc-call (pop (fotp-active-nodes fotp)))
      (return))))
