;;; -*- Mode: LISP; Syntax: Common-lisp; Package: MVL; Base: 10 -*-

;;; ---------------------------------------------------------------------
;;; --                                                                 --
;;; --                    THE MVL DYNAMIC VIEWER                       --
;;; --                                                                 --
;;; --                                                                 --
;;; ---------------------------------------------------------------------

(in-package "MVL")

;;; This file contains the code that implements the MVL dynamic viewer.
;;; This code makes thorough use of the code in the viewer.lisp file.
;;; It is organized as follows:

;;; 0. Global variables.
;;; 1. Advicing the necessary MVL functions.
;;; 2. The Dynamic Viewer
;;;    2.1 Initialization
;;;    2.2 The Main Function
;;;    2.3 Planting A New Subtree
;;;    2.4 Utilities
;;;        2.4.1 Positioning Utilities
;;;        2.4.2 Tree Shifting Utilities
;;;        2.4.3 Information Gathering Utilities (search, etc)...
;;;        2.4.4 Book Keeping Utilities
;;;        2.4.5 Dealing with User Requests During Dynamic Viewing
;;;    2.5 Updating the Overviewer
;;;    2.6 Showing Truth Value Changes
;;;    2.7 Static Display During Dynamic Viewing

;;; [0] Global Vars
;;;     -----------

;;; These variables control the initial area of the extent that is
;;; represented in the overviewer.

(defparameter *viewer-initial-extent-width* 4000)
(defparameter *viewer-initial-extent-height* 3000)

;;; This variable is set to t when what we see in the viewer window has
;;; been corrupted and needs to be repainted.

(defvar *viewer-dirty* nil)

;;; A region is stored in this variable to avoid its creation each time
;;; we need one. It is used to update the truth region of the magnifier
;;; when the node it is showing changes truth value.

(defvar *truth-region* (cw::make-region :height 12))

;;; [1] Advicing the necessary MVL functions
;;;     ------------------------------------

;;; ADVICING init-analysis :BEFORE
;;; This is done to call set-up-mvl-advicing which initializes the 
;;; variables and windows for running the dynamic viewer.

(excl:advise init-analysis :before dynamic-viewer-hook nil
	     (set-up-mvl-advicing (car excl:arglist)))

;;; ADVICING bcs :AFTER
;;; When MVL ends the proof effort, we tell our system that the dynamic
;;; viewing is over.

(excl:advise bcs :after dynamic-viewer-ends nil
	     (dynamic-viewer-ends))

;;; dynamic-viewer-ends. This function is called to transform the dynamic
;;;                      viewer into a static display. We let the user 
;;; know that this has happened by changing the hilite rectangle from
;;; a rectangle to a filled rectangle.

(defun dynamic-viewer-ends ()
  (setf *dynamic-viewer-active* nil)
  (revert-to-reverse-hilite))

;;; set-up-mvl-advicing. If *dynamic-viewing* is set, the user wants the
;;;                      dynamic display. So, we call advise-for-dynamics
;;; to carry out the initializations required.

(defun set-up-mvl-advicing (prop)
  (when *dynamic-viewing*
    (advise-for-dynamics prop)))

;;; advise-for-dynamics. Initializes the system for dynamic viewing. The
;;;                      first thing to do is to dispose of previous viewer
;;; windows. Then we initialize the viewer and the overviewer windows.

(defun advise-for-dynamics (prop)
  (flush-viewer-and-related-windows)
  (dynamic-init-viewer (make-analysis :prop prop))
  (dynamic-init-overviewer))

;;; [2] The Implementation
;;;     ==================

;;; 2.1 Initialization
;;;     --------------

;;; dynamic-init-viewer. Initializes the viewer window for dynamic viewing.
;;;                      Most of the work is done by init-viewer. Note that
;;; here we tell the system to use the function viewer-dynamic-traced-task
;;; to decide what tasks the user wants to see.

(defun dynamic-init-viewer (analysis)
  (setf *dynamic-viewer-active* t)
  (setf *viewer-xedge*  (make-array 320 :element-type 'integer
				    :initial-element 100))
  (init-viewer analysis "Dynamic")
  (setf *display-test* #'viewer-dynamic-traced-task)
  (setf *viewer-spots-taken* (make-array 320 :initial-element nil))
  (setf *viewer-current-root* nil)
  (setf *static-display-signaled* nil)
  (setf *viewer-reusable-proofs* nil)
  (cw::activate *viewer*))

;;; dynamic-init-overviewer. Initializes the overviewer for the dynamic
;;;                          effort. Again, most of the work is done by
;;; the function init-overviewer. Note that the variables that determine
;;; the size of the extent are set from the parameters specifying the
;;; initial size.

(defun dynamic-init-overviewer ()
  (setf *viewer-extent-width* *viewer-initial-extent-width*)
  (setf *viewer-extent-height* *viewer-initial-extent-height*)
  (init-overviewer)
  (init-overviewer-hilite))

;;; 2.2 The Main Function
;;;     -----------------

;;; update-dynamic-display. This is the function called from
;;;                         recompute-truth-value. If dynamic
;;; viewing, we call plant-and-show-truth to do the work.

(defun update-dynamic-display (task)
  (when *dynamic-viewing*
    (plant-and-show-truth task)))

;;; plant-and-show-truth. This function is called from recompute-truth-value.
;;;                       There are two cases to deal with. If the task is
;;; already in the tree that we are building, then it means that its truth
;;; value has changed and needs to be updated. So, if the task is visible,
;;; we call update-the-truth to show the changes.

;;; If the task is not already in our tree, then it is part of a new subtree
;;; that MVL has just created. So, we need to add this tree to our graphical
;;; display. For that we go up from this task until we find an ancestor that
;;; is in our tree already. The new subtree is a child of this node. For each
;;; child of this node, we call dynamic-plant-and-reshape  which adds it to 
;;; our graphical representation of the proof.

(defun plant-and-show-truth (task)
  (when (is-new-tree task)
    (unless (find-gnode task)
      (let* ((new-leaf (find-child-of-planted-ancestor task))
	     (planted-leaf (inference-parent new-leaf)))
	(if planted-leaf
	    (dynamic-plant-and-reshape (inference-children planted-leaf))
	  (dynamic-plant-and-reshape (list new-leaf))))))
  (when (funcall *display-test* task)
    (update-the-truth task)))

;;; 2.3 Planting a new Subtree
;;;     ----------------------

;;; dynamic-plant-and-reshape. This function is called with the roots of all
;;;                            the new subtrees to be added to our gnode tree.

;;; The first problem we encounter is that the roots we got (the children
;;; of the planted ancestor) might not be visible in the gnode tree. If only
;;; inference nodes are visible, then the planted ancestor is an inference
;;; node but its children are not. Thus they are not visible. The real roots
;;; are the first inference children that descend from them. Thus, the
;;; first thing we do is reset the root tasks to the real root tasks. These
;;; we obtain by calling find-visible-gnodes on the original root tasks.

;;; We find the parent-gnode and the y level of the parent and we are ready
;;; to start. 

;;; For each new root R we do the following. The xedge is cleared. Then
;;; we call grow-the-tree which returns a gnode-tree rooted at R. This 
;;; creates all the gnodes for the tree. But grow-the-tree works assuming
;;; the tree it creates is the only thing there is. Our next task is to
;;; shift this new tree so that it is placed below its parent gnode. This
;;; is done by calling plant-new-root. Adding the tree might cause collisions,
;;; and the next step is to correct them. After all this we sort the gnodes
;;; by x position.

;;; Our previous work has given a position to each new node but we have not
;;; yet used this position to set the locations of all the components of the
;;; gnode. We call anchor-completely to do this. Then we correct the location
;;; of the parent of the new tree so that it lies above its kids at the
;;; average of their position.

;;; And we are done. We call redraw-everything to show the changes and 
;;; we call clean the mess to deal with user requests
;;; that we could not attend to during the work.

;;; Note that the code is embedded in setfs of *within-planting*. This
;;; variable is used to tell the system that we are in the middle of
;;; planting and thgerefore it is not safe to let the user scroll and do
;;; other things.

(defun dynamic-plant-and-reshape (root-tasks)
  (setf *within-planting* t)
  (let* ((parent (inference-parent (car root-tasks)))
	 parent-gnode y-level root-gnode)	
    (setf root-tasks (remove-if #'find-gnode (find-visible-roots root-tasks)))
    (if parent
      (setf parent-gnode (find-gnode parent)
	    y-level (y-to-level (gnode-y-pos parent-gnode)))
      (setf y-level -1))
    ;; POSITION EACH ROOT
    (dolist (root root-tasks)
      (clear-xedge)
      (grow-the-tree root parent-gnode (1+ y-level))
      (setf root-gnode (find-gnode root))
      (enroll-tree-in-taken-spots root-gnode)
      (plant-new-root root-gnode parent-gnode (1+ y-level))
      (when parent 
	(check-and-correct-collisions root-gnode (1+ y-level)))
      (sort-gnodes-by-x))
    ;; ANCHOR EACH NEW SUBTREE AND FIX THE PARENTS
    (dolist (root root-tasks)
      (anchor-completely (find-gnode root)))
    (when parent-gnode
      (propagate-locations-to-immediate-parent parent-gnode))
    (redraw-everything *viewer-current-root*)
    (clean-the-mess))
  (setf *within-planting* nil))
          
;;; plant-new-root. This function is called when a new subtree has been
;;;                 created in order to put this subtree below its
;;; planted ancestor. We do this in three steps: we put the new tree
;;; below its planted ancestor, we correct any collisions we might have
;;; with left siblings, and the we try to move all siblings so that they
;;; stay centered around their parent. This last step, if possible, avoids
;;; the tree from growing towards the right all the time.

;;; We find the x position of the parent. If this position is greater than
;;; the position of the new root, then we are done if we increase the x
;;; position of the root (and all its children) by the difference between
;;; the parent's position and its own. On the other hand, if the new root
;;; is located to the right of the parent, then this can be corrected by
;;; shifting the new root to the left. But, by the way grow-the-tree works,
;;; the leftmost node of the new tree is at the left edge of the extent, and
;;; therefore shifting it to the left would push it out of the world. So, the
;;; solution in this case is to shift the whole tree until the new subtree 
;;; can be placed below its planted ancestor without being shifted to the left.

;;; But once we place the new tree below its planted ancestor, we might have
;;; a collision with the older siblings (the ones to the left). So, we find
;;; out if any collisions exist and if so shift the new tree to the right as
;;; mush as is necessary to avoid them.

;;; Since new siblings are added to the right of old siblings, the tree
;;; rooted at the parent will grow to the right all the time. But, if
;;; there is space to the left we can move all the siblings to the left,
;;; thus giving the impression that the tree grows from the center to the
;;; outside. We call shift-kids-to-left to attempt this.

(defun plant-new-root (root parent y-level)
  (when parent
    (setf *left-edge* nil)
    (setf *right-edge* nil)
    (let* ((root-pos (find-root-position parent))
	   (dpos (- root-pos (gnode-x-pos root)))
	   dx left-edge siblings-to-the-left right-edge
	   all-sibs)
      ;; PLACING THE NEW TREE BELOW ITS PLANTED ANCESTOR
      (when (minusp dpos)
	(setf dx (max 1000 (+ *gnode-separation* (- dpos))))
	(setf all-sibs (gn-kids parent))
	(setf (gn-kids parent) (remove root all-sibs))
	(complete-shift-subtree-by-dx (car (aref *viewer-spots-taken* 0)) dx)
	(setf *viewer-extent-width* (+ *viewer-extent-width* dx))
	(setf (gn-kids parent) all-sibs)
	(setf dpos (- (+ root-pos dx) (gnode-x-pos root))))
      (shift-subtree-by-dx root y-level dpos)
      ;; SHIFTING NEW TREE AWAY FROM LEFT SIBS
      (setf left-edge (compile-left-edge root y-level)
	    siblings-to-the-left (find-siblings-to-the-left root y-level)
	    right-edge (compile-right-edge-for-kids 
			(reverse siblings-to-the-left) y-level)
	    dx (worst-collision right-edge left-edge))
      (unless (zerop dx)
	(shift-subtree-by-dx root y-level (+ *gnode-separation* dx)))
      ;; CENTERING SIBLINGS
      (shift-kids-to-left parent))))

;;; shift-kids-to-left. When this function is called, a subtree has just
;;;                     been added to parent, to the right of all the
;;; previous subtrees. This function is called to move all the subtrees
;;; of parent to the left so that the parent is located at the center of
;;; all its children.

;;; We first determine how big the required shift will be and whether
;;; shifting by this amount is possible. If it is, we shift each subtree
;;; and we are done.

;;; The shift is possible when two conditions are met:
;;; 1. The trees to be shifted do not collide with other trees as a
;;;    result of the shift and,
;;; 2. No node belonging to the trees to be shifted gets moved to a
;;;    negative location as a result of the shift (this is tested by
;;;    the predicate shift-plausible).

(defun shift-kids-to-left (parent)
  (let* ((kids (gn-kids parent))
	 (kid-level (y-to-level (gnode-y-pos (car kids))))
	 (bad-root-pos (avg-kids kids))
	 (current-root-pos (gnode-x-pos parent))
	 (desired-dx (- bad-root-pos current-root-pos))
	 left-edge right-edge generation)
    (unless (zerop desired-dx)
      (setf *left-edge* nil
	    *right-edge* nil)
      (setf left-edge (add-to-edge (compile-left-edge-for-kids kids kid-level)
				   (- desired-dx))
	    generation (aref *viewer-spots-taken* kid-level)
	    right-edge (compile-right-edge-for-kids 
			(reverse
			 (subseq generation 
				 0
				 (position (car (last kids)) generation)))
			kid-level))
      (when (and (shift-plausible left-edge desired-dx)
		 (zerop (worst-collision right-edge left-edge)))
	(dolist (child kids)
	  (shift-subtree-by-dx child kid-level (- desired-dx))
	  (anchor-completely child))))))

;;; check-and-correct-collisions. This function is called when the new 
;;;                               subtree has been planted below its
;;; parent, at the right of the older siblings. The new subtree might
;;; have collided with other trees to the right of it. This function
;;; corrects these collisions by shifting the trees to the right away
;;; from the new subtree (ie more to the right).

;;; This is quite easy. We compile the right edge of the new subtree and
;;; the left edge of all subtrees rooted at all the nodes to the right
;;; of the new root. We find out the amount to shift by. If it is non
;;; zero we shift each subtree to the right by the required amount. Each
;;; time we shift a tree we correct the location of its parent so that
;;; it stays immediately above the child.

;;; At the end, we call propagate-locations-to-top on all the trees shifted
;;; to deal with ancestors that have been left very far away from its their
;;; children.

(defun check-and-correct-collisions (root y-level)
  (unless (null (gn-dad root))
    (let ((siblings-to-the-right 
	   (find-planted-siblings-to-the-right root y-level))
	  (dx (find-collision-magnitude root y-level)))
      (unless (zerop dx)
	(dolist (sib siblings-to-the-right)
	  (complete-shift-subtree-by-dx sib dx)
	  (propagate-locations-to-immediate-parent (gn-dad sib))
	  (unless (collisions-persist sib y-level)
	    (return-from check-and-correct-collisions)))
	(propagate-locations-to-top siblings-to-the-right)))))

;;; propagate-locations-to-top. This function is called when the trees rooted
;;;                             at the gnodes gnodes have been shifted to the
;;; right. As a result of this shift, all the ancestors of these roots are
;;; no longer placed to be at the center of their children's positions. This
;;; function corrects this situation by working bottom up changing the
;;; locations of the ancestors to the right thing. The effect is a small shift
;;; of all ancestors to the right.

;;; Since shifting the whole path to the root too often is unnecessary (since
;;; the displacement might be too small to bother the user), for each 
;;; ancestor only relocate it if a threshold displacement is exceeded. For
;;; this we call relocate-gnode-if-necessary.

;;; The loop in this function builds a list which is the union of all the
;;; immediate parents of gnodes (duplicates removed). We recurse on this
;;; list which gets smaller and smaller as we go up the tree.

(defun propagate-locations-to-top (gnodes)
  (cond ((null gnodes) nil)
	(t (let (parent-gnodes)
	     (dolist (gnode gnodes)
	       (unless (or (null (gn-dad gnode))
			   (member (gn-dad gnode) parent-gnodes))
		 (push (gn-dad gnode) parent-gnodes)
		 (relocate-gnode-if-necessary (car parent-gnodes))))
	     (propagate-locations-to-top parent-gnodes)))))

;;; propagate-locations-to-immediate-parent. Shifts dad so that it is located
;;;                                          at the center of its offspring.
;;; Then it calls propagate-if-necessary to shift the ancestors (only if they
;;; are too far off from the correct position).

(defun propagate-locations-to-immediate-parent (dad)
  (when dad (relocate-gnode dad))
  (propagate-if-necessary (gn-dad dad)))

;;; propagate-if-necessary. Shifts ancestors so that they lie at the center
;;;                         of their children, but only if the shift is
;;; necessary. So, for each node on the path from gnode to the root of the
;;; tree we call relocate-gnode-if-necessary to do this.

(defun propagate-if-necessary (gnode)  
  (cond ((null gnode) nil)
	(t (relocate-gnode-if-necessary gnode)
	   (propagate-if-necessary (gn-dad gnode)))))

;;; relocate-gnode-if-necessary. Shifts gnode so that it is at the center of
;;;                              its children's positions only if necessary.
;;; The shift is necessary if gnode is very far away from its correct 
;;; position (3 *gnode-separation*s).

;;; We find the x position where gnode should be and compare it to the
;;; position it is in. If the required displacement is larger than the
;;; threshold, we shift the subtree rooted at gnode by the required amount,
;;; and check and correct any collisions we might have created. When we are
;;; done with this node we recurse on its parent.

(defun relocate-gnode-if-necessary (gnode)
  (let ((kids (gn-kids gnode))
	dx)
    (if (= 1 (length kids))
	(setf dx (- (gnode-x-pos (car kids)) (gnode-x-pos gnode)))
      (setf dx (- (avg-kids kids) (gnode-x-pos gnode))))
    (when (> dx (* 3 *gnode-separation*))
      (shift-gnode-anchoring-by gnode dx)
      (check-and-correct-collisions 
       gnode 
       (y-to-level (gnode-y-pos gnode))))))

;;; relocate-gnode. Shifts gnode so that it is placed at the center of its
;;;                 children's positions. This function is the same as
;;; relocate-gnode-if-necessary except that here gnode is relocated even
;;; if it is very close from where it should be.

(defun relocate-gnode (gnode)
  (let ((kids (gn-kids gnode))
	dx)
    (if (= 1 (length kids))
	(setf dx (- (gnode-x-pos (car kids)) (gnode-x-pos gnode)))
      (setf dx (- (avg-kids kids) (gnode-x-pos gnode))))
    (unless (zerop dx)
      (shift-gnode-anchoring-by gnode dx)
      (check-and-correct-collisions 
       gnode 
       (y-to-level (gnode-y-pos gnode))))))

;;; 2.4 Utilities
;;;     ---------

;;; 2.4.1 Positioning Utilities. Functions used by the functions above to
;;;       ---------------------  determine where to place subtrees.

;;; add-to-edge. Given an edge structure (a list of (level xpos) pairs),
;;;              this function adds dx to the xpos in each pair. By
;;; doing this we can test the effects of shifting the tree whose edge
;;; we are processing by dx.

(defun add-to-edge (edge dx)
  (cond ((null edge) nil)
	(t (cons (list (caar edge) (+ (cadar edge) dx))
		 (add-to-edge (cdr edge) dx)))))

;;; shift-plausible. A shift is plausible if it does not put any gnode
;;;                  to close to the left edge of the extent (x < 200).

(defun shift-plausible (edge dx)
  (cond ((null edge) t)
	((< (cadar edge) 200) nil)
	(t (shift-plausible (cdr edge) dx))))

;;; worst-collision. Returns the amount to shift the tree by 
;;;                  to avoid collisions to the left. 0 is returned 
;;; if no collisions are encountered. The assumption is that the
;;; tree whose left edge we are processing might be colliding with
;;; some tree to its left (right is the right edge of all the trees
;;; to the left of the crasher).

(defun worst-collision (right left)
  (let ((largest-crash 0)
        rpos lpos)
    (setf right (reverse right))
    (setf left (reverse left))
    (loop
      (when (or (null right) (null left))
	(return-from worst-collision largest-crash))
      (setf rpos (cadr (pop right)))
      (setf lpos (cadr (pop left)))
      (when (> (+ rpos *gnode-separation*) lpos)
        (setf largest-crash (max largest-crash 
				 (- (+ rpos *gnode-separation*) lpos)))))))

;;; collisions-persist. Returns t if the tree rooted at root is in a
;;;                     collision state (nodes are position on top of
;;; each other). We find the size of the worst collision. If it is positive
;;; then we know a collision occured and we return t.

(defun collisions-persist (root y-level) 
   (plusp (find-collision-magnitude root y-level)))

;;; find-collision-magnitude. Returns the size of the worst collision
;;;                           between the tree rooted at root and all
;;; subtrees to the right of it.

(defun find-collision-magnitude (root y-level)
  (setf *left-edge* nil)
  (setf *right-edge* nil)
  (let* ((right (compile-right-edge root y-level))
	 (siblings-to-the-right 
	  (find-planted-siblings-to-the-right root y-level))
	 (left (compile-left-edge-for-kids (reverse siblings-to-the-right)
					   y-level)))
	 (worst-collision right left)))

;;; 2.4.2 Tree Shifting Utilities. These functions are used to change the
;;;       -----------------------  location of gnode subtrees.

;;; complete-shift-subtree-by-dx. This function shifts the subtree rooted
;;;                               at root by dx. Note that each node in
;;; the tree is already fully planted. This means that there is a lot of
;;; data to change about each node and this is expensive.

(defun complete-shift-subtree-by-dx (root dx)
  (shift-gnode-anchoring-by root dx)
  (complete-shift-subtree-by-dx-for-kids (gn-kids root) dx))

;;; complete-shift-subtree-by-dx-for-kids. Given a list of gnodes, it calls
;;;                                        complete-shift-subtree-by-dx on
;;; each one.

(defun complete-shift-subtree-by-dx-for-kids (kids dx)
  (dolist (child kids)
    (complete-shift-subtree-by-dx child dx)))

;;; shift-gnode-anchoring-by. This function shifts gnode by dx. This involves
;;;                           changing the positions of many things that are
;;; cached in the gnode structure. When done we check whether the gnode has
;;; been moved out of the extent. If so, we resize the extent and redraw.

(defun shift-gnode-anchoring-by (gnode dx)
  (setf (cw::region-left (gn-area gnode)) ; area
    (+ (cw::region-left (gn-area gnode)) dx))
  (setf (cw::active-region-left (gn-actv-reg gnode)) ; active region
    (+ (cw::active-region-left (gn-actv-reg gnode)) dx))
  (setf (cw::position-x (gn-position gnode)) ; position
    (+ (cw::position-x (gn-position gnode)) dx))
  (setf (cw::position-x (gn-prop-pos gnode)) ; prop position
    (+ (cw::position-x (gn-prop-pos gnode)) dx))
  (setf (cw::position-x (gn-truth-pos gnode)) ; truth pos
    (+ (cw::position-x (gn-truth-pos gnode)) dx))
  (setf (cw::position-x (gn-to-dad gnode)) ; to dad
    (+ (cw::position-x (gn-to-dad gnode)) dx))
  (setf (cw::position-x (gn-to-kids gnode)) ; to kids
    (+ (cw::position-x (gn-to-kids gnode)) dx)) 
  (setf (cw::position-x (cadr (gn-hilite-args gnode))) ; hilite area
    (+ (cw::position-x (cadr (gn-hilite-args gnode))) dx))
  (shift-gnode-draw-args-by gnode dx)
  (check-extent-limits (gnode-x-pos gnode) (gnode-y-pos gnode)))

;;; shift-gnode-draw-args-by. This function shift the draw arguments of gnode
;;;                           by dx.

(defun shift-gnode-draw-args-by (gnode dx)
  (dolist (one-arg (gn-draw-args gnode))
    (when (cw::position-p one-arg)
      (setf (cw::position-x one-arg)
	(+ (cw::position-x one-arg) dx)))))

;;; 2.4.3 Information Gathering Utilities. Used to find and test lots of
;;;       -------------------------------  things...

;;; is-new-tree. Returns t if the task belongs to a tree that does not belong
;;;              to the tree we are building yet. A task belongs to a new
;;; tree if at least one of its successors is visible and does not already
;;; appear in our tree.

;;; This function is very important in making all visible nodes appear in the
;;; tree. The problem is that plant-and-show-truth is called from recompute-
;;; truth-value when we revise the truth value of a task. If anytime trace
;;; is on in its simplest form (all inference tasks are visible), it might
;;; be the case that an inference task is created and its truth value set.
;;; Then recompute-truth-value is called on its parent, and never again on
;;; the inference node. The parent is not visible, but it has a successor
;;; which is and is not already in our tree.

(defun is-new-tree (task)
  (cond ((null task) nil)
	((and (funcall *display-test* task)
	      (not (find-gnode task)))
	 t)
	(t (dolist (child (inference-children task) nil)
	     (when (is-new-tree child)
	       (return-from is-new-tree t))))))

;;; find-visible-roots. This function returns all new roots to be planted.
;;;                     Again, this is used to deal with problems that arise
;;; when anytime-trace is on. In that case, plant-and-show-truth might be
;;; called with a modal task which is not visible. Therefore, this task will
;;; not be a root of our tree. What we need to do is find all its descendants
;;; which are visible. There might be more than one, and so, this function
;;; returns a list of new roots.

;;; It works by going down the tree from the original task. When a visible
;;; task is found we store it and prune its subtree from our search.

(defun find-visible-roots (tasks)
  (let (root-kids kids)
    (dolist (task tasks root-kids)
      (if (funcall *display-test* task)
	  (push task root-kids)
	(progn (setf kids (inference-children task))
	       (setf root-kids
		 (append root-kids (find-visible-roots kids))))))))

;;; find-root-position. Returns the position of a new subtree whose root
;;;                     is a child of parent. If parent has only one kid
;;; (the new root), its position is right below the parent. If there are
;;; more siblings, the new root goes to the right of the last one. Since
;;; kids are stored in reverse order of appeareance (youngest first), the
;;; first kid is the new root and the second is its predecessor (the last
;;; planted child). We get that node's position and add *gnode-separation*.

(defun find-root-position (parent)
  (let ((kids (cdr (gn-kids parent))))
    (if kids
	(+ *gnode-separation* (gnode-x-pos (car kids)))
      (gnode-x-pos parent))))

;;; find-child-of-planted-ancestor. This function is called when plant-and-
;;;                                 show-truth is called with a task. The
;;; task might belong to a new subtree, but may be deep in it. Since we want
;;; to plant the entire subtree, we need to find the first task that is not
;;; in our tree and who is an ancestor of task. This is what this funtion
;;; does.

(defun find-child-of-planted-ancestor (task)
  (let ((parent-task (consider-parent task)))
    (cond ((null parent-task) task)
	  ((find-gnode parent-task) task)
	  (t (find-child-of-planted-ancestor parent-task)))))

;;; find-siblings-to-the-right. Returns all gnodes that are to the right of
;;;                             gnode at level y-level. Since we have the
;;; *viewer-spots-taken* array, this is easy: we find gnode in its level
;;; and return all nodes that are after it.

(defun find-siblings-to-the-right (gnode y-level)
  (let* ((taken-positions (aref *viewer-spots-taken* y-level))
	 (pos (1+ (position gnode taken-positions))))
    (subseq taken-positions pos (length taken-positions))))

;;; find-siblings-to-the-left. Returns all gnodes that are to the left of
;;;                            gnode at level y-level. Since we have the
;;; *viewer-spots-taken* array, this is easy: we find gnode in its level
;;; and return all nodes that are before it.

(defun find-siblings-to-the-left (gnode y-level)
  (let* ((taken-positions (aref *viewer-spots-taken* y-level))
	 (pos (position gnode taken-positions)))
    (subseq taken-positions 0 pos)))

;;; find-planted-siblings-to-the-right. Returns all the subtrees to the
;;;                                     right and below root which have
;;; already been planted. This is done because we do not want to deal with
;;; subtrees that we might be ready to plant after the current one. We will
;;; deal with the collisions of these as they get planted.

(defun find-planted-siblings-to-the-right (root y-level)
  (let ((sibs (find-siblings-to-the-right root y-level)))
    (remove-if #'(lambda (x) (null (gn-prop-pos x))) sibs)))

;;; 2.4.4 Book Keeping Utilities. These functions deal with maintaining the
;;;       ----------------------  data structures the dynamic viewer uses.
;;;                               The array *viewer-spots-taken* is the most
;;;                               important.

;;; clear-xedge. This function sets the variable *viewer-xedge* to a clean
;;;              xedge. This is required to allow grow-the-tree to make
;;; each new tree as if it was the only one in existance.

(defun clear-xedge ()
  (setf *viewer-xedge* (make-array 320 :initial-element 200)))

;;;  enroll-tree-in-taken-spots. Enrolls all the gnodes of the tree rooted
;;;                              at root in the *viewer-spots-taken* array.
;;; Just calls enroll-in-taken-spots for all nodes in the tree.

(defun enroll-tree-in-taken-spots (root)
  (enroll-in-taken-spots root)
  (dolist (child (reverse (gn-kids root)))
    (enroll-tree-in-taken-spots child)))

;;; enroll-in-taken-spots. Adds gnode to the *viewer-spots-taken* array at
;;;                        the appropriate level and position. There are a
;;; number of cases to deal with:

;;; 1. If gnode is the root it is easy. We list gnode and put it in level
;;;    0 of the array.
;;; 2. If the node is not the root, we check to see if it is not the first
;;;    child. If so, it fits right after its previous sibling and we call
;;;    insert-in-taken-positions to do it.
;;; 3. If the node is an only child, we know that it fits before all its
;;;    cousins who are children of uncles to the left of its parent. So we
;;;    call find-spot-by-ancestry to deal with it.

;;; When done we check whether we have placed a node outside of the extent and
;;; take appropriate actions.

(defun enroll-in-taken-spots (gnode)
  (let* ((y (gnode-y-pos gnode))
	 (x (gnode-x-pos gnode))
	 (y-level (y-to-level y))
	 (taken-positions (copy-list (aref *viewer-spots-taken* y-level)))
	 (sibs (if (gn-dad gnode) (gn-kids (gn-dad gnode)) nil))
	 (pos (position gnode sibs))
	 (prev-sib (if (and pos (not (= pos (1- (length sibs)))))
		       (elt sibs (1+ pos))
		     nil)))
    (if (gn-dad gnode)
	(setf (aref *viewer-spots-taken* y-level)
	  (if prev-sib
	      (insert-in-taken-positions taken-positions gnode prev-sib)
	      (find-spot-by-ancestry gnode y-level)))
      (setf (aref *viewer-spots-taken* y-level) (list gnode)))
    (check-extent-limits x y)))

;;; find-spot-by-ancestry. Returns the entry of *viewer-spots-taken*
;;;                        for the level of gnode with gnode inserted in it.
;;; This is called when gnode is an only child. This function, thus, will
;;; place gnode after all its cousins which are children of uncles which
;;; lie to the left of the parent of gnode.

;;; It loops through all the nodes to its left until it finds one whose
;;; dad is not a left uncle. At that point it calls insert-in-taken-positions
;;; to insert the gnode in the list. If we reach the end of the nodes and
;;; they all are left cousins, we call insert-at-end to place the gnode at
;;; the end of the list.

(defun find-spot-by-ancestry (gnode y-level)
  (let* ((taken-positions (copy-list (aref *viewer-spots-taken* y-level)))
	 (dad (gn-dad gnode))
	 (above-taken-positions (aref *viewer-spots-taken* (1- y-level)))
	 (pos (position dad above-taken-positions))
	 (left-uncles (subseq above-taken-positions 0 pos))
	 (my-pos 0))
    (dolist (sib taken-positions (insert-at-end gnode taken-positions))
      (when (not (member (gn-dad sib) left-uncles))
	(return-from find-spot-by-ancestry
	  (insert-in-taken-positions taken-positions gnode 
				     (if (zerop my-pos) 
					 nil 
				       (elt taken-positions (1- my-pos))))))
      (incf my-pos))))

;;; insert-at-end. Inserts gnode at the end of the list taken-positions.

(defun insert-at-end (gnode taken-positions)
  (insert-in-taken-positions taken-positions 
			     gnode 
			     (car (last taken-positions))))

;;; insert-in-taken-positions. Inserts gnode right after prev-sib. 
;;;                            If prev sib is null it inserts gnode
;;; at the head of the list.

(defun insert-in-taken-positions (taken-positions gnode prev-sib)
  (if prev-sib
      (append 
       (subseq taken-positions 0 (1+ (position prev-sib taken-positions)))
       (cons gnode 
	     (nthcdr (1+ (position prev-sib taken-positions))
		     taken-positions)))
    (cons gnode taken-positions)))

;;; enroll-proof-in-reusable-proofs. This function is called each time
;;;                                  a gnode is created to enroll its
;;; proof (if it has one) in the list *viewer-reusable-proofs* which
;;; contains all the proof invocations that are shared by different
;;; tasks. The maintainace of this list is crucial in making the labeling
;;; of proof invocations on gnodes work.

;;; This is a bit complicated because at the time of gnode creation MVL
;;; might not have the proof ready for us. There are 2 cases: 
;;; 1. If we got a proof. Then count the # of tasks and if > 1 
;;;    the proof is shared and we put it in the list.
;;; 2. we got a proposition. Then we look through the list of proofs in 
;;;    the analysis structure. If we find the proposition we enroll the proof
;;;    in our list. This is based on the assumption that if the
;;;    proof exists already it is because another task created
;;;    it BEFORE.
;;; Before we start we reset gn-prover just in case the system has by this
;;; time finished with the creation of the proof structure.

(defun enroll-proof-in-reusable-proofs (gnode)
  (when (inference-p (gn-task gnode))
    (setf (gn-prover gnode) (inference-proof (gn-task gnode)))
    (let ((prover (gn-prover gnode))
	  new-proof)
      (if (proof-p prover)
	  (when (proof-slaves prover)
	    (setf new-proof prover))
	(let ((the-proof (find-its-proof (gn-prop gnode) 
					 (analysis-proofs 
					  (find-the-analysis)))))
	  (when the-proof
	    (setf new-proof the-proof))))
      (when (and new-proof (not (member new-proof *viewer-reusable-proofs*)))
	(setf *viewer-reusable-proofs*
	  (append *viewer-reusable-proofs* (list new-proof)))))))

(defun find-the-analysis ()
  (if (boundp (quote analysis))
      analysis
    (let* ((process (find "Initial Lisp Listener"
			  mp::*all-processes* :key #'mp:process-name
			  :test #'string=))
	   (sgroup (mp:process-stack-group process)))
      (mp:symeval-in-stack-group (quote analysis) sgroup))))

;;; find-its-proof. Returns a proof structure for proving prop is any. NIL is
;;;                 returned otherwise.

(defun find-its-proof (prop all-proofs)
  (find prop all-proofs :key #'proof-prop :test #'samep))


;;; 2.4.5 Dealing with user requests during dynamic viewing.
;;;       -------------------------------------------------

;;; clean-the-mess. During dynamic viewing there are certain pariods in which
;;;                 the data structures needed to do static operations (such
;;; as scrolling), are corrupted. This happens when we are in the process of
;;; creating a new subtree. Allowing user requests during these periods to
;;; go through can result in crashes of our program. Thus, we catch user
;;; requests and process them only when we are done creating a new tree and
;;; the data structures are restored.

;;; This function is called after finishing the planting of a new tree to deal
;;; with user requests. If the user has asked for scrolling the overviewer, we
;;; do so and then clear the variable that signals the request. The user can
;;; also ask for rearrangement of all the gnodes in the tree by calling static
;;; display. If so, we call the static-display-during-dynamic-viewing function.
;;; The break option allows the user to interrupt MVL and the viewer without
;;; crashing Common Windows.

;;; Repainting of the visible part of the extent is only done when it is
;;; necessary. If some new node has been placed in the visible area, or some
;;; connection between nodes that crosses the visible area has changed
;;; location, then we need to repaint the visible area. The variable
;;; *viewer-dirty*, when t states that repainting needs to be done.

;;; Since as a result of the creation of new subtrees visible nodes might have
;;; become inactive, we need to check for this each time and show the changes
;;; to the user. If a node has changed status, we need to change its icon in
;;; the viewer. The function check-life-signs does it.

(defun clean-the-mess ()
  (when *overviewer-scroll-signaled*
    (do-overviewer-scroll *overviewer-scroll-signaled*)
    (setf  *overviewer-scroll-signaled* nil))
  (if *viewer-dirty* 
    (progn (viewer-repaint *viewer*)
	   (setf *viewer-dirty* nil))
    (check-life-signs))
  (when *static-display-signaled*
    (case *static-display-signaled*
      (static-display (static-display-during-dynamic-viewing)
		      (redraw-everything *viewer-current-root*))
      (break (setf *static-display-signaled* 'stoped)
	     (break)
	     (setf *static-display-signaled* nil))))
  (setf *static-display-signaled* nil))

;;; signal-static-display. If this is called during dynamic viewing, we set
;;;                        *static-display-signaled* to 'static-display to
;;; signal that we want static-display done as soon as possible. This turns
;;; out to be when we are done planting the latest subtree. If we are not
;;; viewing dynamically, we can just do the static display right away.

(defun signal-static-display ()
  (if (and *dynamic-viewer-active* *within-planting*)
      (setf *static-display-signaled* 'static-display)
    (progn (static-display-during-dynamic-viewing)
	   (root-on-proposition nil 
	    :gnode (find-gnode (gn-task *viewer-current-root*))))))

;;; signal-break. When dynamic viewing, this function sets 
;;;               *dynamic-viewer-active* to 'break to signal that the
;;; user wants to break as soon as possible.

(defun signal-break ()
  (if *dynamic-viewer-active*
      (setf *static-display-signaled* 'break)))

;;; signal-overviewer-scroll. Sets *overviewer-scroll-signaled* to the 
;;;                           mouse state to signal that the user wants
;;; to scroll as soon as possible. By saving the mouse state we save the
;;; position the user was pointing at.

(defun signal-overviewer-scroll (mouse-state)
  (setf *overviewer-scroll-signaled* mouse-state))

;;; check-life-signs. This function checks all nodes that are visible in the
;;;                   viewer window to see whether they have changed status
;;; (active or inactive). If they have, we need to redraw their icon to let
;;; the user know this has happened. The field drawn-as-active in each
;;; gnode states how the gnode was drawn the last time. If its task's value
;;; of inference-active-p is different from this, we need to redraw the icon
;;; for the gnode.

(defun check-life-signs ()
  (let* ((x (cw::region-left *viewer-region*))
	 (gnodes (collect-exposed-gnodes 
		 *viewer-region*
		 (- x 100)
		 (+ x (cw::region-width *viewer-region*) 100))))
    (dolist (gnode gnodes)
      (when (and (inference-p (gn-task gnode))
		 (not (equal (gn-drawn-as-active gnode)
			     (inference-active-p (gn-task gnode)))))
	(clear-doubtful-icon gnode)
	(draw-gnode gnode)))))

;;; clear-doubtful-icon. This function is called when a gnode is not
;;;                      drawn according to its active-p status. The
;;; node will be redrawn properly by a call to draw-gnode above, but
;;; before we do that we better erase the icon the gnode is using now.
;;; This is what this function does.

(defun clear-doubtful-icon (gnode)
  (let* ((draw-args (gn-draw-args gnode))
	 (draw-pos (cadr draw-args)))
    (cw::clear-rectangle-xy *viewer* 
			    (cw::position-x draw-pos)
			    (cw::position-y draw-pos)
			    (caddr draw-args)
			    (cadddr draw-args))))
 
;;; 2.5 Updating the Overviewer.
;;;     -----------------------

;;; redraw-everything. This function is called to show the new additions
;;;                    to the user. At this point we assume that the extent 
;;; has been resized to the smallest region containing the tree on display. 
;;; Thus all there is to do is to redraw the viewer and the overviewer. Note
;;; that this function is given the node that the user wants to see.
;;; Root-on-proposition is told to make that node the root of the visible
;;; region.

(defun redraw-everything (root-gnode)
  (root-on-proposition nil :gnode root-gnode)
  (update-overviewer t))

;;; anchor-in-overviewer. This function places gnode in the overviewer. This
;;;                       involves the following: changing the value of the
;;; ov-position field, if the gnode has changed positions (as opposed to being
;;; placed for the first time), we need to clear it, together with all its
;;; connections. and the redraw it in the new location with all its
;;; connections. Note that if the gnode was visible in the viewer and it is
;;; not visible as a result of the new position, we set *viewer-dirty* to t,
;;; since the viewer will have to be repainted.

(defun anchor-in-overviewer (gnode)
  (let* ((pos (gn-position gnode))
	 (x (cw::position-x pos))
	 (y (cw::position-y pos))
	 (scaled-x (scale-to-ov-x x))
	 (scaled-y (scale-to-ov-y y))
	 (scaled-pos (cw::make-position :x scaled-x :y scaled-y))
	 (dad (gn-dad gnode))
	 (kids (gn-kids gnode)))
    (unless (and (gn-ov-position gnode)
		 (equalp scaled-pos (gn-ov-position gnode)))
      (unless (null (gn-ov-position gnode))
	(draw-connections gnode dad kids boole-clr t)
	(when (gnode-was-visible gnode)
	  (setf *viewer-dirty* t)))
      (setf (gn-ov-position gnode) scaled-pos)
      (draw-connections gnode dad kids boole-1 nil))))

;;; gnode-was-visible. Returns t if gnode was visible (when the old position of
;;;                    the gnode in the overviewer is contained in the region
;;; of the overviewer that is visible in the viewer.

(defun gnode-was-visible (gnode)
  (if (gn-ov-position gnode)
      (cw::region-contains-position-p 
       *overviewer-region* 
       (gn-ov-position gnode))
    nil))

;;;  clear-anchor-in-overviewer. Clears a gnode from the overviewer.

(defun clear-anchor-in-overviewer (gnode)
  (let* ((pos (gn-position gnode))
	 (x (cw::position-x pos))
	 (y (cw::position-y pos))
	 (scaled-x (scale-to-ov-x x))
	 (scaled-y (scale-to-ov-y y))
	 (scaled-pos (cw::make-position :x scaled-x :y scaled-y))
	 (dad (gn-dad gnode))
	 (kids (gn-kids gnode)))
    (unless (and (gn-ov-position gnode)
		 (equalp scaled-pos (gn-ov-position gnode)))
      (unless (null (gn-ov-position gnode))
	(draw-connections gnode dad kids boole-clr t)
	(check-for-corruption gnode dad kids))
      (setf (gn-ov-position gnode) scaled-pos))))

;;; check-for-corruption. This is called to check whether gnode, by changing
;;;                       positions, has corrupted the display in the viewer
;;; window. This happens when: the gnode was visible before it moved, its
;;; parent was visible (then by moving the child, their connection which is
;;; visible changes position), or when a child of the gnode was visible. In
;;; any of these cases we set *viewer-dirty* to make clean-the-mess repaint
;;; after the planting of the new subtree completes. Note that if 
;;; *viewer-dirty* is set already, we don't bother.

(defun check-for-corruption (gnode dad kids)
  (cond (*viewer-dirty* nil)
	((gnode-was-visible gnode) (setf *viewer-dirty* t))
	((and dad (gnode-was-visible dad)) (setf *viewer-dirty* t))
	(t (dolist (kid kids)
	     (when (gnode-was-visible kid)
	       (setf *viewer-dirty* t)
	       (return-from check-for-corruption))))))

;;; show-anchor-in-overviewer. Calls draw-connections to draws the gnode 
;;;                            in the overviewer.

(defun show-anchor-in-overviewer (gnode)
  (let ((dad (gn-dad gnode))
	(kids (gn-kids gnode)))
    (draw-connections gnode dad kids boole-1 nil)))

;;; draw-connections. Draws the gnode and its connections to parent and kids
;;;                   in the overviewer. Note that drawing of the connections
;;; is always done from parent to child!!!

;;; This function draws the connection from parent to gnode, then from
;;; gnode to its children, and only then the gnode is drawn. Note that
;;; the connections do not have any points in common with the gnodes they
;;; connect. This is done because if we are using xor to draw and erase,
;;; points are left over if there are points in common.

;;; VERY IMPORTANT!!! This function should draw connections in exactly
;;; the same way that ov-connect-to-children does.

(defun draw-connections (gnode dad kids op erasing)
  (let ((source (cw::make-position))
	(dest (cw::make-position)))
    ;; DRAWING CONNECTION TO DAD
    (when dad 
      (setf (cw::position-x source) (cw::position-x (gn-ov-position dad)))
      (setf (cw::position-y source) 
	(- (cw::position-y (gn-ov-position dad))
	   (if (gn-selected dad) 4 3)))
      (setf (cw::position-x dest) (cw::position-x (gn-ov-position gnode)))
      (setf (cw::position-y dest)
	(+ (cw::position-y (gn-ov-position gnode))
	   (if (gn-selected gnode) 3 2)))
      (cw::draw-line *overviewer* source dest :operation op))
    (setf (cw::position-x source) (cw::position-x (gn-ov-position gnode)))
    (setf (cw::position-y source) 
      (- (cw::position-y (gn-ov-position gnode))
	 (if (gn-selected gnode) 4 3)))
    ;; DRAWING CONNECTIONS TO KIDS
    (draw-connections-to-children kids source dest op)
    ;; DRAWING THE GNODE ITSELF
    (if (gn-selected gnode)
	(if erasing 
	    (ov-clear-selected-gnode gnode op)
	  (ov-draw-selected-gnode gnode op))
      (cw::draw-point *overviewer* (gn-ov-position gnode)
		      :brush-width 4
		      :operation op))))

;;; draw-connections-to-children. Draws the connections from gnode to
;;;                               its kids.

(defun draw-connections-to-children (kids source dest op)
  (dolist (child kids)
    (when (gn-ov-position child)
      (setf (cw::position-x dest) (cw::position-x (gn-ov-position child)))
      (setf (cw::position-y dest)
	(+ (cw::position-y (gn-ov-position child)) 
	   (if (gn-selected child) 3 2)))
      (cw::draw-line *overviewer* source dest :operation op))))
  
;;; check-extent-limits. This function is called to check whether a gnode is
;;;                      being placed outside of the extent. If so, the extent
;;; is resized. Since the overviewer shows the whole extent, it needs to be
;;; repainted so that all nodes can be shown.

(defun check-extent-limits (x y)
  (let ((no-change t)
	(beyond-height (< y (- 32000 *viewer-extent-height*)))
	(beyond-width (> x *viewer-extent-width*)))
    (when beyond-height
      (setf *viewer-extent-height* (+ *viewer-extent-height* 2000))
      (setf no-change nil))
    (when beyond-width
      (setf *viewer-extent-width* (+ *viewer-extent-width* 3000))
      (setf no-change nil))
    (when (or beyond-height beyond-width)
      (update-overviewer))))

;;; update-overviewer. Since this is called when the size of the extent
;;;                    changes, the first thing we do is update the scale
;;; variables *overviewer-x/y-scale*. Then we redraw the whole overviewer
;;; to show the tree in relation to the new extent. Before drawing we
;;; clear the rectangle that hilites the visible region. After drawing we
;;; restore the rectangle.

(defun update-overviewer (&optional (show-user nil))
  (setf *overviewer-x-scale* (/ *viewer-extent-width*
				(car *overviewer-size*)))
  (setf *overviewer-y-scale* (/ *viewer-extent-height*
				(cadr *overviewer-size*)))
  (when show-user
    (dynamic-clear-hilite)
    (overviewer-redisplay (prop-find-gnode *viewer-root-prop*))
    (dynamic-show-hilite)))

;;; overviewer-redisplay. Redraws the tree in the overviewer. For this it calls
;;;                       overviewer-clear to clear everything and then,
;;; overviewer-redraw to draw everything again.

(defun overviewer-redisplay (gnode)
  (overviewer-clear gnode)
  (overviewer-redraw gnode))

;;; overviewer-clear. Clears every gnode of the subtree rooted at gnode from
;;;                   the overviewer.

(defun overviewer-clear (gnode)
  (clear-anchor-in-overviewer gnode)
  (dolist (kid (gn-kids gnode))
    (overviewer-clear kid)))

;;;  overviewer-redraw. Draws every gnode of the subtree rooted at gnode in
;;;                     the overviewer.

(defun overviewer-redraw (gnode)
  (show-anchor-in-overviewer gnode)
  (dolist (kid (gn-kids gnode))
    (overviewer-redraw kid)))

;;; 2.6 Showing truth value changes.
;;;     ---------------------------

;;; update-the-truth. This function is called when a task changes truth
;;;                   value. When this happens we do the following:
;;; 1. Change gn-truth-text to the new truth text.
;;; 2. Recompute gn-truth-pos.
;;;    If the gnode is showing in the viewer, redraw it.
;;; 3. Make the gnode blink a number of times in the overviewer.

(defun update-the-truth (task)
  (let ((gnode (find-gnode task)))
    (when gnode
      (let* ((truth (compute-truth-value task))
	     (truth-text (fit-the-text truth))
	     (pos (gn-position gnode))
	     (orig-x (cw::position-x pos))
	     (orig-y (cw::position-y pos))
	     (to-top (/ (cadr *gnode-size*) 2))
	     x y)
	;;; CHECKING if magnifier update is necessary
	(update-the-truth-in-magnifier task truth)
	;;; CLEARING the old truth value
	(setf (cw::region-width *truth-region*)          
	  (+ (* 6 (length (gn-truth-text gnode))) 6))
	(setf (cw::region-bottom *truth-region*)
	  (- (cw::position-y (gn-truth-pos gnode)) 2)) ; these numbers
	(setf (cw::region-left *truth-region*)         ; determined
	  (- (cw::position-x (gn-truth-pos gnode)) 2)) ; empirically...
	(cw::clear-area *viewer* *truth-region*) 
        ;;; CHANGING gn-truth-text
	(setf (gn-truth-text gnode) truth-text)
        ;;; RECOMPUTING gn-truth-pos
	(setf y (+ orig-y to-top 5))
	(setf x (- orig-x (* 3 (length truth-text))))
	(setf (gn-truth-pos gnode) (cw::make-position :x x :y y))
	(draw-gnode gnode)
        ;;; BLINKING IN OVERVIEWER
	(dotimes (i 6)
	  (dotimes (j 10)
	    (cw::draw-point *overviewer* (gn-ov-position gnode)
			    :brush-width (* 4 (- 6 i))
			    :operation boole-xor)))))))


;;; update-the-truth-in-magnifier. If the magnifier window is up and it
;;;                                happens to be showing the task which
;;; is changing truth value, this function changes the truth value that
;;; appears in the truth window of the magnifier.

(defun update-the-truth-in-magnifier (task truth)
  (when (and *magnifier* (equal task (mag-task *magnifier*)))
    (let ((mag (mag-window *magnifier*))
	  (trect (mag-truth-rect *magnifier*))
	  (truth-text (truncate-to-fit (format nil "~a" truth) 21)))
      (cw::clear-area mag (mag-truth-area *magnifier*))
      (cw::set-window-stream-cursor-position mag (mag-truth-pos *magnifier*))
      (format mag truth-text)
      (cw::draw-rectangle mag (car trect)
			  (cadr trect) (caddr trect)
			  :brush-width 2))))


;;; 2.7 Static Display During Dynamic Viewing
;;;     -------------------------------------

;;; static-display-during-dynamic-viewing. This function is called when the
;;;                                        user decides the dynamic display
;;; has become too messy and requests a perfect arrangement of all the nodes
;;; we have so far. The cursor becomes a clock to show the user that nothing
;;; can be done during this period.

(defun static-display-during-dynamic-viewing ()
  (let* ((root-gnode (car (aref *viewer-spots-taken* 0)))
	 (task (gn-task root-gnode)))
    (dynamic-unview)
    (update-mouse-cursor 'busy)
    (lock-viewer)
      (init-viewer-during-dynamic-viewing)
      (plant-and-reshape task)
      (setf *viewer-spots-taken* (make-array 320 :initial-element nil))
      (enroll-tree-in-taken-spots (prop-find-gnode *viewer-root-prop*))
      (reset-selected-list)
      (when *gnodes*
	(unless *dynamic-viewer-active* 
	  (resize-downstairs-world))
	(init-overviewer-during-dynamic-viewing)
	(shrink-forest (prop-find-gnode *viewer-root-prop*)))
    (unlock-viewer)
    (dynamic-view)
    (setf *viewer-current-root* (find-gnode (gn-task *viewer-current-root*)))
    (if *dynamic-viewer-active*
	(overviewer-draw-rect #'cw::draw-rectangle)
      (overviewer-draw-rect #'cw::draw-filled-rectangle))
    (update-mouse-cursor 'active)))

;;; reset-selected-list. This function resets the list of selected gnodes
;;;                      after a static display during dynamic viewing.
;;; This is necessary because the static display creates new gnodes for the
;;; same tasks and therefore, the gnodes in the list *selected-gnodes* do
;;; not refer to any gnode the user sees any longer.

;;; We solve this problem by looking for the new gnode that corresponds
;;; to each selected task and we push that gnode into the list.

(defun reset-selected-list ()
  (let ((old-selected-gnodes *selected-gnodes*)
	new-gnode)
    (setf *selected-gnodes* nil)
    (dolist (old-gnode old-selected-gnodes)
      (setf new-gnode (find-gnode (gn-task old-gnode)))
      (setf (gn-selected new-gnode) t)
      (push new-gnode *selected-gnodes*))))

;;; update-mouse-cursor. Changes the cursor according to situation. If
;;;                      situation is 'busy, the cursor becomes a clock.
;;; If it is 'active it becomes the usual north west arrow. We need to
;;; change the cursor in the viewer and overviewer windows and also in all
;;; the active regions that are contained in the viewer.

(defun update-mouse-cursor (situation)
  (let ((cursor (case situation
		  (busy cw::*mouse-cursor-timer*)
		  (active cw::*mouse-cursor-northwest-arrow*))))
    (setf (cw::window-stream-mouse-cursor *viewer*) cursor)
    (setf (cw::window-stream-mouse-cursor *overviewer*) cursor)
    (dolist (reg (cw::window-stream-active-regions *viewer*))
      (setf (cw::active-region-mouse-cursor reg) cursor))))

;;; overviewer-draw-rect. This function draws a rectangle showing the
;;;                       user what area of the overviewer is currently
;;; being seen. This is necessary after the static display to put the
;;; overviewer in the state the hilite functions expect.

(defun overviewer-draw-rect (draw-function)
  (funcall draw-function
	   *overviewer*
	   (cw::make-position :x (cw::region-left *overviewer-region*)
			      :y (cw::region-bottom *overviewer-region*))
	   (cw::region-width *overviewer-region*)
	   (cw::region-height *overviewer-region*)
	   :operation boole-xor))

;;; init-viewer-during-dynamic-viewing. Reinitializes the data structures
;;;                                     the static display function needs
;;; in order to work.

(defun init-viewer-during-dynamic-viewing ()
 (let* ((width (car *viewer-size*))
	(height (cadr *viewer-size*))
	(bottom (- 32000 height))
	(left 0))
    (setf (cw::window-stream-extent-width *viewer*) 32000)
    (setf (cw::window-stream-extent-height *viewer*) 32000)
    (setf (cw::window-stream-y-offset *viewer*) bottom)
    (setf *viewer-region* (cw::make-region :bottom bottom
				       :left left
				       :width width
				       :height height))
    (setf *gnodes* nil)
    (setf *gprops* (make-hash-table :test #'equal))
    (setf *visible-gnodes* nil)
    (setf *collisions-occured* nil)
    (setf *viewer-xedge*  (make-array 320 :element-type 'integer
				      :initial-element 100))))

;;; init-overviewer-during-dynamic-viewing. Reinitializes the overviewer
;;;                                         to the state it needs to be
;;; for the static display function. It needs to be clear.

(defun init-overviewer-during-dynamic-viewing ()
  (setf *overviewer-x-scale* (/ *viewer-extent-width*
				(car *overviewer-size*)))
  (setf *overviewer-y-scale* (/ *viewer-extent-height*
				(cadr *overviewer-size*)))
  (cw::clear *overviewer*))

;;; ---------------------------------
;;;             THE END
;;; ---------------------------------


