;;; -*- Mode: LISP; Syntax: Common-lisp; Package: MVL; Base: 10 -*-

(in-package "MVL")

#-ANSI (proclaim '(declaration dynamic-extent))

;; functions to produce bilattice of functions from a dag to a
;; bilattice; this is the hardest of all the constructions.  We begin by
;; setting up all of the machinery needed to define dags and functions
;; on them.

;; a dag is defined as a structure with the following fields:
;;  root	The root of the dag
;;  eq 		Tests for equality of two dag elements
;;  leq		Checks to see if arg1 is below arg2 in the dag.  This
;;  		function is optional and will be computed if not
;;  		supplied.
;;  dot		Given two dag elements, returns a list of all their
;;  		glb's
;;  add		Given two dag elements, attempt to combine them into a
;;              single one
;;  sub		Given two dag elelents, try to remove the second from
;;  		the first (the second is assumed to be an instance of the
;;  		first)
;;  long short	Long and short descriptors
;;  inherit	This is a function that takes four arguments -- a bilattice,
;;		two elements of the dag, say d and e, and an element of the
;;		bilattice, say b.  If we have a function f with f(d)=b, then
;;		the result computed by the inherit function should be
;;		f(e).  inherit defaults to a function that ignores d and
;;		e and just returns b.
;;  vars	Returns the vars in a particular dag element.
;;  plug 	Plugging function for dag elements.

(defstruct (dag (:print-function print-dag))
  root						
  eq leq dot (add nil) (sub nil)
  long short					
  (inherit nil)
  (vars nil)
  (plug nil))

(defun print-dag (dag stream print-depth)
  (declare (ignore print-depth))
  (format stream "~a dag" (dag-short dag)))

(defun dag-eql (dag d1 d2)
  (or (eql d1 d2) (funcall (dag-eq dag) d1 d2)))

(defun dag-le (dag d1 d2)
  (funcall (dag-leq dag) d1 d2))

;; inheritance.  Accepts five args: a bilattice, two dag elements, an
;; element of the bilattice, and a dag.  If the inheritance fn is nil,
;; return the bilattice element.  If the dag elements are equal, return
;; the bilattice element.  Otherwise, call the function.

(defun dag-inh (bilattice d1 d2 b dag &aux (inh (dag-inherit dag)))
  (if (or (not inh) (dag-eql dag d1 d2)) b (funcall inh bilattice d1 d2 b)))

;; functions from dags to a bilattice are basically lists; each element
;; on the list is of the form (d b d1 ... dk) where d is a point of the
;; dag, b is the value taken there and the di are other points where the
;; dag takes a value just below d.  In actuality, the list (d b di) is an
;; instance of the dag-entry structure.

;; Dag functions also have dag and bilattice slots, since these are
;; always getting passed around in any case.

(defstruct (dag-fn (:print-function print-dag-fn)
	    (:constructor make-dag-fn (dag bilattice list)))
  dag bilattice list)

(defun print-dag-fn (f stream print-depth)
  (declare (ignore print-depth))
  (fn-print f stream))

;; An element of the dag-fn-list constains information about the dag
;; point, the value, the *immediate* successors to this point, and *all*
;; successors to it.

(defstruct (dag-entry 
	    (:constructor make-dag-entry
			  (pt val imm &optional (succ (copy-list imm)))))
  pt val imm succ)

;; To make the following code more readable, we define some functions to
;; operate on lists of instances of this structure.  dag-root-pt is the
;; dag point of the first element of the list; dag-root-val is the value
;; taken there.  dag-fn-root-val is the value taken by a dag-fn (itself a
;; structure) at the root of the dag.

(defmacro dag-root-pt (x) `(dag-entry-pt (car ,x)))
(defmacro dag-root-val (x) `(dag-entry-val (car ,x)))
(defun dag-fn-root-val (f) (dag-root-val (dag-fn-list f)))

;; to copy a dag-fn, we also copy the entries

(defun copy-entire-fn (fn)
  (setf fn (copy-dag-fn fn)
	(dag-fn-list fn) (copy-dag-entries (dag-fn-list fn)))
  fn)  

;; copy all the dag entries on a list by copying each one ...

(defun copy-dag-entries (list)
  (mapcar #'copy-single-entry list))

;; ... and copy a single entry by copying the structure and the lists of
;; successors.

(defun copy-single-entry (x)
  (make-dag-entry (dag-entry-pt x) (dag-entry-val x)
		  (copy-list (dag-entry-imm x))
		  (copy-list (dag-entry-succ x))))

;; find-entry locates an element in the list of points; the test is eql
;; unless use-dag-eq is non-NIL, in which case it should be the dag-fn
;; from which the list was extracted.  member-entry is the same but uses
;; member instead of find.

(defun find-entry (x list &optional use-dag-eq)
  (find x list :key #'dag-entry-pt
	:test (if use-dag-eq (dag-eq (dag-fn-dag use-dag-eq)) #'eql)))

(defun member-entry (x list &optional use-dag-eq)
  (member x list :key #'dag-entry-pt
	  :test (if use-dag-eq (dag-eq (dag-fn-dag use-dag-eq)) #'eql)))

;; Some other useful utilities

;; get all the points where a dag-fn takes a value.  It's easy -- just
;; cons the root point onto all the successors of the root point!
;; all-dag-pts accepts a dag-fn as an arg; all-dag-pts-1 accepts the
;; first element of the dag-fn-list and is more convenient in some cases.

(defun all-dag-pts (f) (all-dag-pts-1 (car (dag-fn-list f))))

(defun all-dag-pts-1 (root)
  (cons (dag-entry-pt root) (dag-entry-succ root)))

;; Here we recompute the immediate successors for a new dag-entry.
;; keep-fringe-only does the work.

(defun recompute-dag-entry (pt val succ orig)
  (make-dag-entry pt val (keep-fringe-only succ orig t) succ))

;; Here we find the fringe of a set of points.  list is from the
;; original dag list, and contains all the information about what is a
;; successor to what.  keep-dag-greatest indicates whether we want to
;; keep dag-maximal (closest to the root) or dag-minimal elements of
;; points.

(defun keep-fringe-only (points list keep-dag-greatest? &optional (copy? t))
  (delete-subsumed-entries (if copy? (copy-list points) points)
			   (if keep-dag-greatest?
			       #'(lambda (x y) (known-below x y list))
			     #'(lambda (x y) (known-below y x list)))))

;; x is known to be below y given the information in list if x is a
;; member of the successor set of y.

(defun known-below (x y list)
  (member x (dag-entry-succ (find-entry y list))))

;; in the following code, we uniformly use d, d1, d2 etc for elements of
;; the dag.  We use b for an element of the initial bilattice, and f for
;; a function from the dag to the bilattice (i.e., an element of the
;; constructed bilattice).

;; functions defined:
;; 
;; get-val (d f)		returns value of f at dag point d
;; 
;; make-tv (dag bilattice d b &optional root-fn)
;; 				makes fn that has value (root-fn bilattice) at
;;				root and b at d
;; 
;; make-root (dag bilattice b)	makes fn that has value b at root
;; 
;; dag-change-tv (f fn &optional simplify)
;; 				changes function f by applying bilattice
;;				function fn to each point.  If simplify is
;;				non-nil, simplifies the result.
;;
;; dag-change-dag (f fn)	like dag-change-tv, but fn is a function
;;				from the dag to itself
;;
;; dag-list-of-answers (f)	Returns a list of instances of the
;; 				answer structure, where the dag is
;; 				assumed to be that of binding lists
;;
;; dag-accumulate-fn (dag bilattice list &optional modifier)
;;				accepts a list of (dag-pt . value) pairs and
;;				constructs a dag function from them (this
;;				function does the partial sort and little more)
;; 
;; dag-prune (f pred)		prunes from f those entries not satisfying
;; 				predicate pred
;; 
;; dag-drop (f pred)		remove any entries satisfying pred
;;
;; combine-fns (f1 f2 fn)	combines functions f1 and f2 with
;; 				the function fn
;; 
;; add-dag (d b f &optional modifier (simplify t))
;;				given function f, add value b at dag element d
;; 
;; simplify (f)			removes multiple entries from dag-fn

;; given a truth value and a dag element, get the bilattice value.  If
;; there is an explicit value at this point, we use that.  Otherwise, we
;; look for successors to the given point that are above d.  If we find
;; one, we recur.  If not, we return the value at the root.  At the end,
;; we dot everything together.

(defun get-val (d f &aux (list (dag-fn-list f)) temp)
  (if (setq temp (or (find-entry d list) (find-entry d list f)))
      (dag-entry-val temp)
    (gv-1 (dag-fn-dag f) (dag-fn-bilattice f) d (car (dag-fn-list f))
	  (dag-fn-list f))))

;; The recursive part.  Find all immediate successors below the given
;; value.  Then recur and combine with dot.  If there aren't any, inherit
;; the current value.

(defun gv-1 (dag bilattice d curr list
	     &aux (temp (remove-if-not #'(lambda (x) (dag-le dag d x))
				       (dag-entry-imm curr))))
  (if temp
      (reduce (bilattice-dot bilattice)
	      (mapcar #'(lambda (x) 
			  (gv-1 dag bilattice d (find-entry x list) list))
		      temp))
    (dag-inh bilattice (dag-entry-pt curr) d (dag-entry-val curr) dag)))

;; given a dag point and a bilattice point, make a function that is unknown
;; except below the dag point, where it takes the stated value.  There are
;; two cases:
;;  1.  If the stated value is the same as the value at the root would
;;  be anyway, it's easy.  This can happen either because the value is
;;  the root value, or because the dag element d is the root of the dag.
;;  2.  Otherwise, we have to supply values at two points instead of just
;;  one.  We also have to make a dag-entry for d and b a successor to the
;;  value at the root.

(defun make-tv (dag bilattice d b &optional (root-fn #'bilattice-unknown)
		&aux (root-val (funcall root-fn bilattice)))
  (if (or (mvl-eq b root-val bilattice) (dag-eql dag d (dag-root dag)))
      (make-root dag bilattice b)
    (make-dag-fn dag bilattice
		 (list (make-dag-entry (dag-root dag) root-val (list d))
		       (make-dag-entry d b nil)))))

;; same as above, but we assume that the dag element is the root of the dag

(defun make-root (dag bilattice b)
  (make-dag-fn dag bilattice (list (make-dag-entry (dag-root dag) b nil))))

;; given an f and a function from the underlying bilattice to itself,
;; apply the function to every entry in f.  This is really pretty easy;
;; we just apply the function at each point and simplify if necessary.
;; Note that this function is not destructive, so we need to copy the
;; dag-fn before making the change.

(defun dag-change-tv (f fn &optional (simplify t))
  (setq f (copy-entire-fn f))
  (mapc #'(lambda (x) (setf (dag-entry-val x) (funcall fn (dag-entry-val x))))
	(dag-fn-list f))
  (if simplify (simplify f) f))

;; same as dag-change-tv, but fn is from dag to dag.  This is done by
;; working through the dag-fn, applying fn to each dag point in it and
;; adding the information that the function takes the same value at the
;; new dag point.  Then we collect them all with dag-accumulate-fn.

(defun dag-change-dag
    (f fn &optional (bilattice-modify #'bilattice-plus)
     &aux (dag (dag-fn-dag f)) (bilattice (dag-fn-bilattice f)) new-list)
  (declare (dynamic-extent new-list))
  (setq new-list
    (mapcar #'(lambda (x)
		(cons (funcall fn (dag-entry-pt x)) (dag-entry-val x)))
	    (dag-fn-list f)))
  (unless (dag-eql dag (dag-root dag) (caar new-list))
    (push (cons (dag-root dag) (bilattice-unknown bilattice)) new-list))
  (dag-accumulate-fn dag bilattice new-list 
		     (funcall bilattice-modify bilattice)))

;; Accumulate answers in a dag-list in a returned list, where the dag is
;; assumed to be the dag of binding lists.

(defun dag-list-of-answers (f)
  (mapcar #'(lambda (x) (make-answer (dag-entry-pt x) (dag-entry-val x)))
	  (dag-fn-list f)))

;; Dag accumulation function.  Accepts dag, bilattice, list of dotted
;; (dag . bilattice) pairs.  There is also a modifier keyword that tells
;; you what to do if you are trying to overwrite a value that is already
;; specified.  (The default is just to use the new value.)  Just work
;; through the pairs one at a time, calling add-dag at each point.

;; The only subtlety is if we're getting a value at the root.  Since the
;; modifier might not be to keep the new value, we want to construct the
;; value from the beginning with the given value.  That's what the
;; initial make-root call sorts out.

(defun dag-accumulate-fn
    (dag bilattice list &optional (modifier #'return-first-arg) &aux ans temp)
  (setq ans 
    (make-root dag bilattice
	       (if (or (eql modifier #'return-first-arg)
		       (not (setq temp (find (dag-root dag) list
					     :key #'car :test (dag-eq dag)))))
		   (bilattice-unknown bilattice)
		 (cdr temp))))
  (dolist (item list (simplify ans))
    (add-dag (car item) (cdr item) ans modifier)))

(defun return-first-arg (new-val old-val)
  (declare (ignore old-val))
  new-val)

(defun return-second-arg (new-val old-val)
  (declare (ignore new-val))
  old-val)

;; same as dag-change-tv, but prune answers based on some predicate.
;; Just call dag-change-tv, where the funtion either does nothing or, if
;; the value fails to satisfy the predicate, is unknown.

(defun dag-prune (f pred &aux (bilattice (dag-fn-bilattice f))
			      (u (bilattice-unknown bilattice)))
  (dag-change-tv f #'(lambda (x) (if (funcall pred x) x u))))

;; completely drop points given some function on dag-entries saying when
;; to do so.  We tear the list apart, remove points satisfying the test,
;; and then put it back together.

(defun dag-drop (f test-fn &aux list flag)
  (dolist (item (cdr (dag-fn-list f)))
    (if (funcall test-fn item)
	(setq flag t)
      (push item list)))
  (if flag
      (dag-accumulate-fn (dag-fn-dag f) (dag-fn-bilattice f)
			 (napcar #'(lambda (x) 
				     (cons (dag-entry-pt x)
					   (dag-entry-val x)))
				 (cons (car (dag-fn-list f)) list)))
    f))

;; given two truth values and a function to apply to the truth values,
;; combine the functions.  Call combine-1 to do the work.

(defun combine-fns (f1 f2 fn)
  (combine-1 fn '(t t) f1 f2))

;; Here's where we actually do the combination.  We might be able to do
;; it cheaply; that's what combine-quick is about.  If not, combine-slow
;; does the work.

;; There is also a need to handle parametric arguments.  The argument
;; type-info to combine-1 should have 0 in the position of any parametric
;; argument (like the args slot of a modal operator).  The first thing we
;; do is split the given args (the fx) into parameters and real
;; functional arguments.  Then we only call combine-quick or slow with
;; dag points from the real functions.  Before actually applying the
;; function, we will need to splice the parametric arguments back into
;; the list.

(defun combine-1 (fn type-info &rest fx &aux parameters functions)
  (mapc #'(lambda (f type)
	    (if (eql type 0) (push f parameters) (push f functions)))
	fx type-info)
  (setq parameters (nreverse parameters))
  (or (combine-quick fn parameters type-info functions)
      (combine-slow fn parameters type-info functions)))

;; combine-quick handles the case where one of the f's takes values at
;; points that cover all the points where *any* f takes a value.  In this
;; particular case, we can actually use something like dag-change-tv to
;; do all the work!

;; fx is the list of functions to be combined, and is reversed when we
;; get it.  We begin by finding that fi that takes values at as many
;; points as possible, calling it longest.  Now we check to make sure
;; that every fi is either longest or takes values only where longest
;; does (that's what quick-check is for).

;; If this all works, we copy longest, reverse the function list, apply
;; the function at each possible place, and then simplify the result.

(defun combine-quick (fn parameters param-info fx 
		      &aux temp (longest (car fx)) 
			   (max (length (dag-fn-list longest))) 
			   (eq (dag-eq (dag-fn-dag longest))))
  (dolist (f (cdr fx))
    (setq temp (length (dag-fn-list f)))
    (when (> temp max) (setq max temp longest f)))
  (when (every #'(lambda (f)
		   (or (eql f longest)
		       (quick-check (dag-fn-list longest) (dag-fn-list f) eq)))
	       fx)
    (setq longest (copy-entire-fn longest)
	  fx (reverse fx))
    (mapc #'(lambda (x &aux (pt (dag-entry-pt x)))
	      (setf (dag-entry-val x)
		(apply fn (splice-parameters parameters 
					     (mapcar #'(lambda (f)
							 (get-val pt f))
						     fx)
					     param-info))))
	  (dag-fn-list longest))
    (simplify longest)))

(defun quick-check (long short eq)
  (subsetp short long :key #'dag-entry-pt :test eq))

;; Here is the more general case.  The way it works is as follows:
;;  1.  We begin by setting raw-list to a list of (pts . vals) where
;;      pts and vals are the points where the functions take values and
;;      the values taken there.  cs-1 does this.
;;  2.  We then compute the function values.
;;  3.  We commit to using the known value at the root.
;;  4.  For each other value, unless we know that the value will be
;;      subsumed by others, we take all possible dots of points contributing
;;      to it and push them on the accumulated list.
;;  5.  We call dag-accumulate-fn to construct the final function.

(defun combine-slow (fn parameters type-info functions
		     &aux (dag (dag-fn-dag (car functions)))
			  (bilattice (dag-fn-bilattice (car functions)))
			  (dot (dag-dot dag)) (eq (dag-eq dag))
			  (raw-list (cs-1 functions (list nil))) acc)
  (setq functions (nreverse functions))
  (mapc #'(lambda (x) 
	    (setf (cdr x)
	      (apply fn (splice-parameters parameters (cdr x) type-info))))
	raw-list)
  (setq acc (list (cons (dag-root dag) (cdar raw-list))))
  (mapc #'(lambda (x)
	    (unless (combine-subsumed functions (car x) (cdr x) raw-list dag
				      bilattice)
	      (pushconc (mapcar #'(lambda (y) (cons y (cdr x)))
				(get-all-dots dot eq (cdar x) (list (caar x))))
			acc)))
	(cdr raw-list))
  (dag-accumulate-fn dag bilattice acc #'return-second-arg))

;; Here we construct the raw list.  If there are functions left to go,
;; consider the first.  For each point where it takes a value, we take
;; every point on the answer list and pust the point on the <pts> (i.e.,
;; car) and the value on the <vals> (i.e., cdr).

(defun cs-1 (fns ans)
  (if fns 
      (cs-1 (cdr fns)
	    (apply #'nconc
		   (mapcar #'(lambda (pt)
			       (mapcar #'(lambda (x)
					   (cons (cons (dag-entry-pt pt)
						       (car x))
						 (cons (dag-entry-val pt)
						       (cdr x))))
				       ans))
			   (dag-fn-list (car fns)))))
    ans))

;; Here we see if a value is going to be subsumed by other things.  args
;; are functions, dag points in question, overall value to be taken, the
;; original list of (<pts> . val), and the dag and bilattice involved.
;; The story is that if, for every function, if every predecessor of the
;; given dag point has the property that the value inherited using that
;; predecessor would be the same, then this particular value is just
;; going to be inherited from *its* immediate predecessors.

;; To test for this, we work through the fns/pts pairs one at a time.
;; Assuming that the given point isn't the root, we work through the
;; dag-entries of the function.  Whenever the given pt is an immediate
;; successor of an entry, we splice the entry-pt into the original list
;; of points and find it on the original list.  (We find it by looking
;; for something that is list-equal but not eql; eql is just the original
;; entry!)  Then if the value is different, we can return with failure;
;; if not, we keep looking.  At the end, of course, we have to splice the
;; given point back into the list before moving on to the next fn/pt.

(defun combine-subsumed (fns pts val list dag bilattice &aux (ptr pts))
  (do (p)
      ((null fns) t)
    (setq p (car ptr))
    (unless
	(or (dag-eql dag p (dag-root dag))
	    (prog1 
		(dolist (item (dag-fn-list (car fns)) t)
		  (when (member p (dag-entry-imm item))
		    (setf (car ptr) (dag-entry-pt item))
		    (unless
			(mvl-eq val
				(cdr (find pts list :key #'car
					   :test #'(lambda (x y)
						     (and (not (eql x y))
							  (every #'eql x y)))))
				bilattice)
		      (return))))
	      (setf (car ptr) p)))
      (return))
    (setf ptr (cdr ptr) fns (cdr fns))))

;; Here we have a list of dag elements and need to construct all the
;; results that are formed by dotting them all together.  The answer is
;; seeded with a list consisting of the first dag element.

;; To do it, we just work down the list.  Each time, we make ans the
;; result of dotting the given point with each element of ans, then
;; delete duplicates to keep things as simple as possible.

(defun get-all-dots (dot eq list ans)
  (dolist (pt list ans)
    (setq ans
      (delete-duplicates (mapcan #'(lambda (a) (funcall dot pt a)) ans)
			 :test eq))))

;; Here we put the parametric arguments back into the list.  params is a
;; non-reversed list of parameters and args is a non-reversed list of
;; arguments.  Info is the usual list of 0 if parameter or something else
;; if not.

;; If params is NIL, it's easy.  Otherwise, we just walk down info,
;; pushing the args onto the eventual answer unless it's a parameter, in
;; which case we grab a parameter and use that.

(defun splice-parameters (params args info)
  (if params
      (let (ans)
	(dolist (item info (nreverse ans))
	  (push (if (eql item 0) (pop params) (pop args)) ans)))
    args))

;; given a function, add a value at a point.  This function works by
;; side effect *only* -- nothing useful is returned.  The modifier
;; function is used to decide what to do if a value is already present at
;; the given point; simplify is used to simplify the result if needed.
;; We walk down the list; if the function already takes a value at d, we
;; modify the value as appropriate and simplify if required.  The
;; remaining case is handled by add-dag-1.

(defun add-dag (d b f &optional (modifier #'return-first-arg) simplify
		&aux (dag (dag-fn-dag f)) (list (dag-fn-list f)) temp)
  (if (setq temp (find-entry d list f))
      (setf (dag-entry-val temp) (funcall modifier b (dag-entry-val temp)))
    (add-dag-1 dag d b list (all-dag-pts f)))
  (when simplify (simplify f)))

;; add a value at a new point d.  For each point p to consider, there
;; are three cases:
;;  1.  d is above p.  Now we can add p and all of its successors to below-d.
;;  2.  d is below p.  We push p onto above-d.
;;  3.  d is incomparable with the point.  We proceed.

;; At the end, we have a list of points above and below d.  We compute
;; the points just above d and just below it using keep-fringe only, and
;; add in the new entry.

;; To clean up: For any point just above d, we add d to its list of
;; immediate successors, removing from that list any point just below d.
;; For any point above d, we add d to the list of that point's
;; successors.

(defun add-dag-1 (dag d b list remaining 
		  &aux above-d below-d just-above-d just-below-d)
  (do (p temp) 
      ((null remaining))
    (setq p (pop remaining))
    (cond ((dag-le dag p d)
	   (setq temp (intersection (dag-entry-succ (find-entry p list))
				    remaining)
	     remaining (set-difference remaining temp)
	     below-d (cons p (append temp below-d))))
	  ((dag-le dag d p) (push p above-d))))
  (setq just-above-d (keep-fringe-only above-d list nil)
	just-below-d (keep-fringe-only below-d list t))
  (push (make-dag-entry d b just-below-d below-d) (cdr list))
  (mapc #'(lambda (x &aux (pt (find-entry x list)))
	    (setf (dag-entry-imm pt)
	      (cons d (nset-difference (dag-entry-imm pt) just-below-d))))
	just-above-d)
  (mapc #'(lambda (x) (push d (dag-entry-succ (find-entry x list))))
	above-d))

;; to simplify all points of a dag-list, invoke simplify-at-point at
;; each one of them.  We have to be careful to simplify from the top
;; down; so we set remaining to what's left to do, and then gradually
;; work through it.  The point we pick to work on is any one that does
;; not appear as a successor to any other remaining point.

(defun simplify (f &aux (dag (dag-fn-dag f)) (bilattice (dag-fn-bilattice f))
			place (list (dag-fn-list f)))
  (do ((remaining (copy-list (all-dag-pts-1 (car list)))))
      ((null remaining) f)
    (setq remaining
      (delete-if #'(lambda (pt)
		     (unless (some #'(lambda (r)
				       (member pt (dag-entry-imm
						   (find-entry r list))))
				   remaining)
		       (setq place pt) t))
		 remaining :count 1))
    (simplify-at-point dag bilattice place list)))

;; simplify at a point.  There are three separate simplifications that
;; you can do:

;; 1.  Coalesce multiple children of this point that have the same value
;; into a single one.  The successors of the merge are the union of the
;; successors of the originals, and the immediates need to be recomputed.
;; The merge should replace the pair of merged points in any list that
;; includes them both.

;; 2.  Drop a point whose answer is the same as what you'd accumulate
;; from its predecessors anyway.  To do this, we go through the list,
;; removing it from any successor list, and also from any list of
;; immediate successors.  Of course, if we remove it from an immediate
;; list, we potentially have to add all of *its* immediate successors
;; into that immediate list.

;; 3.  If every predecessor of this point has the value x, then any
;; successor of the point that has x can potentially be subtracted from
;; the point.  To do the subtraction, we simply replace the point with
;; the difference in any list where it appears, and drop the point that
;; has been subsumed.

(defun simplify-at-point 
    (dag bilattice d orig &aux (entry (find-entry d orig)))
  (when (and (dag-add dag) (cdr (dag-entry-imm entry)))
    (simplify-coalesce dag bilattice entry orig))
  (when (ok-to-remove dag bilattice entry orig d)
    (simplify-remove entry orig d))
  (when (and (dag-sub dag) (dag-entry-imm entry))
    (simplify-subtract dag bilattice entry orig d)))

;; coalesce.  Go through the immediate successors of place, splitting
;; them by value taken.  Then do it by value.

(defun simplify-coalesce (dag bilattice place orig &aux imm val temp)
  (dolist (item (dag-entry-imm place))
    (setq val (dag-entry-val (find-entry item orig)))
    (if (setq temp (assoc val imm :test (bilattice-eq bilattice)))
	(push item (cdr temp))
      (push (list val item) imm)))
  (mapc #'(lambda (x) (coalesce-by-value dag (car x) (cdr x) orig)) imm))

;; By value: find a pair that coalese.  Then p1 and p2 are the original
;; entries for what coalesced; coalesced is the combined dag point, and
;; centry is the combined entry.  We add centry to the list, and then for
;; any list (including the original list argument!), if it contains both
;; d1 and d2, we remove them and add coalesced.  We then try again with
;; the reduced list.

(define-modify-macro cbv-0 (d1 d2 d) cbv-1)
(defun cbv-1 (list d1 d2 d)
  (if (and (member d1 list) (member d2 list))
      (cons d (delete d1 (delete d2 list :count 1) :count 1))
    list))

(defun coalesce-by-value (dag val list orig &aux p1 p2 coalesced centry)
  (multiple-value-bind (d1 d2) (find-coalescing-pair dag list)
    (when d1
      (setq p1 (find-entry d1 orig) p2 (find-entry d2 orig)
	    coalesced (funcall (dag-add dag) d1 d2)
	    centry (recompute-dag-entry coalesced val 
					(union (dag-entry-succ p1)
					       (dag-entry-succ p2))
					orig))
      (push centry (cdr orig))
      (dolist (item orig)
	(cbv-0 (dag-entry-imm item) d1 d2 coalesced) 
	(cbv-0 (dag-entry-succ item) d1 d2 coalesced))
      (coalesce-by-value dag val (cbv-0 list d1 d2 coalesced) orig))))

;; Here we find a pair of values that coalesce.  Provided there are at
;; least two values, just set d1 to the first and check the rest.  Keep
;; doing that until it works or you run out.

(defun find-coalescing-pair (dag list &aux d1 d2)
  (when (cdr list)
    (setq d1 (pop list)
	  d2 (find-if #'(lambda (d) (funcall (dag-add dag) d1 d)) list))
    (if d2 (values d1 d2)
      (find-coalescing-pair dag list))))

;; is it ok to replace the value at d/place?  Yes if the dot of values
;; from every immediate predecessor is the same as the value at place.
;; (Or if there are no such values because they've been coalesced away
;; or some such.)

(defun ok-to-remove (dag bilattice place orig d &aux ipreds)
  (setq ipreds
    (remove-if-not #'(lambda (x) (member d (dag-entry-imm x))) orig))
  (or (null ipreds)
      (mvl-eq (reduce (bilattice-dot bilattice)
		      (mapcar #'(lambda (y)
				  (dag-inh bilattice (dag-entry-pt y) d
					   (dag-entry-val y) dag))
			      ipreds))
	      (dag-entry-val place) bilattice)))

;; Here we actually do the removal.  We remove place from the
;; dag-fn-list, and then for every item on it, if d was an immediate
;; successor, we remove it and set a flag.  We also remove it from the
;; list of all successors but don't set the flag.  Then if flag is set,
;; we have to recompute the immediate successors, which are the fringe
;; of place's immediate successors and those that are already there.

(defun simplify-remove (place orig d &aux flag)
  (popf (cdr orig) place :count 1)
  (dolist (item orig)
    (setq flag nil)
    (setf (dag-entry-imm item)
      (delete-if #'(lambda (x) (when (eql x d) (setq flag t)))
		 (dag-entry-imm item) :count 1))
    (popf (dag-entry-succ item) d :count 1)
    (when flag
      (setf (dag-entry-imm item)
	(keep-fringe-only (nunion (dag-entry-imm item)
				  (copy-list (dag-entry-imm place)))
			  orig t nil)))))

;; Subtraction.  First we check to see that all the immediate
;; predecessors of place have a common value, setting ipreds to those
;; predecessors and val to the value.  If it worked, then we work through
;; place's immediate successors.  For any that has the same value, we
;; subtract from the current dag point of place, gradually constructing
;; newd for what's left and out as a list of points to remove.

;; Now let's say there *are* points to remove.  We first handle place,
;; changing d to newd and accumulating all the immediate successors into
;; one big list (of which we have to keep the fringe).  (This is done by
;; adjust-for-subtraction, below.)  Now we work through orig.  If any
;; successor list includes d, we remove the points in out and replace d
;; with newd.  In the immediate predecessors of d, we also replace d with
;; newd in the immediate successors list.

(defun simplify-subtract (dag bilattice place orig d &aux (newd d) diff out)
  (multiple-value-bind (ipreds val) (common-predecessors bilattice orig d)
    (when ipreds
      (dolist (item (dag-entry-imm place))
	(when (and (mvl-eq (dag-entry-val (find-entry item orig))
			   val bilattice)
		   (setq diff (funcall (dag-sub dag) newd item)))
	  (setq newd diff)
	  (push item out)))
      (when out
	(adjust-for-subtraction place newd orig out)
	(dolist (item orig)
	  (when (member d (dag-entry-succ item))
	    (setf (dag-entry-succ item)
	      (nsubst newd d (nset-difference (dag-entry-succ item) out)))))
	(dolist (item ipreds)
	  (setf (dag-entry-imm item) (nsubst newd d (dag-entry-imm item))))))))

;; Here we have an original list and a particular point d; do all the
;; immediate predecessors of d have the same value?  If so, return the
;; predecessors and the value.

(defun common-predecessors (bilattice orig d &aux ipreds ans)
  (dolist (item orig (values ipreds ans))
    (when (member d (dag-entry-imm item))
      (cond ((null ipreds) (push item ipreds) (setq ans (dag-entry-val item)))
	    ((mvl-eq (dag-entry-val item) ans bilattice) (push item ipreds))
	    (t (return nil))))))

;; As described above, here place is changing its dag-entry-pt to newd
;; and the elements of out are being removed.

(defun adjust-for-subtraction (place newd orig out)
  (setf (dag-entry-pt place)
    newd
    (dag-entry-imm place)
    (keep-fringe-only
     (reduce #'nunion
	     (mapcar #'(lambda (x) (dag-entry-imm (find-entry x orig))) out)
	     :initial-value (nset-difference (dag-entry-imm place) out))
     orig t nil)))

;; Here are the functions that correspond to distinguished elements of
;; a given bilattice.

(defun fn-elt (dag bilattice b-fn)
  (make-root dag bilattice (funcall b-fn bilattice)))

;; And here are the operators corresponding to and, or, etc.  Each of
;; these is constructed using combine-fns.
  
(defmacro fn-def (name &aux (fname (concsym 'fn- name))
			    (bname (concsym 'bilattice- name)))
  `(defun ,fname (f1 f2) (combine-fns f1 f2 (,bname (dag-fn-bilattice f1)))))

(fn-def and)
(fn-def or)
(fn-def plus)
(fn-def dot)
(fn-def dws)

;; negation is simpler -- we just change the truth values by negating
;; all of them!

(defun fn-not (f1)
  (dag-change-tv f1 (bilattice-not (dag-fn-bilattice f1)) nil))

;; checking for functional equality is rather harder.

(defun fn-eq (f1 f2 &aux (dag (dag-fn-dag f1)) (bilat (dag-fn-bilattice f1)))
  (set-equal (dag-fn-list f1) (dag-fn-list f2)
	     #'(lambda (x y)
		 (and (dag-eql dag (dag-entry-pt x) (dag-entry-pt y))
		      (mvl-eq (dag-entry-val x) (dag-entry-val y) bilat)))))

;; set equality for two lists that are likely to be short.  Cases where
;; either is of length 1 or 2 are handled separately.

(defun set-equal (s1 s2 test)
  (cond ((eq s1 s2))
	((not (and s1 s2)) nil)
	((and (cdr s1) (cdr s2))
	 (and (= (length s1) (length s2)) (subsetp s1 s2 :test test)))
	((or (cdr s1) (cdr s2)) nil)
	((funcall test (car s1) (car s2)))))

;; Comparison function: There is a quick and a slow version, like
;; combining.  The quick version is where f1 or f2 takes values at a
;; superset of the other, in which case we check all the points in the
;; larger set.  It's easy if we're checking f1; if f2, we have to reverse
;; the function passed to fn-comp.  fn-comp-slow is a little harder.

(defun fn-comp (f1 f2 fn &aux (l1 (dag-fn-list f1)) (l2 (dag-fn-list f2))
			      (eq (dag-eq (dag-fn-dag f1))))
  (cond ((> (length l1) (length l2))
	 (if (quick-check l1 l2 eq)
	     (fn-comp-quick l1 f2 fn)
	   (fn-comp-slow f1 f2 fn)))
	((quick-check l2 l1 eq)
	 (fn-comp-quick l2 f1 #'(lambda (x y b) (funcall fn y x b))))
	(t (fn-comp-slow f1 f2 fn))))

(defun fn-comp-quick (long-list short-fn fn 
		      &aux (b (dag-fn-bilattice short-fn)))
  (every #'(lambda (item) (funcall fn (dag-entry-val item) 
				   (get-val (dag-entry-pt item) short-fn)
				   b))
	 long-list))

;; Here's the general case.  We fail as soon as something goes wrong.

(defun fn-comp-slow (f1 f2 fn &aux checked (dag (dag-fn-dag f1))
				   (bilattice (dag-fn-bilattice f1))
				   (dot (dag-dot dag)) (eq (dag-eq dag)))
  (dolist (item1 (dag-fn-list f1) t)
    (dolist (item2 (dag-fn-list f2))
      (dolist (dag-elt (funcall dot (dag-entry-pt item1) (dag-entry-pt item2)))
	(unless (or (member dag-elt checked) (member dag-elt checked :test eq))
	  (if (funcall fn (get-val dag-elt f1) (get-val dag-elt f2) bilattice)
	      (push dag-elt checked)
	    (return-from fn-comp-slow nil)))))))

;; default stash and unstash just take the usual value, at the root of
;; the dag

(defun fn-stash-value (p dag bilattice)
  (make-root dag bilattice (stash-value p bilattice)))

(defun fn-unstash-value (p dag bilattice)
  (make-root dag bilattice (unstash-value p bilattice)))

;; Here's where we actually construct the new bilattice.

;; 1.  True, false, use fn-elt.  cutoff-val is also easy.

;; 2.  The bilattice functions have already been defined above, by and
;; large.  fn-comp is used for the comparison functions.

;; 3.  The modal operators are computed by f-modal-ops.

;; 4.  The long and short descriptions describe the bilattice as the set
;; of functions from the dag to the original bilattice.  The
;; one-character identifier is "e" (for "exponentiation") if available.

;; 5.  The type is "dag" and the components are the dag and the
;; bilattice.

(defun dag-bilattice (dag bilattice)
  (make-standard-modal-ops
    (make-bilattice :true (fn-elt dag bilattice #'bilattice-true)
		    :false (fn-elt dag bilattice #'bilattice-false)
		    :unknown (fn-elt dag bilattice #'bilattice-unknown)
		    :bottom (fn-elt dag bilattice #'bilattice-bottom)
		    :cutoff-val (fn-elt dag bilattice #'bilattice-cutoff-val)
		    :stash-val (take-1-more-arg #'fn-stash-value dag bilattice)
		    :unstash-val (take-1-more-arg
				  #'fn-unstash-value dag bilattice)
		    :plus #'fn-plus :dot #'fn-dot :and #'fn-and :or #'fn-or
		    :not #'fn-not :eq #'fn-eq
		    :kle (take-2-more-args #'fn-comp #'k-le)
		    :tle (take-2-more-args #'fn-comp #'t-le)
		    :dws #'fn-dws
		    :vars (dag-vrs dag bilattice)
		    :plug (dag-plg dag bilattice)
		    :absorb (dag-absorb dag bilattice)
		    :modal-ops (f-modal-ops bilattice)
		    :long-desc (format nil "Functions from ~a dag into ~a."
				       (dot-trim (dag-long dag))
				       (dot-trim
					(bilattice-long-desc bilattice)))
		    :short-desc (format nil "~a dag -> ~a" (dag-short dag)
					(bilattice-short-desc bilattice))
		    :char (bilattice-find-char #\e)
		    :type 'dag :components (list dag bilattice))))

;; The vars in a dag-list are those appearing in any bilattice or dag element

(defun dag-vrs (dag bilattice &aux (d-fn (dag-vars dag)) 
				   (b-fn (bilattice-vars bilattice)))
  (cond ((and b-fn d-fn) #'fn-vars)
	(b-fn #'fn-bvars)
	(d-fn #'fn-dvars)))

(defun fn-vars (f &aux (dag (dag-fn-dag f)) (bilattice (dag-fn-bilattice f)))
  (delete-duplicates
   (mapcan #'(lambda (x) (nconc (funcall (dag-vars dag) (dag-entry-pt x))
				(mvl-vars (dag-entry-val x) bilattice)))
	   (dag-fn-list f))))

(defun fn-bvars (f &aux (bilattice (dag-fn-bilattice f)))
  (delete-duplicates
   (mapcan #'(lambda (x) (mvl-vars (dag-entry-val x) bilattice))
	   (dag-fn-list f))))

(defun fn-dvars (f &aux (dag (dag-fn-dag f)))
  (delete-duplicates (mapcan #'(lambda (x) (funcall (dag-vars dag) x))
			     (all-dag-pts f))))

;; plugging function.  There are four cases, depending upon which of
;; d-fn and b-fn are NIL.

(defun dag-plg (dag bilattice 
		&aux (d-fn (dag-plug dag)) (b-fn (bilattice-plug bilattice)))
  (cond ((and b-fn d-fn) #'db-fn-vf-p)
	(b-fn #'db-fn-vf-pb)
	(d-fn #'db-fn-vf-pd)))

;; case where bilattice is default and dag isn't

(defun db-fn-vf-pd (val bdg &aux (d-fn (dag-plug (dag-fn-dag val))))
  (if (some #'(lambda (pt) (member pt bdg :key #'car)) (fn-dvars val))
      (dag-change-dag val #'(lambda (a) (funcall d-fn a bdg)))
    val))

;; case where dag is default and bilattice isn't

(defun db-fn-vf-pb
    (val bdg &aux (b-fn (bilattice-plug (dag-fn-bilattice val))))
  (dag-change-tv val #'(lambda (a) (funcall b-fn a bdg))))

;; neither is

(defun db-fn-vf-p (val bdg)
  (db-fn-vf-pd (db-fn-vf-pb val bdg) bdg))

;; absorption.  This code doesn't handle the case when the bilattice
;; itself has a nontrivial absorption function, just when plugging does
;; something interesting.  This is a bug, but the code would be awful and
;; there are no relevant instances yet.

(defun dag-absorb (dag bilattice)
  (declare (ignore bilattice))
  (when (dag-plug dag) #'db-absorb))

;; Here's where we actually do the work.  We work through the points in
;; the old value; for each, we plug in the bindings.  If it changes, we
;; add the value from new-val.  Then at the end, we add in all the stuff
;; from the new value and simplify.

(defun db-absorb (old-val bdg new-val
		  &aux (dag (dag-fn-dag old-val)) 
		       (bilattice (dag-fn-bilattice old-val))
		       (plugger (dag-plug dag)) 
		       (unknown (bilattice-unknown bilattice))
		       (plus (bilattice-plus bilattice))
		       plugged-dag-pt (result (copy-entire-fn old-val)))
  (setq new-val (mvl-plug new-val bdg))
  (dolist (dag-elt (dag-fn-list old-val))
    (unless (dag-eql dag (setq plugged-dag-pt
			   (funcall plugger (dag-entry-pt dag-elt) bdg))
		     (dag-entry-pt dag-elt))
      (add-dag plugged-dag-pt (get-val plugged-dag-pt new-val) result)))
  (dolist (dag-elt (dag-fn-list new-val) (simplify result))
    (add-dag (dag-entry-pt dag-elt) (dag-entry-val dag-elt) result plus)))

;; modal operators on functional bilattice.  Just copy over the modal
;; operators on the base bilattice, applying the modal operator function
;; to each point of a dag-list.

(defun f-modal-ops (bilattice)
  (single-modal-ops bilattice #'f-modal-fn))

;; modal operators are simple because combine-1 takes any number of
;; arguments.

(defun f-modal-fn (args op)
  (apply #'combine-1 (modal-op-fn op) (modal-op-args op) args))

;; printing is actually interesting in this case.  The cases that are
;; handled specially are the following:
;;  1.  value at root only
;;  2.  unknown at root and one other value
;; In other cases, we print a suitably formatted list of the answers.

(defun fn-print (f stream &aux (x (dag-fn-list f))
			       (bilattice (dag-fn-bilattice f)))
  (cond ((not (cdr x))
	 (format stream "~a" (mvl-print (dag-root-val x) bilattice)))
	((and (mvl-unk (dag-root-val x) bilattice)
	      (null (cddr x)))
	 (format stream #-ANSI "~a after ~a"
		 #+ANSI "~@<~1I~a~:_ after~:_ ~a~:>"
		 (mvl-print (dag-entry-val (second x)) bilattice)
		 (dag-entry-pt (second x))))
	(t #-ANSI (fn-print-1-old x bilattice stream)
	   #+ANSI (fn-print-1 x bilattice stream))))

(defun fn-print-1-old (x b s)
  (format s "~a"
	  (mapcar #'(lambda (x)
		      (list (dag-entry-pt x) '->
			    (mvl-print (dag-entry-val x) b)))
		  x)))

;; pretty printing.  Indentation to indicate what's under what.

(defun fn-print-1 (x b s)
  (format s "~@<~{~<~a~3I~:_ ->~:_ ~a~:>~^~VI~:@_~}~:>"
	  (cdr (fpl-1 (car x) x b 0))))

(defun fpl-1 (pt list b ind)
  (list* ind (list (dag-entry-pt pt) (mvl-print (dag-entry-val pt) b))
	 (mapcan #'(lambda (z) (fpl-1 (find-entry z list) list b (+ 2 ind)))
		 (dag-entry-imm pt))))
