;;; -*- Mode: LISP; Syntax: Common-lisp; Package: MVL; Base: 10 -*-

(in-package "MVL")

(defvar analysis)
(defvar *active-bilattice*)

;; This code has been written to be similar to the code underlying the
;; first-order prover itself, which is contained in first-order.lisp.
;; There, a structure <fotp> was defined that contained high-level
;; information about the proof effort, what was left to do, and so on.
;; Here, a similar structure <analysis> is defined.  In both cases, a
;; global variable (fotp for first-order, but analysis here) is locally
;; bound to the task being considered in order to avoid passing it around
;; from function to function.

;; There are three separate termination conditions that will cause the
;; prover to return.
;;   cutoffs		Cutoff information when you've got nothing left to do
;;   succeeds		Terminate when you know the final answer will
;;			satisfy this
;;   terminate		Stop immediately if the answer satisfies this (even if
;;			it might not after subtasks finish running)

;; There is also a succeed-with-bound flag that indicates, if an answer
;; passes the succeeds test above, whether you should only return it if
;; you've actually got a sensible lower bound on the answer itself.  This
;; flag defaults to t.

;; An <analysis> has four fields corresponding to the termination
;; criteria, and also the following:
;;
;;   prop		The proposition being analyzed
;;   main-task		The top-level proof task
;;   prototasks		Tasks that have not yet been initialized
;;   proofs		A list of all the proof attempts in use
;;   returned-answers	Answers that have already been returned to the
;;   			user and should not be repeated (an element of
;;   			bdg-to-truth-val)
;;   current-value	The current value to be assigned to the analysis,
;;   			including ignoring exceptions if necessary

(defstruct (analysis (:print-function print-analysis))
  prop
  cutoffs succeeds succeed-with-bound terminate
  main-task (prototasks nil) (proofs nil)
  (returned-answers bdg-unknown)
  (current-value bdg-unknown))

;; An analysis is complete if there are no active proofs and no
;; protatasks that might spawn new proof efforts.

(defun analysis-completed (analysis)
  (or (eql (analysis-prototasks analysis) 'completed)
      (and (every #'completed-proof (analysis-proofs analysis))
	   (null (analysis-prototasks analysis)))))

;; The analysis basically proceeds by maintaining a list of things to
;; think about, each a structure of type <consider>.  The first of these
;; is simply the task of type INFERENCE on which the prover was invoked,
;; but the whole thing is made more complex by the need to do
;; consistency checking and by the need to treat modal operators.  In
;; each case, it is important that a partial answer be maintained at all
;; times so that the prover exhibits some sort of anytime behavior.

;; The consistency check is the simpler of the two.  Here, a task of
;; type INFERENCE corresponding to the check is generated and
;; investigated like any other.

;; Modal operators are much more subtle.  Here, the parent task cannot
;; be used to collect the truth values of the clauses appearing under the
;; scope of the modal operator, so a separate task must be created for
;; this purpose.  This task is of type MODAL.  The modal tasks have as
;; children the tasks responsible for computing the truth values of the
;; modal arguments, provided that only one "side" of the truth value is
;; needed.  If both sides are needed (i.e., the argument type is "x"),
;; then a MODAL-CLAUSE subtask is generated that itself has two inference
;; subtasks to compute the two sides of the truth value.

;; The upshot of this is that there are three types of tasks: inference,
;; modal and modal-clause.  The basic description of the task is given in
;; part by the type of the task in the first place, which is one of three
;; possibilities.

;; Finally, tasks need pointers to their parents and we cache the depth
;; of the task in the analysis tree.  We also record whether or not this
;; task is nonmonotonic, in the sense that a t-increase in its truth
;; value will lead to a t-DEcrease in the truth value of the original
;; query.  And, finally, we need a CACHE field, used to record the value
;; that will be taken by the task given values for its children.  The
;; slots in the basic <consider> structure are therefore as follows:

;;   prop		proposition being proved
;;   parent		parent of a given task
;;   children		children of a given task
;;   depth		depth in the task tree
;;   truth-value	accumulated truth value
;;   nonmon		is this task nonmonotonic?
;;   cache		cached truth values

(defvar binding-dag)
(defvar bdg-to-truth-val)
(defvar bdg-true)
(defvar bdg-unknown)

(defstruct (consider (:print-function print-consider))
  prop
  parent (children nil) (depth (if parent (1+ (consider-depth parent)) 0))
  (truth-value bdg-unknown)
  nonmon (cache nil))

;; There are also less basic structures that include auxiliary
;; information that depends upon the type of the task.  Thus an instance
;; of type <inference> includes the fields in the <consider> structure,
;; together with the following:
;;   just		a list of entries of the form (<bdg> . <just>)
;;   			where <just> is a justification in terms of
;;   			other tasks taken from the ATMS lattice
;;   base-value		Truth value that can be obtained by lookup alone
;;   proof		associated call to first-order theorem prover

;; An instance of type <modal> includes <consider> and the following fields:
;;   parity		Does the clause appear positively (T) or
;;			negatively (NIL) in the parent?
;;   op			The modal operator involved

;; An instance of type <modal-clause> includes just <consider>.  The
;; task considering the clause itself is always the first child, and that
;; considering the negation is always the second.

(defstruct (inference (:include consider (nonmon (inference-flip parent prop)))
	    (:constructor make-inference (prop parent &optional truth-value)))
  (just nil) (base-value bdg-unknown)
  proof)

(defstruct (modal (:include consider (nonmon (modal-flip parent parity)))
	    (:constructor make-modal (prop parent parity op)))
  parity op)

(defstruct (modal-clause (:include consider (nonmon (mc-flip parent)))
	    (:constructor make-modal-clause (prop parent))))

(defun modal-clause-pos (task) (first (consider-children task)))
(defun modal-clause-neg (task) (second (consider-children task)))

;; Here we determine if a task is nonmonotonic relative to the root.  We
;; flip the parity if any of the following conditions holds:
;;  1. The task and its parent are both of type inference.
;;  2. If the parent is modal-clause and this is the negative subtask.
;;  3. The parent is modal and clause is negated.
;;  4. The task is modal and its parity is NIL.

;; In the inference case, if the parent is modal-clause, the task will
;; eventually be the negative one if it's the first one created -- in
;; other words, if the parent has no children yet.  If the parent is
;; modal, we use the special variable CLAUSE, which is set up in the call
;; and is the unnegated version of the proposition.

;; The following code exploits the fact that (eql x y) is the same as
;; (if x y (not y)) for x and y both T or NIL.

(defun inference-flip (parent prop)
  (declare (special clause))
  (when parent
    (eql (typecase parent
	   (modal (eql prop clause))
	   (modal-clause (consider-children parent)))
	 (consider-nonmon parent))))

(defun modal-flip (parent parity)
  (eql parity (consider-nonmon parent)))

(defun mc-flip (parent) (consider-nonmon parent))

;; initialize the analysis process.  Sets up an <analysis> to
;; investigate the given proposition and initializes the main task to
;; instantiate most of its fields.  We have to set up the analysis
;; before calling generate-task because generate-task expects analysis
;; to be globally bound when it is called.

(defparameter *succeed-with-bound* t)

(defun init-analysis
    (prop &key (cutoffs std-cutoffs) (succeeds *never-succeeds*)
	       (succeed-with-bound *succeed-with-bound*)
	       (terminate *never-succeeds*) (initial-value unknown)
	       (analysis (make-analysis
			  :prop prop :cutoffs cutoffs :succeeds succeeds 
			  :succeed-with-bound succeed-with-bound
			  :terminate terminate))
     &aux (task (generate-task prop nil)))
  (unless (eq initial-value unknown)
    (set-base-values task
		     (make-root binding-dag *active-bilattice* initial-value)))
  (setf (analysis-main-task analysis) task)
  analysis)

;; cont-analysis returns two values -- the truth value computed and the
;; analysis used.  This allows the routine to be reinvoked if additional
;; answers are needed.  The way it works is as follows:
;;  1.  If there is nothing left to do, get the truth value of the main
;;  task and return that answer and the (completed) analysis.
;;  2.  Otherwise, take one analysis step.  Either return the result, or
;;  simply try again.

;; This function makes use of the following function in relevance.lisp:
;;
;; return-answer ()	what can you return now?

(defun cont-analysis (analysis)
  (do (temp)
      ((analysis-completed analysis)
       (values (analysis-current-value analysis) analysis))
    (consider-step)
    (unless (mvl-eq (setq temp (return-answer))
		    (analysis-returned-answers analysis)
		    bdg-to-truth-val)
      (return (values temp analysis)))))

;; Stuff to generate a task.  Different depending on whether the
;; proposition is modal or not.

(defun generate-task (prop parent)
  (if (eq (car prop) 'modal)
      (generate-modal-task prop parent)
    (generate-nonmodal-task prop parent)))

;; Generate a modal task.  The proposition is of the form (MODAL <prop>),
;; so that you actually want the modal task to be about the proposition
;; (strip-nots prop), and need to record the parity as well.  It is at
;; this point that we generate the children also.  At the end, we
;; recompute the truth value of the given (modal) task.

(defun generate-modal-task (prop parent)
  (declare (special *prop*))
  (multiple-value-bind (p inv) (strip-nots (second prop))
    (or (find-if #'(lambda (x) (and (modal-p x)
				    (eql (modal-parity x) (not inv))
				    (equal (consider-prop x) p)))
		 (consider-children parent))
	(let* ((op (bdg-modal-op (car p)))
	       (task (make-modal p parent (not inv) op)))
	  (push task (consider-children parent))
	  (setf (consider-children task) (modal-task-children task (cdr p) op))
	  (task-diagnostic task "Node for modal proposition" *prop*
			   "being constructed")
	  (recompute-truth-value task)
	  task))))

;; make the children of a modal task.  We work through the clauses,
;; spawning subtasks as appropriate based on the argument type:
;;   0    do nothing; this clause is a parameter
;;   1    either p or -p based on the parity of the modal operator
;;   -1   either -p or p
;;   t    always p
;;   nil  always -p
;;   x    both p and -p, with a modal-clause task to collect the results

(defun modal-task-children (parent clauses op &aux ans task)
  (mapc #'(lambda (clause arg)
	    (declare (special clause))
	    (unless (eql arg 0)
	      (push (case arg
		      (1 (generate-task (if (modal-parity parent)
					    clause (negate clause))
					parent))
		      (-1 (generate-task (if (modal-parity parent)
					     (negate clause) clause)
					 parent))
		      ((t) (generate-task clause parent))
		      ((nil) (generate-task (negate clause) parent))
		      (x (prog1 (setq task (make-modal-clause clause parent))
			   (generate-task (negate clause) task)
			   (generate-task clause task)
			   (setf (consider-truth-value task) 
			     (compute-truth-value task)))))
		    ans)))
	clauses (modal-op-args op))
  (nreverse ans))

;; Generate an inference task.  There's a catch -- if the parent is of
;; type INFERENCE, we are really interested in the negation of the given
;; proposition (since it's for a consistency check), so we lookup
;; (negate prop), negate the result and initialize the prover to
;; actually do the work.  In this case (unlike generate-modal-task),
;; parent might be NIL if the main task is the one being initialized.

;; We also use the lookuped value if the truth value is true.  Other
;; cases appear to generate too much junk to pass around.

(defun generate-nonmodal-task (prop parent)
  (or (and parent
	   (find-if #'(lambda (x)
			(and (inference-p x) (equal (consider-prop x) prop)))
		    (consider-children parent)))
      (let ((task (make-inference prop parent)))
	(when parent
	  (push task (consider-children parent))
	  (if (inference-p parent)
	      (set-base-values task 
			       (mvl-f-ground
				(lookup-with-vars prop *knowledge-test*)
				bdg-to-truth-val))
	    (when (mvl-eq (lookup-with-vars prop *success-test*)
			  bdg-true bdg-to-truth-val)
	      (set-base-values task bdg-true))))
	(push task (analysis-prototasks analysis))
	task)))

(defun set-base-values (task val)
  (setf (inference-base-value task) val
	(inference-truth-value task) val))

;; lookup a proposition, and construct the truth value as an element of
;; bdg-to-truth-val.  In addition, if the thing is a conjunction, get a
;; truth value by conjoining the truth values of the components.

(defun lookup-with-vars (prop &optional (cutoffs *always-succeeds*))
  (if (eq (car prop) 'not)
      (mvl-not (lookup-with-vars (second prop) (cutoffs-not cutoffs))
	       bdg-to-truth-val)
    (let ((temp (construct-bdg-val (lookups prop :cutoffs cutoffs))))
      (if (eq (car prop) 'and)
	  (mvl-plus temp (conj-lookup-with-vars (cdr prop) cutoffs)
		    bdg-to-truth-val)
	temp))))

;; Here's where we look up a conjunction.  One thing: if we bind a variable
;; in an early conjunct, we want to make sure that the binding is passed to
;; subsequent conjuncts.  So bdgs is a list of all the bindings we've
;; accumulated thus far.
	
(defun conj-lookup-with-vars
    (props cutoffs &aux (ans bdg-true) (bdgs (list nil)))
  (dolist (prop props ans)
    (setq ans (mvl-and ans (lookup-conjunct-with-vars prop bdgs cutoffs)
		       bdg-to-truth-val))
    (unless (setq bdgs (ans-bdgs ans)) (return bdg-unknown))))

;; lookup a single conjunct.  bdgs are the bindings from previous conjuncts

(defun lookup-conjunct-with-vars (prop bdgs cutoffs)
  (reduce (bilattice-plus bdg-to-truth-val)
	  (napcar #'(lambda (b) (lcwv-1 prop b cutoffs)) bdgs)))

(defun lcwv-1 (prop bdg cutoffs)
  (include-bdg-in-truth-value
   bdg (lookup-with-vars (plug prop bdg) cutoffs)))

;; What bindings do we have so far?  The point is that we don't want to
;; include anything more specific than something we've already got, since
;; it won't affect future lookups.

(defun ans-bdgs (val)
  (delete-subsumed-entries
   (napcar #'dag-entry-pt
	   (delete-if #'(lambda (x) (t-le (dag-entry-val x) unknown))
		      (copy-list (dag-fn-list val))))
   #'binding-le))

;; construct an element of bdg-to-truth-val from a list of answers.
;; *** THIS FUNCTION IS DESTRUCTIVE ***

(defun construct-bdg-val (answers)
  (dag-accumulate-fn binding-dag *active-bilattice*
		     (napcar #'(lambda (ans) (cons (answer-binding ans)
						   (answer-value ans)))
			     answers)
		     #'mvl-plus))

;; Invoking the prover is made a bit more subtle by the relevance
;; computations.  These computations basically work by taking a list of
;; tasks, and new answers for these tasks, and returning information
;; about what impact the answers will have on the overall problem.  For
;; each answer returned for a proof effort, we want to develop a
;; structure that includes the following information:

;;  affects	the highest task whose truth value is affected.  If the
;;  		root task, the answer changes but it doesn't help.  If NIL,
;;  		the answer changes and it does help.
;;  value	the value taken by the affected task
;;  assumptions	a list of tasks that were assumed to succeed in producing
;;  		this value

(defstruct (relevance (:constructor make-relevance
				    (affects value assumptions)))
  affects value assumptions)

;; The details of taking a single analysis "step" are now as follows:
;;
;; 1.  We initialize any tasks that don't have associated proof efforts.
;; 2.  We pick a proof effort to work on and see if any of the pending
;;     answers are now relevant.
;; 3.  If that didn't produce anything, we reinvoke the prover.
;; 4.  If that didn't do anything either, we return failure.

(defun consider-step (&aux proof)
  (initialize-prototasks)
  (unless (setq proof (choose-proof))
    (setf (analysis-prototasks analysis) 'completed)
    (return-from consider-step))
  (or (consider-step-recheck proof)
      (consider-step-proof proof)))

;; Here we initialize the "prototasks", which are tasks that are not yet
;; associated with proof efforts.  In order to do this, we need to
;; describe the pending-answers field of a proof.  This is an association
;; list where each car is an answer and each cdr is the pending-status
;; of that answer, one of:
;;   NIL  if the answer is potentially relevant
;;   n    if the answer is currently irrelevant because no task
;;        above depth n would be affected by it

;; For each prototask, we first check to see if there is a suitable
;; existing proof effort; this is a proposition that's samep but not an
;; ancestor of the given task (this is how we deal with stratification).
;; If we find such a task:

;; 1.  We push the task and binding onto the list of tasks for that proof
;;     effort.  (Actually, we push it onto the cdr of this list; the first
;;     entry is always the original task.)
;; 2.  We set the task's proof field to the given proof effort.
;; 3.  We inherit the current value of the original inference task.
;; 4.  For each returned answer, we mark it as potentially relevant.

;; If there is no associated proof effort, we create one.
;; recheck-decendants is defined in relevance.lisp and indicates that if
;; a match is found, all of the decendents of the master task are
;; potentially relevant once again.

(defun initialize-prototasks ()
  (dolist (task (analysis-prototasks analysis))
    (multiple-value-bind (proof bdg) (find-legal-master task)
      (cond (proof
	     (push (cons task (car bdg)) (proof-slaves proof))
	     (recheck-descendants (proof-task proof) nil)
	     (setf (inference-proof task) proof)
	     (recompute-truth-value task))
	    (t (setf proof (init-prover (consider-prop task))
		     (inference-proof task) proof)
	       (push proof (analysis-proofs analysis))
	       (setf (proof-task proof) task)))))
  (setf (analysis-prototasks analysis) nil))

(defun find-legal-master (task &aux proof bdg)
  (when (setq proof
	  (find-if #'(lambda (p &aux (master (proof-task p)))
		       (and (eql (consider-nonmon task) 
				 (consider-nonmon master))
			    (setq bdg (samep (proof-prop p)
					     (consider-prop task) t))
			    (not (unstratified task master))))
		   (analysis-proofs analysis)))
    (values proof bdg)))

;; A task is unstratified with respect to a possible master if the
;; master is an ancestor of the given task.  Unfortunately, the depth
;; of the task is not a reliable indicator, since some intermediate
;; task may itself have masters.  So the stratification check fails
;; if any of the following conditions holds:
;;  1.  The task and master are equal.
;;  2.  The stratification check fails for the parent of the task (if there
;;      is one).
;;  3.  The stratification check fails for any slave of the given task.

(defun unstratified (task master)
  (when task
    (or (eql task master)
	(unstratified (consider-parent task) master)
	(and (inference-p task) (inference-proof task)
	     (some #'(lambda (x)
		       (unstratified (consider-parent (car x)) master))
		   (proof-slaves (inference-proof task)))))))

;; A proof is completed if the associated fotp has terminated and every
;; pending answer is currently believed to be irrelevant.

(defun completed-proof (proof)
  (and (null (proof-fotp proof))
       (every #'cdr (proof-pending-answers proof))))

;; Here we try to advance the state of the analysis by calling the
;; prover.  It's pretty easy; we just call the prover until it returns
;; nothing new.  If it returns some stuff, we call process-answer on each
;; answer returned, doing an ongoing disjuction to see if we've gotten
;; anywhere.  Then we return T if something good has happened.

(defun consider-step-proof (proof)
  (do ((list (cont-prover proof) (cont-prover proof)) (flag nil nil))
      ((null list))
    (mapc #'(lambda (ans) (when (process-answer ans proof) (setq flag t)))
	  list)
    (setf (proof-answers proof) nil)
    (when flag (return t))))

;; Here, we process an answer that has been returned.  We push it onto
;; the list of pending answers for the proof.  consider-step-single is
;; responsible for eventually processing the answer.

(defun process-answer (ans proof)
  (consider-step-single (car (push (cons ans nil)
				   (proof-pending-answers proof)))
			proof))

(defun equal-bcans (bc1 bc2)
  (and (equal-answer bc1 bc2)
       (set-equal (bcanswer-just bc1) (bcanswer-just bc2) #'eq-just)))

;; Here we process a single answer, given a proof that might use it.  We
;; check if it's relevant, either adjusting the entry on the pending list
;; or calling prover-succeeded to process it.  At the end, we return T
;; unless there were no places where the answer was relevant.

;; This function uses the following function in relevance.lisp:

;; relevant-answer (ans proof) Takes an answer and proof and returns a
;; list of <relevance>s describing conditions under which the answer is
;; relevant.

;; It might seem here that we should just process everything, but that
;; is wrong.  After all, we don't want to use all the looked up values
;; because they generate too much junk.  Using all the proved values is
;; quite the same.

(defun consider-step-single (ans proof &aux relevances)
  (cond ((and (not (equal-answer (car ans) (make-answer)))
	      (every #'relevance-affects
		     (setq relevances (relevant-answer (car ans) proof))))
	 (setf (cdr ans)
	   (apply #'min
		  (1+ (proof-min-depth proof))
		  (mapcar #'(lambda (r) (consider-depth (relevance-affects r)))
			  relevances)))
	 nil)
	(t (popf (proof-pending-answers proof) ans :count 1)
	   (prover-succeeded (car ans) (proof-task proof))
	   t)))

(defun proof-min-depth (p)
  (apply #'min
	 (consider-depth (proof-task p))
	 (mapcar #'(lambda (x) (consider-depth (car x)))
		 (proof-slaves p))))

;; Here we try to make progress by rechecking the given list of pending
;; answers for a proof.  This is easy; it's just what
;; consider-step-single does.

(defun consider-step-recheck (proof)
  (some #'(lambda (ans) (unless (cdr ans) (consider-step-single ans proof)))
	(proof-pending-answers proof)))

;; What to do if the prover succeeded.  We split the assumptions used
;; into those conjuncts that only returned a truth value (so we don't
;; need to attempt to process their negations to defeat the original
;; proof), those that are modal, and those that spawn consistency checks.

(defun prover-succeeded (bcans task &aux consistency modal (lookup true))
  (dolist (item (bcanswer-just bcans))
    (case (car item)
      (truth-value (setq lookup (mvl-and lookup (cdr item))))
      (modal (pushnew item modal :test #'equal))
      (t (pushnew item consistency :test #'equal))))
  (spawn-new-task task (answer-binding bcans)
		  (recompute-conj modal consistency) lookup))

;; what conjuncts should be passed off for the next proof task?  The
;; only conjuncts that you need to combine are the consistency checks.

(defun recompute-conj (modal consistency)
  (if consistency (cons (conj-to-logic consistency) modal) modal))

;; tasks resulting from successful proof effort when new tasks need to be
;; spawned.  Arguments are:
;;   task that succeeded
;;   binding returned
;;   conjuncts used (does not include truth value due to looked-up terms)
;;   truth value due to procedurally looked-up terms
;;
;; First, for each item that has to be spawned as a subtask, generate
;; the subtask.  Following this, we add a new justification to the
;; justification list of the given task.  This justification includes
;; a dummy task that records any looked-up truth value.

(defun spawn-new-task (task bdg conj lookup)
  (napcar #'(lambda (x) (generate-task (associated-prop x) task)) conj)
  (recompute-just task bdg (if (mvl-eq lookup true) conj
			     (cons (make-dead-inference task lookup) conj)))
  (setf (consider-cache task) nil)
  (recompute-truth-value task))

(defun associated-prop (x)
  (if (eq (car x) 'modal) x (negate x)))

(defun make-dead-inference (task value)
  (make-inference nil task (make-root binding-dag *active-bilattice* value)))

;; recompute the justification of a specific task, given a new
;; justification and a new binding.  First find any existing
;; justification; if so, just add in the new justification (they are
;; both elements of the atms lattice).  If not, stick a new entry onto
;; the justification list.

(defun recompute-just (task bdg just &aux (temp (find-just task bdg)))
  (if temp (setf (cdr temp) (dj-or (list just) (cdr temp)))
    (push (list bdg just) (inference-just task))))

;; given a task and a binding, is there justification information for
;; that binding already on the task's justification list?  Return that
;; element of the justification list if so or NIL if not.

(defun find-just (task bdg)
  (assoc bdg (inference-just task) :test #'equal-binding))

;; remove leading negations from a proposition.  Returns two values --
;; 2nd is t if an odd number of negations were encountered and nil
;; otherwise.

(defun strip-nots (prop)
  (do ((p prop (second p)) (inv nil (not inv)))
      ((not (eq (car p) 'not)) (values p inv))))
