;;; -*- Mode: LISP; Syntax: Common-lisp; Package: MVL; Base: 10 -*-

(in-package "MVL")

;; constructs ATMS bilattice using disjunctive normal form

;; the bilattice is constructed from the ATMS *lattice* of
;; justifications; each justification is a list of contexts (each of
;; which is a conjunctive list of assumptions)

;; In practice, it appears that the number of assumptions being
;; manipulated at any particular time is quite small.  For this reason,
;; no attempt is made to store them in any sorted order; the overhead
;; will not be justified.

;; the maximal justification is true without any assumptions; the
;; minimal justification is never true.

(defparameter dmax-j '(()) "maximal justification")
(defparameter dmin-j nil "minimal justification")

;; lub is easy -- just append the two big disjunctions, and then call
;; djl-simplify to remove any contexts that are supersets of others.
;; There is a check to see if either disjunction is the empty context,
;; in which case we can simply return that.  For glb, we have to conjoin
;; the contexts individually, so we form the list of all contexts that
;; are obtained by conjoining a context in j1 with one in j2.  Then we
;; delete subsumed contexts as before.

(defun dj-or (j1 j2)
  (atms-or j1 j2 #'djl-simplify))

(defun atms-or (j1 j2 fn)
  (cond ((or (eq j1 dmax-j) (eq j2 dmax-j)) dmax-j)
	((eq j1 dmin-j) j2)
	((eq j2 dmin-j) j1)
	(t (funcall fn (append j1 (copy-list j2))))))

(defun dj-and (j1 j2)
  (atms-and j1 j2 #'djl-simplify #'union))

(defun atms-and (j1 j2 simp-fn combine-fn)
  (cond ((or (eq j1 dmin-j) (eq j2 dmin-j)) dmin-j)
	((eq j1 dmax-j) j2)
	((eq j2 dmax-j) j1)
	(t (funcall simp-fn
		    (mapcan #'(lambda (x)
				(mapcar #'(lambda (y)
					    (funcall combine-fn x y)) j1))
			    j2)))))

;; delete subsumed contexts.  We just walk through the contexts; any
;; that is a superset of some other element is deleted.

(defun djl-simplify (l)
  (delete-subsumed-entries l #'(lambda (x y) (subsetp y x))))

;; remove subsumed entries from a list.  The story is that for x on the
;; list, if there is some y on the list such that (test x y), remove x
;; from the list.  Ans is the accumulated answer and list is what you
;; still have to look at.  So pop the top element off of list and
;;  1.  If some element of ans subsumes it, go on to the next.
;;  2.  Otherwise, remove subsumed elements of ans and push the new item
;;  on.
;; Popcell and pushcell are used to reuse the cons cells in the given
;; list.

(defun delete-subsumed-entries (list test)
  (if (cdr list)
      (do (ans item) ((null list) ans)
	(setq item (popcell list))
	(unless (some #'(lambda (x) (funcall test (car item) x)) ans)
	  (setq ans (delete-if #'(lambda (x) (funcall test x (car item))) ans))
	  (pushcell item ans)))
    list))

;; comparing two values -- set equality on the contexts.

(defun dj-eq (j1 j2)
  (set-equal j1 j2 #'dj-same))

;; two contexts match if they are the same sets.

(defun dj-same (x y)
  (set-equal x y #'eq))

;; j1 is less than j2 if every context in j1 is a superset of some
;; context in j2.

(defun dj-le (j1 j2)
  (atms-le j1 j2 #'subsetp))

(defun atms-le (j1 j2 fn)
  (every #'(lambda (c1) (some #'(lambda (c2) (funcall fn c2 c1)) j2)) j1))

;; dot with star -- take every context in j1, and if it is a superset of
;; some context in j2, remove it.

(defun dj-dws (j1 j2)
  (atms-dws j1 j2 #'subsetp))

(defun atms-dws (j1 j2 fn)
  (remove-if #'(lambda (x) (some #'(lambda (y) (funcall fn y x)) j2)) j1))

;; to stash something, call index-with-negation to insert it into the
;; discrimination net, and then construct a context consisting of that
;; and nothing else.

(defun d-stash (p) `((,(index-with-negation p))))

(defparameter *atms-lattice*
	      (make-lattice :max dmax-j
			    :min dmin-j
			    :stash-val #'d-stash
			    :and #'dj-and
			    :or #'dj-or
			    :eq #'dj-eq
			    :leq #'dj-le
			    :dws #'dj-dws
			    :long-desc "ATMS."
			    :short-desc "ATMS"
			    :char #\a))

;; the ATMS bilattice is constructed in the obvious fashion.

(defparameter *atms-bilattice* (lattice-to-bilattice *atms-lattice*))

;; this adds the bilattice to the list of bilattices about which the
;; system knows

(bilattice *atms-bilattice*)
