;;; -*- Mode: LISP; Syntax: Common-lisp; Package: MARKGRAF-KARL; Base: 10 -*-

(IN-PACKAGE "MARKGRAF-KARL" :use '("CL") :nicknames '("MKRP"))


(DEFUN TERM-TERMINATOR (OLDCLAUSES NEWCLAUSES SUPPORTED.CLAUSES &OPTIONAL FASTFLAG)
						; EDITED:  7-JUL-82 12:42:33
						; INPUT: OLDCLAUSES: A LIST OF CLAUSES PREVIOUSLY
						;        EXAMINED BY THE TERMINATOR.
						;  NEWCLAUSES: A LIST OF CLAUSES NOT YET EXAMINED.
						;  SUPPORTED.CLAUSES: THE SET OF SUPPORT.
						;  FASTFLAG: A BOOLEAN VALUE.
						;
						; VALUE: NIL, IF NOTHING WAS FOUND, ELSE A
						;        DOTTED PAIR (RESULT . SOLUTION).
						;  RESULT IS ONE OF THE SYMBOLS 'PROVED OR 'UNITS.
						;  SOLUTION IS A LIST WITH ELEMENTS OF THE FORM:
						;  (SUBSTITUTION CLAUSE LINKLIST POINTERS LITNOS)
						;  REPRESENTING A REFUTATION TREE.
						;  EACH ELEMENT OF SOLUTION REPRESENTS THE INFORMATION
						;  HOW TO RESOLVE AWAY THE LITERALS IN CLAUSE EXCEPT
						;  LITNOS WITH SUBSTITUTION AS MERGED UNIFIER OF
						;  ALL THE LINKS IN LINKLIST (R- AND RIW-LINKS).
						;  POINTERS IS A LIST OF CONS CELLS POINTING TO
						;  LINKS IN ELEMENTS OF THE TAIL OF SOLUTION.
						;  WHEN WORKING OF THE CURRENT SOLUTION ELEMENT,
						;  THESE LINKS HAVE TO BE REPLACED BY THEIR
						;  DESCENDANTS.
						;  IF RESULT = PROVED, SOLUTION ALLOWS TO DEDUCE
						;  THE EMPTY CLAUSE, IF SOLUTION = UNITS IT ALLOWS
						;  TO DEDUCE UNIT CLAUSES.
						;  IF FASTFLAG = NIL, ONLY 0-LEVEL TERMINATOR
						;  SITUATIONS ARE RECOGNIZED.
  (PROG1
    (PROG ((CLAUSES (UNION OLDCLAUSES NEWCLAUSES)))
	  (MAPC
	    #'(LAMBDA (CLAUSE)
		(COND
		  ((OPT-GET.OPTION TERM_SET.OF.SUPPORT)
		   (COND ((MEMBER CLAUSE SUPPORTED.CLAUSES) (DT-PUTPROP CLAUSE 'NEW T))))
		  (T (DT-PUTPROP CLAUSE 'NEW T))))
	    NEWCLAUSES)
	  (COND
	    ((AND (NOT (ZEROP (OPT-GET.OPTION TERM_ITERATIONS))) (NULL (OPT-GET.OPTION TERM_UNITS))
		  (NOT (OPT-GET.OPTION TERM_SET.OF.SUPPORT)))
	     (SETQ CLAUSES (TERM=TOPOLOGICALLY.RELEVANT.CLAUSES CLAUSES)))) 
	  (RETURN
	    (COND
	      ((TERM=INITIALIZATION CLAUSES SUPPORTED.CLAUSES T)
	       (COND (TERM*SOLUTION (CONS 'PROVED TERM*SOLUTION))
		     (T (TERM=ITERATIONS TERM*NOTUNITCLAUSES FASTFLAG)
			(COND (TERM*SOLUTION (CONS 'PROVED TERM*SOLUTION))
			      (T (TERM=LINEARIZE.UNITS) (COND (TERM*SOLUTION (CONS 'UNITS TERM*SOLUTION)))))))))))
    (TERM=CLEAR) (SETQ TERM*SOLUTION NIL)))

(DEFUN TERM-1_TERMINATOR (OLDCLAUSES NEWCLAUSES CILS ITERATIONS)
						; EDITED: 14-APR-83 18:28:49
						; INPUT:  TWO CLAUSELISTS AND AN INTEGER >=0
						;         CILS IS A LIST WITH ELEMENTS
						;         (CLAUSE IGNORED.LITNOS
						;                 (LINK1 . SUBSTITUTION1) (LINK2 . ...
						;         A CLAUSE MAY OCCUR  ONLY ONCE]
						; VALUE:  NIL <=> NOTHING WAS FOUND,
						;         (PROVED . SOLUTION ) <=> A REFUTATION WAS
						;         FOUND (SEE TERM-TERMINATOR)
						;         ELSE
						;         A LIST WITH ELEMENTS
						; (CLAUSE . ((LINK1 . SUBST1)(LINK2 . SUBST2) ...) .
						;           SOLUTION1) ((LINK . SUBST) ...) ...
						;         (LINK . SUBST) ARE EQ TO THE ELEMENTS IN
						;         CILS. SOLUTION IS A LINK CHAIN WHICH
						;         RESOLVES AWAY ALL LITERALS IN CLAUSE EXCEPT
						;         IGNORED.LITNOS, COMPATIBLE TO THE
						;         SUBSTITUTION.
						; REMARK: IF ITERATIONS = 0, NO UNITCLAUSES ARE
						;         INTERNALLY DEDUCED.
						;         IF ITERATIONS > 0, THE CLAUSES ARE EXAMINED
						;         ITERATIONS TIMES AND UNITCLAUSES ARE DEDUCED
						;
						;         NEWCLAUSES ARE TREATED LIKE A SET OF SUPPORT
  (MAPC #'(LAMBDA (CLAUSE) (DT-PUTPROP CLAUSE 'NEW T)) NEWCLAUSES)
  (PROG1
    (PROG ((CLAUSES (UNION OLDCLAUSES NEWCLAUSES)) (SUPPORTED.CLAUSES (MAPCAR #'CAR CILS)))
	  (COND
	    ((ZEROP ITERATIONS)
	     (SETQ CLAUSES
		   (REMOVE-IF-NOT
		     #'(LAMBDA (CLAUSE) (OR (EQL 1 (DS-CLAUSE.NOLIT CLAUSE)) (MEMBER CLAUSE SUPPORTED.CLAUSES)))
		     CLAUSES))))
	  (RETURN
	    (COND
	      ((TERM=INITIALIZATION CLAUSES SUPPORTED.CLAUSES)
	       (COND (TERM*SOLUTION (CONS 'PROVED TERM*SOLUTION))
		     (T (TERM=1_CREATE.SLTU CILS)
			(COND ((TERM=1_ITERATIONS TERM*NOTUNITCLAUSES ITERATIONS) (CONS 'PROVED  TERM*SOLUTION))
			      (T (TERM=1_LINEARIZE CILS)))))))))
    (TERM=CLEAR) (SETQ TERM*SOLUTION NIL)))

(DEFUN TERM-2_TERMINATOR (OLDCLAUSES NEWCLAUSES SUPPORTED.CLAUSES CONDITION ITERATIONS)
						; EDITED: 18-APR-83 11:01:40
						; INPUT:  THREE CLAUSELISTS,
						;         CONDITION IS A TRUTH VALUE FUNCTION WITH
						;         ARGUMENTS (SIGN PREDICATE TERMLIST).
						;         ITERATIONS IS AN INTEGER >= 0.
						; EFFECT: THE TERMINATOR TRIES TO DEDUCE ONE
						;         UNITCLAUSE SATISFYING THE CONDITION DEFINED
						;         BY THE CONDITION ARGUMENT.
						;         (SUPPORTED.CLAUSES ARE JUST THOSE CLAUSES
						;         CONTAINING A LITERAL OF WHICH AN INSTANCE
						;         CAN SATISFY THE CONDITION.)
						;         IF ITERATIONS = 0, ONLY DEDUCTIONS WITH
						;         INITIAL UNITCLAUSES ARE PREFORMED,
						;         ELSE THE CLAUSES ARE EXAMINED ITERATIONS
						;         TIMES AND INTERNAL UNITS ARE GENERATED.
						; VALUE:  NIL <=> NOTHING WAS FOUND
						;         ('PROVED . SOLUTION-CHAIN) <=> A REFUTATION
						;                                       WAS FOUND.
						;         ((CLAUSE LITNO . TERMLIST) . SOLUTION-CHAIN)
						;         <=> A UNIT (AN INSTANCE OF CLAUSE-LITNO'TH
						;         LITERAL WITH TERMLIST AS THE INSTANCE) IF
						;         A UNITCLAUSE SATISFYING THE CONDITION WAS
						;         FOUND.
  (MAPC #'(LAMBDA (CLAUSE) (DT-PUTPROP CLAUSE 'NEW T)) NEWCLAUSES)
  (PROG1
    (PROG ((CLAUSES (UNION OLDCLAUSES NEWCLAUSES)) RESULT)
	  (COND
	    ((ZEROP ITERATIONS)
	     (SETQ CLAUSES
		   (REMOVE-IF-NOT
		     #'(LAMBDA (CLAUSE) (OR (EQL 1 (DS-CLAUSE.NOLIT CLAUSE)) (MEMBER CLAUSE SUPPORTED.CLAUSES)))
		     CLAUSES))))
	  (RETURN
	    (COND
	      ((TERM=INITIALIZATION CLAUSES SUPPORTED.CLAUSES)
	       (COND (TERM*SOLUTION (CONS 'PROVED  TERM*SOLUTION))
		     ((SETQ RESULT (TERM=2_ITERATIONS CLAUSES SUPPORTED.CLAUSES ITERATIONS CONDITION))
		      (COND ((EQL T RESULT) (CONS 'PROVED  TERM*SOLUTION)) (RESULT (CONS RESULT  TERM*SOLUTION)))))))))
    (TERM=CLEAR) (SETQ TERM*SOLUTION NIL)))

(DEFUN TERM-3_TERMINATOR (CLAUSES CLAUSE.LITNOS RENAMING ITERATIONS)
						; EDITED:  6-SEP-83 13:39:26
						; INPUT:   A LIST OF CLAUSES, A LIST OF DOTTED PAIRS
						;          (CLAUSE . LITNO), A RENAMING SUBSTITUTION
						;          AND A NATURAL NUMBER (>=1)
						; VALUE:   NIL OR A REFUTATION CHAIN WHERE
						;          CLAUSE-LITNO ARE REGARDED AS UNITCLAUSES.
						;          SUBSTITUTIONS INTO THE VARIABLES OCCURRING
						;          IN RENAMING ARE FORBIDDEN (EXCEPT THE
						;          RENAMING ITSELF.                           ")
  (MAPC #'(LAMBDA (CLAUSE) (DT-PUTPROP CLAUSE 'NEW T)) CLAUSES)
  (SETQ TERM*3_VARIABLES.REGARDED.AS.CONSTANTS (REMOVE-DUPLICATES RENAMING)) (SETQ TERM*3_RENAMING RENAMING)
  (SETQ TERM*3_PSEUDOCONSTANTS
	(MAPCAR
	  #'(LAMBDA (VARIABLE)
	      (DT-CONSTANT.CREATE (CONCATENATE 'STRING (PRINC-TO-STRING (DT-VARIABLE.PNAME VARIABLE)) "C")
				  (DT-VARIABLE.SORT VARIABLE)))
	  TERM*3_VARIABLES.REGARDED.AS.CONSTANTS))
  (TERM=3_INITIALIZATION CLAUSES CLAUSE.LITNOS) (TERM=3_ITERATIONS CLAUSES ITERATIONS) (TERM=CLEAR)
  (MAPC
    #'(LAMBDA (ELEMENT)
        (PROG
          ((UNIFIER (SUBPAIR TERM*3_VARIABLES.REGARDED.AS.CONSTANTS TERM*3_PSEUDOCONSTANTS (CAR ELEMENT)))
	   (CLAUSE (SECOND ELEMENT)))
          (MAPC
            #'(LAMBDA (LINK)
                (MEMBER-IF
                  #'(LAMBDA (UNI)
                      (COND ((SETQ UNI (UNI-MERGE.SUBSTITUTIONS UNIFIER UNI)) (SETQ UNIFIER (CAR UNI)) T)))
                  (DS-LINK.UNIFIERS LINK)))
            (REMOVE-IF-NOT #'(LAMBDA (LINK) (ASSOC (DS-LINK.OTHERPAR LINK CLAUSE) CLAUSE.LITNOS)) (THIRD ELEMENT)))
          (RPLACA ELEMENT (CAR (UNI-MERGE.SUBSTITUTIONS UNIFIER RENAMING)))))
    TERM*SOLUTION)
  (MAPC #'DT-CONSTANT.DELETE TERM*3_PSEUDOCONSTANTS) (SETQ TERM*3_RENAMING NIL)
  (SETQ TERM*3_PSEUDOCONSTANTS NIL) (SETQ TERM*3_VARIABLES.REGARDED.AS.CONSTANTS NIL)
  (PROG1 TERM*SOLUTION (SETQ TERM*SOLUTION NIL)))


(DEFUN TERM-DUMP (FILE MESSAGE)
						; EDITED AT 15-MAR-83 13:48
						; EDITED:  14. 3. 1983
						; INPUT:   AN ATOM OR NIL AND AN ARBITRARY
						;          S-EXPRESSION
						; EFFECT:  THE MESSAGE AND ALL UNITS ARE PRINTED
						;          ONTO FILE
						; VALUE:   UNDEFINED.
  (COND (FILE (SETQ FILE (MKRP-OPENOUT (MKRP-MAKE.PATHNAME NIL NIL "text" FILE) NIL)) (LINELENGTH 120 FILE)))
  (COND (MESSAGE (PRINC MESSAGE FILE) (TERPRI FILE) (TERPRI FILE)))
  (PROG (T.PREDICATE T.SIGN UNITS TRANSFORMATIONS)
	(MAPC
	  #'(LAMBDA (PREDICATELIST)
	      (PRINC (CONCATENATE 'STRING "PREDICATE: " (PRINC-TO-STRING (DS-PNAME (CAR PREDICATELIST)))) FILE) (TERPRI FILE)
	      (MAPC
		#'(LAMBDA (SIGN)
		    (SETQ UNITS (MI-TERMSTRUCTURE.TERMLISTS (TERM=UNIT.PREDICATELIST.SIGNLIST PREDICATELIST SIGN)))
		    (COND
		      (UNITS (SETQ TRANSFORMATIONS (TERM=UNIT.PREDICATELIST.TRANSFORMATIONS PREDICATELIST SIGN))
			     (COND
			       (TRANSFORMATIONS (PRINC "TRANSFORMATIONS: " FILE) (TERPRI FILE)
						(MAPC
						  #'(LAMBDA (TRANSFORMATION)
						      (MEMBER-IF
							#'(LAMBDA (PLIST)
							    (COND
							      ((EQL (CAR TRANSFORMATION) (TERM=UNIT.PREDICATELIST.SIGNLIST PLIST '+))
							       (SETQ T.PREDICATE (CAR PLIST)) (SETQ T.SIGN '+) T)
							      ((EQL (CAR TRANSFORMATION) (TERM=UNIT.PREDICATELIST.SIGNLIST PLIST '-))
							       (SETQ T.PREDICATE (CAR PLIST)) (SETQ T.SIGN '-) T)))
							TERM*UNITS)
						      (PRINC
							(CONCATENATE 'STRING (PRINC-TO-STRING T.SIGN) " "
								     (PRINC-TO-STRING (DS-PNAME T.PREDICATE)))
							FILE)
						      (PPRINT (CDR TRANSFORMATION) FILE))
						  TRANSFORMATIONS)))
			     (PRINC "LITERALS: " FILE) (TERPRI FILE)
			     (MAPC
			       #'(LAMBDA (UNIT)
				   (PRINC
				     (CONCATENATE 'STRING (PRINC-TO-STRING SIGN)
						  (PRINC-TO-STRING (DS-PNAME (CAR PREDICATELIST))) " ")
				     FILE)
				   (PRINC (DS-PNAME (TERM=UNIT.MAKE.TERMLIST UNIT)) FILE) (TERPRI FILE))
			       UNITS))))
		'(+ -)))
	  TERM*UNITS))
  (COND (FILE (CLOSEFILE FILE))))

(DEFUN TERM=CLEAR NIL
						; EDITED: 30-JUN-82 10:15:19
						; VALUE : UNDEFINED
						; EFFECT : SETS ALL GLOBAL VARIABLES EXCEPT FOR
						;          TERM*SOLUTION NIL.
						;          DELETETS THE PROPERTIES 'TEST 'UNIT 'HINT
						;          'UNINODE 'NEW.
						;          DELETES ALL NEW VARIABLES WHICH DO NOT
						;          OCCUR IN TERM*SOLUTION
  (MAPC
    #'(LAMBDA (SOLUTION.ELEMENT)
        (RPLACA SOLUTION.ELEMENT
		(DT-ABBREVIATION.EXPAND.TERMLIST
		  (CDR (DT-TERM.RENAMED (CAR SOLUTION.ELEMENT) (DS-CLAUSE.VARIABLES (SECOND SOLUTION.ELEMENT)))))))
    TERM*SOLUTION)
  (MAPC #'(LAMBDA (VAR) (DT-VARIABLE.DELETE VAR)) TERM*NEW.VARIABLES)
  (SETQ TERM*NEW.VARIABLES NIL) (SETQ TERM*UNITS NIL)
  (SETQ TERM*NOTUNITCLAUSES NIL)
  (MAPC
    #'(LAMBDA (CLAUSE) (DT-REMPROPS CLAUSE '(TEST UNIT NEW TULISTS SLTU SLTU.IGNORED.LITNOS SLTU.VARIABLES))
	      (DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
		(DS-CLAUSE.LIT.REMPROPS CLAUSE (1+ RPTN)
					'(UNINODE HINT PATTERN DIRECT.SIMPLIFIERS INDIRECT.SIMPLIFIER.UNITS
						  INDIRECT.SIMPLIFIER.CLAUSES GENERATOR))))
    (CG-CLAUSES ALL))
  (DT-ABBREVIATION.POP))

(DEFUN TERM=INITIALIZATION (CLAUSES SUPPORTED.CLAUSES &OPTIONAL FACTORIZE)
						; EDITED: 21-SEP-82 17:45:23
						; INPUT:  TWO LIST OF CLAUSES AND A BOOLEAN VALUE.
						; EFFECT: TERM*SOLUTION AND TERM*NEW.VARIABLES ARE SET
						;         TO NIL.
						;         TERM*NOTUNITCLAUSES IS SET TO A LIST OF
						;         NOTUNITCLAUSES OCCURING IN CLAUSES
						;         TERM*UNITS IS CREATED FOR THE UNITCLAUSES
						;         IN CLAUSES. EVERY UNITCLAUSE GETS A PROPERTY
						;         'UNIT POINTING TO THE UNIT IN TERM*UNITS.
						;         IF THERE IS AN R-LINK BETWEEN TWO
						;         UNITCLAUSES, TERM*SOLUTION IS CALCULATED.
						;         THE MARKS 'TEST AND 'HINT ARE CREATED.
						;         IF FACTORIZE = T, UNINODES FROM SI-LINKS ARE
						;         CREATED.
						; VALUE:  T IF THERE ARE UNITCLAUSES IN CLAUSES,
						;           ELSE NIL.
  (PROG (UNITCLAUSES REFUTATION.LINK) (DT-ABBREVIATION.PUSH) (SETQ TERM*SOLUTION NIL) (SETQ TERM*NEW.VARIABLES NIL)
	(TERM=RECOGNIZE.SIMPLIFIER CLAUSES)
	(COND ((NULL (SETQ UNITCLAUSES (TERM=CREATE.INITIAL.UNITS CLAUSES))) (RETURN NIL))
	      ((SETQ REFUTATION.LINK (TERM=TRIVIAL.PROOF UNITCLAUSES))
	       (SETQ TERM*SOLUTION
		     (LIST
		       (LIST (CAR (DS-LINK.UNIFIERS REFUTATION.LINK)) (DS-LINK.POSPAR REFUTATION.LINK) (LIST REFUTATION.LINK)))))
	      (T (COND (TERM*GENERATOR.LIMIT (TERM=GENERATOR.LITERALS TERM*NOTUNITCLAUSES)))
		 (TERM=CREATE.PATTERNS TERM*NOTUNITCLAUSES) (TERM=CREATE.MARKS UNITCLAUSES)
		 (COND (FACTORIZE (TERM=UNINODES.FROM.SILINKS TERM*NOTUNITCLAUSES)))
		 (MAPC
		   #'(LAMBDA (CLAUSE.LITNO)
		       (PROG ((CLAUSE (CAR CLAUSE.LITNO)) (LITNO (CDR CLAUSE.LITNO)))
			     (SETQ TERM*NOTUNITCLAUSES (DELETE CLAUSE TERM*NOTUNITCLAUSES))
			     (DS-CLAUSE.ALL.LIT.REMPROP CLAUSE 'HINT)
			     (DS-CLAUSE.LIT.PUTPROP CLAUSE LITNO 'TRACE.UNIFIERS
						    (MI-CREATE.EMPTY.TERMSTRUCTURE (DS-CLAUSE.LIT.GETPROP CLAUSE LITNO 'PATTERN)))))
		   TERM*NORMALIZING.LITERALS)
		 (TERM=SORT.NOTUNITCLAUSES SUPPORTED.CLAUSES)))
	(RETURN T)))

(DEFUN TERM=TOPOLOGICALLY.RELEVANT.CLAUSES (CLAUSES)
						; EDITED:  9-MAY-83 09:53:52
						; INPUT:   A LIST OF CLAUSES
						; VALUE:   A SUBSET WITH THOSE CLAUSES WHICH CAN
						;          POTENTIALLY BE RESOLVED AWAY BY UNIT
						;          RESOLUTION.
  (PROG (CONTINUE NON-UNITS UNITS OTHERPAR NEW PURELIT NEXT)
	(MAPC
	  #'(LAMBDA (CLAUSE)
	      (COND
		((EQL 1 (DS-CLAUSE.NOLIT CLAUSE)) (SETQ UNITS (CONS CLAUSE UNITS)) (SETQ NEW (DT-GETPROP CLAUSE 'NEW))
		 (MAPC
		   #'(LAMBDA (LINK) (SETQ OTHERPAR (DS-LINK.OTHERPAR LINK CLAUSE))
			     (COND
			       ((OR NEW (DT-GETPROP OTHERPAR 'NEW))
				(DS-CLAUSE.LIT.PUTPROP OTHERPAR (DS-LINK.OTHERLITNO LINK CLAUSE) 'M1 T))))
		   (DS-CLAUSE.LINKS 'R CLAUSE 1)))
		(T (SETQ NON-UNITS (CONS  CLAUSE NON-UNITS)))))
	  CLAUSES)
	(SETQ CLAUSES NIL) (SETQ CONTINUE UNITS)
	(WHILE CONTINUE (SETQ CONTINUE NIL)
	       (SMAPL
		 #'(LAMBDA (CLAUSE) (SETQ NEXT (CDR CLAUSE)) (SETQ CLAUSE (CAR CLAUSE)) (SETQ PURELIT NIL)
			   (DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
			     (COND
			       ((NOT (DS-CLAUSE.LIT.GETPROP CLAUSE (1+ RPTN) 'M1))
				(COND ((NULL PURELIT) (SETQ PURELIT (1+ RPTN))) ((NEQ T PURELIT) (SETQ PURELIT T))))))
			   (COND
			     ((NULL PURELIT) (SETQ CLAUSES (CONS CLAUSE CLAUSES)) (SETQ NON-UNITS (DELETE CLAUSE NON-UNITS))
			      (DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
				(COND
				  ((NOT (DS-CLAUSE.LIT.GETPROP CLAUSE (1+ RPTN) 'M2)) (DS-CLAUSE.LIT.PUTPROP CLAUSE (1+ RPTN) 'M2 T)
				   (MAPC
				     #'(LAMBDA (LINK)
					 (DS-CLAUSE.LIT.PUTPROP (DS-LINK.OTHERPAR LINK CLAUSE) (DS-LINK.OTHERLITNO LINK CLAUSE) 'M1
								T))
				     (DS-CLAUSE.LINKS 'R CLAUSE (1+ RPTN)))
				   (SETQ CONTINUE T)))))
			     ((AND (NEQ T PURELIT) (NOT (DS-CLAUSE.LIT.GETPROP CLAUSE PURELIT 'M2)))
			      (DS-CLAUSE.LIT.PUTPROP CLAUSE PURELIT 'M2 T)
			      (MAPC
				#'(LAMBDA (LINK)
				    (DS-CLAUSE.LIT.PUTPROP (DS-LINK.OTHERPAR LINK CLAUSE) (DS-LINK.OTHERLITNO LINK CLAUSE) 'M1 T))
				(DS-CLAUSE.LINKS 'R CLAUSE PURELIT))
			      (SETQ CONTINUE T))))
		 #'(LAMBDA (X) NEXT) NON-UNITS))
	(MAPC #'(LAMBDA (CLAUSE) (DS-CLAUSE.ALL.LIT.REMPROPS CLAUSE '(M1 M2))) (CG-CLAUSES ALL)) (RETURN (NCONC UNITS CLAUSES))))

(DEFUN TERM=CREATE.INITIAL.UNITS (CLAUSES)
						; EDITED: 21-SEP-82 17:59:57
						; INPUT:  A LIST OF CLAUSES
						; EFFECT: TERM*NOTUNITCLAUSES IS SET TO A LIST OF
						;         NOTUNITCLAUSES OCCURING IN CLAUSES.
						;         TERM*UNITS IS CREATED FOR THE UNITCLAUSES
						;         IN CLAUSES.
						;         THE PROPERTY 'UNIT POINTING TO THE UNITS
						;         IN TERM*UNITS IS GENERATED FOR EVERY
						;         UNITCLAUSE.
						; VALUE:  THE LIST OF UNITCLAUSES.
  (SETQ TERM*UNITS NIL) (SETQ TERM*NOTUNITCLAUSES NIL)
  (PROG (UNITCLAUSES)
	(MAPC
	  #'(LAMBDA (CLAUSE)
	      (COND ((NEQ 1 (DS-CLAUSE.NOLIT CLAUSE)) (SETQ TERM*NOTUNITCLAUSES (CONS CLAUSE TERM*NOTUNITCLAUSES)))
		    (T (SETQ UNITCLAUSES (CONS  CLAUSE UNITCLAUSES))
		       (PROG
			 ((PREDICATE (TERM=NORMALIZE.PREDICATE (DS-CLAUSE.PREDICATE CLAUSE 1)))
			  (SIGN (TERM=NORMALIZE.SIGN (DS-CLAUSE.SIGN CLAUSE 1)))
			  (ARITY (LIST-LENGTH (DS-CLAUSE.TERMLIST CLAUSE 1))) PREDICATELIST SIGNLIST UNIT)
			 (SETQ PREDICATELIST (ASSOC PREDICATE TERM*UNITS))
			 (COND
			   ((NULL PREDICATELIST) (SETQ PREDICATELIST (TERM=UNIT.ADD.NEW.PREDICATELIST PREDICATE ARITY))))
			 (SETQ SIGNLIST (TERM=UNIT.PREDICATELIST.SIGNLIST PREDICATELIST SIGN))
			 (SETQ UNIT
			       (MI-INSERT.TERMLIST.INTO.TERMSTRUCTURE SIGNLIST (DS-CLAUSE.TERMLIST CLAUSE 1)
								      (TERM=UNIT.CREATE.PARTIAL NIL CLAUSE NIL) NIL NIL))
			 (DT-PUTPROP CLAUSE 'UNIT UNIT) (TERM=INDIRECT.SIMPLIFIER.UNIT UNIT CLAUSE 1 NIL)))))
	  CLAUSES)
	(RETURN UNITCLAUSES)))

(DEFUN TERM=TRIVIAL.PROOF (UNITCLAUSES)
						; EDITED: 30-JUN-82 14:48:54
						; INPUT : A LIST OF UNITCLAUSES
						; VALUE : THE LINK(IF SUCH ONE EXISTS),WHOSE RESOLENT
						;         CREATES THE EMPTY CLAUSE,ELSE NIL
  (PROG (LINK)
	(MEMBER-IF
	  #'(LAMBDA (UNITCLAUSE)
	      (MEMBER-IF
		#'(LAMBDA (RLINK)
		    (COND ((EQL 1 (DS-CLAUSE.NOLIT (DS-LINK.OTHERPAR RLINK UNITCLAUSE))) (SETQ LINK RLINK))))
		(DS-CLAUSE.LINKS 'R UNITCLAUSE 1)))
	  UNITCLAUSES)
	(RETURN LINK)))

(DEFUN TERM=CREATE.MARKS (UNITCLAUSES)
						; EDITED: 30-JUN-82 10:56:19
						; INPUT : A LIST OF UNITCLAUSES
						; VALUE : UNDEFINED
						; EFFECT : MARKS ALL LITERALS CONNECTED TO A
						;          UNITCLAUSE WITH A PROPERTY 'HINT,CLAUSES
						;          WITH SUCH LITERALS WITH A PROPERTY 'TEST
  (MAPC
    #'(LAMBDA (CLAUSE)
        (MAPC
          #'(LAMBDA (RLINK)
              (PROG ((OTHERPAR (DS-LINK.OTHERPAR RLINK CLAUSE)))
		    (COND ((OR (DT-GETPROP CLAUSE 'NEW) (DT-GETPROP OTHERPAR 'NEW)) (DT-PUTPROP OTHERPAR 'TEST T)))
		    (TERM=CLAUSE.LIT.ADDPROP OTHERPAR (DS-LINK.OTHERLITNO RLINK CLAUSE) 'HINT
					     (TERM=HINT.CREATE (LIST (DT-GETPROP CLAUSE 'UNIT)) RLINK T))))
          (DS-CLAUSE.ALL.LINKS 'R CLAUSE))) 
    UNITCLAUSES))

(DEFUN TERM=SORT.NOTUNITCLAUSES (SUPPORTED.CLAUSES)
					; INPUT:  A LIST OF CLAUSES WHICH ARE SUPPORTED.
					; EFFECT: THE NOTUNITCLAUSES WHICH ARE NOT SUPPORTED
					;         AND THE NOTUNITCLAUSES WHICH ARE SUPPORTED
					;         ARE SORTED ACCORDING TO THE SORT FUNCTION
					;         TERM=SORT AND ARE JOINED LIKE A ZIPPER.
					; VALUE:  UNDEFINED.
  (SETQ SUPPORTED.CLAUSES (INTERSECTION TERM*NOTUNITCLAUSES SUPPORTED.CLAUSES))
  (COND
   ((OR (NULL SUPPORTED.CLAUSES) (EQL (LIST-LENGTH SUPPORTED.CLAUSES) (LIST-LENGTH TERM*NOTUNITCLAUSES)))
    (SETQ TERM*NOTUNITCLAUSES (SORT TERM*NOTUNITCLAUSES #'TERM=SORT)))
   (T (SETQ TERM*NOTUNITCLAUSES (NSET-DIFFERENCE TERM*NOTUNITCLAUSES SUPPORTED.CLAUSES)
	    TERM*NOTUNITCLAUSES (SORT TERM*NOTUNITCLAUSES #'TERM=SORT)
	    SUPPORTED.CLAUSES (SORT SUPPORTED.CLAUSES #'TERM=SORT))
      (PROG (NEWLIST TAIL)
	    (COND
	     ((> (LIST-LENGTH SUPPORTED.CLAUSES) (LIST-LENGTH TERM*NOTUNITCLAUSES))
	      (SETQ NEWLIST
		    (LASTN SUPPORTED.CLAUSES (- (LIST-LENGTH SUPPORTED.CLAUSES) (LIST-LENGTH TERM*NOTUNITCLAUSES)))
		    SUPPORTED.CLAUSES (CAR NEWLIST)
		    TAIL (CDR NEWLIST)
		    NEWLIST NIL)))
	    (RPLACD (LAST SUPPORTED.CLAUSES) SUPPORTED.CLAUSES)
	    (MAPC #'(LAMBDA (C1 C2) (SETQ NEWLIST (NCONC NEWLIST (LIST C1 C2))))
		  SUPPORTED.CLAUSES TERM*NOTUNITCLAUSES)
	    (SETQ TERM*NOTUNITCLAUSES (NCONC NEWLIST TAIL))))))
 
(DEFUN TERM=SORT (CLAUSE1 CLAUSE2)
						; INPUT:  TWO CLAUSES
						; VALUE:  T IF CLAUSE1 HAS FEWER LITERALS RESP. MORE
						;           VARIABLES AS CLAUSE2,
						;         ELSE NIL.
  (COND ((> (DS-CLAUSE.NOLIT CLAUSE2) (DS-CLAUSE.NOLIT CLAUSE1)) T)
	((> (DS-CLAUSE.NOLIT CLAUSE1) (DS-CLAUSE.NOLIT CLAUSE2)) NIL)
	((> (LIST-LENGTH (DS-CLAUSE.VARIABLES CLAUSE1)) (LIST-LENGTH (DS-CLAUSE.VARIABLES CLAUSE2))) NIL) (T T)))

(DEFUN TERM=CREATE.PATTERNS (CLAUSES)		; edited: 24-feb-83 16:39:43
						; input:  a list of clauses
						; effect: the patterns necessary for the MI-module
						;         are generated and stored as literal property 'pattern.
						; value: undefined
  (MAPC #'(LAMBDA (CLAUSE)
	    (LET ((NOLIT (DS-CLAUSE.NOLIT CLAUSE)) (VARIABLES (DS-CLAUSE.VARIABLES CLAUSE)))
	      (DODOWN (RPTN NOLIT)
		(LET (PATTERN (VARS (DS-CLAUSE.LIT.VARIABLES CLAUSE (1+ RPTN))))
		  (MAPC #'(LAMBDA (VARIABLE) (SETQ PATTERN (CONS (IF (MEMBER VARIABLE VARS) T NIL) PATTERN)))
			VARIABLES)
		  (DS-CLAUSE.LIT.PUTPROP CLAUSE (1+ RPTN) 'PATTERN (NREVERSE PATTERN))))))
	CLAUSES))

(DEFUN TERM=UNINODES.FROM.SILINKS (CLAUSES)
						; EDITED: 15. 9. 1984
						; INPUT:  A LIST OF CLAUSES.
						; EFFECT: ALL UNINODES FROM THE SI-LINKS ARE CREATED.
						;         A PSEUDOUNIT IS CONNECTED TO POSLITNO.
						;         THE UNIT-CELL OF SUCH A UNINODE IS FILLED
						;         WITH A DOTTED PAIR (POSLITNO . NEGLITNO).
						; VALUE:  UNDEFINED.
  (PROG (VARIABLES NEW POSLITNO NEGLITNO)
	(MAPC
	  #'(LAMBDA (CLAUSE) (SETQ VARIABLES (DS-CLAUSE.VARIABLES CLAUSE)) (SETQ NEW (DT-GETPROP CLAUSE 'NEW))
		    (DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
		      (MAPC
			#'(LAMBDA (LINK)
			    (COND
			      ((EQL (1+ RPTN) (SETQ POSLITNO (DS-LINK.POSLITNO LINK))) (SETQ NEGLITNO (DS-LINK.NEGLITNO LINK))
			       (MAPC
				 #'(LAMBDA (UNIFIER) (SETQ UNIFIER (TERM=UNIFIER.NORMALIZE UNIFIER VARIABLES))
					   (DS-CLAUSE.LIT.PUTPROP CLAUSE (1+ RPTN) 'UNINODE
								  (QCONC1 (DS-CLAUSE.LIT.GETPROP CLAUSE (1+ RPTN) 'UNINODE)
									  (TERM=UNINODE.CREATE UNIFIER LINK (CONS POSLITNO NEGLITNO)
											       NEW))))
				 (DS-LINK.UNIFIERS LINK)))))
			(DS-CLAUSE.LINKS 'SI CLAUSE (1+ RPTN)))))
	  CLAUSES)))

(DEFUN TERM=ITERATIONS (CLAUSES &OPTIONAL FASTFLAG)
						; EDITED: 30-JUN-82 10:59:28
						; INPUT : A LIST OF CLAUSES AND A BOOLEAN VALUE.
						; VALUE : UNDEFINED
						; EFFECT : THIS IS THE TOP LEVEL LOOP FOR THE SEARCH
						;          FOR TERMINATOR SITUATIONS RESP. FOR
						;          DEDUCING NEW UNITCLAUSES.
  (PROG
    (READY (COUNTER 0)
     (ITERATIONS
       (COND (FASTFLAG 1) ((ZEROP (OPT-GET.OPTION TERM_ITERATIONS)) 1) (T (OPT-GET.OPTION TERM_ITERATIONS)))))
    (DECLARE (SPECIAL COUNTER))
    (SETQ FASTFLAG (OR FASTFLAG (ZEROP (OPT-GET.OPTION TERM_ITERATIONS))))
    (WHILE (AND (< COUNTER ITERATIONS) (NULL READY))
      (MEMBER-IF
	#'(LAMBDA (CLAUSE)
	    (COND ((DT-GETPROP CLAUSE 'TEST) (SETQ READY (TERM=EXAMINE.CLAUSE CLAUSE FASTFLAG)))))
	CLAUSES)
      (SETQ COUNTER (1+ COUNTER))
      (COND
	((OPT-GET.OPTION TERM_BREADTH.FIRST)
	 (MAPC
	   #'(LAMBDA (CLAUSE)
	       (DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
		 (MAPC #'(LAMBDA (HINT) (TERM=HINT.ACTIVATE HINT)) (DS-CLAUSE.LIT.GETPROP CLAUSE (1+ RPTN) 'HINT))))
	   CLAUSES))))))

(DEFUN TERM=CREATE.UNITS (CLAUSE VARIABLES LITNO TRACE.UNIFIERS &OPTIONAL CONDITION FASTMODE)
						; EDITED: 22-SEP-82 18:08:44
						; INPUT:  A CLAUSE, VARIABLES REPRESENTING THE OTHER
						;         SIDE OF THE T-UNIFIERS, A LITERAL NUMBER,
						;         A DATA STRUCTURE TRACE.UNIFIERS,
						;         A FUNCTION (SEE TERM-2_TERMINATOR) AND A
						;         BOOLEAN VALUE.
						; EFFECT: EVERY TRACE.UNIFIER IS APPLIED TO THE
						;         LITNO'S LITERAL IN CLAUSE AND A NEW UNIT
						;         IS CREATED, IF IT IS NOT SUBSUMED BY AN
						;         ALREADY EXISTING ONE, IT HAS AT LEAST ONE
						;         NEW UNINODE, IT DOES NOT EXCEED THE
						;         TERM DEPTH BOUND, IT IS NO TAUTOLOGY AND
						;         NOT TO MANY SKOLEM FUNCTIONS ARE NESTED.
						;         IF FASTMODE =//= NIL
						;         FOR EVERY NEW UNIT, THE NEW HINTS ARE
						;         GENERATED.
						; VALUE:  IF CONDITION =//= NIL AND A UNIT IS CREATED
						;         SATISFYING THE CONDITION THEN
						;         ((CLAUSE LITNO . TERMLIST) . UNIT)
						;         ELSE NIL.
  (COND ((NULL TRACE.UNIFIERS) NIL)
	(T
	 (PROG
	   ((SIGN (TERM=NORMALIZE.SIGN (DS-CLAUSE.SIGN CLAUSE LITNO)))
	    (PREDICATE (TERM=NORMALIZE.PREDICATE (DS-CLAUSE.PREDICATE CLAUSE LITNO)))
	    (TERMLIST (DS-CLAUSE.TERMLIST CLAUSE LITNO))
	    (TERM.DEPTH (COND ((OPT-GET.OPTION STR_TERM.DEPTH) (1+ (OPT-GET.OPTION STR_TERM.DEPTH))))) (UNITS (LIST NIL))
	    PREDICATELIST SIGNLIST UNIT TRANSFORMATIONS)
	   (SETQ PREDICATELIST (ASSOC PREDICATE TERM*UNITS))
	   (COND
	     ((NULL PREDICATELIST)
	      (SETQ PREDICATELIST (TERM=UNIT.ADD.NEW.PREDICATELIST PREDICATE (LIST-LENGTH TERMLIST)))))
	   (SETQ SIGNLIST (TERM=UNIT.PREDICATELIST.SIGNLIST PREDICATELIST SIGN))
	   (SETQ TRANSFORMATIONS (TERM=UNIT.PREDICATELIST.TRANSFORMATIONS PREDICATELIST SIGN))
	   (MAPC
	     #'(LAMBDA (TRACE.UNIFIER)
		 (PROG
		   ((TUNIFIER (MI-TERMLIST.STANDARD.FULL TRACE.UNIFIER)) NEW.TERMLIST NODES NESTING.COUNTER)
		   (SETQ NEW.TERMLIST (TERM=APPLY.UNIFIER TUNIFIER TERMLIST VARIABLES))
		   (COND
		     ((AND
			(MEMBER-IF #'(LAMBDA (X) (TERM=UNINODE.IS.NEW X))
				   (SETQ NODES (TERM=TRACE.UNIFIER.UNINODES TRACE.UNIFIER)))
			(COND (TERM.DEPTH (NOT (> (DT-TERMLIST.MAXDEPTH NEW.TERMLIST) TERM.DEPTH))) (T T))
			(NOT
			  (AND TERM*GENERATOR.LIMIT (DS-CLAUSE.LIT.GETPROP CLAUSE LITNO 'GENERATOR)
			       (ZEROP (SETQ NESTING.COUNTER (TERM=NESTING.COUNTER NODES)))))
			(NOT (TERM=IS.TAUTOLOGY SIGN PREDICATE NEW.TERMLIST)))
		      (SETQ NEW.TERMLIST (CAR (DT-ABBREVIATION.COMPRESS.TERMLIST NEW.TERMLIST)))
		      (SETQ UNIT
			    (TERM=SIMPLIFY.UNIT
			      (MI-INSERT.TERMLIST.INTO.TERMSTRUCTURE SIGNLIST NEW.TERMLIST
								     (TERM=UNIT.CREATE.PARTIAL TUNIFIER CLAUSE NODES)
								     TRANSFORMATIONS T)
			      CLAUSE LITNO))
		      (COND
			(UNIT
			 (COND
			   ((AND CONDITION
				 (FUNCALL CONDITION SIGN PREDICATE
					  (DT-ABBREVIATION.EXPAND.TERMLIST (COPY-TREE NEW.TERMLIST))))
                            (RETURN-FROM TERM=CREATE.UNITS
			      (CONS (CONS CLAUSE (CONS LITNO (DT-ABBREVIATION.EXPAND.TERMLIST (COPY-TREE NEW.TERMLIST))))
				    UNIT))))
			 (TERM=UNIT.SET.COUNTER UNIT NESTING.COUNTER)
			 (COND ((TERM=NORMALIZE.UNIT UNIT NEW.TERMLIST CLAUSE LITNO)) (T (QCONC1 UNITS UNIT)))))))))
	     (MI-TERMSTRUCTURE.TERMLISTS TRACE.UNIFIERS))
	   (COND
	     ((AND (CAR UNITS) (OR (NULL CONDITION) (NOT FASTMODE))) (TERM=CREATE.NEW.HINTS CLAUSE LITNO (CAR UNITS))))))))

(DEFUN TERM=CREATE.NEW.HINTS (CLAUSE LITNO UNITS)
						; EDITED: 30-JUN-82 10:33:07
						; INPUT : CLAUSE, LITERALNUMBER, A LIST OF UNITS.
						; VALUE : UNDEFINED
						; EFFECT : MARKS ALL LITERALS CONNECTED WITH THE
						;          LITERAL (CLAUSE LITNO) WITH A PROPERTY
						;          'HINT,CLAUSES WITH SUCH LITERALS WITH
						;          'TEST
  (SETQ UNITS (TERM=NO.NORMALIZING.UNITS CLAUSE LITNO UNITS))
  (MAPC
    #'(LAMBDA (RLINK)
        (PROG ((OTHERPAR (DS-LINK.OTHERPAR RLINK CLAUSE)))
	      (COND
		((NEQ 1 (DS-CLAUSE.NOLIT OTHERPAR)) (DT-PUTPROP OTHERPAR 'TEST T)
		 (TERM=CLAUSE.LIT.ADDPROP OTHERPAR (DS-LINK.OTHERLITNO RLINK CLAUSE) 'HINT
					  (TERM=HINT.CREATE UNITS RLINK))))))
    (DS-CLAUSE.LINKS 'R CLAUSE LITNO))
  (MAPC
    #'(LAMBDA (LINK)
        (PROG
          ((OTHERLITNO
             (COND ((EQL LITNO (DS-LINK.POSLITNO LINK)) (DS-LINK.NEGLITNO LINK)) (T (DS-LINK.POSLITNO LINK)))))
          (DT-PUTPROP CLAUSE 'TEST T)
          (TERM=CLAUSE.LIT.ADDPROP CLAUSE OTHERLITNO 'HINT (TERM=HINT.CREATE UNITS LINK))))
    (DS-CLAUSE.LINKS 'RIW CLAUSE LITNO)))

(DEFUN TERM=SET.UNINODE (LITNO CLAUSE HINT VARIABLES)
						; EDITED: 30-JUN-82 10:05:36
						; INPUT : LITERALNUMBER CLAUSE HINT
						; VALUE : T <=> A NEW UNINODE WAS CREATED
						; EFFECT : CREATES(IF POSSIBLE) FROM THE INPUT HINT
						;          THE CORRESPONDING UNINODES AND JOINS THEM
						;          TO THE LITNO'TH LITERAL OF CLAUSE
  (PROG
    ((LINK (TERM=HINT.LINK HINT)) (UNITS (TERM=HINT.UNITS HINT)) (UNINODES (LIST NIL)) (RENAMING (DS-CLAUSE.RENAMING CLAUSE))
     PREDICATE VARS TUNIFIER COLOUR RENAMED)
    (SETQ PREDICATE (DS-CLAUSE.PREDICATE CLAUSE LITNO)) (SETQ RENAMED (UNI-UNIFIER.CODOMAIN RENAMING))
    (SETQ COLOUR (DS-LINK.COLOUR LINK)) (SETQ VARS (DS-CLAUSE.VARIABLES (DS-LINK.OTHERPAR LINK CLAUSE)))
    (MAPC
      #'(LAMBDA (UNIT)
          (COND
            ((TERM=UNIT.NODES UNIT)
						; INTERNAL UNITS
	     (SETQ TUNIFIER (TERM=UNIT.UNIFIER UNIT))
	     (TERM=BIND TUNIFIER VARS)
	     (CASE COLOUR
	       (R
		 (MAPC #' (LAMBDA (UNIFIER)
			    (MAPC #'(LAMBDA (UNIFIER)
				      (QCONC1 UNINODES
					      (TERM=UNINODE.CREATE (TERM=UNIFIER.NORMALIZE UNIFIER VARIABLES) LINK UNIT T)))
				  (UNI-UNIFY.MIXED.TERMLIST UNIFIER)))
		       (DS-LINK.UNIFIERS LINK)))
	       (RIW
		 (COND
		   ((EQL LITNO (DS-LINK.POSLITNO LINK))
		    (MAPC
		      #'(LAMBDA (UNIFIER)
			  (MAPC
			    #'(LAMBDA (UNIFIER)
				(QCONC1 UNINODES
					(TERM=UNINODE.CREATE (TERM=UNIFIER.NORMALIZE UNIFIER RENAMED) LINK UNIT T)))
			    (UNI-UNIFY.MIXED.TERMLIST UNIFIER)))
		      (DS-LINK.UNIFIERS LINK)))
		   (T
		    (MAPC
		      #'(LAMBDA (UNIFIER)
			  (MAPC
			    #'(LAMBDA (UNIFIER)
				(QCONC1 UNINODES
					(TERM=UNINODE.CREATE (TERM=UNIFIER.NORMALIZE UNIFIER RENAMED) LINK UNIT T)))
			    (UNI-UNIFY.MIXED.TERMLIST UNIFIER)))
		      (UNI-SWITCH RENAMING (COPY-TREE (DS-LINK.UNIFIERS LINK)))))))
	       (OTHERWISE (ERROR "ILLEGAL LINK COLOUR IN TERM=SET.UNINODE: ~A" COLOUR)))
	     (TERM=UNBIND VARS))
            (TERM*3_PSEUDOCONSTANTS
	     (MAPC
	       #'(LAMBDA (UNIFIER)
		   (QCONC1 UNINODES
			   (TERM=UNINODE.CREATE (TERM=UNIFIER.NORMALIZE UNIFIER VARIABLES) LINK UNIT T)))
	       (UNI-UNIFY.ATOMS PREDICATE
				(SUBPAIR TERM*3_PSEUDOCONSTANTS TERM*3_VARIABLES.REGARDED.AS.CONSTANTS
					 (UNI-APPLY.SUBSTITUTION TERM*3_RENAMING
								 (DS-CLAUSE.TERMLIST (DS-LINK.POSPAR LINK)
										     (DS-LINK.POSLITNO LINK))
								 T))
				PREDICATE
				(SUBPAIR TERM*3_PSEUDOCONSTANTS TERM*3_VARIABLES.REGARDED.AS.CONSTANTS
					 (UNI-APPLY.SUBSTITUTION TERM*3_RENAMING
								 (DS-CLAUSE.TERMLIST (DS-LINK.NEGPAR LINK) (DS-LINK.NEGLITNO LINK))
								 T)))))
            (T					; EXTERNAL UNITS
	     (MAPC
	       #'(LAMBDA (UNIFIER)
		   (QCONC1 UNINODES
			   (TERM=UNINODE.CREATE (TERM=UNIFIER.NORMALIZE UNIFIER VARIABLES) LINK UNIT
						(OR (DT-GETPROP (DS-LINK.POSPAR LINK) 'NEW)
						    (DT-GETPROP (DS-LINK.NEGPAR LINK) 'NEW)))))
	       (DS-LINK.UNIFIERS LINK)))))
      UNITS)
    (COND
      ((CAR UNINODES)
       (DS-CLAUSE.LIT.PUTPROP CLAUSE LITNO 'UNINODE
			      (QCONC (DS-CLAUSE.LIT.GETPROP CLAUSE LITNO 'UNINODE) (CAR UNINODES)))
       (RETURN T)))))

(DEFUN TERM=EXAMINE.CLAUSE (CLAUSE &OPTIONAL FASTMODE CONDITION)
						; EDITED:  7-APR-83 15:37:26
						; INPUT:   A CLAUSE, A FLAG AND A FUNCTION
						;          (SEE TERM-2_TERMINATOR).
						; VALUE:   T <=> A REFUTATION WAS FOUND.
						;          IF CONDITION =//= NIL AND A UNIT CAN BE
						;          DEDUCED SATISFYING THE CONDITION THEN
						;          ((CLAUSE LITNO . TERMLIST) . UNIT) ELSE NIL
						; EFFECT:  IF NO REFUTATION CAN BE FOUND, BUT
						;          FASTMODE IS NOT NIL, ALL UNITCLAUSES WHICH
						;          CAN BE DEDUCED FROM THIS CLAUSE AND THE
						;          EXISTING UNITS ARE GENERATED.
  (CATCH 'TERM=EXAMINE.CLAUSE
    (PROG1
      (PROG ((NEWLITS (TERM=PREPARE.CLAUSE CLAUSE (OPT-GET.OPTION TERM_BREADTH.FIRST)))
	     PURELITS UNIT)
	    (SETQ PURELITS (CAR NEWLITS)) (SETQ NEWLITS (CDR NEWLITS))
	    (COND ((OR (CDR PURELITS) (NULL NEWLITS)))
		  (PURELITS
		   (COND
		     ((OR CONDITION (NOT FASTMODE)) (TERM=CREATE.INITIAL.TULISTS CLAUSE)
		      (RETURN (TERM=DEDUCE.UNITS CLAUSE (CAR PURELITS) NEWLITS CONDITION FASTMODE)))))
		  ((PROG2 (TERM=CREATE.INITIAL.TULISTS CLAUSE) (TERM=COMPATIBILITY.TEST.FULL CLAUSE NEWLITS)) (RETURN T))
		  ((OR CONDITION (NOT FASTMODE))
		   (DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
		     (PROGN
		       (COND
			 ((SETQ UNIT (TERM=DEDUCE.UNITS CLAUSE (1+ RPTN) NEWLITS CONDITION FASTMODE))
			  (RETURN-FROM TERM=EXAMINE.CLAUSE UNIT)))
		       (TERM=CUT.TULISTS CLAUSE (1+ RPTN)))))))
      (DT-REMPROP CLAUSE 'TULISTS)
      (DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
	(MAPC #'(LAMBDA (NODE) (TERM=UNINODE.MARK.OLD NODE))
	      (CAR (DS-CLAUSE.LIT.GETPROP CLAUSE (1+ RPTN) 'UNINODE)))))))

(DEFUN TERM=DEDUCE.UNITS (CLAUSE LITNO NEWLITS &OPTIONAL CONDITION FASTMODE)
						; EDITED:  7-APR-83 14:33:12
						; INPUT:   A CLAUSE, A LITERAL NUMBER, A LIST OF
						;          LITERALS WITH NEW UNINODES,
						;          A FUNCTION (SEE TERM-2_TERMINATOR)
						;          AND A BOOLEAN VALUE.
						;          NO LITERALS OF CLAUSE EXCEPT LITNO ARE
						;          PURE, THE TULISTS ARE CREATED.
						; EFFECT:  THE UNITCLAUSES INHERITED FROM CLAUSE-LITNO
						;          ARE GENERATED.
						; VALUE:   IF CONDITION =//= NIL AND A UNIT CAN BE
						;          DEDUCED SATISFYING THE CONDITION THEN
						;          ((CLAUSE LITNO . TERMLIST) . UNIT) ELSE NIL
  (COND
    ((AND (OR (CDR NEWLITS) (NEQ LITNO (CAR NEWLITS)))
	  (OR (OPT-GET.OPTION TERM_UNITS) (DS-CLAUSE.LINKS 'RIW CLAUSE LITNO)
	      (MEMBER-IF #'(LAMBDA (LINK) (NEQ 1 (DS-CLAUSE.NOLIT (DS-LINK.OTHERPAR LINK CLAUSE))))
			 (DS-CLAUSE.LINKS 'R CLAUSE LITNO))))
     (TERM=CREATE.UNITS CLAUSE (DS-CLAUSE.VARIABLES CLAUSE) LITNO
			(TERM=TULIST.TRACE.UNIFIERS
			  (TERM=MERGE.LIST.OF.TULISTS CLAUSE
						      (TERM=GET.TULISTS (DT-GETPROP CLAUSE 'TULISTS) (DS-CLAUSE.NOLIT CLAUSE)
									(LIST LITNO)) NEWLITS))
			CONDITION FASTMODE))))

(DEFUN TERM=COMPATIBILITY.TEST.FULL (CLAUSE NEWLITS)
						; EDITED:  7-APR-83 14:17:57
						; INPUT:   A CLAUSE AND A LIST OF LITERALS WITH
						;          NEW UNINODES.
						;          NO LITERAL OF CLAUSE IS 'PURE' AND THE
						;          TULISTS ARE CREATED.
						; VALUE:   T IF AT LEAST ONE MERGED UNIFIER HAS BEEN
						;          FOUND.
						; EFFECT:  IF THE MERGING HAS BEEN SUCCESSFULL, THE
						;          SOLUTION CHAIN IS GENERATED.
  (PROG
    ((TRACE.UNIFIERS
       (TERM=TULIST.TRACE.UNIFIERS
         (TERM=MERGE.LIST.OF.TULISTS CLAUSE
				     (TERM=GET.TULISTS (DT-GETPROP CLAUSE 'TULISTS) (DS-CLAUSE.NOLIT CLAUSE) NIL) NEWLITS)))
     TRACE.UNIFIER)
    (COND
      (TRACE.UNIFIERS (SETQ TRACE.UNIFIER (CAR (MI-TERMSTRUCTURE.TERMLISTS TRACE.UNIFIERS)))
		      (TERM=LINEARIZATION
			(TERM=CREATE.PSEUDOUNIT (MI-TERMLIST.STANDARD.FULL TRACE.UNIFIER) CLAUSE
						(TERM=TRACE.UNIFIER.UNINODES TRACE.UNIFIER)))
		      (RETURN T)))))

(DEFUN TERM=CREATE.INITIAL.TULISTS (CLAUSE)
						; EDITED:  7-APR-83 12:13:41
						; INPUT:   A CLAUSE
						; EFFECT:  THE UNINODES ARE TRANSFORMED INTO TULISTS
						; VALUE:   UNDEFINED
  (PROG (TULISTS NODES)
	(DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
	  (COND
	    ((SETQ NODES (CAR (DS-CLAUSE.LIT.GETPROP CLAUSE (1+ RPTN) 'UNINODE)))
	     (SETQ TULISTS
		   (CONS
		     (TERM=TULIST.CREATE (LIST (1+ RPTN))
					 (TERM=CREATE.INITIAL.TRUNIFIERS NODES (DS-CLAUSE.LIT.GETPROP CLAUSE (1+ RPTN) 'PATTERN)))
		     TULISTS)))))
	(DT-PUTPROP CLAUSE 'TULISTS TULISTS)))

(DEFUN TERM=PREPARE.CLAUSE (CLAUSE BREADTH.FIRST)
						; EDITED:  7-APR-83 11:10:41
						; INPUT:   A CLAUSE AND A FLAG INDICATING
						;          BREADTH FIRST SEARCH.
						; EFFECT:  THE HINTS ARE TRANSFORMED INTO UNINODES.
						;          THE CLAUSE PROPERTY TEST IS UPDATED.
						; VALUE:   (PURELITS . NEWLITS)
						;          PURELITS IS A LIST OF LITERALS WITHOUT
						;          UNINODES, AND NEWLITS IS A LIST OF LITERALS
						;          WITH NEW UNINODES.
  (PROG (PURELITS NEWLITS FOUND (VARIABLES (DS-CLAUSE.VARIABLES CLAUSE))) (DT-REMPROP CLAUSE 'TEST)
	(DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
	  (PROGN
	    (SETQ FOUND
		  (MEMBER-IF #'(LAMBDA (X) (TERM=UNINODE.IS.NEW X)) (CAR (DS-CLAUSE.LIT.GETPROP CLAUSE (1+ RPTN) 'UNINODE))))
	    (COND
	      (BREADTH.FIRST
	       (DS-CLAUSE.LIT.PUTPROP CLAUSE (1+ RPTN) 'HINT
				      (REMOVE-IF-NOT
					#'(LAMBDA (HINT)
					    (COND
					      ((TERM=HINT.IS.ACTIVE HINT)
					       (SETQ FOUND (OR (TERM=SET.UNINODE (1+ RPTN) CLAUSE HINT VARIABLES) FOUND)) NIL)
					      (T)))
					(DS-CLAUSE.LIT.GETPROP CLAUSE (1+ RPTN) 'HINT)))
	       (COND ((DS-CLAUSE.LIT.GETPROP CLAUSE (1+ RPTN) 'HINT) (DT-PUTPROP CLAUSE 'TEST T))))
	      (T
	       (MAPC
		 #'(LAMBDA (HINT) (SETQ FOUND (OR (TERM=SET.UNINODE (1+ RPTN) CLAUSE HINT VARIABLES) FOUND)))
		 (DS-CLAUSE.LIT.GETPROP CLAUSE (1+ RPTN) 'HINT))
	       (DS-CLAUSE.LIT.REMPROP CLAUSE (1+ RPTN) 'HINT)))
	    (COND (FOUND (SETQ NEWLITS (CONS (1+ RPTN) NEWLITS))))
	    (COND ((NULL (DS-CLAUSE.LIT.GETPROP CLAUSE (1+ RPTN) 'UNINODE)) (SETQ PURELITS (CONS (1+ RPTN) PURELITS))))))
	(RETURN (CONS  PURELITS NEWLITS))))

(DEFUN TERM=MERGE.LIST.OF.TULISTS (CLAUSE TULISTS NEWLITS)
						; EDITED:  7-APR-83 10:17:55
						; INPUT:   A CLAUSE, A LIST OF TULISTS AND A LIST
						;          OF LITERALS WITH NEW UNINODES.
						; VALUE:   A NEW TULIST CONTAINING THE MERGED
						;          UNIFIERS OF TULISTS.
						; EFFECT:  THE INTERMEDIATE RESULTS ARE STORED
						;          UNDER THE CLAUSE PROPERTY 'TULISTS.
  (COND
    ((CDR TULISTS)
     (COND
       ((NOT (TERM=TULIST.IS (CAR TULISTS)))
	(RPLACA TULISTS (TERM=MERGE.LIST.OF.TULISTS CLAUSE (CAR TULISTS) NEWLITS))))
     (COND
       ((NOT (TERM=TULIST.IS (SECOND TULISTS)))
	(RPLACA (CDR TULISTS) (TERM=MERGE.LIST.OF.TULISTS CLAUSE (SECOND TULISTS) NEWLITS))))
     (PROG (NEW.TULIST SILINKS)
	   (DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
	     (COND ((DS-CLAUSE.LINKS 'SI CLAUSE (1+ RPTN)) (SETQ SILINKS T) (SETQ RPTN 0))))
	   (SETQ NEW.TULIST
		 (TERM=TULIST.CREATE
		   (APPEND (TERM=TULIST.LITNOS (CAR TULISTS)) (COPY-LIST (TERM=TULIST.LITNOS (SECOND TULISTS))))
		   (TERM=MERGE (TERM=TULIST.TRACE.UNIFIERS (CAR TULISTS)) (TERM=TULIST.TRACE.UNIFIERS (SECOND TULISTS))
			       (TERM=NOT.OLD.WITH.OLD (CAR TULISTS) (SECOND TULISTS) NEWLITS (DS-CLAUSE.NOLIT CLAUSE))
			       (TERM=TARGETPATTERN CLAUSE (CAR TULISTS) (SECOND TULISTS)) SILINKS TULISTS)))
	   (DT-PUTPROP CLAUSE 'TULISTS
		       (SORT (CONS NEW.TULIST (DT-GETPROP CLAUSE 'TULISTS))
			     #'(LAMBDA (TULIST1 TULIST2)
				 (> (LIST-LENGTH (TERM=TULIST.LITNOS TULIST1)) (LIST-LENGTH (TERM=TULIST.LITNOS TULIST2))))))
	   (RETURN NEW.TULIST)))
    (T (CAR TULISTS))))

(DEFUN TERM=NOT.OLD.WITH.OLD (TULIST1 TULIST2 NEWLITS NOLITS)
						; EDITED:  7-APR-83 13:09:24
						; INPUT:   TWO TULISTS AND THE LITERALS WITH NEW
						;          UNINODES AND THE LENGTH OF THE EXAMINED
						;          CLAUSE
						; VALUE:   T IF NO NEW UNINODES OCCUR AT THE LITERALS
						;          NOT COVERED BY THE TULISTS
  (SETQ TULIST1 (APPEND (TERM=TULIST.LITNOS TULIST1) (COPY-LIST (TERM=TULIST.LITNOS TULIST2))))
  (OR (EQL (LIST-LENGTH TULIST1) (1- NOLITS)) (SUPERSET TULIST1 NEWLITS)))

(DEFUN TERM=TARGETPATTERN (CLAUSE TULIST1 TULIST2)
						; EDITED:  7-APR-83 09:08:19
						; INPUT:   A CLAUSE AND TWO TULISTS
						; VALUE:   THE PATTERN CORRESPONDING TO THE LITERALS
						;          NOT COVERED BY THE TWO TULISTS.
  (PROG ((LITNOS (APPEND (TERM=TULIST.LITNOS TULIST1) (COPY-LIST (TERM=TULIST.LITNOS TULIST2)))) VARIABLES PATTERN)
	(DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
	  (COND
	    ((NOT (MEMBER (1+ RPTN) LITNOS)) (SETQ VARIABLES (APPEND (DS-CLAUSE.LIT.VARIABLES CLAUSE (1+ RPTN))
								     (COPY-LIST VARIABLES))))))
	(MAPC #'(LAMBDA (VARIABLE) (SETQ PATTERN (CONS (COND ((MEMBER VARIABLE VARIABLES) T)) PATTERN)))
	      (DS-CLAUSE.VARIABLES CLAUSE))
	(RETURN (NREVERSE PATTERN))))

(DEFUN TERM=QUOTIENT (N M)
						; EDITED:  6-APR-83 20:02:52
						; INPUT:   TWO INTEGERS
						; VALUE:   SEE CODE.
  (COND ((ZEROP (REM N M)) (MOD N M)) (T (1+ (MOD N M)))))

(DEFUN TERM=GET.TULISTS (TULIST NOLIT IGNORED.LITNOS &OPTIONAL NOSORTFLAG)
						; EDITED:  6-APR-83 19:18:47
						; INPUT:   A LIST OF TULISTS, AN INTEGER AND A LIST OF
						;          INTEGERS  SMALLER THAN NOLIT AND A FLAG.
						;          THE TULIST IS EXPECTED TO CONTAIN AT
						;          LEAST ALL SINGLE LITNOS SMALLER THAN NOLIT.
						; VALUE:   A LIST WITH A MINIMUM NUMBER OF TULISTS
						;          NECESSARY TO COVER THE REQUIRED LITERALS.
						;          IF NOSORTFLAG = NIL,
						;          THE TULISTS ARE SORTED AND PAIRWISE NESTED
						;          IN SUCH A WAY THAT OPTIMIZED CHAINS OF
						;          COVERED LITERALS ARE PRODUCED.
  (PROG (LITNOS TULISTS TRIAL TAIL REMAINDER (LENGTH 0))
	(DODOWN (RPTN NOLIT)
	  (COND ((MEMBER (1+ RPTN) IGNORED.LITNOS)) (T (SETQ LITNOS (CONS (1+ RPTN) LITNOS)) (SETQ LENGTH (1+ LENGTH)))))
	(WHILE
	  (AND TULIST
	       (OR (NULL TULISTS)
		   (< (TERM=QUOTIENT LENGTH (LIST-LENGTH (TERM=TULIST.LITNOS (CAR TULIST)))) (LIST-LENGTH TULISTS))))
	  (COND
	    ((SUPERSET LITNOS (SETQ REMAINDER (TERM=TULIST.LITNOS (CAR TULIST))))
	     (SETQ REMAINDER (SET-DIFFERENCE LITNOS REMAINDER)) (SETQ TRIAL (LIST (CAR TULIST))) (SETQ TAIL (CDR TULIST))
	     (WHILE (AND REMAINDER TAIL)
	       (COND
		 ((SUPERSET REMAINDER (TERM=TULIST.LITNOS (CAR TAIL)))
		  (SETQ REMAINDER (NSET-DIFFERENCE REMAINDER (TERM=TULIST.LITNOS (CAR TAIL)))) (SETQ TRIAL (CONS (CAR TAIL) TRIAL))))
	       (SETQ TAIL (CDR TAIL)))
	     (COND
	       ((NULL REMAINDER)
		(COND
		  ((CDR TRIAL)
		   (COND
		     ((OR (NULL TULISTS) (< (LIST-LENGTH TRIAL) (LIST-LENGTH TULISTS))) (SETQ TULISTS TRIAL))))
		  (T (RETURN-FROM TERM=GET.TULISTS TRIAL)))))))
	  (SETQ TULIST (CDR TULIST)))
	(COND (TULISTS (RETURN (COND (NOSORTFLAG TULISTS) (T (TERM=SORT.TULISTS TULISTS NOLIT)))))
	      (T (ERROR "TERM=GET.TULISTS: NO PROPER TULIST: ~A" NIL)))))

(DEFUN TERM=SORT.TULISTS (TULISTS NOLIT)
						; EDITED:  8-APR-83 09:47:25
						; INPUT:   A LIST OF TULISTS AND AN INTEGER
						; VALUE:   THE TULISTS ARE SORTED AND NESTED IN
						;          SUCH A WAY THAT PAIRWISE MERGING OF
						;          THEM PRODUCES OPTIMIZED CHAINS OF
						;          COVERED LITERALS.
  (COND
    ((CDDR TULISTS)
     (SETQ TULISTS
	   (SORT TULISTS
		 #'(LAMBDA (TULIST1 TULIST2) (< (CAR (TERM=TULIST.LITNOS TULIST1)) (CAR (TERM=TULIST.LITNOS TULIST2))))))
     (PROG (LITLIST TAIL NOT.FOUND PAIR CODE POSITION NUMBER (COUNTER 1))
	   (DODOWN (RPTN NOLIT) (SETQ LITLIST (CONS NIL LITLIST)))
	   (MAPC
	     #'(LAMBDA (TULIST)
		 (MAPC #'(LAMBDA (LITNO) (RPLACA (NTHCDR (1- LITNO) LITLIST) COUNTER)) (TERM=TULIST.LITNOS TULIST))
		 (SETQ COUNTER (1+ COUNTER)))
	     TULISTS)
	   (SETQ LITLIST (NREVERSE LITLIST))
	   (WHILE (NULL (CAR LITLIST))
	     (SETQ LITLIST (CDR LITLIST))) (SETQ LITLIST (NREVERSE LITLIST))
	   (SMAPL #'IGNORE
		  #'(LAMBDA (TAIL) (WHILE (EQL (CAR TAIL) (SECOND TAIL)) (RPLACD TAIL (CDDR TAIL))) (CDR TAIL))
		  LITLIST)
	   (COND ((NULL (CAR LITLIST)) (SETQ LITLIST (CDR LITLIST)))) (SETQ NUMBER (1+ NOLIT)) (SETQ COUNTER 2)
	   (WHILE (AND (< COUNTER (LIST-LENGTH TULISTS)) (CDDR (REMOVE NIL LITLIST))) (SETQ NOT.FOUND T)
		  (WHILE NOT.FOUND (SETQ TAIL LITLIST)
			 (WHILE TAIL (SETQ PAIR NIL)
				(SEVERY
				  #'(LAMBDA (ELEMENT)
				      (COND
					(ELEMENT
					 (COND ((< (LIST-LENGTH PAIR) COUNTER) (SETQ PAIR (INSERT ELEMENT PAIR)))
					       ((MEMBER ELEMENT PAIR))))))
				  #'(LAMBDA (TAIL) (SETQ POSITION (CDR TAIL))) TAIL)
				(COND
				  ((AND (EQL COUNTER (LIST-LENGTH PAIR))
					(NOTANY #'(LAMBDA (ELEMENT) (MEMBER ELEMENT PAIR))
						(NCONC (LDIFF LITLIST TAIL) POSITION)))
				   (SETQ PAIR (NREVERSE PAIR))
				   (WHILE (CDDR PAIR) (RPLACA PAIR (LIST (CAR PAIR) (SECOND PAIR))) (RPLACD PAIR (CDDR PAIR)))
				   (SETQ CODE (CONS (CONS NUMBER PAIR) CODE)) (RPLACA TAIL NUMBER) (RPLACD TAIL POSITION)
				   (SETQ NUMBER (1+ NUMBER))
				   (SETQ TAIL NIL) (SETQ NOT.FOUND NIL) (SETQ COUNTER 2))
				  ((< (LIST-LENGTH (REMOVE NIL TAIL)) COUNTER) (SETQ NOT.FOUND NIL) (SETQ COUNTER (1+ COUNTER))
				   (SETQ TAIL NIL))
				  (T (SETQ TAIL (CDR TAIL)))))))
	   (SETQ LITLIST (REMOVE-DUPLICATES (DELETE NIL LITLIST)))
	   (WHILE (CDDR LITLIST) (RPLACA LITLIST (LIST (CAR LITLIST) (SECOND LITLIST))) (RPLACD LITLIST (CDDR LITLIST)))
	   (TERM=SORT.EXPAND LITLIST CODE) (TERM=SORT.REPLACE LITLIST TULISTS) (RETURN LITLIST)))
    (T TULISTS)))

(DEFUN TERM=SORT.EXPAND (LITLIST CODE)
						; EDITED:  8-APR-83 09:29:01
						; INPUT:   A TWO-ELEMENT LIST AND AN ASSOCIATION LIST
						; EFFECT:  THE ELEMENTS OF LITLIST ARE REPLACED
						;          BY THOSE DEFINED IN CODE.
						; VALUE:   UNDEFINED. LITLIST IS DESTRUCTIVELY CHANGED
  (MAPL
    #'(LAMBDA (LITLIST)
        (COND
          ((NUMBERP (CAR LITLIST))
	   (COND
	     ((ASSOC (CAR LITLIST) CODE) (RPLACA LITLIST (CDR (ASSOC (CAR LITLIST) CODE))) (TERM=SORT.EXPAND (CAR LITLIST) CODE))))
          (T (TERM=SORT.EXPAND (CAR LITLIST) CODE))))
    LITLIST))

(DEFUN TERM=SORT.REPLACE (LITLIST TULISTS)
						; EDITED:  8-APR-83 10:25:49
						; INPUT:   LITLIST DEFINED IN TERM=SORT.TULISTS
						;          AND THE CORRESPONDING TULISTS.
						; EFFECT:  THE NUMBERS IN LITLIST ARE REPLACED BY
						;          THE CORRESONDING TULIST.
						; VALUE:   UNDEFINED.
  (MAPL
    #'(LAMBDA (LITLIST)
        (COND ((NUMBERP (CAR LITLIST)) (RPLACA LITLIST (CAR (NTHCDR (1- (CAR LITLIST)) TULISTS))))
	      (T (TERM=SORT.REPLACE (CAR LITLIST) TULISTS))))
    LITLIST))

(DEFUN TERM=CUT.TULISTS (CLAUSE LITNO)
						; EDITED:  8-APR-83 13:05:15
						; INPUT:   A CLAUSE AND A LITERAL NUMBER
						; EFFECT:  EACH  TULIST WHICH IS NOT NECESSARY FOR
						;          DEDUCING UNITS FROM LITERALS LOWER
						;          THAN LITNO IS DELETED.
						; VALUE:   UNDEFINED.
  (PROG ((TULISTS (DT-GETPROP CLAUSE 'TULISTS)) (NOLIT (DS-CLAUSE.NOLIT CLAUSE)) NEEDED)
	(DODOWN (RPTN (1- LITNO)) (SETQ NEEDED (NCONC (TERM=GET.TULISTS TULISTS NOLIT (LIST (1+ RPTN)) T) NEEDED)))
	(DT-PUTPROP CLAUSE 'TULISTS (REMOVE-IF-NOT #'(LAMBDA (TULIST) (MEMBER TULIST NEEDED)) TULISTS))))

(DEFUN TERM=LINEARIZE.UNITS NIL
						; EDITED: 23-SEP-82 13:45:11
						; EFFECT: TERM*SOLUTION IS SET ALLOWING TO RECONSTRUCT
						;         ALL INTERNALLY GERNERATED UNITCLAUSES.
						; VALUE:  UNDEFINED.
  (MAPC
    #'(LAMBDA (PREDICATELIST)
        (MAPC #'(LAMBDA (UNIT) (COND ((TERM=UNIT.NODES UNIT) (TERM=LINEARIZATION UNIT))))
	      (MI-TERMSTRUCTURE.TERMLISTS (TERM=UNIT.PREDICATELIST.SIGNLIST PREDICATELIST '+)))
        (MAPC #'(LAMBDA (UNIT) (COND ((TERM=UNIT.NODES UNIT) (TERM=LINEARIZATION UNIT))))
	      (MI-TERMSTRUCTURE.TERMLISTS (TERM=UNIT.PREDICATELIST.SIGNLIST PREDICATELIST '-))))
    TERM*UNITS))

(DEFUN TERM=LINEARIZATION (UNIT)
						; EDITED:  7-JUL-82 12:56:57
						; INPUT : A UNIT OR A PSEUDOUNIT(LIT=NIL)
						; VALUE : UNDEFINED
						; EFFECT : SETS TERM*SOLUTION,WHICH IS THE SOLUTION
						;          IN THE VALUE OF THE INTERFACE FUNCTION.
						;          (SPECIFICATION OF TERM*SOLUTION UNDER
						;           TERM-TERMINATOR
  (PROG
    ((NODES (TERM=SUBSET.NODES (TERM=UNIT.NODES UNIT))) (CLAUSE (TERM=UNIT.CLAUSE UNIT)) UNIT.1 CURRENT.ELEMENT LINK.CELL
     LITNOS LITERALS)
    (MAPC
      #'(LAMBDA (UNINODE) (SETQ UNIT.1 (TERM=UNINODE.UNIT UNINODE))
		(COND ((CONSP (CDR UNIT.1)) (COND ((TERM=UNIT.NODES UNIT.1) (TERM=LINEARIZATION UNIT.1))))
		      (T (TERM=UNINODE.PUT.UNIT UNINODE NIL))))
      NODES)
    (SETQ CURRENT.ELEMENT (LIST (TERM=STANDARD.UNIFIER (TERM=UNIT.UNIFIER UNIT) CLAUSE) CLAUSE NIL NIL))
    (SETQ TERM*SOLUTION (NCONC1 TERM*SOLUTION CURRENT.ELEMENT))
    (MAPC
      #'(LAMBDA (NODE) (SETQ LITNOS (CONS (TERM=LINEARIZATION.LITNO CLAUSE NODE) LITNOS))
		(RPLACA (CDDR CURRENT.ELEMENT) (SETQ LINK.CELL (CONS (TERM=UNINODE.LINK NODE) (THIRD CURRENT.ELEMENT))))
		(PROG ((POINTER (TERM=UNIT.AUXILIARY (TERM=UNINODE.UNIT NODE))))
		      (COND ((CONSP POINTER) (RPLACA (CDDDR POINTER) (CONS LINK.CELL (FOURTH POINTER)))))))
      NODES)
    (DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
      (COND ((NOT (MEMBER (1+ RPTN) LITNOS)) (SETQ LITERALS (CONS (1+ RPTN) LITERALS)))))
    (NCONC1 CURRENT.ELEMENT LITERALS) (TERM=UNIT.DELETE.NODES UNIT) (TERM=UNIT.SET.AUXILIARY UNIT CURRENT.ELEMENT)))

(DEFUN TERM=LINEARIZATION.LITNO (CLAUSE UNINODE)
						; EDITED: 24. 2. 1984
						; INPUT:  A CLAUSE AND A UNINODE CONNECTED TO IT.
						; VALUE:  THE LITERAL NUMBER TO WHICH THIS UNINODE
						;         IS CONNECTED.
  (PROG ((LINK (TERM=UNINODE.LINK UNINODE)))
	(RETURN
	  (CASE (DS-LINK.COLOUR LINK)
	    (R (COND ((EQL CLAUSE (DS-LINK.POSPAR LINK)) (DS-LINK.POSLITNO LINK)) (T (DS-LINK.NEGLITNO LINK))))
	    (RIW
	      (COND
		((MEMBER (DS-LINK.POSLITNO LINK) (CAR (CDDDDR (TERM=UNIT.AUXILIARY (TERM=UNINODE.UNIT UNINODE)))))
		 (DS-LINK.NEGLITNO LINK))
		(T (DS-LINK.POSLITNO LINK))))
	    (SI (DS-LINK.POSLITNO LINK))
	    (OTHERWISE (ERROR "ILLEGAL LINK COLOUR IN TERM=LINEARIZATION.LITNO: ~A" (DS-LINK.COLOUR LINK)))))))

(DEFUN TERM=SUBSET.NODES (NODES)
						; EDITED: 30-JUN-83 14:24:12
						; INPUT:  A LIST OF UNINODES
						; VALUE:  A LIST OF NODES WITH NO FLINKS OR A LIST
						;         WITH ONE NODE WITH AN FLINK.
  (COND
    ((REMOVE-IF-NOT
       #'(LAMBDA (NODE) (SETQ NODE (TERM=UNINODE.LINK NODE)) (OR (CONSP NODE) (NEQ 'F (DS-LINK.COLOUR NODE))))
       NODES))
    (T (CAR NODES))))

(DEFMACRO TERM=BIND (SUBSTITUTES VARIABLES)
						; EDITED: 24. 2. 1984
						; INPUT:  A LIST OF TERMS AND A LIST OF VARIABLES,
						;         BOTH OF THE SAME LENGTH.
						; EFFECT: THE VARIABLES ARE BOUND WITH THE
						;         CORRESPONDING TERMS.
						; VALUE:  UNDEFINED.
  `(MAPC #'(LAMBDA (VAR VAL) (UNLESS (EQ VAR VAL)(DT-VARIABLE.PUT.BINDING VAR VAL))) ,VARIABLES ,SUBSTITUTES))

(DEFMACRO TERM=UNBIND (VARIABLES)
						; EDITED: 24. 2. 1984
						; INPUT:  A LIST OF VARIABLES.
						; EFFECT: THE VARIABLES ARE UNBOUND.
						; VALUE:  UNDEFINED.
  `(MAPC #'(LAMBDA (VAR) (DT-VARIABLE.DELETE.BINDING VAR)) ,VARIABLES))

(DEFUN TERM=MERGE (TRACE.UNIFIERS1 TRACE.UNIFIERS2 &OPTIONAL NOT.OLD.WITH.OLD TARGETPATTERN SILINKS.EXPECTED TULISTS)
  (DECLARE (IGNORE TULISTS))
						; EDITED AT 16-MAR-83 13:59
						; EDITED: 26-JUL-82 17:29:01
						; INPUT:  TWO STRUCTURES OF THE FORM TRACE.UNIFIERS,
						;         A BOOLEAN VALUE, A LIST WITH T'S AND NIL'S
						;         AND A FLAG
						; EFFECT: THE MERGED UNIFIERS OF TRACE.UNIFIERS1
						;         AND TRACE.UNIFIERS2 ARE CALCULATED AND
						;         A NEW DATA STRACTURE NEW.TRACE.UNIFIERS IS
						;         GENERATED.
						;         IF NOT.OLD.WITH.OLD = T UNIFIERS BELONGING
						;         TO OLD NODES ARE NOT MERGED.
						; VALUE:  THE NEW TRACE.UNIFIERS.
  (PROG (CONNECTION CONDITION)
	(COND
	  (SILINKS.EXPECTED
	   (COND
	     (NOT.OLD.WITH.OLD
	      (SETQ CONDITION
		    #'(LAMBDA (NODES) (CONS (MEMBER-IF #'(LAMBDA (X) (TERM=UNINODE.IS.NEW X)) NODES)
					    (MAPCAN #'(LAMBDA (DUMMY)
							(SETQ DUMMY (TERM=UNINODE.UNIT DUMMY))
							(IF (NUMBERP (REST DUMMY))
							    (LIST DUMMY)
							    NIL))
						    NODES))))
	      (SETQ CONNECTION
		    #'(LAMBDA (CONDITION1 CONDITION2)
			(AND (OR (CAR CONDITION1) (CAR CONDITION2))
			     (OR (NULL (CDR CONDITION1)) (NULL (CDR CONDITION2))
				 (NOT (TERM=CYCLE (NCONC (CDR CONDITION1) (CDR CONDITION2)))))))))
	     (T
	      (SETQ CONDITION
		    #'(LAMBDA (NODES)
			(MAPCAN
			  #'(LAMBDA (DUMMY) (SETQ DUMMY (TERM=UNINODE.UNIT DUMMY))
				    (COND ((NUMBERP (CDR DUMMY)) (LIST DUMMY))))
			  NODES)))
	      (SETQ CONNECTION
		    #'(LAMBDA (CONDITION1 CONDITION2)
			(OR (NULL (CDR CONDITION1)) (NULL (CDR CONDITION2))
			    (NOT (TERM=CYCLE (NCONC (CDR CONDITION1) (CDR CONDITION2))))))))))
	  (NOT.OLD.WITH.OLD
	   (SETQ CONDITION #'(LAMBDA (NODES) (MEMBER-IF #'(LAMBDA (X) (TERM=UNINODE.IS.NEW X)) NODES)))
	   (SETQ CONNECTION #'(LAMBDA (X Y) (OR X Y))))
	  (T (SETQ CONNECTION #'(LAMBDA (X Y) (DECLARE (IGNORE X Y)) T))
	     (SETQ CONDITION  #'NULL))) 
	(RETURN
	  (MI-MERGE.TERMSTRUCTURES TRACE.UNIFIERS1 TRACE.UNIFIERS2 TARGETPATTERN CONNECTION CONDITION CONDITION
				   #'(LAMBDA (PROPERTY1 PROPERTY2 &OPTIONAL X Y)
				       (DECLARE (IGNORE X Y))
				       (APPEND PROPERTY1 (COPY-LIST PROPERTY2))) TERM*INSTANCETEST T))))

(DEFUN TERM=CYCLE (ALIST)
						; EDITED: 15. 9. 1983
						; INPUT:  A LIST OF DOTTED.PAIRS
						;         (FROM.LITNO . TO.LITNO)
						; VALUE:  T IF ALIST CONTAINS A CYCLE, ELSE NIL.
  (PROG (FROM TO COUNTER (LENGTH (LIST-LENGTH ALIST)))
	(MEMBER-IF
	  #'(LAMBDA (ELEMENT) (SETQ FROM (CAR ELEMENT)) (SETQ TO (CDR ELEMENT)) (SETQ COUNTER LENGTH)
		    (WHILE (SETQ TO (CDR (ASSOC TO ALIST))) (SETQ COUNTER (1- COUNTER))
			   (COND ((OR (EQL FROM TO) (ZEROP COUNTER)) (RETURN-FROM TERM=CYCLE T)))))
	  ALIST)))

(DEFUN TERM=CREATE.INITIAL.TRUNIFIERS (UNINODES PATTERN)
						; EDITED: 26-JUL-82 16:12:32
						; INPUT:  A LIST OF UNINODES AND AN INTEGER.
						;         VARLENGTH IS THE NUMBER OF VARIABLES OF
						;         THE EXAMINED CLAUSE.
						; EFFECT: A DATA STRUCTURE TRACE.UNIFIERS IS CREATED
						;         AND EVERY UNINODE IS INSERTED.
						; VALUE:  THE NEW TRACE.UNIFIERS.
  (PROG ((TRACE.UNIFIERS (MI-CREATE.EMPTY.TERMSTRUCTURE PATTERN)))
	(MAPC
	  #'(LAMBDA (NODE)
	      (MI-INSERT.TERMLIST.INTO.TERMSTRUCTURE TRACE.UNIFIERS (TERM=UNINODE.UNIFIER NODE) (LIST NODE) NIL
						     (DS-RULES)))
	  UNINODES)
	(RETURN TRACE.UNIFIERS)))

(DEFMACRO TERM=TRACE.UNIFIER.UNINODES (TRACE.UNIFIER)
						; EDITED:  7-MAR-83 17:16:45
						; INPUT:   TRACE.UNIFIER IS AN MI-TERMLIST
						; VALUE:   THE UNIFIER-NODES
  ` (MI-TERMLIST.PROPERTY ,TRACE.UNIFIER))

(DEFUN TERM=UNIFIER.NORMALIZE (UNIFIER VARIABLES)
						; EDITED: 22-JUL-82 12:31:06
						; INPUT:  A STANDARD UNIFIER AND A LIST OF VARIABLES.
						; VALUE:  A NORMALIZED FORM OF THE UNIFIER
						;         ACCORDING TO VARIABLES.
						; EXAMPLE: UNIFIER: (X1 A X3 B X4 X8)
						;         VARIABLES (X1 X2 X3 X4)
						;         VALUE:    (V1 B  X2 A ). (V1 IS NEW])
						;         AFTER VARIABLES IS REVERSED, VARIABLES AND
						;         THE NORMALIZED UNIFIER MAY BE DIRECTLY
						;         USED IN (SUBPAIR VARIABLES UNIFIER TERMLIST)
  (PROG (NORM.UNIFIER TAIL)
	(MAPC
	  #'(LAMBDA (VARIABLE)
	      (COND
		((SETQ TAIL (SSOMEL #'(LAMBDA (UTAIL) (AND (EQL (FIRST UTAIL) VARIABLE) UTAIL))
				    #'CDDR
				    UNIFIER))
		 (SETQ NORM.UNIFIER (CONS (SECOND TAIL) NORM.UNIFIER)))
		(T (SETQ NORM.UNIFIER (CONS VARIABLE NORM.UNIFIER)))))
	  VARIABLES)
	(SETQ NORM.UNIFIER (DT-TERM.RENAMED NORM.UNIFIER NIL))
	(SETQ TERM*NEW.VARIABLES (NCONC (POP NORM.UNIFIER) TERM*NEW.VARIABLES))
	(RETURN (CAR (DT-ABBREVIATION.COMPRESS.TERMLIST (NREVERSE (COPY-TREE NORM.UNIFIER)))))))

(DEFUN TERM=IS.TAUTOLOGY (SIGN PREDICATE TERMLIST)
						; EDITED: 18-APR-83 09:20:30
						; INPUT:  A LITERAL
						; VALUE:  T <=> THE LITERAL IS AN INSTANCE OF THE
						;         REFLEXIVITY RESP. IRREFLEXIVITY AXIOM.
  (AND
    (OR (AND (DT-PREDICATE.IS.MARKED REFLEXIVE PREDICATE) (DS-SIGN.IS.POSITIVE SIGN))
	(AND (DT-PREDICATE.IS.MARKED IRREFLEXIVE PREDICATE) (DS-SIGN.IS.NEGATIVE SIGN)))
    (UNI-EQUAL.TERMS (CAR TERMLIST) (SECOND TERMLIST))))

(DEFUN TERM=1_ITERATIONS (CLAUSES ITERATIONS)
						; EDITED: 30-JUN-82 10:59:28
						; INPUT : A LIST OF CLAUSES AND AN INTEGER.
						; VALUE : T <=> A REFUTATION WAS FOUND.
						; EFFECT : THIS IS THE TOP LEVEL LOOP FOR THE SEARCH
						;          FOR TERMINATOR SITUATIONS RESP. FOR
						;          DEDUCING NEW UNITCLAUSES AND CLOSING THE
						;          SLTU-LISTS.
  (PROG
    ((COUNTER 0) (FASTMODE (ZEROP ITERATIONS)) (ITERATIONS (COND ((ZEROP ITERATIONS) 1) (T ITERATIONS))) READY CLOSED)
    (DECLARE (SPECIAL COUNTER))
    (WHILE (AND (< COUNTER ITERATIONS) (NULL READY))
      (MEMBER-IF
	#'(LAMBDA (CLAUSE)
	    (COND
	      ((AND (DT-GETPROP CLAUSE 'TEST) (NOT (MEMBER CLAUSE CLOSED)))
	       (SETQ READY (TERM=1_EXAMINE.CLAUSE CLAUSE FASTMODE))
	       (COND ((EQL READY 'CLOSED) (SETQ CLOSED (CONS CLAUSE CLOSED)) (SETQ READY NIL))) READY)))
	CLAUSES)
      (SETQ COUNTER (1+ COUNTER))
      (COND
	((OPT-GET.OPTION TERM_BREADTH.FIRST)
	 (MAPC
	   #'(LAMBDA (CLAUSE)
	       (DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
		 (MAPC #'(LAMBDA (HINT) (TERM=HINT.ACTIVATE HINT)) (DS-CLAUSE.LIT.GETPROP CLAUSE (1+ RPTN) 'HINT))))
	   CLAUSES))))
    (RETURN READY)))

(DEFUN TERM=1_EXAMINE.CLAUSE (CLAUSE FASTMODE)
						; EDITED:  7-APR-83 15:37:26
						; INPUT:   A CLAUSE AND A FLAG
						; VALUE:   T <=> A REFUTATION WAS FOUND.
						;          CLOSED <=> THE SLTU-LIST OF CLAUSE IS READY
						; EFFECT:  IF NO REFUTATION CAN BE FOUND, BUT
						;          FASTMODE IS NOT NIL, ALL UNITCLAUSES WHICH
						;          CAN BE DEDUCED FROM THIS CLAUSE AND THE
						;          EXISTING UNITS ARE GENERATED.
						;          IF THE SLTU-LIST IS CLOSED, NO MORE UNITS
						;          ARE CREATED.
  (PROG1
    (PROG
      ((NEWLITS (TERM=PREPARE.CLAUSE CLAUSE (OPT-GET.OPTION TERM_BREADTH.FIRST)))
       (IGNORED.LITNOS (DT-GETPROP CLAUSE 'SLTU.IGNORED.LITNOS)) (NOT.CREATED T) PURELITS)
      (SETQ PURELITS (POP NEWLITS))
      (COND
        (NEWLITS
	 (COND
	   (IGNORED.LITNOS
	    (COND
	      ((AND (SUPERSET IGNORED.LITNOS PURELITS) (NOT (SUPERSET IGNORED.LITNOS NEWLITS)))
	       (TERM=CREATE.INITIAL.TULISTS CLAUSE) (SETQ NOT.CREATED NIL)
	       (COND ((TERM=1_COMPATIBILITY.TEST.PARTIAL CLAUSE NEWLITS) (RETURN 'CLOSED)))))))
	 (COND ((CDR PURELITS))
	       (PURELITS
		(COND
		  ((NOT FASTMODE) (COND (NOT.CREATED (TERM=CREATE.INITIAL.TULISTS CLAUSE)))
		   (TERM=DEDUCE.UNITS CLAUSE (CAR PURELITS) NEWLITS))))
	       ((PROG2 (COND (NOT.CREATED (TERM=CREATE.INITIAL.TULISTS CLAUSE)))
		       (TERM=COMPATIBILITY.TEST.FULL CLAUSE NEWLITS))
		(RETURN T))
	       ((NOT FASTMODE)
		(DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
		  (PROGN (TERM=DEDUCE.UNITS CLAUSE (1+ RPTN) NEWLITS) (TERM=CUT.TULISTS CLAUSE (1+ RPTN)))))))))
    (DT-REMPROP CLAUSE 'TULISTS)
    (DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
      (MAPC #'(LAMBDA (NODE) (TERM=UNINODE.MARK.OLD NODE))
	    (CAR (DS-CLAUSE.LIT.GETPROP CLAUSE (1+ RPTN) 'UNINODE))))))

(DEFUN TERM=1_CREATE.SLTU (CILS)
						; EDITED: 11-APR-83 19:27:06
						; INPUT:  A LIST WITH ELEMENTS
						;         (CLAUSE IGNORED.LITNOS
						;                 (LINK1 . SUBSTITUTION1) ...)
						; EFFECT: FOR EVERY CLAUSE IN CILS CLAUSE-PROPERTIES
						;         'IGNORED.LITNOS, 'SLTU.VARIABLES, 'SLTU
						;         ARE CREATED.
						;         SLTU IS A LIST WITH MI-TERMSTRUCTURES
						;         CONTAINING ONE P-UNIFIER CORRESPONDING TO
						;         A SUBSTITUTION IN CILS. THE PROPERTY-CELL
						;         OF THIS MI-TERMLIST IS
						;         (LIST (LINK . SUBSTITUTION)) AND EQ TO THE
						;         ELEMENT IN CILS.
						;         SLTU.VARIABLES IS A LIST OF VARIABLES
						;         OCCURRING IN ALL P-UNIFIERS OF THE CLAUSE,
						;         BUT NOT IN THE CLAUSE ITSELF.
						; VALUE:  UNDEFINED.
  (MAPC
    #'(LAMBDA (CILS)
        (PROG
          ((CLAUSE (CAR CILS)) (IGNORED.LITNOS (SECOND CILS)) (LINK.SUBSTITUTION.S (CDDR CILS)) SLTU-LIST SLTU PATTERN VARIABLES
	   TUNIFIER)
          (MAPC
            #'(LAMBDA (LITNO) (SETQ VARIABLES (COPY-LIST (DS-CLAUSE.LIT.VARIABLES CLAUSE LITNO)))) IGNORED.LITNOS)
          (MAPC #'(LAMBDA (VARIABLE) (SETQ PATTERN (CONS (COND ((MEMBER VARIABLE VARIABLES) T)) PATTERN)))
		(DS-CLAUSE.VARIABLES CLAUSE))
          (SETQ PATTERN (NREVERSE PATTERN)) (SETQ VARIABLES NIL)
          (MAPC
            #'(LAMBDA (LINK.SUBSTITUTION)
                (SETQ SLTU-LIST (CONS (SETQ SLTU (MI-CREATE.EMPTY.TERMSTRUCTURE PATTERN)) SLTU-LIST))
                (SETQ TUNIFIER (TERM=UNIFIER.NORMALIZE (CDR LINK.SUBSTITUTION) (DS-CLAUSE.VARIABLES CLAUSE)))
                (SETQ VARIABLES
		      (NCONC (NSET-DIFFERENCE (DT-TERMLIST.VARIABLES TUNIFIER) (DS-CLAUSE.VARIABLES CLAUSE))))
                (MI-INSERT.TERMLIST.INTO.TERMSTRUCTURE SLTU TUNIFIER (LIST LINK.SUBSTITUTION NIL) NIL NIL))
            LINK.SUBSTITUTION.S)
          (DT-PUTPROP CLAUSE 'SLTU.IGNORED.LITNOS IGNORED.LITNOS) (DT-PUTPROP CLAUSE 'SLTU.VARIABLES VARIABLES)
          (DT-PUTPROP CLAUSE 'SLTU SLTU-LIST)))
    CILS))

(DEFUN TERM=1_LINEARIZE (CILS)
						; EDITED: 14-APR-83 10:13:43
						; INPUT:  A LIST WITH ELEMENTS
						;         (CLAUSE IGNORED.LITNOS
						;                 (LINK1 . SUBSTITUTION1)  ...)
						; EFFECT: FROM THE SLTU-LISTS, STORED AS CLAUSE
						;         PROPERTY 'SLTU, THE SOLUTION CHAINS ARE
						;         CALCULATED.
						; VALUE:  A LIST WITH ELEMENTS:
						;
						; (CLAUSE . ((LINK1 . SUBST1)(LINK2 . SUBST2) ...) .
						;           CHAIN1) ((LINK . SUBST) ...) . CHAIN2) ...
						;
						;         (LINK1 . SUBST1) ETC ARE EQ TO THE ELEMENTS
						;         IN CILS
  (PROG (RESULT)
	(MAPC
	  #'(LAMBDA (CILS)
	      (PROG ((CLAUSE (CAR CILS)) SLTU-LIST DUMMY VARIABLES DOUBLE)
		    (COND
		      ((SETQ SLTU-LIST (REMOVE-IF-NOT #'TERM=SLTU.TUNIFIER (DT-GETPROP CLAUSE 'SLTU)))
		       (SETQ VARIABLES (DS-CLAUSE.VARIABLES CLAUSE))
		       (MAPC
			 #'(LAMBDA (SLTU)
			     (UNI-DECLARE.VARIABLES.AS.CONSTANTS
			       (NSET-DIFFERENCE (DT-TERMLIST.VARIABLES (TERM=SLTU.TUNIFIER SLTU)) VARIABLES))
			     (COND
			       ((SETQ DOUBLE
				      (CAR
					(MEMBER-IF
					  #'(LAMBDA (ELEMENT)
					      (UNI-UNIFY.TERMLISTS (TERM=SLTU.TUNIFIER SLTU) (SECOND ELEMENT) T))
					  DUMMY)))
				(RPLACA DOUBLE (CONS (TERM=SLTU.LINK.SUBSTITUTION SLTU) (CAR DOUBLE))))
			       (T
				(SETQ DUMMY
				      (CONS 
					(LIST (LIST (TERM=SLTU.LINK.SUBSTITUTION SLTU)) (TERM=SLTU.TUNIFIER SLTU)
					      (TERM=SLTU.UNINODES SLTU)) DUMMY)))))
			 SLTU-LIST)
		       (SETQ SLTU-LIST NIL)
		       (MAPC
			 #'(LAMBDA (LSTU)
			     (UNI-DECLARE.VARIABLES.AS.CONSTANTS
			       (NSET-DIFFERENCE (DT-TERMLIST.VARIABLES (SECOND LSTU)) VARIABLES))
			     (COND
			       ((SETQ DOUBLE
				      (CAR
					(MEMBER-IF
					  #'(LAMBDA (ELEMENT) (UNI-UNIFY.TERMLISTS (SECOND ELEMENT) (SECOND LSTU) T))
					  SLTU-LIST)))
				(RPLACA DOUBLE (CONS (CAR LSTU) (FIRST DOUBLE))))
			       (T (SETQ SLTU-LIST (CONS LSTU SLTU-LIST)))))
			 DUMMY)
		       (UNI-CLEAR.VARIABLES.AS.CONSTANTS)
		       (MAPC
			 #'(LAMBDA (LSTU) (SETQ DUMMY (LAST TERM*SOLUTION))
				   (TERM=LINEARIZATION (TERM=CREATE.PSEUDOUNIT (SECOND LSTU) CLAUSE (THIRD LSTU)))
				   (RPLACD LSTU (COPY-LIST (COND (DUMMY (CDR DUMMY)) (T TERM*SOLUTION)))))
			 SLTU-LIST)
		       (SETQ RESULT (NCONC1 RESULT (CONS CLAUSE SLTU-LIST)))))))
	  CILS)
	(RETURN RESULT)))

(DEFUN TERM=1_COMPATIBILITY.TEST.PARTIAL (CLAUSE NEWLITS)
						; EDITED: 13-APR-83 10:13:03
						; INPUT:  A CLAUSE AND A LIST OF LITERALS WITH
						;         NEW UNINODES.
						; EFFECT: THE SLTU-LIST OF THE CLAUSE IS COMPLETED
						;         AS FAR AS POSSIBLE I.E. FOR EVERY SLTU
						;         A COMPATIBLE TRACE UNIFIER IS COMPUTED
						;         AND INSERTED INTO THE SLTU
						;         (TOGETHER WITH ITS NODES.)
						; VALUE:  T <=> NO OPEN (WITHOUT TRACE UNIFIER) SLTU
						;         REMAINS FOR THIS CLAUSE.
  (PROG
    ((IGNORED.LITNOS (DT-GETPROP CLAUSE 'SLTU.IGNORED.LITNOS)) (SLTU-LIST (DT-GETPROP CLAUSE 'SLTU)) TRACE.UNIFIERS
     TARGETPATTERN)
    (COND
      ((AND IGNORED.LITNOS
	    (SETQ TRACE.UNIFIERS
		  (TERM=TULIST.TRACE.UNIFIERS
		    (TERM=MERGE.LIST.OF.TULISTS CLAUSE
						(TERM=GET.TULISTS (DT-GETPROP CLAUSE 'TULISTS) (DS-CLAUSE.NOLIT CLAUSE)
								  IGNORED.LITNOS) NEWLITS))))
       (DODOWN (RPTN (LIST-LENGTH (DS-CLAUSE.VARIABLES CLAUSE))) (SETQ TARGETPATTERN (CONS NIL TARGETPATTERN)))
       (UNI-DECLARE.VARIABLES.AS.CONSTANTS (DT-GETPROP CLAUSE 'SLTU.VARIABLES))
       (MAPC
	 #'(LAMBDA (SLTU)
	     (PROG
	       ((PUNIFIER
		  (MI-MERGE.TERMSTRUCTURES SLTU TRACE.UNIFIERS TARGETPATTERN #'(LAMBDA (X) (DECLARE (IGNORE X)) T) NIL NIL
					   #'(LAMBDA (X Y Z TERMLIST2)
					       (DECLARE (IGNORE X Y Z))
					       TERMLIST2)
					   T T)))
	       (COND
		 (PUNIFIER (SETQ PUNIFIER (MI-TERMLIST.PROPERTY (CAR (MI-TERMSTRUCTURE.TERMLISTS PUNIFIER))))
			   (TERM=SLTU.PUT.TUNIFIER SLTU (MI-TERMLIST.STANDARD.FULL PUNIFIER))
			   (TERM=SLTU.PUT.UNINODES SLTU (TERM=TRACE.UNIFIER.UNINODES PUNIFIER)) (RETURN T)))))
	 SLTU-LIST)
       (UNI-CLEAR.VARIABLES.AS.CONSTANTS)
       (COND
	 ((EVERY #'TERM=SLTU.TUNIFIER SLTU-LIST) (DT-REMPROPS CLAUSE '(SLTU.IGNORED.LITNOS SLTU.VARIABLES))
	  (RETURN T)))))))

(DEFUN TERM=2_ITERATIONS (CLAUSES SUPPORTED.CLAUSES ITERATIONS CONDITION)
						; EDITED: 18-APR-83 11:39:38
						; INPUT:  TWO CLAUSELISTS, AN INTEGER >= 0 AND A
						;         FUNCTION (SEE TERM-2_TERMINATOR)
						; EFFECT: THIS IS THE TOP LEVEL LOOP FOR THE SEARCH
						;         FOR A TERMINATOR SITUATION OR A UNIT
						;         SATISFYING THE CONDITION DEFINED BY THE
						;         CONDITION ARGUMENT.
						; VALUE:  T <=> A REFUTATION WAS FOUND .
						;         (CLAUSE LITNO . TERMLIST) A UNIT
						;         SATISFYING THE CONDITION WAS FOUND,
						;         ELSE NIL.
  (PROG
    ((COUNTER 0) (FASTMODE (ZEROP ITERATIONS)) (ITERATIONS (COND ((ZEROP ITERATIONS) 1) (T ITERATIONS))) READY)
    (DECLARE (SPECIAL COUNTER))
    (WHILE (AND (< COUNTER ITERATIONS) (NULL READY))
      (MEMBER-IF
	#'(LAMBDA (CLAUSE)
	    (COND
	      ((DT-GETPROP CLAUSE 'TEST)
	       (SETQ READY
		     (TERM=EXAMINE.CLAUSE CLAUSE FASTMODE (COND ((MEMBER CLAUSE SUPPORTED.CLAUSES) CONDITION)))))))
	CLAUSES)
      (SETQ COUNTER (1+ COUNTER))
      (COND
	((OPT-GET.OPTION TERM_BREADTH.FIRST)
	 (MAPC
	   #'(LAMBDA (CLAUSE)
	       (DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
		 (MAPC #'(LAMBDA (HINT) (TERM=HINT.ACTIVATE HINT)) (DS-CLAUSE.LIT.GETPROP CLAUSE (1+ RPTN) 'HINT))))
	   CLAUSES))))
    (COND ((EQL T READY) (RETURN READY)) (READY (TERM=LINEARIZATION (CDR READY)) (RETURN (CAR READY))))))

(DEFUN TERM=3_INITIALIZATION (CLAUSES CLAUSE.LITNOS)
						; EDITED: 21-SEP-82 17:45:23
						; INPUT : A LIST OF UNITCLAUSES,
						;         A LIST OF DOTTED PAIRS (CLAUSE . LITNO)
						;         OF LITERALS WHICH SHALL BE REGARDED AS UNITS
						; EFFECT: TERM*SOLUTION AND TERM*NEW.VARIABLES ARE SET
						;         TO NIL.
						;         TERM*NOTUNITCLAUSES IS SET TO A LIST OF
						;         NOTUNITCLAUSES OCCURING IN CLAUSES
						;         TERM*UNITS IS CREATED FOR THE UNITCLAUSES
						;         IN CLAUSES. EVERY UNITCLAUSE GETS A PROPERTY
						;         'UNIT POINTING TO THE UNIT IN TERM*UNITS.
						;         IF THERE IS AN R-LINK BETWEEN TWO
						;         UNITCLAUSES, TERM*SOLUTION IS CALCULATED.
						;         THE MARKS 'TEST AND 'HINT ARE CREATED.
						; VALUE:  UNDEFINED.
  (PROG (UNITCLAUSES) (DT-ABBREVIATION.PUSH) (SETQ TERM*SOLUTION NIL) (SETQ TERM*NEW.VARIABLES NIL)
	(SETQ UNITCLAUSES (TERM=3_CREATE.INITIAL.UNITS CLAUSES CLAUSE.LITNOS))
	(TERM=CREATE.PATTERNS TERM*NOTUNITCLAUSES) (TERM=3_CREATE.MARKS UNITCLAUSES CLAUSE.LITNOS)
	(TERM=SORT.NOTUNITCLAUSES NIL)))

(DEFUN TERM=3_CREATE.INITIAL.UNITS (CLAUSES CLAUSE.LITNOS)
						; EDITED: 21-SEP-82 17:59:57
						; INPUT:  A LIST OF CLAUSES
						;         A LIST OF DOTTED PAIRS (CLAUSE . LITNO)
						;         OF LITERALS WHICH SHALL BE REGARDED AS UNITS
						; EFFECT: TERM*NOTUNITCLAUSES IS SET TO A LIST OF
						;         NOTUNITCLAUSES OCCURING IN CLAUSES.
						;         TERM*UNITS IS CREATED FOR THE UNITCLAUSES
						;         IN CLAUSES.
						;         THE PROPERTY 'UNIT POINTING TO THE UNITS
						;         IN TERM*UNITS IS GENERATED FOR EVERY
						;         UNITCLAUSE.
						; VALUE:  THE LIST OF UNITCLAUSES.
  (SETQ TERM*UNITS NIL) (SETQ TERM*NOTUNITCLAUSES NIL)
  (MAPC
    #'(LAMBDA (CLAUSE.LITNO)
        (PROG ((CLAUSE (CAR CLAUSE.LITNO)) (LITNO (CDR CLAUSE.LITNO)))
	      (PROG
		((PREDICATE (TERM=NORMALIZE.PREDICATE (DS-CLAUSE.PREDICATE CLAUSE LITNO)))
		 (SIGN (TERM=NORMALIZE.SIGN (DS-CLAUSE.SIGN CLAUSE LITNO)))
		 (ARITY (LIST-LENGTH (DS-CLAUSE.TERMLIST CLAUSE LITNO))) PREDICATELIST SIGNLIST UNIT)
		(SETQ PREDICATELIST (ASSOC PREDICATE TERM*UNITS))
		(COND
		  ((NULL PREDICATELIST) (SETQ PREDICATELIST (TERM=UNIT.ADD.NEW.PREDICATELIST PREDICATE ARITY))))
		(SETQ SIGNLIST (TERM=UNIT.PREDICATELIST.SIGNLIST PREDICATELIST SIGN))
		(SETQ UNIT
		      (MI-INSERT.TERMLIST.INTO.TERMSTRUCTURE SIGNLIST (DS-CLAUSE.TERMLIST CLAUSE LITNO)
							     (TERM=UNIT.CREATE.PARTIAL NIL CLAUSE NIL) NIL NIL))
		(DT-PUTPROP CLAUSE 'UNIT UNIT))))
    CLAUSE.LITNOS)
  (PROG (UNITCLAUSES)
	(MAPC
	  #'(LAMBDA (CLAUSE)
	      (COND ((NEQ 1 (DS-CLAUSE.NOLIT CLAUSE)) (SETQ TERM*NOTUNITCLAUSES (CONS CLAUSE TERM*NOTUNITCLAUSES)))
		    (T (SETQ UNITCLAUSES (CONS CLAUSE UNITCLAUSES))
		       (PROG
			 ((PREDICATE (TERM=NORMALIZE.PREDICATE (DS-CLAUSE.PREDICATE CLAUSE 1)))
			  (SIGN (TERM=NORMALIZE.SIGN (DS-CLAUSE.SIGN CLAUSE 1)))
			  (ARITY (LIST-LENGTH (DS-CLAUSE.TERMLIST CLAUSE 1))) PREDICATELIST SIGNLIST UNIT)
			 (SETQ PREDICATELIST (ASSOC PREDICATE TERM*UNITS))
			 (COND
			   ((NULL PREDICATELIST) (SETQ PREDICATELIST (TERM=UNIT.ADD.NEW.PREDICATELIST PREDICATE ARITY))))
			 (SETQ SIGNLIST (TERM=UNIT.PREDICATELIST.SIGNLIST PREDICATELIST SIGN))
			 (SETQ UNIT
			       (MI-INSERT.TERMLIST.INTO.TERMSTRUCTURE SIGNLIST (DS-CLAUSE.TERMLIST CLAUSE 1)
								      (TERM=UNIT.CREATE.PARTIAL NIL CLAUSE NIL) NIL NIL))
			 (DT-PUTPROP CLAUSE 'UNIT UNIT)))))
	  CLAUSES)
	(RETURN UNITCLAUSES)))

(DEFUN TERM=3_CREATE.MARKS (UNITCLAUSES CLAUSE.LITNOS)
						; EDITED: 30-JUN-82 10:56:19
						; INPUT : A LIST OF UNITCLAUSES,
						;         A LIST OF DOTTED PAIRS (CLAUSE . LITNO)
						;         OF LITERALS WHICH SHALL BE REGARDED AS UNITS
						; VALUE : UNDEFINED
						; EFFECT : MARKS ALL LITERALS CONNECTED TO A
						;          UNITCLAUSE WITH A PROPERTY 'HINT,CLAUSES
						;          WITH SUCH LITERALS WITH A PROPERTY 'TEST
  (MAPC
    #'(LAMBDA (CLAUSE)
        (MAPC
          #'(LAMBDA (RLINK)
              (PROG ((OTHERPAR (DS-LINK.OTHERPAR RLINK CLAUSE))) (DT-PUTPROP OTHERPAR 'TEST T)
		    (TERM=CLAUSE.LIT.ADDPROP OTHERPAR (DS-LINK.OTHERLITNO RLINK CLAUSE) 'HINT
					     (TERM=HINT.CREATE (LIST (DT-GETPROP CLAUSE 'UNIT)) RLINK T))))
          (DS-CLAUSE.ALL.LINKS 'R CLAUSE)))
    UNITCLAUSES)
  (UNI-DECLARE.VARIABLES.AS.CONSTANTS TERM*3_VARIABLES.REGARDED.AS.CONSTANTS)
  (MAPC
    #'(LAMBDA (CLAUSE.LITNO)
        (PROG ((CLAUSE (CAR CLAUSE.LITNO)) (LITNO (CDR CLAUSE.LITNO)) OTHERPAR OTHERLITNO PREDICATE)
	      (SETQ PREDICATE (DS-CLAUSE.PREDICATE CLAUSE LITNO))
	      (MAPC
		#'(LAMBDA (RLINK) (SETQ OTHERPAR (DS-LINK.OTHERPAR RLINK CLAUSE))
			  (SETQ OTHERLITNO (DS-LINK.OTHERLITNO RLINK CLAUSE))
			  (COND
			    ((UNI-UNIFY.ATOMS PREDICATE (DS-CLAUSE.TERMLIST CLAUSE LITNO) PREDICATE
					      (DS-CLAUSE.TERMLIST OTHERPAR OTHERLITNO) T)
			     (DT-PUTPROP OTHERPAR 'TEST T)
			     (TERM=CLAUSE.LIT.ADDPROP OTHERPAR OTHERLITNO 'HINT
						      (TERM=HINT.CREATE (LIST (DT-GETPROP CLAUSE 'UNIT)) RLINK T)))))
		(DS-CLAUSE.LINKS 'R CLAUSE LITNO))))
    CLAUSE.LITNOS)
  (UNI-CLEAR.VARIABLES.AS.CONSTANTS))

(DEFUN TERM=3_ITERATIONS (CLAUSES ITERATIONS)
						; EDITED: 30-JUN-82 10:59:28
						; INPUT : A LIST OF CLAUSES AND AN INTEGER
						; VALUE : UNDEFINED
						; EFFECT : THIS IS THE TOP LEVEL LOOP FOR THE SEARCH
						;          FOR TERMINATOR SITUATIONS RESP. FOR
						;          DEDUCING NEW UNITCLAUSES.
  (PROG (READY (COUNTER 0))
	(DECLARE (SPECIAL COUNTER))
	(WHILE (AND (< COUNTER ITERATIONS) (NULL READY))
	  (MEMBER-IF
	    #'(LAMBDA (CLAUSE) (COND ((DT-GETPROP CLAUSE 'TEST) (SETQ READY (TERM=EXAMINE.CLAUSE CLAUSE NIL)))))
	    CLAUSES)
	  (SETQ COUNTER (1+ COUNTER))
	  (COND
	    ((OPT-GET.OPTION TERM_BREADTH.FIRST)
	     (MAPC
	       #'(LAMBDA (CLAUSE)
		   (DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
		     (MAPC #'(LAMBDA (HINT) (TERM=HINT.ACTIVATE HINT)) (DS-CLAUSE.LIT.GETPROP CLAUSE (1+ RPTN) 'HINT))))
	       CLAUSES))))))

(DEFUN TERM=GENERATOR.LITERALS (CLAUSES)
						; EDITED: 16-AUG-83 08:59:25
						; INPUT:  A LIST OF CLAUSES
						; EFFECT: 'GENERATOR LITERALS' ARE MARKED.
						;         A GENERATOR LITERAL L SATISFIES THE
						;         FOLLOWING CONDITIONS:
						;         1. THERE IS AN AUTO-R-LINK CONNECTED TO L
						;         2. WEAK MATCHING WITH THE OTHER SIDE OF THE
						;            AUTO-R-LINK IS POSSIBLE.
						;         EXAMPLE:    NOT P(X) OR P(F(X))
						;                                  L
						; VALUE:  UNDEFINED
  (MAPC
    #'(LAMBDA (CLAUSE)
        (MEMBER-IF
          #'(LAMBDA (LINK)
              (PROG
                ((POSLITNO (DS-LINK.POSLITNO LINK)) (NEGLITNO (DS-LINK.NEGLITNO LINK)) POSTERMLIST NEGTERMLIST VARIABLES
		 MATCHER FOUND)
                (SETQ POSTERMLIST (DS-CLAUSE.TERMLIST CLAUSE POSLITNO))
                (SETQ NEGTERMLIST (DT-TERM.RENAMED (DS-CLAUSE.TERMLIST CLAUSE NEGLITNO)))
                (SETQ VARIABLES (POP NEGTERMLIST))
                (COND
                  ((AND (SETQ MATCHER (CAR (UNI-UNIFY1.TERMLISTS POSTERMLIST NEGTERMLIST VARIABLES)))
			(SSOME #'(LAMBDA (TERM) (AND (CONSP TERM) (DT-TERMLIST.VARIABLES TERM)))
			       #'CDDR
			       (CDR MATCHER)))
		   (DS-CLAUSE.LIT.PUTPROP CLAUSE NEGLITNO 'GENERATOR T) (SETQ FOUND T))
                  ((AND
                     (SETQ MATCHER
			   (CAR
			     (UNI-UNIFY1.TERMLISTS NEGTERMLIST POSTERMLIST
						   (DS-CLAUSE.LIT.VARIABLES CLAUSE POSLITNO))))
                     (SSOME #'(LAMBDA (TERM) (AND (CONSP TERM) (DT-TERMLIST.VARIABLES TERM)))
			    #'CDDR
			    (CDR MATCHER)))
		   (DS-CLAUSE.LIT.PUTPROP CLAUSE POSLITNO 'GENERATOR T) (SETQ FOUND T)))
                (MAPC #'(LAMBDA (VAR) (DT-VARIABLE.DELETE VAR)) VARIABLES) (RETURN FOUND)))
          (DS-CLAUSE.ALL.LINKS 'RIW CLAUSE)))
    CLAUSES))

(DEFUN TERM=NESTING.COUNTER (UNINODES)
						; EDITED: 16-AUG-83 09:53:28
						; INPUT:  THE NODES BELONGING TO A NEW UNIT, DEDUCED
						;         FROM A GENERATOR LITERAL
						; VALUE:  THE NESTING COUNTER FOR THE NEW UNIT.
  (PROG (COUNTER UNITS (SIBLING.COUNTER 0))
	(MAPC #'(LAMBDA (UNINODE) (SETQ UNITS (CONS (TERM=UNINODE.UNIT UNINODE) UNITS))) UNINODES)
	(MAPC
	  #'(LAMBDA (UNIT)
	      (COND
		((AND (SETQ COUNTER (TERM=UNIT.COUNTER UNIT))
		      (EVERY
			#'(LAMBDA (UNIT1)
			    (OR (EQL UNIT UNIT1)
				(MEMBER-IF #'(LAMBDA (NODE) (EQL UNIT1 (TERM=UNINODE.UNIT NODE))) (TERM=UNIT.NODES UNIT))))
			UNITS))
		 (RETURN-FROM TERM=NESTING.COUNTER (1- COUNTER)))
		(COUNTER
		 (MAPC
		   #'(LAMBDA (UNIT1)
		       (COND
			 ((AND (NEQ UNIT UNIT1) (TERM=UNIT.COUNTER UNIT1)) (SETQ SIBLING.COUNTER (1+ SIBLING.COUNTER)))))
		   UNITS)
		 (SETQ COUNTER (- COUNTER SIBLING.COUNTER))
		 (COND ((ZEROP SIBLING.COUNTER))
		       (T (RETURN-FROM TERM=NESTING.COUNTER (COND ((< COUNTER 0) 0) (T COUNTER))))))))
	  UNITS))
  TERM*GENERATOR.LIMIT)

(DEFUN TERM=UNIT.ADD.NEW.PREDICATELIST (PREDICATE ARITY)
						; EDITED: 21-SEP-82 16:59:33
						; INPUT:  A PREDICATE ADDRESS AND A NATURAL NUMBER
						; EFFECT: A NEW PREDICATELIST IS INSERTED INTO
						;         TERM*UNITS. EMPTY SIGNLISTS AND
						;         POSITIONLISTS ARE CREATED.
						; VALUE:  THE NEW PREDICATELIST.
  (PROG (PATTERN +SIGNLIST -SIGNLIST TRANSFORMATION)
	(DODOWN (RPTN ARITY) (SETQ PATTERN (CONS T PATTERN)))
	(SETQ +SIGNLIST (LIST (MI-CREATE.EMPTY.TERMSTRUCTURE PATTERN)))
	(SETQ -SIGNLIST (LIST (MI-CREATE.EMPTY.TERMSTRUCTURE PATTERN)))
	(SETQ TERM*UNITS (CONS (LIST PREDICATE +SIGNLIST -SIGNLIST) TERM*UNITS))
	(COND
	  ((DT-PREDICATE.IS.MARKED SYMMETRIC PREDICATE)
	   (SETQ TRANSFORMATION #'(LAMBDA (TERMLIST) (CONS (SECOND TERMLIST) (CONS (FIRST TERMLIST) (CDDR TERMLIST)))))
	   (RPLACD +SIGNLIST (LIST (CONS (CAR +SIGNLIST) TRANSFORMATION)))
	   (RPLACD -SIGNLIST (LIST (CONS (CAR -SIGNLIST) TRANSFORMATION))))))
  (CAR TERM*UNITS))

(DEFMACRO TERM=UNIT.PREDICATELIST.SIGNLIST (PREDICATELIST SIGN)
						; EDITED: 21-SEP-82 17:20:24
						; INPUT:  A PREDICATELIST AND A NORMALIZED SIGN
						; VALUE:  THE SIGNLIST BELONGING TO SIGN
  `(COND ((EQL ,SIGN '+) (CAADR ,PREDICATELIST)) (T (CAADDR ,PREDICATELIST))))

(DEFMACRO TERM=UNIT.PREDICATELIST.TRANSFORMATIONS (PREDICATELIST SIGN)
						; EDITED:  7-MAR-83 15:16:00
						; INPUT:   PREDICATELIST IS
						;          (PREDICATE (+UNITS . TRANSFORMATIONS)
						;                     (-UNITS . TRANSFORMATIONS)
						;          SIGN IS EITHER + OR -
						; VALUE:   THE TRANSFORMATIONS BELONGING TO +UNITS
						;          IF SIGN = + , ELSE THE TRANSFORMATIONS
						;          BELONGING TO -UNITS
  `(COND ((EQL ,SIGN '+) (CDADR ,PREDICATELIST)) (T (CDADDR ,PREDICATELIST))))

(DEFMACRO TERM=UNIT.CREATE.PARTIAL (UNIFIER CLAUSE NODES)
						; EDITED:  7-MAR-83 15:21:38
						; INPUT:   A T-UNIFIER, A CLAUSE AND A LIST OF
						;          UNIFIERNODES.
						; VALUE:   THE PART OF THE DATASTRUCTURE UNITS
						;          CONSITING OF UNIFIER, CLAUSE AND NODES.
  `(CONS ,UNIFIER (CONS ,CLAUSE (CONS NIL ,NODES))))

(DEFMACRO TERM=UNIT.MAKE.TERMLIST (UNIT)
						; EDITED: 22-SEP-82 18:03:33
						; INPUT:  A UNIT
						; VALUE:  ITS TERMLIST IN STANDARD FORM
  `(MI-TERMLIST.STANDARD ,UNIT))

(DEFMACRO TERM=UNIT.SET.AUXILIARY (UNIT VALUE)
						; EDITED: 23-SEP-82 13:53:30
						; INPUT:  A UNIT OF TERM*UNITS AND AN S-EXPRESSION
						; EFFECT: THE AUXILIARY CELL OF UNIT IS SET TO VALUE.
						; VALUE:  UNDEFINED.
  `(MI-TERMLIST.PUT.AUXILIARY ,UNIT ,VALUE))

(DEFMACRO TERM=UNIT.AUXILIARY (UNIT)
						; EDITED: 23-SEP-82 13:53:30
						; INPUT:  A UNIT OF TERM*UNITS
						; VALUE:  THE AUXILIARY CELL OF UNIT
  `(MI-TERMLIST.AUXILIARY ,UNIT))

(DEFMACRO TERM=UNIT.UNIFIER (UNIT)
						; EDITED: 30-JUN-82 09:33:01
						; INPUT : A UNIT
						; VALUE : ITS UNIFIER
  `(CAR (MI-TERMLIST.PROPERTY ,UNIT)))

(DEFMACRO TERM=UNIT.CLAUSE (UNIT)
						; EDITED: 30-JUN-82 09:40:11
						; INPUT : A UNIT
						; VALUE : ITS CLAUSE
  `(SECOND (MI-TERMLIST.PROPERTY ,UNIT)))

(DEFMACRO TERM=UNIT.NODES (UNIT)
						; EDITED: 30-JUN-82 09:34:27
						; INPUT : A UNIT
						; VALUE : THE LIST OF ITS UNINODES
  `(CDDDR (MI-TERMLIST.PROPERTY ,UNIT)))

(DEFMACRO TERM=UNIT.COUNTER (UNIT)
						; EDITED: 25-MAR-83 13:00:01
						; INPUT:  A UNIT
						; VALUE:  IT'S SKOLEM MARK
  `(THIRD (MI-TERMLIST.PROPERTY ,UNIT)))

(DEFMACRO TERM=UNIT.SET.COUNTER (UNIT MARK)
						; EDITED: 25-MAR-83 13:01:24
						; INPUT:  A UNIT AND AN S-EXPRESSION
						; EFFECT: THE MARK IS FIXED TO THE SKOLEM MARK CELL
						;         OF UNIT.
						; VALUE:  UNDEFINED.
  `(RPLACA (CDDR (MI-TERMLIST.PROPERTY ,UNIT)) ,MARK))

(DEFMACRO TERM=UNIT.DELETE.NODES (UNIT)
						; EDITED: 30-JUN-82 09:42:06
						; INPUT : A UNIT
						; VALUE : UNDEFINED
						; EFFECT : THE UNINODES OF UNIT ARE DELETED
  `(RPLACD (CDDR (MI-TERMLIST.PROPERTY ,UNIT)) NIL))

(DEFMACRO TERM=CREATE.PSEUDOUNIT (UNIFIER CLAUSE NODES)
						; EDITED:  7-MAR-83 15:06:13
						; INPUT:   A T-UNIFIER, A CLAUSE AND A LIST OF
						;          UNIFIERNODES.
						; VALUE:   A UNIT WITH TERMLIST = NIL
  `(MI-INSERT.TERMLIST.INTO.TERMSTRUCTURE (MI-CREATE.EMPTY.TERMSTRUCTURE NIL) NIL
					  (TERM=UNIT.CREATE.PARTIAL ,UNIFIER ,CLAUSE ,NODES) NIL NIL))

(DEFMACRO TERM=HINT.CREATE (UNITS LINK &OPTIONAL ACTIVE.PASSIVE)
						; EDITED: 30-JUN-82 09:47:14
						; INPUT : A LINK, SOME UNITS AND A FLAG
						; VALUE : THE CORRESPONDING HINT,THAT MEANS A
						;         DOTTED PAIR WITH ELEMENTS UNIT AND LINK
  `(CONS ,UNITS (CONS ,LINK ,ACTIVE.PASSIVE)))

(DEFMACRO TERM=HINT.UNITS (HINT)
						; EDITED: 30-JUN-82 09:49:31
						; INPUT : A HINT
						; VALUE : ITS UNIT
  `(CAR ,HINT))

(DEFMACRO TERM=HINT.LINK (HINT)
						; EDITED: 30-JUN-82 09:50:49
						; INPUT : A HINT
						; VALUE : ITS LINK
  `(SECOND ,HINT))

(DEFMACRO TERM=HINT.IS.ACTIVE (HINT)
						; INPUT:  A HINT
						; VALUE:  T IF IT IS ACTIVE, ELSE NIL.
  `(CDDR ,HINT))

(DEFMACRO TERM=HINT.ACTIVATE (HINT)
						; INPUT:  A HINT
						; EFFECT: THE HINT IS ACTIVATED.
  `(RPLACD (CDR ,HINT) T))

(DEFMACRO TERM=UNINODE.CREATE (UNI LINK UNIT &OPTIONAL IS.NEW)
						; EDITED: 30-JUN-82 09:53:12
						; INPUT : UNIFIER LINK UNIT AND A BOOLEAN VARIABLE
						;         IS.NEW
						; VALUE : THE CORRESPONDING UNINODE,THAT MEANS A LIST
						;         WITH TOP LEVEL ELEMENTS UNI LINK AND THE
						;         DOTTED PAIR (UNIT.IS.NEW)
  `(CONS ,UNI (CONS ,LINK (CONS ,UNIT ,IS.NEW))))

(DEFMACRO TERM=UNINODE.UNIFIER (UNINODE)
						; EDITED: 30-JUN-82 09:57:28
						; INPUT : A UNINODE
						; VALUE : ITS UNIFIER
  `(CAR ,UNINODE))

(DEFMACRO TERM=UNINODE.LINK (UNINODE)
						; EDITED: 30-JUN-82 09:58:48
						; INPUT : A UNINODE
						; VALUE : ITS LINK
  `(SECOND ,UNINODE))

(DEFMACRO TERM=UNINODE.UNIT (UNINODE)
						; EDITED: 30-JUN-82 10:00:06
						; INPUT : A UNINODE
						; VALUE : ITS UNIT
  `(THIRD ,UNINODE))

(DEFMACRO TERM=UNINODE.PUT.UNIT (UNINODE UNIT)
						; EDITED: 15. 9. 1984
						; INPUT:  A UNINODE AND AN S-EXPRESSION
						; EFFECT: UNIT IS SET INTO THE UNIT-CELL OF UNINODE
						; VALUE:  UNDEFINED.
  `(RPLACA (CDDR ,UNINODE) ,UNIT))

(DEFMACRO TERM=UNINODE.IS.NEW (UNINODE)
						; EDITED: 30-JUN-82 10:01:40
						; INPUT : A UNINODE
						; VALUE : ITS IS.NEW FLAG
  `(CDDDR ,UNINODE))

(DEFMACRO TERM=UNINODE.MARK.OLD (UNINODE)
						; EDITED: 30-JUN-82 10:03:38
						; INPUT : A UNINODE
						; VALUE : UNDEFINED
						; EFFECT : SETS THE NEW.FLAG OF UNINODE NIL
  `(RPLACD (CDDR ,UNINODE) NIL))

(DEFMACRO TERM=TULIST.CREATE (LITNOS &OPTIONAL TRACE.UNIFIERS)
						; EDITED:  8-APR-83 14:22:31
						; INPUT:   A LIST OF LITERAL NUMBERS AND
						;          TRACE.UNIFIERS.
						; VALUE:   THE TULIST.
  (DECLARE (IGNORE TRACE.UNIFIERS))
  `(CONS (SORT (COPY-TREE ,LITNOS) #'ALPHORDER) ,TRACE.UNIFIERS))

(DEFMACRO TERM=TULIST.LITNOS (TULIST)
						; EDITED:  8-APR-83 14:24:33
						; INPUT:   A TULIST
						; VALUE:   THE LITERAL NUMBERS.
  `(CAR ,TULIST))

(DEFMACRO TERM=TULIST.TRACE.UNIFIERS (TULIST)
						; EDITED:  8-APR-83 14:24:33
						; INPUT:   A TULIST
						; VALUE:   THE TRACE.UNIFIERS.
  `(CDR ,TULIST))

(DEFMACRO TERM=TULIST.IS (TULIST)
						; EDITED:  8-APR-83 14:24:33
						; INPUT:   A LIST
						; VALUE:   T <=> IT IS A TULIST.
  `(EVERY #'NUMBERP (TERM=TULIST.LITNOS ,TULIST)))

(DEFMACRO TERM=SLTU.LINK.SUBSTITUTION (SLTU)
						; EDITED: 15-APR-83 11:09:34
						; INPUT:  AN SLTU (MI-TERMSTRUCTURE)
						; VALUE:  THE (LINK . SUBSTITUTION) CELL
  `(CAR (MI-TERMLIST.PROPERTY (CAR (MI-TERMSTRUCTURE.TERMLISTS ,SLTU)))))

(DEFMACRO TERM=SLTU.TUNIFIER (SLTU)
						; EDITED: 15-APR-83 11:09:34
						; INPUT:  AN SLTU (MI-TERMSTRUCTURE)
						; VALUE:  THE TRACE UNIFIER (T-REPRESENTATION)
  `(SECOND (MI-TERMLIST.PROPERTY (CAR (MI-TERMSTRUCTURE.TERMLISTS ,SLTU)))))

(DEFMACRO TERM=SLTU.UNINODES (SLTU)
						; EDITED: 15-APR-83 11:09:34
						; INPUT:  AN SLTU (MI-TERMSTRUCTURE)
						; VALUE:  THE UNINODES
  `(CDDR (MI-TERMLIST.PROPERTY (CAR (MI-TERMSTRUCTURE.TERMLISTS ,SLTU)))))

(DEFMACRO TERM=SLTU.PUT.TUNIFIER (SLTU TUNIFIER)
						; EDITED: 15-APR-83 11:12:56
						; INPUT:  AN SLTU (MI-TERMSTRUCTURE) AND A T-UNIFIER
						; EFFECT: THE T-UNIFIER IS INSERTED
						; VALUE:  UNDEFINED
  `(RPLACA (CDR (MI-TERMLIST.PROPERTY (CAR (MI-TERMSTRUCTURE.TERMLISTS ,SLTU)))) ,TUNIFIER))

(DEFMACRO TERM=SLTU.PUT.UNINODES (SLTU UNINODES)
						; EDITED: 15-APR-83 11:12:56
						; INPUT:  AN SLTU (MI-TERMSTRUCTURE) AND UNINODES
						; EFFECT: THE UNINODES ARE INSERTED
						; VALUE:  UNDEFINED
  `(RPLACD (CDR (MI-TERMLIST.PROPERTY (CAR (MI-TERMSTRUCTURE.TERMLISTS ,SLTU)))) ,UNINODES))

(DEFUN TERM=STANDARD.UNIFIER (TUNIFIER CLAUSE)
						; EDITED: 22-JUL-82 14:10:16
						; INPUT:  AN UNFIER IN TERMINATOR NORMAL FORM AND A
						;         CLAUSE ADDRESS.
						; EFFECT: THE UNIFIER IS TRANSFORMED INTO THE
						;         STANDARD FORM.
						; VALUE:  THE TRANSFORMED UNIFIER.
  (PROG (STANDARD.UNIFIER)
	(MAPC #'(LAMBDA (VAR TERM) (COND ((NEQ VAR TERM) (SETQ STANDARD.UNIFIER (CONS VAR (CONS TERM STANDARD.UNIFIER))))))
	      (DS-CLAUSE.VARIABLES CLAUSE) TUNIFIER)
	(RETURN STANDARD.UNIFIER)))

(DEFMACRO TERM=APPLY.UNIFIER (TUNIFIER TERMLIST VARIABLES)
						; EDITED: 22-JUL-82 12:57:27
						; INPUT:  AN UNIFIER IN T-FORM.
						;         AN ARBITRARY TERMLIST AND THE LIST OF
						;         VARIABLES REPRESENTING THE OTHER PART OF
						;         THE UNIFIER.
						; VALUE:  A COPY OF THE TERMLIST CHANGED BY THE
						;         UNIFIER.
  `(SUBPAIR ,TUNIFIER ,VARIABLES ,TERMLIST))

(DEFUN TERM=CLAUSE.LIT.ADDPROP (CLAUSE LITNO IND VALUE)
						; EDITED: 30-JUN-82 10:53:07
						; INPUT CLAUSE LITERALNUMBER INDICATOR VALUE
						; VALUE : UNDEFINED
						; EFFECT : ANALOGEOUS TO LISP FUNCTION ADDPROP FOR
						;          THE LITNO'TH LITERAL OF CLAUSE
  (DS-CLAUSE.LIT.PUTPROP CLAUSE LITNO IND (CONS VALUE (DS-CLAUSE.LIT.GETPROP CLAUSE LITNO IND))))

(DEFMACRO TERM=NORMALIZE.SIGN (SIGN)
						; EDITED: 21-SEP-82 17:17:41
						; INPUT:  A DATASTRUCTURE OBJECT SIGN
						; VALUE:  A NORMALIZED REPRESENTATION OF SIGN: + OR -
  `(COND ((DS-SIGN.IS.POSITIVE ,SIGN) '+) (T '-)))

(DEFMACRO TERM=NORMALIZE.PREDICATE (PREDICATE)
						; EDITED: 22-SEP-82 17:56:42
						; INPUT:  A PREDICATE ADDRESS
						; VALUE:  IF PREDICATE IS AN EQUALITY PREDICATE,
						;         THE FIRST EQUALITY PREDICATE ADDRESS, ELSE
						;         PREDICATE.
  `(LET ((PREDICATE ,PREDICATE))
     (COND ((DT-PREDICATE.IS.EQUALITY ,PREDICATE) (CAR (DT-PREDICATE.EQUALITIES))) (T PREDICATE))))

(DEFUN TERM=NORMALIZE.UNIT (UNIT TERMLIST CLAUSE LITNO)
						; EDITED: 28-MAR-83 17:15:41
						; INPUT:  A UNIT AND ITS TERMLIST
						;         CLAUSE-LITNO IS THE PARENTLITERAL OF UNIT
						; EFFECT: IF THERE EXITS A NORMALIZING LITERAL WHICH
						;         NORMALIZES THIS UNIT, IT IS NORMALIZED,
						;         INSERTED INTO THE SIGNLISTS AND THE HINTS
						;         ARE CREATED.
						; VALUE:  T IF IT IS NORMALIZED,
						;         ELSE NIL.
						; REMARK: PRELIMINARY IMPLEMENTATION
  (PROG (RESULT)
	(MAPC
	  #'(LAMBDA (CLAUSE.LITNO)
	      (PROG
		((NORM.CLAUSE (CAR CLAUSE.LITNO)) (NORM.LITNO (CDR CLAUSE.LITNO)) LINK UNINODES VARIABLES LITNOS
		 TRACE.UNIFIERS OTHERLITNO)
		(SETQ VARIABLES (DS-CLAUSE.VARIABLES NORM.CLAUSE))
		(COND
		  ((AND
		     (SETQ LINK
			   (CAR
			     (MEMBER-IF
			       #'(LAMBDA (LINK)
				   (AND (EQL NORM.CLAUSE (DS-LINK.OTHERPAR LINK CLAUSE))
					(NEQ NORM.LITNO (SETQ OTHERLITNO (DS-LINK.OTHERLITNO LINK CLAUSE)))))
			       (DS-CLAUSE.LINKS 'R CLAUSE LITNO))))
		     (SETQ UNINODES
			   (MAPCAR
			     #'(LAMBDA (UNIFIER)
				 (TERM=UNINODE.CREATE (TERM=UNIFIER.NORMALIZE UNIFIER VARIABLES) LINK UNIT T))
			     (UNI-UNIFY.ATOMS (DS-CLAUSE.PREDICATE CLAUSE LITNO) TERMLIST
					      (DS-CLAUSE.PREDICATE NORM.CLAUSE OTHERLITNO)
					      (DS-CLAUSE.TERMLIST NORM.CLAUSE OTHERLITNO)))))
		   (SETQ TRACE.UNIFIERS
			 (TERM=CREATE.INITIAL.TRUNIFIERS UNINODES
							 (DS-CLAUSE.LIT.GETPROP NORM.CLAUSE OTHERLITNO 'PATTERN)))
		   (COND
		     ((SETQ TRACE.UNIFIERS
			    (TERM=MERGE TRACE.UNIFIERS (DS-CLAUSE.LIT.GETPROP NORM.CLAUSE NORM.LITNO 'TRACE.UNIFIERS) NIL
					(MAPCAR #'(LAMBDA (X) T) VARIABLES)))
		      (DODOWN (RPTN (DS-CLAUSE.NOLIT NORM.CLAUSE))
			(COND
			  ((AND (NEQ (1+ RPTN) OTHERLITNO) (NEQ (1+ RPTN) NORM.LITNO)) (SETQ LITNOS (CONS (1+ RPTN) LITNOS)))))
		      (COND ((CDR LITNOS)
			     (C ONLY 3-LITERAL CLAUSES ARE POSSIBLE *)
			     )
			    (T (TERM=CREATE.UNITS NORM.CLAUSE VARIABLES (CAR LITNOS) TRACE.UNIFIERS)
			       (DS-CLAUSE.ALL.LIT.REMPROP NORM.CLAUSE 'HINT) (SETQ RESULT T)))))))))
	  TERM*NORMALIZING.LITERALS)
	(RETURN RESULT)))

(DEFUN TERM=NO.NORMALIZING.UNITS (CLAUSE LITNO UNITS)
						; EDITED: 28-MAR-83 16:19:32
						; INPUT:  A CLAUSE, A LITERALNUMBER AND A LIST OF
						;         UNITS CREATED FROM CLAUSE-LITNO
						; EFFECT: ALL UNITS WHICH ARE COMPLEMENTARY INSTANCES
						;         OF NORMALIZING LITERALS ARE INSERTED INTO
						;         THE TRACE-UNIFIERS OF THIS LITERALS
						; VALUE:  THE SUBSET OF UNITS WHICH ARE NOT INSERTED
						;         IN SUCH A LIST.
  (PROG (CLL)
	(MAPCAR
	  #'(LAMBDA (CLAUSE.LITNO)
	      (LET ((NORM.CLAUSE (CAR CLAUSE.LITNO)) (NORM.LITNO (CDR CLAUSE.LITNO)) LINK)
		(WHEN (SETQ LINK (CAR (MEMBER-IF #'(LAMBDA (LINK)
						     (AND (EQL NORM.CLAUSE (DS-LINK.OTHERPAR LINK CLAUSE))
							  (EQL NORM.LITNO (DS-LINK.OTHERLITNO LINK CLAUSE))))
						 (DS-CLAUSE.LINKS 'R CLAUSE LITNO))))
		  (SETQ CLL (CONS (LIST NORM.CLAUSE NORM.LITNO LINK) CLL)))))
	  TERM*NORMALIZING.LITERALS)
	(COND
	  (CLL
	   (RETURN
	     (REMOVE-IF-NOT
	       #'(LAMBDA (UNIT)
		   (NOTANY
		     #'(LAMBDA (CLL)
			 (PROG
			   ((NORM.CLAUSE (CAR CLL)) (NORM.LITNO (SECOND CLL)) (LINK (THIRD CLL)) UNINODES VARIABLES TRACE.UNIFIERS)
			   (SETQ VARIABLES (DS-CLAUSE.VARIABLES NORM.CLAUSE))
			   (SETQ TRACE.UNIFIERS (DS-CLAUSE.LIT.GETPROP NORM.CLAUSE NORM.LITNO 'TRACE.UNIFIERS))
			   (COND
			     ((SETQ UNINODES
				    (MAPCAR
				      #'(LAMBDA (UNIFIER)
					  (TERM=UNINODE.CREATE (TERM=UNIFIER.NORMALIZE UNIFIER VARIABLES) LINK UNIT T))
				      (UNI-UNIFY1.ATOMS (DS-CLAUSE.PREDICATE NORM.CLAUSE NORM.LITNO)
							(DS-CLAUSE.TERMLIST NORM.CLAUSE NORM.LITNO) (DS-CLAUSE.PREDICATE CLAUSE LITNO)
							(TERM=UNIT.MAKE.TERMLIST UNIT) T)))
			      (MAPC
				#'(LAMBDA (NODE)
				    (MI-INSERT.TERMLIST.INTO.TERMSTRUCTURE TRACE.UNIFIERS
									   (TERM=UNINODE.UNIFIER NODE) (LIST NODE) NIL NIL))
				UNINODES)
			      (RETURN T)))))
		     CLL))
	       UNITS)))
	  (T (RETURN UNITS)))))


(DEFVAR TERM*UNITS NIL)

(DEFVAR TERM*SOLUTION NIL)

(DEFVAR TERM*NEW.VARIABLES NIL)

(DEFVAR TERM*NOTUNITCLAUSES NIL)

(DEFVAR TERM*UNITCOUNTER NIL)

(DEFVAR TERM*UNITCOUNTER.PARTIAL NIL)

(DEFVAR TERM*UNITCOUNTER.TOTAL NIL)

(DEFVAR TERM*UNIFIERCOUNTER.PARTIAL NIL)

(DEFVAR TERM*UNIFIERCOUNTER.TOTAL NIL)

(DEFVAR TERM*FILE NIL)

(DEFVAR TERM*TERMINAL NIL)

(DEFVAR TERM*PRINT.UNITS NIL)

(DEFVAR TERM*INSTANCETEST NIL)

(DEFVAR TERM*GENERATOR.LIMIT NIL)

(DEFVAR TERM*NORMALIZING.LITERALS NIL)

(DEFVAR TERM*3_PSEUDOCONSTANTS NIL)

(DEFVAR TERM*3_VARIABLES.REGARDED.AS.CONSTANTS NIL)

(DEFVAR TERM*3_RENAMING NIL)


(DEFSTRUCT (IND.SIMPL.UNIT (:TYPE LIST) :CONC-NAME) IND.SIMPLIFIER UNIT SIMP.TERMLIST ACCESSORS UNINODE LINK)

(DEFSTRUCT (DIRECT.SIMPL (:TYPE LIST) :CONC-NAME) CLAUSE FROM.LITNO TO.LITNO REST.LITNOS ACCESSORS LINK)

(DEFSTRUCT (IND.SIMPL (:TYPE LIST) :CONC-NAME) CLAUSE SIMP.LITNO FROM.LITNO FROM.ACCESSORS TO.LITNO TO.ACCESSORS
	   REST.LITNOS LINK)



(DEFVAR TERM*SIMPLIFICATION NIL)

(DEFUN TERM=SIMPLIFY.UNIT (UNIT PARENTCLAUSE PARENTLITNO)
  (COND
    ((AND UNIT TERM*SIMPLIFICATION)
     (PROG ((NEW.UNIT (TERM=DIRECT.SIMPLIFY.UNIT UNIT PARENTCLAUSE PARENTLITNO T)) RENAMEFLAG)
	   (SETQ RENAMEFLAG (EQL UNIT NEW.UNIT)) (SETQ UNIT NEW.UNIT)
	   (COND ((NULL NEW.UNIT) (RETURN NIL))
		 ((SETQ NEW.UNIT (TERM=INDIRECT.SIMPLIFY.UNIT UNIT PARENTCLAUSE PARENTLITNO RENAMEFLAG)))
		 ((NULL NEW.UNIT) (RETURN NIL)))
	   (WHILE (NEQ NEW.UNIT UNIT) (SETQ RENAMEFLAG NIL) (SETQ UNIT NEW.UNIT)
		  (SETQ NEW.UNIT (TERM=INDIRECT.SIMPLIFY.UNIT NEW.UNIT PARENTCLAUSE PARENTLITNO NIL)))
	   (TERM=INDIRECT.SIMPLIFIER.UNIT NEW.UNIT PARENTCLAUSE PARENTLITNO RENAMEFLAG) (RETURN NEW.UNIT)))
    (T UNIT)))

(DEFUN TERM=ACCESS.FCT (SUBTERM SUPERTERM-LIST)
  (PROG (ACCESS.FCT (COUNTER 1))
	(DECLARE (SPECIAL ACCESS.FCT))
	(MAPC
	  #'(LAMBDA (TERM)
	      (COND ((EQUAL TERM SUBTERM) (SETQ ACCESS.FCT (CONS (LIST COUNTER) ACCESS.FCT)))
		    ((CONSP TERM) (TERM=EXTRACT.ACCESS.FCT SUBTERM TERM (LIST COUNTER))))
	      (SETQ COUNTER (1+ COUNTER)))
	  SUPERTERM-LIST)
	(RETURN ACCESS.FCT)))

(DEFUN TERM=EXTRACT.ACCESS.FCT (SUBTERM SUPERTERM CURRENT.FCT)
  (DECLARE (SPECIAL ACCESS.FCT))
  (PROG ((COUNTER 1))
	(MAPC
	  #'(LAMBDA (TERM)
	      (COND ((EQUAL TERM SUBTERM) (SETQ ACCESS.FCT (CONS (NCONC1 (COPY-TREE CURRENT.FCT) COUNTER) ACCESS.FCT)))
		    ((CONSP TERM) (TERM=EXTRACT.ACCESS.FCT SUBTERM TERM (NCONC1 (COPY-TREE CURRENT.FCT) COUNTER))))
	      (SETQ COUNTER (1+ COUNTER)))
	  (CDR SUPERTERM))))

(DEFUN TERM=RECOGNIZE.SIMPLIFIER (CLAUSES)
  (COND (TERM*SIMPLIFICATION
	 (PROG (POSLITNO NEGLITNO POSTERMLIST NEGTERMLIST ACCESSORS LINKS ACCESSOR1 ACCESSOR2 REST.LITNOS SIMP.TERMLIST)
	       (MAPC #'(LAMBDA (CLAUSE)
			 (MEMBER-IF
			   #'(LAMBDA (LINK)
			       (COND ((DS-LINK.RULE LINK) NIL)
				     (T (SETQ POSLITNO (DS-LINK.POSLITNO LINK)) (SETQ NEGLITNO (DS-LINK.NEGLITNO LINK))
					(SETQ POSTERMLIST (DS-CLAUSE.TERMLIST CLAUSE POSLITNO))
					(SETQ NEGTERMLIST (DS-CLAUSE.TERMLIST CLAUSE NEGLITNO))
					(COND ((CDR POSTERMLIST) NIL)
					      (T (SETQ REST.LITNOS NIL)
						 (DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
						   (COND
						     ((AND (NEQ (1+ RPTN) POSLITNO) (NEQ (1+ RPTN) NEGLITNO))
						      (SETQ REST.LITNOS (CONS (1+ RPTN) REST.LITNOS)))))
						 (COND
						   ((SETQ ACCESSORS (TERM=ACCESS.FCT (CAR POSTERMLIST) NEGTERMLIST))
						    (MAPC
						      #'(LAMBDA (LINK)
							  (COND ((DS-LINK.RULE LINK) NIL)
								(T
								 (TERM=CLAUSE.LIT.ADDPROP
								   (DS-LINK.OTHERPAR LINK CLAUSE)
								   (DS-LINK.OTHERLITNO LINK CLAUSE NEGLITNO) 'DIRECT.SIMPLIFIERS
								   (MAKE-DIRECT.SIMPL :CLAUSE CLAUSE :FROM.LITNO NEGLITNO :TO.LITNO
										      POSLITNO :REST.LITNOS REST.LITNOS
										      :ACCESSORS ACCESSORS :LINK LINK)))))
						      (DS-CLAUSE.LINKS '(RIW R) CLAUSE NEGLITNO))
						    T)
						   ((SETQ ACCESSORS (TERM=ACCESS.FCT (CAR NEGTERMLIST) POSTERMLIST))
						    (MAPC
						      #'(LAMBDA (LINK)
							  (COND ((DS-LINK.RULE LINK) NIL)
								(T
								 (TERM=CLAUSE.LIT.ADDPROP
								   (DS-LINK.OTHERPAR LINK CLAUSE)
								   (DS-LINK.OTHERLITNO LINK CLAUSE POSLITNO) 'DIRECT.SIMPLIFIERS
								   (MAKE-DIRECT.SIMPL :CLAUSE CLAUSE :FROM.LITNO POSLITNO :TO.LITNO
										      NEGLITNO :REST.LITNOS REST.LITNOS
										      :ACCESSORS ACCESSORS :LINK LINK))))
							  T)
						      (DS-CLAUSE.LINKS '(RIW R) CLAUSE POSLITNO)))))))))
			   (SETQ LINKS (DS-CLAUSE.ALL.LINKS 'RIW CLAUSE)))
			 (COND
			   ((> (DS-CLAUSE.NOLIT CLAUSE) 2)
			    (MEMBER-IF
			      #'(LAMBDA (LINK)
				  (COND ((DS-LINK.RULE LINK) NIL)
					(T (SETQ POSLITNO (DS-LINK.POSLITNO LINK)) (SETQ NEGLITNO (DS-LINK.NEGLITNO LINK))
					   (SETQ POSTERMLIST (DS-CLAUSE.TERMLIST CLAUSE POSLITNO))
					   (SETQ NEGTERMLIST (DS-CLAUSE.TERMLIST CLAUSE NEGLITNO))
					   (COND ((CDR POSTERMLIST) NIL)
						 (T
						  (DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
						    (COND
						      ((AND (NEQ (1+ RPTN) POSLITNO) (NEQ (1+ RPTN) NEGLITNO))
						       (SETQ SIMP.TERMLIST (DS-CLAUSE.TERMLIST CLAUSE (1+ RPTN)))
						       (COND
							 ((AND
							    (SETQ ACCESSOR1 (TERM=ACCESS.FCT (CAR POSTERMLIST) SIMP.TERMLIST))
							    (SETQ ACCESSOR2 (TERM=ACCESS.FCT (CAR NEGTERMLIST) SIMP.TERMLIST)))
							  (SETQ REST.LITNOS NIL) (SETQ ACCESSORS (1+ RPTN))
							  (DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
							    (COND
							      ((AND (NEQ (1+ RPTN) POSLITNO) (NEQ (1+ RPTN) NEGLITNO)
								    (NEQ (1+ RPTN) ACCESSORS))
							       (SETQ REST.LITNOS (CONS (1+ RPTN) REST.LITNOS)))))
							  (MAPC
							    #'(LAMBDA (LINK)
								(COND ((DS-LINK.RULE LINK) NIL)
								      (T
								       (TERM=CLAUSE.LIT.ADDPROP
									 (DS-LINK.OTHERPAR LINK CLAUSE)
									 (DS-LINK.OTHERLITNO LINK CLAUSE POSLITNO)
									 'INDIRECT.SIMPLIFIER.CLAUSES
									 (MAKE-IND.SIMPL :CLAUSE CLAUSE :SIMP.LITNO (1+ RPTN)
											 :FROM.LITNO POSLITNO
											 :FROM.ACCESSORS ACCESSOR1 :TO.LITNO
											 NEGLITNO :TO.ACCESSORS ACCESSOR2
											 :REST.LITNOS
											 REST.LITNOS :LINK LINK)))))
							    (DS-CLAUSE.LINKS '(RIW R) CLAUSE POSLITNO))
							  (MAPC
							    #'(LAMBDA (LINK)
								(COND ((DS-LINK.RULE LINK) NIL)
								      (T
								       (TERM=CLAUSE.LIT.ADDPROP
									 (DS-LINK.OTHERPAR LINK CLAUSE)
									 (DS-LINK.OTHERLITNO LINK CLAUSE NEGLITNO)
									 'INDIRECT.SIMPLIFIER.CLAUSES
									 (MAKE-IND.SIMPL :CLAUSE CLAUSE :SIMP.LITNO (1+ RPTN)
											 :FROM.LITNO NEGLITNO
											 :FROM.ACCESSORS ACCESSOR2 :TO.LITNO
											 POSLITNO :TO.ACCESSORS ACCESSOR1
											 :REST.LITNOS
											 REST.LITNOS :LINK LINK)))))
							    (DS-CLAUSE.LINKS '(RIW R) CLAUSE NEGLITNO))
							  T))))))))))
			      LINKS))))
		     CLAUSES)))))

(DEFUN TERM=INDIRECT.SIMPLIFIER.UNIT (UNIT PARENTCLAUSE PARENTLITNO RENAMEFLAG)
  (MAPC
    #'(LAMBDA (SIMPLIFIER)
        (PROG
          ((CLAUSE (IND.SIMPL-CLAUSE SIMPLIFIER)) (FROM.LITNO (IND.SIMPL-FROM.LITNO SIMPLIFIER))
	   (TERMLIST (TERM=RENAME (TERM=UNIT.MAKE.TERMLIST UNIT) RENAMEFLAG)) FROM.TERM TO.TERM FROM.VARS TO.VARS VARS
	   ACCESSORS UNINODE UNIFIERS)
          (COND
            ((SETQ UNIFIERS
		   (UNI-UNIFY.TERMLISTS TERMLIST (DS-CLAUSE.TERMLIST CLAUSE (IND.SIMPL-SIMP.LITNO SIMPLIFIER))))
	     (SETQ UNINODE
		   (TERM=UNINODE.CREATE (TERM=UNIFIER.NORMALIZE (CAR UNIFIERS) (DS-CLAUSE.VARIABLES CLAUSE))
					(IND.SIMPL-LINK SIMPLIFIER) UNIT T))
	     (MAPC
	       #'(LAMBDA (FROM.ACCESSOR)
		   (MAPC
		     #'(LAMBDA (TO.ACCESSOR) (SETQ FROM.TERM (DT-ACCESS FROM.ACCESSOR TERMLIST))
			       (SETQ TO.TERM (DT-ACCESS TO.ACCESSOR TERMLIST))
			       (COND
				 ((NOT (EQUAL FROM.TERM TO.TERM)) (SETQ FROM.VARS (DT-TERMLIST.VARIABLES FROM.TERM))
				  (SETQ TO.VARS (DT-TERMLIST.VARIABLES TO.TERM))
				  (SETQ VARS (INTERSECTION FROM.VARS TO.VARS))
				  (COND
				    ((SET-DIFFERENCE FROM.VARS TO.VARS)
				     (SETQ TO.TERM
					   (TERM=DIRECT.SIMPLIFY.TERM TO.TERM VARS PARENTCLAUSE PARENTLITNO))
				     (SETQ TERMLIST
					   (DT-REPLACE.TERM.IN.TERMLIST TO.TERM TO.ACCESSOR (COPY-TREE TERMLIST)))))
				  (COND
				    ((AND (TERM=SIMP.UNIT.IS.NOT.SUBSUMED TERMLIST PARENTCLAUSE PARENTLITNO)
					  (SETQ ACCESSORS (TERM=ACCESS.FCT TO.TERM (LIST FROM.TERM))))
				     (MAPC
				       #'(LAMBDA (LINK)
					   (COND ((DS-LINK.RULE LINK) NIL)
						 (T
						  (TERM=CLAUSE.LIT.ADDPROP (DS-LINK.OTHERPAR LINK CLAUSE)
									   (DS-LINK.OTHERLITNO LINK CLAUSE FROM.LITNO)
									   'INDIRECT.SIMPLIFIER.UNITS
									   (MAKE-IND.SIMPL.UNIT :IND.SIMPLIFIER SIMPLIFIER
												:UNIT UNIT
												:SIMP.TERMLIST TERMLIST
												:ACCESSORS ACCESSORS
												:UNINODE UNINODE
												:LINK LINK)))))
				       (DS-CLAUSE.LINKS '(RIW R) CLAUSE FROM.LITNO)))))))
		     (IND.SIMPL-TO.ACCESSORS SIMPLIFIER)))
	       (IND.SIMPL-FROM.ACCESSORS SIMPLIFIER))))))
    (DS-CLAUSE.LIT.GETPROP PARENTCLAUSE PARENTLITNO 'INDIRECT.SIMPLIFIER.CLAUSES)))

(DEFUN TERM=DIRECT.SIMPLIFY.TERM (TERM VARIABLES PARENTCLAUSE PARENTLITNO)
  (MEMBER-IF
    #'(LAMBDA (SIMPLIFIER)
	(LET ((CLAUSE (DIRECT.SIMPL-CLAUSE SIMPLIFIER)) (FROM.LITNO (DIRECT.SIMPL-FROM.LITNO SIMPLIFIER))
	      (TO.LITNO (DIRECT.SIMPL-TO.LITNO SIMPLIFIER)) (REST.LITNOS (DIRECT.SIMPL-REST.LITNOS SIMPLIFIER)) UNIFIERS
	      C-TERMLIST U-TERMLIST C0-TERMLIST U0-TERMLIST VARS)
	  (DECLARE (SPECIAL C-TERMLIST U-TERMLIST C0-TERMLIST U0-TERMLIST UNIFIERS CLAUSE VARS))
	  (SETQ VARS (DS-CLAUSE.VARIABLES CLAUSE))
	  (MEMBER-IF #'(LAMBDA (ACCESSOR)
			 (UNI-DECLARE.VARIABLES.AS.CONSTANTS (UNION VARIABLES
								    (DT-TERMLIST.VARIABLES (DT-ACCESS ACCESSOR (LIST TERM)))))
			 (SETQ C0-TERMLIST (DS-CLAUSE.TERMLIST CLAUSE FROM.LITNO)) (SETQ U0-TERMLIST (LIST TERM))
			 (SETQ UNIFIERS (IF REST.LITNOS
					    (CARTESIAN.LOOP (MAPCAR #'(LAMBDA (LITNO)
									(CAR (DS-CLAUSE.LIT.GETPROP CLAUSE LITNO 'UNINODE)))
								    REST.LITNOS)
							    #'(LAMBDA (UNINODES)
								(DECLARE (SPECIAL UNINODES ))
								(SETQ C-TERMLIST C0-TERMLIST)
								(SETQ U-TERMLIST U0-TERMLIST)
								(MAPC #'(LAMBDA (UNINODES)
									  (SETQ C-TERMLIST (APPEND VARS (COPY-LIST C-TERMLIST)))
									  (SETQ U-TERMLIST
										(APPEND (TERM=UNINODE.UNIFIER (CAR UNINODES))
											(COPY-LIST U-TERMLIST))))
								      UNINODES)
								(WHEN (SETQ UNIFIERS (UNI-UNIFY.TERMLISTS C-TERMLIST U-TERMLIST))
								  (THROW 'CARTESIAN.LOOP UNIFIERS))))
					    (UNI-UNIFY.TERMLISTS C0-TERMLIST U0-TERMLIST)))
			 (WHEN UNIFIERS
			   (UNI-CLEAR.VARIABLES.AS.CONSTANTS)
			   (SETQ TERM (TERM=DIRECT.SIMPLIFY.TERM (UNI-APPLY.SUBSTITUTION (CAR UNIFIERS)
											 (CAR (DS-CLAUSE.TERMLIST CLAUSE TO.LITNO))
											 T)
								 VARIABLES PARENTCLAUSE PARENTLITNO)))
			 (UNI-CLEAR.VARIABLES.AS.CONSTANTS))
		     (DIRECT.SIMPL-ACCESSORS SIMPLIFIER))))
    (DS-CLAUSE.LIT.GETPROP PARENTCLAUSE PARENTLITNO 'DIRECT.SIMPLIFIERS))
  TERM)

(DEFUN TERM=DIRECT.SIMPLIFY.UNIT (UNIT* PARENTCLAUSE* PARENTLITNO* RENAMEFLAG)
  (COND
    (UNIT*
     (MEMBER-IF
       #'(LAMBDA (SIMPLIFIER)
	   (PROG
	     ((UNIT UNIT*)
	      (PARENTCLAUSE PARENTCLAUSE*)
	      (PARENTLITNO PARENTLITNO*)
	      (CLAUSE (DIRECT.SIMPL-CLAUSE SIMPLIFIER)) (FROM.LITNO (DIRECT.SIMPL-FROM.LITNO SIMPLIFIER))
	      (TO.LITNO (DIRECT.SIMPL-TO.LITNO SIMPLIFIER)) (REST.LITNOS (DIRECT.SIMPL-REST.LITNOS SIMPLIFIER))
	      (LINK (DIRECT.SIMPL-LINK SIMPLIFIER))
	      (TERMLIST (TERM=RENAME (TERM=UNIT.MAKE.TERMLIST UNIT*) RENAMEFLAG))
	      (SIGNLIST
		(TERM=UNIT.PREDICATELIST.SIGNLIST
		  (ASSOC (TERM=NORMALIZE.PREDICATE (DS-CLAUSE.PREDICATE PARENTCLAUSE* PARENTLITNO*)
						   ) TERM*UNITS)
		  (TERM=NORMALIZE.SIGN (DS-CLAUSE.SIGN PARENTCLAUSE* PARENTLITNO*)
				       )))
	      UNIFIERS C-TERMLIST U-TERMLIST NEW.UNIT C0-TERMLIST U0-TERMLIST VARIABLES)
	     (DECLARE
	       (SPECIAL C-TERMLIST U-TERMLIST CLAUSE UNIFIERS NEW.UNIT LINK UNIT PARENTCLAUSE PARENTLITNO
			SIGNLIST C0-TERMLIST U0-TERMLIST VARIABLES TO.LITNO))
	     (SETQ VARIABLES (DS-CLAUSE.VARIABLES CLAUSE))
	     (MEMBER-IF
	       #'(LAMBDA (ACCESSOR) (UNI-CONSTANTIFY (DT-ACCESS ACCESSOR TERMLIST))
			 (SETQ C0-TERMLIST (DS-CLAUSE.TERMLIST CLAUSE FROM.LITNO)) (SETQ U0-TERMLIST TERMLIST)
			 (COND
			   (REST.LITNOS
			    (CARTESIAN.LOOP
			      (MAPCAR
				#'(LAMBDA (LITNO) (CAR (DS-CLAUSE.LIT.GETPROP CLAUSE LITNO 'UNINODE))) REST.LITNOS)
			      #'(LAMBDA (UNINODES) (SETQ C-TERMLIST C0-TERMLIST) (SETQ U-TERMLIST U0-TERMLIST)
					(MAPC
					  #'(LAMBDA (UNINODES) (SETQ C-TERMLIST (APPEND VARIABLES (COPY-LIST C-TERMLIST)))
						    (SETQ U-TERMLIST
							  (APPEND (TERM=UNINODE.UNIFIER (CAR UNINODES)) (COPY-LIST U-TERMLIST))))
					  UNINODES)
					(COND
					  ((SETQ UNIFIERS (UNI-UNIFY.TERMLISTS C-TERMLIST U-TERMLIST))
					   (UNI-CLEAR.VARIABLES.AS.CONSTANTS)
					   (SETQ NEW.UNIT
						 (MI-INSERT.TERMLIST.INTO.TERMSTRUCTURE
						   SIGNLIST
						   (UNI-APPLY.SUBSTITUTION (CAR UNIFIERS)
									   (DS-CLAUSE.TERMLIST CLAUSE TO.LITNO) T)
						   (TERM=UNIT.CREATE.PARTIAL
						     (SETQ UNIFIERS
							   (TERM=UNIFIER.NORMALIZE (CAR UNIFIERS) (DS-CLAUSE.VARIABLES CLAUSE)))
						     CLAUSE (CONS (TERM=UNINODE.CREATE UNIFIERS LINK UNIT T)
								  (MAPCAR UNINODES #'CAR)))
						   NIL T))
					   (TERM=SIMPLIFICATION.PRINT UNIT NEW.UNIT "dir.")
					   (TERM=TEST.UNIT NEW.UNIT PARENTCLAUSE PARENTLITNO)
					   (RETURN-FROM TERM=DIRECT.SIMPLIFY.UNIT
					     (TERM=DIRECT.SIMPLIFY.UNIT NEW.UNIT PARENTCLAUSE PARENTLITNO NIL)))))))
			   ((SETQ UNIFIERS (UNI-UNIFY.TERMLISTS C0-TERMLIST U0-TERMLIST))
			    (UNI-CLEAR.VARIABLES.AS.CONSTANTS)
			    (SETQ NEW.UNIT
				  (MI-INSERT.TERMLIST.INTO.TERMSTRUCTURE
				    SIGNLIST
				    (UNI-APPLY.SUBSTITUTION (CAR UNIFIERS) (DS-CLAUSE.TERMLIST CLAUSE TO.LITNO) T)
				    (TERM=UNIT.CREATE.PARTIAL
				      (SETQ UNIFIERS
					    (TERM=UNIFIER.NORMALIZE (CAR UNIFIERS) (DS-CLAUSE.VARIABLES CLAUSE)))
				      CLAUSE (LIST (TERM=UNINODE.CREATE UNIFIERS LINK UNIT T)))
				    NIL T))
			    (TERM=SIMPLIFICATION.PRINT UNIT NEW.UNIT "dir.")
			    (TERM=TEST.UNIT NEW.UNIT PARENTCLAUSE PARENTLITNO)
			    (RETURN-FROM TERM=DIRECT.SIMPLIFY.UNIT
			      (TERM=DIRECT.SIMPLIFY.UNIT NEW.UNIT PARENTCLAUSE PARENTLITNO NIL)))))
	       (DIRECT.SIMPL-ACCESSORS SIMPLIFIER))))
       (DS-CLAUSE.LIT.GETPROP PARENTCLAUSE* PARENTLITNO* 'DIRECT.SIMPLIFIERS))
     (UNI-CLEAR.VARIABLES.AS.CONSTANTS) UNIT*)
    ))

(DEFUN TERM=INDIRECT.SIMPLIFY.UNIT (UNIT* PARENTCLAUSE* PARENTLITNO* RENAMEFLAG)
  (COND
    (UNIT*
     (MEMBER-IF
       #'(LAMBDA (SIMPLIFIER)
	   (DECLARE (SPECIAL SIMPLIFIER))
	   (PROG
	     ((UNIT UNIT*)
	      (PARENTCLAUSE PARENTCLAUSE*)
	      (PARENTLITNO PARENTLITNO*)
	      (IND.SIMPL (IND.SIMPL.UNIT-IND.SIMPLIFIER SIMPLIFIER))
	      (TERMLIST (TERM=RENAME (TERM=UNIT.MAKE.TERMLIST UNIT*) RENAMEFLAG))
	      (UNINODE (IND.SIMPL.UNIT-UNINODE SIMPLIFIER))
	      (SIGNLIST
		(TERM=UNIT.PREDICATELIST.SIGNLIST
		  (ASSOC (TERM=NORMALIZE.PREDICATE (DS-CLAUSE.PREDICATE PARENTCLAUSE* PARENTLITNO*)
						   ) TERM*UNITS)
		  (TERM=NORMALIZE.SIGN (DS-CLAUSE.SIGN PARENTCLAUSE* PARENTLITNO*)
				       )))
	      CLAUSE FROM.LITNO TO.LITNO REST.LITNOS NEW.TERMLIST NEW.UNINODE UNIFIERS NEW.UNIT
	      C0-TERMLIST U0-TERMLIST)
	     (DECLARE
	       (SPECIAL C-TERMLIST U-TERMLIST CLAUSE UNIFIERS NEW.TERMLIST NEW.UNINODE UNIT C0-TERMLIST
			U0-TERMLIST PARENTCLAUSE PARENTLITNO SIGNLIST UNINODE NEW.UNIT TO.LITNO))
	     (SETQ CLAUSE (IND.SIMPL-CLAUSE IND.SIMPL))
	     (SETQ FROM.LITNO (IND.SIMPL-FROM.LITNO IND.SIMPL)) (SETQ TO.LITNO (IND.SIMPL-TO.LITNO IND.SIMPL))
	     (SETQ REST.LITNOS (IND.SIMPL-REST.LITNOS IND.SIMPL))
	     (MEMBER-IF
	       #'(LAMBDA (ACCESSOR) (UNI-CONSTANTIFY (DT-ACCESS ACCESSOR TERMLIST))
			 (SETQ C0-TERMLIST
			       (APPEND (DS-CLAUSE.TERMLIST CLAUSE FROM.LITNO) (COPY-LIST (DS-CLAUSE.VARIABLES CLAUSE))))
			 (SETQ U0-TERMLIST (APPEND TERMLIST (COPY-LIST (TERM=UNINODE.UNIFIER UNINODE))))
			 (COND
			   (REST.LITNOS
			    (CARTESIAN.LOOP
			      (MAPCAR
				#'(LAMBDA (LITNO) (CAR (DS-CLAUSE.LIT.GETPROP CLAUSE LITNO 'UNINODE))) REST.LITNOS)
			      #'(LAMBDA (UNINODES) (SETQ C-TERMLIST C0-TERMLIST) (SETQ U-TERMLIST U0-TERMLIST)
					(MAPC
					  #'(LAMBDA (UNINODES)
					      (SETQ C-TERMLIST (APPEND (DS-CLAUSE.VARIABLES CLAUSE) (COPY-LIST C-TERMLIST)))
					      (SETQ U-TERMLIST
						    (APPEND (TERM=UNINODE.UNIFIER (CAR UNINODES)) (COPY-LIST U-TERMLIST))))
					  UNINODES)
					(COND
					  ((SETQ UNIFIERS (UNI-UNIFY.TERMLISTS C-TERMLIST U-TERMLIST))
					   (SETQ NEW.TERMLIST
						 (UNI-APPLY.SUBSTITUTION (CAR UNIFIERS) (DS-CLAUSE.TERMLIST CLAUSE TO.LITNO) T))
					   (SETQ NEW.UNINODE
						 (TERM=UNINODE.CREATE
						   (SETQ UNIFIERS
							 (TERM=UNIFIER.NORMALIZE (CAR UNIFIERS) (DS-CLAUSE.VARIABLES CLAUSE)))
						   (IND.SIMPL.UNIT-LINK SIMPLIFIER) UNIT T))
					   (UNI-CLEAR.VARIABLES.AS.CONSTANTS)
					   (SETQ NEW.UNIT
						 (MI-INSERT.TERMLIST.INTO.TERMSTRUCTURE
						   SIGNLIST NEW.TERMLIST
						   (TERM=UNIT.CREATE.PARTIAL UNIFIERS CLAUSE
									     (CONS UNINODE
										   (CONS NEW.UNINODE (MAPCAR UNINODES #'CAR))))
						   NIL T))
					   (TERM=SIMPLIFICATION.PRINT UNIT NEW.UNIT "ind.")
					   (TERM=TEST.UNIT NEW.UNIT PARENTCLAUSE PARENTLITNO)
					   (SETQ UNIT
						 (TERM=DIRECT.SIMPLIFY.UNIT NEW.UNIT PARENTCLAUSE PARENTLITNO NIL))
					   (RETURN-FROM TERM=INDIRECT.SIMPLIFY.UNIT UNIT))))))
			   ((SETQ UNIFIERS (UNI-UNIFY.TERMLISTS C0-TERMLIST U0-TERMLIST))
			    (SETQ NEW.TERMLIST
				  (UNI-APPLY.SUBSTITUTION (CAR UNIFIERS) (DS-CLAUSE.TERMLIST CLAUSE TO.LITNO) T))
			    (SETQ NEW.UNINODE
				  (TERM=UNINODE.CREATE
				    (SETQ UNIFIERS
					  (TERM=UNIFIER.NORMALIZE (CAR UNIFIERS) (DS-CLAUSE.VARIABLES CLAUSE)))
				    (IND.SIMPL.UNIT-LINK SIMPLIFIER) UNIT T))
			    (UNI-CLEAR.VARIABLES.AS.CONSTANTS)
			    (SETQ NEW.UNIT
				  (MI-INSERT.TERMLIST.INTO.TERMSTRUCTURE SIGNLIST NEW.TERMLIST
									 (TERM=UNIT.CREATE.PARTIAL UNIFIERS CLAUSE
												   (LIST UNINODE NEW.UNINODE)) NIL T))
			    (TERM=SIMPLIFICATION.PRINT UNIT NEW.UNIT "ind.")
			    (TERM=TEST.UNIT NEW.UNIT PARENTCLAUSE PARENTLITNO)
			    (SETQ UNIT (TERM=DIRECT.SIMPLIFY.UNIT NEW.UNIT PARENTCLAUSE PARENTLITNO NIL))
			    (RETURN-FROM TERM=INDIRECT.SIMPLIFY.UNIT UNIT))))
	       (IND.SIMPL.UNIT-ACCESSORS SIMPLIFIER))))
       (DS-CLAUSE.LIT.GETPROP PARENTCLAUSE* PARENTLITNO* 'INDIRECT.SIMPLIFIER.UNITS))
     (UNI-CLEAR.VARIABLES.AS.CONSTANTS) UNIT*)
    ))

(DEFUN TERM=SIMP.UNIT.IS.NOT.SUBSUMED (TERMLIST PARENTCLAUSE PARENTLITNO) (UNI-CONSTANTIFY TERMLIST)
       (PROG1
	 (NOTANY
	   #'(LAMBDA (SIMPLIFIER) (UNI-UNIFY1.TERMLISTS (IND.SIMPL.UNIT-SIMP.TERMLIST SIMPLIFIER) TERMLIST NIL T))
	   (DS-CLAUSE.LIT.GETPROP PARENTCLAUSE PARENTLITNO 'INDIRECT.SIMPLIFIER.UNITS))
	 (UNI-CLEAR.VARIABLES.AS.CONSTANTS)))

(DEFUN TERM=RENAME (EXPR RENAMEFLAG)
  (COND
    (RENAMEFLAG (SETQ EXPR (DT-TERM.RENAMED EXPR)) (SETQ TERM*NEW.VARIABLES (NCONC (CAR EXPR) TERM*NEW.VARIABLES)) (CDR EXPR))
    (T EXPR)))

(DEFUN TERM=TEST.UNIT (UNIT PARENTCLAUSE PARENTLITNO)
  (COND
    (UNIT
     (PROG ((UNIFIER (TERM=UNIT.UNIFIER UNIT)))
	   (MAPC
	     #'(LAMBDA (UNINODE)
		 (COND
		   ((UNI-UNIFY.TERMLISTS UNIFIER (TERM=UNINODE.UNIFIER UNINODE)) (TERM=LINEARIZATION UNIT)
		    (THROW 'TERM=EXAMINE.CLAUSE T))))
	     (CAR (DS-CLAUSE.LIT.GETPROP PARENTCLAUSE PARENTLITNO 'UNINODE)))))))

(DEFVAR TERM*SIMPLIFICATION.PRINT T)

(DEFVAR TERM*SIMPLIFICATION.STREAM T)

(DEFUN TERM=SIMPLIFICATION.PRINT (OLD.UNIT NEW.UNIT COMMENT)
  (WHEN TERM*SIMPLIFICATION.PRINT
    (TERPRI TERM*SIMPLIFICATION.STREAM)(PRINC COMMENT TERM*SIMPLIFICATION.STREAM)
    (PRINC " Simplifcation " TERM*SIMPLIFICATION.STREAM)
    (PRINC (DS-PNAME (TERM=UNIT.MAKE.TERMLIST OLD.UNIT)) TERM*SIMPLIFICATION.STREAM)
    (COND
      (NEW.UNIT (PRINC " => " TERM*SIMPLIFICATION.STREAM)
		(PRINC (DS-PNAME (TERM=UNIT.MAKE.TERMLIST NEW.UNIT)) TERM*SIMPLIFICATION.STREAM))
      (T (PRINC " subsumed." TERM*SIMPLIFICATION.STREAM)))))







