;;; -*- Package: MKRP; Syntax: Common-lisp; Mode: LISP -*-

(IN-PACKAGE "MKRP" :use '("CL"))

(DEFVAR TWO*RULES.MAXLEVEL 0)

(DEFVAR TWO*CONTROL NIL)

(DEFVAR TWO*SUPPRESS.NORULES NIL)

(DEFVAR TWO*DIALOG NIL)

(DEFVAR TWO*EXTERNAL.RULE.COLOURS NIL)

(DEFVAR TWO*INTERNAL.RULE.COLOURS NIL)

(DEFVAR TWO*RULES NIL)

(DEFVAR TWO*ADDED.RULES NIL)

(DEFVAR TWO*EMPTY.CLAUSE.FOUND NIL)

(DEFVAR TWO*NEW.RULES NIL)

(DEFVAR TWO*NORULE.CLAUSES NIL)

(DEFVAR TWO*TR.RULES NIL)

(DEFVAR TWO*TR.NORULES NIL)

(DEFUN TWO-UPDATE.RULES (CLAUSE)
						; EDITED: 30-JAN-84 15:35:06        MW
						; INPUT:  A TWO-LITERAL CLAUSE TO BECOME A NEW RULE.
						; EFFECT: BUILTS CLAUSE INTO THE RULE SET, COMPLETES
						;         IT AND ADDS CORRESPONDING LINKS TO THE
						;         CONNECTION GRAPH.
						; VALUE:  A LIST OF CLAUSES TO BE BUILT INTO THE
						;         CONNECTION GRAPH.
						;         OR: JUST THE EMPTY CLAUSE IF IT WAS FOUND.
  (COND
    ((TWO=ADMISSIBLE.RULE.IS CLAUSE)
     (let (NON.RULE.CLAUSES ADJACENT.CLAUSES)
       (SETQ ADJACENT.CLAUSES (TWO=GET.ADJACENT.CLAUSES.OF CLAUSE))
       (RED-REMOVE.CLAUSE CLAUSE 'RULE)
       (CG-PUSH)
       (CG-RECALL.GRAPH.REGISTER)
       (CG-INSERT.CLAUSE CLAUSE NIL NIL T 'RULE)
       (SETQ NON.RULE.CLAUSES (TWO=COMPLETE.RULES CLAUSE))
       (SETQ NON.RULE.CLAUSES
	     (NCONC NON.RULE.CLAUSES
		    (MAPCAN (FUNCTION (LAMBDA (NEW.RULE) (TWO=FACTORIZE NEW.RULE))) TWO*NEW.RULES)))
       (MAPC #'(LAMBDA (CLAUSE) (CG-REMOVE.CLAUSE CLAUSE "NORULE FROM RULEGRAPH")) NON.RULE.CLAUSES)
       (MAPC #'(LAMBDA (NEW.RULE) (DS-RULE.INSERT NEW.RULE)) TWO*NEW.RULES)
       (CG-FIX T)
       (CG-STORE.GRAPH.REGISTER)
       (CG-POP)
       (SETQ TWO*TR.RULES TWO*NEW.RULES)
       (SETQ TWO*TR.NORULES NON.RULE.CLAUSES)
       (COND
	 (TWO*EMPTY.CLAUSE.FOUND (SETQ TWO*TR.NORULES (LIST TWO*EMPTY.CLAUSE.FOUND))
				 TWO*EMPTY.CLAUSE.FOUND)
	 (T (TWO=UPDATE.GRAPH ADJACENT.CLAUSES NIL TWO*NEW.RULES)
	    (append NON.RULE.CLAUSES
		    (mapcar-not-nil #'(lambda (clause) (if (eql 'equality (two=classify.clause (first clause)))
							   (first clause)
							   nil))
				    two*rules))))))))

(DEFUN TWO-RESET NIL
						; EDITED: 16-FEB-84 20:59:18        MW
						; EFFECT: RESETS THE MODULE TO ITS INITIAL STATE.
						; NOTE:   THE RULE INDUCED CHANGES OF THE DS MODULE
						;         ARE NOT RESET.
  (SETQ TWO*CONTROL (OPT-GET.OPTION TWO_RULES))
  (SETQ TWO*RULES.MAXLEVEL (OPT-GET.OPTION TWO_RULES.MAXLEVEL))
  (SETQ TWO*SUPPRESS.NORULES (OPT-GET.OPTION TWO_SUPPRESS.NORULES))
  (SETQ TWO*RULES NIL)
  (SETQ TWO*ADDED.RULES NIL)
  (SETQ TWO*NEW.RULES NIL)
  (SETQ TWO*TR.RULES NIL)
  (SETQ TWO*TR.NORULES NIL)
  (SETQ TWO*EMPTY.CLAUSE.FOUND NIL)
  (TWO=INIT)
  (CG-PUSH) (CG-STORE.GRAPH.REGISTER) (CG-POP))

(DEFUN TWO-SAVE (FILE)
						; EDITED:  25.1.1984    MW
						; INPUT:   EITHER NIL OR THE NAME OF A FILE WHICH IS
						;          OPEN FOR OUTPUT.
						; EFFECT:  COMPUTES AN S-EXPRESSION REPRESENTING THE
						;          STATE OF THIS MODULE.
						;          IF FILE <> NIL, THE S-EXPRESSION IS WRITTEN
						;          ON FILE, OTHERWISE THE S-EXPRESSION IS THE
						;          VALUE OF THIS FUNCTION.
						; VALUE:   IF FILE = NIL THEN THE COMPUTED
						;          S-EXPRESSION, ELSE NIL.
  (PROG
    ((S-EXPRESSION
       (CONS 'PROGN
	     (MAPCAR (FUNCTION (LAMBDA (ATOM) (LIST 'setq ATOM `',(EVAL ATOM))))
		     '(TWO*RULES.MAXLEVEL TWO*CONTROL TWO*SUPPRESS.NORULES TWO*DIALOG TWO*EXTERNAL.RULE.COLOURS
					  TWO*INTERNAL.RULE.COLOURS TWO*RULES TWO*ADDED.RULES TWO*EMPTY.CLAUSE.FOUND TWO*NEW.RULES
					  TWO*NORULE.CLAUSES TWO*TR.RULES TWO*TR.NORULES)))))
    (COND (FILE (PROGN (PRINC S-EXPRESSION FILE) (TERPRI FILE))) (T (RETURN S-EXPRESSION)))))


(DEFUN TWO=INIT NIL
  (SETQ TWO*EXTERNAL.RULE.COLOURS
    (INTERSECTION (SET-DIFFERENCE (DS-LINK.COLOURS.FOR 'RULES) (DS-LINK.COLOURS.FOR 'AUTOLINKS))
      (DS-LINK.COLOURS.FOR 'INITIAL)))
  (SETQ TWO*INTERNAL.RULE.COLOURS
    (INTERSECTION (INTERSECTION (DS-LINK.COLOURS.FOR 'RULES) (DS-LINK.COLOURS.FOR 'AUTOLINKS))
      (DS-LINK.COLOURS.FOR 'INITIAL))))

(DEFUN TWO-END NIL
						; EDITED: 19-JUN-84 11:37:14        MW
						; INPUT:  NONE.
						; EFFECT: RELEASES ALL DATA RELATED TO THE (TWO-LIT.)
						;         RULE CONNECTION GRAPH.       - HOWEVER
						;         THE RULES ARE PRESERVED AS DATASTRUCTURES.
						; VALUE:  UNDEF'D.
						; NOTE:   CALL ONLY WHEN THE CREATION OF RULES HAS
						;         BEEN COMPLETED.
  (CG-PUSH) (CG-RECALL.GRAPH.REGISTER)
  (MAPC (FUNCTION (LAMBDA (CLAUSE) (CG-REMOVE.CLAUSE CLAUSE "RELEASE OF RULEGRAPH" NIL))) (CG-CLAUSES ALL)) (CG-FIX T)
  (CG-STORE.GRAPH.REGISTER) (CG-POP))

(DEFUN TWO=COMPLETE.RULES (NEW.RULE)
						; EDITED:  1-FEB-84    MW
						; INPUT:   A (TWO-LITERAL) RULE CLAUSE
						; EFFECT:  THE SET OF RULE CLAUSES IS COMPLETED UNTIL
						;          TWO*RULES.MAXLEVEL IS REACHED. SEE ALSO
						;          TWO=ADD.AND.LEVEL.SATURATE. TWO*RULES AND
						;          TWO*ADDED.RULES ARE THEREFORE UPDATED.
						; VALUE:   NEW NON-RULE CLAUSES GENERATED BY THE
						;          COMPLETION ALGORITHM.
  (SETQ TWO*NEW.RULES (LIST NEW.RULE))
  (SETQ TWO*ADDED.RULES (LIST TWO*NEW.RULES)) (SETQ TWO*NORULE.CLAUSES NIL)
  (COND
    ((NOT TWO*RULES) (OP-CONSTRUCT.LITERAL.LINKS NEW.RULE NIL TWO*INTERNAL.RULE.COLOURS NIL 'NORULES)
     (TWO=ADD.AND.LEVEL.SATURATE TWO*RULES.MAXLEVEL))
    (T (TWO=CONSTRUCT.LINKS NEW.RULE (FLATTEN TWO*RULES) 'NORULES) (TWO=ADD.AND.LEVEL.SATURATE TWO*RULES.MAXLEVEL)))
  (SETQ TWO*RULES (TWO=MERGE TWO*RULES TWO*ADDED.RULES)) TWO*NORULE.CLAUSES)

(DEFUN TWO=ADD.AND.LEVEL.SATURATE (LEVEL#)
						; INPUT:  A NONNEGATIVE INTEGER.
						; EFFECT: TO A GIVEN, POSSIBLY EMPTY LIST OF ALREADY
						;         EXISTING RULE CLAUSES TWO*RULES PREVIOUSLY
						;         LEVEL SATURATED UP TO LEVEL#, AND A NEW
						;         LIST TWO*ADDED.RULES CONTAINING AT LEAST ONE
						;         NEW CLAUSE ALL ADDITIONAL LEVEL SATURATION
						;         CLAUSES UP TO LEVEL# ARE COMPUTED.
						;         ACCORDING TO TWO*SUPPRESS.NORULES, THE LAST
						;         LEVEL IS EITHER MADE NORULE CLAUSES OR IT IS
						;         SUPPRESSED. TWO*RULES AND TWO*ADDED.RULES
						;         ARE LISTS OF THE FORMAT:
						;         <(Cc0,1e..Cc0,M1e)..(CcN,1e..CcN,MIe)>
						;         WHERE I IS THE SATURATION LEVEL OF THE
						;         CLAUSE C[I,K].
  (COND ((EQL LEVEL# 0) NIL)
	((EQL LEVEL# 1)
	 (PROG (RESULT)
	       (COND ((EQL LEVEL# TWO*RULES.MAXLEVEL)
		      (COND (TWO*SUPPRESS.NORULES NIL)
			    (T (SETQ RESULT (NCONC (TWO=RESOLVE.ALL (TWO=S 0) (TWO=S.ADDED 0))
						   (TWO=RESOLVE.ALL (TWO=S.ADDED 0) (TWO=S.ADDED 0))))
			       (SETQ TWO*NORULE.CLAUSES (NCONC RESULT TWO*NORULE.CLAUSES)))))
		     (T (TWO=ADD.NEW.LEVEL
			  (TWO=SELECT.RULES
			    (NCONC (TWO=RESOLVE.ALL (TWO=S 0) (TWO=S.ADDED 0))
				   (TWO=RESOLVE.ALL (TWO=S.ADDED 0) (TWO=S.ADDED 0)))))))))
	((NOT TWO*EMPTY.CLAUSE.FOUND) (TWO=ADD.AND.LEVEL.SATURATE (1- LEVEL#))
	 (COND ((EQL LEVEL# TWO*RULES.MAXLEVEL)
		(PROG (RESULT)
		      (COND (TWO*SUPPRESS.NORULES NIL)
			    (T (SETQ RESULT
				     (NCONC (TWO=RESOLVE.ALL (TWO=S (1- LEVEL#)) (TWO=S.ADDED (1- LEVEL#)))
					    (TWO=RESOLVE.ALL (TWO=S.UNIONS (1- (1- LEVEL#))) (TWO=S.ADDED (1- LEVEL#)))
					    (TWO=RESOLVE.ALL (TWO=S.ADDED.UNIONS (1- (1- LEVEL#))) (TWO=S (1- LEVEL#)))
					    (TWO=RESOLVE.ALL (TWO=S.ADDED.UNIONS (1- LEVEL#)) (TWO=S.ADDED (1- LEVEL#)))))
			       (SETQ TWO*NORULE.CLAUSES (NCONC RESULT TWO*NORULE.CLAUSES))))))
	       (T (TWO=ADD.NEW.LEVEL
		    (TWO=SELECT.RULES
		      (NCONC (TWO=RESOLVE.ALL (TWO=S (1- LEVEL#)) (TWO=S.ADDED (1- LEVEL#)))
			     (TWO=RESOLVE.ALL (TWO=S.UNIONS (1- (1- LEVEL#))) (TWO=S.ADDED (1- LEVEL#)))
			     (TWO=RESOLVE.ALL (TWO=S.ADDED.UNIONS (1- (1- LEVEL#))) (TWO=S (1- LEVEL#)))
			     (TWO=RESOLVE.ALL (TWO=S.ADDED.UNIONS (1- LEVEL#)) (TWO=S.ADDED (1- LEVEL#)))))))))))

(DEFUN TWO=ADD.NEW.LEVEL (CLAUSES)
						; EDITED:  4-SEP-84 11:03:06        MW
						; INPUT:  A LIST OF CLAUSES CONSTITUTING A NEW LEVEL.
						; EFFECT: ADD THEM TO TWO*ADDED.RULES.
						; VALUE:  UNDEFINED.
  (COND (CLAUSES (RPLACD (LAST TWO*ADDED.RULES) (LIST CLAUSES)))))

(DEFUN TWO=RESOLVE.ALL (CLIST1 CLIST2)
						; EDITED:  4-SEP-84 10:26:54        MW
						; INPUT:   TWO LIST OF (2L-) CLAUSES
						; EFFECT:  ALL POSSIBLE RESOLUTIONS OF CLAUSES IN
						;          CLIST1 WITH CLIST2.
						; VALUE:   A LIST OF RESOLVENTS.
  (COND ((AND CLIST1 CLIST2)
	 (let (RESOLVENTS LINK)
	   (MAPC #'(LAMBDA (CLAUSE1)
		     (MAPC #'(LAMBDA (CLAUSE2)
			       (COND ((EQL CLAUSE1 CLAUSE2)
				      (when (SETQ LINK (first (DS-CLAUSE.LINKS 'RIW CLAUSE1 1)))
					(SETQ RESOLVENTS (NCONC (TWO=RESOLVE LINK) RESOLVENTS))))
				     (T (DO ((LITNO.CLAUSE1 2 (1- litno.clause1)))
					    ((zerop litno.clause1))
					  (DO ((LITNO.CLAUSE2 2 (1- litno.clause2)))
					      ((zerop litno.clause2))
					    (when (SETQ LINK (TWO=ADJACENT.LITERALS CLAUSE1 LITNO.CLAUSE1 CLAUSE2 LITNO.CLAUSE2))
					      (SETQ RESOLVENTS (NCONC (TWO=RESOLVE LINK) RESOLVENTS))))))))
			   CLIST2))
		 CLIST1)
	   RESOLVENTS))))

(DEFUN TWO=SELECT.RULES (CLAUSES)
						; EDITED: 29-AUG-84 17:31:52        MW
						; INPUT:  A LIST OF CLAUSES.
						; EFFECT: TWO*NORULE.CLAUSES, TWO*NEW.RULES ARE
						;         UPDATED.
						; VALUE:  THE LIST OF RULE CLAUSES OF CLAUSES.
  (REMOVE-IF-NOT #'(LAMBDA (CLAUSE)
		     (CASE (TWO=CLASSIFY.CLAUSE CLAUSE)
		       (RULE (SETQ TWO*NEW.RULES (CONS CLAUSE TWO*NEW.RULES)) T)
		       ((NO2L EQUALITY SIW NOT.PARTIAL) (SETQ TWO*NORULE.CLAUSES (CONS CLAUSE TWO*NORULE.CLAUSES)) NIL)
		       ((TERM.DEPTH MANUAL)
			(if TWO*SUPPRESS.NORULES
			    (CG-REMOVE.CLAUSE CLAUSE "SUPPRESSED NORULE")
			    (SETQ TWO*NORULE.CLAUSES (CONS CLAUSE TWO*NORULE.CLAUSES)))
			NIL)
		       (OTHERWISE (ERROR "ILLEGAL CLAUSE INDICATOR IN TWO=CLASSIFY.CLAUSE:: ~A" NIL))))
		 CLAUSES))

(DEFMACRO TWO=S (N)
						; EDITED:  4-SEP-84 10:01:12        MW
						; INPUT:  NO. OF LEVEL, A NONNEGATIVE INTEGER
						; VALUE:  NTH LEVEL OF EXISTING LEVEL.SATURATED RULES.
  `(CAR (NTHCDR ,N TWO*RULES)))

(DEFMACRO TWO=S.ADDED (N)
						; EDITED:  4-SEP-84 10:01:12        MW
						; INPUT:  NO. OF LEVEL, A NONNEGATIVE INTEGER
						; VALUE:  NTH LEVEL OF NEWLY ADDED LEVEL SATUR. RULES.
  `(CAR (NTHCDR ,N TWO*ADDED.RULES)))

(DEFUN TWO=S.UNIONS (UPPER.BOUND)
						; EDITED:  4-SEP-84 10:07:21        MW
						; INPUT: A NONNEGATIVE INTEGER
						; VALUE: A LIST OF ALREADY EXISTING RULE CLAUSES OF
						;        ALL SATURATION LEVELS UP TO UPPER.BOUND.
  (COND
    (TWO*RULES
     (FLATTEN
       (PROG ((LEVEL# 0))
	     (RETURN
	       (REMOVE-IF-NOT
		 (FUNCTION (LAMBDA (LEVEL)
			     (declare (ignore level))
			     (PROG1 (NOT (> LEVEL# UPPER.BOUND)) (SETQ LEVEL# (1+ LEVEL#))))) TWO*RULES)))))))

(DEFUN TWO=S.ADDED.UNIONS (UPPER.BOUND)
						; EDITED:  4-SEP-84 10:07:21        MW
						; INPUT: A NONNEGATIVE INTEGER
						; VALUE: A LIST OF NEWLY ADDED RULE CLAUSES OF
						;        ALL SATURATION LEVELS UP TO UPPER.BOUND.
  (COND
    (TWO*ADDED.RULES
     (FLATTEN
       (PROG ((LEVEL# 0))
	     (RETURN
	       (REMOVE-IF-NOT
		 (FUNCTION (LAMBDA (LEVEL)
			     (declare (ignore level))
			     (PROG1 (NOT (> LEVEL# UPPER.BOUND))
				    (SETQ LEVEL# (1+ LEVEL#)))))
		 TWO*ADDED.RULES)))))))

(DEFUN TWO=MERGE (LIST1 LIST2)
						; EDITED:  4-SEP-84 14:11:38        MW
						; INPUT:  TWO LISTS OF THE FORM
						;         <(Cc1,1e..Cc1,M1e)..(CcN,MNe)>.
						; VALUE:  A LIST OF THE SAME FORMAT WITH THE ELEMENTS
						;         CcI,Je FROM EACH LIST FOR THE SAME I IN THE
						;         SAME SUBLIST.
  (PROG ((TAIL1 LIST1) (TAIL2 LIST2) RESULT)
	(DODOWN (RPTN (MAX (LIST-LENGTH LIST1) (LIST-LENGTH LIST2)))
	  (PROG1 (SETQ RESULT (NCONC RESULT (LIST (NCONC (CAR TAIL1) (CAR TAIL2)))))
		 (SETQ TAIL1 (CDR TAIL1)) (SETQ TAIL2 (CDR TAIL2))))
	(RETURN RESULT)))

(DEFUN TWO=FACTORIZE (RULE)
						; EDITED: 31-JAN-84 17:34:01        MW
						; INPUT:  A (TWO-LITERAL) RULE CLAUSE.
						; EFFECT: THE FACTORS OF RULE ARE GENERATED AND ADDED
						;         TO THE CONNECTION GRAPH.
						; VALUE:  FACTORS.
  (let (SI_LINKS FACTORS)
    (SETQ SI_LINKS (DS-CLAUSE.LINKS 'SI RULE 1))
    (MAPC
      (FUNCTION
	(LAMBDA (LINK)
	  (PROG ((UNIFIERS (DS-LINK.UNIFIERS LINK)))
		(SETQ FACTORS
		      (NCONC FACTORS
			     (MAPCAR
			       (FUNCTION
				 (LAMBDA (UNIFIER) (PROG (FACTOR) (SETQ FACTOR (OP-FACTORIZE LINK UNIFIER)) (RETURN FACTOR))))
			       UNIFIERS))))))
      SI_LINKS)
    factors))

(DEFUN TWO=RESOLVE (LINK)
						; EDITED:  2-FEB-84 21:17:12        MW
						; INPUT:   LINK OF RESOLVABLE COLOUR.
						; EFFECT:  ALL RESOLUTIONS POSSIBLE WITH LINK.
						; VALUE:   LIST OF RESOLVENTS OF LINK, ELSE NIL.
  (let (RESOLVENTS UNIFIERS UNIFIER RESOLVENT)
    (WHILE (AND (DS-TYPE LINK) (SETQ UNIFIERS (DS-LINK.UNIFIERS LINK)))
      (SETQ UNIFIER (CAR UNIFIERS))
      (SETQ RESOLVENT (OP-RESOLVE LINK UNIFIER))
      (CG-FIX)
      (when (AND RESOLVENT (TWO=REDUCE `(,RESOLVENT ,LINK ,UNIFIER . "RULEGRAPH RESOLUTION")))
	(push RESOLVENT RESOLVENTS)))
    RESOLVENTS))

(DEFUN TWO=REDUCE (RESOLUTION.RESULT)
						; EDITED: 21-FEB-84 23:20:13        MW
						; INPUT: RESOLUTION RESULT AS LIST OF THE FORM
						;        <RESOLVENT> <LINK> (<UNIFIERS>.<REASON>).
						; EFFECT: REDUCTION.
						; VALUE: THE RESOLVENT IF IT STILL EXISTS AFTER
						;        REDUCTION, OTHERWISE NIL.
  (let ((CLAUSE (CAR RESOLUTION.RESULT)))
    (RED-REDUCE.RULE.GRAPH NIL (LIST (CDR RESOLUTION.RESULT)) (LIST CLAUSE) NIL)
    (when (AND (ZEROP (DS-CLAUSE.NOLIT CLAUSE)) (NOT TWO*EMPTY.CLAUSE.FOUND)) (SETQ TWO*EMPTY.CLAUSE.FOUND CLAUSE))
    (CG-FIX)
    (if (MEMBER CLAUSE (CG-CLAUSES ALL)) CLAUSE NIL)))

(DEFUN TWO=GET.ADJACENT.CLAUSES.OF (CLAUSE)
						; EDITED: 14-FEB-84 20:41:37        MW
						; INPUT:  A CLAUSE.
						; VALUE: ALL ADJACENT.CLAUSES OF CLAUSE.
  (let (LINKS CLAUSES)
    (MAPC #'(LAMBDA (COLOUR)
	      (let (NEW)
		(SETQ NEW (DS-CLAUSE.ALL.LINKS COLOUR CLAUSE))
		(COND (NEW (SETQ LINKS (APPEND NEW (copy-list LINKS)))))))
	  TWO*EXTERNAL.RULE.COLOURS)
    (MAPC (FUNCTION (LAMBDA (LINK) (SETQ CLAUSES (INS (DS-LINK.OTHERPAR LINK CLAUSE) CLAUSES)))) LINKS)
    CLAUSES))

(DEFUN TWO=ADJACENT.LITERALS (CL1 LIT1 CL2 LIT2)
						; EDITED: 29-AUG-84 15:33:54        MW
						; INPUT: CLAUSE1, LITNO1, CLAUSE2, LITNO2
						; VALUE: THE R-LINK CONNECTING THE GIVEN LITERALS.
						;        ELSE NIL.
						; NOTE:  IT IS ASSUMED THAT THERE EXISTS ONE AND ONLY
						;        ONE R-LINK CONNECTING TWO LITERALS (WHICH IS
						;        NOT GENERALLY TRUE OUTSIDE THE RULE GRAPH]).
  (let (LINKS)
    (SETQ LINKS (INTERSECTION (DS-CLAUSE.LINKS 'R CL1 LIT1) (DS-CLAUSE.LINKS 'R CL2 LIT2)))
    (if LINKS (first LINKS))))

(DEFUN TWO=CONSTRUCT.LINKS (CLAUSE READY.CLAUSES CLASS)
						; EDITED:  8-FEB-84 23:43:25        MW
						; INPUT:  A CLAUSE, A CLAUSLIST AND A BOOLEAN FLAG.
						; EFFECT: ALL RULE COLOUR LINKS FOR CLAUSES AMONG
						;         THEMSELVES AND TO READY.CLAUSES ARE CREATED.
						;         IF CLASS = 'NORULES, UNIFICATION WITHOUT
						;         RULES ONLY IS ATTEMPTED (STANDARD FOR THE
						;         RULE GRAPH]).
						; VALUE:  UNDEFINED.
  (OP-CONSTRUCT.LITERAL.LINKS CLAUSE NIL TWO*INTERNAL.RULE.COLOURS NIL CLASS)
  (OP-CONSTRUCT.LITERAL.LINKS CLAUSE NIL TWO*EXTERNAL.RULE.COLOURS READY.CLAUSES CLASS))

(DEFUN TWO=UPDATE.GRAPH (CLAUSES READY.CLAUSES RULES)
						; EDITED:  8-FEB-84 23:43:25        MW
						; INPUT:  THREE LISTS OF CLAUSES
						; EFFECT: ALL NOT MISSING LINKS FOR CLAUSES AMONG
						;         THEMSELVES AND TO READY.CLAUSES ARE CREATED
						;         FOR THE GIVEN RULES.
						; VALUE:  UNDEFINED.
  (PROG ((READY.NEW.CLAUSES READY.CLAUSES))
	(MAPC
	  (FUNCTION
	    (LAMBDA (CLAUSE) (OP-CONSTRUCT.LITERAL.LINKS CLAUSE NIL TWO*INTERNAL.RULE.COLOURS NIL RULES)
		    (OP-CONSTRUCT.LITERAL.LINKS CLAUSE NIL TWO*EXTERNAL.RULE.COLOURS READY.NEW.CLAUSES RULES)
		    (SETQ READY.NEW.CLAUSES (CONS CLAUSE READY.NEW.CLAUSES))))
	  CLAUSES)))

(DEFUN TWO=ADMISSIBLE.RULE.IS (CLAUSE)
						; EDITED: 23-NOV-83 19:46:53
						; INPUT:  A CLAUSE.
						; VALUE:  T IF CLAUSE IS A PERMITTED RULE CLAUSE.
  (AND TWO*CONTROL (member (TWO=CLASSIFY.CLAUSE CLAUSE) '(RULE equality))))

(DEFUN TWO=CLASSIFY.CLAUSE (CLAUSE)
						; EDITED: 11-SEP-84 11:43:30        MW
						; INPUT:  A CLAUSE.
						; VALUE:  AN ATOM INDICATING THE CLASSIFICATION OF
						;         CLAUSE, EITHER 'RULE' FOR ADMISSIBLE TWO
						;         LITERAL RULES OR THE REASON FOR CLAUSE
						;         BEING A NON-RULE CLAUSE.
  (COND ((NEQ (DS-CLAUSE.NOLIT CLAUSE) 2) 'NO2L)
	((AND (DT-PREDICATE.EQUALITIES)
	      (OR (MEMBER-IF (FUNCTION (LAMBDA (TERM) (NOT (DT-VARIABLE.IS TERM)))) (DS-CLAUSE.TERMLIST CLAUSE 1))
		  (MEMBER-IF (FUNCTION (LAMBDA (TERM) (NOT (DT-VARIABLE.IS TERM)))) (DS-CLAUSE.TERMLIST CLAUSE 2))
		  (DS-CLAUSE.IS.EQUATION CLAUSE 1) (DS-CLAUSE.IS.EQUATION CLAUSE 2)))
	 'EQUALITY)
	((PROG ((TERM.DEPTH (OPT-GET.OPTION STR_TERM.DEPTH)))
	       (RETURN
		 (COND
		   (TERM.DEPTH
		    (OR (> (DT-TERMLIST.MAXDEPTH (DS-CLAUSE.TERMLIST CLAUSE 1)) TERM.DEPTH)
			(> (DT-TERMLIST.MAXDEPTH (DS-CLAUSE.TERMLIST CLAUSE 2)) TERM.DEPTH))))))
	 'TERM.DEPTH)
	((AND (EQL TWO*CONTROL 'PARTIAL) (TWO=NOT.PARTIAL.IS CLAUSE)) 'NOT.PARTIAL)
	((AND TWO*DIALOG (NOT (TWO=DIALOG CLAUSE))) 'MANUAL) (T 'RULE)))

(DEFUN TWO=ADMISSIBLE.SIW.HAS (CLAUSE)
						; EDITED:  6-MAY-84 13:28:01        MW
						; INPUT:   A TWO-LITERAL CLAUSE WITH SIW-LINK.
						; EFFECT:  CLAUSE IS MARKED WITH ORIENTATION.
						; VALUE:   T IF CLAUSE CAN BE USED AS RULE (E.G.
						;          ASSYMETRY).
  (let (UNI)
    (PROG1 (AND (SETQ UNI (first (UNI-UNIFY.TERMLISTS
				   (UNI-APPLY.SUBSTITUTION (DS-CLAUSE.RENAMING CLAUSE) (DS-CLAUSE.TERMLIST CLAUSE 1) T)
				   (DS-CLAUSE.TERMLIST CLAUSE 2))))
		(UNI-EQUAL.TERMLISTS
		  (UNI-APPLY.SUBSTITUTION UNI (DS-CLAUSE.TERMLIST CLAUSE 1) T)
		  (UNI-APPLY.SUBSTITUTION UNI (UNI-APPLY.SUBSTITUTION (DS-CLAUSE.RENAMING CLAUSE)
								      (DS-CLAUSE.TERMLIST CLAUSE 2)
								      T)
					  NIL)))
	   (DT-PUTPROP CLAUSE 'TWO*ORIENTATION 'NONE))))

(DEFUN TWO=NOT.PARTIAL.IS (CLAUSE)
						; EDITED:  2-JUL-84 18:43:45        MW
						; INPUT:  AN ADMISSIBLE RULE CLAUSE.
						; VALUE:  T   IF CLAUSE HAS SAME SIGNS AND PREDICATES,
						;         OTHERWISE NIL.
  (AND (DT-PREDICATE.ARE.SAME (DS-CLAUSE.PREDICATE CLAUSE 1) (DS-CLAUSE.PREDICATE CLAUSE 2))
       (DS-SIGN.ARE.EQUAL (DS-CLAUSE.SIGN CLAUSE 1) (DS-CLAUSE.SIGN CLAUSE 2))))

(DEFUN TWO=DIALOG (CLAUSE)
						; EDITED:  3-JUL-84 09:09:26        MW
						; INPUT:  AN ADMISSIBLE RULE CLAUSE.
						; VALUE:  T IF CLAUSE IS MANUALLY SELECTED AS RULE.
  (CG-DUMP T
	   `((MESSAGE
	       ,(CONCATENATE 'STRING "?2?2? DO YOU WISH " (PRINC-TO-STRING (DS-PNAME CLAUSE)) " AS TWO-LITERAL RULE?"
			     (PRINC-TO-STRING (CHARACTER 47))))
	     (CLAUSES (LIST ,CLAUSE))))
  (PROG NIL LOOP (PROGN (PRINC "?2?2? REPLY YES//NO (Y//N)") (TERPRI))
	(RETURN (CASE (READ T) ((Y YES) T) ((N NO) NIL) (OTHERWISE (GO LOOP))))))

(DEFUN TWO-LAST.CALL.NEW.RULES NIL
						; EDITED: 19-JUN-84 12:59:11        MW
						; EFFECT: DESTROYS OWN INFORMATION]
						; VALUE:  NEW RULE CLAUSE ACCEPTED AND NEW RULE
						;         CLAUSES GENERATED SINCE THE LAST CALL OF
						;         TWO-UPDATE.RULES.
  (PROG1 TWO*TR.RULES (SETQ TWO*TR.RULES NIL)))

(DEFUN TWO-LAST.CALL.NEW.NORULES NIL
						; EDITED: 19-JUN-84 12:59:11        MW
						; EFFECT: DESTROYS OWN INFORMATION]
						; VALUE:  NEW NORULE CLAUSES GENERATED SINCE THE
						;         LAST CALL OF TWO-UPDATE.RULES.
						;         TWO-UPDATE.RULES.
  (PROG1 TWO*TR.NORULES (SETQ TWO*TR.NORULES NIL)))

