;;; -*- Package: MKRP; Syntax: Common-lisp; Mode: LISP -*-

(IN-PACKAGE "MKRP" :use '("CL"))

(DEFUN RED.SERVICE-BINDING.RENAMES.VARIABLES (VARIABLES)
						; EDITED:  5-JUN-84 19:47:31
						; INPUT:  'VARIABLES' IS A SET OF VARIABLES IN FORM
						;         OF A LIST.
						; EFFECT: -
						; VALUE:  NOT NIL IFF THE CURRENT BINDING IS A
						;         VARIABLE RENAMING FOR THE VARIABLES
						;         'VARIABLES', I.E. |BINDING('VARIABLES')| =
						;         |'VARIABLES'| AND SORT(BINDING(V)) =
						;         SORT(V) FOR V IN 'VARIABLES'.
  (let ((VARSCOD NIL) BIND)
    (EVERY #'(LAMBDA (VAR)
	       (SETQ BIND (DT-VARIABLE.GET.BINDING VAR))
	       (COND ((NOT BIND) (COND ((NOT (MEMBER VAR VARSCOD))
					(SETQ VARSCOD (CONS VAR VARSCOD)))))
		     ((AND (DT-VARIABLE.IS BIND) (NOT (MEMBER BIND VARSCOD))
			   (EQL (DT-TERM.SORT VAR) (DT-TERM.SORT BIND)))
		      (SETQ VARSCOD (CONS BIND VARSCOD)))))
	   VARIABLES)))

(DEFUN RED.SERVICE-NIL.IN.UNIFIERS (UNIFIERS)
						; EDITED:  5-JUN-84 19:53:06
						; INPUT:  A LIST OF UNIFIERS.
						; EFFECT: -
						; VALUE:  NOT NIL IFF ONE OF THE UNIFIERS IS THE
						;         NIL-UNIFIER UNDER THE CURRENT BINDING.
  (MEMBER-IF #'(lambda (uni) (UNI-BINDING.IS.INSTANCE.OF uni)) UNIFIERs))

(DEFUN RED.SERVICE-SWITCHED.UNIFIERS (LINK THISLITNO CLAUSE)
						; EDITED: 10-JUL-84 09:16:52
						; INPUT:  'THISLITNO' IS THE NUMBER OF A LITERAL OF
						;         CLAUSE 'CLAUSE', 'LINK' IS A LINK INCIDENT
						;         WITH THIS LITERAL.
						; EFFECT: -
						; VALUE:  THE LIST OF UNIFIERS OF LINK 'LINK'. THE
						;         UNFIERS ARE COPIED AND DOMAIN AND CODOMAIN
						;         VARIABLES OF THE RENAMING OF CLAUSE
						;         'CLAUSE' ARE SWITCHED IN THE UNIFIERS, IF
						;         'THISLITNO' IS THE POSITIVE
						;         PARENT LITERAL OF LINK 'LINK'
						;         AND COLOUR OF 'LINK' IS A WEAK UNIFIABILITY
						;         COLOUR (WHERE RENAMING IS USED).
						; EXAMPLE:'UNIFIERS':            ((X<-FY)(X<-Y'))
						;         RENAMING OF 'CLAUSE':  (X<-X' Y<-Y')
						;         'VALUE':               ((X'<-FY')(X'<-Y))
  (if (AND (EQL THISLITNO (DS-LINK.POSLITNO LINK))
	   (MEMBER (DS-LINK.COLOUR LINK) (DS-LINK.COLOURS.FOR 'RENAMED)))
      (MAPCAR (FUNCTION (LAMBDA (UNI2) (UNI-SWITCH (DS-CLAUSE.RENAMING CLAUSE) (COPY-TREE UNI2))))
	      (DS-LINK.UNIFIERS LINK))
      (DS-LINK.UNIFIERS LINK)))

(defmacro RED.SERVICE-CLAUSE.LINKS (COLOURS CLAUSE LITNO)
						; EDITED:  5-JUN-84 19:55:12
						; INPUT:  'COLOURS' IS A LIST OF LINK COLOURS, 'LITNO'
						;         IS THE NUMBER OF A LITERAL OF CLAUSE
						;         'CLAUSE'.
						; EFFECT: -
						; VALUE:  THE LIST OF LISTS OF ALL LINKS (L1 ... LN)
						;         OF COLOURS (C1 ... CN) OF LITERAL 'LITNO'
						;         OF CLAUSE 'CLAUSE', SO THAT LINKS OF LIST
						;         LI HAVE THE COLOUR CI.
  (setq colours (macroexpand colours))
  (COND
    ((and (constantp colours)
	  (EVERY (FUNCTION (LAMBDA (COLOUR) (MEMBER COLOUR (DS-LINK.COLOURS.FOR 'ALL)))) (eval COLOURS)))
     (CONS 'LIST
	   (MAPCAR (FUNCTION (LAMBDA (COLOUR) (LIST 'DS-CLAUSE.LINKS (KWOTE COLOUR) CLAUSE LITNO))) (eval COLOURS))))
    (T (list 'RED.SERVICE==CLAUSE.LINKS colours clause litno))))

(defun red.service==clause.links (colours clause litno)
  (MAPCAR (FUNCTION (LAMBDA (COLOUR) (DS-CLAUSE.LINKS COLOUR CLAUSE LITNO))) COLOURS))

(DEFUN RED.SERVICE-OTHERPAR.EXTERNAL (LINK CLAUSE)
						; EDITED:  5-JUN-84 20:00:02
						; INPUT:  LINK 'LINK' IS AN EXTERNAL LINK OF CLAUSE
						;         'CLAUSE'.
						; EFFECT: -
						; VALUE:  THE OTHER PARENT CLAUSE OF LINK 'LINK'.
  (let ((POSPAR (DS-LINK.POSPAR LINK)))
    (COND ((EQL CLAUSE POSPAR) (DS-LINK.NEGPAR LINK))
	  (T POSPAR))))

(DEFUN RED.SERVICE-OTHERLITNO.EXTERNAL (LINK CLAUSE)
						; EDITED:  5-JUN-84 20:00:02
						; INPUT:  LINK 'LINK' IS AN EXTERNAL LINK OF CLAUSE
						;         'CLAUSE'.
						; EFFECT: -
						; VALUE:  THE NUMBER OF LITERAL INCIDENT WITH LINK
						;         'LINK', WHICH IS NOT IN CLAUSE 'CLAUSE'.
  (COND ((EQL CLAUSE (DS-LINK.POSPAR LINK)) (DS-LINK.NEGLITNO LINK))
	(T (DS-LINK.POSLITNO LINK))))

(DEFUN RED.SERVICE-OTHERLITNO.INTERNAL (LINK LITNO)
						; EDITED:  5-JUN-84 20:05:02
						; INPUT:  LINK 'LINK' IS AN INTERNAL LINK OF A
						;         LITERAL WITH NUMBER 'LITNO' IN A CLAUSE.
						; EFFECT: -
						; VALUE:  THE NUMBER OF THE OTHER PARENT LITERAL OF
						;         THE LINK IN THIS CLAUSE.
  (let ((POSLITNO (DS-LINK.POSLITNO LINK)))
    (COND ((EQL LITNO POSLITNO) (DS-LINK.NEGLITNO LINK))
	  (T POSLITNO))))

(DEFUN RED.SERVICE-LINK.NEGPAR (LINK POSPAR.FLAG)
						; EDITED:  5-JUN-84 20:06:14
						; INPUT:  'LINK' IS A LINK, 'POSPAR.FLAG' IS A FLAG.
						; EFFECT: -
						; VALUE:  THE NEGATIVE PARENT CLAUSE OF LINK 'LINK',
						;         IF THE LINK HAS THIS COMPONENT,
						;         THE POSITIVE PARENT CLAUSE, IF NOT AND
						;         'POSPAR.FLAG' IS NOT NIL,
						;         NIL ELSE.
  (COND ((MEMBER (DS-LINK.COLOUR LINK) (DS-LINK.COLOURS.WITH 'NEGPAR)) (DS-LINK.NEGPAR LINK))
	(POSPAR.FLAG (DS-LINK.POSPAR LINK))))

(DEFUN RED.SERVICE-LINK.PARENTS (LINK)
						; EDITED:  5-JUN-84 20:13:24
						; INPUT:  A LINK.
						; EFFECT: -
						; VALUE:  THE LIST OF THE PARENT CLAUSES OF THE LINK.
  (NCONC (LIST (DS-LINK.POSPAR LINK))
	 (COND ((MEMBER (DS-LINK.COLOUR LINK) (DS-LINK.COLOURS.WITH 'NEGPAR)) (LIST (DS-LINK.NEGPAR LINK))))))

(DEFUN RED.SERVICE-LINK.ORIENTED (LINK)
						; EDITED:  5-JUN-84 20:14:55
						; INPUT:  A LINK.
						; EFFECT: -
						; VALUE:  NOT NIL IFF THE LINK HAS AN ORIENTED RULE.
  (DS-RULE.ORIENTED (DS-LINK.RULE LINK) (DS-LINK.COLOUR LINK)))

(DEFUN RED.SERVICE-TRUE.LITERAL.IS (CLAUSE LITNO &optional BINDING)
  (declare (ignore binding))
						; EDITED:  5-JUN-84 20:19:45
						; INPUT:  'LITNO' IS THE NUMBER OF A POSITIVE
						;         LITERAL WITH REFLEXIVE PREDICATE OR A
						;         NEGATIVE LITERAL WITH IRRELEXIVE PREDICATE
						;         IN THE CLAUSE 'CLAUSE'.
						;         'BINDING' IS THE CURRENT BINDING.
						; EFFECT: -
						; VALUE:  'IMPOSSIBLE' IF THE TWO TERMS OF THE
						;         LITERAL CAN NOT BE UNIFIED,
						;         'T' IF THEY ARE EQUAL,
						;         NIL IF THEY ARE NOT EQUAL BUT UNIFIABLE.
  (let* ((TERMLIST (DS-CLAUSE.TERMLIST CLAUSE LITNO))
	 (UNIFIERS (UNI-UNIFY.TERMS (CAR TERMLIST) (SECOND TERMLIST))))
    (COND ((NULL UNIFIERS) 'IMPOSSIBLE)
	  ((RED.SERVICE-NIL.IN.UNIFIERS UNIFIERS) T)
	  (T nil))))

(DEFUN RED.SERVICE-FALSE.LITERAL.IS (CLAUSE LITNO &optional BINDING)
  (declare (ignore binding))
						; EDITED:  5-JUN-84 20:16:37
						; INPUT:  'LITNO' IS THE NUMBER OF A NEGATIVE
						;         LITERAL WITH REFLEXIVE PREDICATE OR A
						;         POSITIVE LITERAL WITH IRREFLEXIVE PREDICATE
						;         IN THE CLAUSE 'CLAUSE'.
						;         'BINDING' IS THE CURRENT BINDING.
						; EFFECT: -
						; VALUE:  'IMPOSSIBLE'  IF THE TWO TERMS OF THE
						;         LITERAL CAN NOT BE UNIFIED,
						;         'T' IF THEY ARE EQUAL,
						;         NIL IF THEY ARE NOT EQUAL BUT UNIFIABLE.
  (let ((TERMLIST (DS-CLAUSE.TERMLIST CLAUSE LITNO)) UNIFIERS)
    (SETQ UNIFIERS (UNI-UNIFY.TERMS (CAR TERMLIST) (SECOND TERMLIST)))
    (COND ((NULL UNIFIERS) 'IMPOSSIBLE)
	  ((RED.SERVICE-NIL.IN.UNIFIERS UNIFIERS) T)
	  (T nil))))

(DEFUN RED.SERVICE-TRUE.CLAUSE (CLAUSE LITNO &optional BINDING)
  (declare (ignore binding))
						; EDITED:  5-JUN-84 20:20:58
						; INPUT:  'LITNO' IS THE NUMBER OF A POSITIVE
						;         LITERAL WITH REFLEXIVE PREDICATE OR A
						;         NEGATIVE LITERAL WITH IRREFLEXIVE PREDICATE
						;         IN THE CLAUSE 'CLAUSE'.
						;         'BINDING' IS THE CURRENT BINDING.
						; EFFECT: -
						; VALUE:  THE REFLEXIVITY OR IRREFLEXIVITY CLAUSE OF
						;         THE PREDICATE OF THE LITERAL, IF ITS TWO
						;         TERMS ARE EQUAL,
						;         NIL ELSE.
  (let ((TERMLIST (DS-CLAUSE.TERMLIST CLAUSE LITNO)))
    (if (UNI-EQUAL.TERMS (CAR TERMLIST) (SECOND TERMLIST) T)
	(DT-PREDICATE.REFL.CLAUSE (DS-CLAUSE.PREDICATE CLAUSE LITNO))
	nil)))

(DEFUN RED.SERVICE-FALSE.CLAUSE (CLAUSE LITNO &optional BINDING)
  (declare (ignore binding))
						; EDITED:  5-JUN-84 20:24:17
						; INPUT:  'LITNO' IS THE NUMBER OF A NEGATIVE
						;         LITERAL WITH REFLEXIVE PREDICATE OR A
						;         POSITIVE LITERAL WITH IRREFLEXIVE PREDICATE
						;         IN THE CLAUSE 'CLAUSE'.
						;         'BINDING' IS THE CURRENT BINDING.
						; EFFECT: -
						; VALUE:  THE REFLEXIVITY OR IRREFLEXIVITY CLAUSE OF
						;         THE PREDICATE OF THE LITERAL, IF ITS TWO
						;         TERMS ARE EQUAL,
						;         NIL ELSE.
  (let ((TERMLIST (DS-CLAUSE.TERMLIST CLAUSE LITNO)))
    (if (UNI-EQUAL.TERMS (CAR TERMLIST) (SECOND TERMLIST) T)
	(DT-PREDICATE.REFL.CLAUSE (DS-CLAUSE.PREDICATE CLAUSE LITNO))
	nil)))

(DEFVAR RED*SERVICE_RENAME.VARIABLES NIL)

(DEFUN RED.SERVICE-RENAME.POINT NIL
						; EDITED:  2-AUG-84 16:57:12
						; INPUT:  -
						; EFFECT: AN EMPTY LIST () IS INSERTED IN THE LIST
						;         OF LIST OF VARIABLES
						;         'RED*SERVICE_RENAME.VARIABLES' AT FIRST
						;         POSITION.
						; VALUE:  NEW 'RED*SERVICE_RENAME.VARIABLES'.
						; REMARK: ALL VARIABLES NEWLY CREATED BY
						;         'RED.SERVICE-RENAME' SINCE THE LAST CALL
						;         OF 'RED.SERVICE-RENAME.POINT' CAN BE
						;         REMOVED BY 'RED.SERVICE-RENAME.DELETE'.
  (SETQ RED*SERVICE_RENAME.VARIABLES (CONS NIL RED*SERVICE_RENAME.VARIABLES)))

(DEFUN RED.SERVICE-RENAME.DELETE ()
						; EDITED:  2-AUG-84 16:54:34
						; INPUT:  -
						; EFFECT: ALL VARIABLES IN THE FIRST LIST OF THE
						;         LISTS OF VARIABLES 'RED*SERVICE_RENAME.-
						;         VARIABLES' WILL BE PHYSICALLY DELETED.
						;         THIS LIST IS REMOVED FROM
						;         'RED*SERVICE_RENAME.VARIABLES'.
						; VALUE:  NEW 'RED*SERVICE_RENAME.VARIABLES'.
  (MAPC #'(lambda (var) (DT-VARIABLE.DELETE var)) (CAR RED*SERVICE_RENAME.VARIABLES))
  (SETQ RED*SERVICE_RENAME.VARIABLES (CDR RED*SERVICE_RENAME.VARIABLES)))

(DEFUN RED.SERVICE-RENAME (UNIFIERS VARIABLES)
						; EDITED:  2-AUG-84 16:50:49
						; INPUT:  'UNIFIERS' IS A LIST OF UNIFIERS.
						;         'VARIABLES' IS A LIST OF VARIABLES.
						; EFFECT: ALL OCCURRENCES OF ELEMENTS OF 'VARIABLES'
						;         IN 'UNIFIERS' ARE CONSISTENTLY REPLACED
						;         BY NEW VARIABLES. THE LATTER ARE INSERTED
						;         INTO THE FIRST SUBLIST OF
						;         'RED*SERVICE_RENAME.VARIABLES'.
						; VALUE:  THE SO MODIFIED UNIFIER LIST.
  (MAPC #'(LAMBDA (VARIABLE)
	    (when (in VARIABLE UNIFIERS)
	      (NSUBST (let ((NEW.VAR (DT-VARIABLE.CREATE (DT-VARIABLE.SORT VARIABLE))))
			(push NEW.VAR (CAR RED*SERVICE_RENAME.VARIABLES))
			NEW.VAR)
		      VARIABLE UNIFIERS)))
	VARIABLES)
  unifiers)



