;;; -*- Package: MKRP; Syntax: Common-lisp; Mode: LISP -*-

(IN-PACKAGE "MKRP" :use '("CL"))

(DEFUN PPR-CODE (CODE code.file)		; edited:  2-oct-84 15:12:51  by cl
						; input :  an s-expression (the proof) and a list
						;          of dotted pairs specifying the version.
						; effect:  cuts down the code into small units and
						;          makes preparatory changes to it.
						; value :  undefined
  (PPR=RESET)
  (let ((HEAD (CAR CODE)) result)
    (PDS-PUT.PROTOCOL.TYPE (first  HEAD))
    (PDS-PUT.PROOF.VERSION (SECOND HEAD))
    (PDS-PUT.RUN.DATE (THIRD HEAD))
    (PDS-PUT.COMMENT (fourth HEAD))
    (sMAPC #'(LAMBDA (CODE.PART)
	       (CASE (CAR CODE.PART)
		 (REFUTATION      (PDS-ADD.SPLITPART (PPR=SPLITPART (CDR CODE.PART) (opt-get.option pr_options)
								    nil code.file)))
		 (AXIOMS.INFIX    (when  (opt-get.option pr_infix.form) (PDS-PUT.AXIOMS.INFIX (SECOND CODE.PART))))
		 (THEOREMS.INFIX  (when  (opt-get.option pr_infix.form)  (PDS-PUT.THEOREMS.INFIX (SECOND CODE.PART))))
		 (AXIOMS.PREFIX   (when  (opt-get.option pr_prefix.form) (PDS-PUT.AXIOMS.PREFIX (SECOND CODE.PART))))
		 (THEOREMS.PREFIX (when  (opt-get.option pr_prefix.form) (PDS-PUT.THEOREMS.PREFIX (SECOND CODE.PART))))
		 (OPTIONS         (when (AND (opt-get.option pr_options) (NEQ (PDS-GET.PROTOCOL.TYPE) 'C&S))
				    (PDS-PUT.OPTIONS (CDR CODE.PART))))
		 (indices         (pds-put.indices (mapcar #'(lambda (l)
							       (cons (pds-clause.pname (cassoc (first l)
											       PPR*OLD.NEW.CLAUSE.ADDRESSES.initial))
								     (rest l)))
							   (first (rest code.part)))))
		 (LINK.COLOURS    )
		 (AXIOMS          (setq result (PPR=AXIOMS.GRAPH (CDR CODE.PART) nil)))
		 (THEOREMS        (PDS-ADD.PROOF.PART (PPR=PROOF.PART (CDR CODE.PART) nil)))
		 (SPLITPARTS      (PDS-PUT.PROTOCOL.TYPE 'C&S))
		 (OTHERWISE       (ERROR "Wrong key word ~S in code file." (CAR CODE.PART)))))
	   #'(lambda (rest.code) (if (eq result 'success) nil (rest rest.code)))
	   (CDR CODE)))
  (ppr=mark.used.operations (pds-get.ax.operations)))

(DEFparameter PPR*NEW.PNAMES T)

(DEFVAR PPR*INITIAL.CLAUSE.NUMBER 1)

(DEFVAR PPR*DEDUCED.CLAUSE.NUMBER 1)

(DEFUN PPR=RESET NIL
						; EDITED:  4-SEP-84 17:00:33       BY CL
						; INPUT :  NONE
						; EFFECT:  SETS ALL THE COMMON VARIABLES TO NIL, AND
						;          RESETS MEMORY AND SYMBOLS
						; VALUE :  UNDEFINED
  
  (SETQ PPR*AXIOMS.O.N.C.ADDRESSES   (LIST NIL))
  (SETQ PPR*OLD.NEW.CLAUSE.ADDRESSES (LIST NIL))
  (SETQ PPR*OLD.NEW.CLAUSE.ADDRESSES.initial (LIST NIL))
						; THE ABOVE VARIABLES ARE AUGMENTED BY PUTASSOC IN THE PROGRAM.
  (SETQ PPR*INITIAL.CLAUSE.NUMBER 1)
  (SETQ PPR*DEDUCED.CLAUSE.NUMBER 1)
  (SETQ PPR*NEW.PNAMES T)			; MAY BE SET TO NIL BY AN OPTION
  (PDS-RESET))

(defun ppr=result.handling (result reason proof.part proof.tree final.clauses)
  (declare (special used.clause.list))
  (COND ((NULL RESULT) NIL)
	((EQL RESULT 'FAILURE) (SETQ PROOF.TREE NIL) (SETQ USED.CLAUSE.LIST NIL))
	((EQL RESULT 'SUCCESS)
	 (SETQ REASON     (CAR REASON)
	       PROOF.PART (PDS-ALLOCATE.MEMORY.ADDRESS 'PROOF.PART))
	 (COND ((AND (INTEGERP REASON) (PDS-CLAUSE.IS (PPR=GET.NEW.CLAUSE.ADDRESS REASON)))
		(SETQ REASON (PPR=GET.NEW.CLAUSE.ADDRESS REASON))
		(SETQ PROOF.TREE (PPR=CONSTRUCT.DEDUCTION.TREE REASON PROOF.PART)))
						; ALSO MARKS USED CLAUSES AND DEFINES USED.CLAUSE.LIST (a special variable)
	       ((OR (EQL REASON 'AXIOMS.UNSATISFIABLE) (EQL REASON 'THEOREMS.VALID))
		(SETQ PROOF.TREE (PPR=CONSTRUCT.DEDUCTION.TREE (CAR (LAST FINAL.CLAUSES)) PROOF.PART)))
						; ALSO MARKS USED CLAUSES AND DEFINES USED.CLAUSE.LIST (a special variable)
	       (T (ERROR "Unknown reason, ~S, of successful proof termination." REASON))))
	((EQL RESULT 'SPLIT) (SETQ REASON (SECOND REASON)) (SETQ PROOF.TREE NIL) (SETQ USED.CLAUSE.LIST NIL))
	(T (ERROR "Unknown result, ~S, of proof part" RESULT)))
  (values proof.tree final.clauses))

(DEFUN PPR=AXIOMS.GRAPH (CODE.OF.AXIOMS.GRAPH STATISTICS.FLAG)
						; EDITED: 10-SEP-84 09:28:56               BY CL
						; INPUT : A LIST CONTAINING THE CODE OF THE
						;                                    AXIOMS-GRAPH
						; EFFECT: PREPARES THE CODE AND STORES THE INFO
						; VALUE : UNDEFINED
  (let (START END RESULT REASON final.clauses used.clause.list)
    (declare (special used.clause.list))
    (MAPC #'(LAMBDA (CODE.PART)
	      (CASE (CAR CODE.PART)
		(PARTIAL    (PDS-ADD.AXIOMS (PPR=CLAUSES (CDR CODE.PART) 'AXIOM)))
		(OPERATION  (PDS-ADD.AX.OPERATION (PPR=OPERATION (CDR CODE.PART))))
		(STATISTICS (when sTATISTICS.FLAG
			      (cerror "Continue without statistics." "Statistics are not supported by the protocol")))
		(START.TIME (SETQ START (SECOND CODE.PART)))
		(END.TIME   (SETQ END   (SECOND CODE.PART)))
		(FINAL      (setq final.clauses (PDS-PUT.FINAL.AXIOM.CLAUSES (CDR CODE.PART))))
		(RESULT     (SETQ RESULT (SECOND CODE.PART))
			    (SETQ REASON (cddr CODE.PART)))
		(SYMBOLS    (PDS-PUT.AX.SYMBOLS (SECOND CODE.PART)))
		(OTHERWISE  (ERROR "Wrong key word ~S in code of the axioms." (CAR CODE.PART)))))
	  CODE.OF.AXIOMS.GRAPH)
    (ppr=result.handling result reason nil nil final.clauses)
    (PDS-PUT.AXIOMS.TIME (- END START))
    (SETQ PPR*AXIOMS.O.N.C.ADDRESSES (COPY-TREE PPR*OLD.NEW.CLAUSE.ADDRESSES))
    result))

(DEFUN PPR=PROOF.PART (CODE.OF.PROOF.PART STATISTICS.FLAG)
						; EDITED:  2-OCT-84 15:14:02               BY CL
						; INPUT : A LIST CONTAINING THE CODE OF AN INITIAL
						;           GRAPH (AXIOMS OR THEOREMS) AND A FLAG
						; EFFECT: PREPARES THE CODE AND STORES THE INFO
						; VALUE : THE ADDRESS OF THE INITIAL GRAPH
  (let (IDENTIFIER CHANGED.OPTIONS INITIAL.CLAUSES OPERATIONS FINAL.CLAUSES USED.CLAUSE.LIST START END RESULT
	REASON TIME PROOF.TREE SYMBOLS PROOF.PART)
    (declare (special used.clause.list))	;
    (SETQ PPR*OLD.NEW.CLAUSE.ADDRESSES (COPY-TREE PPR*AXIOMS.O.N.C.ADDRESSES))
    (MAPC #'(LAMBDA (CODE.PART)
	      (CASE (CAR CODE.PART)
		(SPLITPART.IDENTIFIER (SETQ IDENTIFIER      (SECOND CODE.PART)))
		(OPTIONS              (SETQ CHANGED.OPTIONS (PPR=CHANGED.OPTIONS (CDR CODE.PART))))
		(PARTIAL              (SETQ INITIAL.CLAUSES (NCONC INITIAL.CLAUSES (PPR=CLAUSES (CDR CODE.PART) 'THEOREM))))
		(OPERATION            (SETQ OPERATIONS      (NCONC OPERATIONS (PPR=OPERATION (CDR CODE.PART)))))
		(START.TIME           (SETQ START           (SECOND CODE.PART)))
		(END.TIME             (SETQ END             (SECOND CODE.PART)))
		(FINAL                (SETQ FINAL.CLAUSES   (MAPCAR (FUNCTION PPR=GET.NEW.CLAUSE.ADDRESS) (CDR CODE.PART))))
		(RESULT               (SETQ RESULT          (SECOND CODE.PART))
				      (SETQ REASON          (cddr CODE.PART)))
		(SYMBOLS              (SETQ SYMBOLS (SECOND CODE.PART)))
		(INITIAL   NIL)
		(OTHERWISE (ERROR "Wrong key word ~S in the code of a proof part (axioms or theorems)." (CAR CODE.PART)))))
	  CODE.OF.PROOF.PART)
    (SETQ TIME (- END START))
    (multiple-value-setq (proof.tree final.clauses) (ppr=result.handling result reason proof.part proof.tree final.clauses))
    (PROGN (SETQ PROOF.PART (PDS-PROOF.PART.CREATE
			      IDENTIFIER CHANGED.OPTIONS INITIAL.CLAUSES OPERATIONS "NO STATISTICS"
			      "RESERVED FOR TOTAL STATISTICS" FINAL.CLAUSES RESULT REASON TIME USED.CLAUSE.LIST PROOF.TREE
			      SYMBOLS PPR*OLD.NEW.CLAUSE.ADDRESSES PROOF.PART))
	   (PPR=MARK.USED.OPERATIONS OPERATIONS)
	   (COND (STATISTICS.FLAG (cerror "Continue without statistics." "Statistics are not supported by the protocol"))))
    PROOF.PART))

(DEFVAR PPR*OLD.NEW.CLAUSE.ADDRESSES NIL)
(defvar PPR*OLD.NEW.CLAUSE.ADDRESSES.initial nil)

(DEFVAR PPR*AXIOMS.O.N.C.ADDRESSES NIL)

(DEFUN PPR=SPLITPART (CODE.OF.SPLITPART OPTIONS.FLAG STATISTICS.FLAG code.file)
						; edited:  2-oct-84 15:28:26  by cl
						; input : a list containing info about a splitpart
						; effect: prepares the info and stores it into memory
						; value : an integer (the internal address)
  (let (AXIOMS THEOREMS INITIAL.GRAPH CHANGED.OPTIONS OPERATIONS RESULT USED.CLAUSE.LIST REASON START END
	PROOF.TREE SYMBOLS IDENTIFIER SPLITPART (PRECEDING.PROOF.PART NIL) (PROTOCOL.TYPE (PDS-GET.PROTOCOL.TYPE)))
    (DECLARE (SPECIAL USED.CLAUSE.LIST))
						; test whether refutation is empty
    
    (cond ((not (or (PDS-GET.AXIOMS) (PDS-ALL.PROOF.PARTS)	; non-empty graph
		    (some #'(lambda (code.part)
			      (and (eq (first code.part) 'initial)
				   (rest code.part)))
			  code.of.splitpart)))
	   (cerror "Delete ~A.~%~
               Later try to construct a protocol again from previous initial refutation."
		   "The code file ~A is empty.~%~
               Presumably it was written by REFUTE when the proof had already been found initially."                            
		   (truename (pathname code.file)))
	   (delete-file code.file))
    
						; initialization
    (t (if (EQL PROTOCOL.TYPE 'C&S)
	   (let (PRECEDING.IDENTIFIER)
	     (SETQ IDENTIFIER (CADAR (MEMBER-IF #'(LAMBDA (ELEMENT) (EQL (CAR ELEMENT) 'SPLITPART.IDENTIFIER)) CODE.OF.SPLITPART)))
	     (SETQ PRECEDING.IDENTIFIER (PPR=PRECEDING.IDENTIFIER IDENTIFIER))
	     (SMAPC #'(LAMBDA (PROOF.PART)
			(when (EQL (PDS-PROOF.PART.IDENTIFIER PROOF.PART) PRECEDING.IDENTIFIER)
			  (SETQ PRECEDING.PROOF.PART PROOF.PART)))
		    #'(LAMBDA (REST) (if PRECEDING.PROOF.PART NIL (CDR REST)))
		    (PDS-ALL.PROOF.PARTS))
	     (SETQ PPR*OLD.NEW.CLAUSE.ADDRESSES (PDS-PROOF.PART.O.N.C.ADDRESSES PRECEDING.PROOF.PART))
	     (SETQ THEOREMS (PDS-PROOF.PART.INITIAL.CLAUSES PRECEDING.PROOF.PART)
		   AXIOMS   (MAPCAR #'PPR=GET.NEW.CLAUSE.ADDRESS (PDS-GET.FINAL.AXIOM.CLAUSES))))
	   (SETQ PPR*OLD.NEW.CLAUSE.ADDRESSES (LIST NIL)))
    
						; preparation of splitpart code
       (MAPC #'(LAMBDA (CODE.PART)
		 (CASE (CAR CODE.PART)
		   (INITIAL    (SETQ INITIAL.GRAPH (COND ((EQ (PDS-GET.PROTOCOL.TYPE) 'SPLITPARTS)
							  (PPR=CLAUSES (CDR CODE.PART) 'INITIAL))
							 (T  (MAPCAR #'(LAMBDA (CLAUSE.CODE)
									 (PPR=GET.NEW.CLAUSE.ADDRESS
									   (PPR=GET.CLAUSE.ADDRESS CLAUSE.CODE)))
								     (CDR CODE.PART))))))
		   (OPTIONS    (COND ((EQL PROTOCOL.TYPE 'C&S)
				      (SETQ CHANGED.OPTIONS (PPR=CHANGED.OPTIONS (CDR CODE.PART))))
				     (OPTIONS.FLAG (PDS-PUT.OPTIONS (CDR CODE.PART)))))
		   (OPERATION  (SETQ OPERATIONS (NCONC OPERATIONS (PPR=OPERATION (CDR CODE.PART)))))
		   (RESULT     (SETQ RESULT  (CDR CODE.PART)))
		   (START.TIME (SETQ START   (SECOND CODE.PART)))
		   (END.TIME   (SETQ END     (SECOND CODE.PART)))
		   (SYMBOLS    (SETQ SYMBOLS (SECOND CODE.PART)))
		   (SPLITPART.IDENTIFIER  (unless (EQL PROTOCOL.TYPE 'C&S)	; see initialization above	
					    (SETQ IDENTIFIER (CDR CODE.PART))))
		   (OTHERWISE (ERROR "Wrong keyword ~S in code of a splitpart. " (CAR CODE.PART)))))
	     CODE.OF.SPLITPART)
						; result & reason with proof tree
    
       (PROGN (SETQ REASON    (SECOND RESULT)
		    RESULT    (first RESULT)
		    SPLITPART (PDS-ALLOCATE.MEMORY.ADDRESS 'SPLITPART))
	      (COND ((OR (NULL RESULT) (EQL RESULT 'FAILURE)))	; in all of the following cases result = SUCCESS. con-
						; struction of deduction tree also marks used clauses
						; and builds used.clause.list (a special variable)
		    ((PDS-CLAUSE.IS (PPR=GET.NEW.CLAUSE.ADDRESS REASON))
		     (SETQ REASON (PPR=GET.NEW.CLAUSE.ADDRESS REASON))
		     (SETQ PROOF.TREE (PPR=CONSTRUCT.DEDUCTION.TREE REASON SPLITPART)))
		    ((EQL REASON 'AXIOMS.UNSATISFIABLE)
		     (SETQ PROOF.TREE (PPR=CONSTRUCT.DEDUCTION.TREE (CAR (LAST AXIOMS)) SPLITPART)))
		    ((EQL REASON 'THEOREMS.VALID)
		     (let (EMPTY.CLAUSE (CLAUSES (PDS-OPERATION.CLAUSES (CAR (LAST OPERATIONS)))))
		       (SMAPC (FUNCTION (LAMBDA (CLAUSE)
					  (unless (PDS-CLAUSE.LITERALS CLAUSE) (SETQ EMPTY.CLAUSE CLAUSE))))
			      (FUNCTION (LAMBDA (REST) (COND (EMPTY.CLAUSE NIL) (T (CDR REST)))))
			      CLAUSES)
		       (SETQ PROOF.TREE (PPR=CONSTRUCT.DEDUCTION.TREE EMPTY.CLAUSE SPLITPART))))
		    (T (ERROR "Unknown reason, ~S, of proof end." REASON))))
    
						; MARKING OF PARTS USED IN PROOF AND STATISTICS
    
       (PROGN (SETQ SPLITPART (PDS-SPLITPART.CREATE AXIOMS THEOREMS INITIAL.GRAPH CHANGED.OPTIONS OPERATIONS RESULT REASON
						    USED.CLAUSE.LIST PROOF.TREE (- end start) "RESERVED FOR TOTAL STATISTICS"
						    SYMBOLS IDENTIFIER SPLITPART))
	      (PPR=MARK.USED.OPERATIONS OPERATIONS)
	      (when (EQL PROTOCOL.TYPE 'C&S)
		(PPR=MARK.USED.OPERATIONS (pds-proof.part.OPERATIONS preceding.proof.part)))
	      (when STATISTICS.FLAG (cerror "Continue without statistics." "Statistics are not supported by the protocol")))
       SPLITPART))))

(DEFUN PPR=PRECEDING.IDENTIFIER (IDENTIFIER)
						; EDITED: 30-JUN-84 11:03:27          BY CL
						; INPUT : A SPLITPART IDENTIFIER
						; EFFECT: FINDS OUT THE IDENTIFIER OF THE PRECEDING
						;         PROOF PART
						; VALUE : IDENTIFIER OF PRECEDING PROOF PART
  (COND ((NUMBERP IDENTIFIER) IDENTIFIER) ((CONSP IDENTIFIER) (CAR (LASTN IDENTIFIER 1)))
	(T (ERROR "Wrong proof part identifier ~S" IDENTIFIER))))

(DEFUN PPR=CHANGED.OPTIONS (OPTIONS.LIST)
						; EDITED: 14-SEP-83 09:56:38              BY CL
						; INPUT : A LIST OF DOTTED PAIRS (NEW OPTIONS)
						; VALUE : SUBSET OF INPUT LIST CONTAINING ALL THE
						;         OPTIONS WITH CHANGED VALUES
  (PROG (CHANGED.OPTIONS)
	(MAPC
	  (FUNCTION
	    (LAMBDA (NEW.OPTION OLD.OPTION)
	      (COND
		((NOT (EQUAL NEW.OPTION OLD.OPTION)) (SETQ CHANGED.OPTIONS (NCONC1 CHANGED.OPTIONS NEW.OPTION))))))
	  OPTIONS.LIST (PDS-GET.OPTIONS))
	(RETURN CHANGED.OPTIONS)))

(DEFUN PPR=CONSTRUCT.DEDUCTION.TREE (CLAUSE SPLITPART)
  (declare (special used.clause.list))
						; EDITED:  2-OCT-84 15:29:13           BY CL
						; INPUT : A CLAUSE ADDRESS AND A SPLITPART ADDRESS
						;                           OR A PROOF PART ADDRESS
						;    REM: THE GLOBAL VARIABLE USED.CLAUSE.LIST IS AUG-
						;         MENTED IN THIS FUNCTION. IT IS SUPPOSED TO
						;         BE NIL, WHEN THIS FUNCTION IS CALLED FROM
						;         OUTSIDE.
						; EFFECT: CONSTRUCTS DEDUCTION TREE OF CLAUSE AND,
						;         IF SPLITPART <> NIL, MARKS THE CLAUSES USED
						;         IN THIS DEDUCTION. BESIDES LIST OF USED
						;         CLAUSES IS DEFINED.
						; VALUE : THE DEDUCTION TREE AS A LIST, IDENTICAL
						;         SUBTREES BEING EQ (I.E. SHARING STRUCTURE)
  (if (MEMBER SPLITPART (PDS-CLAUSE.USE CLAUSE))
      (PDS-CLAUSE.PROPERTY CLAUSE 'DEDUCTION.TREE)
      (let ((PARENTS (PDS-CLAUSE.PARENTS CLAUSE)))
	(PDS-CLAUSE.ADD.USE CLAUSE SPLITPART)
	(SETQ USED.CLAUSE.LIST (NCONC1 USED.CLAUSE.LIST CLAUSE))
	(COND ((ATOM PARENTS) (PDS-CLAUSE.ADD.PROPERTY CLAUSE 'DEDUCTION.TREE CLAUSE))
	      ((CONSP PARENTS)
	       (PDS-CLAUSE.ADD.PROPERTY CLAUSE 'DEDUCTION.TREE
					(CONS CLAUSE (MAPCAR #'(LAMBDA (PARENT)
								 (PPR=CONSTRUCT.DEDUCTION.TREE PARENT SPLITPART))
							     (FLATTEN PARENTS)))))))))

(DEFUN PPR=CONSTRUCT.LIST.OF.ANCESTORS (DEDUCTION.TREE)
						; EDITED: 16-SEP-83 08:55:03             BY CL
						; INPUT : A LIST REPRESENTING A DEDUCTION TREE
						; VALUE : A SORTED LIST OF ALL ELEMENTS (OF ANY LEVEL)
						;         OF THE INPUT TREE
  (SORT (REMOVE-DUPLICATES (FLATTEN DEDUCTION.TREE)) #'ALPHORDER))

#| Not used
(DEFUN PPR=MARK.USED.CLAUSES (USED.CLAUSE.LIST SPLITPART)
  ; EDITED: 16-SEP-83 13:55:58            BY CL
  ; INPUT : A LIST OF CLAUSE ADDRESSES AND AN INTEGER
  ; EFFECT: ADDS THE SPLITPART TO THE LISTS STORED
  ;         AS 'CLAUSE.USE FOR ALL THE CLAUSES IN LIST
  ; VALUE : UNDEFINED
  (MAPC (FUNCTION (LAMBDA (CLAUSE) (PDS-CLAUSE.ADD.USE CLAUSE SPLITPART))) USED.CLAUSE.LIST))|#

(DEFUN PPR=MARK.USED.OPERATIONS (OPERATIONS)
						; EDITED: 22-AUG-84 15:28:25
						; INPUT : A LIST OF ADDRESSES (OPERATIONS)
						; EFFECT: MARKS THE OPERATIONS NEEDED IN THE PROOF
						; VALUE : UNDEFINED
  (MAPC #'(LAMBDA (OPERATION)
	    (some #'(LAMBDA (CLAUSE)
		      (if (PDS-CLAUSE.USE CLAUSE)
			  (progn (PDS-OPERATION.INSERT.USE OPERATION T) T)
			  NIL))
		  (PDS-OPERATION.CLAUSES OPERATION)))
	OPERATIONS))

(DEFVAR PPR*SPLITTABLE.OPERATIONS '(REPLACEMENT.OPERATION))

(DEFVAR PPR*LENGTH.OF.CLAUSE.CHANGED)

(DEFUN PPR=OPERATION (OPERATION.CODE)		; edited: 11-sep-84 10:39:48  by cl
						; input :  the code as put out by pr-operation
						;          clauses=nil, unless called recursively, then a list of integers
						; effect:  prepares the operation and stores the info
						; value :  list of addresses of operations
  (let (CLAUSES PARENTS DESCRIPTORS (OPERATIONS NIL) (TYPE (CAAR (LAST OPERATION.CODE))))
    (MAPC #'(LAMBDA (CODE.PART)
	      (CASE (CAR CODE.PART)
		(CLAUSE (unless (MEMBER TYPE PPR*SPLITTABLE.OPERATIONS)
			  (SETQ CLAUSES (NCONC1 CLAUSES (PPR=CLAUSE CODE.PART TYPE)))))
						; The clause must be evaluated before the operation.
		((DOUBLE.LITERAL instantiation instantiate RESOLUTION REPL.RES PARAMODULATION
				 FACTORIZATION REWRITE REWRITE.SYMMETRY)
		 (SETQ DESCRIPTORS (PPR=OPERATION.DESCRIPTION TYPE (CDR CODE.PART))))
		(REPLACEMENT.OPERATION (SETQ OPERATIONS  (PPR=SPLIT.REPLACEMENT.OPERATION (CDR CODE.PART))))
		(OTHERWISE (ERROR "Wrong entry or unknown operation type ~S" (CAR CODE.PART)))))
	  OPERATION.CODE)
    (unless OPERATIONS				; i.e. the operation has not been splitted
      (SETQ PARENTS (REMOVE-DUPLICATES (MAPCAN #'(LAMBDA (CLAUSE) (COPY-TREE (PDS-CLAUSE.PARENTS CLAUSE))) CLAUSES)))
      (SETQ OPERATIONS (LIST (PDS-OPERATION.CREATE CLAUSES TYPE PARENTS NIL DESCRIPTORS))))
    OPERATIONS))

(DEFUN PPR=OPERATION.DESCRIPTION (OP.TYPE DESCRIPTORS)
						; EDITED: 13-SEP-84 13:44:49              BY CL
						; INPUT :  AN ATOM AND A LIST
						; EFFECT:  PREPARES THE DESCRIPTION OF AN OPERATION
						;          OF TYPE OP.TYPE
						; VALUE :  THE PREPARED LIST OF DESCRIPTORS
  (CASE OP.TYPE
    (INSTANTIATION         (list "hole" (first descriptors)))	; the unifier
    (instantiate
      (let ((PAR (second DESCRIPTORS))
	    (UNIFIER (first DESCRIPTORS)))
	(SETQ PAR (PPR=GET.NEW.CLAUSE.ADDRESS PAR))
	(LIST UNIFIER PAR)))
    ((RESOLUTION REPL.RES) (let ((POSPAR (first DESCRIPTORS)) (POSLITNO (SECOND DESCRIPTORS))
				 (NEGPAR (THIRD DESCRIPTORS)) (NEGLITNO (FOURTH DESCRIPTORS))
				 (RULE    (fifth   DESCRIPTORS))
				 (UNIFIER (sixth   DESCRIPTORS))
				 (CLAUSE  (seventh DESCRIPTORS)))
			     (SETQ POSPAR (PPR=GET.NEW.CLAUSE.ADDRESS POSPAR)
				   NEGPAR (if (eql negpar clause)
					      ppr*clause.changed
					      (PPR=GET.NEW.CLAUSE.ADDRESS NEGPAR))
				   CLAUSE (PPR=GET.NEW.CLAUSE.ADDRESS CLAUSE)
				   RULE   (PPR=RULE RULE))
			     (when (AND RULE (NEQ RULE 'SYMMETRIC) (NEQ RULE 'ASYMMETRIC))
			       (PDS-CLAUSE.ADD.PARENT CLAUSE RULE))			       
			     (COND
			       ((EQL CLAUSE POSPAR)
				(SETQ POSPAR (CAR (PDS-CLAUSE.PARENTS CLAUSE)))
				(PDS-CLAUSE.ADD.PARENT CLAUSE NEGPAR))
			       ((EQL CLAUSE NEGPAR)
				(SETQ NEGPAR (CAR (PDS-CLAUSE.PARENTS CLAUSE)))
				(PDS-CLAUSE.ADD.PARENT CLAUSE POSPAR))
			       (T NIL))
						; USES THE FACT, THAT CLAUSE HAS BEEN CONSTRUCTED
						; BEFORE THE DESCRIPTION OF THE OPERATION.
			     (LIST RULE UNIFIER POSPAR POSLITNO "HOLE" NEGPAR NEGLITNO)))
    (PARAMODULATION
      (let ((EQPAR (first DESCRIPTORS))  (EQLITNO (SECOND DESCRIPTORS)) (EQFCT (THIRD DESCRIPTORS))
	    (PAR   (FOURTH DESCRIPTORS)) (LITNo   (fifth DESCRIPTORS))  (FCT   (sixth DESCRIPTORS))
	    (RULE (THIRD (CDDDDR DESCRIPTORS)))
	    (UNIFIER (FOURTH (CDDDDR DESCRIPTORS))))
        (SETQ EQPAR (PPR=GET.NEW.CLAUSE.ADDRESS EQPAR)
	      PAR   (PPR=GET.NEW.CLAUSE.ADDRESS PAR)
	      RULE  (PPR=RULE RULE))
        (when (AND RULE (NEQ RULE 'SYMMETRIC) (NEQ RULE 'ASYMMETRIC))
	  (PDS-CLAUSE.ADD.PARENT (PPR=GET.NEW.CLAUSE.ADDRESS (CAR (LAST DESCRIPTORS))) RULE))
        (LIST RULE UNIFIER EQPAR EQLITNO EQFCT PAR LITNO FCT)))
    (FACTORIZATION
      (PROG ((PAR (CAR DESCRIPTORS)) (RULE (SECOND DESCRIPTORS)) (UNIFIER (THIRD DESCRIPTORS)))
	    (SETQ PAR (PPR=GET.NEW.CLAUSE.ADDRESS PAR)) (SETQ RULE (PPR=RULE RULE))
	    (COND
	      ((AND RULE (NEQ RULE 'SYMMETRIC) (NEQ RULE 'ASYMMETRIC))
	       (PDS-CLAUSE.ADD.PARENT (PPR=GET.NEW.CLAUSE.ADDRESS (CAR (LAST DESCRIPTORS))) RULE)))
	    (RETURN (LIST RULE UNIFIER PAR))))
    (DOUBLE.LITERAL
      (PROG
        ((REMAINING.LITNO (CAR DESCRIPTORS)) (REMOVED.LITNO (SECOND DESCRIPTORS)) (RULE (THIRD DESCRIPTORS)))
        (SETQ RULE (PPR=RULE RULE))
        (COND
          ((AND RULE (NEQ RULE 'SYMMETRIC) (NEQ RULE 'ASYMMETRIC))
	   (PDS-CLAUSE.ADD.PARENT (PPR=GET.NEW.CLAUSE.ADDRESS (CAR (LAST DESCRIPTORS))) RULE)))
        (RETURN (LIST RULE REMAINING.LITNO REMOVED.LITNO))))
    (REWRITE
      (PROG ((RULE (CAR DESCRIPTORS))
	     (LITNO (SECOND DESCRIPTORS))
	     (CLAUSE (THIRD DESCRIPTORS)))
	    (SETQ RULE (PPR=RULE RULE T))
	    (SETQ CLAUSE (PPR=GET.NEW.CLAUSE.ADDRESS CLAUSE))
	    (COND ((NEQ RULE 'SYMMETRIC) (PDS-CLAUSE.ADD.PARENT CLAUSE RULE))) (RETURN (LIST RULE LITNO))))
    (REWRITE.SYMMETRY NIL)
    (OTHERWISE (ERROR "Unknown operation type ~S" OP.TYPE))))

(DEFUN PPR=RULE (RULE &OPTIONAL (REWRITE NIL))
						; EDITED:  3-JUL-84 17:45:56          BY CL
						; INPUT :  A RULE AS GIVEN IN DS, E.G. ( 9 1 2 2 1 )
						; EFFECT:  CHANGES THE RULE TO ITS CLAUSE ADDRESS
						;          USED IN PDS.
						; VALUE :  ADDRESS OF RULE CLAUSE IN PDS
						;          OR ONE OF THE ATOMS 'SYMMETRIC 'ASYMMETRIC
  (COND ((NULL RULE) NIL)
	((CONSP RULE)
	 (IF REWRITE
	     (MAPCAR #'PPR=GET.NEW.CLAUSE.ADDRESS RULE)
	     (PPR=GET.NEW.CLAUSE.ADDRESS (CAR RULE))))
	((INTEGERP RULE) (PPR=GET.NEW.CLAUSE.ADDRESS RULE))
	((OR (EQL RULE 'SYMMETRIC) (EQL RULE 'ASYMMETRIC)) RULE)
	(T (ERROR "Unknown syntax of rule ~S" RULE))))

(defvar ppr*clause.changed)
(defvar ppr*clause.symbols nil)

(defun ppr=change.unifier (unifier vars.sorts.of.codomain clause.changed)
						; Edited:  15-AUG-1990 13:05
						; Authors: ap + cl
						; Input:   
						; Effect:  keiner mehr
						; Value:   inverted unifier with resp. to renaming components,
						;          and new.vars.sorts.of.codomain
  (let ((sorts.vars (pds-clause.sorts.variables clause.changed))
	(new.vars.sorts.of.codomain (copy-tree vars.sorts.of.codomain)))
    (while (let ((renaming nil))
	     (ssomel #'(lambda (rest.uni)
			 (when (and (assoc (second rest.uni) vars.sorts.of.codomain)	; is variable
				    (some #'(lambda (sort.vars)
					      (member (first rest.uni) (rest sort.vars)))
					  sorts.vars))
			   (push (first rest.uni) renaming)
			   (push (second rest.uni) renaming)
			   (setf (first (assoc (second rest.uni) new.vars.sorts.of.codomain))
				 (first rest.uni))
			   (setq unifier (uni-switch renaming unifier))))
		     #'cddr
		     unifier)))
    (values unifier new.vars.sorts.of.codomain)))

(DEFUN PPR=SPLIT.REPLACEMENT.OPERATION (OP.CODE)
						; EDITED: 13-SEP-84 13:47:03            BY CL
						; INPUT : A LIST, CODE AS PUT OUT BY PR-OPERATION:
						;         (<UNIFIER> <CLAUSES.INVOLVED> <RESOLUTIONS>
						;          <DOUBLE.LIT.LISTS> <VARS.SORTS OF CODOMAIN>)
						; EFFECT: CREATES A (SERIES OF) OPERATION(S) CORRES-
						;         PONDING TO THE REPLACEMENT.OPERATION
						; VALUE : LIST OF OPERATION ADDRESSES
  (let ((UNIFIER                (first  OP.CODE))
	(CLAUSES.INVOLVED       (SECOND OP.CODE))
	(RESOLUTIONS            (THIRD  OP.CODE))
	(DOUBLE.LIT.LISTS       (FOURTH OP.CODE))
	(VARS.SORTS.OF.CODOMAIN (fifth  OP.CODE))
	CLAUSE.CHANGED
	AFFECTED.VARS
	(OPERATIONS (LIST NIL)))
    (declare (ignore affected.vars))
    (SETQ CLAUSE.CHANGED               (CAR CLAUSES.INVOLVED)
	  ppr*clause.changed           (PPR=GET.NEW.CLAUSE.ADDRESS clause.changed)
	  PPR*LENGTH.OF.CLAUSE.CHANGED (LIST-LENGTH (PDS-CLAUSE.LITERALS (PPR=GET.NEW.CLAUSE.ADDRESS CLAUSE.CHANGED))))
    (multiple-value-setq (unifier vars.sorts.of.codomain) (ppr=change.unifier unifier vars.sorts.of.codomain ppr*clause.changed))
    (WHEN (MEMBER CLAUSE.CHANGED (CDR CLAUSES.INVOLVED))
      (setq ppr*clause.symbols nil)
      (SETQ DOUBLE.LIT.LISTS
	    (CONS (FIRST DOUBLE.LIT.LISTS)
		  (MAPCAR #'(LAMBDA (RES DBL.LITS)
			      (IF (AND (OR (= CLAUSE.CHANGED (CAR (FIRST RES)))
					   (= CLAUSE.CHANGED (CAR (SECOND RES))))
				       (OR (= (SECOND CLAUSES.INVOLVED) (CAR (FIRST RES)))
					   (= (SECOND CLAUSES.INVOLVED) (CAR (SECOND RES)))))
				  (let ((new.symbol (gentemp (format nil "PPR-~A-" clause.changed))))
				    (setq ppr*clause.symbols (nconc1 ppr*clause.symbols new.symbol))
				    (PUTASSOC new.symbol
					      (cassoc clause.changed PPR*OLD.NEW.CLAUSE.ADDRESSES)
					      PPR*OLD.NEW.CLAUSE.ADDRESSES)
				    (dotimes (litno PPR*LENGTH.OF.CLAUSE.CHANGED)
				      (unless (= (1+ litno) (if (= CLAUSE.CHANGED (CAR (FIRST RES)))
								(cdr (first res))
								(cdr (second res))))
					(setq dbl.lits (NCONC1 DBL.LITS (list (cons clause.changed (1+ litno))
									      (cons new.symbol (1+ litno))
									      nil)))))
				    dbl.lits)
				  (progn (setq ppr*clause.symbols (nconc1 ppr*clause.symbols nil))
					 DBL.LITS)))
			  (CDR RESOLUTIONS)
			  (CDR DOUBLE.LIT.LISTS)))))
    (PPR=BUILD.OLDLITS.NEWLITS resolutions DOUBLE.LIT.LISTS (remove nil ppr*clause.symbols))
    (COND
      ((null resolutions) ;(AND UNIFIER (NULL RESOLUTIONS)) Sometimes there are pure double lit removeals here ! Axel
						; I.E. PURE REPLACEMENT FACTORIZATION
       (NCONC OPERATIONS
	      (PPR=OPERATION
		(LIST (PPR=SIMULATE.INSTANTIATION CLAUSE.CHANGED UNIFIER VARS.SORTS.OF.CODOMAIN)
		      (LIST 'INSTANTIATION UNIFIER CLAUSE.CHANGED)))
	      (PPR=DOUBLE.LITERAL.OPERATIONS (CAR DOUBLE.LIT.LISTS) CLAUSE.CHANGED)))
      (T (let ((ppr*clause.symbols nil))
	   (NCONC OPERATIONS
		  (PPR=RESOLUTION.OPERATION      (first RESOLUTIONs)      CLAUSE.CHANGED UNIFIER VARS.SORTS.OF.CODOMAIN)
		  (PPR=DOUBLE.LITERAL.OPERATIONS (first DOUBLE.LIT.lists) CLAUSE.CHANGED)
		  (pop ppr*clause.symbols)))
	 (MAPC #'(LAMBDA (RESOLUTION DOUBLE.LITERALS)
		   (NCONC OPERATIONS
			  (PPR=RESOLUTION.OPERATION      RESOLUTION      CLAUSE.CHANGED UNIFIER VARS.SORTS.OF.CODOMAIN)
			  (PPR=DOUBLE.LITERAL.OPERATIONS DOUBLE.LITERALS CLAUSE.CHANGED))
		   (pop ppr*clause.symbols))
	       (rest RESOLUTIONS) (rest DOUBLE.LIT.LISTS))))
    (setq ppr*clause.changed nil)
    (CDR OPERATIONS)))

(DEFVAR PPR*OLDLITS.NEWLITS nil)

(DEFUN PPR=BUILD.OLDLITS.NEWLITS (resolutions DOUBLE.LIT.LISTS news)
						; EDITED: 25-AUG-84 10:11:22           BY CL
						; INPUT : RES. IS A LIST OF LISTS BEGINNING WITH TWO
						;         'LITERALS' EACH. A LITERAL IS A DOTTED PAIR
						;                       (CLAUSE.ADDRESS  .  LITNO)
						;         DBL.LIT.LISTS IS A LIST OF LISTS EACH HAVING
						;         THE SAME STRUCTURE AS RESOLUTIONS.
						; VALUE :  A LIST OF DOTTED PAIRS OF EACH LITERAL IN-
						;          VOLVED WITH
  (let ((clause.changed (first (first (first resolutions)))))
    (setq resolutions (cons (first resolutions)
			    (mapcar #'(lambda (res) (if (eql clause.changed (caadr res))
							(let ((new.symbol (pop news)))
							  (list (first res) (cons new.symbol (cdadr res)) (caddr res)))
							res))
				    (rest resolutions)))))
  (SETQ PPR*OLDLITS.NEWLITS
	(MAPCAN #'(LAMBDA (LIT.LIT.RULE)
		    (list (cons (CAR LIT.LIT.RULE) (CAR LIT.LIT.RULE)) (cons (SECOND LIT.LIT.RULE) (SECOND LIT.LIT.RULE))))
		(APPEND RESOLUTIONS (MAPCAN #'copy-list DOUBLE.LIT.LISTS)))))

(DEFUN PPR=RESOLUTION.OPERATION (RESOLUTION CLAUSE UNIFIER VARS.SORTS.OF.CODOMAIN)
						; EDITED: 13-SEP-84 13:51:33
						; INPUT : A LIST DESCRIBING THE RESOLUTION STEP
						;         FORM: < (23 . 1) (43 . 1) (9 1 2 2 1) >
						;         AN ADDRESS, A UNIFIER, AND A LIST OF DOTTED
						;         PAIRS (ALL CONTAINIG ADDRESSES FROM PROOF).
						; EFFECT: DOES THE RESOLUTION STEP DESCRIBED
						; VALUE : A LIST CONTAINIG THE ADDRESS OF THE RESUL-
						;         TING RESOLUTION STEP
  (when (and (first ppr*clause.symbols) (eql clause (caadr resolution)))
    (setq resolution (list (first resolution)
			   (cons (first ppr*clause.symbols) (cdadr resolution))
			   (caddr resolution))))
  (let (CLAUSE.CODE RESOLUTION.CODE
	(1ST.LITERAL (CAR RESOLUTION))
	POSPAR POSLITNO
	(2ND.LITERAL (SECOND RESOLUTION))
	NEGPAR NEGLITNO
	(RULE (THIRD RESOLUTION)))
    (SETQ 1ST.LITERAL (PPR=ACTUAL.LITERAL 1ST.LITERAL)
						; THE 2ND LITERAL IS NOT ACTUALIZED SINCE IT IS ALWAYS
						; FROM AN ORIGINAL CLAUSE. IT MAY, HOWEVER, BE A COPY
						; OF A CLAUSE ALREADY USED IN THE SAME REPLACEMENT
						; STEP (WITH RENAMED VARIABLES).
	  pospar   (CAR 1ST.LITERAL)
	  POSLITNO (CDR 1ST.LITERAL)
	  NEGPAR   (CAR 2ND.LITERAL)
	  NEGLITNO (CDR 2ND.LITERAL))
    (SETQ CLAUSE.CODE
	  (PPR=SIMULATE.RESOLUTION UNIFIER VARS.SORTS.OF.CODOMAIN CLAUSE POSPAR POSLITNO NEGPAR NEGLITNO))
    (SETQ RESOLUTION.CODE (LIST 'REPL.RES POSPAR POSLITNO NEGPAR NEGLITNO RULE UNIFIER CLAUSE))
    (PPR=UPDATE.OLD.NEW.LITS 'RESOLUTION RESOLUTION)
    (SETQ PPR*LENGTH.OF.CLAUSE.CHANGED
	  (+ PPR*LENGTH.OF.CLAUSE.CHANGED
	     (LIST-LENGTH (PDS-CLAUSE.LITERALS (if (eql negpar clause)
						   ppr*clause.changed
						   (PPR=GET.NEW.CLAUSE.ADDRESS NEGPAR))))
	     -2))
    (PPR=OPERATION (LIST CLAUSE.CODE RESOLUTION.CODE))))

(DEFUN PPR=SIMULATE.RESOLUTION (UNIFIER VARS.SORTS.OF.CODOMAIN CLAUSE.CHANGED 1ST.CLAUSE 1ST.LITNO 2ND.CLAUSE 2ND.LITNO)
						; EDITED: 13-SEP-84 13:52:22           BY CL
						; INPUT :  A LIST WITH EVEN NUMBER OF ELEMENTS, A LIST
						;          OF DOTTED PAIRS, AND FOUR ADDRESSES
						; EFFECT:  CONSTRUCTS THE CLAUSE RESULTING FROM THE
						;          RESOLUTION
						; VALUE :  THE CODE OF THE RESULTING CLAUSE (AS PUT
						;          OUT BY PR=CLAUSE)
  (let (LITERALS SORTS.VARS (VARS.SORTS NIL) LENGTH.OF.1ST.CLAUSE (LITERAL.NUMBER 0)
	(NEW.1ST.CLAUSE (PPR=GET.NEW.CLAUSE.ADDRESS 1ST.CLAUSE))
	(NEW.2ND.CLAUSE (if (eql 2nd.clause clause.changed)
			    ppr*clause.changed
			    (PPR=GET.NEW.CLAUSE.ADDRESS 2nd.clause))))
    (declare (ignore literal.number))
    (PROGN					;              LITERALS
      (SETQ LITERALS (PDS-CLAUSE.LITERALS NEW.1ST.CLAUSE))
      (SETQ LENGTH.OF.1ST.CLAUSE (LIST-LENGTH LITERALS))
      (SETQ LITERALS (PPR=APPLY.SUBSTITUTION UNIFIER (APPEND LITERALS (PDS-CLAUSE.LITERALS NEW.2ND.CLAUSE))))
      (SETQ LITERALS (delete-nth LITERALS (+ LENGTH.OF.1ST.CLAUSE 2ND.LITNO -1)))
      (SETQ LITERALS (delete-NTH LITERALS (1- 1ST.LITNO))))
    (PROGN					;                   VARIABLES & SORTS
      (SETQ SORTS.VARS
	    (APPEND (PDS-CLAUSE.SORTS.VARIABLES NEW.1ST.CLAUSE) (PDS-CLAUSE.SORTS.VARIABLES NEW.2ND.CLAUSE)))
      (MAPC
	(FUNCTION
	  (LAMBDA (SORT.VARS)
	    (PROG ((SORT (CAR SORT.VARS)))
		  (MAPC (FUNCTION (LAMBDA (VAR) (SETQ VARS.SORTS (NCONC1 VARS.SORTS (CONS VAR SORT))))) (CDR SORT.VARS)))))
	SORTS.VARS)
      (SETQ VARS.SORTS (NCONC VARS.SORTS VARS.SORTS.OF.CODOMAIN))
      (SETQ VARS.SORTS (REMOVE-DUPLICATES VARS.SORTS :test #'equal))
      (SETQ VARS.SORTS (PPR=VARS.SORTS.USED VARS.SORTS LITERALS)))
    (LIST 'CLAUSE CLAUSE.CHANGED (PDS-CLAUSE.PNAME NEW.1ST.CLAUSE) (LIST 1ST.CLAUSE 2ND.CLAUSE) VARS.SORTS LITERALS)))

(DEFUN PPR=SIMULATE.INSTANTIATION (CLAUSE UNIFIER VARS.SORTS.OF.CODOMAIN)
						; EDITED: 25-SEP-84 17:08:23           BY CL
						; INPUT : AN INTEGER (OLD CLAUSE ADDRESS), A UNIFIER,
						;         AND AN ASSOC-LIST.
						; EFFECT: CONSTRUCTS THE CLAUSE RESULTING FROM APPLI-
						;         CATION OF THE UNIFIER TO THE CLAUSE.
						; VALUE : THE CODE OF THE RESULTING CLAUSE AS PUT OUT
						;         BY PR=CLAUSE.
  (let (VARS.SORTS AFFECTED.VARS (NEW.ADDRESS (PPR=GET.NEW.CLAUSE.ADDRESS CLAUSE))
	(DOMAIN (SMAPCAR (FUNCTION (LAMBDA (VAR) VAR)) (FUNCTION CDDR) UNIFIER)))
    (MAPC #'(LAMBDA (SORT.VARS)
	      (let ((SORT (CAR SORT.VARS)))
		(MAPC #'(LAMBDA (VAR)
			  (COND ((MEMBER VAR DOMAIN) (SETQ AFFECTED.VARS (NCONC1 AFFECTED.VARS VAR)))
				(T (SETQ VARS.SORTS (NCONC1 VARS.SORTS (CONS VAR SORT))))))
		      (CDR SORT.VARS))))
	  (PDS-CLAUSE.SORTS.VARIABLES NEW.ADDRESS))
    (LIST 'CLAUSE CLAUSE (PDS-CLAUSE.PNAME NEW.ADDRESS) (LIST CLAUSE)
	  (REMOVE-DUPLICATES
	    (APPEND VARS.SORTS (PPR=IMPORTED.VARS.SORTS VARS.SORTS.OF.CODOMAIN UNIFIER AFFECTED.VARS))
	    :test #'equal)
	  (PPR=APPLY.SUBSTITUTION UNIFIER (PDS-CLAUSE.LITERALS NEW.ADDRESS)))))

(DEFUN PPR=UPDATE.OLD.NEW.LITS (OP.TYPE DESCRIPTION)
						; EDITED: 31-AUG-84 16:22:05
						; INPUT : LIST OF DOTTED PAIRS (OF DOTTED PAIRS), AN
						;         ATOM, AN S-EXPRESSION (DESCRIBED BELOW)
						;         AND AN INTEGER.
						; EFFECT: UPDATES THE CORRESPONDENCE OF OLD//NEW LITS
						; VALUE : UNDEFINED
  (CASE OP.TYPE
    (DOUBLE.LITERAL				;  NOW DESCRIPTION = REMOVED LITERAL
      (let ((NEW.REMOVED.LITERAL (PPR=ACTUAL.LITERAL DESCRIPTION)) NEW.REMOVED.CLAUSE NEW.REMOVED.LITNO (removed nil))
        (SETQ NEW.REMOVED.CLAUSE (CAR NEW.REMOVED.LITERAL)) (SETQ NEW.REMOVED.LITNO (CDR NEW.REMOVED.LITERAL))
        (SETQ PPR*OLDLITS.NEWLITS
	      (MAPCAN #'(LAMBDA (OLDLIT.NEWLIT)
			  (let ((NEW.LITNO (CDDR OLDLIT.NEWLIT)))
			    (COND ((OR (NEQ NEW.REMOVED.CLAUSE (SECOND OLDLIT.NEWLIT)) (< NEW.LITNO NEW.REMOVED.LITNO))
				   (LIST OLDLIT.NEWLIT))
				  (T (COND ((and (not removed) (EQL NEW.LITNO NEW.REMOVED.LITNO))
					    (setq removed t) NIL)
					   ((EQL NEW.LITNO NEW.REMOVED.LITNO) (list oldlit.newlit))
					   (T	; I.E. NEW LITNO > REMOVED LITNO" * *)
					    (LIST (CONS (CAR OLDLIT.NEWLIT) (CONS (SECOND OLDLIT.NEWLIT) (1- NEW.LITNO)))))))))) 
		      PPR*OLDLITS.NEWLITS))))
    (RESOLUTION					; NOW DESCRIPTION = LITERALS RESOLVED UPON
      (let* ((1ST.LITERAL (CAR DESCRIPTION))
	     (2ND.LITERAL (SECOND DESCRIPTION))
	     (NEW.1ST.LITERAL (PPR=ACTUAL.LITERAL 1ST.LITERAL))
	     (NEW.2ND.LITERAL (PPR=ACTUAL.LITERAL 2ND.LITERAL))
	     (RESULTING.CLAUSE (CAR NEW.1ST.LITERAL))
	     (1removed nil) (2removed nil))
        (SETQ PPR*OLDLITS.NEWLITS
	      (MAPCAN #'(LAMBDA (OLDLIT.NEWLIT)
			  (let ((NEW.CLAUSE (SECOND OLDLIT.NEWLIT))
				(NEW.LITNO (CDDR OLDLIT.NEWLIT)))
			    (COND
			      ((EQL NEW.CLAUSE (CAR NEW.1ST.LITERAL))
			       (COND ((and (EQL NEW.LITNO (CDR NEW.1ST.LITERAL)) (not 1removed))
				      (setq 1removed t) NIL)
				     ((EQL NEW.LITNO (CDR NEW.1ST.LITERAL)) (list oldlit.newlit))
				     ((< NEW.LITNO (CDR NEW.1ST.LITERAL))
				      (LIST (CONS (CAR OLDLIT.NEWLIT) (CONS RESULTING.CLAUSE NEW.LITNO))))
				     (T (LIST (CONS (CAR OLDLIT.NEWLIT) (CONS RESULTING.CLAUSE (1- NEW.LITNO)))))))
			      ((EQL NEW.CLAUSE (CAR NEW.2ND.LITERAL))
			       (COND ((and (EQL NEW.LITNO (CDR NEW.2nd.LITERAL)) (not 2removed)) (setq 2removed t) NIL)
				     ((EQL NEW.LITNO (CDR NEW.2nd.LITERAL)) (list oldlit.newlit))
				     ((< NEW.LITNO (CDR NEW.2ND.LITERAL))
				      (LIST
					(CONS (CAR OLDLIT.NEWLIT)
					      (CONS RESULTING.CLAUSE (+ PPR*LENGTH.OF.CLAUSE.CHANGED -1 NEW.LITNO)))))
				     (T (LIST
					  (CONS (CAR OLDLIT.NEWLIT)
						(CONS RESULTING.CLAUSE (+ PPR*LENGTH.OF.CLAUSE.CHANGED -2 NEW.LITNO)))))))
			      (T (LIST OLDLIT.NEWLIT)))))
		      PPR*OLDLITS.NEWLITS))))
    (OTHERWISE (ERROR "No updating of literals is needed after a ~S operation." OP.TYPE))))

(DEFUN PPR=DOUBLE.LITERAL.OPERATIONS (DOUBLE.LITERALS CLAUSE)
						; EDITED: 18-SEP-84 14:06:55           BY CL
						; INPUT : A LIST (CF. PPR=DOUBLE.LITERAL.OPERATION )
						;         AND A CLAUSE ADDRESS (FROM PROOF)
						; EFFECT: DOES THE DOUBLE LITERAL OPERATIONS
						; VALUE : A LIST OF THE RESULTING OPERATION ADDRESSES
  (when (and ppr*clause.symbols
	     (not (eql (caadar double.literals) (caaar double.literals)))
	     (or (eql clause (caadar double.literals)) (eql clause (caaar double.literals))))
    (setq double.literals (mapcar #'(lambda (dl) (list (if (not (eql clause (caar dl)))
							   (cons (first ppr*clause.symbols) (cdar dl))
							   (first dl))
						       (if (not (eql clause (caadr dl)))
							   (cons (first ppr*clause.symbols) (cdadr dl))
							   (second dl))
						       (caddr dl)))
				  double.literals)))
  (MAPCAN (FUNCTION (LAMBDA (DOUBLE.LITERAL) (PPR=DOUBLE.LITERAL.OPERATION DOUBLE.LITERAL CLAUSE)))
	  (SORT DOUBLE.LITERALS (FUNCTION PPR=REM.DBL.LIT.EARLIER))))

(DEFUN PPR=REM.DBL.LIT.EARLIER (DBL.LIT.1 DBL.LIT.2)
						; EDITED: 18-SEP-84 13:57:41           BY CL
						; INPUT : TWO LISTS ( LIT1  LIT2  RULE )
						;         LIT IS A DOTTED PAIR ( ADDRESS . LITNO )
						;         RULE IS NOT USED.
						; EFFECT: FINDS OUT IF THE FIRST DBL.LIT HAS TO BE
						;         REMOVED BEFORE THE SECOND.
						; VALUE : T IF YES, NIL ELSE.
  (EQUAL (CAR DBL.LIT.1) (SECOND DBL.LIT.2)))

(DEFUN PPR=DOUBLE.LITERAL.OPERATION (DOUBLE.LITERAL CLAUSE)
						; EDITED: 25-AUG-84 10:31:07           BY CL
						; INPUT : DOUBLE.LITERAL
						;         FORM: < (17 . 3) (20 . 1) (12 1 2 2 1) >
						;         AND THE ADDRESS OF THE CLAUSE CHANGED
						; EFFECT: DOES THE DOUBLE.LITERAL OPERATION
						; VALUE : A LIST CONTAINING THE ADDRESS OF THE
						;         DOUBLE.LITERAL OPERATION
  (PROG
    (CLAUSE.CODE DOUBLE.LIT.CODE (ACTUAL.1ST.LITNO (PPR=ACTUAL.LITNO (CAR DOUBLE.LITERAL)))
     (ACTUAL.2ND.LITNO (PPR=ACTUAL.LITNO (SECOND DOUBLE.LITERAL))) (RULE (THIRD DOUBLE.LITERAL)))
    (SETQ CLAUSE.CODE (PPR=SIMULATE.DOUBLE.LITERALS CLAUSE ACTUAL.2ND.LITNO))
    (SETQ DOUBLE.LIT.CODE (LIST 'DOUBLE.LITERAL ACTUAL.1ST.LITNO ACTUAL.2ND.LITNO RULE CLAUSE))
    (PPR=UPDATE.OLD.NEW.LITS 'DOUBLE.LITERAL (SECOND DOUBLE.LITERAL))
    (SETQ PPR*LENGTH.OF.CLAUSE.CHANGED (1- PPR*LENGTH.OF.CLAUSE.CHANGED))
    (RETURN (PPR=OPERATION (LIST CLAUSE.CODE DOUBLE.LIT.CODE)))))

(DEFUN PPR=SIMULATE.DOUBLE.LITERALS (CLAUSE REMOVED.LITNO)
						; EDITED:  5-JUL-84 13:29:02           BY CL
						; INPUT : TWO INTEGERS, CLAUSE = ADDRESS IN PROOF
						;         REMOVED.LITNO = ACTUAL LITERAL NUMBER
						; EFFECT: CONSTRUCTS THE CLAUSE RESULTING AFTER
						;         DELETING OF THE SPECIFIED LITERAL
						; VALUE : THE CODE OF THE RESULTING CLAUSE AS PUT OUT
						;         BY PR=CLAUSE
  (PROG (VARS.SORTS (NEW.ADDRESS (PPR=GET.NEW.CLAUSE.ADDRESS CLAUSE)))
	(MAPC
	  (FUNCTION
	    (LAMBDA (SORT.VARS)
	      (PROG ((SORT (CAR SORT.VARS)))
		    (MAPC (FUNCTION (LAMBDA (VAR) (SETQ VARS.SORTS (NCONC1 VARS.SORTS (CONS VAR SORT))))) (CDR SORT.VARS)))))
	  (COPY-LIST (PDS-CLAUSE.SORTS.VARIABLES NEW.ADDRESS)))
	(RETURN
	  (LIST 'CLAUSE CLAUSE (PDS-CLAUSE.PNAME NEW.ADDRESS) (LIST CLAUSE) VARS.SORTS
		(delete-NTH (COPY-LIST (PDS-CLAUSE.LITERALS NEW.ADDRESS)) (1- REMOVED.LITNO))))))

(DEFUN PPR=ACTUAL.LITERAL (OLD.LITERAL) (CDR (ASSOC OLD.LITERAL PPR*OLDLITS.NEWLITS :TEST (FUNCTION EQUAL))))

(DEFUN PPR=ACTUAL.LITNO (OLD.LITERAL) (CDDR (ASSOC OLD.LITERAL PPR*OLDLITS.NEWLITS :TEST (FUNCTION EQUAL))))

(DEFUN PPR=APPLY.SUBSTITUTION (SUBSTITUTION LIST)
						; EDITED: 29-MAR-84 08:57:21           BY CL
						; INPUT : A LIST WITH EVEN NUMBER OF ELEMENTS,
						;         AND AN ARBITRARY LIST
						; EFFECT: ANY OF THE ODD ELEMENTS OF SUBSTITUTION
						;         APPEARING IN LIST ARE SUBSTITUTED BY THEIR
						;         SUCCESSOR IN SUBSTITUTION.
						; VALUE : THE LIST AFTER SUBSTITUTION
						; REMARK: CF. UNI=APPLY.SUBSTITUTION ]
  (PROG ((RESULT (COPY-TREE LIST)))
	(SMAPL
	  (FUNCTION (LAMBDA (SUBSTITUTES) (SETQ RESULT (NSUBST (SECOND SUBSTITUTES) (CAR SUBSTITUTES) RESULT))))
	  (FUNCTION CDDR) SUBSTITUTION)
	(RETURN RESULT)))

(DEFUN PPR=VARS.SORTS.USED (VARS.SORTS LITERALS)
						; EDITED: 10-SEP-84 11:06:13
						; INPUT : A LIST OF DOTTED PAIRS AND A LIST OF
						;         LITERALS, BOTH CONTAINING ADDRESSES FOR
						;         VARIABLE, CONSTANT, AND FUNCTION SYMBOLS.
						; EFFECT: FINDS OUT, WHICH OF THE VARIABLES ACTUALLY
						;         APPEAR IN THE LITERALS.
						; VALUE : A SUBSET OF VARS.SORTS
  (REMOVE-IF-NOT (FUNCTION (LAMBDA (VAR.SORT) (INSIDE (CAR VAR.SORT) LITERALS))) VARS.SORTS))

(DEFUN PPR=IMPORTED.VARS.SORTS (VARS.SORTS.OF.CODOMAIN UNIFIER AFFECTED.VARS)
						; EDITED: 10-SEP-84 10:13:39           BY CL
						; INPUT : AN ASSOC-LIST, A UNIFIER, AND A LIST OF
						;         INTEGERS (OLD VARIABLE ADDRESSES)
						; EFFECT: FINDS THE VARIABLES INTRODUCED BY THE UNI-
						;         FIER AND THEIR RESPSCTIVE SORTS.
						; VALUE : A LIST OF DOTTED PAIRS ( VAR . SORT )
  (PROG (IMPORTED.TERMS)
	(SMAPL
	  (FUNCTION
	    (LAMBDA (REST.UNIFIER)
	      (COND ((MEMBER (CAR REST.UNIFIER) AFFECTED.VARS) (SETQ IMPORTED.TERMS (SECOND REST.UNIFIER))))))
	  (FUNCTION CDDR) UNIFIER)
	(RETURN
	  (REMOVE-IF-NOT (FUNCTION (LAMBDA (VAR.SORT) (INSIDE (CAR VAR.SORT) IMPORTED.TERMS))) VARS.SORTS.OF.CODOMAIN))))

(DEFUN PPR=CLAUSES (CLAUSES.CODE ORIGIN)
						; EDITED:  2-JUL-84 15:57:34             BY CL
						; INPUT : A LIST OF SUBLISTS ('CLAUSE ...........) ,
						; EFFECT: PREPARES THE CLAUSES AND STORES THE INFO
						; VALUE : LIST OF CLAUSE ADDRESSES
  (MAPCAR #'(LAMBDA (CLAUSE.CODE) (PPR=CLAUSE CLAUSE.CODE ORIGIN))
	  CLAUSES.CODE))

(DEFUN PPR=CLAUSE (CLAUSE.CODE ORIGIN)
						; EDITED: 28-SEP-84 18:23:33           BY CL
						; INPUT : A CLAUSE AS PUT OUT BY PR=CLAUSE
						;         AND AN ATOM DENOTING THE TYPE OF OPERATION
						;         THAT LED TO THE CONSTRUCTION OF THE CLAUSE.
						; EFFECT: PREPARES THE CLAUSE AND STORES THE INFO
						; VALUE : THE NEW CLAUSE ADDRESS (INTEGER)
  (let ((PNAME (PPR=GET.CLAUSE.PNAME CLAUSE.CODE))
	SORTS.VARS (LITERALS (PPR=GET.CLAUSE.LITERALS CLAUSE.CODE))
	(OLD.ADDRESS (PPR=GET.CLAUSE.ADDRESS CLAUSE.CODE))
	(NEW.ADDRESS (PDS-ALLOCATE.MEMORY.ADDRESS 'CLAUSE))
	PARENTS)
    (SETQ SORTS.VARS (PPR=CLAUSE.SORTS.VARIABLES CLAUSE.CODE))
    (SETQ PNAME (PPR=ALLOCATE.CLAUSE.PNAME PNAME ORIGIN))
    (SETQ PARENTS
	  (COND ((eq origin 'initial) 
		 'initial)
		((MEMBER ORIGIN PPR*DESTRUCTIVE.OPERATIONS) (LIST (PPR=GET.NEW.CLAUSE.ADDRESS OLD.ADDRESS)))
		(T (PUTASSOC OLD.ADDRESS NEW.ADDRESS PPR*OLD.NEW.CLAUSE.ADDRESSES.initial)
		   (PPR=CLAUSE.PARENTS (PPR=GET.CLAUSE.PARENTS CLAUSE.CODE)))))
    (PUTASSOC OLD.ADDRESS NEW.ADDRESS PPR*OLD.NEW.CLAUSE.ADDRESSES)
    (PDS-CLAUSE.CREATE PNAME PARENTS SORTS.VARS LITERALS NIL NEW.ADDRESS)))

(DEFUN PPR=CLAUSE.PARENTS (PARENTS)
						; EDITED:  3-AUG-83 15:03:44            BY CL
						; INPUT :  A LIST OF INTEGERS (OLD ADDRESSES) OR ONE
						;          OF THE ATOMS 'AXIOM OR 'THEOREM
						; EFFECT:  PREPARES CLAUSE PARENTS
						; VALUE :  A LIST OF INTEGERS (NEW.ADDRESSES) OR ONE
						;          OF THE ATOMS 'AXIOM 'THEOREM
  (if (CONSP PARENTS)
      (MAPCAR #'PPR=GET.NEW.CLAUSE.ADDRESS PARENTS)
      PARENTS))

(DEFUN PPR=CLAUSE.SORTS.VARIABLES (CLAUSE.CODE)
						; EDITED:  3-JUL-84 09:19:18            BY CL
						; INPUT :  CLAUSE, AS PUT OUT BY PR=CLAUSE, AND FLAG
						; EFFECT:  PREPARES CLAUSE-VARIABLES AND -SORTS.
						;          RENAMES VARIABLES.
						; VALUE :  A LIST E.G < (SORT1 2 8) (SORT2 7) >
  (let (SORTS.VARS SORTS (VARS.SORTS (PPR=GET.CLAUSE.VARIABLES.SORTS CLAUSE.CODE)))
    (when VARS.SORTS
      (SETQ SORTS      (LIST NIL)
	    SORTS.VARS (LIST NIL))		; START TCONC
      (MAPC #'(LAMBDA (VAR.SORT)
		(let ((VAR (CAR VAR.SORT)) (SORT (CDR VAR.SORT)) END.FLAG)
		  (if (MEMBER SORT (CAR SORTS))
		      (SMAPC #'(LAMBDA (SORT.VARS)
				 (COND ((EQL SORT (CAR SORT.VARS)) (NCONC1 SORT.VARS VAR) (SETQ END.FLAG T)) (T NIL)))
			     #'(LAMBDA (TAIL) (if END.FLAG (SETQ END.FLAG NIL) (CDR TAIL)))
			     (CAR SORTS.VARS))
		      (progn (QCONC1 SORTS SORT)
			     (QCONC1 SORTS.VARS (LIST SORT VAR))))))
	    VARS.SORTS)
      (SETQ SORTS.VARS (CAR SORTS.VARS)))	; END TCONC
    SORTS.VARS))

(DEFUN PPR=ALLOCATE.CLAUSE.PNAME (PNAME ORIGIN)
						; EDITED: 13-SEP-84 13:53:54          BY CL
						; INPUT :  A STRING (PNAME FROM PROOF) AND AN ATOM
						;          INDICATING, WHERE THE CLAUSE STEMS FROM.
						; EFFECT:  ALLOCATES A PNAME FOR THE CLAUSE
						; VALUE :  THE NEW PNAME
  (PROG (NUMBER NEW.PNAME)
	(COND
	  (PPR*NEW.PNAMES
	   (COND
	     ((MEMBER ORIGIN PPR*INITIAL.CLAUSE.INDICATORS) (SETQ NUMBER PPR*INITIAL.CLAUSE.NUMBER)
	      (SETQ PPR*INITIAL.CLAUSE.NUMBER (1+ NUMBER)))
	     (T (SETQ NUMBER PPR*DEDUCED.CLAUSE.NUMBER) (SETQ PPR*DEDUCED.CLAUSE.NUMBER (1+ NUMBER))))
	   (SETQ NEW.PNAME
		 (CONCATENATE 'STRING
			      (CASE ORIGIN
				(AXIOM "A") (THEOREM "T") (GENERATED "G") (INITIAL "IC")
				((INSTANTIATION instantiate) "I")
				((RESOLUTION REPL.RES) "R")
				(PARAMODULATION "P") (FACTORIZATION "F") (DOUBLE.LITERAL "D") (REWRITE "RW")
				(REWRITE.SYMMETRY "RS")
				(OTHERWISE (ERROR "Unknown origin, ~S, of clause" ORIGIN)))
			      (PRINC-TO-STRING NUMBER))))
	  (T (SETQ NEW.PNAME (STRING PNAME))))
	(RETURN NEW.PNAME)))

(DEFVAR PPR*DESTRUCTIVE.OPERATIONS '(DOUBLE.LITERAL INSTANTIATION REPL.RES REWRITE REWRITE.SYMMETRY))

(DEFVAR PPR*INITIAL.CLAUSE.INDICATORS '(AXIOM THEOREM))

(DEFUN PPR=GET.CLAUSE.PNAME (CLAUSE.CODE) (THIRD CLAUSE.CODE))

(DEFUN PPR=GET.CLAUSE.ADDRESS (CLAUSE.CODE) (SECOND CLAUSE.CODE))

(DEFUN PPR=GET.CLAUSE.PARENTS (CLAUSE.CODE) (FOURTH CLAUSE.CODE))

(DEFUN PPR=GET.CLAUSE.VARIABLES.SORTS (CLAUSE.CODE) (CAR (CDDDDR CLAUSE.CODE)))

(DEFUN PPR=GET.CLAUSE.LITERALS (CLAUSE.CODE) (SECOND (CDDDDR CLAUSE.CODE)))


(DEFUN PPR=GET.NEW.CLAUSE.ADDRESS (OLD.ADDRESS)	; edited: 13-sep-84 13:58:00  by cl
						; input :  an integer, clause address as used in proof
						; value :  an integer, clause address used in protocol
  (let (ADDRESS.PAIR)
    (SETQ ADDRESS.PAIR (ASSOC OLD.ADDRESS PPR*OLD.NEW.CLAUSE.ADDRESSES))
    (if ADDRESS.PAIR
	(CDR ADDRESS.PAIR)
	(ERROR "The address ~S is not known as a clause (in proof)." OLD.ADDRESS))))


(DEFmacro PPR=PREPARE.PRINT.FORMAT (FILE)
  (declare (ignore file))
						; EDITED:  1-JUN-83 18:28:39         BY CL
						; INPUT :  A FILE NAME
						; EFFECT:  SETS ALL THE VALUES TO FORMAT THE OUTPUT,
						;          SETS ALL VIRTUAL PRINT SIGNS.
						; VALUE :  UNDEFINED
  NIL)




