;;; -*- mode: lisp; syntax: common-lisp; package: mkrp -*-

;;; Copyright (C) 1991 AG Siekmann, 
;;;                       Fachbereich Informatik, Universitaet des Saarlandes, 
;;;                       Saarbruecken, Germany
;;; 
;;; This file is part of Markgraf Karl Refutation Procedure (MKRP).
;;; 
;;; MKRP is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY.  No author or distributor
;;; accepts responsibility to anyone for the consequences of using it
;;; or for whether it serves any particular purpose or works at all,
;;; unless he says so in writing.  
;;; 
;;; Everyone is granted permission to copy, modify and redistribute
;;; MKRP, but only if the it is not used for military purposes or any
;;; military research. It is also forbidden to use MKRP in nuclear plants
;;; or nuclear research, and for verifying programs in military 
;;; and nuclear research.  A copy of this license is
;;; supposed to have been given to you along with MKRP so you
;;; can know your rights and responsibilities.  
;;; Among other things, the copyright notice
;;; must be preserved on all copies.  

(IN-PACKAGE "MARKGRAF-KARL" :use '("CL") :nicknames '("MKRP"))

(defvar po*indices nil "To record old literals")
(DEFVAR PO*FILE NIL)
(defvar PO*INDENTATION 0 "an integer, describing the number of blanks needed at the beginning of a line for formatted output")

(DEFUN PO-CONSTRUCT.START (FILE SYSTEM.VERSION COMMENT)
						      ; Edited:  08-APR-1992 19:12
						      ; Authors: MKRP CL
						      ; input : code file, version of system, a string comment
						      ; effect: sets linelength of code file to 120 and
						      ;         prints <CONSTRUCTION system.version date (C comment)>
						      ;         top-level elements of comment will be prin-
						      ;         in separate lines.
						      ; value : undefined
  (when (SETQ PO*FILE FILE)    
    (setq po*indices nil)
    (setq comment (mapcar #'(lambda (c.line) (if (stringp c.line) c.line (princ-to-string c.line))) comment))
    (format PO*FILE "~%(CONSTRUCTION  ~S ~S ~%~14T(~{~S~%~15T~}))"
	    system.version (date) comment)))


(DEFUN PO-INFIX.FORM (AXIOMS.INFIX THEOREMS.INFIX)
						; edited:  5-sep-84 16:13:46  by cl
						; input : two lists
						; effect: prints <AXIOMS.INFIX   (...) ... (...)>
						;                <THEOREMS.INFIX (...) ... (...)>
						; value : undefined
  (when PO*FILE
    (format  PO*FILE "~&~%(AXIOMS.INFIX    (~S~{~%~18T~S~}))~%~%(THEOREMS.INFIX (~S~{~%~18T~S~}))"
	     (first AXIOMS.INFIX) (rest axioms.infix) (first THEOREMS.INFIX) (rest theorems.infix))))

					
(DEFUN PO-PREFIX.FORM (AXIOMS.PREFIX THEOREMS.PREFIX)
						; edited: 21-apo-83 17:27:16  by cl
						; input : two lists
						; effect: prints <'axioms.prefix   (...) ... (...)>
						;                <'theorems.prefix (...) ... (...)>
						; value : undefined
  (WHEN PO*FILE
    (format  PO*FILE "~&~%(AXIOMS.PREFIX   (~S~{~%~18T~S~}))~%~%(THEOREMS.PREFIX (~S~{~%~18T~S~}))"
	     (first AXIOMS.PREFIX) (rest axioms.prefix) (first THEOREMS.PREFIX) (rest theorems.prefix))))

					
(DEFUN PO-OPTIONS NIL				; edited: 22-apo-83 08:05:46  by cl
						; input : none
						; effect: prints <OPTIONS ... all options and their
						;                      values as dotted pairs ... >
						; value : undefined
  (WHEN PO*FILE (PO=OPTIONS PO*FILE)))


(DEFUN PO-AXIOMS.START NIL			; edited: 22-apo-83 11:15:44  by cl
						; input : none
						; effect: prints <LINK.COLOURS R ... PER>
						;                <AXIOMS (START.TIME time in sec/977)
  (when PO*FILE
    (PO=LINK.COLOURS PO*FILE)
    (setq PO*INDENTATION 8)
    (format PO*file "~&~%(AXIOMS (START.TIME ~A)" (GET-INTERNAL-RUN-TIME))))

					
(DEFUN PO-AXIOMS.END (RESULT)			; edited: 23-mar-84 10:05:44  by cl
						; input : a list (SUCCESS AXIOMS.UNSATISFIABLE)
						;             or (SUCCESS   ..empty clause..)  or nil
						; effect: prints (END.TIME  time in sec/997 )
						;                (FINAL     actual clauses  )
						;                (RESULT    result reason   )>
						; value : undefined
  (when PO*FILE
    (PO=AXIOMS.OR.THEOREMS.END RESULT PO*FILE)
    (PRINC ")" PO*FILE)				; closes the bracket opened in po-axioms.start
    (setq PO*INDENTATION 0)))


(DEFUN PO-THEOREMS.START (SPLITPART.IDENTIFIER &optional SPLITFLAG)
  (declare (ignore splitflag))			; edited: 23-mar-84 10:23:02             by cl
						; input : a list of integers and a boolean value
						; effect: prints <THEOREMS (SPLITPART.IDENTIFIER ..)
						;                          (START.TIME    ..time.. )
						; value : undefined
  (WHEN PO*FILE
    (setq PO*INDENTATION 10)
    (format po*file "~&~%(THEOREMS (SPLITPART.IDENTIFIER~{ ~A~})~%~vT(START.TIME ~D)"
	    SPLITPART.IDENTIFIER PO*INDENTATION (GET-INTERNAL-RUN-TIME))))

					
(DEFUN PO-THEOREMS.END (RESULT)			; edited: 23-mar-84 10:32:37  by cl
						; input : one of the following lists:
						;          (SUCCESS THEOREMS.VALID)
						;          (SUCCESS ..empty.clause..)
						;          (FAILURE GRAPH.COLLAPSED)
						;          (FAILURE GRAPH.SATISFIABLE ..model..)
						;          (SPLIT   ..file.. ..splitpart.indicators..)
						; effect: prints (END.TIME time in sec/997)
						;                (FINAL    actual clauses )
						;                (RESULT   result reason  )>
						; value : undefined
  (when PO*FILE
    (PO=AXIOMS.OR.THEOREMS.END RESULT PO*FILE)
    (PRINC ")" PO*FILE)				; closes the bracket opened in po-theorems.start    
    (setq PO*INDENTATION 0)))


(DEFun PO-CONSTRUCT.END ()
  (testeval (format po*file "~%(indices ~A)" po*indices)))


(DEFUN PO-SPLITPARTS.START (FILE SYSTEM.VERSION &optional COMMENT)
						      ; Edited:  22-APR-1992 11:17
						      ; Authors: MKRP 
						      ; input : code file and version of system 
						      ; effect: sets linelength of code file to 120 and 
						      ;         prints <SPLITPARTS system.version date> 
						      ; value : undefined
  (SETQ PO*FILE FILE)
  (when po*file
    (linelength 117 po*file)			      ; length suitable for lineprinter
    (format PO*file "~&~%(SPLITPARTS ~S ~S ~S)" SYSTEM.VERSION (DATE) comment)))

					
(DEFUN PO-REFUTATION.START ()
						; edited: 28-aug-84 10:50:39  by cl
						; input : a list of lists of integers, a flag indica-
						;         ting if splitpart is continued, and an in-
						;         teger (or nil, if new splitpart begins)
						; effect: prints    <'refutation
						;                       ('start.time ..in msec..)
						; value : undefined
  (when PO*FILE
    (format PO*FILE "~&~%(REFUTATION (START.TIME ~D)" (GET-INTERNAL-RUN-TIME))
    (setq PO*INDENTATION 12)
    (PO=OPTIONS PO*FILE)))

					
(DEFUN PO-PARTIAL.GRAPH (PARTIAL.CLAUSE.LIST)	; edited: 22-apo-83 08:10:18  by cl
						; input : list of clauses
						; effect: prints  <PARTIAL ... list of clauses as
						;                          described in po=clauses ...>
						; value : undefined
  (when (AND PO*FILE PARTIAL.CLAUSE.LIST)
    (format PO*FILE "~&~vT(PARTIAL   " PO*INDENTATION)
    (setq PO*INDENTATION (+ PO*indentation 9))
    (MAPC (FUNCTION (LAMBDA (CLAUSE) (PO=CLAUSE CLAUSE PO*FILE t))) PARTIAL.CLAUSE.LIST)
    (PRINC ")" PO*FILE)
    (setq PO*INDENTATION (- PO*indentation 9))))


(DEFUN PO-INITIAL.GRAPH ()			; edited: 10-jul-84 16:00:47  by cl
						; input : none
						; effect: prints  <INITIAL ... list of clauses as
						;                          described in po=clauses ...>
 						; value : undefined
  (when PO*FILE
    (format po*file "~&~vT(INITIAL   " PO*INDENTATION)
    (setq PO*INDENTATION (+ PO*indentation 9))
    (MAPC #'(LAMBDA (CLAUSE)
	      (PO=CLAUSE CLAUSE PO*FILE t)
	      (format po*file "~%~vT" po*indentation))
	  (DS-RULES))
    (MAPL #'(LAMBDA (rest.CLAUSEs)
	      (PO=CLAUSE (first rest.CLAUSEs) PO*FILE)
	      (format po*file "~:[~%~;~vT~]" (endp (cdr rest.clauses)) po*indentation))
	  (CG-CLAUSES ALL))
    (PRINC ")" PO*FILE)
    (setq PO*INDENTATION (- PO*indentation 9))))


(DEFUN Po-OPERATION (OPERATION.TYPE ARGUMENTS)
						      ; Edited:  02-APO-1992 20:58
						      ; Authors: PRCKLN
						      ; input : an atom, and arguments according to the type of operation
						      ; effect: prints <OPERATION <CLAUSE  ..see PO=CLAUSE..>
						      ;                           <op.type ..see PO=op.type..>
						      ; value : undefined
  (when PO*FILE
    (CASE OPERATION.TYPE
      (R.CHAIN   (PO=R.CHAIN ARGUMENTS PO*FILE))
      (OTHERWISE (setq PO*INDENTATION (+ PO*indentation 11))
		 (CASE OPERATION.TYPE		      ; in any case PO=CLAUSE is called first for the resulting clause
		   (RESOLUTION             (PO=RESOLUTION             ARGUMENTS PO*FILE))
		   (PARAMODULATION         (PO=PARAMODULATION         ARGUMENTS PO*FILE))
		   (FACTORIZATION          (PO=FACTORIZATION          ARGUMENTS PO*FILE))
		   (instantiate            (PO=instantiate            ARGUMENTS PO*FILE))
		   (REPLACEMENT.OPERATION  (PO=REPLACEMENT.OPERATION  ARGUMENTS PO*FILE))
		   (DOUBLE.LITERAL         (PO=DOUBLE.LITERAL         ARGUMENTS PO*FILE))
		   (REWRITE                (PO=REWRITE                ARGUMENTS PO*FILE))
		   (REWRITE.SYMMETRY       (PO=REWRITE.SYMMETRY       ARGUMENTS PO*FILE))
		   (REPLACEMENT.RESOLUTION (PO=REPLACEMENT.RESOLUTION ARGUMENTS PO*FILE))
		   (OTHERWISE              (ERROR "Non-existing operation type: ~S" OPERATION.TYPE)))
		 (PRINC ")" PO*FILE)
		 (setq PO*INDENTATION (- PO*indentation 11))))))


(DEFUN PO-STATISTICS (TIME)
  (declare (ignore time))			; edited:  3-jul-84 17:19:28  by cl
						; input : an integer (time in sec/997)
						; effect: prints the following (example) :
						;   (STATISTICS (CLAUSES (ALL . 44) (INSERTED . 2) (REMOVED . 1) (CHANGED . 0))
						;               (LINKS (ALL . (223 12 ... 0 2))   (INSERTED . (12 0 ... 0 1))
						;               (REMOVED . (20 2 ... 1 1)) (CHANGED . (1 0 ... 0 0))))
						;         numbers separated for different link colours.
						; value : undefined
  nil)


(DEFUN PO-REFUTATION.END (SPLITPART.IDENTIFIER RESULT)
						; edited: 28-aug-84 11:43:46         by cl
						; input : a list of lists of integers and a list:
						;            (SUCCESS ..empty.clause..)
						;         OR (FAILURE GRAPH.SATISFIABLE ..model..),
						;            (FAILURE LINKS.INOPERABLE),
						;            (FAILURE ABORTED.MAXSTEPS),
						;            (FAILURE ABORTED.MANUALLY),
						;            (SPLIT ..file.. ..splitpart.indicators..)
						; effect: prints  (END.TIME ..in sec/977..)
						;                 (SYMBOLS  ....)
						;                 (SPLITPART.IDENTIFIER  ....)
						;                 (RESULT result reason)   >
						; value : undefined
  (when PO*FILE
    (format po*file "~&~vT(END.TIME ~D)" PO*INDENTATION (GET-INTERNAL-RUN-TIME))
    (PO=SYMBOLS PO*FILE)
    (format po*file "~&~vT(SPLITPART.IDENTIFIER~{ ~A~})" PO*INDENTATION SPLITPART.IDENTIFIER)
    (format po*file "~&~vT(RESULT~{ ~S~})~%" po*indentation RESULT)
    (PRINC ")" PO*FILE)				; closes bracket opened in po-refutation.start
    (setq PO*INDENTATION 0)))


(DEFUN PO-SPLITPARTS.END () nil)


(DEFUN PO=AXIOMS.OR.THEOREMS.END (RESULT FILE)	; edited: 28-aug-84 11:49:47  by cl
						; input : a list, see po-axioms.end or po-theorems.end
						;         and a file open for output
						; effect: prints (END.TIME ..time in sec/977..)
						;                (FINAL    ..actual.clauses.. )
						;                (SYMBOLS  .....)
						;                (RESULT   result reason)
  (FORMAT PO*FILE "~&~vT(END.TIME ~D)~%~vT(FINAL~{ ~A~})"
	  PO*INDENTATION (GET-INTERNAL-RUN-TIME) PO*INDENTATION (CG-CLAUSES ALL))
  (PO=SYMBOLS FILE)
  (format po*file "~&~vT(RESULT~{ ~A~})" PO*INDENTATION RESULT))


(DEFUN PO=OPTIONS (FILE)
  (format file "~&~%(OPTIONS~{ ~A~%~9T~})" (OPT-GET.LIST.OPTIONS)))


(DEFUN PO=SYMBOLS (FILE)			; edited: 28-aug-84 10:16:45  by cl
						; input : a file name (open for output)
						; effect: prints the info about all constants,
						;         functions, and predicates to the code file.
						;         <SYMBOLS (LET (NEW.ADDRESS)
						;                    (LIST .....))>
						; value : undefined
  (format file "~&~vT(SYMBOLS " PO*INDENTATION)
  (DT-SAVE.SYMBOLS FILE (+ PO*INDENTATION 9))
  (PRINC ")" FILE))


(DEFUN PO=RESOLUTION (LINK.UNI.CLAUSE FILE)
						      ; Edited:  02-APO-1992 21:22
						      ; Authors: PRCKLN
						      ; input : a list (link unifier clause) and a file
						      ; effect: prints information as described in PO=CLAUSE and PO=LINK.RES
						      ; value : undefined
  (let ((link (first link.uni.clause))
	(uni (second link.uni.clause))
	(clause (third link.uni.clause)))
    (format FILE "~%(~A " clause)
    (PO=CLAUSE clause FILE)
			 (format file "~% (RESOLUTION ")
    (PO=LINK.RES link FILE)
    (format file " (substitution ~A ~A)))" (dt-pname (uni-unifier.domain uni)) (dt-pname (uni-unifier.codomain uni)))))


(DEFUN PO=R.CHAIN (ARGUMENTS FILE)		; EDITED:  5-SEP-84 16:23:08
						; INPUT:  A LIST (RESOLVENT TERMLISTS CHAIN-ELEMENT)
						;         RESOLVENT IS THE NEW CLAUSE,
						;         CHAIN-ELEMENT IS A LIST
						;         (UNIFIER CLAUSE LINKS POINTERS LITNOS)
						;         WHERE CLAUSE IS THE PARENTCLAUSE OF
						;         RESOLVENT (WHICH IS CREATED BY SIMULTANEUS
						;         RESOLUTION UPON LINKS WITH 'UNIFIER')
						;         POINTERS AND LITNOS IS IRRELEVANT.
						;         TERMLISTS IS THE LIST OF TERMLISTS OF
						;         THE INTERMEDIATE CLAUSE WHICH WOULD BE
						;         CREATED BY RESOLUTION UPON THE FIRST ELEMENT
						;         OF LINKS.
						; EFFECT: THIS OPERATION IS PROTOCOLLED AS A
						;         RESOLUTION UPON THE FIRST ELEMENT OF LINKS
						;         FOLLOWED BY A REPLACEMENT RESOLUTION STEP.
						; VALUE:  UNDEFINED.
  (PROG ((RESOLVENT (CAR ARGUMENTS))
	 (TERMLISTS (SECOND ARGUMENTS))
	 (UNIFIER (CAR (THIRD ARGUMENTS)))
	 (CLAUSE (SECOND (THIRD ARGUMENTS)))
	 (LINKS (THIRD (THIRD ARGUMENTS)))	;; (LITNOS (CAR (CDDDR (THIRD ARGUMENTS))))
	 (LITNO 0)
	 FIRST COLOUR RLINKS DOUBLE.LITS)
	(SETQ COLOUR (DS-LINK.COLOUR (CAR LINKS)))
	(SETQ FIRST  (CASE COLOUR
		       (R  (DS-LINK.THISLITNO (CAR LINKS) CLAUSE))
		       (SI (DS-LINK.POSLITNO  (CAR LINKS)))
		       (OTHERWISE (ERROR "ILLEGAL COLOUR IN PO=R.CHAIN: ~A" COLOUR))))
	(format file "(OPERATION (CLAUSE ~A ~A ~A "
		RESOLVENT (DS-PNAME RESOLVENT)
		(CASE COLOUR (R (LIST CLAUSE (DS-LINK.OTHERPAR (CAR LINKS) CLAUSE))) (OTHERWISE (LIST CLAUSE))))
	
	;; (PRINC "(OPERATION (CLAUSE " FILE) (PRINC RESOLVENT FILE) (PRINC " " FILE) (PRINC (DS-PNAME RESOLVENT) FILE)
	;; (PRINC (CASE COLOUR (R (LIST CLAUSE (DS-LINK.OTHERPAR (CAR LINKS) CLAUSE))) (OTHERWISE (LIST CLAUSE))) FILE)
	;; (PRINC " " FILE)
						; variables.sort
	(PROGN (PRINC "(" FILE)
	       (MAPC (FUNCTION (LAMBDA (VAR) (PRINC "(" FILE) (PRINC VAR FILE) (PRINC " . " FILE)
				       (PRINC (DT-VARIABLE.SORT VAR) FILE) (PRINC ")" FILE)))
		     (DT-TERMLIST.VARIABLES TERMLISTS))
	       (PRINC ")" FILE))
	
	(progn (PRINC "(" FILE)
	       (DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
		 (PROGN (SETQ LITNO (1+ LITNO))
			(COND ((NEQ LITNO FIRST) (PRINC "(" FILE) (PRINC (DS-CLAUSE.SIGN CLAUSE LITNO) FILE) (PRINC " " FILE)
			       (PRINC (DS-CLAUSE.PREDICATE CLAUSE LITNO) FILE) (PRINC " " FILE) (PRINC (CAR TERMLISTS) FILE)
			       (SETQ TERMLISTS (CDR TERMLISTS)) (PRINC ")" FILE)))))
	       (PRINC "))(" FILE))
	
	(CASE COLOUR (SI (PRINC "FACTORIZATION " FILE) (PO=LINK.FAC (CAR LINKS) FILE))
	      (OTHERWISE (PROGN (PRINC "RESOLUTION " FILE) (PO=LINK.RES (CAR LINKS) FILE))))
	(PRINC " " FILE) (PRINC UNIFIER FILE) (PRINC " " FILE) (PRINC RESOLVENT FILE) (PRINC "))" FILE)
	(MAPC
	  (FUNCTION
	    (LAMBDA (LINK)
	      (COND ((EQL 'R (DS-LINK.COLOUR LINK)) (SETQ RLINKS (NCONC1 RLINKS LINK)))
		    (T
		     (SETQ DOUBLE.LITS
			   (NCONC1 DOUBLE.LITS
				   (LIST
				     (CONS RESOLVENT
					   (COND ((< (DS-LINK.NEGLITNO LINK) FIRST) (DS-LINK.NEGLITNO LINK))
						 (T (1- (DS-LINK.NEGLITNO LINK)))))
				     (CONS RESOLVENT
					   (COND ((< (DS-LINK.POSLITNO LINK) FIRST) (DS-LINK.POSLITNO LINK))
						 (T (1- (DS-LINK.POSLITNO LINK)))))
				     (DS-LINK.RULE LINK))))))))
	  (CDR LINKS))
	(COND
	  ((OR RLINKS DOUBLE.LITS) (PRINC "(OPERATION " FILE) (PO=CLAUSE RESOLVENT FILE)
	   (PRINC "(REPLACEMENT.OPERATION NIL (" FILE) (PRINC RESOLVENT FILE) (PRINC " " FILE)
	   (MAPC (FUNCTION (LAMBDA (LINK) (PRINC (DS-LINK.OTHERPAR LINK CLAUSE) FILE) (PRINC " " FILE))) RLINKS) (PRINC ")" FILE)
	   (COND
	     (RLINKS (PRINC "(" FILE)
		     (MAPC
		       (FUNCTION
			 (LAMBDA (LINK) (PRINC "((" FILE) (PRINC RESOLVENT FILE) (PRINC " . " FILE)
				 (SETQ LITNO (DS-LINK.THISLITNO LINK CLAUSE))
				 (PRINC (COND ((< LITNO FIRST) LITNO) (T (1- LITNO))) FILE) (PRINC ")(" FILE)
				 (PRINC (DS-LINK.OTHERPAR LINK CLAUSE) FILE) (PRINC " . " FILE)
				 (PRINC (DS-LINK.OTHERLITNO LINK CLAUSE litno) FILE) (PRINC ")" FILE)
				 (PRINC (DS-LINK.RULE LINK) FILE)
				 (PRINC ")" FILE)))
		       RLINKS)
		     (PRINC ")" FILE))
	     (T (PRINC " NIL " FILE)))
	   (PRINC "(" FILE) (PRINC (OR DOUBLE.LITS " NIL ") FILE)
	   (MAPC (FUNCTION (LAMBDA (LINK) (PRINC " NIL " FILE))) (CDR RLINKS)) (PRINC ")" FILE)
	   (PRINC (PO=VARIABLES.SORTS.OF.CODOMAIN NIL) FILE) (PRINC "))" FILE)))))

(DEFUN PO=LINK.RES (LINK FILE)			; edited:  3-jul-84 10:58:56               by cl
						; input : link address and a file opened for output
						; effect: prints :  pospar poslitno negpar neglitno rule
						; value : undefined
  (format file "((~A ~D) (~A ~D) ~A)"
	  (DS-LINK.POSPAR LINK) (DS-LINK.POSLITNO LINK)
	  (if (member (ds-link.colour link) (ds-link.colours.for 'autolinks))
	      (DS-LINK.POSPAR LINK)
	      (DS-LINK.NEGPAR LINK))
	  (DS-LINK.NEGLITNO LINK)
	  (DS-LINK.RULE LINK)))


(DEFUN PO=REPLACEMENT.RESOLUTION (ARGUMENTS FILE)
						; edited: 29-oct-83 14:03:08
						; input:  a list  (replaced.parent litno other.parent litno unifier)
						;         and the code file
						; effect: the information is printed to the code file in the same way
						;         as PO=RESOLUTION would.
						; value:  undefined.
  (PO=CLAUSE (first ARGUMENTS) FILE)
  (format file "~&~vT(RESOLUTION ~{~A ~}~A)" PO*indentation arguments (first arguments)))


(DEFUN PO=PARAMODULATION (LINK.UNI.CLAUSE FILE)	
						      ; Edited:  08-APR-1992 19:06
						      ; Authors: PRCKLN CL
						      ; input : a list (link unifier clause)  and a file
						      ; effect: prints information as described in po=clause
						      ;                   and po=link.par respectively
						      ; value : undefined
  (let ((link (first link.uni.clause))
	(uni (second link.uni.clause))
	(clause (third link.uni.clause)))
    (format FILE "~%(~A " clause)
    (PO=CLAUSE clause FILE)
    (format file "~% (paramodulation ")
    (PO=LINK.par link FILE)
    (format file " (substitution ~A ~A)))" (dt-pname (uni-unifier.domain uni)) (dt-pname (uni-unifier.codomain uni)))))


(DEFUN PO=LINK.PAR (LINK FILE)			
						      ; Edited:  08-APR-1992 20:07;
						      ; Authors: MKRP CL
						      ; input : link# and a file opened for output
						      ; effect: prints :  eqpar eqlitno eqfct par litno fct
						      ; value : undefined
  (format file "((~A ~D) (~A ~D ~A) ~A) ~A"
	  (DS-LINK.NEGPAR LINK) (DS-LINK.NEGLITNO LINK)
	  (DS-LINK.POSPAR LINK) (DS-LINK.POSLITNO LINK)
	  (if (DT-TAF.IS.LEFT (ds-link.posfct link)) 'lr 'rl)
	  (DS-LINK.RULE LINK) (DS-LINK.NEGFCT LINK)))


(DEFUN PO=FACTORIZATION (LINK.UNI.CLAUSE FILE)	; edited: 22-apo-83 08:10:18  by cl
						; input : a list (link unifier clause)  and a file
						; effect: prints information as described in po=clause
						;                   and po=link.fac respectively
						; value : undefined
  (PO=CLAUSE (THIRD LINK.UNI.CLAUSE) FILE)
  (format FILE "~&~vT(FACTORIZATION " po*indentation)
  (PO=LINK.FAC (CAR LINK.UNI.CLAUSE) FILE)
  (format file " ~A ~A)" (SECOND LINK.UNI.CLAUSE) (THIRD LINK.UNI.CLAUSE)))

(DEFUN PO=instantiate (UNI.CLAUSE.old FILE)	; edited: 22-apo-83 08:10:18  by cl
						; input : a list (link unifier clause)  and a file
						; effect: prints information as described in po=clause
						;                   and po=link.fac respectively
						; value : undefined
  (PO=CLAUSE (third UNI.CLAUSE.old) FILE)
  (format FILE "~&~vT(instantiate " po*indentation)
  (format file " ~A ~A)" (first UNI.CLAUSE.old) (second UNI.CLAUSE.old)))

(DEFUN PO=LINK.FAC (LINK FILE)			; edited:  3-jul-84 11:01:00  by cl
						; input : link# and a file opened for output
						; effect: prints par
						; value : undefined
  (format file "~A ~A"
	  (DS-LINK.POSPAR LINK) (DS-LINK.RULE LINK)))


(DEFUN PO=REPLACEMENT.OPERATION (CLAUSE.DESCRIPTION FILE)
						; edited: 24-mar-84 20:00:16  by cl
						; input : a list of two elements, an integer (the clause) and a
						;         list (unifier clauses resolutions multiples), and a file open for output.
						; effect: prints the info to the code file
						; value : undefined
  (PO=CLAUSE (CAR CLAUSE.DESCRIPTION) FILE)
  (let ((UNIFIER (CAADR CLAUSE.DESCRIPTION)))
    (format file "~&~vT(REPLACEMENT.OPERATION ~A ~{~A~%  ~}~A)"
	    po*indentation unifier (rest (SECOND CLAUSE.DESCRIPTION)) (PO=VARIABLES.SORTS.OF.CODOMAIN UNIFIER))))


(DEFUN PO=VARIABLES.SORTS.OF.CODOMAIN (UNIFIER)	; edited: 15-may-84 09:18:06  by cl
						; input :  a list with an even number of elements
						; effect:  finds all the variables appearing in the
						;          codomain of the unifier
						; value :  a list of dotted pairs (var . sort)
  (let (CODOMAIN VARIABLES)
    (SETQ CODOMAIN  (SMAPCAR #'identity #'CDDR (CDR UNIFIER))
	  VARIABLES (DT-TERMLIST.VARIABLES CODOMAIN))
    (MAPCAR #'(LAMBDA (VAR) (CONS VAR (DT-VARIABLE.SORT VAR))) VARIABLES)))


(DEFUN PO=DOUBLE.LITERAL (CLAUSE.REMAININGLITNO.DELETEDLITNO.RULE FILE)
  (PO=CLAUSE (CAR CLAUSE.REMAININGLITNO.DELETEDLITNO.RULE) FILE)
  (format file "~&~vT(DOUBLE.LITERAL ~{~A ~}~A)"
	  po*indentation (REST CLAUSE.REMAININGLITNO.DELETEDLITNO.RULE)
	  (FIRST CLAUSE.REMAININGLITNO.DELETEDLITNO.RULE)))


(DEFUN PO=REWRITE (RULE.CLAUSE.LITNO FILE)	; edited:  6-jul-83 15:47:07  by cl
						; input :  a list and a file open for output
						; effect:  prints clause as described in po=clause,
						;          prints (REWRITE rule litno clause)
						; value :  undefined
  (let ((rule (first rule.clause.litno))
	(clause (second rule.clause.litno))
	(litno (third rule.clause.litno)))
    (format FILE "~%(~A " clause)
    (PO=CLAUSE clause FILE)
    (format file "~% (rewrite (~A ~A) ~A))" clause litno rule)))


(DEFUN PO=REWRITE.SYMMETRY (CLAUSE. FILE)	; edited:  9-jul-84 10:24:06  by cl
						; input :  a list and a file open for output
						; effect:  prints clause as described in PO=CLAUSE,
						;          prints (REWRITE.SYMMETRY clause)
						; value :  undefined
  (PO=CLAUSE (CAR CLAUSE.) FILE)
  (format file "~&~vT(REWRITE.SYMMETRY ~A)" po*indentation (CAR CLAUSE.)))


(DEFUN PO=CLAUSE (CLAUSE FILE &optional record.flag)
						      ; Edited:  02-APO-1992 21:24
						      ; Authors: PRCKLN
						      ; input:  a clause, and a file name.
						      ; effect: prints the given information in the format:
						      ;         (CLAUSE address pname parents variables.sorts literals)
						      ; remark: it is assumed that the position is correct before.
						      ; value:  undefined.
  (format file "(CLAUSE (")
  (PROGN (MAPC #'(LAMBDA (VAR)			      ; Variables and sorts
		   (format file "(~A ~A)" (dt-pname var) (DT-VARIABLE.SORT VAR)))
	       (DS-CLAUSE.VARIABLES CLAUSE))
	 (PRINC ")" FILE))
  (let ((indices nil))
    (dotimes (litno (DS-CLAUSE.NOLIT CLAUSE))
      (push (ds-clause.lit.getprop clause (1+ litno) 'index) indices)
      (let ((pname (dt-pname (cons (ds-clause.predicate  clause (1+ litno))
				   (ds-clause.termlist clause (1+ litno))))))
	(when (ds-sign.is.negative (DS-CLAUSE.SIGN CLAUSE (1+ LITNO)))
	  (setq pname (list 'not pname)))
	(format file " ~A" pname)))
    (when record.flag (push (cons clause (nreverse indices)) po*indices)))

  (PRINC ")" FILE))



(DEFUN PO=LINK.COLOURS (FILE)
  (format file "~&~%(LINK.COLOURS ~A)" (DS-LINK.COLOURS.FOR 'ALL)))
