;;; -*- Package: MKRP; Syntax: Common-Lisp; Mode: LISP -*-

#| Copyright (C) 1991 AG Siekmann, 
                      Fachbereich Informatik, Universitaet des Saarlandes, 
                      Saarbruecken, Germany

This file is part of Markgraf Karl Refutation Procedure (MKRP).

MKRP is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  

Everyone is granted permission to copy, modify and redistribute
MKRP, but only if the it is not used for military purposes or any
military research. It is also forbidden to use MKRP in nuclear plants
or nuclear research, and for verifying programs in military 
and nuclear research.  A copy of this license is
supposed to have been given to you along with MKRP so you
can know your rights and responsibilities.  
Among other things, the copyright notice
must be preserved on all copies.  |#

(IN-PACKAGE "MARKGRAF-KARL" :use '("CL") :nicknames '("MKRP"))


(DEFUN PSIM-PRESIMPLIFICATION (EXPRESSION)
						; INPUT:  FOPC EXPRESSION CONTAINING LITERALS, THE
						;         JUNCTORS NOT, AND, OR, IMPL, EQV,  THE
						;         QUANTIFIERS ALL, EX.
						;         SOME OF THE LITERALS MAY BE TRUTH VALUES,
						;         I.E. (+ TRUE NIL ) OR (+ FALSE NIL).
						;         EXPRESSION IS IN PREFIX FORM EXACTLY
						;         PARANTHESIZED.
						; VALUE:  EQUIVALENT EXPRESSION CONTAINING NEITHER
						;         TRUTH VALUES NOR 'OBVIOUSLY' TRUE OR FALSE
						;         SUBEXPRESSIONS, OR CONSISTING OF EXACTLY
						;         ONE TRUTH VALUE. IN THE LATTER CASE THE
						;         PARENTHESES ARE OMITTED, I.E. THE VALUE IS
						;         EITHER OF THE ATOMS TRUE OR FALSE.
						; REMARK: DESTRUCTIVE ON EXPRESSION
  (cond ((opt-get.option gen_presimplification)
	 (when EXPRESSION (SETQ EXPRESSION (PSIM=SIMPLIFY.TRUTH.VALUES (PSIM=CREATE.TRUTH.VALUES EXPRESSION)))) 
	 (COND ((DT-PREDICATE.IS.TRUE (PSIM=LEFT EXPRESSION)) 'TRUE)
	       ((DT-PREDICATE.IS.FALSE (PSIM=LEFT EXPRESSION)) 'FALSE)
	       (T EXPRESSION)))
	(t expression)))

(DEFUN PSIM-APPLY.DEFINITIONS (AXIOMS THEOREMS)
						; EDITED: 3. 5. 1984
						; INPUT:  TWO LISTS OF FOPC EXPRESSIONS
						; EFFECT: AXIOMS REPRESENTING PREDICATE DEFINITIONS
						;         ARE REMOVED AND THE DEFINITIONS ARE
						;         EXPANDED.
						; VALUE:  (LIST AXIOMS THEOREMS)
  (if (eq (opt-get.option gen_presimplification) t)
      (let (DEFINITION DEF)
	(WHILE (SETQ DEF (CAR (MEMBER-IF #'(LAMBDA (AXIOM) (SETQ DEFINITION (PSIM=DEFINITION AXIOM))) AXIOMS)))
	  (SETQ AXIOMS (DELETE DEF AXIOMS))
	  (MAPL #'(LAMBDA (AXIOM) (RPLACA AXIOM (PSIM=REPLACE DEFINITION (CAR AXIOM)))) AXIOMS)
	  (MAPL #'(LAMBDA (AXIOM) (RPLACA AXIOM (PSIM=REPLACE DEFINITION (CAR AXIOM)))) THEOREMS))
	(LIST AXIOMS THEOREMS))
      (list axioms theorems)))

(DEFUN PSIM=DEFINITION (AXIOM)
  ; EDITED: 3. 5. 1984
  ; INPUT:  AN FOPC AXIOM
  ; VALUE:  NIL IF AXIOM IS NO PREDICATE DEFINITION,
  ;         ELSE A LIST
  ;         ((PREDICATE . TERMLIST) DEFINIENDUM)
  (CASE (PSIM=OP AXIOM)
    (ALL (let ((def (psim=definition (PSIM=RIGHT AXIOM))))
	   (if (and def (member (psim=left axiom) (rest (first def))))
	       def
	       nil)))	     
    (EQV (let (DEF)
	   (COND ((SETQ DEF (PSIM=DEFINITION.IS (PSIM=LEFT AXIOM) (PSIM=RIGHT AXIOM)))
		  (LIST DEF (PSIM=REMOVE.PROPERTIES (PSIM=RIGHT AXIOM))))
		 ((SETQ DEF (PSIM=DEFINITION.IS (PSIM=RIGHT AXIOM) (PSIM=LEFT AXIOM)))
		  (LIST DEF (PSIM=REMOVE.PROPERTIES (PSIM=LEFT AXIOM)))))))
    (OTHERWISE NIL)))

(DEFUN PSIM=DEFINITION.IS (LEFT RIGHT)
  ; EDITED: 3. 5. 1984
  ; INPUT:  TWO SIDES OF AN EQUIVALENCE
  ; VALUE:  (PREDICATE . TERMLIST) IF THE EQUIVALENCE
  ;         REPRESENTS A PREDICATE DEFINITION, ELSE NIL.
  (cond ((eql (PSIM=op LEFT) '+)
	 (let* ((PREDICATE (PSIM=LEFT LEFT))
		(TERMLIST (PSIM=RIGHT LEFT))
		(SORTS (DT-PREDICATE.DOMAINSORTS PREDICATE))
		TERM)
	   (COND ((AND (EVERYL #'(LAMBDA (TAIL sortstail)
				   (SETQ TERM (CAR TAIL))
				   (AND (DT-VARIABLE.IS TERM)
					(NOT (MEMBER TERM (CDR TAIL)))
					(EQL (DT-VARIABLE.SORT TERM) (CAR SORTStail))))
			       TERMLIST sorts)
		       (NOT (INSIDE PREDICATE RIGHT)))
		  (CONS PREDICATE TERMLIST)))))))

(DEFUN PSIM=REPLACE (DEFINITION EXPRESSION)
  ; EDITED: 3. 5. 1984
  ; INPUT:  THE OUTPUT OF PSIM=DEFINITION AND A
  ;         FOPC AXIOM.
  ; EFFECT: THE DEFINITION IS APPLIED TO AXIOM
  ; VALUE:  the new expression.
  (CASE (PSIM=OP EXPRESSION)
    (NOT (PSIM=PUT=LEFT EXPRESSION (PSIM=REPLACE DEFINITION (PSIM=LEFT EXPRESSION))))
    ((ALL EX) (PSIM=PUT=RIGHT EXPRESSION (PSIM=REPLACE DEFINITION (PSIM=RIGHT EXPRESSION))))
    ((AND OR IMPL EQV)
     (PSIM=PUT=LEFT EXPRESSION (PSIM=REPLACE DEFINITION (PSIM=LEFT EXPRESSION)))
     (PSIM=PUT=RIGHT EXPRESSION (PSIM=REPLACE DEFINITION (PSIM=RIGHT EXPRESSION))))
    ((+ -) (when (EQL (PSIM=LEFT EXPRESSION) (CAAR DEFINITION))
	     (let ((NEW (SUBPAIR (PSIM=RIGHT EXPRESSION) (CDAR DEFINITION)
				 (norm-COPY (SECOND DEFINITION)))))
	       (SETQ EXPRESSION (CASE (PSIM=OP EXPRESSION) (+ NEW) (- (NORM-CREATE.FORMULA 'NOT NEW)) (OTHERWISE NIL))))))
    (OTHERWISE (ERROR "illegal expression in PSIM=REPLACE: ~A" EXPRESSION)))
  EXPRESSION)

(DEFUN PSIM=REMOVE.PROPERTIES (EXPRESSION)
  ; EDITED: 2. 8. 1984
  ; INPUT:  A FOPC EXPRESSION
  ; EFFECT: THE PROPERTY CELLS OF THE LITERALS ARE
  ;         DESTRUCTIVELY SET TO NIL.
  ; VALUE:  THE MODIFIED EXPRESSION.
  (CASE (PSIM=OP EXPRESSION) (NOT (PSIM=PUT=LEFT EXPRESSION (PSIM=REMOVE.PROPERTIES (PSIM=LEFT EXPRESSION))))
    ((ALL EX) (PSIM=PUT=RIGHT EXPRESSION (PSIM=REMOVE.PROPERTIES (PSIM=RIGHT EXPRESSION))))
    ((AND OR IMPL EQV) (PSIM=PUT=LEFT EXPRESSION (PSIM=REMOVE.PROPERTIES (PSIM=LEFT EXPRESSION)))
      (PSIM=PUT=RIGHT EXPRESSION (PSIM=REMOVE.PROPERTIES (PSIM=RIGHT EXPRESSION))))
    ((+ -) (RPLACa (CdDDR EXPRESSION) NIL)) (OTHERWISE (ERROR "illegal expression in psim=remove.properties: ~A" EXPRESSION)))
  EXPRESSION)

(DEFMACRO PSIM=CREATE.TRUTH.VALUES (EXPRESSION)
  ; INPUT:  EXPRESSION CONTAINING JUNCTORS, QUANTIFIERS
  ;         LITERALS, TRUTH VALUES.
  ; VALUE:  EQUIVALENT EXPRESSION WHERE 'OBVIOUSLY'
  ;         TRUE OR FALSE SUBEXPRESSIONS ARE REPLACED
  ;         BY THE RESPECTIVE TRUTH VALUES.
  ; REMARK: RULES IMPLEMENTED SO FAR ARE:
  ;         NONE
  `(PSIM=NORMALIZE.SIGN ,EXPRESSION))

(DEFUN PSIM=SIMPLIFY.TRUTH.VALUES (EXPRESSION)
						; INPUT:  EXPRESSION CONTAINING JUNCTORS, QUANTIFIERS
						;         LITERALS, TRUTH VALUES.
						; VALUE:  EQUIVALENT EXPRESSION CONTAINING NO TRUTH
						;         VALUES OR CONSISTING OF EXACTLY ONE TRUTH
						;         VALUE.
  (CASE (PSIM=OP EXPRESSION)
    (NOT (PSIM=SIMPLIFY.TRUTH.VALUES.NOT EXPRESSION))
    (AND (PSIM=SIMPLIFY.TRUTH.VALUES.AND EXPRESSION))
    (OR (PSIM=SIMPLIFY.TRUTH.VALUES.OR EXPRESSION))
    (IMPL (PSIM=SIMPLIFY.TRUTH.VALUES.IMPL EXPRESSION))
    (EQV (PSIM=SIMPLIFY.TRUTH.VALUES.EQV EXPRESSION))
    ((ALL EX) (PSIM=SIMPLIFY.TRUTH.VALUES.QUANTIFIER EXPRESSION))
    (+ EXPRESSION)
    (OTHERWISE (ERROR "ILLEGAL EXPRESSION IN PSIM=SIMPLIFY.TRUTH.VALUES:~A" EXPRESSION))))

(DEFUN PSIM=NORMALIZE.SIGN (EXPRESSION)
  ; EDITED: 10. 8. 1984
  ; INPUT:  A FOPC EXPRESSION IN PREFIX FORM
  ; EFFECT: LITERALS (- TRUE) ARE REPLACED BY (+ FALSE),
  ;         LITERALS (- FALSE) ARE REPLACED BY (+ TRUE).
  ; VALUE:  THE NORMALIZED EXPRESSION.
  (CASE (PSIM=OP EXPRESSION)
    (-
      (COND ((DT-PREDICATE.IS.TRUE (PSIM=LEFT EXPRESSION)) (NORM-CREATE.FORMULA '+ (DT-PREDICATE.FALSE)))
        ((DT-PREDICATE.IS.FALSE (PSIM=LEFT EXPRESSION)) (NORM-CREATE.FORMULA '+ (DT-PREDICATE.TRUE))) (T EXPRESSION)))
    (+ EXPRESSION)
    ((AND OR IMPL EQV) (PSIM=PUT=LEFT EXPRESSION (PSIM=NORMALIZE.SIGN (PSIM=LEFT EXPRESSION)))
      (PSIM=PUT=RIGHT EXPRESSION (PSIM=NORMALIZE.SIGN (PSIM=RIGHT EXPRESSION))) EXPRESSION)
    (NOT (PSIM=PUT=LEFT EXPRESSION (PSIM=NORMALIZE.SIGN (PSIM=LEFT EXPRESSION))) EXPRESSION)
    ((ALL EX) (PSIM=PUT=RIGHT EXPRESSION (PSIM=NORMALIZE.SIGN (PSIM=RIGHT EXPRESSION))) EXPRESSION)
    (OTHERWISE (ERROR "ILLEGAL OPERATOR IN PSIM=NORMALIZE.SIGN~A" (PSIM=OP EXPRESSION)))))

(DEFUN PSIM=SIMPLIFY.TRUTH.VALUES.NOT (EXPRESSION)
  ; INPUT:  EXPRESSION HAS FORM (NOT LEFT)
  ;         WHERE LEFT IS AN EXPRESSION CONTAINING
  ;         JUNCTORS, QUANTIFIERS, LITERALS, TRUTH
  ;         VALUES.
  ; VALUE:  EQUIVALENT EXPRESSION CONTAINING NO TRUTH
  ;         VALUE OR CONSISTING OF EXACTLY ONE TRUTH
  ;         VALUE.
  (let ((LEFT (PSIM=SIMPLIFY.TRUTH.VALUES (PSIM=LEFT EXPRESSION))))
    (PSIM=PUT=LEFT EXPRESSION LEFT)
    (SETQ LEFT (PSIM=LEFT LEFT))
    (COND ((DT-PREDICATE.IS.TRUE LEFT) (NORM-CREATE.FORMULA '+ (DT-PREDICATE.FALSE)))
	  ((DT-PREDICATE.IS.FALSE LEFT) (NORM-CREATE.FORMULA '+ (DT-PREDICATE.TRUE)))
	  (T EXPRESSION))))

(DEFUN PSIM=SIMPLIFY.TRUTH.VALUES.AND (EXPRESSION)
  ; INPUT:  EXPRESSION HAS FORM  (AND LEFT RIGHT)
  ;         WHERE LEFT AND RIGHT ARE EXPRESSIONS
  ;         CONTAINING JUNCTORS, QUANTIFIERS, LITERALS,
  ;         AND TRUTH VALUES.
  ; VALUE:  EQUIVALENT EXPRESSION CONTAINING NO TRUTH
  ;         VALUE OR CONSISTING OF EXACTLY ONE TRUTH
  ;         VALUE.
  (let ((LEFT (PSIM=SIMPLIFY.TRUTH.VALUES (PSIM=LEFT EXPRESSION)))
	(RIGHT (PSIM=SIMPLIFY.TRUTH.VALUES (PSIM=RIGHT EXPRESSION))))
    (COND ((DT-PREDICATE.IS.TRUE (PSIM=LEFT LEFT)) RIGHT) ((DT-PREDICATE.IS.FALSE (PSIM=LEFT LEFT)) LEFT)
	  ((DT-PREDICATE.IS.TRUE (PSIM=LEFT RIGHT)) LEFT) ((DT-PREDICATE.IS.FALSE (PSIM=LEFT RIGHT)) RIGHT)
	  (T (PSIM=PUT=LEFT EXPRESSION LEFT) (PSIM=PUT=RIGHT EXPRESSION RIGHT) EXPRESSION))))

(DEFUN PSIM=SIMPLIFY.TRUTH.VALUES.OR (EXPRESSION)
  ; INPUT:  EXPRESSION HAS FORM  (OR  LEFT RIGHT)
  ;         WHERE LEFT AND RIGHT ARE EXPRESSIONS
  ;         CONTAINING JUNCTORS, QUANTIFIERS, LITERALS,
  ;         AND TRUTH VALUES.
  ; VALUE:  EQUIVALENT EXPRESSION CONTAINING NO TRUTH
  ;         VALUE OR CONSISTING OF EXACTLY ONE TRUTH
  ;         VALUE.
  (let ((LEFT (PSIM=SIMPLIFY.TRUTH.VALUES (PSIM=LEFT EXPRESSION)))
	(RIGHT (PSIM=SIMPLIFY.TRUTH.VALUES (PSIM=RIGHT EXPRESSION))))
    (COND ((DT-PREDICATE.IS.TRUE (PSIM=LEFT LEFT)) LEFT) ((DT-PREDICATE.IS.FALSE (PSIM=LEFT LEFT)) RIGHT)
	  ((DT-PREDICATE.IS.TRUE (PSIM=LEFT RIGHT)) RIGHT) ((DT-PREDICATE.IS.FALSE (PSIM=LEFT RIGHT)) LEFT)
	  (T (PSIM=PUT=LEFT EXPRESSION LEFT) (PSIM=PUT=RIGHT EXPRESSION RIGHT) EXPRESSION))))

(DEFUN PSIM=SIMPLIFY.TRUTH.VALUES.IMPL (EXPRESSION)
  ; INPUT:  EXPRESSION HAS FORM  (IMPL LEFT RIGHT)
  ;         WHERE LEFT AND RIGHT ARE EXPRESSIONS
  ;         CONTAINING JUNCTORS, QUANTIFIERS, LITERALS,
  ;         AND TRUTH VALUES.
  ; VALUE:  EQUIVALENT EXPRESSION CONTAINING NO TRUTH
  ;         VALUE OR CONSISTING OF EXACTLY ONE TRUTH
  ;         VALUE.
  (let ((LEFT (PSIM=SIMPLIFY.TRUTH.VALUES (PSIM=LEFT EXPRESSION)))
	(RIGHT (PSIM=SIMPLIFY.TRUTH.VALUES (PSIM=RIGHT EXPRESSION))))
    (COND ((DT-PREDICATE.IS.TRUE (PSIM=LEFT LEFT)) RIGHT)
	  ((DT-PREDICATE.IS.FALSE (PSIM=LEFT LEFT)) (NORM-CREATE.FORMULA '+ (DT-PREDICATE.TRUE)))
	  ((DT-PREDICATE.IS.TRUE (PSIM=LEFT RIGHT)) RIGHT)
	  ((DT-PREDICATE.IS.FALSE (PSIM=LEFT RIGHT)) (NORM-CREATE.FORMULA 'NOT LEFT))
	  (T (PSIM=PUT=LEFT EXPRESSION LEFT) (PSIM=PUT=RIGHT EXPRESSION RIGHT) EXPRESSION))))

(DEFUN PSIM=SIMPLIFY.TRUTH.VALUES.EQV (EXPRESSION)
  ; INPUT:  EXPRESSION HAS FORM  (EQV LEFT RIGHT)
  ;         WHERE LEFT AND RIGHT ARE EXPRESSIONS
  ;         CONTAINING JUNCTORS, QUANTIFIERS, LITERALS,
  ;         AND TRUTH VALUES.
  ; VALUE:  EQUIVALENT EXPRESSION CONTAINING NO TRUTH
  ;         VALUE OR CONSISTING OF EXACTLY ONE TRUTH
  ;         VALUE.
  (let ((LEFT (PSIM=SIMPLIFY.TRUTH.VALUES (PSIM=LEFT EXPRESSION)))
	(RIGHT (PSIM=SIMPLIFY.TRUTH.VALUES (PSIM=RIGHT EXPRESSION)))
	LEFT.OPERAND
	RIGHT.OPERAND)
    (SETQ LEFT.OPERAND (PSIM=LEFT LEFT)) (SETQ RIGHT.OPERAND (PSIM=LEFT RIGHT))
    (COND ((AND (OR (EQL LEFT.OPERAND (DT-PREDICATE.TRUE)) (EQL LEFT.OPERAND (DT-PREDICATE.FALSE)))
		(OR (EQL RIGHT.OPERAND (DT-PREDICATE.TRUE)) (EQL RIGHT.OPERAND (DT-PREDICATE.FALSE)))
		(COND ((EQL LEFT.OPERAND RIGHT.OPERAND) (NORM-CREATE.FORMULA '+ (DT-PREDICATE.TRUE))) (T (NORM-CREATE.FORMULA '+ (DT-PREDICATE.FALSE))))))
	  ((DT-PREDICATE.IS.TRUE LEFT.OPERAND) RIGHT) ((DT-PREDICATE.IS.FALSE LEFT.OPERAND) (NORM-CREATE.FORMULA 'NOT RIGHT))
	  ((DT-PREDICATE.IS.TRUE RIGHT.OPERAND) LEFT) ((DT-PREDICATE.IS.FALSE RIGHT.OPERAND) (NORM-CREATE.FORMULA 'NOT LEFT))
	  (T (PSIM=PUT=LEFT EXPRESSION LEFT) (PSIM=PUT=RIGHT EXPRESSION RIGHT) EXPRESSION))))

(DEFUN PSIM=SIMPLIFY.TRUTH.VALUES.QUANTIFIER (EXPRESSION)
  ; INPUT:  EXPRESSION HAS FORM (QUANTIFIER VAR RIGHT)
  ;         WHERE RIGHT IS AN EXPRESSION CONTAINING
  ;         JUNCTORS, QUANTIFIERS, LITERALS, TRUTH
  ;         VALUES.
  ; VALUE:  EQUIVALENT EXPRESSION CONTAINING NO TRUTH
  ;         VALUE OR CONSISTING OF EXACTLY ONE TRUTH
  ;         VALUE.
  (PROG ((RIGHT (PSIM=SIMPLIFY.TRUTH.VALUES (PSIM=RIGHT EXPRESSION)))) (PSIM=PUT=RIGHT EXPRESSION RIGHT)
    (RETURN
      (COND ((OR (DT-PREDICATE.IS.TRUE (PSIM=LEFT RIGHT)) (DT-PREDICATE.IS.FALSE (PSIM=LEFT RIGHT))) RIGHT) (T EXPRESSION)))))

(DEFMACRO PSIM=OP (X)
  ; |EDITED:| "23-NOV-81 13:27:07")
  ; INPUT:  X HAS FORM (OPERATOR LEFT RIGHT).
  ; VALUE:  OPERATOR.
  `(CAR ,X))

(DEFMACRO PSIM=PUT=OP (X OP)
  ; |EDITED:| "23-NOV-81 13:34:57")
  ; INPUT:  X HAS FORM (OPERATOR LEFT RIGHT).
  ; EFFECT: REPLACES OPERATOR IN X BY OP.
  ; VALUE:  CHANGED X.
  `(RPLACA ,X ,OP))

(DEFMACRO PSIM=LEFT (X)
						; EDITED: 23-NOV-81 13:28:04
						; INPUT:  X HAS FORM (OPERATOR LEFT RIGHT).
						; VALUE:  LEFT.
  `(SECOND ,X))

(DEFMACRO PSIM=PUT=LEFT (X LEFT)
  ; |EDITED:| "23-NOV-81 13:35:43")
  ; INPUT:  X HAS FORM (OPERATOR OLDLEFT RIGHT).
  ; EFFECT: REPLACES OLDLEFT IN X BY LEFT.
  ; VALUE:  CHANGED X.
  `(RPLACA (CDR ,X) ,LEFT))

(DEFMACRO PSIM=RIGHT (X)
  ; |EDITED:| "23-NOV-81 13:28:33")
  ; INPUT:  X HAS FORM (OPERATOR LEFT RIGHT).
  ; VALUE:  RIGHT.
  `(THIRD ,X))

(DEFMACRO PSIM=PUT=RIGHT (X RIGHT)
  ; |EDITED:| "23-NOV-81 13:36:09")
  ; INPUT:  X HAS FORM (OPERATOR LEFT OLDRIGHT).
  ; EFFECT: REPLACES OLDRIGHT IN X BY RIGHT.
  ; VALUE:  CHANGED X.
  `(RPLACA (CDDR ,X) ,RIGHT))

