;;; -*- Package: MKRP; Syntax: Common-Lisp; Mode: LISP -*-

(IN-PACKAGE "MKRP" :use '("CL"))

(defparameter norm*literal.index 0)

(defun norm=set.index (expression)
  (CASE (NORM=OP EXPRESSION)
    ((+ -)
     (norm=put=index expression (incf norm*literal.index))
     expression)
    ((AND OR IMPL EQV AND! OR! IMPL! EQV!)
     (norm=set.index (NORM=LEFT EXPRESSION))
     (norm=set.index (NORM=right EXPRESSION))
     EXPRESSION)
    (NOT (norm=set.index (NORM=LEFT EXPRESSION))
	 EXPRESSION)
    ((ALL EX)
     (norm=set.index (NORM=right EXPRESSION))
     expression)
    (OTHERWISE (ERROR "Illegal expression: ~A" EXPRESSION))))

(DEFUN NORM-NORMALIZATION (EXPRESSION SPLIT.DEPTH)
  ;; INPUT: EXPRESSION IS A FOPC FORMULA ACCORDING TO THE
  ;;        FOLLOWING GRAMMAR:
  ;;          <FORMULA> ::=  <LITERAL>
  ;;                         | (NOT <FORMULA>)
  ;;                         | (<JUNCTOR> <FORMULA> <FORMULA>)
  ;;                         | (<QUANTIFIER> <VARIABLE> <FORMULA>)
  ;;          <LITERAL> ::=
  ;;             (<SIGN> <PREDICATE> (<TERMS>) <S-EXPR>)
  ;;          <SIGN> ::=  + | -
  ;;          <TERMS> ::= <EMPTY> | <TERM> <TERMS>
  ;;          <TERM> ::= <VARIABLE> | <CONSTANT>
  ;;             | (<FUNCTION> <TERMS>)
  ;;          <JUNCTOR> ::= AND | OR | IMPL | EQV
  ;;          <QUANTIFIER> ::= ALL | EX
  ;;        <VARIABLE>, <CONSTANT>, <FUNCTION>, AND
  ;;        <PREDICATE> DENOTE DT-OBJECTS OF THE RESPEC-
  ;;        TIVE KIND.
  ;;        ALL VARIABLES ARE BOUND BY QUANTIFIERS.
  ;;        DIFFERENT SUBEXPRESSIONS MAY BE EQUAL BUT
  ;;        NOT EQ.
  ;;        SPLIT.DEPTH IS T, INTEGER >= 0 OR NIL.
  ;; EFFECT:EXPRESSION IS CONVERTED TO CLAUSAL FORM.
  ;;        IF SPLIT.DEPTH IS NON-NIL, A FORM ALLOWING
  ;;        FOR A MAXIMUM OF TOP-OR-SPLITS IN THE UPPER
  ;;        SPLIT.DEPTH LEVELS IS CREATED (SPLIT.DEPTH
  ;;        DEFAULTS TO NORM*MAX.SPLIT.DEPTH IF IT IS T).
  ;;        TO THAT END THE EXPRESSION MAY BE PARTIALLY
  ;;        TRANSFORMED TO DNF. EQUIVALENCE REPLACEMENT
  ;;        IS OPTIMIZED WITH RESPECT TO CNF OR DNF.
  ;;        NEW OBJECTS ARE CREATED FOR VARIABLES AND
  ;;        SKOLEM CONSTANTS AND FUNCTIONS.
  ;; VALUE: LIST OF USUALLY ONE CLAUSELIST, FOR NON-NIL
  ;;        SPLIT.DEPTH POSSIBLY SEVERAL CLAUSELISTS.
  ;;        EACH CLAUSELIST IS A LIST OF CLAUSES WHICH IN
  ;;        TURN IS A LIST OF LITERALS IN INPUT FORMAT.
  ;;        ALL CLAUSES ARE VARIABLE DISJOINT.
  ;; REMARK:DESTRUCTIVE ON EXPRESSION.
  (setq expression (norm=set.index expression))
  (SETQ EXPRESSION (NORM=NORMALIZE.1 EXPRESSION SPLIT.DEPTH))
  (SETQ EXPRESSION (MAPCAR (FUNCTION NORM=NORMALIZE.2) EXPRESSION)) EXPRESSION)

(DEFUN NORM-NEW.FORMULA (EXPRESSION SPLIT.DEPTH)
  ;; INPUT:  FOPC FORMULA AND BOOLEAN // INTEGER >= 0
  ;;         AS IN NORM-NORMALIZATION.
  ;; EFFECT: MODIFIES THE MODULE STATE SUCH THAT
  ;;         SUBSEQUENT CALLS OF NORM-NEXT.SPLITPART
  ;;         WILL RETURN THE SPLIT PARTS OF EXPRESSION
  ;;         ONE AT A TIME.
  ;;         IF A FORMULA EXCEEDS THE LIMITS GIVEN IN THE
  ;;         COND CONDITION WHILE SPLIT.DEPTH IS T, IT IS
  ;;         WRITTEN TO THE FILE (EVAL NORM*FILE.NAME).
  ;; VALUE:  INTERMEDIATE FORM OF EXPRESSION.
  (setq expression (norm=set.index expression))
  (SETQ EXPRESSION (NORM=NORMALIZE.1 EXPRESSION SPLIT.DEPTH))
  (let ((EXPRESSION.LENGTH (LIST-LENGTH EXPRESSION))
	(EXPRESSION.SIZE (CONSES EXPRESSION))
	STREAM)
    (SETQ NORM*EXPRESSION.LENGTH EXPRESSION.LENGTH
	  NORM*EXPRESSION.LENGTH.ORIGINAL EXPRESSION.LENGTH)
    (COND ((AND SPLIT.DEPTH
		(opt-get.option gen_MIN.EXPRESSION.LENGTH.FOR.FILE)
		(opt-get.option gen_MIN.EXPRESSION.SIZE.FOR.FILE)
		(or (not (numberp (opt-get.option gen_MIN.EXPRESSION.LENGTH.FOR.FILE)))
		    (< (opt-get.option gen_MIN.EXPRESSION.LENGTH.FOR.FILE) EXPRESSION.LENGTH))
		(or (not (numberp (opt-get.option gen_MIN.EXPRESSION.SIZE.FOR.FILE)))
		    (< (opt-get.option gen_MIN.EXPRESSION.SIZE.FOR.FILE) EXPRESSION.SIZE)))
	   (SETQ STREAM (mkrp-OPENOUT (mkrp-make.pathname t "tempnormsplits" "lisp" nil) nil))
	   (MAPC #'(LAMBDA (SPLITPART) (format stream "~A~%" SPLITPART)) EXPRESSION)
	   (SETQ NORM*EXPRESSION 'FILE)
	   (CLOSEFILE STREAM)
	   (setq norm*file.version (mkrp-OPENIN (mkrp-make.pathname t "tempnormsplits" "lisp" nil))))
	  (T (SETQ NORM*EXPRESSION EXPRESSION)))
    EXPRESSION))

(DEFUN NORM-NEXT.SPLITPART ()
  ;; INPUT:  NONE.
  ;;         NORM-NEW.FORMULA OUGHT TO HAVE BEEN CALLED
  ;;         PRIOR TO THIS CALL, ENTERING A FOPC
  ;;         EXPRESSION X.
  ;; EFFECT: X IS FURTHER NORMALIZED TO AN EXTENT
  ;;         SUFFICING TO GENERATE ONE SPLIT PART. THE
  ;;         REMAINDER OF X IS KEPT SUCH THAT THE NEXT
  ;;         SPLIT PART CAN BE PRODUCED BY ANOTHER CALL
  ;;         TO THIS FUNCTION.
  ;; VALUE:  A CLAUSE LIST REPRESENTING ONE SPLIT PART
  ;;         OF X, FORMAT AS IN NORM-NORMALIZATION;
  ;;         NIL IF THERE WAS NO FURTHER PART OF X.
  (let (RESULT)
    (when NORM*EXPRESSION
      (SETQ NORM*EXPRESSION.LENGTH (1- NORM*EXPRESSION.LENGTH))
      (CASE NORM*EXPRESSION
	(FILE
	  (COND
	    ((mkrp-INSTREAMP NORM*FILE.VERSION) (SETQ RESULT (NORM=NORMALIZE.2 (READ NORM*FILE.VERSION)))
	     (COND ((ZEROP NORM*EXPRESSION.LENGTH) (CLOSEFILE NORM*FILE.VERSION))))
	    (T NIL)))
	(OTHERWISE
	  (PROGN (SETQ RESULT (NORM=NORMALIZE.2 (CAR NORM*EXPRESSION)))
		 (SETQ NORM*EXPRESSION (CDR NORM*EXPRESSION))))))
    result))
#|
(DEFUN NORM-NEXT.SPLITPART ()
  ;; INPUT:  NONE.
  ;;         NORM-NEW.FORMULA OUGHT TO HAVE BEEN CALLED
  ;;         PRIOR TO THIS CALL, ENTERING A FOPC
  ;;         EXPRESSION X.
  ;; EFFECT: X IS FURTHER NORMALIZED TO AN EXTENT
  ;;         SUFFICING TO GENERATE ONE SPLIT PART. THE
  ;;         REMAINDER OF X IS KEPT SUCH THAT THE NEXT
  ;;         SPLIT PART CAN BE PRODUCED BY ANOTHER CALL
  ;;         TO THIS FUNCTION.
  ;; VALUE:  A CLAUSE LIST REPRESENTING ONE SPLIT PART
  ;;         OF X, FORMAT AS IN NORM-NORMALIZATION;
  ;;         NIL IF THERE WAS NO FURTHER PART OF X.
  (let (RESULT)
    (when NORM*EXPRESSION
      (SETQ NORM*EXPRESSION.LENGTH NORM*EXPRESSION.LENGTH)
      (CASE NORM*EXPRESSION
	(FILE
	  (COND
	    ((INSTREAMP NORM*FILE.VERSION) (SETQ RESULT (NORM=NORMALIZE.2 (READ NORM*FILE.VERSION)))
	     (COND ((ZEROP NORM*EXPRESSION.LENGTH) (CLOSEFILE NORM*FILE.VERSION))))
	    (T NIL)))
	(OTHERWISE
	  (PROGN (SETQ RESULT (NORM=NORMALIZE.2 (CAR NORM*EXPRESSION)))
		 (if (rest NORM*EXPRESSION)
		     (SETQ NORM*EXPRESSION (norm=split.lazy (rest NORM*EXPRESSION) t))
		     (SETQ NORM*EXPRESSION.LENGTH 0))))))
    result))|#

(DEFUN NORM-SEVERAL.SPLITPARTS? ()
  ;; EDITED: 22-APR-83 11:27:03        MW
  ;; INPUT: NONE.
  ;; VALUE: T IF PRESENTLY THERE ARE TWO OR MORE
  ;;        SPLITPARTS LEFT, OTHERWISE NIL.
  (> NORM*EXPRESSION.LENGTH 1))

(DEFUN NORM-more.SPLITPARTS? ()
  ;; EDITED: 22-APR-83 11:27:03        MW
  ;; INPUT: NONE.
  ;; VALUE: T IF PRESENTLY THERE ARE TWO OR MORE
  ;;        SPLITPARTS LEFT, OTHERWISE NIL.
  (> NORM*EXPRESSION.LENGTH 0))

(DEFUN NORM-PRENEX.FORM (EXPRESSION)
  ;; EDITED: 31-JUL-84 16:23:34        NE
  ;; INPUT:  EXPRESSION AS IN NORM-NORMALIZATION.
  ;; EFFECT: DESTRUCTIVELY CONVERTS EXPRESSION TO
  ;;         PRENEX FORM.
  ;; VALUE:  MODIFIED EXPRESSION.
  (SETQ EXPRESSION (NORM=REPLACE.EQV EXPRESSION NIL NIL)) (SETQ EXPRESSION (NORM=MOVE.QUANTIFIERS.OUTSIDE EXPRESSION)) EXPRESSION)

(DEFUN NORM-RESET ()
 ;; EDITED:| "19-NOV-81 13:09:50")
 ;; EFFECT: RESETS THE MODULE TO ITS INITIAL STATE
 ;; VALUE: UNDEFINED.
 (SETQ NORM*FUNCTION.COUNTER 0
       norm*literal.index 0
       NORM*CONSTANT.COUNTER 0
       NORM*EXPRESSION NIL)
 (COND
  ((AND NORM*FILE.VERSION (mkrp-INSTREAMP NORM*FILE.VERSION)) (CLOSEFILE NORM*FILE.VERSION) (SETQ NORM*FILE.VERSION NIL))))



(DEFUN NORM=NORMALIZE.1 (EXPRESSION SPLIT.DEPTH)
  ;; INPUT:  SAME AS IN NORM-NORMALIZATION.
  ;; EFFECT: DESTRUCTIVELY NORMALIZES EXPRESSION UP TO
  ;;         THE POINT AFTER SPLITTING.
  ;; VALUE:  LIST OF SUBFORMULAE REPRESENTING THE SPLIT
  ;;         PARTS. THE SUBFORMULAE CONTAIN LITERALS,
  ;;         ALL, AND, OR.
  (when (EQL SPLIT.DEPTH T) (SETQ SPLIT.DEPTH NORM*MAX.SPLIT.DEPTH))
  ;; NOW SPLIT.DEPTH IS NIL OR A NONNEGATIVE INTEGER
  (when (CONSP EXPRESSION)
    (COND
      (SPLIT.DEPTH (SETQ EXPRESSION (NORM=MOVE.QUANTIFIERS.INSIDE EXPRESSION))
		   ;; FIRST ATTEMPT TO MOVE QUANTIFIERS INSIDE
		   (SETQ EXPRESSION (NORM=MARK.SPLIT EXPRESSION SPLIT.DEPTH NIL))
		   ;; EXPRESSION PARTS TO BE SPLIT MARKED
		   ))
    (SETQ EXPRESSION (NORM=REPLACE.EQV EXPRESSION T NIL))
    ;; EQV REMOVED
    (SETQ EXPRESSION (NORM=REPLACE.IMPL EXPRESSION))
    ;; IMPL REMOVED
    (SETQ EXPRESSION (NORM=MOVE.NEGATIONS.TO.LITERALS EXPRESSION))
    ;; NOT REMOVED
    (SETQ EXPRESSION (NORM=MOVE.QUANTIFIERS.INSIDE EXPRESSION))
    ;; QUANTIFIERS MOVED TO INNERMOST SCOPES
    (SETQ EXPRESSION (NORM=SKOLEMIZE EXPRESSION NIL))
    ;; EX REMOVED, EX VARIABLES DROPPED.
    (COND
      (SPLIT.DEPTH (SETQ EXPRESSION (NORM=MOVE.QUANTIFIERS.INSIDE EXPRESSION))
		   ;; ATTEMPT TO MOVE ALL FURTHER INSIDE
		   (SETQ EXPRESSION (NORM=PREPARE.SPLIT EXPRESSION T))
		   ;; PSEUDO DNF CREATED, SPLITMARKS REMOVED
		   (SETQ EXPRESSION (NORM=SPLIT EXPRESSION))
		   ;; SPLITTING DONE
		   )
      (T (SETQ EXPRESSION (LIST EXPRESSION)))))
  EXPRESSION)


#|
(DEFUN NORM=NORMALIZE.1 (EXPRESSION SPLIT.DEPTH)
  ;; INPUT:  SAME AS IN NORM-NORMALIZATION.
  ;; EFFECT: DESTRUCTIVELY NORMALIZES EXPRESSION UP TO
  ;;         THE POINT AFTER SPLITTING.
  ;; VALUE:  LIST OF SUBFORMULAE REPRESENTING THE SPLIT
  ;;         PARTS. THE SUBFORMULAE CONTAIN LITERALS,
  ;;         ALL, AND, OR.
  (when (EQL SPLIT.DEPTH T) (SETQ SPLIT.DEPTH NORM*MAX.SPLIT.DEPTH))
  ;; NOW SPLIT.DEPTH IS NIL OR A NONNEGATIVE INTEGER
  (when (CONSP EXPRESSION)
    (COND
      (SPLIT.DEPTH (SETQ EXPRESSION (NORM=MOVE.QUANTIFIERS.INSIDE EXPRESSION))
		   ;; FIRST ATTEMPT TO MOVE QUANTIFIERS INSIDE
		   (SETQ EXPRESSION (NORM=MARK.SPLIT EXPRESSION SPLIT.DEPTH NIL))
		   ;; EXPRESSION PARTS TO BE SPLIT MARKED
		   ))
    (SETQ EXPRESSION (NORM=REPLACE.EQV EXPRESSION T NIL))
    ;; EQV REMOVED
    (SETQ EXPRESSION (NORM=REPLACE.IMPL EXPRESSION))
    ;; IMPL REMOVED
    (SETQ EXPRESSION (NORM=MOVE.NEGATIONS.TO.LITERALS EXPRESSION))
    ;; NOT REMOVED
    (SETQ EXPRESSION (NORM=MOVE.QUANTIFIERS.INSIDE EXPRESSION))
    ;; QUANTIFIERS MOVED TO INNERMOST SCOPES
    (SETQ EXPRESSION (NORM=SKOLEMIZE EXPRESSION NIL))
    ;; EX REMOVED, EX VARIABLES DROPPED.
    (COND
      (SPLIT.DEPTH (SETQ EXPRESSION (NORM=MOVE.QUANTIFIERS.INSIDE EXPRESSION))
		   ;; ATTEMPT TO MOVE ALL FURTHER INSIDE
		   (SETQ EXPRESSION (NORM=SPLIT.lazy (list EXPRESSION) T))
		   ;; SPLITTING DONE
		   )
      (T (SETQ EXPRESSION (LIST EXPRESSION)))))
  EXPRESSION)
|#

(DEFUN NORM=NORMALIZE.2 (EXPRESSION)
  ;; INPUT:  EXPRESSION CONTAINING LITERALS, ALL,AND,OR.
  ;; EFFECT: DESTRUCTIVELY TRANSFORMS EXPRESSION TO
  ;;         CLAUSAL NORMAL FORM.
  ;; VALUE:  LIST OF CLAUSES REPRESENTING EXPRESSION.
  ;;         FORMAT AS IN NORM-NORMALIZATION.  
  (mkrp-gc.start (opt-get.option gen_lisp.garbage.collection))
  (SETQ EXPRESSION (NORM=MOVE.QUANTIFIERS.OUTSIDE EXPRESSION))
  ;; PRENEX FORM CREATED.
  (mkrp-gc.start (opt-get.option gen_lisp.garbage.collection))
  (SETQ EXPRESSION (NORM=TRANSFORM.TO.PSEUDO.CNF EXPRESSION))
  ;; CNF OF MATRIX CREATED.
  (mkrp-gc.start (opt-get.option gen_lisp.garbage.collection))
  (SETQ EXPRESSION (NORM=MOVE.QUANTIFIERS.INSIDE EXPRESSION))
  ;; SUBEXPRESSIONS OF CONJUNCTIONS VARIABLE DISJOINT.
  (mkrp-gc.start (opt-get.option gen_lisp.garbage.collection))
  (SETQ EXPRESSION (NORM=REMOVE.QUANTIFIERS EXPRESSION))
  ;; ALL QUANTIFIERS REMOVED.
  (mkrp-gc.start (opt-get.option gen_lisp.garbage.collection))
  (SETQ EXPRESSION (NORM=FLATTEN.CLAUSELIST EXPRESSION))
  ;; AND, OR REMOVED; CLAUSAL FORM OBTAINED.
  (mkrp-gc.start (opt-get.option gen_lisp.garbage.collection))
  EXPRESSION)
       
(DEFUN NORM=MOVE.QUANTIFIERS.INSIDE (EXPRESSION)
  ;; INPUT:  EXPRESSION, JUNCTORS MAY BE MARKED BY '!' .
  ;; VALUE:  EQUIVALENT EXPRESSION WHERE QUANTIFIERS ARE
  ;;         MOVED TO INNERMOST POSSIBLE SCOPES.
  (CASE (NORM=OP EXPRESSION)
    ((+ -) EXPRESSION)
    ((AND OR IMPL EQV AND! OR! IMPL! EQV!)
     (NORM=PUT=LEFT EXPRESSION (NORM=MOVE.QUANTIFIERS.INSIDE (NORM=LEFT EXPRESSION)))
     (NORM=PUT=RIGHT EXPRESSION (NORM=MOVE.QUANTIFIERS.INSIDE (NORM=RIGHT EXPRESSION)))
     EXPRESSION)
    (NOT (NORM=PUT=LEFT EXPRESSION (NORM=MOVE.QUANTIFIERS.INSIDE (NORM=LEFT EXPRESSION))) EXPRESSION)
    (ALL (NORM=MOVE.ALL.INSIDE EXPRESSION))
    (EX (NORM=MOVE.EX.INSIDE EXPRESSION))
    (OTHERWISE (ERROR "ILLEGAL EXPRESSION IN NORM=MOVE.QUANTIFIERS.INSIDE:~A" EXPRESSION))))
       
(DEFUN NORM=MOVE.ALL.INSIDE (EXPRESSION)
  ;; INPUT:  EXPRESSION HAS FORM (ALL VAR RIGHT)
  ;;         WHERE RIGHT IS AN EXPRESSION.
  ;;         JUNCTORS MAY BE MARKED BY '!'.
  ;; VALUE:  EQUIVALENT EXPRESSION WHERE EVERY QUANTIFIER
  ;;         IS MOVED TO THE INNERMOST POSSIBLE SCOPE.
  (let ((VARIABLE (NORM=LEFT EXPRESSION))
	(RIGHT (NORM=MOVE.QUANTIFIERS.INSIDE (NORM=RIGHT EXPRESSION))))
    (NORM=PUT=RIGHT EXPRESSION RIGHT)
    (CASE (NORM=OP RIGHT)
      ((+ -) (COND ((NORM=VARIABLE.OCCURS.IN VARIABLE right) EXPRESSION)
		   (T (NORM=DROP.VARIABLE VARIABLE) RIGHT)))
      ((AND AND!)
       ;; ALL X (P AND Q)   --->   (ALL X P) AND (ALL X Q)
       (PROG ((NEWVAR (NORM=NEW.VARIABLE VARIABLE)))
	     (NORM=PUT=LEFT RIGHT (NORM=MOVE.ALL.INSIDE (NORM=CREATE.FORMULA 'ALL VARIABLE (NORM=LEFT RIGHT))))
	     (NORM=PUT=RIGHT RIGHT
			     (NORM=MOVE.ALL.INSIDE (NORM=CREATE.FORMULA 'ALL NEWVAR (NORM=REPLACE.VARIABLE VARIABLE NEWVAR
											    (NORM=RIGHT RIGHT)))))
	     (RETURN RIGHT)))
      ((OR OR!)
       (PROG
	 ((NOT-IN-RIGHT.LEFT (NOT (NORM=VARIABLE.OCCURS.IN VARIABLE (NORM=LEFT RIGHT))))
	  (NOT-IN-RIGHT.RIGHT (NOT (NORM=VARIABLE.OCCURS.IN VARIABLE (NORM=RIGHT RIGHT)))))
	 (RETURN
	   (COND
	     ((AND NOT-IN-RIGHT.LEFT NOT-IN-RIGHT.RIGHT)
	      ;; ALL X (P OR Q)    --->   (P OR Q)
	      (NORM=DROP.VARIABLE VARIABLE) RIGHT)
	     (NOT-IN-RIGHT.LEFT ;; ALL X (P OR QX)   --->   P OR (ALL X QX)
	      (NORM=PUT=RIGHT RIGHT (NORM=MOVE.ALL.INSIDE (NORM=CREATE.FORMULA 'ALL VARIABLE (NORM=RIGHT RIGHT)))) RIGHT)
	     (NOT-IN-RIGHT.RIGHT ;; ALL X (PX OR Q)   --->   (ALL X PX) OR Q
	      (NORM=PUT=LEFT RIGHT (NORM=MOVE.ALL.INSIDE (NORM=CREATE.FORMULA 'ALL VARIABLE (NORM=LEFT RIGHT)))) RIGHT)
	     (T EXPRESSION)))))
      ((IMPL IMPL!)
       (PROG
	 ((NOT-IN-RIGHT.LEFT (NOT (NORM=VARIABLE.OCCURS.IN VARIABLE (NORM=LEFT RIGHT))))
	  (NOT-IN-RIGHT.RIGHT (NOT (NORM=VARIABLE.OCCURS.IN VARIABLE (NORM=RIGHT RIGHT)))))
	 (RETURN
	   (COND
	     ((AND NOT-IN-RIGHT.LEFT NOT-IN-RIGHT.RIGHT)
	      ;; ALL X (P IMPL Q)  ---> (P IMPL Q)
	      (NORM=DROP.VARIABLE VARIABLE) RIGHT)
	     (NOT-IN-RIGHT.LEFT ;; ALL X (P IMPL QX)  --->  P IMPL (ALL X QX)
	      (NORM=PUT=RIGHT RIGHT (NORM=MOVE.ALL.INSIDE (NORM=CREATE.FORMULA 'ALL VARIABLE (NORM=RIGHT RIGHT)))) RIGHT)
	     (NOT-IN-RIGHT.RIGHT ;; ALL X (PX IMPL Q) --->  (EX X  PX) IMPL Q))
	      (NORM=PUT=LEFT RIGHT (NORM=MOVE.EX.INSIDE (NORM=CREATE.FORMULA 'EX VARIABLE (NORM=LEFT RIGHT)))) RIGHT)
	     (T EXPRESSION)))))
      ((EQV EQV!)
       (COND
	 ((AND (NOT (NORM=VARIABLE.OCCURS.IN VARIABLE (NORM=LEFT RIGHT)))
	       (NOT (NORM=VARIABLE.OCCURS.IN VARIABLE (NORM=RIGHT RIGHT))))
	  ;; ALL X (P EQV Q) ---> (P EQV Q)
	  (NORM=DROP.VARIABLE VARIABLE) RIGHT)
	 (T EXPRESSION)))
      (ALL ;; ALL X (ALL Y P)   --->   ALL Y (ALL X P)
	(PROG ((INNERVAR (NORM=LEFT RIGHT))) (NORM=PUT=LEFT EXPRESSION INNERVAR) (NORM=PUT=LEFT RIGHT VARIABLE)
	      (NORM=PUT=RIGHT EXPRESSION (NORM=MOVE.ALL.INSIDE RIGHT)) (RETURN EXPRESSION)))
      (EX
	(COND ((NORM=VARIABLE.OCCURS.IN VARIABLE (NORM=RIGHT RIGHT)) EXPRESSION) (T (NORM=DROP.VARIABLE VARIABLE) RIGHT)))
      (NOT ;; ALL X (NOT P)     --->   NOT (EX X P)
	(SETQ EXPRESSION (NORM=CREATE.FORMULA 'NOT (NORM=MOVE.EX.INSIDE (NORM=CREATE.FORMULA 'EX VARIABLE (NORM=LEFT RIGHT))))))
      (OTHERWISE (ERROR "ILLEGAL EXPRESSION IN NORM=MOVE.ALL.INSIDE:~A" EXPRESSION)))))
       
(DEFUN NORM=MOVE.EX.INSIDE (EXPRESSION)
  ;; INPUT:  EXPRESSION HAS FORM  (EX VAR RIGHT)
  ;;         WHERE RIGHT IS AN EXPRESSION.
  ;;         JUNCTORS MAY BE MARKED BY '!'.
  ;; VALUE:  EQUIVALENT EXPRESSION WHERE EVERY QUANTIFIER
  ;;         IS MOVED TO THE INNERMOST POSSIBLE SCOPE.
  (PROG ((VARIABLE (NORM=LEFT EXPRESSION)) (RIGHT (NORM=MOVE.QUANTIFIERS.INSIDE (NORM=RIGHT EXPRESSION))))
	(NORM=PUT=RIGHT EXPRESSION RIGHT)
	(RETURN
	  (CASE (NORM=OP RIGHT)
	    ((+ -) (COND ((IN VARIABLE (NORM=RIGHT RIGHT)) EXPRESSION) (T (NORM=DROP.VARIABLE VARIABLE) RIGHT)))
	    ((OR OR!)
	     ;; EX X (P OR Q)     --->   (EX X P) OR (EX X Q)
	     (PROG ((NEWVAR (NORM=NEW.VARIABLE VARIABLE)))
		   (NORM=PUT=LEFT RIGHT (NORM=MOVE.EX.INSIDE (NORM=CREATE.FORMULA 'EX VARIABLE (NORM=LEFT RIGHT))))
		   (NORM=PUT=RIGHT RIGHT
				   (NORM=MOVE.EX.INSIDE (NORM=CREATE.FORMULA 'EX NEWVAR (NORM=REPLACE.VARIABLE VARIABLE NEWVAR
												(NORM=RIGHT RIGHT)))))
		   (RETURN RIGHT)))
	    ((AND AND!)
	     (PROG
	       ((NOT-IN-RIGHT.LEFT (NOT (NORM=VARIABLE.OCCURS.IN VARIABLE (NORM=LEFT RIGHT))))
		(NOT-IN-RIGHT.RIGHT (NOT (NORM=VARIABLE.OCCURS.IN VARIABLE (NORM=RIGHT RIGHT)))))
	       (RETURN
		 (COND
		   ((AND NOT-IN-RIGHT.LEFT NOT-IN-RIGHT.RIGHT)
		    ;; EX X (P AND Q)    --->   (P AND Q)
		    (NORM=DROP.VARIABLE VARIABLE) RIGHT)
		   (NOT-IN-RIGHT.LEFT ;; EX X (P AND QX)   --->   P AND (EX X QX)
		    (NORM=PUT=RIGHT RIGHT (NORM=MOVE.EX.INSIDE (NORM=CREATE.FORMULA 'EX VARIABLE (NORM=RIGHT RIGHT)))) RIGHT)
		   (NOT-IN-RIGHT.RIGHT ;; EX X (PX AND Q)   --->   (EX X PX) AND Q
		    (NORM=PUT=LEFT RIGHT (NORM=MOVE.EX.INSIDE (NORM=CREATE.FORMULA 'EX VARIABLE (NORM=LEFT RIGHT)))) RIGHT)
		   (T EXPRESSION)))))
	    ((IMPL IMPL!)
	     ;; EX X (P IMPL Q)   --->   (ALL X P) IMPL (EX X Q)
	     (PROG ((NEWVAR (NORM=NEW.VARIABLE VARIABLE)))
		   (NORM=PUT=LEFT RIGHT (NORM=MOVE.ALL.INSIDE (NORM=CREATE.FORMULA 'ALL VARIABLE (NORM=LEFT RIGHT))))
		   (NORM=PUT=RIGHT RIGHT
				   (NORM=MOVE.EX.INSIDE (NORM=CREATE.FORMULA 'EX NEWVAR (NORM=REPLACE.VARIABLE VARIABLE NEWVAR (NORM=RIGHT RIGHT)))))
		   (RETURN RIGHT)))
	    ((EQV EQV!)
	     (COND
	       ((AND (NOT (NORM=VARIABLE.OCCURS.IN VARIABLE (NORM=LEFT RIGHT)))
		     (NOT (NORM=VARIABLE.OCCURS.IN VARIABLE (NORM=RIGHT RIGHT))))
		(NORM=DROP.VARIABLE VARIABLE) RIGHT)
	       (T EXPRESSION)))
	    (EX ;; EX X (EX Y P)     --->   EX Y (EX X P)
	      (PROG ((INNERVAR (NORM=LEFT RIGHT))) (NORM=PUT=LEFT EXPRESSION INNERVAR) (NORM=PUT=LEFT RIGHT VARIABLE)
		    (NORM=PUT=RIGHT EXPRESSION (NORM=MOVE.EX.INSIDE RIGHT)) (RETURN EXPRESSION)))
	    (ALL
	      (COND ((NORM=VARIABLE.OCCURS.IN VARIABLE (NORM=RIGHT RIGHT)) EXPRESSION) (T (NORM=DROP.VARIABLE VARIABLE) RIGHT)))
	    (NOT ;; EX  X (NOT P)     --->   NOT (ALL X P)
	      (SETQ EXPRESSION (NORM=CREATE.FORMULA 'NOT (NORM=MOVE.ALL.INSIDE (NORM=CREATE.FORMULA 'ALL VARIABLE (NORM=LEFT RIGHT))))))
	    (OTHERWISE (ERROR "ILLEGAL EXPRESSION IN NORM=MOVE.EX.INSIDE:~A" EXPRESSION))))))
       
(DEFUN NORM=MOVE.QUANTIFIERS.OUTSIDE (EXPRESSION)
  ;; EDITED:  5-AUG-84 14:37:49        NE
  ;; INPUT:  EXPRESSION NOT CONTAINING EQV.
  ;; VALUE:  PRENEX FORM OF EXPRESSION.
  (SETQ EXPRESSION (NORM=EXTRACT.PREFIX.MATRIX EXPRESSION NIL))
  (NSUBST (CDR EXPRESSION) 'MATRIX (CAR EXPRESSION)))
       
(DEFUN NORM=EXTRACT.PREFIX.MATRIX (EXPRESSION INVERT.FLAG)
  ;; EDITED:  5-AUG-84 14:41:35        NE
  ;; INPUT:  EXPRESSION NOT CONTAINING EQV; BOOLEAN.
  ;; EFFECT: COLLECTS THE QUANTIFIERS OF EXPRESSION IN
  ;;         THE PREFIX LEAVING THE QUANTIFIER FREE
  ;;         MATRIX.
  ;;         IF INVERT.FLAG IS NON-NIL, THE QUANTIFIERS
  ;;         IN THE PREFIX ARE INVERTED.
  ;;         THE PREFIX HAS THE FORMAT OF AN EXPRESSION
  ;;         BUT WITH THE ATOM 'MATRIX AT THE POSITION
  ;;         THE MATRIX HAS TO BE SUBSTITUTED IN IN ORDER
  ;;         TO OBTAIN THE PREFIX FORM OF EXPRESSION.
  ;; VALUE:  DOTTED PAIR (PREFIX . MATRIX)
  (PROG (LEFT LEFT.PREFIX LEFT.MATRIX RIGHT RIGHT.PREFIX RIGHT.MATRIX PREFIX MATRIX)
	(CASE (NORM=OP EXPRESSION) ((+ -) (SETQ PREFIX 'MATRIX) (SETQ MATRIX EXPRESSION))
	      ((ALL EX) (SETQ RIGHT (NORM=EXTRACT.PREFIX.MATRIX (NORM=RIGHT EXPRESSION) INVERT.FLAG)) (SETQ MATRIX (CDR RIGHT))
	       (SETQ PREFIX EXPRESSION) (NORM=PUT=RIGHT PREFIX (CAR RIGHT))
	       (COND
		 (INVERT.FLAG
		  (CASE (NORM=OP PREFIX) (ALL (NORM=PUT=OP PREFIX 'EX)) (EX (NORM=PUT=OP PREFIX 'ALL)) (OTHERWISE NIL)))))
	      (NOT (SETQ LEFT (NORM=EXTRACT.PREFIX.MATRIX (NORM=LEFT EXPRESSION) (NOT INVERT.FLAG))) (SETQ PREFIX (CAR LEFT))
		   (SETQ MATRIX EXPRESSION) (NORM=PUT=LEFT MATRIX (CDR LEFT)))
	      ((AND OR IMPL)
	       (SETQ LEFT
		     (NORM=EXTRACT.PREFIX.MATRIX (NORM=LEFT EXPRESSION)
						 (CASE (NORM=OP EXPRESSION) (IMPL (NOT INVERT.FLAG)) (OTHERWISE INVERT.FLAG))))
	       (SETQ RIGHT (NORM=EXTRACT.PREFIX.MATRIX (NORM=RIGHT EXPRESSION) INVERT.FLAG)) (SETQ LEFT.PREFIX (CAR LEFT))
	       (SETQ LEFT.MATRIX (CDR LEFT)) (SETQ RIGHT.PREFIX (CAR RIGHT)) (SETQ RIGHT.MATRIX (CDR RIGHT))
	       (SETQ PREFIX (NSUBST RIGHT.PREFIX 'MATRIX LEFT.PREFIX)) (SETQ MATRIX EXPRESSION) (NORM=PUT=LEFT MATRIX LEFT.MATRIX)
	       (NORM=PUT=RIGHT MATRIX RIGHT.MATRIX))
	      (OTHERWISE (ERROR "ILLEGAL EXPRESSION IN NORM=EXTRACT.PREFIX.MATRIX: ~A" EXPRESSION)))
	(RETURN (CONS PREFIX MATRIX))))
       
(DEFUN NORM=MARK.SPLIT (EXPRESSION LEVELCOUNTER PRIOR.OP)
  ;; input:  expression;
  ;;         levelcounter is a natural number,
  ;;         prior.op is a junctor or negation symbol.
  ;; effect: the junctors of the upper levelcounter
  ;;         levels of expression are marked by '!'.
  ;;         the level is computed as if and,or were
  ;;         n-ary instead of binary junctors.
  ;; value:  modified expression.
  (let ((OP (NORM=OP EXPRESSION)))
    (CASE OP
      ((+ -) EXPRESSION)
      (NOT (NORM=PUT=LEFT EXPRESSION (NORM=MARK.SPLIT (NORM=LEFT EXPRESSION) LEVELCOUNTER OP)) EXPRESSION)
      ((ALL EX) (NORM=PUT=RIGHT EXPRESSION (NORM=MARK.SPLIT (NORM=RIGHT EXPRESSION) LEVELCOUNTER OP)) EXPRESSION)
      ((AND OR IMPL EQV)
       (COND ((AND (MEMBER PRIOR.OP '(AND OR)) (EQL PRIOR.OP OP))
	      ;; SUPPRESS LEVEL DECREMENT
	      )
	     (T (SETQ LEVELCOUNTER (1- LEVELCOUNTER))))
       (when (> LEVELCOUNTER -1)
	 (NORM=PUT=OP EXPRESSION (INTERN (format nil "~A!" op) (find-package "MKRP"))))
       (NORM=PUT=LEFT EXPRESSION (NORM=MARK.SPLIT (NORM=LEFT EXPRESSION) LEVELCOUNTER OP))
       (NORM=PUT=RIGHT EXPRESSION (NORM=MARK.SPLIT (NORM=RIGHT EXPRESSION) LEVELCOUNTER OP)) EXPRESSION)
      (OTHERWISE (ERROR "Illegal expression in norm=mark.split:~a" EXPRESSION)))))
	 
(DEFUN NORM=REPLACE.EQV (EXPRESSION SPLITTABLE NEGATION.FLAG)
  ;; Input:  EXPRESSION, junctors may be marked by '!' .
  ;;         SPLITTABLE is nil iff expression occurs in
  ;;         an environment where splitting is impossible
  ;;         or not desired.
  ;;         NEGATION.FLAG is nil iff expression lies in
  ;;         the scope of an even number of negations.
  ;; effect: depending on splittable, negation.flag, and
  ;;         the presence of the !-marked junctors, eqv
  ;;         is resolved in a way providing a maximum
  ;;         number of possible splits and a relative
  ;;         minimum of literals when multiplied later
  ;;         to cnf.
  ;; value:  equivalent expression without eqv.
  (CASE (NORM=OP EXPRESSION)
    (ALL (COND ((NOT NEGATION.FLAG) (SETQ SPLITTABLE NIL)))
	 (NORM=PUT=RIGHT EXPRESSION (NORM=REPLACE.EQV (NORM=RIGHT EXPRESSION) SPLITTABLE NEGATION.FLAG)) EXPRESSION)
    (EX (COND (NEGATION.FLAG (SETQ SPLITTABLE NIL)))
	(NORM=PUT=RIGHT EXPRESSION (NORM=REPLACE.EQV (NORM=RIGHT EXPRESSION) SPLITTABLE NEGATION.FLAG)) EXPRESSION)
    (NOT (NORM=PUT=LEFT EXPRESSION (NORM=REPLACE.EQV (NORM=LEFT EXPRESSION) SPLITTABLE (NOT NEGATION.FLAG))) EXPRESSION)
    ((AND OR AND! OR!)
     (NORM=PUT=LEFT EXPRESSION (NORM=REPLACE.EQV (NORM=LEFT EXPRESSION) SPLITTABLE NEGATION.FLAG))
     (NORM=PUT=RIGHT EXPRESSION (NORM=REPLACE.EQV (NORM=RIGHT EXPRESSION) SPLITTABLE NEGATION.FLAG)) EXPRESSION)
    ((IMPL IMPL!) (NORM=PUT=LEFT EXPRESSION (NORM=REPLACE.EQV (NORM=LEFT EXPRESSION) SPLITTABLE (NOT NEGATION.FLAG)))
     (NORM=PUT=RIGHT EXPRESSION (NORM=REPLACE.EQV (NORM=RIGHT EXPRESSION) SPLITTABLE NEGATION.FLAG)) EXPRESSION)
    ((EQV EQV!)
     (PROG
       ((+LEFT (NORM=REPLACE.EQV (NORM=COPY (NORM=LEFT EXPRESSION)) SPLITTABLE NEGATION.FLAG))
	(-LEFT (NORM=REPLACE.EQV (NORM=CREATE.FORMULA 'NOT (NORM=LEFT EXPRESSION)) SPLITTABLE NEGATION.FLAG))
	(+RIGHT (NORM=REPLACE.EQV (NORM=COPY (NORM=RIGHT EXPRESSION)) SPLITTABLE NEGATION.FLAG))
	(-RIGHT (NORM=REPLACE.EQV (NORM=CREATE.FORMULA 'NOT (NORM=RIGHT EXPRESSION)) SPLITTABLE NEGATION.FLAG)))
       (CASE (NORM=OP EXPRESSION)
	 (EQV
	   (COND
	     (NEGATION.FLAG (NORM=PUT=OP EXPRESSION 'OR) (NORM=PUT=LEFT EXPRESSION (NORM=CREATE.FORMULA 'AND +LEFT +RIGHT))
			    (NORM=PUT=RIGHT EXPRESSION (NORM=CREATE.FORMULA 'AND -LEFT -RIGHT)))
	     (T (NORM=PUT=OP EXPRESSION 'AND) (NORM=PUT=LEFT EXPRESSION (NORM=CREATE.FORMULA 'OR +LEFT -RIGHT))
                (NORM=PUT=RIGHT EXPRESSION (NORM=CREATE.FORMULA 'OR -LEFT +RIGHT)))))
	 (EQV!
	   (COND
	     ((ANTIVALENT SPLITTABLE NEGATION.FLAG) (NORM=PUT=OP EXPRESSION 'OR!)
	      (NORM=PUT=LEFT EXPRESSION (NORM=CREATE.FORMULA 'AND! +LEFT +RIGHT)) (NORM=PUT=RIGHT EXPRESSION (NORM=CREATE.FORMULA 'AND! -LEFT -RIGHT)))
	     (T (NORM=PUT=OP EXPRESSION 'AND!) (NORM=PUT=LEFT EXPRESSION (NORM=CREATE.FORMULA 'OR! +LEFT -RIGHT))
                (NORM=PUT=RIGHT EXPRESSION (NORM=CREATE.FORMULA 'OR! -LEFT +RIGHT)))))
	 (OTHERWISE (ERROR "Illegal expression in NORM=REPLACE.EQV:~A" EXPRESSION)))
       (RETURN EXPRESSION)))
    ((+ -) EXPRESSION) (OTHERWISE (ERROR "Illegal expression in NORM=REPLACE.EQV:~A" EXPRESSION))))

(DEFUN NORM=REPLACE.IMPL (EXPRESSION)
  ;; Edited:  2-aug-84 17:52:05        ne
  ;; input:  EXPRESSION, junctors may be marked by '!' .
  ;; value:  equivalent expression where impl is replaced
  ;;         by not and or.
  (CASE (NORM=OP EXPRESSION)
    ((ALL EX) (NORM=PUT=RIGHT EXPRESSION (NORM=REPLACE.IMPL (NORM=RIGHT EXPRESSION))) EXPRESSION)
    (NOT (NORM=PUT=LEFT EXPRESSION (NORM=REPLACE.IMPL (NORM=LEFT EXPRESSION))) EXPRESSION)
    ((AND OR IMPL EQV AND! OR! IMPL! EQV!) (NORM=PUT=LEFT EXPRESSION (NORM=REPLACE.IMPL (NORM=LEFT EXPRESSION)))
     (NORM=PUT=RIGHT EXPRESSION (NORM=REPLACE.IMPL (NORM=RIGHT EXPRESSION)))
     (CASE (NORM=OP EXPRESSION)
       (IMPL (NORM=PUT=OP EXPRESSION 'OR) (NORM=PUT=LEFT EXPRESSION (NORM=CREATE.FORMULA 'NOT (NORM=LEFT EXPRESSION))))
       (IMPL! (NORM=PUT=OP EXPRESSION 'OR!) (NORM=PUT=LEFT EXPRESSION (NORM=CREATE.FORMULA 'NOT (NORM=LEFT EXPRESSION)))) (OTHERWISE NIL))
     EXPRESSION)
    ((+ -) EXPRESSION)
    (OTHERWISE (ERROR "Illegal expression in NORM=REPLACE.IMPL: ~A" EXPRESSION))))

(DEFUN NORM=MOVE.NEGATIONS.TO.LITERALS (EXPRESSION)
  ;; Input:  expression not containing impl, eqv.
  ;;         junctors may be marked by '!'.
  ;; value:  equivalent expression without not, i.e.
  ;;         negations are reflected by minus signs in
  ;;         literals only.
  (CASE (NORM=OP EXPRESSION)
    ((ALL EX) (NORM=PUT=RIGHT EXPRESSION (NORM=MOVE.NEGATIONS.TO.LITERALS (NORM=RIGHT EXPRESSION))) EXPRESSION)
    ((AND OR AND! OR!) (NORM=PUT=LEFT EXPRESSION (NORM=MOVE.NEGATIONS.TO.LITERALS (NORM=LEFT EXPRESSION)))
      (NORM=PUT=RIGHT EXPRESSION (NORM=MOVE.NEGATIONS.TO.LITERALS (NORM=RIGHT EXPRESSION))) EXPRESSION)
    (NOT (SETQ EXPRESSION (NORM=LEFT EXPRESSION))
      (CASE (NORM=OP EXPRESSION)
        (ALL (NORM=PUT=OP EXPRESSION 'EX)
          (NORM=PUT=RIGHT EXPRESSION (NORM=MOVE.NEGATIONS.TO.LITERALS (NORM=CREATE.FORMULA 'NOT (NORM=RIGHT EXPRESSION)))) EXPRESSION)
        (EX (NORM=PUT=OP EXPRESSION 'ALL)
          (NORM=PUT=RIGHT EXPRESSION (NORM=MOVE.NEGATIONS.TO.LITERALS (NORM=CREATE.FORMULA 'NOT (NORM=RIGHT EXPRESSION)))) EXPRESSION)
        (AND (NORM=PUT=OP EXPRESSION 'OR)
          (NORM=PUT=LEFT EXPRESSION (NORM=MOVE.NEGATIONS.TO.LITERALS (NORM=CREATE.FORMULA 'NOT (NORM=LEFT EXPRESSION))))
          (NORM=PUT=RIGHT EXPRESSION (NORM=MOVE.NEGATIONS.TO.LITERALS (NORM=CREATE.FORMULA 'NOT (NORM=RIGHT EXPRESSION)))) EXPRESSION)
        (AND! (NORM=PUT=OP EXPRESSION 'OR!)
          (NORM=PUT=LEFT EXPRESSION (NORM=MOVE.NEGATIONS.TO.LITERALS (NORM=CREATE.FORMULA 'NOT (NORM=LEFT EXPRESSION))))
          (NORM=PUT=RIGHT EXPRESSION (NORM=MOVE.NEGATIONS.TO.LITERALS (NORM=CREATE.FORMULA 'NOT (NORM=RIGHT EXPRESSION)))) EXPRESSION)
        (OR (NORM=PUT=OP EXPRESSION 'AND)
          (NORM=PUT=LEFT EXPRESSION (NORM=MOVE.NEGATIONS.TO.LITERALS (NORM=CREATE.FORMULA 'NOT (NORM=LEFT EXPRESSION))))
          (NORM=PUT=RIGHT EXPRESSION (NORM=MOVE.NEGATIONS.TO.LITERALS (NORM=CREATE.FORMULA 'NOT (NORM=RIGHT EXPRESSION)))) EXPRESSION)
        (OR! (NORM=PUT=OP EXPRESSION 'AND!)
          (NORM=PUT=LEFT EXPRESSION (NORM=MOVE.NEGATIONS.TO.LITERALS (NORM=CREATE.FORMULA 'NOT (NORM=LEFT EXPRESSION))))
          (NORM=PUT=RIGHT EXPRESSION (NORM=MOVE.NEGATIONS.TO.LITERALS (NORM=CREATE.FORMULA 'NOT (NORM=RIGHT EXPRESSION)))) EXPRESSION)
        (NOT (NORM=MOVE.NEGATIONS.TO.LITERALS (NORM=LEFT EXPRESSION))) (+ (NORM=PUT=OP EXPRESSION '-) EXPRESSION)
        (- (NORM=PUT=OP EXPRESSION '+) EXPRESSION)
        (OTHERWISE (ERROR "ILLEGAL EXPRESSION IN NORM=MOVE.NEGATIONS.TO.LITERALS:~A" EXPRESSION))))
    ((+ -) EXPRESSION) (OTHERWISE (ERROR "ILLEGAL EXPRESSION IN NORM=MOVE.NEGATIONS.TO.LITERALS:~A" EXPRESSION))))

(DEFUN NORM=SKOLEMIZE (EXPRESSION ENVIRONMENT)
  ;; INPUT:  EXPRESSION NOT CONTAINING IMPL, EQV, NOT.
  ;;         JUNCTORS MAY BE MARKED BY '!'.
  ;;         ENVIRONMENT: LIST OF UNIVERSALLY QUANTIFIED
  ;;         VARIABLES IN WHOSE SCOPE EXPRESSION LIES.
  ;; VALUE:  EXPRESSION WITHOUT EXISTENTIAL QUANTIFIERS.
  ;;         THE CORRESPONDING VARIABLES ARE REPLACED BY
  ;;         SKOLEM TERMS.
  (CASE (NORM=OP EXPRESSION)
    ((AND OR AND! OR!) (NORM=PUT=LEFT EXPRESSION (NORM=SKOLEMIZE (NORM=LEFT EXPRESSION) ENVIRONMENT))
      (NORM=PUT=RIGHT EXPRESSION (NORM=SKOLEMIZE (NORM=RIGHT EXPRESSION) ENVIRONMENT)) EXPRESSION)
    (ALL (NORM=PUT=RIGHT EXPRESSION (NORM=SKOLEMIZE (NORM=RIGHT EXPRESSION) (CONS (NORM=LEFT EXPRESSION) ENVIRONMENT))) EXPRESSION)
    (EX (let* ((EX.VARIABLE (NORM=LEFT EXPRESSION))
	       (SKOLEM.TERM (NORM=NEW.TERM ENVIRONMENT EX.VARIABLE))
	       (RIGHT (NORM=SKOLEMIZE (NORM=RIGHT EXPRESSION) ENVIRONMENT)))
	  (SETQ RIGHT (NORM=REPLACE.VARIABLE EX.VARIABLE SKOLEM.TERM RIGHT))
	  (prog1 (if (opt-get.option sort_literals)
		     (NORM=CREATE.FORMULA 'AND RIGHT
					  (norm=create.formula '+ (dt-predicate.element)
							       (list skolem.term (dt-variable.sort ex.variable))))
		     right)
		 (NORM=DROP.VARIABLE EX.VARIABLE))))
    ((+ -) EXPRESSION) (OTHERWISE (ERROR "ILLEGAL EXPRESSION IN NORM=SKOLEMIZE:~A" EXPRESSION))))

(DEFUN NORM=SPLIT.lazy (EXPRESSIONs SPLITTABLE)
  ;; Input:  expression not containing eqv,impl,not,ex.
  ;;         junctors may be marked by '!'.
  ;;         splittable is nil iff splitting is
  ;;         impossible or not required.
  ;; effect: causes expression to be multiplied to dnf
  ;;         as far as it is splittable (i.e., if all-
  ;;         quantifers permit) and marked by '!'.
  ;;         the '!'-marks are removed.
  ;; value:  modified expression.
  (let ((expression (first expressions)))
      (CASE (NORM=OP EXPRESSION)
	((+ -) expressions)
	(ALL (SETQ SPLITTABLE NIL)
	     (NORM=PUT=RIGHT EXPRESSION (NORM=PREPARE.SPLIT.lazy (NORM=RIGHT EXPRESSION) SPLITTABLE))
	     (norm=split.lazy expressions splittable))
	(AND (NORM=PUT=LEFT EXPRESSION (NORM=PREPARE.SPLIT.lazy (NORM=LEFT EXPRESSION) SPLITTABLE))
	     (NORM=PUT=RIGHT EXPRESSION (NORM=PREPARE.SPLIT.lazy (NORM=RIGHT EXPRESSION) SPLITTABLE))
	     expressions)
	(OR  (setf (first expressions) (NORM=RIGHT EXPRESSION))
	     (push (NORM=LEFT EXPRESSION) expressions)
	     (norm=split.lazy expressions splittable))
	(AND! (NORM=PUT=OP EXPRESSION 'AND)
	      (NORM=PUT=LEFT EXPRESSION (NORM=PREPARE.sPLIT.lazy (NORM=LEFT EXPRESSION) SPLITTABLE))
	      (NORM=PUT=RIGHT EXPRESSION (NORM=PREPARE.SPLIT.lazy (NORM=RIGHT EXPRESSION) SPLITTABLE))
	      (COND (SPLITTABLE (setf (first expressions) (NORM=MULTIPLY.TO.DNF1 EXPRESSION)))
		    (T EXPRESSION))
	      (norm=split.lazy expressions splittable))
	(OR! (NORM=PUT=OP EXPRESSION 'OR)
	     (norm=split.lazy expressions splittable))
	(OTHERWISE (ERROR "ILLEGAL EXPRESSION IN NORM=PREPARE.SPLIT:~A" EXPRESSION)))))

(defun norm=prepare.split.lazy (EXPRESSION SPLITTABLE)
  (CASE (NORM=OP EXPRESSION)
    ((+ -) EXPRESSION)
    (ALL (SETQ SPLITTABLE NIL)
	 (NORM=PUT=RIGHT EXPRESSION (NORM=PREPARE.SPLIT.lazy (NORM=RIGHT EXPRESSION) SPLITTABLE))
	 EXPRESSION)
    ((AND OR)
     (NORM=PUT=LEFT EXPRESSION (NORM=PREPARE.SPLIT.lazy (NORM=LEFT EXPRESSION) SPLITTABLE))
     (NORM=PUT=RIGHT EXPRESSION (NORM=PREPARE.SPLIT.lazy (NORM=RIGHT EXPRESSION) SPLITTABLE))
     EXPRESSION)
    (AND! (NORM=PUT=OP EXPRESSION 'AND)
	  (NORM=PUT=LEFT EXPRESSION (NORM=PREPARE.sPLIT.lazy (NORM=LEFT EXPRESSION) SPLITTABLE))
	  (NORM=PUT=RIGHT EXPRESSION (NORM=PREPARE.SPLIT.lazy (NORM=RIGHT EXPRESSION) SPLITTABLE))
	  (COND (SPLITTABLE (NORM=MULTIPLY.TO.DNF EXPRESSION))
		(T EXPRESSION)))
    (OR! (NORM=PUT=OP EXPRESSION 'OR) (NORM=PUT=LEFT EXPRESSION (NORM=PREPARE.SPLIT.lazy (NORM=LEFT EXPRESSION) SPLITTABLE))
	 (NORM=PUT=RIGHT EXPRESSION (NORM=PREPARE.SPLIT.lazy (NORM=RIGHT EXPRESSION) SPLITTABLE)) EXPRESSION)
    (OTHERWISE (ERROR "ILLEGAL EXPRESSION IN NORM=PREPARE.SPLIT:~A" EXPRESSION))))

(DEFUN NORM=PREPARE.SPLIT (EXPRESSION SPLITTABLE)
  ;; Input:  expression not containing eqv,impl,not,ex.
  ;;         junctors may be marked by '!'.
  ;;         splittable is nil iff splitting is
  ;;         impossible or not required.
  ;; effect: causes expression to be multiplied to dnf
  ;;         as far as it is splittable (i.e., if all-
  ;;         quantifers permit) and marked by '!'.
  ;;         the '!'-marks are removed.
  ;; value:  modified expression.
  (CASE (NORM=OP EXPRESSION)
    ((+ -) EXPRESSION)
    (ALL (SETQ SPLITTABLE NIL)
	 (NORM=PUT=RIGHT EXPRESSION (NORM=PREPARE.SPLIT (NORM=RIGHT EXPRESSION) SPLITTABLE))
	 EXPRESSION)
    ((AND OR)
     (NORM=PUT=LEFT EXPRESSION (NORM=PREPARE.SPLIT (NORM=LEFT EXPRESSION) SPLITTABLE))
     (NORM=PUT=RIGHT EXPRESSION (NORM=PREPARE.SPLIT (NORM=RIGHT EXPRESSION) SPLITTABLE))
     EXPRESSION)
    (AND! (NORM=PUT=OP EXPRESSION 'AND) (NORM=PUT=LEFT EXPRESSION (NORM=PREPARE.SPLIT (NORM=LEFT EXPRESSION) SPLITTABLE))
	  (NORM=PUT=RIGHT EXPRESSION (NORM=PREPARE.SPLIT (NORM=RIGHT EXPRESSION) SPLITTABLE))
	  (COND (SPLITTABLE (NORM=MULTIPLY.TO.DNF EXPRESSION)) (T EXPRESSION)))
    (OR! (NORM=PUT=OP EXPRESSION 'OR) (NORM=PUT=LEFT EXPRESSION (NORM=PREPARE.SPLIT (NORM=LEFT EXPRESSION) SPLITTABLE))
	 (NORM=PUT=RIGHT EXPRESSION (NORM=PREPARE.SPLIT (NORM=RIGHT EXPRESSION) SPLITTABLE)) EXPRESSION)
    (OTHERWISE (ERROR "ILLEGAL EXPRESSION IN NORM=PREPARE.SPLIT:~A" EXPRESSION))))

(DEFUN NORM=SPLIT (EXPRESSION)
  ;; INPUT: EXPRESSION.
  ;; VALUE:  LIST OF EXPRESSIONS OBTAINED BY SPLITTING
  ;;         THE GIVEN ONE TOP DOWN INTO SUBEXPRESSIONS
  ;;         CONNECTED BY OR.
  (if (EQL (NORM=OP EXPRESSION) 'OR)
      (NCONC (NORM=SPLIT (NORM=LEFT EXPRESSION))
	     (NORM=SPLIT (NORM=RIGHT EXPRESSION)))
      (LIST EXPRESSION)))

(DEFUN NORM=REMOVE.QUANTIFIERS (EXPRESSION)
  ;; EDITED:  5-AUG-84 14:29:04        NE
  ;; INPUT:  EXPRESSION NOT CONTAINING EQV.
  ;; VALUE:  EXPRESSION WITHOUT ANY QUANTIFIERS.
  (CASE (NORM=OP EXPRESSION) ((+ -) EXPRESSION) ((ALL EX) (NORM=REMOVE.QUANTIFIERS (NORM=RIGHT EXPRESSION)))
    (NOT (NORM=PUT=LEFT EXPRESSION (NORM=REMOVE.QUANTIFIERS (NORM=LEFT EXPRESSION))) EXPRESSION)
    ((AND OR IMPL) (NORM=PUT=LEFT EXPRESSION (NORM=REMOVE.QUANTIFIERS (NORM=LEFT EXPRESSION)))
      (NORM=PUT=RIGHT EXPRESSION (NORM=REMOVE.QUANTIFIERS (NORM=RIGHT EXPRESSION))) EXPRESSION)
    (OTHERWISE (ERROR "ILLEGAL EXPRESSION IN NORM=REMOVE.QUANTIFIERS: ~A" EXPRESSION))))

(DEFUN NORM=TRANSFORM.TO.PSEUDO.CNF (EXPRESSION)
  ;; INPUT:  EXPRESSION NOT CONTAINING EQV, IMPL, NOT.
  ;; VALUE:  EQUIVALENT EXPRESSION IN PSEUDO CNF, I.E.
  ;;         QUANTIFIED SUBEXPRESSIONS ARE TREATED LIKE
  ;;         LITERALS BUT THEIR SUBEXPRESSIONS IN TURN
  ;;         ARE ALSO TRANSFORMED TO CNF.
  (CASE (NORM=OP EXPRESSION) ((+ -) EXPRESSION)
    ((ALL EX) (NORM=PUT=RIGHT EXPRESSION (NORM=TRANSFORM.TO.PSEUDO.CNF (NORM=RIGHT EXPRESSION))) EXPRESSION)
    (AND (NORM=PUT=LEFT EXPRESSION (NORM=TRANSFORM.TO.PSEUDO.CNF (NORM=LEFT EXPRESSION)))
      (NORM=PUT=RIGHT EXPRESSION (NORM=TRANSFORM.TO.PSEUDO.CNF (NORM=RIGHT EXPRESSION))) EXPRESSION)
    (OR (NORM=PUT=LEFT EXPRESSION (NORM=TRANSFORM.TO.PSEUDO.CNF (NORM=LEFT EXPRESSION)))
      (NORM=PUT=RIGHT EXPRESSION (NORM=TRANSFORM.TO.PSEUDO.CNF (NORM=RIGHT EXPRESSION))) (NORM=MULTIPLY.TO.CNF EXPRESSION))
    (OTHERWISE (ERROR "ILLEGAL EXPRESSION IN NORM=TRANSFORM.TO.PSEUDO.CNF:~A" EXPRESSION))))

(DEFUN NORM=MULTIPLY.TO.CNF (EXPRESSION)
  ;; INPUT:  EXPRESSION HAS FORM (OR LEFT RIGHT)
  ;; VALUE:  CNF OF EXPRESSION WHERE ANY SUBEXPRESSION
  ;;         WITH AN OPERATOR OTHER THAN AND,OR IS
  ;;         TREATED AS IF IT WERE A LITERAL.
  (PROG ((LEFT (NORM=LEFT EXPRESSION)) (RIGHT (NORM=RIGHT EXPRESSION)))
    (RETURN
      (COND
        ((EQL (NORM=OP LEFT) 'AND)
  ;; (OR (AND L.L L.R) R) ==> (AND (OR L.L R)(OR L.R R))
  (NORM=PUT=OP EXPRESSION 'AND)
          (NORM=PUT=LEFT EXPRESSION (NORM=MULTIPLY.TO.CNF (NORM=CREATE.FORMULA 'OR (NORM=LEFT LEFT) RIGHT)))
          (NORM=PUT=RIGHT EXPRESSION (NORM=MULTIPLY.TO.CNF (NORM=CREATE.FORMULA 'OR (NORM=RIGHT LEFT) (NORM=COPY RIGHT)))) EXPRESSION)
        ((EQL (NORM=OP RIGHT) 'AND)
  ;; (OR L (AND R.L R.R)) ==> (AND (OR L R.L)(OR L R.R))
  (NORM=PUT=OP EXPRESSION 'AND)
          (NORM=PUT=LEFT EXPRESSION (NORM=MULTIPLY.TO.CNF (NORM=CREATE.FORMULA 'OR LEFT (NORM=LEFT RIGHT))))
          (NORM=PUT=RIGHT EXPRESSION (NORM=MULTIPLY.TO.CNF (NORM=CREATE.FORMULA 'OR (NORM=COPY LEFT) (NORM=RIGHT RIGHT)))) EXPRESSION)
        (T EXPRESSION)))))

(DEFUN NORM=TRANSFORM.TO.PSEUDO.DNF (EXPRESSION)
  ;; INPUT:  EXPRESSION NOT CONTAINING EQV, IMPL, NOT.
  ;; VALUE:  EQUIVALENT EXPRESSION IN PSEUDO DNF, I.E.
  ;;         QUANTIFIED SUBEXPRESSIONS ARE TREATED LIKE
  ;;         LITERALS BUT THEIR SUBEXPRESSIONS IN TURN
  ;;         ARE ALSO TRANSFORMED TO DNF.
  (CASE (NORM=OP EXPRESSION) ((+ -) EXPRESSION)
    ((ALL EX) (NORM=PUT=RIGHT EXPRESSION (NORM=TRANSFORM.TO.PSEUDO.DNF (NORM=RIGHT EXPRESSION))) EXPRESSION)
    (AND (NORM=PUT=LEFT EXPRESSION (NORM=TRANSFORM.TO.PSEUDO.DNF (NORM=LEFT EXPRESSION)))
      (NORM=PUT=RIGHT EXPRESSION (NORM=TRANSFORM.TO.PSEUDO.DNF (NORM=RIGHT EXPRESSION))) (NORM=MULTIPLY.TO.DNF EXPRESSION))
    (OR (NORM=PUT=LEFT EXPRESSION (NORM=TRANSFORM.TO.PSEUDO.DNF (NORM=LEFT EXPRESSION)))
      (NORM=PUT=RIGHT EXPRESSION (NORM=TRANSFORM.TO.PSEUDO.DNF (NORM=RIGHT EXPRESSION))) EXPRESSION)
    (OTHERWISE (ERROR "ILLEGAL EXPRESSION IN NORM=TRANSFORM.TO.PSEUDO.DNF:~A" EXPRESSION))))

(DEFUN NORM=MULTIPLY.TO.DNF (EXPRESSION)
  ;; INPUT:  EXPRESSION HAS FORM (AND  LEFT RIGHT)
  ;; VALUE:  DNF OF EXPRESSION, WHERE ANY SUBEXPRESSION
  ;;         WITH AN OPERATOR OTHER THAN AND, OR, IS
  ;;         TREATED AS IF IT WERE AN ATOM.
  (mkrp-gc.start (opt-get.option gen_lisp.garbage.collection))
  (let ((LEFT (NORM=LEFT EXPRESSION))
	(RIGHT (NORM=RIGHT EXPRESSION)))
    (COND
      ((EQL (NORM=OP LEFT) 'OR)
       ;; (AND (OR L.L L.R) R) ==> (OR (AND L.L R)(AND L.R R))
       (NORM=PUT=OP EXPRESSION 'OR)
       (NORM=PUT=LEFT EXPRESSION (NORM=MULTIPLY.TO.DNF (NORM=CREATE.FORMULA 'AND (NORM=LEFT LEFT) RIGHT)))
       (NORM=PUT=RIGHT EXPRESSION (NORM=MULTIPLY.TO.DNF (NORM=CREATE.FORMULA 'AND (NORM=RIGHT LEFT) (NORM=COPY RIGHT))))
       EXPRESSION)
      ((EQL (NORM=OP RIGHT) 'OR)
       ;; (AND L (OR R.L R.R)) ==> (OR (AND L R.L)(AND L R.R))
       (NORM=PUT=OP EXPRESSION 'OR)
       (NORM=PUT=LEFT EXPRESSION (NORM=MULTIPLY.TO.DNF (NORM=CREATE.FORMULA 'AND LEFT (NORM=LEFT RIGHT))))
       (NORM=PUT=RIGHT EXPRESSION (NORM=MULTIPLY.TO.DNF (NORM=CREATE.FORMULA 'AND (NORM=COPY LEFT) (NORM=RIGHT RIGHT))))
       EXPRESSION)
      (T EXPRESSION))))

(DEFUN NORM=MULTIPLY.TO.DNF1 (EXPRESSION)
  ;; INPUT:  EXPRESSION HAS FORM (AND  LEFT RIGHT)
  ;; VALUE:  DNF OF EXPRESSION, WHERE ANY SUBEXPRESSION
  ;;         WITH AN OPERATOR OTHER THAN AND, OR, IS
  ;;         TREATED AS IF IT WERE AN ATOM.
  (mkrp-gc.start (opt-get.option gen_lisp.garbage.collection))
  (let ((LEFT (NORM=LEFT EXPRESSION))
	(RIGHT (NORM=RIGHT EXPRESSION)))
    (COND
      ((EQL (NORM=OP LEFT) 'OR)
       ;; (AND (OR L.L L.R) R) ==> (OR (AND L.L R)(AND L.R R))
       (NORM=PUT=OP EXPRESSION 'OR)
       (NORM=PUT=LEFT EXPRESSION (NORM=CREATE.FORMULA 'AND! (NORM=LEFT LEFT) RIGHT))
       (NORM=PUT=RIGHT EXPRESSION (NORM=CREATE.FORMULA 'AND! (NORM=RIGHT LEFT) (NORM=COPY RIGHT)))
       EXPRESSION)
      ((EQL (NORM=OP RIGHT) 'OR)
       ;; (AND L (OR R.L R.R)) ==> (OR (AND L R.L)(AND L R.R))
       (NORM=PUT=OP EXPRESSION 'OR)
       (NORM=PUT=LEFT EXPRESSION (NORM=CREATE.FORMULA 'AND! LEFT (NORM=LEFT RIGHT)))
       (NORM=PUT=RIGHT EXPRESSION (NORM=CREATE.FORMULA 'AND! (NORM=COPY LEFT) (NORM=RIGHT RIGHT)))
       EXPRESSION)
      (T EXPRESSION))))

(DEFUN NORM=FLATTEN.CLAUSELIST (EXPRESSION)
  ;; INPUT: EXPRESSION IN CNF CONTAINING AND,OR,LITERALS
  ;; VALUE: EXPRESSION AS LIST OF CLAUSES. AND,OR REMOVED
  (CASE (NORM=OP EXPRESSION)
    (AND (NCONC (NORM=FLATTEN.CLAUSELIST (NORM=LEFT EXPRESSION)) (NORM=FLATTEN.CLAUSELIST (NORM=RIGHT EXPRESSION))))
    (OTHERWISE (LIST (NORM=FLATTEN.LITLIST EXPRESSION)))))

(DEFUN NORM=FLATTEN.LITLIST (EXPRESSION)
  ;; INPUT: EXPRESSION IN CNF CONTAINING OR, LITERALS
  ;; VALUE: LIST OF ALL LITERALS.
  (CASE (NORM=OP EXPRESSION)
    (OR (NCONC (NORM=FLATTEN.LITLIST (NORM=LEFT EXPRESSION)) (NORM=FLATTEN.LITLIST (NORM=RIGHT EXPRESSION))))
    (OTHERWISE (LIST EXPRESSION))))

(defun norm-copy (expression)
  (norm=copy expression))

(DEFUN NORM=COPY (EXPRESSION)
  ;; Input:  expression, junctors may be marked by '!' .
  ;; value:  copy of expression, where all variables
  ;;         defined inside are renamed.
  (CASE (NORM=OP EXPRESSION)
    ((ALL EX)
     (let ((OLDVAR (NORM=LEFT EXPRESSION)) (NEWVAR (NORM=NEW.VARIABLE (NORM=LEFT EXPRESSION))))
       (LIST (NORM=OP EXPRESSION) NEWVAR (NORM=REPLACE.VARIABLE OLDVAR NEWVAR (NORM=COPY (NORM=RIGHT EXPRESSION))))))
    (NOT (NORM=CREATE.FORMULA 'NOT (NORM=COPY (NORM=LEFT EXPRESSION))))
    ((AND OR IMPL EQV AND! OR! IMPL! EQV!)
     (LIST (NORM=OP EXPRESSION) (NORM=COPY (NORM=LEFT EXPRESSION)) (NORM=COPY (NORM=RIGHT EXPRESSION))))
    ((+ -) (COPY-TREE EXPRESSION))
    (OTHERWISE (ERROR "Illegal expression in NORM=COPY: ~A" EXPRESSION))))

(DEFUN NORM=VARIABLE.OCCURS.IN (VARIABLE EXPRESSION)
  ;; EDITED:  5-AUG-84 14:16:50        NE
  ;; INPUT:  EXPRESSION, JUNCTORS MAY BE MARKED BY '!'.
  ;;         VARIABLE SELF EXPLAINING.
  ;; VALUE:  IF VARIABLE OCCURS IN SOME TERMLIST OF
  ;;         EXPRESSION THEN T ELSE NIL.
  (CASE (NORM=OP EXPRESSION)
    ((ALL EX) (AND (NEQ VARIABLE (NORM=LEFT EXPRESSION)) (NORM=VARIABLE.OCCURS.IN VARIABLE (NORM=RIGHT EXPRESSION))))
    (NOT (NORM=VARIABLE.OCCURS.IN VARIABLE (NORM=LEFT EXPRESSION)))
    ((AND OR IMPL EQV AND! OR! IMPL! EQV!)
      (OR (NORM=VARIABLE.OCCURS.IN VARIABLE (NORM=LEFT EXPRESSION)) (NORM=VARIABLE.OCCURS.IN VARIABLE (NORM=RIGHT EXPRESSION))))
    ((+ -) (some #'(lambda (term) (dt-variable.in VARIABLE term)) (NORM=RIGHT EXPRESSION)))
    (OTHERWISE (ERROR "ILLEGAL EXPRESSION IN NORM=VARIABLE.OCCURS.IN: ~A" EXPRESSION))))

(DEFUN NORM=REPLACE.VARIABLE (VARIABLE TERM EXPRESSION)
  ;; Edited:  2-aug-84 18:12:09        ne
  ;; input:  EXPRESSION, junctors may be marked by '!'.
  ;;         VARIABLE, TERM selfexplaining.
  ;; effect: each occurrence of variable in any termlist
  ;;         of expression is destructively replaced by
  ;;         a copy of term.
  ;; value:  modified expression.
  (CASE (NORM=OP EXPRESSION)
    ((ALL EX) (NORM=PUT=RIGHT EXPRESSION (NORM=REPLACE.VARIABLE VARIABLE TERM (NORM=RIGHT EXPRESSION))) EXPRESSION)
    (NOT (NORM=PUT=LEFT EXPRESSION (NORM=REPLACE.VARIABLE VARIABLE TERM (NORM=LEFT EXPRESSION))) EXPRESSION)
    ((+ -) (NORM=PUT=RIGHT EXPRESSION (NORM=REPLACE.VARIABLE.IN.TERMLIST VARIABLE TERM (NORM=RIGHT EXPRESSION))) EXPRESSION)
    ((AND OR IMPL EQV AND! OR! IMPL! EQV!)
      (NORM=PUT=LEFT EXPRESSION (NORM=REPLACE.VARIABLE VARIABLE TERM (NORM=LEFT EXPRESSION)))
      (NORM=PUT=RIGHT EXPRESSION (NORM=REPLACE.VARIABLE VARIABLE TERM (NORM=RIGHT EXPRESSION))) EXPRESSION)
    (OTHERWISE (ERROR "Illegal expression in NORM=REPLACE.VARIABLE: ~A" EXPRESSION))))

(DEFUN NORM=REPLACE.VARIABLE.IN.TERMLIST (VARIABLE TERM TERMLIST)
  ;; EDITED:  2-AUG-84 18:24:58        NE
  ;; INPUT:  SELF EXPLAINING.
  ;; EFFECT: EACH OCCURRENCE OF VARIABLE IN TERMLIST IS
  ;;         DESTRUCTIVELY REPLACED BY A COPY OF TERM.
  ;; VALUE:  MODIFIED TERMLIST.
  (MAPL
    (FUNCTION
      (LAMBDA (RESTTERMLIST)
        (COND ((EQL VARIABLE (CAR RESTTERMLIST)) (RPLACA RESTTERMLIST (COPY-TREE TERM)))
	      ((CONSP (CAR RESTTERMLIST)) (NORM=REPLACE.VARIABLE.IN.TERMLIST VARIABLE TERM (CDR (CAR RESTTERMLIST)))))))
    TERMLIST)
  TERMLIST)

(defmacro norm=create.formula (sign predicate &optional termlist props)
						; Edited:  24-JUL-1990 18:55
						; Authors: PRCKLN
						; Input:   SIGN, PREDICATE, TERMLIST, and a propertylist PROPS
						; Effect:  -
						; Value:   A formula.
  `(list ,sign ,predicate ,termlist ,props))

(defun norm-create.formula (sign predicate &optional termlist props)
						; Edited:  24-JUL-1990 18:55
						; Authors: PRCKLN
						; Input:   SIGN, PREDICATE, TERMLIST, and a propertylist PROPS
						; Effect:  -
						; Value:   A formula.
  (norm=create.formula sign predicate termlist props))

(DEFMACRO NORM=OP (X)
  ;; EDITED:| "23-NOV-81 13:27:07")
  ;; INPUT:  X HAS FORM (OPERATOR LEFT RIGHT).
  ;; VALUE:  OPERATOR.
  `(CAR ,X))

(DEFMACRO NORM=PUT=OP (X OP)
  ;; EDITED:| "23-NOV-81 13:34:57")
  ;; INPUT:  X HAS FORM (OPERATOR LEFT RIGHT).
  ;; EFFECT: REPLACES OPERATOR IN X BY OP.
  ;; VALUE:  UNDEFINED.
  `(RPLACA ,X ,OP))

(DEFMACRO NORM=LEFT (X)
  ;; EDITED:| "23-NOV-81 13:28:04")
  ;; INPUT:  X HAS FORM (OPERATOR LEFT RIGHT).
  ;; VALUE:  LEFT.
  `(SECOND ,X))

(DEFMACRO NORM=PUT=LEFT (X LEFT)
  ;; EDITED:| "23-NOV-81 13:35:43")
  ;; INPUT:  X HAS FORM (OPERATOR OLDLEFT RIGHT).
  ;; EFFECT: REPLACES OLDLEFT IN X BY LEFT.
  ;; VALUE:  UNDEFINED.
  `(RPLACA (CDR ,X) ,LEFT))

(DEFMACRO NORM=RIGHT (X)
  ;; EDITED:| "23-NOV-81 13:28:33")
  ;; INPUT:  X HAS FORM (OPERATOR LEFT RIGHT).
  ;; VALUE:  RIGHT.
  `(THIRD ,X))

(DEFMACRO NORM=PUT=RIGHT (X RIGHT)
  ;; EDITED:| "23-NOV-81 13:36:09")
  ;; INPUT:  X HAS FORM (OPERATOR LEFT OLDRIGHT).
  ;; EFFECT: REPLACES OLDRIGHT IN X BY RIGHT.
  ;; VALUE:  UNDEFINED.
  `(RPLACA (CDDR ,X) ,RIGHT))

(defun norm=put=index (x index)
  (setf (getf (fourth x) 'Index) index))

(DEFVAR NORM*FUNCTION.COUNTER 0)

(DEFVAR NORM*CONSTANT.COUNTER 0)

(DEFVAR NORM*MAX.SPLIT.DEPTH 270435455)

(DEFVAR NORM*EXPRESSION NIL)

(DEFVAR NORM*EXPRESSION.LENGTH NIL)

(DEFVAR NORM*EXPRESSION.LENGTH.ORIGINAL NIL)

(DEFVAR NORM*FILE.VERSION NIL)





(DEFMACRO NORM=DROP.VARIABLE (OLD.VARIABLE)
  ;; EDITED:  2-AUG-84 12:23:51        NE
  ;; INPUT:  A VARIABLE.
  ;; EFFECT: DELETES THE VARIABLE.
  ;; VALUE:  UNDEFINED.
  `(DT-VARIABLE.DELETE ,OLD.VARIABLE))

(DEFMACRO NORM=NEW.VARIABLE (OLD.VARIABLE)
  ;; EDITED:  2-AUG-84 12:25:44        NE
  ;; INPUT:  A VARIABLE.
  ;; EFFECT: CREATES A NEW VARIABLE OF THE SAME SORT.
  ;; VALUE:  THE NEW VARIABLE.
  `(DT-VARIABLE.CREATE (DT-VARIABLE.SORT ,OLD.VARIABLE)))

(DEFUN NORM=NEW.TERM (DOMAIN.VARIABLES RANGE.VARIABLE)
  ;; EDITED:  2-AUG-84 12:27:16        NE
  ;; INPUT:  A LIST OF VARIABLES AND A VARIABLE.
  ;; EFFECT: CREATES A NEW CONSTANT OR FUNCTION OBJECT.
  ;; VALUE:  NEW SKOLEM TERM OF THE RESPECTIVE DOMAIN
  ;;         AND RANGE.
  (COND ((NULL DOMAIN.VARIABLES)
	 (SETQ NORM*CONSTANT.COUNTER (1+ NORM*CONSTANT.COUNTER))
	 (DT-CONSTANT.CREATE (format nil "C_~A" NORM*CONSTANT.COUNTER)
			     (DT-VARIABLE.SORT RANGE.VARIABLE) T))
	(T (SETQ NORM*FUNCTION.COUNTER (1+ NORM*FUNCTION.COUNTER))
	   (CONS
	     (DT-FUNCTION.CREATE (format nil "F_~A" NORM*FUNCTION.COUNTER)
				 (DT-VARIABLE.SORT RANGE.VARIABLE)
				 (MAPCAR #'(lambda (var) (DT-VARIABLE.SORT var)) DOMAIN.VARIABLES)
				 NIL T)
	     DOMAIN.VARIABLES))))

