;;; -*- Mode: LISP; Syntax: Common-lisp; Package: MARKGRAF-KARL; Base: 10 -*-

(IN-PACKAGE "MARKGRAF-KARL" :use '("CL") :nicknames '("MKRP"))

(DEFMACRO DT-SORT.TRANSITIVE.CLOSURE (SORT)
  ;; EDITED: 11-FEB-83 17:43:08
  ;; INPUT:  A SORT SYMBOL
  ;; VALUE:  A LIST WITH ALL SUBSORTS OF SORT.
  `(GET ,SORT 'DT*TRANSITIVE.CLOSURE))

(DEFMACRO DT-SORT.DIRECT.SUBSORTS (SORT)
  ;; INPUT: A SORT
  ;; VALUE: THE DIRECT SUBSORTS OF THIS SORT
  `(GET ,SORT 'DT*DIRECT.SUBSORTS))

(DEFMACRO DT-SORT.DIRECT.SUPERSORTS (SORT)
  ;;  INPUT: A SORT
  ;;  VALUE: THE DIRECT SUPERSORTS OF THIS SORT
  `(GET ,SORT 'DT*DIRECT.SUPERSORTS))

(defparameter dt*commons.to.save
	'(DT*SORT.ALL
	   DT*SORT.NR 
	   DT*SORT.PROPERTIES
	   DT*SORT.COMMON.COMPUTE.FLAG
	   dt*element.predicate 
	   dt*omega.constant
	   DT*VARIABLE.COUNTER
	   DT*CONSTANT.COUNTER
	   DT*CONSTANT.ALL
	   DT*ABBREVIATIONS
	   DT*FUNCTION.COUNTER 
	   DT*FUNCTION.ALL 
	   DT*FUNCTION.ADMISSIBLE.THEORIES
	   DT*FUNCTION.ACTUAL.THEORIES
	   DT*FUNCTION.WITH.ARGUMENT.SYMMETRIES
	   DT*FUNCTION.COMPONENTS
	   DT*PREDICATE.ADMISSABLE.ATTRIBUTES
	   DT*PREDICATE.COUNTER
	   DT*EQUALITY.SYMBOLS
	   DT*EQUALITY.PREDICATES
	   DT*NONEQUALITY.PREDICATES
	   DT*PREDICATE.ALL
	   DT*PREDICATE.WITH.ATTRIBUTES
	   DT*PREDICATE.COMPONENTS
	   DT*TRUE.PREDICATE
	   DT*FALSE.PREDICATE
	   DT*UNI.CREATES.VARIABLES
	   DT*SIGN.MINUS.SYMBOLS
	   DT*SIGN.PLUS.SYMBOLS
	   DT*SYMBOL.KINDS)
  "These names  are used in dt-save")

(DEFVAR DT*SORT.ALL NIL)

(DEFVAR DT*SORT.NR 1)

(DEFvar DT*SORT.PROPERTIES
	'(DT*MAX.SUBSORTS DT*LEAST.SUPERSORTS DT*TRANSITIVE.CLOSURE DT*INVERSE.TRANSITIVE.CLOSURE DT*MINIMAL.SUBSORTS
			  DT*DIRECT.SUBSORTS DT*DIRECT.SUPERSORTS DT*DISJOINT.SORTS DT*COMPLETION.SORT))

(DEFVAR DT*SORT.COMMON.COMPUTE.FLAG NIL)


(DEFMACRO DT-SORT.IS.SUBSORT (SUBSORT SUPERSORT)
  ;; EDITED: 11-FEB-83 16:35:00
  ;; INPUT:  TWO SORTS
  ;; VALUE:  T IF SUBSORT IS INDEED A SUBSORT OF
  ;;         SUPERSORT, ELSE NIL.
  `(MEMBER ,SUBSORT (GET ,SUPERSORT 'DT*TRANSITIVE.CLOSURE)))

(DEFMACRO DT-SORT.INSERT (SORT SUPERSORT SUBSORT)
  ;; INPUT: A SORT,ITS NEW DIRECT.SUPERSORT AND THE
  ;;      NEW DIRECT SUBSORTS
  ;;  EFFECT: THE SORT WILL BE INTRODUCED AND ALL PROPERTIES")
  ;;           ARE UPDATED
  ;; REMARK: SORT IS A NEW SYMBOL OR CREATED WITH
  ;;  (DT-SORT.NEW.SYMBOL).                              " )
  `(DT=SORT.INSERT ,SORT ,SUPERSORT ,SUBSORT))



(DEFMACRO DT-SORT.ALL NIL
  ;; VALUE: ALL ACTUAL SORTS
  'DT*SORT.ALL)

(DEFMACRO DT-SORT.MINIMAL.SUBSORTS (SORT)
  ;; INPUT: A SORT
  ;; VALUE: ALL MINIMAL SUBSORTS OF SORT.
  `(GET ,SORT 'DT*MINIMAL.SUBSORTS))

(DEFMACRO DT-SORT.NUMBER (SORT) `(GET ,SORT 'DT*SORT.NUMBER))



(DEFun DT-SORT.ST.PUT.DIRECT.SUPERSORTS (SORT SUPERSORTS)
  ;; INPUT:   A SORT AND ITS NEW DIRECT SUPERSORTS
  ;; EFFECT:  THE SORT HAS NOW SUPERSORTS AS SUPERSORTS
  ;;          AND THE SORT-STRUCTURE IS UPDATED
  ;; Remark:  THIS FUNCTION SHOULD BE USED ONLY IN ST.
  (setf (get SORT 'DT*DIRECT.SUPERSORTS) (DELETE 'ANY SUPERSORTS))
  (DT=SORT.ST.UPDATE.SORTS))


(DEFMACRO DT-SORT.INVERSE.TRANSITIVE.CLOSURE (SORT)
  ;; INPUT: A SORT
  ;; VALUE: THE INVERSE TRANSITIVE.CLOSURE OF SORT
  `(GET ,SORT 'DT*INVERSE.TRANSITIVE.CLOSURE))



(DEFMACRO DT-SORT.COMMON.COMPUTE.FLAG NIL 'DT*SORT.COMMON.COMPUTE.FLAG)


(DEFMACRO DT-SORT.DISJOINT.SORTS (SORT)
  ;; INPUT: A SORT
  ;; VALUE : ALL DISJOINT SORTS ")
  `(GET ,SORT 'DT*DISJOINT.SORTS))

(DEFMACRO DT-SORT.IS.DISJOINT.WITH (SORT1 SORT2)
  ;; INPUT: TWO SORTS
  ;; VALUE: T, IF THE SORTS ARE DISJOINT; NIL ELSE
  `(MEMBER ,SORT1 (GET ,SORT2 'DT*DISJOINT.SORTS)))


(DEFMACRO DT-SORT.GREATEST.COMMON.SUBSORT (SORT1 SORT2)
  ;; INPUT: TWO SORTS
  ;; VALUE: THE GREATEST COMMON SUBSORT , OR NIL
  ;; REMARK: CALL ONLY IF SORT-LATTICE IS COMPLETED
  `(CAR (DT-SORT.GREATEST.COMMON.SUBSORTS ,SORT1 ,SORT2)))

(DEFMACRO DT-SORT.LEAST.COMMON.SUPERSORT (SORT1 SORT2)
  ;; INPUT: TWO SORTS
  ;; VALUE: THE LEAST  OMMON SUPERSORT , OR NIL
  ;; REMARK: CALL ONLY IF SORT-LATTICE IS COMPLETED
  `(CAR (DT-SORT.LEAST.COMMON.SUPERSORTS ,SORT1 ,SORT2)))



(DEFMACRO DT-SORT.PUT.DISJOINTS (SORT SORTLIST)
  ;; INPUT: A SORT AND A LIST OF DISJOINT SORTS
  ;; VALUE: UNDEFINED
  ;; EFFECT: THE PROPERTY DT*DISJOINT.SORTS IS CHANGED
  `(SETF (get ,SORT 'DT*DISJOINT.SORTS) ,SORTLIST))

(DEFMACRO DT-SORT.PUT.INV.TRANS.CLOSURE (SORT SORTLIST)
  ;; INPUT: A SORT AND IT'S NEW INV.TRANSITIVE CLOSURE ")
  ;; EFFECT: THE INV.TRANSITIVE CLOSURE OF SORT IS CHANGED ")
  ;; REMARK: USE ONLY IN MODUL SO.
  `(SETF (get ,SORT 'DT*INVERSE.TRANSITIVE.CLOSURE) ,SORTLIST))

(DEFMACRO DT-SORT.PUT.TRANS.CLOSURE (SORT SORTLIST)
  ;; INPUT: A SORT AND IT'S NEW TRANSITIVE CLOSURE ")
  ;; EFFECT: THE TRANSITIVE CLOSURE OF SORT IS CHANGED ")
  ;; REMARK: USE ONLY IN MODUL SO.
  `(SETF (get ,SORT 'DT*TRANSITIVE.CLOSURE) ,SORTLIST))

(DEFMACRO DT-SORT.UPDATE.ALL NIL
  ;; INPUT; NIL ; VALUE: NIL
  ;; EFFECT: ALL SORT PROPERTIES ARE COMPUTED
  ;;  ON THE BASIS OF DIRECT SUBSORTS
  `(progn (DT=SORT.UPDATE.MAX.SUBSORTS) (DT=SORT.UPDATE.LEAST.SUPERSORTS)
	  (DT=SORT.UPDATE.MINIMAL.SUBSORTS)))


(DEFMACRO DT-SORT.UPDATE.DIRECT.SUBSORTS NIL
  ;; INPUT: NIL
  ;; EFFECT: ALL DIRECT.SUBSORTS ARE COMPUTED OUT OF
  ;;         THE TRANSITIVE CLOSURE
  `(progn (DT=SORT.UPDATE.DIRECT.SUBSORTS.OUT.OF.TRANS.CLOSURE)
	  (DT=SORT.UPDATE.MAX.SUBSORTS T) (DT=SORT.UPDATE.LEAST.SUPERSORTS T) (DT=SORT.UPDATE.MINIMAL.SUBSORTS)))

(DEFUN DT-SORT.CREATE (SORT DIRECT.SUBSORTS NO.LEAST.UPDATE)
  ;; EDITED: 11-FEB-83 09:38:54
  ;; INPUT:  A SORT SYMBOL AND ITS DIRECT SUBSORTS
  ;;         (NOT THE TRANSITIVE CLOSURE])
  ;; EFFECT: THE SORT IS INSERTED INTO DT*SORT.ALL,
  ;;         THE DIRECT SUBSORTS ARE INSERTED INTO
  ;;         THE PROPERTYLIST OF SORT AND THE
  ;;         GREATEST COMMON SUBSORTS ARE UPDATED.
  ;; VALUE:  SORT
  (REMPROPS SORT DT*SORT.PROPERTIES)
  (SETf DT*SORT.ALL (CONS SORT DT*SORT.ALL)
	(get SORT 'DT*DIRECT.SUBSORTS) DIRECT.SUBSORTS
	(get SORT 'DT*SORT.NuMBER) DT*SORT.NR)
  (incf DT*SORT.NR)
  (when (EVERY #'(LAMBDA (SORT) (MEMBER SORT DT*SORT.ALL)) DIRECT.SUBSORTS)
    (DT=SORT.UPDATE.MAX.SUBSORTS)
    (DT=SORT.UPDATE.MINIMAL.SUBSORTS)
    (unless NO.LEAST.UPDATE (DT=SORT.UPDATE.LEAST.SUPERSORTS)))
  SORT)

(DEFUN DT-SORT.GREATEST.COMMON.SUBSORTS (SORT1 SORT2)
  ;; EDITED: 11-FEB-83 09:35:59
  ;; INPUT:  TWO SORT SYMBOLS
  ;; VALUE:  THE GREATEST COMMON SUBSORTS OF THESE TWO
  ;;         SORTS, RESP. NIL.
  (COND (DT*SORT.COMMON.COMPUTE.FLAG (DT=SORT.MAX.SUBSORTS SORT1 SORT2))
	(T (CdR (ASSOC SORT1 (GET SORT2 'DT*MAX.SUBSORTS))))))





(DEFUN DT-SORT.GREATEST.COMMON.SUBSORT.OF.LIST (SORTLIST)
  ;; INPUT: A LIST OF SORTS
  ;; VALUE: THE GREATEST COMMON SUBSORT  OF   THIS SORTS
  (PROG ((MAX.SORT (CAR SORTLIST)))
	(MAPC #'(LAMBDA (SORT) (SETQ MAX.SORT (DT-SORT.GREATEST.COMMON.SUBSORT MAX.SORT SORT))) (CDR SORTLIST))
	(RETURN MAX.SORT)))



(DEFUN DT-SORT.PUT.COMMON.COMPUTE.FLAG (FLAG) (SETQ DT*SORT.COMMON.COMPUTE.FLAG FLAG))

(DEFUN DT-SORT.UPDATE.MINIMAL.SUBSORTS (SORT)
  ;; NO INPUT, VALUE UNDEFINED
  ;; SIDEEFFECTS: THE PROPERTY MINIMAL SUBSORTS FOR
  ;;              EVERY SORT IS UPDATED
  (COND (SORT (SETF (get SORT 'DT*MINIMAL.SUBSORTS) (DT=SORT.MINIMAL.SUBSORTS SORT)))
	(T
	 (MAPC
	   #'(LAMBDA (SORT)
	       (SETF (get SORT 'DT*MINIMAL.SUBSORTS) (DT=SORT.MINIMAL.SUBSORTS SORT)))
	   (DT-SORT.ALL)))))


(DEFUN DT-SORT.ST.REMOVE (SORT)
  ;; INPUT: A SORT
  ;; EFFECT: THE SORT IS REMOVED FROM THE SORT-STRUCTURE
  ;;         AND ALL NEEDED SORT-PROPERTIES ARE UPDATED
  (SETQ DT*SORT.ALL (DELETE SORT DT*SORT.ALL))
  (MAPC
    #'(LAMBDA (ALL.SORT)
        (SETF (get ALL.SORT 'DT*DIRECT.SUBSORTS) (DELETE SORT (DT-SORT.DIRECT.SUBSORTS ALL.SORT))))
    (DT-SORT.ALL))
  (REMPROPS SORT DT*SORT.PROPERTIES) (DT=SORT.UPDATE.MAX.SUBSORTS) (DT=SORT.UPDATE.LEAST.SUPERSORTS)
  (DT=SORT.UPDATE.MINIMAL.SUBSORTS))


(DEFUN DT-SORT.LATTICE.COMPLETION NIL
  ;; INPUT: NO INPUT
  ;; VALUE: UNDEFINED
  ;; SIDE-EFFECTS: THE SORT-STRUCTURE IS CHANGED.
  ;;               NEW SORTS ARE ADDED, AND AFTER THIS
  ;;               THE GREATEST COMMON SUBSORT OF TWO
  ;;               SORTS IS UNIQUE,IF IT EXISTS.
  ;; I.E. THE SORT STRUCTURE IS A SEMILATTICE.
  (PROG (SUBSORTS (NEW.SORTS (DT=SORT.CREATE.ALL.INTERSECTIONS)))
	(MAPC #'(LAMBDA (SORT) (SETF (get SORT 'DT*COMPLETION.SORT) T)) NEW.SORTS)
	(SETQ DT*SORT.ALL (NCONC DT*SORT.ALL NEW.SORTS))
	(MAPC
	  #'(LAMBDA (SORT1)
	      (MAPC
		#'(LAMBDA (SORT2)
		    (COND
		      ((AND (SUPERSET (GET SORT1 'DT*SORT.SET) (GET SORT2 'DT*SORT.SET)) (NEQ SORT1 SORT2))
		       (SETQ SUBSORTS (CONS SORT2 SUBSORTS)))))
		DT*SORT.ALL)
	      (SETF (get SORT1 'DT*DIRECT.SUBSORTS)
		    (MAXIMA SUBSORTS
			    #'(LAMBDA (SORT.REMOVE SORT.COMPARE)
				(SUPERSET (GET SORT.COMPARE 'DT*SORT.SET) (GET SORT.REMOVE 'DT*SORT.SET)))))
	      (SETQ SUBSORTS NIL))
	  DT*SORT.ALL)
	(MAPC #'(LAMBDA (SORT1) (REMPROP SORT1 'DT*SORT.SET)) DT*SORT.ALL)
	(DT=SORT.UPDATE.MAX.SUBSORTS)
	(DT=SORT.UPDATE.LEAST.SUPERSORTS) (DT=SORT.UPDATE.MINIMAL.SUBSORTS)))



(DEFUN DT-SORT.DELETE.ALL NIL
  ;; NO INPUT, VALUE UNDEFINED
  ;; SIDEEFFECTS: ALL SORTS ARE DELETED
  (MAPC #'(LAMBDA (SORT) (REMPROPS SORT DT*SORT.PROPERTIES)) DT*SORT.ALL) (SETQ DT*SORT.ALL NIL))

(DEFUN DT-SORT.LEAST.COMMON.SUPERSORTS (SORT1 SORT2)
  ;; INPUT: TWO SORTS
  ;; VALUE: THE LIST OF THE LEAST COMMON SUPERSORTS
  (COND ((AND SORT1 SORT2) (CdR (ASSOC SORT1 (GET SORT2 'DT*LEAST.SUPERSORTS))))
	((NULL SORT1) (LIST SORT2))
	((NULL SORT2) (LIST SORT1))))


(DEFUN DT-SORT.ADD.TO.INV.TRANS.CLOSURE (ADD.SORT SORT)
  ;;  INPUT: A SORT OR A SORT LIST TO BE ADDED.
  ;;        AND A SORT, WHICH PROPERTIES ARE TO BE CHANGD
  ;; EFFECT: ADD.SORT IS ADDED TO TRANS.CLOSURE OF SORT
  ;; THIS FUNCTION SHOULD BE USED ONLY IN MODULE SO.
  (COND
    ((ATOM ADD.SORT)
     (SETF (get SORT 'DT*INVERSE.TRANSITIVE.CLOSURE)
	   (INSERT ADD.SORT (GET SORT 'DT*INVERSE.TRANSITIVE.CLOSURE))))
    (T
     (PROG ((PROPLIST (GET SORT 'DT*INVERSE.TRANSITIVE.CLOSURE)))
	   (MAPC #'(LAMBDA (ADD.SORT.EL) (SETQ PROPLIST (INSERT ADD.SORT.EL PROPLIST))) ADD.SORT)
	   (SETF (get SORT 'DT*INVERSE.TRANSITIVE.CLOSURE) PROPLIST)))))

(DEFUN DT-SORT.ADD.TO.TRANS.CLOSURE (ADD.SORT SORT)
  ;;  INPUT: A SORT OR A SORT LIST TO BE ADDED.
  ;;        AND A SORT, WHICH PROPERTIES ARE TO BE CHANGD
  ;; EFFECT: ADD.SORT IS ADDED TO TRANS.CLOSURE OF SORT
  ;; THIS FUNCTION SHOULD BE USED ONLY IN MODULE SO.
  (COND
    ((ATOM ADD.SORT)
     (SETF (get SORT 'DT*TRANSITIVE.CLOSURE) (INSERT ADD.SORT (GET SORT 'DT*TRANSITIVE.CLOSURE))))
    (T
     (PROG ((PROPLIST (GET SORT 'DT*TRANSITIVE.CLOSURE)))
	   (MAPC #'(LAMBDA (ADD.SORT.EL) (SETQ PROPLIST (INSERT ADD.SORT.EL PROPLIST))) ADD.SORT)
	   (SETF (get SORT 'DT*TRANSITIVE.CLOSURE) PROPLIST)))))

(DEFUN DT-SORT.CLEAR.SORTS (DELETED.SORTS)
  ;; INPUT: A LIST OF SORTS TO BE DELETED ")
  ;;  EFFECT: ALL PROPERTIES OF SORT ARE DELETED ")
  ;;          AND SORTS ARE REMOVED FROM DT*SORT.ALL ")
  (MAPC
    #'(LAMBDA (DEL.SORT)
        (REMPROPS DEL.SORT
		  '(DT*MAX.SUBSORTS DT*LEAST.SUPERSORTS DT*TRANSITIVE.CLOSURE DT*INVERSE.TRANSITIVE.CLOSURE DT*MINIMAL.SUBSORTS
				    DT*DIRECT.SUBSORTS DT*DIRECT.SUPERSORTS DT*DISJOINT.SORTS)))
    DELETED.SORTS)
  (SETQ DT*SORT.ALL (SET-DIFFERENCE (DT-SORT.ALL) DELETED.SORTS)))




(DEFUN DT-SORT.NEW.SYMBOL (SORT)
  ;; INPUT: NIL
  ;; VALUE: A NEW SORT SYMBOL ")
  ;; EFFECT: THE SORT IS ADDED TO DT*SORT.ALL
  (COND ((NULL SORT) (SETQ SORT (DT=SORT.NEW.SYMBOL)))
	(T (SETF (get SORT 'DT*SORT.NUMBER) DT*SORT.NR) (SETQ DT*SORT.NR (1+ DT*SORT.NR))))
  (SETQ DT*SORT.ALL (INSERT SORT DT*SORT.ALL)) SORT)




(DEFUN DT-SORT.UPDATE.MAX.SUBSORTS NIL
  ;; EDITED: 11-FEB-83 09:31:39
  ;; EFFECT: FOR EACH SORT AN ASSOCIATION LIST IS CREATED
  ;;         KEY: A SORT, VALUE: THE GREATEST COMMON
  ;;         SUBSORTS.
  ;; VALUE:  UNDEFINED.
  (MAPC
    #'(LAMBDA (SORT1)
        (SETF (get SORT1 'DT*MAX.SUBSORTS)
	      (MAPCAR #'(LAMBDA (SORT2) (CONS SORT2 (DT=SORT.MAX.SUBSORTS SORT1 SORT2))) (DT-SORT.ALL))))
    (DT-SORT.ALL)))

(DEFUN DT=SORT.CREATE.ALL.INTERSECTIONS NIL
  ;; INPUT: NIL
  ;; VALUE: THE NAMES OF THE NEW CREATED SORTS
  ;; SIDEEFFECTS: EVERY SORT HAS WITH PROPERTY-KEY
  ;;             DT*SORT.SET THE TRANSITIVE.CLOSURE
  ;;             OF THE RELATION SUBSORT (INCLUDING THE
  ;;             NEW CREATED SORTS).
  (PROG
    ((SORT.SET.OLD (MAPCAR #'(LAMBDA (OLD.SORT) (GET OLD.SORT 'DT*TRANSITIVE.CLOSURE)) DT*SORT.ALL)) SORT.SET.ALL.NEW
     (SORT.SET.NEW T) SORT.SET.LAST INTERSECTION.SET SORT.SYMBOL)
    (SETQ SORT.SET.LAST SORT.SET.OLD)
    (MAPC
      #'(LAMBDA (OLD.SORT)
          (SETF (get OLD.SORT 'DT*SORT.SET) (GET OLD.SORT 'DT*TRANSITIVE.CLOSURE)))
      DT*SORT.ALL)
    (WHILE SORT.SET.NEW (SETQ SORT.SET.NEW NIL)
	   (MAPC
	     #'(LAMBDA (OLD.SET)
		 (MAPC
		   #'(LAMBDA (LAST.SET) (SETQ INTERSECTION.SET (INTERSECTION OLD.SET LAST.SET))
			     (COND
			       ((AND (CONSP INTERSECTION.SET) (NOT (MEMBER* INTERSECTION.SET SORT.SET.ALL.NEW))
				     (NOT (MEMBER* INTERSECTION.SET SORT.SET.OLD)) (NOT (MEMBER* INTERSECTION.SET SORT.SET.NEW)))
				(SETQ SORT.SET.NEW (CONS INTERSECTION.SET SORT.SET.NEW)))))
		   SORT.SET.LAST))
	     SORT.SET.OLD)
	   (SETQ SORT.SET.LAST SORT.SET.NEW) (SETQ SORT.SET.ALL.NEW (NCONC SORT.SET.ALL.NEW SORT.SET.NEW)))
    (RETURN
      (MAPCAR
        #'(LAMBDA (SORT.SET)
            (PROG1 (SETQ SORT.SYMBOL (DT=SORT.NEW.SYMBOL))
		   (SETF (get SORT.SYMBOL 'DT*SORT.SET) SORT.SET)))
        SORT.SET.ALL.NEW))))

(DEFUN DT=SORT.DIRECT.SUPERSORTS (SORT)
  ;; INPUT: A SORT
  ;; VALUE: ALL DIRECT SUPERSORTS OF SORT
  (REMOVE-DUPLICATES (MAPCAN #'(LAMBDA (SORT.LOOP) (COND ((MEMBER SORT (GET SORT.LOOP 'DT*DIRECT.SUBSORTS)) (LIST SORT.LOOP))))
			     DT*SORT.ALL)))

(DEFUN DT=SORT.INSERT (SORT DIRECT.SUPERSORT DIRECT.SUBSORTS)
  ;; INPUT: A NEW SORT, THE DIRECT SUPERSORT
  ;;        AND THE DIRECT SUBSORTS.
  ;; VALUE: UNDEFINED
  ;; SIDEEFFECTS:  THE NEW SORTS IS ADDED, AND THE
  ;;               SORT.STRUCTURE IS UPDATED
  (SETQ DT*SORT.ALL (INSERT SORT DT*SORT.ALL))
  (SETF (get SORT 'DT*DIRECT.SUBSORTS) DIRECT.SUBSORTS)
  (SETF (get DIRECT.SUPERSORT 'DT*DIRECT.SUBSORTS)
	(SET-DIFFERENCE (CONS SORT (GET DIRECT.SUPERSORT 'DT*DIRECT.SUBSORTS)) DIRECT.SUBSORTS))
  (DT=SORT.UPDATE.MAX.SUBSORTS) (DT=SORT.UPDATE.MINIMAL.SUBSORTS) (DT=SORT.UPDATE.LEAST.SUPERSORTS))

(DEFUN DT=SORT.UPDATE.LEAST.SUPERSORTS (&optional INVERSE.UPDATE.FLAG)
  ;; EDITED: 11-FEB-83 09:31:39
  ;; EFFECT: FOR EACH SORT AN ASSOCIATION LIST IS CREATED
  ;;         KEY: A SORT, VALUE: THE LEAST    COMMON
  ;;         SUPERSORTS.
  ;; VALUE:  UNDEFINED.
  (unless INVERSE.UPDATE.FLAG (DT=SORT.UPDATE.INVERSE.TRANS.CLOSURE))
  (MAPC
    #'(LAMBDA (SORT1)
        (SETF (get SORT1 'DT*LEAST.SUPERSORTS)
	      (MAPCAR #'(LAMBDA (SORT2) (CONS SORT2 (DT=SORT.LEAST.SUPERSORTS SORT1 SORT2))) DT*SORT.ALL)))
    DT*SORT.ALL))

(DEFUN DT=SORT.UPDATE.INVERSE.TRANS.CLOSURE NIL
  ;; NO INPUT,VALUE UNDEFINED
  ;; SIDEFFECTS: FOR ALL SORTS, INVERSE.TRANS.CLOSURE
  ;;             IS UPDATED, DUE TO DIRECT.SUPERSORTS
  (MAPC
    #'(LAMBDA (SORT1)
        (SETF (get SORT1 'DT*DIRECT.SUPERSORTS) (DT=SORT.DIRECT.SUPERSORTS SORT1)))
    DT*SORT.ALL)
  (MAPC
    #'(LAMBDA (SORT1)
        (SETF (get SORT1 'DT*INVERSE.TRANSITIVE.CLOSURE)
	      (REMOVE-DUPLICATES (DT=SORT.INVERSE.TRANSITIVE.CLOSURE SORT1))))
    DT*SORT.ALL))

(DEFUN DT=SORT.INVERSE.TRANSITIVE.CLOSURE (SORT)
  ;; EDITED: 10-FEB-83 17:04:36
  ;; INPUT:  A SORT SYMBOL
  ;; VALUE:  A LIST OF ALL SUBSORTS OF SORT.
  (PROG ((TRANS.CLO.NEW (LIST SORT)) TRANS.CLO.OLD)
	(WHILE (NOT (SET= TRANS.CLO.OLD TRANS.CLO.NEW)) (SETQ TRANS.CLO.OLD TRANS.CLO.NEW)
	       (MAPC
		 #'(LAMBDA (TRANS.SORT) (SETQ TRANS.CLO.NEW (UNION (GET TRANS.SORT 'DT*DIRECT.SUPERSORTS) TRANS.CLO.NEW)))
		 TRANS.CLO.OLD))
	(RETURN TRANS.CLO.OLD)))

(DEFUN DT=SORT.LEAST.SUPERSORTS (SORT1 SORT2)
  ;; EDITED: 20-FEB-84 17:27:36
  ;; INPUT:  TWO SORT SYMBOLS
  ;; VALUE:  THE LEAST COMMON SUPERSORTS OF THE TWO
  ;;         SORTS.
  ;; THIS FUNCTION SHOULD BE CALLED ONLY IN FUNCTION
  ;;      DT=SORT.UPDATE.LEAST.SUPERSORTS.
  (COND ((EQL SORT1 SORT2) (LIST SORT1)) ((MEMBER SORT1 (GET SORT2 'DT*INVERSE.TRANSITIVE.CLOSURE)) (LIST SORT1))
	((MEMBER SORT2 (GET SORT1 'DT*INVERSE.TRANSITIVE.CLOSURE)) (LIST SORT2))
	(T
	 (PROG (LEAST.SUPERSORT.LIST)
	       (SETQ LEAST.SUPERSORT.LIST
		     (INTERSECTION (GET SORT1 'DT*INVERSE.TRANSITIVE.CLOSURE) (GET SORT2 'DT*INVERSE.TRANSITIVE.CLOSURE)))
	       (RETURN
		 (MAXIMA LEAST.SUPERSORT.LIST #'(LAMBDA (SORT.REM SORT.COMP) (DT-SORT.IS.SUBSORT SORT.COMP SORT.REM))))))))

(DEFUN DT=SORT.ST.UPDATE.SORTS NIL
  ;; NO INPUT
  ;; EFFECT: AFTER A NEW SETTING OF DIRECT SUPERSORTS
  ;;         THE SORT-STRUCTURE IS UPDATED FOR SYMBOLTBL
  (let (TRANS.CLOSURE TRANS.C CONTINUE)
    (MAPC #'(LAMBDA (SORT)
	      (COND ((AND (NULL (DT-SORT.DIRECT.SUPERSORTS SORT)) (NEQ SORT 'ANY))
		     (SETF (get SORT 'DT*DIRECT.SUPERSORTS) (LIST 'ANY)))))
	  (DT-SORT.ALL))
    (MAPC #'(LAMBDA (SORT)
	      (SETQ TRANS.CLOSURE (UNION (LIST SORT) (DT-SORT.DIRECT.SUPERSORTS SORT))) (SETQ CONTINUE T)
	      (WHILE CONTINUE
		(SETQ CONTINUE NIL)
		(MAPC #'(LAMBDA (SORT.IN.TRANS.C)
			  (SETQ TRANS.C (UNION TRANS.CLOSURE (DT-SORT.DIRECT.SUPERSORTS SORT.IN.TRANS.C)))
			  (COND ((NOT (SET= TRANS.C TRANS.CLOSURE)) (SETQ TRANS.CLOSURE TRANS.C) (SETQ CONTINUE T))))
		      TRANS.CLOSURE))
	      (SETF (get SORT 'DT*INVERSE.TRANSITIVE.CLOSURE) TRANS.CLOSURE))
	  (DT-SORT.ALL))
    (MAPC #'(LAMBDA (SORT)
	      (SETF (get SORT 'DT*DIRECT.SUBSORTS)
		    (REMOVE-IF-NOT #'(LAMBDA (SUBSORT) (MEMBER SORT (DT-SORT.DIRECT.SUPERSORTS SUBSORT))) (DT-SORT.ALL))))
	  (DT-SORT.ALL))
    (DT=SORT.UPDATE.MAX.SUBSORTS)))

(DEFUN DT=SORT.MINIMAL.SUBSORTS (SORT)
  ;; INPUT: A SORT
  ;; VALUE: ALL MINIMAL SUBSORTS OF SORT (COMPUTED)
  (MAXIMA (COPY-TREE (GET SORT 'DT*TRANSITIVE.CLOSURE)) #'(LAMBDA (SORT1 SORT2) (DT-SORT.IS.SUBSORT SORT2 SORT1))))

(DEFun DT=SORT.UPDATE.TRANS.CLOSURE NIL
  ;; NO INPUT, VALUE UNDEFINED
  ;; SIDEEFFECTS: THE TRANSITIVE CLOSURE OF ALL SORTS
  ;;              IS UPDATED DUE TO DIRECT.SUBSORTS
  (MAPC #'(LAMBDA (SORT1)
	    (SETF (get SORT1 'DT*TRANSITIVE.CLOSURE)
		  (REMOVE-DUPLICATES (DT=SORT.TRANSITIVE.CLOSURE SORT1))))
	DT*SORT.ALL))

(DEFMACRO DT=SORT.UPDATE.MINIMAL.SUBSORTS NIL
  ;; NO INPUT, VALUE UNDEFINED
  ;; SIDEEFFECTS: THE PROPERTY MINIMAL SUBSORTS FOR
  ;;              EVERY SORT IS UPDATED
  `(MAPC #'(LAMBDA (SORT) (SETF (get SORT 'DT*MINIMAL.SUBSORTS) (DT=SORT.MINIMAL.SUBSORTS SORT)))
	 (DT-SORT.ALL)))

(DEFUN DT=SORT.UPDATE.DIRECT.SUBSORTS.OUT.OF.TRANS.CLOSURE NIL
  ;;  SEE FUNCTION DT-SORT.UPDATE.DIRECT.SUBSORTS
  (MAPC #'(LAMBDA (SORT) (SETF (get SORT 'DT*DIRECT.SUBSORTS)
			       (MAXIMA (DELETE SORT (COPY-TREE (GET SORT 'DT*TRANSITIVE.CLOSURE)))
				       #'(LAMBDA (SORT1 SORT2) (DT-SORT.IS.SUBSORT SORT2 SORT1)))))
	(DT-SORT.ALL)))

(DEFUN DT=SORT.DELETE.ALL NIL
  ;; EDITED: 11-FEB-83 10:08:17
  ;; EFFECT: ALL SORT SYMBOLS ARE CLEARED.
  ;; VALUE:  UNDEFINED
  (MAPC
    #'(LAMBDA (SORT)
        (REMPROPS SORT
		  '(DT*DIRECT.SUBSORTS DT*MAX.SUBSORTS DT*TRANSITIVE.CLOSURE DT*DISJOINT.SORTS DT*LEAST.SUPERSORTS
				       DT*INVERSE.TRANSITIVE.CLOSURE DT*DIRECT.SUPERSORTS
				       DT*MINIMAL.SUBSORTS DT*CORRESPONDING.SORTS)))
    DT*SORT.ALL)
  (SETQ DT*SORT.ALL NIL))

(DEFUN DT-PUT.UNI.CREATES.VARIABLES (VALUE)
  ;; EDITED: 25.4.84
  ;; EFFECT: UPDATING THE VALUE OF DT*UNI.CREATES-
  ;;         VARIABLES.
  (SETQ DT*UNI.CREATES.VARIABLES VALUE))

(DEFUN DT=SORT.UPDATE.MAX.SUBSORTS (&REST UPDATE.FLAG)
  ;; EDITED: 11-FEB-83 09:31:39
  ;; EFFECT: FOR EACH SORT AN ASSOCIATION LIST IS CREATED
  ;;         KEY: A SORT, VALUE: THE GREATEST COMMON
  ;;         SUBSORTS.
  ;; VALUE:  UNDEFINED.
  (unless UPDATE.FLAG (DT=SORT.UPDATE.TRANS.CLOSURE))
  (MAPC #'(LAMBDA (SORT1)
	    (get 'element-of-u-union-v  'DT*DIRECT.SUBSORTS)
	    (setf (get SORT1 'DT*MAX.SUBSORTS)
		  (MAPCAR #'(LAMBDA (SORT2)
			      (CONS SORT2 (let ((MAX.SUBSORTS (DT=SORT.MAX.SUBSORTS SORT1 SORT2)))
					    (COND ((CDR (DT=SORT.MAX.SUBSORTS SORT1 SORT2))
						   (DT-PUT.UNI.CREATES.VARIABLES T)))
					    MAX.SUBSORTS)))
			  DT*SORT.ALL)))
	DT*SORT.ALL))

(DEFUN DT=SORT.MAX.SUBSORTS (SORT1 SORT2)
						; Authors: PRCKLN UNKNOWN
						; EDITED:  26-SEP-1989 17:17
						; INPUT:  TWO SORT SYMBOLS
						; VALUE:  THE MAXIMUM COMMON SUBSORTS OF THE TWO
						;         SORTS.
						; Remark: THIS FUNCTION SHOULD BE CALLED ONLY IN FUNCTION
						;         DT=SORT.UPDATE.MAX.SUBSORTS.
  (COND ((EQL SORT1 SORT2) (LIST SORT1))
	((MEMBER SORT1 (GET SORT2 'DT*TRANSITIVE.CLOSURE)) (LIST SORT1))
	((MEMBER SORT2 (GET SORT1 'DT*TRANSITIVE.CLOSURE)) (LIST SORT2))
	(T (let (MAX.SUBSORT.LIST)
	     (SETQ MAX.SUBSORT.LIST (INTERSECTION (GET SORT1 'DT*TRANSITIVE.CLOSURE) (GET SORT2 'DT*TRANSITIVE.CLOSURE)))
	     (MAXIMA (copy-list MAX.SUBSORT.LIST) #'(lambda (x y) (DT-SORT.IS.SUBSORT x y)))))))

(DEFUN DT=SORT.TRANSITIVE.CLOSURE (SORT)
  ;; EDITED: 10-FEB-83 17:04:36
  ;; INPUT:  A SORT SYMBOL
  ;; VALUE:  A LIST OF ALL SUBSORTS OF SORT.
  (let ((TRANS.CLO.NEW (LIST SORT)) TRANS.CLO.OLD)
    (WHILE (NOT (SET= TRANS.CLO.OLD TRANS.CLO.NEW)) (SETQ TRANS.CLO.OLD TRANS.CLO.NEW)
	   (MAPC #'(LAMBDA (TRANS.SORT)
		     (SETQ TRANS.CLO.NEW (UNION (DT-SORT.DIRECT.SUBSORTS TRANS.SORT) TRANS.CLO.NEW)))
		 TRANS.CLO.OLD))
    TRANS.CLO.OLD))

(DEFMACRO DT=SORT.NEW.SYMBOL NIL
  ;; NO INPUT
  ;; VALUE: THE NEXT NEW SORT-SYMBOL:   SORT_NR
  ;; SIDEEFFECTS: DT*SORT.NR WILL BE INCREASED.
  '(PROG ((SORT.SYMBOL (INTERN (CONCATENATE 'STRING (PRINC-TO-STRING '"SORT_") (PRINC-TO-STRING DT*SORT.NR))
			       (find-package "MKRP"))))
	 (REMPROPS SORT.SYMBOL DT*SORT.PROPERTIES)
	 (SETF (get SORT.SYMBOL 'DT*SORT.NUMBER) DT*SORT.NR)
	 (SETQ DT*SORT.NR (1+ (SYMBOL-VALUE 'DT*SORT.NR))) (RETURN SORT.SYMBOL)))

(DEFVAR DT*VARIABLE.COUNTER 0)

(DEFVAR DT*VARIABLE.BUFFER NIL)


#|(defstruct (dt=a_variable :named (:type :list) (:conc-name dt=a_variable.) (:constructor dt=a_variable.create))
  sort binding)|#

(DEFUN DT-VARIABLE.CREATE (SORT)
						; EDITED: 11-AUG-81
						; INPUT:  'SORT' IS A SORT. ARBITRARY is a flag
						; EFFECT: A VARIABLE IS CREATED AND ITS COMPONENTS ARE
						;         INITIALIZED. Iff ARBITRARY is true a defstruct variable is created
						; VALUE:  THE VARIABLE ADDRESS.
  (let ((VARIABLE (DT=VARIABLE.STORAGE)))	;(if (eql variable 50) (break "create 50"))
    (DT=VARIABLE.PUTSORT VARIABLE SORT)
    (DT=VARIABLE.PUTBINDING VARIABLE NIL)
    VARIABLE))

(DEFMACRO DT-VARIABLE.PNAME (VARIABLE)
  ;; EDITED: 29-SEP-83 20:02:20
  ;; INPUT:  VARIABLE ADDRESS.
  ;; EFFECT:  -
  ;; VALUE:  NAME OF VARIABLE, E.G. X_'VARIABLE'.
  `(DT=VARIABLE.GETPNAME ,VARIABLE))

(defmacro DT-VARIABLE.SORT (VARIABLE)
  ;; EDITED: 11-AUG-81 14:18:04
  `(DT=VARIABLE.GETSORT ,VARIABLE))

(DEFMACRO DT-VARIABLE.PUTSORT (VARIABLE SORT)
  ;; INPUT: A VARIABLE AND A SORT
  ;; EFFECT: THE VARIABLE IS SET TO THIS SORT
  `(DT=VARIABLE.PUTSORT ,VARIABLE ,SORT))

(DEFMACRO DT-VARIABLE.GET.BINDING (VARIABLE)
  ;; EDITED:2-AUG-83.
  ;; INPUT: 'VARIABLE' IS A VARIABLE ADDRESS.
  ;; VALUE: THE FIRST ELEMENT OF BINDING LIST OF 'VAR.'.
  ;; REMARK:COMMENT AND FUNCTION DIFFERENT. PRELIMENARY.
  `(DT=VARIABLE.GETBINDING ,VARIABLE))

(DEFMACRO DT-VARIABLE.PUT.BINDING (VARIABLE TERM)
  ;; EDITED:2-AUG-83.
  ;; INPUT: 'VARIABLE' IS A VARIABLE ADDRESS.
  ;;        'TERM' IS A TERM
  ;; EFFECT:PUTS 'TERM' AT FIRST POSITION IN
  ;;        THE BINDING LIST OF 'VARIABLE.
  ;; VALUE: UNDEFINED
  ;; REMARK:COMMENT AND FUNCTION DIFFERENT. PRELIMENARY.
  `(DT=VARIABLE.PUTBINDING ,VARIABLE ,TERM))

(DEFMACRO DT-VARIABLE.DELETE.BINDING (VARIABLE)
  ;; EDITED:2-AUG-83.
  ;; INPUT: 'VARIABLE' IS A VARIABLE ADDRESS.
  ;; EFFECT:THE FIRST ELEMENT OF THE BINDING LIST OF
  ;;        VARIABLE 'VARIABLE' WILL BE REMOVED.
  ;; VALUE: THE FIRST ELEMENT OF OLD BINDING LIST.
  ;; REMARK:COMMENT AND FUNCTION DIFFERENT. PRELIMENARY.
  `(DT=VARIABLE.PUTBINDING ,VARIABLE NIL))

(DEFUN DT-VARIABLE.RENAMING.SUBSTITUTION (VARIABLELIST)
						; EDITED: 10-NOV-83 08:33:52
						; INPUT:  A LISTS OF VARIABLES
						; EFFECT: FOR EACH VARIABLE OF AN INPUT LIST A NEW
						;         VARIABLE IS CREATED or taken from VARIABLES
						; VALUES: A VARIABLE RENAMING SUBSTITUTION FOR ALL
						;         VARIABLES OF THE INPUT LISTS.
  (MAPCAN #'(LAMBDA (VARIABLE) (LIST VARIABLE (DT-VARIABLE.CREATE (copy-tree (DT-VARIABLE.SORT VARIABLE)))))
	  VARIABLELIST))

(DEFMACRO DT-VARIABLE.IS (VARIABLE)
						; EDITED: 12-NOV-79 15:03:21
						; VALUE:  T IF VARIABLE IS A VARIABLE SYMBOL NOT YET DELETED, ELSE NIL .
  `(DT=VARIABLE.IS ,VARIABLE))

(defun dt-variable.in (variable term)
						; Edited:  12-MAR-1991 20:23
						; Authors: PRCKLN
						; Input:   
						; Effect:  
						; Value:   True iff variable occurs in TERM
  (if (dt-variable.is term)
      (if (= variable term)
	  t
	  (if (opt-get.option sort_literals)
	      (dt-variable.in variable (dt-variable.sort term))
	      nil))
      (if (dt-constant.is term)
	  nil
	  (some #'(lambda (subterm) (dt-variable.in variable subterm))
		(dt-term_arguments term)))))

(defmacro DT-VARIABLE.DELETE (VARIABLE)
						; EDITED: 12-NOV-79 15:04:13
						; VALUE:  UNDEFINED 
  `(MEM-ERASE ,VARIABLE T))

(DEFMACRO DT=VARIABLE.STORAGE NIL
  ;; EDITED:    11-AUG-81 14:40:04
  ;; VALUE:     POINTER TO A NEW STORAGE UNIT FOR A VARIABLE
  ;; STRUCTURE: CELL    COMPONENT
  ;;               1    SORT
  ;;               2    BINDING
  `(MEM-NEW 'VARIABLE 2))

(defun DT=VARIABLE.GETPNAME (VARIABLE)
									      ; Edited:  24-JAN-1992 19:43
									      ; Authors: KKL
									      ; Input:   VARIABLE ADDRESS.
									      ; Effect:  -
									      ; Value:   X_'VARIABLE' or with sort.
  (if (and (opt-get.option sort_literals) (opt-get.option sort_show.variable.sorts))
      (format nil "X_~A:~A" VARIABLE (dt-pname (DT=VARIABLE.GETSORT variable)))
      (format nil "X_~A" VARIABLE)))

(DEFMACRO DT=VARIABLE.GETSORT (VARIABLE)
  ;; EDITED: 11-AUG-81 14:31:58
  ;; INPUT:  VARIABLE- A VARIABLE ADDRESS
  ;; VALUE:  SORT OF VARIABLE.
  `(MEM-GET ,VARIABLE 1))

(DEFMACRO DT=VARIABLE.PUTSORT (VARIABLE SORT)
  ;; EDITED:| "11-AUG-81 14:34:29")
  ;; INPUT:  VARIABLE- A VARIABLE ADDRESS
  ;; EFFECT: WRITES THE SORT-COMPONENT OF THE GIVEN
  ;;         VARIABLE.
  ;; VALUE:  UNDEFINED.
  `(MEM-PUT ,VARIABLE 1 ,SORT))

(DEFMACRO DT=VARIABLE.GETBINDING (VARIABLE)
  ;; EDITED:2-AUG-83.
  ;; INPUT: 'VARIABLE' IS A VARIABLE ADDRESS.
  ;; VALUE: THE BINDING COMPONENT OF 'VARIABLE'.
  `(MEM-GET ,VARIABLE 2))

(DEFMACRO DT=VARIABLE.PUTBINDING (VARIABLE BINDING)
  ;; EDITED:2-AUG-83.
  ;; INPUT: 'VARIABLE' IS A VARIABLE ADDRESS.
  ;;        'BINDING' IS A S-EXPRESSION.
  ;; EFFECT:'BINDING' WILL BECOME THE BINDING COMPONENT
  ;;        OF VARIABLE 'VARIABLE'.
  ;; VALUE: UNDEFINED.
  `(MEM-PUT ,VARIABLE 2 ,BINDING))

(DEFMACRO DT=VARIABLE.IS (VAR)
  ;; EDITED:| "18-JAN-80 12:35:52")
  ;; VALUE:| T IF VAR IS A VAR SYMBOL NOT YET |DELETED,| ELSE NIL 
  `(EQL (MEM-TYPE ,VAR) 'VARIABLE))

(DEFVAR DT*CONSTANT.COUNTER 0)

(DEFVAR DT*CONSTANT.ALL NIL)

(defparameter DT*omega.CONSTANT nil)

(DEFMACRO DT-CONSTANT.omega NIL
 ;; VALUE: THE ADDRESS OF THE 'TRUE' CONSTANT
  'DT*omega.CONSTANT)

(DEFUN DT-CONSTANT.CREATE (PNAME &optional SORT SKOLEM.FLAG)
  ;; EDITED: 19-NOV-79 16:20:33
  ;; INPUT:  A CONSTANT NAME, A SORT, AND 
  ;;         A FLAG TO MARK SKOLEM CONSTANTS 
  ;; VALUE:  NEW CONSTANT SYMBOL 
  ;; EFFECT: IF PNAME IS ATOMIC, IT WILL BE USED. OTHERWISE A NEW NAME IS CREATED 
  (SETQ PNAME (COND ((NULL PNAME) (DT=CONSTANT.CREATE.PNAME))
		    (T            (princ-to-STRING PNAME))))
  ;; Pname created 
  (let ((CONSTANT (DT=CONSTANT.STORAGE)))
    (DT=CONSTANT.PUTPNAME CONSTANT PNAME)
    (DT=CONSTANT.PUTSORT CONSTANT SORT)
    (COND (SKOLEM.FLAG (DT-PUTPROP CONSTANT 'DT*ST-KIND 'SYS-CONST)))
    (SETQ DT*CONSTANT.ALL (CONS CONSTANT DT*CONSTANT.ALL))
    (COND ((STRING= "OMEGA" (STRING PNAME)) (SETQ DT*omega.constant constant)))
    CONSTANT))

(DEFMACRO DT-CONSTANT.SORT (CONSTANT)
  ;; EDITED:| "11-AUG-81 14:31:25")
  ;; INPUT:  CONSTANT- A CONSTANT ADDRESS
  ;; VALUE:  SORT OF CONSTANT.
  `(DT=CONSTANT.GETSORT ,CONSTANT))

(DEFMACRO DT-CONSTANT.PUTSORT (CONSTANT SORT)
  ;; INPUT: A CONSTANT AND A SORT
  ;; EFFECT: THE CONSTANT IS SET TO THIS SORT
  `(DT=CONSTANT.PUTSORT ,CONSTANT ,SORT))

(DEFMACRO DT-CONSTANT.PUTPNAME (CONSTANT P.NAME) `(DT=CONSTANT.PUTPNAME ,CONSTANT ,P.NAME))

(DEFMACRO DT-CONSTANT.PNAME (CONSTANT)
  ;; EDITED:| "12-NOV-79 14:47:49")
  ;; VALUE:| PNAME OF CONSTANT SYMBOL 
  `(DT=CONSTANT.GETPNAME ,CONSTANT))

(DEFMACRO DT-CONSTANT.IS (CONSTANT)
  ;; EDITED: 12-NOV-79 15:03:21
  ;; VALUE:  T IF CONSTANT IS A CONSTANT SYMBOL NOT YET |DELETED,| ELSE NIL 
  `(DT=CONSTANT.IS ,CONSTANT))

(DEFMACRO DT-CONSTANT.ALL NIL
  ;; INPUT:  NONE
  ;; VALUE:  LIST OF ALL CONSTANT-ADDRESSES
  ;; EFFECT: RETURNS VALUE
  `DT*CONSTANT.ALL)

(DEFUN DT-CONSTANT.DELETE (CONSTANT)
  ;; EDITED:| "12-NOV-79 15:04:13")
  ;; VALUE:| UNDEFINED *)
  (SETQ DT*CONSTANT.ALL (DELETE CONSTANT (SYMBOL-VALUE 'DT*CONSTANT.ALL))) (MEM-ERASE CONSTANT NIL))

(DEFMACRO DT-CONSTANT.IS.SKOLEM (CONSTANT)
  ;; INPUT:  A CONSTANT
  ;; VALUE: T, IF CONSTANT IS CREATED WITH SKOLEMFLAG
  `(DT-GETPROP ,CONSTANT 'DT*ST-KIND))

(DEFMACRO DT=CONSTANT.STORAGE NIL
  ;; EDITED:| "11-AUG-81 14:40:21")
  ;; VALUE:| POINTER TO A NEW STORAGE UNIT FOR A CONSTANT *)
  ;; STRUCTURE: CELL    COMPONENT
  ;;               1    PNAME
  ;;               2    SORT
  `(MEM-NEW 'CONSTANT 2))

(DEFMACRO DT=CONSTANT.CREATE.PNAME NIL
  ;; EDITED: 19-NOV-79 16:25:00
  ;; VALUE:  NEW NAME FOR A CONSTANT
  `(CONCATENATE 'STRING "CONST" (PRINC-TO-STRING (SETQ DT*CONSTANT.COUNTER (1+ DT*CONSTANT.COUNTER)))))

(DEFMACRO DT=CONSTANT.GETPNAME (CONSTANT)
  ;; EDITED: 12-NOV-79 15:04:57
  `(MEM-GET ,CONSTANT 2))

(DEFMACRO DT=CONSTANT.GETSORT (CONSTANT)
  ;; EDITED: 11-AUG-81 14:29:34
  ;; INPUT:  CONSTANT- A CONSTANT ADDRESS
  ;; VALUE:  SORT OF CONSTANT.
  `(MEM-GET ,CONSTANT 1))

(DEFMACRO DT=CONSTANT.PUTPNAME (CONSTANT PNAME)
  ;; EDITED: 12-NOV-79 15:05:22
  `(MEM-PUT ,CONSTANT 2 ,PNAME))

(DEFMACRO DT=CONSTANT.PUTSORT (CONSTANT SORT)
  ;; EDITED:| "11-AUG-81 14:35:29")
  ;; INPUT:  CONSTANT- A CONSTANT ADDRESS
  ;; EFFECT: WRITES THE SORT-COMPONENT OF THE GIVEN
  ;;         CONSTANT.
  ;; VALUE:  UNDEFINED.
  `(MEM-PUT ,CONSTANT 1 ,SORT))

(DEFMACRO DT=CONSTANT.IS (CONST)
  ;; EDITED:| "18-JAN-80 12:35:52")
  ;; VALUE:| T IF CONST IS A CONST SYMBOL NOT YET |DELETED,| ELSE NIL *)
  `(EQL (MEM-TYPE ,CONST) 'CONSTANT))

(DEFVAR DT*ABBREVIATIONS NIL)

(DEFUN DT-ABBREVIATION.PUSH NIL
  ;; EDITED: 28-OCT-82 13:14:07
  ;; EFFECT: A NEW EMPTY ABBREVIATION SCHEME IS CREATED.
  ;; VALUE:  THE NEW SCHEME.
  (let ((SCHEME (GENSYM)))
    (SETF (SYMBOL-VALUE SCHEME) (LIST NIL)
	  (SYMBOL-PLIST SCHEME) NIL
	  DT*ABBREVIATIONS (CONS SCHEME DT*ABBREVIATIONS))
    nil))

(DEFUN DT-ABBREVIATION.POP NIL
  ;; EDITED AT 10-NOV-83 |20:30|)
  ;; EDITED: 28-OCT-82 13:24:30
  ;; INPUT:  NO INPUT
  ;; EFFECT: ALL ABBREVIATIONS OF THE ACTUAL SCHEME ARE
  ;;         DELETED.
  ;; VALUE:  UNDEFINED.
  (PROG ((SCHEME (CAR DT*ABBREVIATIONS)))
	(COND
	  (SCHEME (MAPC #'(LAMBDA (VAR) (MEM-ERASE VAR T)) (CAR (SYMBOL-VALUE SCHEME)))
		  (SETQ DT*ABBREVIATIONS (DELETE SCHEME DT*ABBREVIATIONS))
		  (SETF (SYMBOL-VALUE SCHEME) 'NOBIND)
		  (SETF (SYMBOL-PLIST SCHEME) NIL)))))

(DEFUN DT-ABBREVIATION.COMPRESS.TERM (TERM)
  ;; EDITED AT 14-DEC-83 |12:02|)

  ;; EDITED: 28-OCT-82 18:37:07
  ;; INPUT:  A TERM
  ;; EFFECT: FOR TERM AND FOR EVERY SUBTERM
  ;;         REPRESENTING A GROUND INSTANCE, AN
  ;;         ABBREVIATION IS GENERATED.
  ;; VALUE:  (ABBREVIATED-TERM . NEW-ABBREVIATIONS)
  ;;         NEW-ABBREVIATIONS IS A LIST OF ADDRESS WITH
  ;;         JUST GENERATED ABBREVIATIONS.
  ;; REMARK: TERM IS DESTRUCTIVELY CHANGED.
  (COND ((ATOM TERM) (CONS TERM NIL))
	(T
	 (PROG ((NEW.ABBREVIATIONS (LIST NIL)))
	       (when DT*FUNCTION.ACTUAL.THEORIES (SETQ TERM (DT=ABBREVIATION.EXPAND.FOR.COMPRESS TERM)))
	       (COND
		 ((OR (SYMBOL-VALUE 'DT*FUNCTION.WITH.ARGUMENT.SYMMETRIES) (SYMBOL-VALUE 'DT*FUNCTION.ACTUAL.THEORIES))
		  (SETQ TERM (DT=ABBREVIATION.NORMALFORM.FOR.COMPRESS TERM))))
	       (SETQ TERM (DT=ABBREVIATION.GENERATE TERM (CAR (SYMBOL-VALUE 'DT*ABBREVIATIONS)) NEW.ABBREVIATIONS))
	       (RETURN (CONS TERM (CAR NEW.ABBREVIATIONS)))
	       ;; NEW.ABBREVIATIONS IS CHANGED IN DT=ABBREVIATION.CREATE.TREE
	       ))))

(DEFUN DT-ABBREVIATION.COMPRESS.TERMLIST (TERMLIST)
  ;; EDITED AT 11-NOV-83 |10:34|)

  ;; EDITED: 28-OCT-82 18:37:07
  ;; INPUT:  A TERMLIST
  ;; EFFECT: FOR TERM AND FOR EVERY SUBTERM
  ;;         REPRESENTING A GROUND INSTANCE, AN
  ;;         ABBREVIATION IS GENERATED, IF THE FUNCTION
  ;;         SYMBOL HAS NO ATTRIBUTES (THEORIES).
  ;; VALUE:  (ABBREVIATED-TERMLIST . NEW-ABBREVIATIONS)
  ;;         NEW-ABBREVIATIONS IS A LIST OF ADDRESS WITH
  ;;         JUST GENERATED ABBREVIATIONS.
  ;; REMARK: TERM IS DESTRUCTIVELY CHANGED.
  (PROG (NEW.ABBREVIATIONS)
	(RETURN
	  (CONS
	    (MAPCAR
	      #'(LAMBDA (TERM) (SETQ TERM (DT-ABBREVIATION.COMPRESS.TERM TERM))
			(SETQ NEW.ABBREVIATIONS (NCONC (CDR TERM) NEW.ABBREVIATIONS)) (CAR TERM))
	      TERMLIST)
	    NEW.ABBREVIATIONS))))

(DEFUN DT-ABBREVIATION.EXPAND.TERM (TERM NUMBER.OR.NIL)
  ;; EDITED: 29-OCT-82 17:33:09
  ;; INPUT:  A TERM AND NIL OR AN INTERGER > 0
  ;; EFFECT: THE TERM IS EXPANDED UP TO A LEVEL GIVEN
  ;;         BY DEPTH. IF DEPTH = NIL , THE TERM IS
  ;;         COMPLETELY EXPANDED.
  ;; VALUE:  THE EXPANDED TERM.
  ;; REMARK: TERM IS DESTURCTIVELY MODIFIED.
  (COND (NUMBER.OR.NIL (DT=ABBREVIATION.EXPAND TERM NUMBER.OR.NIL)) (T (DT=ABBREVIATION.EXPAND TERM 10000))))

(DEFUN DT-ABBREVIATION.EXPAND.TERMLIST (TERMLIST &optional (NUMBER.OR.NIL 10000))

  ;; EDITED: 29-OCT-82 17:33:09
  ;; INPUT:  A TERMLIST AND NIL OR AN INTERGER > 0
  ;; EFFECT: THE TERMLIST IS EXPANDED UP TO A LEVEL GIVEN
  ;;         BY DEPTH. IF DEPTH = NIL , THE TERM IS
  ;;         COMPLETELY EXPANDED.
  ;; VALUE:  THE EXPANDED TERM.
  ;; REMARK: TERM IS DESTURCTIVELY MODIFIED.
  (unless NUMBER.OR.NIL (setq NUMBER.OR.NIL 10000))
  (MAPCAR #'(LAMBDA (TERM) (DT=ABBREVIATION.EXPAND TERM NUMBER.OR.NIL)) TERMLIST))

(DEFMACRO DT-ABBREVIATION.SCHEMES NIL
  ;; EDITED: 28-OCT-82 17:07:07
  ;; VALUE:  A LIST OF ALL ABBREVIATION SCHEMES.
  'DT*ABBREVIATIONS)

(DEFMACRO DT-ABBREVIATION.ALL NIL
  ;; EDITED AT 11-NOV-83 |10:40|)
  ;; EDITED: 28-OCT-82 17:07:07
  ;; INPUT:  NO INPUT
  ;; VALUE:  A LIST OF ALL ABBREVIATIONS, DEFINED IN THE
  ;;         ACTUAL SCHEME.
  `(CAR (SYMBOL-VALUE (CAR DT*ABBREVIATIONS))))

(DEFMACRO DT-ABBREVIATION.TERM (ABBREVIATION)
  ;; INPUT:  AN ABBREVIATION ADDRESS.
  ;; VALUE:  THE TERM REPRESENTED BY THIS ABBREVIATION
  ;;         EXPANDED UP TO DEPTH 1.
  ;; WARNING: IT'S NOT ALLOWED TO WORK DESTRUCTIVELY
  ;;          UPON THIS TERM.
  `(DT=ABBREVIATION.GETTERM ,ABBREVIATION))

(DEFMACRO DT-ABBREVIATION.SORT (ABBREVIATION)
  ;; EDITED: 28-OCT-82 16:45:41
  ;; INPUT:  AN ABBREVIATION ADDRESS.
  ;; VALUE:  THE SORT OF THE ABBREVIATED.TERM
  `(DT=ABBREVIATION.GETSORT ,ABBREVIATION))

(DEFMACRO DT-ABBREVIATION.IS (ABBREVIATION)
  ;; EDITED: 28-OCT-82 17:01:37
  ;; INPUT:  AN S-EXPRESSION
  ;; VALUE:  T IF IT IS AN ABBREVIATION ADDRESS, ELSE
  ;;         NIL.
  `(DT=ABBREVIATION.IS ,ABBREVIATION))


(DEFUN DT=ABBREVIATION.EXPAND.ASS (TERM FUNCTION)
  ;; EDITED 10.3.84         M.SS
  ;; INPUT: A LIST OF ABBREVIATIONS OR CONSTANTS
  ;; EFFECT: THE LIST WILL BE EXPANDED UNTIL EVERY
  ;;         ABBREVIATION OF THE FORM (F A B)
  ;;         F  ASSOCIATIVE, A OR B IS ABBREVIATION
  ;;         OR CONSTANT
  ;;         HAS THE FOLLOWING PROPERTY
  ;;         NEITHER THE TERM OF A NOR OF B HAS
  ;;         TOPLEVEL FUNCTIONSYMBOL F.
  ;; VALUE:  THE PARTIAL EXPANDED TERM
  (PROG ((CONTINUE T) SUBTERM)
	(WHILE CONTINUE
	  (SETQ CONTINUE NIL)
	  (SETQ TERM
		(MAPCAN
		  #'(LAMBDA (ELEMENT)
		      (COND
			((AND (DT=ABBREVIATION.IS ELEMENT)
			      (EQL (CAR (SETQ SUBTERM (DT=ABBREVIATION.GETTERM ELEMENT))) FUNCTION))
			 (SETQ CONTINUE T)
			 (COPY-TREE (CDR SUBTERM)))
			(T (LIST ELEMENT))))
		  TERM)))
	(RETURN TERM)))



(DEFUN DT=FUNCTION.TUPLE.FIT (TUPLE SORT.LIST)
  ;; EDITED AT 5-DEC-83 |17:34| *)
  ;; THE GIVEN TUPLE DOES NOT CONTRADICT THE GIVEN SET
  ;; OF TUPLES, IF THE RANGESORT IS A MONOTONE FUNCTION
  ;; OF THE DOMAINSORTS.
  ;; VALUE T IF OK,  ELSE NIL
  (EVERY
    #'(LAMBDA (TUPLE2)
        (AND
          (IMPLIES (DT-FUNCTION.TUPLE.LESS TUPLE TUPLE2 T) (DT-SORT.IS.SUBSORT (CAR (LAST TUPLE)) (CAR (LAST TUPLE2))))
          (IMPLIES (DT-FUNCTION.TUPLE.LESS TUPLE2 TUPLE T) (DT-SORT.IS.SUBSORT (CAR (LAST TUPLE2)) (CAR (LAST TUPLE))))))
    SORT.LIST))

(DEFUN DT=FUNCTION.TUPLE.MINIMIZE (SORT.LIST)
  ;; EDITED AT 5-DEC-83 |18:31| *)
  ;; TUPLES OF SORTLIST ARE ELIMINATED, IF THERE IS
  ;; A TUPLE, WHICH IS TUPLE.GREATER AND HAS EQUAL
  ;; RANGE.SORT.
  ;; VALUE: THE MINIMIZED SORT.LIST
  (PROG (SORT.LIST2)
	(MAPC
	  #'(LAMBDA (TUPLE1)
	      (COND
		((AND (NOT (MEMBER TUPLE1 SORT.LIST2 :TEST (FUNCTION EQUAL)))
		      (NOTANY
			#'(LAMBDA (TUPLE2)
			    (AND (DT-FUNCTION.TUPLE.LESS TUPLE1 TUPLE2 T) (NOT (EQUAL TUPLE1 TUPLE2))
				 (EQUAL (CAR (LAST TUPLE1)) (CAR (LAST TUPLE2)))))
			SORT.LIST))
		 (SETQ SORT.LIST2 (NCONC1 SORT.LIST2 TUPLE1)))))
	  SORT.LIST)
	(RETURN SORT.LIST2)))

(DEFUN DT-ABBREVIATION.EXPAND.ASS (ABBREVIATION FUNCTION)
  ;; EDITED AT 10-APR-84 09:35
  ;; INPUT: A ABBREVIATION AND A FUNCTION WHICH IS
  ;;        ASSOCIATIVE
  ;; EFFECT: THE ABBR WILL BE EXPANDED UNTIL EVERY
  ;;         SUBTERM OF THE FORM  (F A B)
  ;;         F  ASSOCIATIVE, A OR B IS ABBREVIATION
  ;;         OR CONSTANT
  ;;         HAS THE FOLLOWING PROPERTY
  ;;         NEITHER THE TERM OF A NOR OF B HAS
  ;;         TOPLEVEL FUNCTIONSYMBOL F.
  ;; VALUE:  THE PARTIAL EXPANDED TERM WITHOUT F
  (PROG ((TERM (DT-ABBREVIATION.TERM ABBREVIATION)))
	(COND ((AND (CONSP TERM) (EQL FUNCTION (CAR TERM))) (RETURN (DT=ABBREVIATION.EXPAND.ASS (CDR TERM) FUNCTION)))) (RETURN ABBREVIATION)))

(DEFMACRO DT-ABBREVIATION.PUTSORT (ABBREVIATION SORT)
  ;; INPUT: A ABBREVIATION ADDRESS AND A SORT
  ;; VALUE: NOT DEFINED
  ;; EFFECT: THE SORT OF THE ABBREVIATION IS CHANGED
  `(DT=ABBREVIATION.PUTSORT ,ABBREVIATION ,SORT))

(DEFMACRO DT=ABBREVIATION.STORAGE NIL
  ;; EDITED: 28-OCT-82 12:51:57
  ;; VALUE:  A POINTER TO THE NEW STORAGE UNIT.
  ;; STRUCTURE:   CELL      COMPONENT
  ;;               1        TERM
  ;;               2        DEPTH
  ;;               3        SORT OF TERM
  `(MEM-NEW 'ABBREVIATION 3))

(DEFUN DT=ABBREVIATION.CREATE (TERM)
  ;; EDITED: 28-OCT-82 17:27:53
  ;; INPUT:  A NON-ATOMIC TERM.
  ;; EFFECT: A NEW MEMORY LOCATION IS GENERATED.
  ;; VALUE:  THE ADDRESS OF THIS NEW LOCATION.
  (PROG ((DEPTH 0) (ABBREVIATION (DT=ABBREVIATION.STORAGE))) (DT=ABBREVIATION.PUTTERM ABBREVIATION TERM)
	(DT=ABBREVIATION.PUTSORT ABBREVIATION (DT-TERM.SORT TERM))
	(MAPC
	  #'(LAMBDA (ELEMENT)
	      (COND
		((DT=ABBREVIATION.IS ELEMENT) (SETQ ELEMENT (DT=ABBREVIATION.GETDEPTH ELEMENT))
		 (COND ((> ELEMENT DEPTH) (SETQ DEPTH ELEMENT))))))
	  (CDR TERM))
	(DT=ABBREVIATION.PUTDEPTH ABBREVIATION (COND ((ZEROP DEPTH) 2) (T (1+ DEPTH)))) (RETURN ABBREVIATION)))

(DEFMACRO DT=ABBREVIATION.PUTTERM (ABBREVIATION TERM)
  ;; EDITED: 28-OCT-82 12:54:55
  ;; INPUT:  AN ABBREVIATION ADDRESS AND AN S-EXPRESSION.
  ;; VALUE:  UNDEFINED.
  `(MEM-PUT ,ABBREVIATION 3 ,TERM))

(DEFMACRO DT=ABBREVIATION.GETTERM (ABBREVIATION)
  ;; EDITED: 28-OCT-82 12:54:55
  ;; INPUT:  AN ABBREVIATION ADDRESS
  ;; VALUE:  THE CONTENTS OR THE TERM-CELL.
  `(MEM-GET ,ABBREVIATION 3))

(DEFMACRO DT=ABBREVIATION.PUTDEPTH (ABBREVIATION DEPTH)
  ;; EDITED: 29-OCT-82 09:46:48
  ;; INPUT:  AN ABBREVIATION ADDRESS AND AN INTEGER
  ;; VALUE:  UNDEFINED.
  `(MEM-PUT ,ABBREVIATION 2 ,DEPTH))

(DEFMACRO DT=ABBREVIATION.GETDEPTH (ABBREVIATION)
  ;; EDITED: 29-OCT-82 09:46:48
  ;; INPUT:  AN ABBREVIATION ADDRESS
  ;; VALUE:  THE CORRESPONDING TERM DEPTH
  `(MEM-GET ,ABBREVIATION 2))

(DEFMACRO DT=ABBREVIATION.IS (ABBR)
  ;; EDITED: 28-OCT-82 17:01:37
  ;; INPUT:  AN S-EXPRESSION
  ;; VALUE:  T IF IT IS AN ABBREVIATION ADDRESS, ELSE
  ;;         NIL.
  `(EQL (MEM-TYPE ,ABBR) 'ABBREVIATION))

(DEFUN DT=ABBREVIATION.GENERATE (TERM SCHEME NEW.ABBREVIATIONS)
  (declare (ignore scheme))
  ;; EDITED AT 11-NOV-83 |11:38|)
  ;; INPUT:  A NON-ATOMIC TERM,
  ;;         AND A TCONC LIST.
  ;; EFFECT: THE TERM IS ABBREVIATED AS FAR AS POSSIBLE.
  ;;         I.E. ONLY GOUND INSTANCES OF TERMS WITH
  ;;         FUNCTIONS FOR WHICH NO THEORY IS DEFINED
  ;;         ARE ABBREVIATED.
  ;; VALUE:  THE ABBREVIATED TERM.
  (PROG (NO.ABBREVIATION.POSSIBLE ELEMENT)
	(MAPL
	  #'(LAMBDA (TAIL) (SETQ ELEMENT (CAR TAIL))
		    (COND ((ATOM ELEMENT) (COND ((DT=VARIABLE.IS ELEMENT) (SETQ NO.ABBREVIATION.POSSIBLE T))))
			  (T (SETQ ELEMENT (DT=ABBREVIATION.GENERATE ELEMENT (CAR DT*ABBREVIATIONS) NEW.ABBREVIATIONS))
			     (COND ((CONSP ELEMENT) (SETQ NO.ABBREVIATION.POSSIBLE T))) (RPLACA TAIL ELEMENT))))
	  (CDR TERM))
	(RETURN
	  (COND (NO.ABBREVIATION.POSSIBLE TERM) (T (DT=ABBREVIATION.INSERT TERM (CAR DT*ABBREVIATIONS) NEW.ABBREVIATIONS))))))

(DEFUN DT=ABBREVIATION.INSERT (TERM SCHEME NEW.ABBREVIATIONS &optional ABBREVIATION.IN.OLD.SCHEMES)
  (declare (ignore scheme))
  ;; EDITED AT 11-NOV-83 |12:32|)
  ;; INPUT:  A NON-ATOMIC TERM WITH ATOMIC ELEMENTS,
  ;;        AND A TCONC LIST.
  ;; EFFECT: THE ABBREVIATION FOR THIS TERM IS SEARCHED
  ;;         RESP. GENERATED.
  ;; VALUE:  THE ABBREVIATION ADDRESS.
  (PROG (TREE (SCHEME (CAR DT*ABBREVIATIONS))) (SETQ TREE (ASSOC (CAR TERM) (SYMBOL-PLIST SCHEME)))
	(COND
	  (TREE
	   (RETURN
	     (PROG ((TERM-TAIL (CDR TERM)) (FOUND T) ELEMENT CONTINUE NEXT ABBREVIATION.IN.OLD.SCHEMES) (SETQ ELEMENT (CAR TERM-TAIL))
		   (WHILE (AND FOUND ELEMENT) (SETQ FOUND NIL) (SETQ CONTINUE T)
			  (WHILE CONTINUE
			    (COND
			      ((SETQ NEXT (CAADR TREE))
			       (COND ((EQL NEXT ELEMENT) (SETQ TREE (SECOND TREE)) (SETQ CONTINUE NIL) (SETQ FOUND T))
				     ((> NEXT ELEMENT) (SETQ CONTINUE NIL)) (T (SETQ TREE (CDR TREE)))))
			      (T (SETQ CONTINUE NIL))))
			  (COND (FOUND (SETQ TERM-TAIL (CDR TERM-TAIL)) (SETQ ELEMENT (CAR TERM-TAIL)))))
		   (COND (FOUND (RETURN (CDR TREE)))
			 ((SETQ ABBREVIATION.IN.OLD.SCHEMES (DT=ABBREVIATION.SEARCH.OLD.SCHEMES TERM))
			  (RETURN ABBREVIATION.IN.OLD.SCHEMES))
			 (T (SETQ ELEMENT (DT=ABBREVIATION.CREATE.TREE TERM TERM-TAIL SCHEME NEW.ABBREVIATIONS))
			    (RPLACD TREE (CONS (CAR ELEMENT) (CDR TREE))) (RETURN (CDR ELEMENT)))))))
	  (T
	   (COND
	     ((SETQ ABBREVIATION.IN.OLD.SCHEMES (DT=ABBREVIATION.SEARCH.OLD.SCHEMES TERM)) (RETURN ABBREVIATION.IN.OLD.SCHEMES))
	     (T (SETQ TREE (DT=ABBREVIATION.CREATE.TREE TERM TERM SCHEME NEW.ABBREVIATIONS))
		(SETF (SYMBOL-PLIST SCHEME) (CONS (CAR TREE) (SYMBOL-PLIST SCHEME))) (RETURN (CDR TREE))))))))

(DEFUN DT=ABBREVIATION.CREATE.TREE (TERM TERM-TAIL SCHEME NEW.ABBREVIATIONS)
  (declare (ignore scheme))
  ;; EDITED AT 11-NOV-83 |10:50|)
  ;; INPUT:  A NON-ATOMIC TERM WITH ATOMIC ELEMENTS,
  ;;      A TAIL OF THIS TERM, AND A TCONC LIST
  ;; EFFECT: A NEW ABBREVIATION IS CREATED AND INSERTED
  ;;       INTO THE ACTUAL SCHEME AND NEW.ABBREVIATIONS,
  ;;         TERM-TAIL IS TRANSFORMED INTO A TREE
  ;;         WITH ITS ELEMENTS AS NODES AND THE NEW
  ;;         ABBREVIATION AS LEAF.
  ;; VALUE:  (TREE . ABBREVIATION)
  (PROG ((ABBREVIATION (DT=ABBREVIATION.CREATE TERM))) (QCONC1 NEW.ABBREVIATIONS ABBREVIATION)
	(QCONC1 (SYMBOL-VALUE (CAR DT*ABBREVIATIONS)) ABBREVIATION) (SETQ TERM-TAIL (COPY-TREE TERM-TAIL)) (SETQ TERM TERM-TAIL)
	(WHILE (CDR TERM-TAIL) (RPLACD TERM-TAIL (LIST (CDR TERM-TAIL))) (SETQ TERM-TAIL (SECOND TERM-TAIL))) (RPLACD TERM-TAIL ABBREVIATION)
	(RETURN (CONS TERM ABBREVIATION))))

(DEFUN DT=ABBREVIATION.EXPAND (TERM DEPTH)
  ;; EDITED: 29-OCT-82 17:33:09
  ;; INPUT:  A TERM AND AN INTEGER >= 0
  ;; EFFECT: THE TERM IS EXPANDED UP TO A LEVEL GIVEN
  ;;         BY DEPTH.
  ;; VALUE:  THE EXPANDED TERM.
  (COND ((ZEROP DEPTH) TERM)
	((DT=ABBREVIATION.IS TERM) (DT=ABBREVIATION.EXPAND (COPY-TREE (DT=ABBREVIATION.GETTERM TERM)) (1- DEPTH))) ((ATOM TERM) TERM)
	((EQL 1 DEPTH) TERM)
	(T (MAPL #'(LAMBDA (TAIL) (RPLACA TAIL (DT=ABBREVIATION.EXPAND (CAR TAIL) (1- DEPTH)))) (CDR TERM)) TERM)))



(DEFMACRO DT=ABBREVIATION.PUTSORT (ABBREVIATION SORT)
  ;; EDITED: 28-OCT-82 12:54:55
  ;; INPUT:  AN ABBREVIATION ADDRESS AND A SORT
  ;; VALUE:  UNDEFINED.
  `(MEM-PUT ,ABBREVIATION 1 ,SORT))

(DEFMACRO DT=ABBREVIATION.GETSORT (ABBREVIATION)
  ;; EDITED: 28-OCT-82 12:54:55
  ;; INPUT:  AN ABBREVIATION ADDRESS
  ;; VALUE:  THE CONTENTS OF THE TERM-CELL (THE SORT).
  `(MEM-GET ,ABBREVIATION 1))

(DEFUN DT=ABBREVIATION.NORMALFORM (TERM)
  ;; EDITED AT 14-DEC-83 |18:01|)
  ;; INPUT : ANY TERM
  ;; EFFECT: THE TERM IS CHANGED INTO A NORMALFORM
  ;;  WITH RESPECT TO THE THEORIES DEFINED
  ;; REMARK: THIS NORMALFORM HAS NOT THE SYNTAX OF A TERM
  ;;      FOR ASSOCIATIVE FUNCTIONS THE BRACKETS ARE REMOVED
  ;; VALUE: THE NORMALFORM OF THE TERM
  ;;      THE TERM IS CHANGED DESTRUCTIVELY
  (PROG NIL (COND ((ATOM TERM) (RETURN TERM)))
	(COND
	  ((DT-FUNCTION.IS.MARKED ASSOCIATIVE (CAR TERM))
	   (PROG ((CONTINUE T) NEWTERM)
		 (WHILE CONTINUE (SETQ NEWTERM (LIST (CAR TERM))) (SETQ CONTINUE NIL)
			(MAPC
			  #'(LAMBDA (ARGUMENT)
			      (COND ((ATOM ARGUMENT) (NCONC1 NEWTERM ARGUMENT))
				    ((EQL (CAR TERM) (CAR ARGUMENT)) (SETQ NEWTERM (APPEND NEWTERM (copy-list (CDR ARGUMENT))))
				     (COND
				       ((OR (AND (CONSP (SECOND ARGUMENT)) (EQL (CAR TERM) (CAADR ARGUMENT)))
					    (AND (CONSP (THIRD ARGUMENT)) (EQL (CAR TERM) (CAADDR ARGUMENT))))
					(SETQ CONTINUE T))))
				    (T (NCONC1 NEWTERM ARGUMENT))))
			  (CDR TERM))
			(SETQ TERM NEWTERM))
		 (RETURN TERM))))
	(COND ((MEMBER-IF #'LISTP TERM) (RPLACD TERM (MAPCAR #'DT=ABBREVIATION.NORMALFORM (CDR TERM)))))
	(COND ((DT-FUNCTION.ARGUMENT.SYMMETRIES (CAR TERM)) (RPLACD TERM (DT=ABBREVIATION.ARGUMENT.NORMALFORM TERM))))
	(COND
	  ((DT-FUNCTION.IS.MARKED COMMUTATIVE (CAR TERM)) (RPLACD TERM (SORT (CDR TERM) #'DT=ABBREVIATION.TERM.LESS))
	   (COND
	     ((DT-FUNCTION.IS.MARKED IDEMPOTENT (CAR TERM))
	      (PROG ((NEWTERM (LIST (CAR TERM))))
		    (MAPL
		      #'(LAMBDA (TAIL)
			  (COND ((NOT (EQUAL (CAR TAIL) (SECOND TAIL))) (SETQ NEWTERM (NCONC1 NEWTERM (CAR TAIL))))
				((NULL (CDR TAIL)) (SETQ NEWTERM (NCONC1 NEWTERM (CAR TAIL))))))
		      (CDR TERM))
		    (COND ((NULL (CDDR NEWTERM)) (SETQ TERM (SECOND NEWTERM))) (T (SETQ TERM NEWTERM))))
	      (COND ((AND (EQL 3 (LIST-LENGTH TERM)) (EQUAL (SECOND TERM) (THIRD TERM))) (SETQ TERM (SECOND TERM))))))))
	(COND ((ATOM TERM) (RETURN TERM)))
	(COND
	  ((AND (DT-FUNCTION.IS.MARKED IDEMPOTENT (CAR TERM)) (NOT (DT-FUNCTION.IS.MARKED ASSOCIATIVE (CAR TERM)))
		(NOT (DT-FUNCTION.IS.MARKED COMMUTATIVE (CAR TERM))) (EQUAL (SECOND TERM) (THIRD TERM)))
	   (RETURN (SECOND TERM))))
	(COND
	  ((AND (DT-FUNCTION.IS.MARKED ASSOCIATIVE (CAR TERM)) (DT-FUNCTION.IS.MARKED IDEMPOTENT (CAR TERM))
		(NOT (DT-FUNCTION.IS.MARKED COMMUTATIVE (CAR TERM))))
	   (RPLACD TERM (DT=ABBREVIATION.NORMALFORM.XX.TO.X (CDR TERM)))
	   (COND ((EQL 2 (LIST-LENGTH TERM)) (RETURN (SECOND TERM)))
		 ((> (LIST-LENGTH TERM) 7) (RPLACD TERM (DT=ABBREVIATION.NORMALFORM.XYZ.TO.XZ (CDR TERM)))
		  (RPLACD TERM (DT=ABBREVIATION.NORMALFORM.XX.TO.X (CDR TERM)))))))
	(RETURN TERM)))

(DEFUN DT=ABBREVIATION.NORMALFORM.XX.TO.X (TERM)
  ;; EDITED AT 14-NOV-83 |20:43|)
  (PROG ((TAIL TERM) (STRINGLENGTH 1) (BEGIN 1))
	(WHILE (NOT (> (* 2 STRINGLENGTH) (LIST-LENGTH TERM))) (SETQ BEGIN 1) (SETQ TAIL TERM)
	       (WHILE (< (+ BEGIN STRINGLENGTH STRINGLENGTH (- 2)) (LIST-LENGTH TERM))
		 (COND
		   ((EQUAL (FIRSTN TAIL STRINGLENGTH) (CDR (LASTN (FIRSTN TAIL (* 2 STRINGLENGTH)) STRINGLENGTH)))
		    (SETQ TERM (NCONC (FIRSTN TERM (+ BEGIN -1)) (CDR (LASTN TAIL (- (LIST-LENGTH TAIL) STRINGLENGTH)))))
		    (SETQ TAIL (CDR (LASTN TAIL (- (LIST-LENGTH TAIL) STRINGLENGTH)))))
		   (T (SETQ BEGIN (1+ BEGIN)) (SETQ TAIL (CDR TAIL)))))
	       (SETQ STRINGLENGTH (1+ STRINGLENGTH)))
	(RETURN TERM)))

(DEFUN DT=ABBREVIATION.NORMALFORM.XYZ.TO.XZ (TERM &optional CDRFLAG TERMSETLENGTHOLD)
  ;; EDITED AT 11-NOV-83 |11:13|)
  ;;  INPUT:  THE ARGUMENTLIST OF A FUNCTION
  ;;  EFFECT: THIS ARGUMENTLIST IS REDUCED
  ;;          BY THE XYZ TO XZ - RULE
  ;;          SEE SIEKMANN. GWAI 81  S. 235
  ;;  VALUE:  THE REDUCED ARGUMENTLIST
  (PROG (TERMSET TERMSETLENGTH) (COND ((> 7 (LIST-LENGTH TERM)) (RETURN TERM)))
	(SETQ TERMSET
	      (PROG (RESULT)
		    (MAPC
		      #'(LAMBDA (X) (COND ((NOT (MEMBER X RESULT :TEST #'EQUAL)) (SETQ RESULT (NCONC1 RESULT X)))))
		      TERM)
		    (RETURN RESULT)))
	(SETQ TERMSETLENGTH (LIST-LENGTH TERMSET)) (COND ((< TERMSETLENGTH 3) (ERROR "'XX.TO.XX-IST-FALSCH~A" NIL)))
	(COND
	  ((AND (> (LIST-LENGTH TERM) (* 2 TERMSETLENGTH)) (OR (NOT TERMSETLENGTHOLD) (< TERMSETLENGTH TERMSETLENGTHOLD)))
	   (PROG
	     ((STRINGX (FIRSTN TERM TERMSETLENGTH)) (STRINGZ (REVERSE (CDR (LASTN TERM TERMSETLENGTH)))) STRINGXSET
	      (TAIL (CDR (LASTN TERM (- (LIST-LENGTH TERM) TERMSETLENGTH)))) RESULT)
	     (SETQ STRINGXSET
		   (PROG2
		     (MAPC
		       #'(LAMBDA (X) (COND ((NOT (MEMBER X RESULT :TEST #'EQUAL)) (SETQ RESULT (NCONC1 RESULT X)))))
		       STRINGX)
		     RESULT))
	     (WHILE (< (LIST-LENGTH STRINGXSET) TERMSETLENGTH) (SETQ STRINGX (NCONC1 STRINGX (CAR TAIL)))
		    (SETQ STRINGXSET (UNION (LIST (CAR TAIL)) STRINGXSET)) (SETQ TAIL (CDR TAIL)))
	     (COND ((> (+ (LIST-LENGTH STRINGX) TERMSETLENGTH) (LIST-LENGTH TERM)) (RETURN NIL)))
	     (SETQ TAIL (CDDDDR (LASTN (REVERSE TERM) (- (LIST-LENGTH TERM) TERMSETLENGTH))))
	     (SETQ STRINGXSET
		   (PROG2
		     (MAPC
		       #'(LAMBDA (X) (COND ((NOT (MEMBER X RESULT :TEST #'EQUAL)) (SETQ RESULT (NCONC1 RESULT X)))))
		       STRINGZ)
		     RESULT))
	     (WHILE (< (LIST-LENGTH STRINGXSET) TERMSETLENGTH) (SETQ STRINGZ (NCONC1 STRINGZ (CAR TAIL)))
		    (SETQ STRINGXSET (UNION (LIST (CAR TAIL)) STRINGXSET)) (SETQ TAIL (CDR TAIL)))
	     (COND ((> (+ (LIST-LENGTH STRINGZ) TERMSETLENGTH) (LIST-LENGTH TERM)) (RETURN NIL)))
	     (SETQ TERM (NCONC STRINGX (REVERSE STRINGZ))))))
	(COND
	  ((NOT CDRFLAG)
	   (SETQ TERM
		 (NCONC (DT=ABBREVIATION.NORMALFORM.XYZ.TO.XZ (FIRSTN TERM (1- (LIST-LENGTH TERM))) NIL TERMSETLENGTH)
			(NTHCDR (1- (LIST-LENGTH TERM)) TERM)))))
	(RPLACD TERM (DT=ABBREVIATION.NORMALFORM.XYZ.TO.XZ (CDR TERM) T TERMSETLENGTH)) (RETURN TERM)))

(DEFUN DT=ABBREVIATION.CONTAINS.VARIABLE (TERM)
  ;; EDITED AT 11-NOV-83 |11:18|)
  ;;  INPUT:  A TERM
  ;;  VALUE: T , IF TERM CONTAINS A VARIABLE
  ;;         NIL , ELSE
  (COND ((ATOM TERM) (DT=VARIABLE.IS TERM)) (T (MEMBER-IF #'DT=ABBREVIATION.CONTAINS.VARIABLE (CDR TERM)))))

(DEFUN DT=ABBREVIATION.NORMALFORM.INSERT.BRACKETS (TERM)
  ;; EDITED AT 11-NOV-83 |11:24|)
  ;;  INPUT: A TERM IN INTERNAL NORMALFORM
  ;;         I.E.  WITHOUT BRACKETS
  ;;  EFFECT: FOR ASSOCIATIVE FUNCTIONS,
  ;;          BRACKETS AND FUNCTIONSYMBOLS ARE
  ;;          INSERTED FROM LEFT
  ;;  VALUE: THE TERM WITH RIGHT TERM-SYNTAX
  (COND
    ((AND (CDDDR TERM) (DT-FUNCTION.IS.MARKED ASSOCIATIVE (CAR TERM))) (RPLACD (CDR TERM) (LIST (CONS (CAR TERM) (CDDR TERM))))))
  (RPLACD TERM
	  (MAPCAR
	    #'(LAMBDA (ELEMENT) (COND ((ATOM ELEMENT) ELEMENT) (T (DT=ABBREVIATION.NORMALFORM.INSERT.BRACKETS ELEMENT))))
	    (CDR TERM)))
  TERM)

(DEFUN DT=ABBREVIATION.NORMALFORM.FOR.COMPRESS (TERM)
  ;; EDITED AT 14-DEC-83 |12:05|)
  ;; INPUT: A TERM
  ;; EFFECT: EVERY SUBTERM, WHICH DOES NOT CONTAIN
  ;;         ANY VARIABLE, IS CHANGED INTO ITS
  ;;         NORMALFORM WITH RIGHT SYNTAX
  ;; VALUE: THE CHANGED TERM
  (COND ((ATOM TERM) TERM)
	((MEMBER-IF #'DT=ABBREVIATION.CONTAINS.VARIABLE (CDR TERM))
	 (RPLACD TERM (MAPCAR #'DT=ABBREVIATION.NORMALFORM.FOR.COMPRESS (CDR TERM))) TERM)
	(T (SETQ TERM (DT=ABBREVIATION.NORMALFORM TERM))
	   (COND
	     ((MEMBER 'ASSOCIATIVE (SYMBOL-VALUE 'DT*FUNCTION.ACTUAL.THEORIES))
	      (SETQ TERM (DT=ABBREVIATION.NORMALFORM.INSERT.BRACKETS TERM))))
	   TERM)))

(DEFUN DT=ABBREVIATION.EXPAND.FOR.COMPRESS (TERM)
  ;; EDITED AT 11-NOV-83 |11:35|)
  ;; INPUT: A TERM
  ;; EFFECT: THE TERM WILL BE EXPANDED UNTIL EVERY
  ;;         SUBTERM OF THE FORM  (F A B)
  ;;         F  ASSOCIATIVE, A OR B IS ABBREVIATION
  ;;         HAS THE FOLLOWING PROPERTY
  ;;         NEITHER THE TERM OF A NOR OF B HAS
  ;;         TOPLEVEL FUNCTIONSYMBOL F.
  ;; VALUE:  THE PARTIAL EXPANDED TERM
  (SETQ TERM
	(RPLACD TERM
		(MAPCAR
		  #'(LAMBDA (ELEMENT)
		      (COND
			((AND (ATOM ELEMENT) (DT=ABBREVIATION.IS ELEMENT) (EQL (CAR (DT=ABBREVIATION.GETTERM ELEMENT)) (CAR TERM))
			      (DT-FUNCTION.IS.MARKED ASSOCIATIVE (CAR TERM)))
			 (SETQ ELEMENT (DT=ABBREVIATION.EXPAND ELEMENT 1)))
			((ATOM ELEMENT) ELEMENT) ((CONSP ELEMENT) (SETQ ELEMENT (DT=ABBREVIATION.EXPAND.FOR.COMPRESS ELEMENT)))))
		  (CDR TERM)))))

(DEFUN DT=ABBREVIATION.SEARCH.OLD.SCHEMES (TERM)
  ;; EDITED AT 11-NOV-83 |11:45|)
  ;; INPUT:  A NON-ATOMIC TERM WITH ATOMIC ELEMENTS,
  ;;         AND A TCONC LIST.
  ;; EFFECT: THE ABBREVIATION FOR THIS TERM IS SEARCHED
  ;;         RESP. GENERATED.
  ;; VALUE:  THE ABBREVIATION ADDRESS.
  (MEMBER-IF
    #'(LAMBDA (SCHEME)
	(PROG ((TERM-TAIL (CDR TERM)) (FOUND T) ELEMENT CONTINUE NEXT TREE) (SETQ TREE (ASSOC (CAR TERM) (SYMBOL-PLIST SCHEME)))
	      (SETQ ELEMENT (CAR TERM-TAIL))
	      (WHILE (AND FOUND ELEMENT) (SETQ FOUND NIL) (SETQ CONTINUE T)
		     (WHILE CONTINUE
		       (COND
			 ((SETQ NEXT (CAADR TREE))
			  (COND ((EQL NEXT ELEMENT) (SETQ TREE (SECOND TREE)) (SETQ CONTINUE NIL) (SETQ FOUND T))
				((> NEXT ELEMENT) (SETQ CONTINUE NIL)) (T (SETQ TREE (CDR TREE)))))
			 (T (SETQ CONTINUE NIL))))
		     (COND (FOUND (SETQ TERM-TAIL (CDR TERM-TAIL)) (SETQ ELEMENT (CAR TERM-TAIL)))))
	      (COND (FOUND (RETURN-FROM DT=ABBREVIATION.SEARCH.OLD.SCHEMES (CDR TREE))))))
    (CDR (SYMBOL-VALUE 'DT*ABBREVIATIONS))))

(DEFUN DT=ABBREVIATION.TERM.LESS (SUB.TERM SUPER.TERM)
  ;; EDITED AT 14-DEC-83 |17:43| *)
  ;; INPUT : TWO TERMS
  ;; VALUE: T, IF TERM1 IS LESS THAN TERM2.
  ;;        LESS THAN IS A ORDERING ON TERMS, WHICH
  ;;        RESPECTS ASSOCIATIVITY, AND ARGUMENT.SYMMETRY
  ;;        IDEMPOTENCE, AND ALLOWS TO SELECT THE LEAST
  ;;        TERM OF A EQUIVALENCE CLASS OF ARG.SYMMETRY
  (PROG ((SUBTERM SUB.TERM) (SUPERTERM SUPER.TERM)) (COND ((EQUAL SUBTERM SUPERTERM) (RETURN T)))
	(COND
	  ((AND (ATOM SUBTERM) (DT=ABBREVIATION.IS SUBTERM)) (SETQ SUBTERM (DT=ABBREVIATION.EXPAND SUBTERM 1))
	   (COND
	     ((MEMBER 'ASSOCIATIVE (SYMBOL-VALUE 'DT*FUNCTION.ACTUAL.THEORIES))
	      (SETQ SUBTERM (DT=ABBREVIATION.EXPAND.FOR.COMPRESS SUBTERM))))))
	(COND
	  ((AND (ATOM SUPERTERM) (DT=ABBREVIATION.IS SUPERTERM)) (SETQ SUPERTERM (DT=ABBREVIATION.EXPAND SUPERTERM 1))
	   (COND
	     ((MEMBER 'ASSOCIATIVE (SYMBOL-VALUE 'DT*FUNCTION.ACTUAL.THEORIES))
	      (SETQ SUPERTERM (DT=ABBREVIATION.EXPAND.FOR.COMPRESS SUPERTERM))))))
	(COND ((EQUAL SUBTERM SUPERTERM) (RETURN T)))
	(COND ((ATOM SUBTERM) (COND ((ATOM SUPERTERM) (RETURN (< SUBTERM SUPERTERM))) (T (RETURN T))))
	      (T
	       (COND ((ATOM SUPERTERM) (RETURN NIL))
		     (T
		      (COND
			((EQL (LIST-LENGTH SUBTERM) (LIST-LENGTH SUPERTERM))
			 (MAPC
			   #'(LAMBDA (SUBTERM.CAR)
			       (COND
				 ((NOT (DT=ABBREVIATION.TERM.LESS SUBTERM.CAR (CAR SUPERTERM))) (RETURN-FROM DT=ABBREVIATION.TERM.LESS nil))
				 ((NOT (DT=ABBREVIATION.TERM.EQUAL SUBTERM.CAR (CAR SUPERTERM)))
				  (RETURN-FROM DT=ABBREVIATION.TERM.LESS nil)))
			       (SETQ SUPERTERM (CDR SUPERTERM)))
			   SUBTERM)
			 (RETURN T))
			(T (RETURN (< (LIST-LENGTH SUBTERM) (LIST-LENGTH SUPERTERM)))))))))))

(DEFUN DT=ABBREVIATION.ARGUMENT.NORMALFORM (TERM)
  ;; EDITED AT 14-DEC-83 |18:01| *)
  ;; INPUT: A TERM,WHICH IS NORMALIZED BY ABB.NORMALFORM
  ;; VALUE: THE MINIMAL FORM WITH RESPECT TO
  ;;        ARGUMENT-SYMMETRIES
  (PROG (ARGUMENT.POSSIBILITIES MINIMAL.ARGUMENT)
	(SETQ ARGUMENT.POSSIBILITIES
	      (CONS (CDR TERM)
		    (MAPCAR
		      #'(LAMBDA (SYMMETRY) (MAPCAR #'(LAMBDA (ARGPOS) (CAR (NTHCDR (1- ARGPOS) (CDR TERM)))) SYMMETRY))
		      (DT-FUNCTION.ARGUMENT.SYMMETRIES (CAR TERM)))))
	(SETQ MINIMAL.ARGUMENT (CAR ARGUMENT.POSSIBILITIES))
	(MAPC
	  #'(LAMBDA (ARGUMENT) (COND ((DT=ABBREVIATION.TERM.LESS ARGUMENT MINIMAL.ARGUMENT) (SETQ MINIMAL.ARGUMENT ARGUMENT))))
	  (CDR ARGUMENT.POSSIBILITIES))
	(RETURN MINIMAL.ARGUMENT)))

(DEFUN DT=ABBREVIATION.TERM.EQUAL (SUB.TERM SUPER.TERM)
  ;; EDITED AT 14-DEC-83 |17:57| *)
  ;; INPUT : TWO TERMS
  ;; VALUE: T, IF TERM1 IS EQUAL TO  TERM2.
  ;;        EQUAL MEANS EQUIVALENT RESPECTING
  ;;        ASSOCIATIVITY,  ARGUMENT.SYMMETRY AND
  ;;        IDEMPOTENCE.
  (PROG ((SUBTERM SUB.TERM) (SUPERTERM SUPER.TERM)) (COND ((EQUAL SUBTERM SUPERTERM) (RETURN T)))
	(COND
	  ((AND (ATOM SUBTERM) (DT=ABBREVIATION.IS SUBTERM)) (SETQ SUBTERM (DT=ABBREVIATION.EXPAND SUBTERM 1))
	   (COND
	     ((MEMBER 'ASSOCIATIVE (SYMBOL-VALUE 'DT*FUNCTION.ACTUAL.THEORIES))
	      (SETQ SUBTERM (DT=ABBREVIATION.EXPAND.FOR.COMPRESS SUBTERM))))))
	(COND
	  ((AND (ATOM SUPERTERM) (DT=ABBREVIATION.IS SUPERTERM)) (SETQ SUPERTERM (DT=ABBREVIATION.EXPAND SUPERTERM 1))
	   (COND
	     ((MEMBER 'ASSOCIATIVE (SYMBOL-VALUE 'DT*FUNCTION.ACTUAL.THEORIES))
	      (SETQ SUPERTERM (DT=ABBREVIATION.EXPAND.FOR.COMPRESS SUPERTERM))))))
	(COND ((EQUAL SUBTERM SUPERTERM) (RETURN T)))
	(COND ((ATOM SUBTERM) (COND ((ATOM SUPERTERM) (RETURN (EQUAL SUBTERM SUPERTERM))) (T (RETURN NIL))))
	      (T
	       (COND ((ATOM SUPERTERM) (RETURN NIL))
		     (T
		      (COND
			((EQL (LIST-LENGTH SUBTERM) (LIST-LENGTH SUPERTERM))
			 (MAPL
			   #'(LAMBDA (SUBTERM.CAR)
			       (COND
				 ((NOT (DT=ABBREVIATION.TERM.EQUAL SUBTERM.CAR (CAR SUPERTERM)))
				  (return-from DT=ABBREVIATION.term.EQUAL nil)))
			       (SETQ SUPERTERM (CDR SUPERTERM)))
			   SUBTERM)
			 (RETURN T))
			(T (RETURN NIL)))))))))

(DEFVAR DT*FUNCTION.COUNTER 0)

(DEFVAR DT*FUNCTION.ALL NIL)

(DEFVAR DT*FUNCTION.ADMISSIBLE.THEORIES '(ASSOCIATIVE))

(DEFVAR DT*FUNCTION.ACTUAL.THEORIES NIL)

(DEFVAR DT*FUNCTION.WITH.ARGUMENT.SYMMETRIES NIL)

(DEFVAR DT*FUNCTION.COMPONENTS
	'(PNAME ATTRIBUTES MAX.RANGE.SORT MAX.DOMAIN.SORTS MIN.RANGE.SORTS SORT.LIST SORT.LIST.INVERSE SORT.TREE.CALC
		ARGUMENT.SYMMETRIES ARITY G.L.B.OF.RANGES))

(DEFUN DT-FUNCTION.CREATE (PNAME MAX.RANGE.SORT MAX.DOMAIN.SORTS &optional COMPLETE.SORT.LIST SKOLEM.FLAG)
  ;; EDITED AT 13-DEC-83 11:16 
  ;; VALUE: NEW FUNCTION SYMBOL 
  ;; EFFECT: IF PNAME IS ATOMIC, IT WILL BE USED. OTHERWISE A NEW NAME IS CREATED *)
  ;; IF COMPLETE.SORT.LIST IS NIL, THE CREATED FUNCTION
  ;; IS NOT POLYMORPHIC
  ;; IF SKOLEMFLAG = T,THEN FUNCTION IS MARKED SKOLEM
  (SETQ PNAME (COND ((NULL PNAME) (DT=FUNCTION.CREATE.PNAME)) (T (STRING PNAME))))
  ;; PNAME CREATED 
  (let ((FUNCTION (DT=FUNCTION.STORAGE)))
    (DT=FUNCTION.PUT 'PNAME FUNCTION PNAME)
    (DT=FUNCTION.PUT 'ATTRIBUTES FUNCTION NIL)
    (DT=FUNCTION.PUT 'MAX.RANGE.SORT FUNCTION
		     (if (opt-get.option sort_literals)
			 'any
			 MAX.RANGE.SORT))
    (DT=FUNCTION.PUT 'MAX.DOMAIN.SORTS FUNCTION
		     (if (opt-get.option sort_literals)
			 (mapcar #'(lambda (x) 'any) MAX.DOMAIN.SORTS)
			 MAX.DOMAIN.SORTS))
    (DT=FUNCTION.PUT 'ARGUMENT.SYMMETRIES FUNCTION)
    (DT=FUNCTION.PUT 'ARITY FUNCTION (LENGTH MAX.DOMAIN.SORTS))
    (DT=FUNCTION.PUT 'SORT.LIST FUNCTION
		     (if (opt-get.option sort_literals)
			 (mapcar #'(lambda (x) 'any) COMPLETE.SORT.LIST)
			 COMPLETE.SORT.LIST))
    (DT=FUNCTION.UPDATE.POLYMORPHIC FUNCTION)
    (COND (SKOLEM.FLAG (DT-PUTPROP FUNCTION 'DT-ST*KIND 'SYS-FUNCT))) (SETQ DT*FUNCTION.ALL (CONS FUNCTION DT*FUNCTION.ALL))
    FUNCTION))

(defun dt-function.c.tree (function)
						; Edited:  22-JUN-1990 00:55
						; Authors: PRCKLN
						; Input:   A function
						; Effect:  -
						; Value:   The rewrite compilation tree  for FUNCTION.
  (dt-getprop function 'dt*c.tree))

(defun dt-function.c.c.tree (function)
						; Edited:  26-FEB-1992 16:55
						; Authors: MKRP PRCKLN
						; Input:   A function
						; Effect:  -
						; Value:   The compiled rewrite compilation tree for FUNCTION.
  (dt-getprop function 'dt*c.c.tree))

(defun dt-function.put.c.tree (function c.tree)
						; Edited:  22-JUN-1990 00:55
						; Authors: PRCKLN
						; Input:   A function and a compilation tree
						; Effect:  C.TREE is inserted into the function
						; Value:   Undefined
  (dt-putprop function 'dt*c.tree c.tree))

(defun dt-function.put.c.c.tree (function)
						; Edited:  26-FEB-1992 16:56
						; Authors: MKRP PRCKLN
						; Input:   A functio
						; Effect:  The compiled c tree is inserted into the function
						; Value:   Undefined
  (dt-putprop function 'dt*c.c.tree (compile nil (red=rw_compile.proc (dt-function.c.tree function)))))

(DEFUN DT-FUNCTION.SORT (FUNCTION &optional DOMAINSORTS)
  ;; EDITED AT 12-DEC-83 |16:58|)
  ;; EDITED:| "11-AUG-81 14:30:42")
  ;; INPUT:  FUNCTION- A FUNCTION ADDRESS
  ;; VALUE:  MAX.RANGE OF FUNCTION.
  (COND ((NULL (DT=FUNCTION.GET 'MIN.RANGE.SORTS FUNCTION))
	 (DT=FUNCTION.GET 'MAX.RANGE.SORT FUNCTION))
	((NULL DOMAINSORTS) (ERROR "FUNCTION IS POLYMORPHIC, BUT CALLED WRONG~A" NIL))
	(T (let ((TAIL (DT=FUNCTION.GET 'SORT.TREE.CALC FUNCTION))
		 (DOMAINS DOMAINSORTS))
	     (DODOWN (RPTN (LENGTH DOMAINSORTS))
	       (PROG1 NIL (SETQ TAIL (ASSOC (CAR DOMAINS) TAIL)) (SETQ TAIL (CDR TAIL)) (SETQ DOMAINS (CDR DOMAINS))))
	     TAIL))))

(DEFMACRO DT-FUNCTION.PUTPNAME (FUNCTION P.NAME) `(DT=FUNCTION.PUT 'PNAME ,FUNCTION ,P.NAME))

(DEFMACRO DT-FUNCTION.PNAME (FUNCTION)
  ;; EDITED AT 7-DEC-83 |11:04|)
  ;; EDITED:| "12-NOV-79 14:47:49")
  ;; VALUE:| PNAME OF FUNCTION SYMBOL *)
  `(DT=FUNCTION.GET 'PNAME ,FUNCTION))

(DEFMACRO DT-FUNCTION.ATTRIBUTES (FUNCTION)
  ;; EDITED AT 7-DEC-83 |11:02|)
  ;; EDITED:| "29-JAN-80 18:28:55")
  ;; VALUE:| ATTRIBUTE LIST OF FUNCTION SYMBOL *)
  `(DT=FUNCTION.GET 'ATTRIBUTES ,FUNCTION))

(DEFUN DT-FUNCTION.ADD.ATTRIBUTES (FUNCTION ATTRIBUTES)
  ;; EDITED AT 7-DEC-83 |11:07|)
  ;; EDITED:| "29-JAN-80 18:30:09")
  ;; VALUE:| UNDEFINED *)
  (DT=FUNCTION.PUT 'ATTRIBUTES FUNCTION (APPEND ATTRIBUTES (copy-list (DT=FUNCTION.GET 'ATTRIBUTES FUNCTION))))
  (MAPC #'(LAMBDA (ATTRIBUTE)
	    (when (EQL ATTRIBUTE 'ASSOCIATIVE) (SETQ DT*UNI.CREATES.VARIABLES T))
	    (when (MEMBER ATTRIBUTE DT*FUNCTION.ADMISSIBLE.THEORIES)
	      (SETQ DT*FUNCTION.ACTUAL.THEORIES (INSERT ATTRIBUTE DT*FUNCTION.ACTUAL.THEORIES))))
	ATTRIBUTES))

(DEFMACRO DT-FUNCTION.DOMAINSORTS (FUNCTION)
  ;; EDITED AT 1-DEC-83 |11:48|)

  ;; EDITED: 11-FEB-83 11:25:00
  ;; INPUT:  A FUNCTION ADDRESS.
  ;; VALUE:  ITS MAXIMAL  DOMAINSORTS.
  `(DT=FUNCTION.GET 'MAX.DOMAIN.SORTS ,FUNCTION))

(DEFMACRO DT-FUNCTION.ARITY (FUNCTION)
  ;; EDITED:  2-OCT-83 14:45:44
  ;; INPUT:   A FUNCTION ADDRESS
  ;; VALUE:   THE ARITY OF THIS FUNCTION
  `(DT=FUNCTION.GET 'ARITY ,FUNCTION))

(DEFMACRO DT-FUNCTION.THEORIES NIL
  ;; EDITED: 11-FEB-83 11:30:22
  ;; VALUE:  A LIST WITH ALL FUNCTION-THEORIES WHICH
  ;;         HAVE BEEN DEFINED.
  `DT*FUNCTION.ACTUAL.THEORIES)

(DEFMACRO DT-FUNCTION.IS.MARKED (ATTRIBUTE DT.FUNCTION)
  ;; EDITED: 11-FEB-83 11:32:17
  ;; INPUT:  AN ATTRIBUTE WHICH HAS BEEN DEFINED IN
  ;;         DT-FUNCTION.ADD.ATTRIBUTES
  ;;         AND A FUNCTION.
  ;; VALUE:  T IF THIS FUNCTION HAS THIS ATTRIBUTE,
  ;;         ELSE NIL.
  `(member (QUOTE ,ATTRIBUTE) (dt=FUNCTION.GET 'ATTRIBUTES ,DT.FUNCTION)))

(DEFMACRO DT-FUNCTION.IS (FUNCTION)
  ;; EDITED:| "12-NOV-79 15:03:21")
  ;; VALUE:| T IF FUNCTION IS A FUNCTION SYMBOL NOT YET |DELETED,| ELSE NIL *)
  `(DT=FUNCTION.IS ,FUNCTION))

(DEFMACRO DT-FUNCTION.ALL NIL
  ;; INPUT:  NONE
  ;; VALUE:  LIST OF ALL FUNCTION-ADDRESSES
  ;; EFFECT: RETURNS VALUE
  `DT*FUNCTION.ALL)

(DEFUN DT-FUNCTION.DELETE (FUNCTION)
  ;; EDITED:| "12-NOV-79 15:04:13")
  ;; VALUE:| UNDEFINED *)
  (SETQ DT*FUNCTION.ALL (DELETE FUNCTION DT*FUNCTION.ALL)) (DT-UPDATE.STRANGE.COMMONS 'DT*UNI.CREATES.VARIABLES)
  (DT-UPDATE.STRANGE.COMMONS 'DT*FUNCTION.ACTUAL.THEORIES) (MEM-ERASE FUNCTION NIL))

(DEFUN DT-FUNCTION.PUT.ATTRIBUTES (ARG.FUNCTION ARG.ATTRIBUTES)
  ;; INPUT: A FUNCTION AND ITS ATTRIBUTES
  ;; EFFECT: THE FUNCTIONS ATTRIBUTES ARE REPLACED BY
  ;;         THE GIVEN ONES. ALL RELATED COMMONS ARE
  ;;         UPDATED
  (DT=FUNCTION.PUT 'ATTRIBUTES ARG.FUNCTION ARG.ATTRIBUTES) (DT-UPDATE.STRANGE.COMMONS 'DT*FUNCTION.ACTUAL.THEORIES)
  (DT-UPDATE.STRANGE.COMMONS 'DT*UNI.CREATES.VARIABLE))

(DEFUN DT-FUNCTION.CHANGE (FUNCTION MAX.RANGE.SORT MAX.DOMAIN.SORTS COMPLETE.SORT.LIST)
  ;; INPUT: A FUNCTION AND ITS NEW SORT-DESCRIPTION
  ;; EFFECT: THE FUNCTION IS SET TO THIS NEW DEWCRIPTION
  ;;         AND RELATED COMMONS ARE UPDATED
  ;; EDITED: "19-NOV-79 16:21:23
  ;; VALUE: NEW FUNCTION SYMBOL 
  ;; EFFECT: IF PNAME IS |ATOMIC,| IT WILL BE USED. OTHERWISE A NEW NAME IS CREATED
  ;; IF COMPLETE.SORT.LIST IS NIL, THE CREATED FUNCTION
  ;; IS NOT POLYMORPHIC
  (PROG NIL (DT=FUNCTION.PUT 'ATTRIBUTES FUNCTION NIL) (DT=FUNCTION.PUT 'MAX.RANGE.SORT FUNCTION MAX.RANGE.SORT)
	(DT=FUNCTION.PUT 'MAX.DOMAIN.SORTS FUNCTION MAX.DOMAIN.SORTS) (DT=FUNCTION.PUT 'ARGUMENT.SYMMETRIES FUNCTION)
	(DT=FUNCTION.PUT 'ARITY FUNCTION (LENGTH MAX.DOMAIN.SORTS)) (DT=FUNCTION.PUT 'SORT.LIST FUNCTION COMPLETE.SORT.LIST)
	(DT=FUNCTION.UPDATE.POLYMORPHIC FUNCTION)))

(DEFUN DT-FUNCTION.CHANGE.ONE.ENTRY (FUNCTION ENTRYNAME ENTRY)
  ;; INPUT: A FUNCTION AN ENTRYNAME AND AN ENTRY
  ;;        ENTRYNAME IS ONE OF MAX.RANGE.SORT
  ;;                            MAX.DOMAINSORTS
  ;;                            COMPLETE.SORTLIST
  ;; EFFECT: THE FUNCTION IS UPDATED
  (CASE ENTRYNAME (MAX.RANGESORT (DT=FUNCTION.PUT 'MAX.RANGE.SORT FUNCTION ENTRY))
	(MAX.DOMAINSORTS (DT=FUNCTION.PUT 'MAX.DOMAIN.SORTS FUNCTION ENTRY))
	(COMPLETE.SORTLIST (DT=FUNCTION.PUT 'SORT.LIST FUNCTION ENTRY)) (OTHERWISE (ERROR "WRONG ENTRYNAME~A" NIL)))
  (DT=FUNCTION.UPDATE.POLYMORPHIC FUNCTION))

(DEFUN DT-FUNCTION.IS.SKOLEM (FUNCTION)
  ;; INPUT: A FUNCTION
  ;; VALUE: T, IF FUNCTION IS CREATED WITH SKOLEMFLAG
  (DT-GETPROP FUNCTION 'DT-ST*KIND))

(DEFMACRO DT-FUNCTION.SORTLIST (FUNCTION)
  ;;  INPUT: A FUNCTION
  ;;  VALUE: THE COMPLETE SORTLIST OF FUNCTION ")
  `(DT=FUNCTION.GET 'SORT.LIST ,FUNCTION))

(DEFMACRO DT-FUNCTION.ARGUMENT.SYMMETRIES (FUNCTION)
  ;; EDITED AT 14-DEC-83 |09:44| *)
  ;; INPUT: A FUNCTION
  ;; VALUE: THE ARGUMENT.SYMMETRIES OF THIS FUNCTION
  ;;        ALL POSSIBLE PERMUTATION NOT EQUAL TO ID.
  ;;        IF F(X1 X2 X3 X4) = F(X2 X1 X4 X3) THIS IS
  ;;        DENOTED BY (2 1 4 3)
  `(DT=FUNCTION.GET 'ARGUMENT.SYMMETRIES ,FUNCTION))

(DEFUN DT-FUNCTION.BUILD.TREE.CALC (SORTLIST FUNCTION)
  ;; EDITED AT 13-DEC-83 |10:37| *)
  ;; INPUT : A COMPLETED TUPLE.LIST
  ;; OUTPUT: THE TREE FOR THE CALCULATION OF THE
  ;;         RANGESORT, IF THE DOMAINSORT ARE KNOWN.
  (let ((TREE (MAPCAR #'LIST (DT-SORT.TRANSITIVE.CLOSURE (CAR (MAXIMA (MAPCAR (FUNCTION CAR) SORTLIST)
								      #'(LAMBDA (SORT1 SORT2)
									  (DT-SORT.IS.SUBSORT SORT2 SORT1))))))))
    (MAPC #'(LAMBDA (SUB.TREE)
	      (DT=FUNCTION.BUILD.TREE.CALC.R SUB.TREE SORTLIST
					     (NCONC1 (COPY-TREE (DT=FUNCTION.GET 'MAX.DOMAIN.SORTS FUNCTION))
						     (DT=FUNCTION.GET 'MAX.RANGE.SORT FUNCTION))
					     (COPY-TREE SUB.TREE) 1))
	  TREE)
    TREE))

(DEFMACRO DT-FUNCTION.GLB.OF.RANGES (FUNCTION)
  ;; INPUT : A FUNCTION
  ;; VALUE: THE GREATEST LOWER BOUND OF ALL RANGESORTS
  ;;        OF FUNCTION.
  `(DT=FUNCTION.GET 'G.L.B.OF.RANGES ,FUNCTION))

(DEFMACRO DT-FUNCTION.IS.POLYMORPHIC (FUNCTION)
  ;; INPUT: A FUNCTION
  ;; VALUE: T, IF THE FUNCTION IS POLYMORPHIC
  ;;        NIL ELSE
  `(DT=FUNCTION.GET 'MIN.RANGE.SORTS ,FUNCTION))

(DEFMACRO DT-FUNCTION.UPDATE.POLYMORPHIC (FUNCTION)
  `(let ((function ,function))
     (COND ((DT-FUNCTION.IS.POLYMORPHIC FUNCTION) (DT=FUNCTION.UPDATE.POLYMORPHIC FUNCTION)))))

(DEFMACRO DT-FUNCTION.MAX.DOMAINS.OF.RANGE (FUNCTION RANGE.SORT)
  ;; INPUT: A FUNCTION AND A RANGE
  ;; VALUE: A LIST OF THE MAXIMAL DOMAINS FOR THAT RANGE.
  `(CdR (ASSOC ,RANGE.SORT (DT=FUNCTION.GET 'SORT.LIST.INVERSE ,FUNCTION))))


(DEFMACRO DT-FUNCTION.MAX.RANGE.SORT (FUNCTION)
  ;; INPUT: A FUNCTION
  ;; VALUE: THE MAX RANGE-SORT OF FUNCTION
  `(DT=FUNCTION.GET 'MAX.RANGE.SORT ,FUNCTION))

(DEFMACRO DT-FUNCTION.MIN.RANGES (FUNCTION)
  ;; INPUT: A FUNCTION
  ;; VALUE: A LIST OF THE MINIMAL RANGES OF THE FUNCTION.
  ;;        NIL, IF THE FUNCTION IS NOT POLYMORPHIC.
  `(DT=FUNCTION.GET 'MIN.RANGE.SORTS ,FUNCTION))



(DEFMACRO DT-FUNCTION.SORT.LIST.INVERSE (FUNCTION)
  ;; INPUT : A FUNCTION
  ;; VALUE: THE INVERSE SORT.LIST OF FUNCTION, IF THIS
  ;;        FUNCTION IS POLYMORPHIC.   ELSE NIL
  `(DT=FUNCTION.GET 'SORT.LIST.INVERSE ,FUNCTION))

(DEFUN DT-FUNCTION.TUPLE.COMPLETION (SORTLIST)
  ;; EDITED AT 7-DEC-83 |11:00| *)
  ;; INPUT: A LIST OF TUPLES OF EQUAL LENGTH.
  ;; OUTPUT: A LIST OF TUPLES (THE COMPLETION), WHICH HAS
  ;;         THE PROPERTY: FOR EVERY PAIR FO TUPLES
  ;;         THIS TUPLE IS COMPARABLE BY TUPLE.LESS OR
  ;;         THERE IS A THIRD TUPLE IN THE LIST, WHICH
  ;;         IS A SUBTUPLE OF BOTH.
  (PROG ((MAX.TUPLE (CAR SORTLIST)) (LIST.NEW SORTLIST) (LIST.NEW.NEW T) LIST.OLD MAX.SUBSORT.LIST)
	(MAPC #'(LAMBDA (TUPLE1) (COND ((NOT (DT=FUNCTION.TUPLE.FIT TUPLE1 SORTLIST)) (RETURN NIL)))) SORTLIST)
	(MAPC #'(LAMBDA (TUPLE) (COND ((DT-FUNCTION.TUPLE.LESS MAX.TUPLE TUPLE T) (SETQ MAX.TUPLE TUPLE)))) SORTLIST)
	(MAPC #'(LAMBDA (TUPLE)
		  (COND ((AND (NOT (EQUAL MAX.TUPLE TUPLE)) (NOT (DT-FUNCTION.TUPLE.LESS TUPLE MAX.TUPLE T)))
			 (RETURN NIL))))
	      SORTLIST)
	(WHILE LIST.NEW.NEW
	  (SETQ LIST.NEW.NEW NIL)
	  (MAPC #'(LAMBDA (TUPLE1)
		    (MAPC #'(LAMBDA (TUPLE2)
			      (COND ((AND (NOT (DT-FUNCTION.TUPLE.LESS TUPLE1 TUPLE2 T))
					  (NOT (DT-FUNCTION.TUPLE.LESS TUPLE2 TUPLE1 T)))
				     (SETQ MAX.SUBSORT.LIST (MAPCAR (FUNCTION DT-SORT.GREATEST.COMMON.SUBSORTS) TUPLE1 TUPLE2))
				     (COND
				       ((NOT (MEMBER NIL MAX.SUBSORT.LIST))
					(SETQ LIST.NEW.NEW
					      (NCONC LIST.NEW.NEW
						     (DT=FUNCTION.TUPLE.RETURN.ADMISSIBLE MAX.SUBSORT.LIST LIST.NEW))))))))
			  LIST.NEW))
		SORTLIST)
	  (SETQ LIST.OLD (NCONC LIST.OLD LIST.NEW)) (SETQ LIST.NEW LIST.NEW.NEW))
	(RETURN
	  (SORT (DT=FUNCTION.TUPLE.MINIMIZE LIST.OLD)
		#'(LAMBDA (TUPLE1 TUPLE2) (DT-SORT.IS.SUBSORT (CAR (LAST TUPLE2)) (CAR (LAST TUPLE1))))))))

(DEFUN DT-FUNCTION.TUPLE.LESS (TUPLE1 TUPLE2 &optional COMPLETE.FLAG)
  ;; EDITED AT 5-DEC-83 |17:16| *)
  ;; TUPLE1 IS LESS THAN TUPLE2, IF TUPLE1 IS
  ;; COMPONENT-WISE LESS THAN TUPLE2 IN THE GIVEN
  ;; SORT-ORDERING.
  ;; IF COMPLETE-FLAG IS NIL, THE LAST COMPONENT IS
  ;; IGNORED.
  ;; VALUE:  T  OR NIL
  (SEVERY #'(LAMBDA (SORT1) (PROG1 (DT-SORT.IS.SUBSORT SORT1 (CAR TUPLE2)) (SETQ TUPLE2 (CDR TUPLE2))))
	  #'(LAMBDA (LIST) (COND ((OR (CDDR LIST) COMPLETE.FLAG) (CDR LIST)))) TUPLE1))

(DEFUN DT=FUNCTION.BUILD.TREE.CALC.R (SUB.TREE SORT.LIST MAX.TUPLE OLD.TUPLE DEPTH)
  (DECLARE (SPECIAL DT*NEW.SORT.LIST))
  ;; EDITED AT 13-DEC-83 |10:42| *)
  ;; THE RECURSIVE FUNCTION FOR BUILDING THE SORT TREE
  ;;  FOR RANGESORT.CALCULATION.THE TREE HAS ALL POSSIBLE
  ;;  NODES. FOR EVERY POSSIBLE DOMAINSORT-COMBINATION
  ;;  THER IS A LEAF WITH THE CORRESPONDING RANGESORT.
  (PROG (RANGES NEW.TUPLE TRANS.CLOSURE (CAR.OLD.TUPLE (CAR OLD.TUPLE)))
	(COND
	  ((EQL DEPTH (1- (LIST-LENGTH MAX.TUPLE)))
	   (MAPC
	     #'(LAMBDA (TUPLE)
		 (COND
		   ((DT-SORT.IS.SUBSORT CAR.OLD.TUPLE (CAR TUPLE))
                    (COND ((DT-FUNCTION.TUPLE.LESS OLD.TUPLE TUPLE) (SETQ RANGES (INSERT (CAR (LAST TUPLE)) RANGES)))))))
	     SORT.LIST)
	   (RETURN
	     (RPLACD SUB.TREE (CAR (MAXIMA RANGES #'(LAMBDA (SORT1 SORT2) (DT-SORT.IS.SUBSORT SORT2 SORT1))))))))
	(COND ((> DEPTH (LIST-LENGTH MAX.TUPLE)) (ERROR "ERROR IN SORT-STRUCTURE~A" NIL)))
	(SETQ TRANS.CLOSURE (DT-SORT.TRANSITIVE.CLOSURE (CAR (NTHCDR (1- (1+ DEPTH)) MAX.TUPLE))))
	(C
	  (MAPC
	    #'(LAMBDA (NTH.SORT) (SETQ NEW.TUPLE (APPEND OLD.TUPLE (copy-list (LIST NTH.SORT))))
		      (SETQ SORT.LIST
			    (REMOVE-IF-NOT
			      #'(LAMBDA (TUPLE)
				  (OR (DT-FUNCTION.TUPLE.LESS NEW.TUPLE (FIRSTN TUPLE (1+ DEPTH)) T)
				      (DT-FUNCTION.TUPLE.LESS (FIRSTN TUPLE (1+ DEPTH)) NEW.TUPLE T)))
			      SORT.LIST)))
	    TRANS.CLOSURE))
	(RPLACD SUB.TREE (MAPCAR (FUNCTION LIST) TRANS.CLOSURE))
	(MAPC
	  #'(LAMBDA (SUB.SUB.TREE) (SETQ NEW.TUPLE (COPY-TREE (APPEND OLD.TUPLE (copy-list SUB.SUB.TREE))))
		    (DT=FUNCTION.BUILD.TREE.CALC.R SUB.SUB.TREE DT*NEW.SORT.LIST MAX.TUPLE NEW.TUPLE (1+ DEPTH)))
	  (CDR SUB.TREE))))

(DEFUN DT=FUNCTION.GLB.OF.RANGES (FUNCTION)
  ;; INPUT : A FUNCTION
  ;; VALUE: THE GREATEST LOWER BOUND OF ALL RANGESORTS
  ;;        OF FUNCTION.
  (PROG (SORT.SET (ALL.LOWER.BOUNDS (DT-SORT.ALL)))
	(SETQ SORT.SET
	      (MAXIMA (MAPCAR #'(LAMBDA (TUPLE) (CAR TUPLE)) (DT=FUNCTION.GET 'SORT.LIST.INVERSE FUNCTION))
		      #'(LAMBDA (SORT1 SORT2) (DT-SORT.IS.SUBSORT SORT2 SORT1))))
	(MAPC
	  #'(LAMBDA (SORT) (SETQ ALL.LOWER.BOUNDS (INTERSECTION (DT-SORT.TRANSITIVE.CLOSURE SORT) ALL.LOWER.BOUNDS)))
	  SORT.SET)
	(RETURN (MAXIMA ALL.LOWER.BOUNDS #'(LAMBDA (SORT1 SORT2) (DT-SORT.IS.SUBSORT SORT2 SORT1))))))


(DEFUN DT=FUNCTION.MINIMAL.SORTS (DT*FUNCTION SORTLIST)
  (DECLARE (SPECIAL DT*NEW.SORT DT*MINIMAL.SORTS DT*FUNCTION))
  ;; INPUT: A FUNCTION
  ;;      : A LIST OF SORTS WITH STRUCTURE:
  ;;       ((S11 S12 ... S1M1) ... ( SN1 SN2 ... SNMN ))
  ;;       THE NTH SUBLIST ARE THE POSSIBLE SORTS OF
  ;;       THE NTH ARGUMENT OF FUNCTION.
  ;; VALUE: THE MINIMAL RANGE-SORTS,WHICH CAN BE
  ;;        GENERATED BY ALL TERMS OF THE FORM
  ;;        F(S1I1 S2I2 ...  SNIN)
  (SETQ DT*NEW.SORT NIL)
  (SETQ DT*MINIMAL.SORTS
	(LIST (DT-FUNCTION.SORT DT*FUNCTION (MAPCAR (FUNCTION DT-SORT.GREATEST.COMMON.SUBSORT.OF.LIST) SORTLIST))))
  (CARTESIAN.LOOP SORTLIST
		  #'(LAMBDA (CART.LIST) (SETQ DT*NEW.SORT (DT-FUNCTION.SORT DT*FUNCTION (MAPCAR (FUNCTION CAR) CART.LIST)))
			    (COND
			      ((NOTANY #'(LAMBDA (MIN.SORT) (DT-SORT.IS.SUBSORT MIN.SORT DT*NEW.SORT)) DT*MINIMAL.SORTS)
			       (SETQ DT*MINIMAL.SORTS (CONS DT*NEW.SORT DT*MINIMAL.SORTS))
			       (SETQ DT*MINIMAL.SORTS
				     (MAXIMA DT*MINIMAL.SORTS #'(LAMBDA (SORT1 SORT2) (DT-SORT.IS.SUBSORT SORT2 SORT1))))))
			    (COND
			      ((OR (SET= DT*MINIMAL.SORTS (DT-SORT.MINIMAL.SUBSORTS 'ANY))
				   (SET= DT*MINIMAL.SORTS (DT-FUNCTION.MIN.RANGES DT*FUNCTION)))
			       (RETURN-FROM DT=FUNCTION.MINIMAL.SORTS DT*MINIMAL.SORTS)))))
  DT*MINIMAL.SORTS)

(DEFUN DT=FUNCTION.TUPLE.FIND.INV.TUPLES (TREE.CALC SORT.LIST)
  ;; EDITED AT 13-DEC-83 |11:23| *)
  ;; INPUT: THE SORT.CALCULATION.TREE AND A TUPLE.LIST
  ;; VALUE: A LIST WITH THE STRUCTURE:
  ;;      .. (R (DS11 DS12 ..)(DS21 DS22 ..) ..) ..
  ;;      R IS RANGE.SORT AND (DS11 DS12 ..) IS A
  ;;      COMBINATION OF DOMAINSORTS WITH RANGE R.
  ;;      DOMAINSORT-TPLES ARE MAXIMAL.
  (PROG ((DEPTH (1- (LIST-LENGTH (CAR SORT.LIST)))) RANGE.SORTS RANGE.SORT.MAX.TUPLES OLD.TUPLE)
	(MAPC #'(LAMBDA (TUPLE) (SETQ RANGE.SORTS (UNION RANGE.SORTS (LAST TUPLE)))) SORT.LIST)
	(RETURN
	  (MAPCAR
	    #'(LAMBDA (RANGE.SORT)
		(SETQ RANGE.SORT.MAX.TUPLES
		      (MAPCAN
			#'(LAMBDA (SUB.TREE.CALC) (SETQ OLD.TUPLE (LIST (CAR SUB.TREE.CALC)))
				  (DT=FUNCTION.TUPLE.FIND.INV.TUPLES.R (CDR SUB.TREE.CALC) (1- DEPTH) RANGE.SORT OLD.TUPLE))
			TREE.CALC))
		(MAXIMA RANGE.SORT.MAX.TUPLES (FUNCTION DT-FUNCTION.TUPLE.LESS)) (CONS RANGE.SORT RANGE.SORT.MAX.TUPLES))
	    RANGE.SORTS))))

(DEFUN DT=FUNCTION.TUPLE.FIND.INV.TUPLES.R (SUB.TREE.CALC DEPTH RANGE.SORT OLD.TUPLE)
  ;; EDITED AT 13-DEC-83 |10:32| *)
  ;; THIS IS THE RECURSIVE WORKING FUNCTION OF
  ;;  THE FUNCTION DT=FUNCTION.TUPLE.FIND.INV.TUPLES.
  (COND ((EQUAL DEPTH 0) (COND ((EQUAL RANGE.SORT SUB.TREE.CALC) (LIST (COPY-TREE OLD.TUPLE)))))
	(T
	 (MAPCAN
	   #'(LAMBDA (SUB.SUB.TREE)
	       (DT=FUNCTION.TUPLE.FIND.INV.TUPLES.R (CDR SUB.SUB.TREE) (1- DEPTH) RANGE.SORT
						    (APPEND OLD.TUPLE (LIST (CAR SUB.SUB.TREE)))))
	   SUB.TREE.CALC))))



(DEFUN DT=FUNCTION.TUPLE.MIN.RANGES (SORTLIST)
  ;; EDITED AT 12-DEC-83 |17:34| *)
  ;; INPUT: A LIST OF TUPLES
  ;; VALUE: A LIST OF THE MINIMAL RANGESORTS.
  (PROG (MIN.RANGES) (MAPC #'(LAMBDA (TUPLE) (SETQ MIN.RANGES (UNION MIN.RANGES (LAST TUPLE)))) SORTLIST)
	(RETURN (MAXIMA MIN.RANGES #'(LAMBDA (SORT1 SORT2) (DT-SORT.IS.SUBSORT SORT2 SORT1))))))



(DEFUN DT=FUNCTION.TUPLE.RETURN.ADMISSIBLE (MAX.SUBSORT.LIST SORT.LIST.OLD)
  ;; EDITED AT 5-DEC-83 |19:44| *)
  ;; INPUT: A LIST OF TUPLES, WHICH FIT TOGETHER.
  ;; OUTPUT: THE LIST OF ALL TUPLES, WHICH CAN BE
  ;;         COMPUTED OUT OF A PAIR OF THE INPUT.LIST,
  ;;         REGARDING THAT TEH RANGESORT IS A MONOTONE
  ;;         FUNCTION OF THE DOMAINSORTS.
  ;;         THE UNION OF THE INPUT AND OUTPUTLIST FIT
  ;;          TOGETHER.
  ;;         IF AN ERROR OCCURS, THIS FUNCTION CALL
  ;;         ERROR. IF THIS FUNCTION IS CALLED BY THE
  ;;         COMPILER, THIS HAS TO BE CHANGED.
  (PROG
    (PARMLIST SORT.LIST.NEW (TUPLE.LENGTH (LIST-LENGTH MAX.SUBSORT.LIST)) (CONTINUE T) (CONTINUE.PARM.UPD T) TEST.TUPLE NTH.SORT)
    (DODOWN (RPTN TUPLE.LENGTH) (SETQ PARMLIST (NCONC1 PARMLIST 1)))
    (WHILE CONTINUE (SETQ TEST.TUPLE NIL) (SETQ NTH.SORT 0)
	   (DODOWN (RPTN TUPLE.LENGTH)
	     (PROG NIL (SETQ NTH.SORT (1+ NTH.SORT))
		   (SETQ TEST.TUPLE
			 (NCONC1 TEST.TUPLE
				 (CAR (NTHCDR (1- (CAR (NTHCDR (1- NTH.SORT) PARMLIST)))
					      (CAR (NTHCDR (1- NTH.SORT) MAX.SUBSORT.LIST))))))))
	   (COND
	     ((DT=FUNCTION.TUPLE.FIT TEST.TUPLE SORT.LIST.OLD)
	      (COND
		((DT=FUNCTION.TUPLE.FIT TEST.TUPLE SORT.LIST.NEW) (SETQ SORT.LIST.NEW (NCONC1 SORT.LIST.NEW TEST.TUPLE)))
		(T (ERROR "TUPLE PASST NICHT~A" NIL)))))
	   (SETQ NTH.SORT TUPLE.LENGTH) (SETQ CONTINUE.PARM.UPD T)
	   (WHILE CONTINUE.PARM.UPD
	     (COND
	       ((NEQ (LIST-LENGTH (CAR (NTHCDR (1- NTH.SORT) MAX.SUBSORT.LIST))) (CAR (NTHCDR (1- NTH.SORT) PARMLIST)))
		(RPLACA (NTHCDR (1- NTH.SORT) PARMLIST) (1+ (CAR (NTHCDR (1- NTH.SORT) PARMLIST)))) (SETQ CONTINUE.PARM.UPD NIL))
	       (T
		(COND ((EQL NTH.SORT 1) (SETQ CONTINUE NIL) (SETQ CONTINUE.PARM.UPD NIL))
		      (T (RPLACA (NTHCDR (1- NTH.SORT) PARMLIST) 1)))))
	     (SETQ NTH.SORT (1- NTH.SORT))))
    (RETURN SORT.LIST.NEW)))

(DEFUN DT=FUNCTION.UPDATE.POLYMORPHIC (FUNCTION)
  (DECLARE (SPECIAL DT*NEW.SORT.LIST))
  ;; INPUT: A FUNCTION
  ;; VALUE: UNDEFINED
  ;; SIDEEFFECTS: ALL MEMORY-CELLS FOR POLYMORPHIC
  ;;              FUNCTIONS ARE UPDATED WITH THE
  ;;              INFORMATION OF SORTLIST.
  (SETQ DT*NEW.SORT.LIST (DT=FUNCTION.GET 'SORT.LIST FUNCTION))
  (COND
    ((CDR DT*NEW.SORT.LIST) (SETQ DT*UNI.CREATES.VARIABLES T)
     (DT=FUNCTION.PUT 'MIN.RANGE.SORTS FUNCTION (DT=FUNCTION.TUPLE.MIN.RANGES DT*NEW.SORT.LIST))
     (DT=FUNCTION.PUT 'SORT.TREE.CALC FUNCTION (DT-FUNCTION.BUILD.TREE.CALC DT*NEW.SORT.LIST FUNCTION))
     (DT=FUNCTION.PUT 'SORT.LIST.INVERSE FUNCTION
		      (DT=FUNCTION.TUPLE.FIND.INV.TUPLES (DT=FUNCTION.GET 'SORT.TREE.CALC FUNCTION) DT*NEW.SORT.LIST))
     (DT=FUNCTION.PUT 'G.L.B.OF.RANGES FUNCTION (DT=FUNCTION.GLB.OF.RANGES FUNCTION)))
    (T (DT=FUNCTION.PUT 'SORT.LIST FUNCTION) (DT=FUNCTION.PUT 'MIN.RANGE.SORTS FUNCTION)
       (DT=FUNCTION.PUT 'SORT.TREE.CALC FUNCTION) (DT=FUNCTION.PUT 'SORT.LIST.INVERSE FUNCTION)
       (DT=FUNCTION.PUT 'G.L.B.OF.RANGES FUNCTION (DT=FUNCTION.GET 'MAX.RANGE.SORT FUNCTION)))))

(DEFMACRO DT=FUNCTION.STORAGE NIL
  ;; EDITED:  9-APR-84 15:57:13   SYNTHETIC
  ;;  VALUE: POINTER TO A NEW STORAGE UNIT FOR A FUNCTION
  ;;  STRUCTURE: CELL   COMPONENT
  ;;                1   PNAME
  ;;                2   ATTRIBUTES
  ;;                3   MAX.RANGE.SORT
  ;;                4   MAX.DOMAIN.SORTS
  ;;                5   MIN.RANGE.SORTS
  ;;                6   SORT.LIST
  ;;                7   SORT.LIST.INVERSE
  ;;                8   SORT.TREE.CALC
  ;;                9   ARGUMENT.SYMMETRIES
  ;;                10   ARITY
  ;;                11   G.L.B.OF.RANGES
  `(MEM-NEW 'FUNCTION 11))

(DEFMACRO DT=FUNCTION.CREATE.PNAME NIL
  ;; EDITED:| "19-NOV-79 16:26:32")
  ;; VALUE:| NEW NAME FOR A FUNCTION *)
  `(CONCATENATE 'STRING "SKO" (PRINC-TO-STRING (SETQ DT*FUNCTION.COUNTER (1+ DT*FUNCTION.COUNTER)))))

(DEFMACRO DT=FUNCTION.IS (FCT)
  ;; EDITED:| "18-JAN-80 12:35:52")
  ;; VALUE:| T IF FUNCTION IS A FUNCTION SYMBOL NOT YET |DELETED,| ELSE NIL *)
  `(EQL (MEM-TYPE ,FCT) 'FUNCTION))

(DEFMACRO DT=FUNCTION.GET (COMPONENT FUNCTION)
  (if (OR (ATOM COMPONENT) (NEQ (CAR COMPONENT) 'QUOTE))
      `(DT==FUNCTION.GET ,COMPONENT ,FUNCTION)
      `(meM-GET ,FUNCTION
		,(OR (LISTPOS (SECOND COMPONENT) DT*FUNCTION.COMPONENTS)
		     (ERROR "illegal component in dt=function.get~A"
			    COMPONENT)))))


(DEFUN DT==FUNCTION.GET (COMPONENT FUNCTION)
  ;; EDITED:  9-APR-84 15:57:13   SYNTHETIC
  ;; INPUT:  COMPONENT IS ONE OF THE ATOMS:
  ;;         PNAME, ATTRIBUTES, MAX.RANGE.SORT,
  ;;         MAX.DOMAIN.SORTS, MIN.RANGE.SORTS,
  ;;         SORT.LIST, SORT.LIST.INVERSE,
  ;;         SORT.TREE.CALC, ARGUMENT.SYMMETRIES, ARITY,
  ;;         G.L.B.OF.RANGES,
  ;;         FUNCTION IS A FUNCTION ADDRESS
  ;; VALUE:  CONTENTS OF THE MEMORY ARRAY ELEMENT
  ;;         FOR FUNCTION, DENOTED BY COMPONENT.
  (CASE COMPONENT
    (PNAME (MEM-GET FUNCTION 1))
    (ATTRIBUTES (MEM-GET FUNCTION 2))
    (MAX.RANGE.SORT (MEM-GET FUNCTION 3))
    (MAX.DOMAIN.SORTS (MEM-GET FUNCTION 4))
    (MIN.RANGE.SORTS (MEM-GET FUNCTION 5))
    (SORT.LIST (MEM-GET FUNCTION 6))
    (SORT.LIST.INVERSE (MEM-GET FUNCTION 7))
    (SORT.TREE.CALC (MEM-GET FUNCTION 8))
    (ARGUMENT.SYMMETRIES (MEM-GET FUNCTION 9))
    (ARITY (MEM-GET FUNCTION 10)) (G.L.B.OF.RANGES (MEM-GET FUNCTION 11))
    (OTHERWISE (ERROR "DT=FUNCTION.GET: ILLEGAL COMPONENT~A" COMPONENT))))

(defmacro DT=FUNCTION.PUT (COMPONENT FUNCTION &OPTIONAL VALUE)
  (COND
    ((OR (ATOM COMPONENT) (NEQ 'QUOTE (CAR COMPONENT)))
     `(DT==FUNCTION.PUT ,COMPONENT ,FUNCTION ,VALUE))
    (T `(MEM-PUT ,FUNCTION
		 ,(OR (LISTPOS (SECOND COMPONENT) DT*FUNCTION.COMPONENTS)
		      (ERROR "Illegal component in dt=function.put~A" COMPONENT))
		 ,VALUE))))

(DEFUN DT==FUNCTION.PUT (COMPONENT FUNCTION VALUE)
  ;; EDITED:  9-APR-84 15:57:13   SYNTHETIC
  ;; INPUT:  COMPONENT IS ONE OF THE ATOMS:
  ;;         PNAME, ATTRIBUTES, MAX.RANGE.SORT,
  ;;         MAX.DOMAIN.SORTS, MIN.RANGE.SORTS,
  ;;         SORT.LIST, SORT.LIST.INVERSE,
  ;;         SORT.TREE.CALC, ARGUMENT.SYMMETRIES, ARITY,
  ;;         G.L.B.OF.RANGES,
  ;;         FUNCTION IS A FUNCTION ADDRESS
  ;;         VALUE IS ANY S-EXPRESSION
  ;; EFFECT: VALUE IS ASSIGNED TO MEMORY ARRAY ELEMENT
  ;;         FOR FUNCTION, DENOTED BY COMPONENT.
  (CASE COMPONENT
    (PNAME (MEM-PUT FUNCTION 1 VALUE))
    (ATTRIBUTES (MEM-PUT FUNCTION 2 VALUE))
    (MAX.RANGE.SORT (MEM-PUT FUNCTION 3 VALUE))
    (MAX.DOMAIN.SORTS (MEM-PUT FUNCTION 4 VALUE))
    (MIN.RANGE.SORTS (MEM-PUT FUNCTION 5 VALUE))
    (SORT.LIST (MEM-PUT FUNCTION 6 VALUE))
    (SORT.LIST.INVERSE (MEM-PUT FUNCTION 7 VALUE))
    (SORT.TREE.CALC (MEM-PUT FUNCTION 8 VALUE))
    (ARGUMENT.SYMMETRIES (MEM-PUT FUNCTION 9 VALUE))
    (ARITY (MEM-PUT FUNCTION 10 VALUE))
    (G.L.B.OF.RANGES (MEM-PUT FUNCTION 11 VALUE))
    (OTHERWISE (ERROR "DT=FUNCTION.PUT: ILLEGAL COMPONENT~A" COMPONENT))))

(DEFVAR DT*PREDICATE.ADMISSABLE.ATTRIBUTES '(SYMMETRIC DEFINED REFLEXIVE))

(DEFVAR DT*PREDICATE.COUNTER 0)

(DEFVAR DT*EQUALITY.SYMBOLS '("=" ":=" "=:" ":=:"))

(DEFVAR DT*EQUALITY.PREDICATES nil)

(DEFVAR DT*NONEQUALITY.PREDICATES NIL)

(DEFVAR DT*PREDICATE.ALL NIL)

(DEFVAR DT*PREDICATE.WITH.ATTRIBUTES NIL)

(DEFVAR DT*PREDICATE.COMPONENTS
	'(PNAME POSITIVE.OCCURRENCES NEGATIVE.OCCURRENCES DOMAINSORTS ATTRIBUTES REFL.CLAUSE +ROTHERSIDES -ROTHERSIDES
		+SOTHERSIDES -SOTHERSIDES +TOTHERSIDES -TOTHERSIDES))

(DEFVAR DT*TRUE.PREDICATE NIL)

(DEFVAR DT*FALSE.PREDICATE NIL)

(DEFUN DT-PREDICATE.CREATE (PNAME DOMAINSORTS)
  ;; EDITED: 1-FEB-82 16:33:57
  ;; VALUE: NEW PREDICATE SYMBOL
  ;; EFFECT: IF PNAME IS ATOMIC, IT WILL BE USED. OTHERWISE A NEW NAME IS CREATED 
  (SETQ PNAME (cOND ((NULL PNAME)
		     (DT=PREDICATE.CREATE.PNAME))
		    (T (STRING PNAME))))
  ;; PNAME CREATED 
  (let ((PREDICATE (DT=PREDICATE.STORAGE)))
    (DT-PREDICATE.PUT 'PNAME PREDICATE PNAME)
    (DT-PREDICATE.PUT 'DOMAINSORTS PREDICATE DOMAINSORTS)
    (DT-PREDICATE.PUT 'ATTRIBUTES PREDICATE NIL)
    (COND ((MEMBER PNAME DT*EQUALITY.SYMBOLS :test #'string=)
	   (DT-PREDICATE.PUT 'ATTRIBUTES PREDICATE (LIST 'SYMMETRIC 'REFLEXIVE))
	   (SETQ DT*EQUALITY.PREDICATES (CONS PREDICATE DT*EQUALITY.PREDICATES))
	   (SETF (get 'DT*PREDICATE.WITH.ATTRIBUTES 'SYMMETRIC)
		 (CONS PREDICATE (GET 'DT*PREDICATE.WITH.ATTRIBUTES 'SYMMETRIC)))
	   (SETF (get 'DT*PREDICATE.WITH.ATTRIBUTES 'REFLEXIVE)
		 (CONS PREDICATE (GET 'DT*PREDICATE.WITH.ATTRIBUTES 'REFLEXIVE))))
	  (T (SETQ DT*NONEQUALITY.PREDICATES (CONS PREDICATE DT*NONEQUALITY.PREDICATES))))
    (DT=PREDICATE.INSERT.OTHERSIDES PREDICATE)
    (DT-PREDICATE.PUT 'POSITIVE.OCCURRENCES PREDICATE NIL)
    (DT-PREDICATE.PUT 'NEGATIVE.OCCURRENCES PREDICATE NIL)
    (DT-PREDICATE.PUT 'REFL.CLAUSE PREDICATE NIL)
    (SETQ DT*PREDICATE.ALL (CONS PREDICATE DT*PREDICATE.ALL))
    (COND ((STRING= "FALSE" (STRING PNAME)) (SETQ DT*FALSE.PREDICATE PREDICATE)))
    (COND ((STRING= "TRUE" (STRING PNAME)) (SETQ DT*TRUE.PREDICATE PREDICATE)))
    (COND ((STRING= "E" (STRING PNAME)) (SETQ DT*element.PREDICATE PREDICATE)))
    PREDICATE))
#|(DEFUN DT-PREDICATE.CREATE (PNAME DOMAINSORTS)
  ;; EDITED: 1-FEB-82 16:33:57
  ;; VALUE: NEW PREDICATE SYMBOL
  ;; EFFECT: IF PNAME IS ATOMIC, IT WILL BE USED. OTHERWISE A NEW NAME IS CREATED 
  (SETQ PNAME (cOND ((NULL PNAME)
		     (DT=PREDICATE.CREATE.PNAME))
		    (T (STRING PNAME))))
  ;; PNAME CREATED 
  (let ((PREDICATE (DT=PREDICATE.STORAGE)))
    (DT-PREDICATE.PUT 'PNAME PREDICATE PNAME)
    (DT-PREDICATE.PUT 'DOMAINSORTS PREDICATE DOMAINSORTS)
    (DT-PREDICATE.PUT 'ATTRIBUTES PREDICATE NIL)
    (COND ((MEMBER PNAME DT*EQUALITY.SYMBOLS :test #'string=)
	   (DT-PREDICATE.PUT 'ATTRIBUTES PREDICATE (LIST 'SYMMETRIC 'REFLEXIVE))
	   (SETQ DT*EQUALITY.PREDICATES (CONS PREDICATE DT*EQUALITY.PREDICATES))
	   (SETF (get 'DT*PREDICATE.WITH.ATTRIBUTES 'SYMMETRIC)
		 (CONS PREDICATE (GET 'DT*PREDICATE.WITH.ATTRIBUTES 'SYMMETRIC)))
	   (SETF (get 'DT*PREDICATE.WITH.ATTRIBUTES 'REFLEXIVE)
		 (CONS PREDICATE (GET 'DT*PREDICATE.WITH.ATTRIBUTES 'REFLEXIVE))))
	  (T (SETQ DT*NONEQUALITY.PREDICATES (CONS PREDICATE DT*NONEQUALITY.PREDICATES))))
    (DT=PREDICATE.INSERT.OTHERSIDES PREDICATE)
    (DT-PREDICATE.PUT 'POSITIVE.OCCURRENCES PREDICATE NIL)
    (DT-PREDICATE.PUT 'NEGATIVE.OCCURRENCES PREDICATE NIL)
    (DT-PREDICATE.PUT 'REFL.CLAUSE PREDICATE NIL)
    (SETQ DT*PREDICATE.ALL (CONS PREDICATE DT*PREDICATE.ALL))
    (COND ((STRING= "FALSE" (STRING PNAME)) (SETQ DT*FALSE.PREDICATE PREDICATE)))
    (COND ((STRING= "TRUE" (STRING PNAME)) (SETQ DT*TRUE.PREDICATE PREDICATE)))
    PREDICATE))|#

(DEFMACRO DT-PREDICATE.PUTPNAME (PREDICATE P.NAME) `(DT-PREDICATE.PUT 'PNAME ,PREDICATE ,P.NAME))

(DEFMACRO DT-PREDICATE.PNAME (PREDICATE)
  ;; EDITED: 12-NOV-79 14:47:49
  ;; VALUE:  PNAME OF PREDICATE SYMBOL
  `(DT-PREDICATE.GET 'PNAME ,PREDICATE))

(DEFMACRO DT-PREDICATE.IS.SYMMETRIC (PREDICATE)
  ;; EDITED: 2-FEB-82 13:29:02
  ;; INPUT:  A ADRESS OF A PREDICATE SYMBOL.
  ;; VALUE:  T IF PREDICATE IS SYMMETRIC ELSE NIL.
  `(MEMBER 'SYMMETRIC (DT-PREDICATE.GET 'ATTRIBUTES ,PREDICATE)))

(DEFMACRO DT-PREDICATE.IS.EQUALITY (PREDICATE)
  ;; EDITED:  1-FEB-82 13:53:02
  ;; INPUT:  ADDRESS OF A PREDICATE SYMBOL
  ;; EFFECT: RETURNS VALUE
  ;; VALUE:  T IF THE GIVEN PREDICATE DENOTES THE
  ;;         EQUALITY , ELSE NIL.
  `(MEMBER ,PREDICATE DT*EQUALITY.PREDICATES))

(DEFUN DT-PREDICATE.ARE.SAME (PREDICATE1 PREDICATE2)
  ;; EDITED: 8-JUL-80 09:27:19
  ;; INPUT:  ADDRESS OF TWO PREDICATE SYMBOLS
  ;; EFFECT: RETURNS VALUE
  ;; VALUE:  T IF BOTH SYMBOLS DENOTE THE SAME PREDICATE
  ;;         ELSE NIL.
  (OR (EQL PREDICATE1 PREDICATE2)
      (AND (MEMBER PREDICATE1 DT*EQUALITY.PREDICATES)
	   (MEMBER PREDICATE2 DT*EQUALITY.PREDICATES))))

(DEFMACRO DT-PREDICATE.IS (PREDICATE)
  ;; EDITED: 12-NOV-79 15:03:21
  ;; VALUE:  T IF PREDICATE IS A PREDICATE SYMBOL NOT YET DELETED, ELSE NIL.
  `(DT=PREDICATE.IS ,PREDICATE))

(DEFMACRO DT-PREDICATE.MARKED.ALL (ATTRIBUTE)
  ;; EDITED: 17-FEB-83 15:12:17
  ;; INPUT:  AN ATOM
  ;; VALUE:  ALL PREDICATE ADDRESSES WHICH POSSESS THIS
  ;;         ATTRIBUTE.
  `(GET 'DT*PREDICATE.WITH.ATTRIBUTES ,ATTRIBUTE))

(DEFMACRO DT-PREDICATE.ALL NIL
  ;; INPUT:  NONE
  ;; VALUE:  LIST OF ALL FUNCTION-ADDRESSES
  ;; EFFECT: RETURNS VALUE
  'DT*PREDICATE.ALL)

(DEFMACRO DT-PREDICATE.EQUALITIES NIL
  ;; INPUT:  NONE
  ;; VALUE:  A LIST OF EQUALITY-PREDICATE-ADDRESSES
  ;; EFFECT: RETURNS VALUE
  `DT*EQUALITY.PREDICATES)

(defparameter DT*element.PREDICATE nil)

(DEFMACRO DT-PREDICATE.element NIL
 ;; VALUE: THE ADDRESS OF THE 'TRUE' PREDICATE
  'DT*element.PREDICATE)

(DEFMACRO DT-PREDICATE.NONEQUALITIES NIL
  ;; EDITED: 11-FEB-83 13:04:05
  ;; VALUE:  LIST OF ALL NON-EQUALITY PREDICATES.
  `DT*NONEQUALITY.PREDICATES)

(DEFMACRO DT-PREDICATE.PUT.POSITIVE.OCCURRENCES (PREDICATE LIST)
  ;; INPUT:  ADDRESS OF AN PREDICATE SYMBOL AND A
  ;;         LIST OF THE POSITIVE OCCURENCES OF THE
  ;;         PREDICATE
  ;; EFFECT: STORES THE LIST IN CELL 3 OF THE
  ;;         PREDICATE
  ;; VALUE:  LIST
  `(DT-PREDICATE.PUT 'POSITIVE.OCCURRENCES ,PREDICATE ,LIST))

(defun DT-PREDICATE.DELete.POSITIVE.OCCURRENCES (clause predicate)
						; Edited:  19-NOV-1991 23:25
						; Authors: PRCKLN
						; Input:   A clause and a predicate
						; Effect:  Deletes the clause from the positive occurrence lists
						; Value:   Undefined.
  (DT-PREDICATE.PUT.POSITIVE.OCCURRENCES predicate (remASSOC CLAUSE (DT-PREDICATE.POSITIVE.OCCURRENCES PREDICATE))))

(defun DT-PREDICATE.DELete.POSITIVE.OCCURRENCE (clause litno predicate)
						; Edited:  19-NOV-1991 23:25
						; Authors: PRCKLN
						; Input:   A clause literal and a predicate
						; Effect:  Deletes the clause literal from the positive occurrence lists
						; Value:   Undefined.
  (DT-PREDICATE.PUT.POSITIVE.OCCURRENCES
    predicate (REMVALUESASSOC CLAUSE (list litno) (DT-PREDICATE.POSITIVE.OCCURRENCES PREDICATE))))

(DEFMACRO DT-PREDICATE.POSITIVE.OCCURRENCES (PREDICATE)
  ;; INPUT:  ADDRESS OF AN PREDICATE SYMBOL
  ;; EFFECT: RETURNS VALUE
  ;; VALUE:  LIST OF THE POSITIVE OCCURENCES OF THE
  ;;         PREDICATE SYMBOL
  `(DT-PREDICATE.GET 'POSITIVE.OCCURRENCES ,PREDICATE))

(DEFMACRO DT-PREDICATE.PUT.NEGATIVE.OCCURRENCES (PREDICATE LIST)
  ;; INPUT:  ADDRESS OF AN PREDICATE SYMBOL AND A
  ;;         LIST OF THE NEGATIVE OCCURENCES OF THE
  ;;         PREDICATE
  ;; EFFECT: STORES THE LIST IN CELL 4 OF THE
  ;;         PREDICATE
  ;; VALUE:  LIST
  `(DT-PREDICATE.PUT 'NEGATIVE.OCCURRENCES ,PREDICATE ,LIST))

(defun DT-PREDICATE.DELete.NEGATIVE.OCCURRENCES (clause predicate)
						; Edited:  19-NOV-1991 23:25
						; Authors: PRCKLN
						; Input:   A clause and a predicate
						; Effect:  Deletes the clause from the negative occurrence lists
						; Value:   Undefined.
  (DT-PREDICATE.PUT.NEGATIVE.OCCURRENCES predicate (remASSOC CLAUSE (DT-PREDICATE.NEGATIVE.OCCURRENCES PREDICATE))))

(defun DT-PREDICATE.DELete.NEGATIVE.OCCURRENCE (clause litno predicate)
						; Edited:  19-NOV-1991 23:25
						; Authors: PRCKLN
						; Input:   A clause literal and a predicate
						; Effect:  Deletes the clause literal from the negative occurrence lists
						; Value:   Undefined.
  (DT-PREDICATE.PUT.NEGATIVE.OCCURRENCES
    predicate (REMVALUESASSOC CLAUSE (list litno) (DT-PREDICATE.NEGATIVE.OCCURRENCES PREDICATE))))

(DEFMACRO DT-PREDICATE.NEGATIVE.OCCURRENCES (PREDICATE)
  ;; INPUT:  ADDRESS OF AN PREDICATE SYMBOL
  ;; EFFECT: RETURNS VALUE
  ;; VALUE:  LIST OF THE NEGATIVE OCCURENCES OF THE
  ;;         PREDICATE SYMBOL
  `(DT-PREDICATE.GET 'NEGATIVE.OCCURRENCES ,PREDICATE))

(DEFMACRO DT-PREDICATE.DOMAINSORTS (PREDICATE)
  ;; EDITED: 11-FEB-83 12:56:25
  ;; INPUT:  A PREDICATE ADDRESS
  ;; VALUE:  ITS DOMAINSORTS
  `(DT-PREDICATE.GET 'DOMAINSORTS ,PREDICATE))

(DEFMACRO DT-PREDICATE.PUTSORT (PREDICATE DOMAIN.SORTS)
 ;; EDITED AT 4-OCT-84 |19:44|)
  ;; INPUT: A PREDICATE AND THE NEW DOMAINSORTS
  ;; VALUE: UNDEFINED
  ;; SIDEEFFECTS: THE DOMAINSORTS OF THE PREDICATE ARE
  ;;              CHANGED
  `(DT-PREDICATE.PUT 'DOMAINSORTS ,PREDICATE ,DOMAIN.SORTS))

(DEFMACRO DT-PREDICATE.ATTRIBUTES (PREDICATE)
  ;; EDITED: 11-FEB-83 12:56:25
  ;; INPUT:  A PREDICATE ADDRESS
  ;; VALUE:  ITS ATTRIBUTES
  `(DT-PREDICATE.GET 'ATTRIBUTES ,PREDICATE))

(DEFUN DT-PREDICATE.PUT.ATTRIBUTES (PREDICATE ATTRIBUTES)
 ;; INPUT: A PREDICATE AND ITS NEW ATTRIBUTES
  ;; EFFECT: THE PREDICATES ATTRIBUTES ARE SET TO THE
  ;;         GIVEN ONES. RELATED COMMONS ARE UPDATED
  (SETF (SYMBOL-PLIST 'DT*PREDICATE.WITH.ATTRIBUTES)
    (SMAPL
      #'(LAMBDA (PROPTAIL)
          (COND
            ((MEMBER (CAR PROPTAIL) DT*PREDICATE.ADMISSABLE.ATTRIBUTES)
              (RPLACA (CDR PROPTAIL) (DELETE PREDICATE (SECOND PROPTAIL))))))
      #'(LAMBDA (LIST) (SETQ LIST (CDDR LIST))) (SYMBOL-PLIST 'DT*PREDICATE.WITH.ATTRIBUTES)))
  (DT-PREDICATE.PUT 'ATTRIBUTES PREDICATE NIL) (DT-PREDICATE.ADD.ATTRIBUTES PREDICATE ATTRIBUTES))

(DEFUN DT-PREDICATE.ADD.ATTRIBUTES (PREDICATE ATTRIBUTES)
  ;; EDITED: 11-FEB-83 12:56:25
  ;; INPUT:  A PREDICATE ADDRESS AND A LIST OF ATOMS
  ;; EFFECT: THE ATTRIBUTES ARE APPENDED.
  ;; VALUE:  UNDEFINED
  (DT-PREDICATE.PUT 'ATTRIBUTES PREDICATE (APPEND ATTRIBUTES (copy-list (DT-PREDICATE.GET 'ATTRIBUTES PREDICATE))))
  (MAPC #'(LAMBDA (ATTRIBUTE)
	    (SETF (get 'DT*PREDICATE.WITH.ATTRIBUTES ATTRIBUTE)
		  (CONS PREDICATE (GET 'DT*PREDICATE.WITH.ATTRIBUTES ATTRIBUTE))))
	ATTRIBUTES)
  (COND ((MEMBER 'SYMMETRIC ATTRIBUTES) (DT=PREDICATE.OTHERSIDES.ADD.SYMMETRY PREDICATE))
	((MEMBER 'aSYMMETRIC ATTRIBUTES) (DT=PREDICATE.OTHERSIDES.ADD.aSYMMETRY PREDICATE))))

(DEFMACRO DT-PREDICATE.IS.MARKED (ATTRIBUTE DT.PREDICATE)

  ;; EDITED: 11-FEB-83 12:57:47
  ;; INPUT:  AN ATTRIBUTE LIKE SYMMETRIC, REFLEXIVE ETC.
  ;;         AND A PREDICATE ADDRESS
  ;; VALUE:  T IF PREDICATE HAS THIS ATTRIBUTE, ELSE NIL.
  `(MEMBer (QUOTE ,ATTRIBUTE) (DT-PREDICATE.GET 'ATTRIBUTES ,DT.PREDICATE)))

(DEFMACRO DT-PREDICATE.REFL.CLAUSE (PREDICATE)
  ;; EDITED: 14-FEB-83 11:35:42
  ;; INPUT:  A PREDICATE ADDRESS
  ;; VALUE:  THE REFLEXIVITY RESP. IRREFLEXIVITY CLAUSE
  ;;         BELONGING TO THIS PREDICATE.
  `(DT-PREDICATE.GET 'REFL.CLAUSE ,PREDICATE))

(DEFUN DT-PREDICATE.DELETE (PREDICATE)
  ;; INPUT: A PREDICATE
  ;; EFFECT: THE PREDICATES IS DELETED
  ;;                     RELATED COMMONS ARE UPDATED
  ;; EDITED:| "12-NOV-79 15:04:13")
  ;; VALUE:| UNDEFINED *)
  (COND
    ((MEMBER PREDICATE DT*EQUALITY.PREDICATES) (SETQ DT*EQUALITY.PREDICATES (DELETE PREDICATE DT*EQUALITY.PREDICATES)))
    (T (SETQ DT*NONEQUALITY.PREDICATES (DELETE PREDICATE DT*NONEQUALITY.PREDICATES))))
  (COND ((EQL PREDICATE DT*TRUE.PREDICATE) (SETQ DT*TRUE.PREDICATE NIL))
	((EQL PREDICATE DT*FALSE.PREDICATE) (SETQ DT*FALSE.PREDICATE NIL)))
  (SETQ DT*PREDICATE.ALL (DELETE PREDICATE DT*PREDICATE.ALL))
  (SETF (SYMBOL-PLIST 'DT*PREDICATE.WITH.ATTRIBUTES)
	(SMAPL
	  #'(LAMBDA (PROPTAIL)
	      (COND
		((MEMBER (CAR PROPTAIL) DT*PREDICATE.ADMISSABLE.ATTRIBUTES)
		 (RPLACA (CDR PROPTAIL) (DELETE PREDICATE (SECOND PROPTAIL))))))
	  #'(LAMBDA (LIST) (SETQ LIST (CDDR LIST))) (SYMBOL-PLIST 'DT*PREDICATE.WITH.ATTRIBUTES)))
  (MEM-ERASE PREDICATE NIL))

(DEFMACRO DT-PREDICATE.CONVERTSET (PREDICATES)
  ;; EDITED:  8-APR-83 14:21:32
  ;; INPUT:   A LIST OF PREDICATES
  ;; VALUE:   THE SET OF ELEMENTS OF THE INPUTLIST IN
  ;;          FORM OF A LIST (I.E. MULTIBLE OCCURRENCES
  ;;          OF THE SAME ELEMENT ARE ELIMINATED). IF
  ;;          THE INPUT LIST CONTAINS SOME EQUALITY
  ;;          PREDICATES THEN THE VALUE CONTAINS ONLY
  ;;          ONE EQUALITY PREDICATE.
  `(REMOVE-DUPLICATES ,PREDICATES))

(DEFMACRO DT-PREDICATE.GET (COMPONENT PREDICATE)
  (COND ((OR (ATOM COMPONENT) (NEQ 'QUOTE (CAR COMPONENT)))
	 `(dT=PREDICATE.GET ,COMPONENT ,PREDICATE))
	(T `(MEM-GET ,PREDICATE
		     ,(OR (LISTPOS (SECOND COMPONENT) DT*PREDICATE.COMPONENTS)
			  (ERROR "illegal component in dt-predicate.get: ~A" COMPONENT))))))

(DEFUN DT=PREDICATE.GET (COMPONENT PREDICATE)
  ;; EDITED: 29-SEP-83 11:20:48
  ;; INPUT:  ONE OF THE ATOMS LISTED IN
  ;;         DT*PREDICATE.COMPONENTS AND A PREDICATE
  ;; VALUE:  THE CONTENTS OF THE CORRESPONDING MEMORY
  ;;         CELL.
  (CASE COMPONENT
    (PNAME (MEM-GET PREDICATE 1))
    (POSITIVE.OCCURRENCES (MEM-GET PREDICATE 2))
    (NEGATIVE.OCCURRENCES (MEM-GET PREDICATE 3))
    (DOMAINSORTS (MEM-GET PREDICATE 4))
    (ATTRIBUTES (MEM-GET PREDICATE 5))
    (REFL.CLAUSE (MEM-GET PREDICATE 6))
    (+ROTHERSIDES (MEM-GET PREDICATE 7))
    (-ROTHERSIDES (MEM-GET PREDICATE 8))
    (+SOTHERSIDES (MEM-GET PREDICATE 9))
    (-SOTHERSIDES (MEM-GET PREDICATE 10))
    (+TOTHERSIDES (MEM-GET PREDICATE 11))
    (-TOTHERSIDES (MEM-GET PREDICATE 12))
    (OTHERWISE (ERROR "ILLEGAL COMPONENT IN DT=PREDICATE.GET: ~A" COMPONENT))))

(DEFMACRO DT-PREDICATE.PUT (COMPONENT PREDICATE &OPTIONAL VALUE)
  (COND
    ((OR (ATOM COMPONENT) (NEQ 'QUOTE (CAR COMPONENT)))
     `(DT=PREDICATE.PUT ,COMPONENT ,PREDICATE ,VALUE))
    (T `(MEM-PUT ,PREDICATE
		 ,(OR (LISTPOS (SECOND COMPONENT) DT*PREDICATE.COMPONENTS)
		      (ERROR "illegal component in dt-predicate.put: ~A" COMPONENT))
		 ,VALUE))))

(DEFUN DT=PREDICATE.PUT (COMPONENT PREDICATE VALUE)
  ;; EDITED: 29-SEP-83 11:32:40
  ;; INPUT:  ONE OF THE ATOMS LISTED IN
  ;;         DT*PREDICATE.COMPONENTS, A PREDICATE AND
  ;;         AN S-EXPRESSION
  ;; EFFECT: VALUE IS PUT INTO THE CORRESPONDING
  ;;         MEMORY CELL.
  ;; VALUE:  UNDEFINED
  (CASE COMPONENT
    (PNAME (MEM-PUT PREDICATE 1 VALUE))
    (POSITIVE.OCCURRENCES (MEM-PUT PREDICATE 2 VALUE))
    (NEGATIVE.OCCURRENCES (MEM-PUT PREDICATE 3 VALUE))
    (DOMAINSORTS (MEM-PUT PREDICATE 4 VALUE))
    (ATTRIBUTES (MEM-PUT PREDICATE 5 VALUE))
    (REFL.CLAUSE (MEM-PUT PREDICATE 6 VALUE))
    (+ROTHERSIDES (MEM-PUT PREDICATE 7 VALUE))
    (-ROTHERSIDES (MEM-PUT PREDICATE 8 VALUE))
    (+SOTHERSIDES (MEM-PUT PREDICATE 9 VALUE))
    (-SOTHERSIDES (MEM-PUT PREDICATE 10 VALUE))
    (+TOTHERSIDES (MEM-PUT PREDICATE 11 VALUE))
    (-TOTHERSIDES (MEM-PUT PREDICATE 12 VALUE))
    (OTHERWISE (ERROR "ILLEGAL COMPONENT IN DT=PREDICATE.PUT: ~A" COMPONENT))))

(DEFMACRO DT-PREDICATE.IS.TRUE (PREDICATE)
 ;; INPUT: A PREDICATE
  ;; VALUE: T, IF PREDICATE IS THE 'TRUE' PREDICATE
  `(EQL ,PREDICATE DT*TRUE.PREDICATE))

(DEFMACRO DT-PREDICATE.IS.FALSE (PREDICATE)
 ;; INPUT: A PREDICATE
  ;; VALUE: T, IF PREDICATE IS THE 'FALSE' PREDICATE
  `(EQL ,PREDICATE DT*FALSE.PREDICATE))

(DEFMACRO DT-PREDICATE.TRUE NIL
 ;; VALUE: THE ADDRESS OF THE 'TRUE' PREDICATE
  'DT*TRUE.PREDICATE)

(DEFMACRO DT-PREDICATE.FALSE NIL
 ;; VALUE: THE ADDRESS OF THE 'FALSE' PREDICATE
  'DT*FALSE.PREDICATE)

(DEFMACRO DT=PREDICATE.STORAGE NIL
  ;; EDITED:  15-NOV-79 01:55:39
  ;; VALUE: POINTER TO A NEW STORAGE UNIT FOR A PREDICATE
  ;; STRUCTURE: CELL    COMPONENT
  ;;               1    PNAME
  ;;               2   POSITIVE OCCURRENCES
  ;;               3   NEGATIVE OCCURRENCES
  ;;               4   DOMAINSORTS
  ;;               5   ATTRIBUTES
  ;;               6   REFL. OR IRREFL.CLAUSE
  ;;               7   +ROTHERSIDES
  ;;               8   -ROTHERSIDES
  ;;              9   +SOTHERSIDES
  ;;              10   -SOTHERSIDES
  ;;              11   +TOTHERSIDES
  ;;              12   -TOTHERSIDES
  `(MEM-NEW 'PREDICATE 12))

(DEFMACRO DT=PREDICATE.CREATE.PNAME NIL
 ;; EDITED:| "19-NOV-79 16:27:55")
 ;; VALUE:| NEW NAME FOR APREDICATE *)
  `(CONCATENATE 'STRING "PRED" (PRINC-TO-STRING (SETQ DT*PREDICATE.COUNTER (1+ DT*PREDICATE.COUNTER)))))

(DEFMACRO DT=PREDICATE.IS (PRED)
 ;; EDITED:| "18-JAN-80 12:35:52")
  ;; VALUE:| T IF PRED IS A PRED SYMBOL NOT YET |DELETED,| ELSE NIL *)
  `(EQL (MEM-TYPE ,PRED) 'PREDICATE))

(DEFUN DT=PREDICATE.INSERT.OTHERSIDES (PREDICATE)
  ;; EDITED: 29-SEP-83 13:00:38
  ;; INPUT:  A NEW PREDICATE
  ;; EFFECT: THE OTHERSIDES CORRESPONDING TO THE
  ;;         DIFFERENT LINK TYPES ARE INSERTED.
  ;; VALUE:  UNDEFINED.
  (DT-PREDICATE.PUT '+ROTHERSIDES PREDICATE (LIST (LIST '- PREDICATE (LIST 'POSITIVE))))
  (DT-PREDICATE.PUT '-ROTHERSIDES PREDICATE (LIST (LIST '+ PREDICATE (LIST 'NEGATIVE))))
  (DT-PREDICATE.PUT '+SOTHERSIDES PREDICATE (LIST (LIST '+ PREDICATE (LIST NIL))))
  (DT-PREDICATE.PUT '-SOTHERSIDES PREDICATE (LIST (LIST '- PREDICATE (LIST NIL))))
  (DT-PREDICATE.PUT '+TOTHERSIDES PREDICATE NIL)
  (DT-PREDICATE.PUT '-TOTHERSIDES PREDICATE NIL))

(DEFUN DT=PREDICATE.OTHERSIDES.ADD.SYMMETRY (PREDICATE)
  ;; EDITED:  30. 9. 1983
  ;; INPUT:   A PREDICATE SYMBOL
  ;; EFFECT:  THE SYMMETRY RULE IS ADDED TO THE
  ;;          LINK OTHERSIDES DESCRIPTION
  ;; VALUE:   UNDEFINED
  (MAPC #'(LAMBDA (ELEMENT)
	    (COND ((AND (EQL PREDICATE (SECOND ELEMENT)) (EQL '- (CAR ELEMENT))) (RPLACD (THIRD ELEMENT) 'SYMMETRIC))))
	(DT-PREDICATE.GET '+ROTHERSIDES PREDICATE))
  (MAPC #'(LAMBDA (ELEMENT)
	    (COND ((AND (EQL PREDICATE (SECOND ELEMENT)) (EQL '+ (CAR ELEMENT))) (RPLACD (THIRD ELEMENT) 'SYMMETRIC))))
	(DT-PREDICATE.GET '-ROTHERSIDES PREDICATE))
  (MAPC #'(LAMBDA (ELEMENT)
	    (COND ((AND (EQL PREDICATE (SECOND ELEMENT)) (EQL '+ (CAR ELEMENT))) (RPLACD (THIRD ELEMENT) 'SYMMETRIC))))
	(DT-PREDICATE.GET '+SOTHERSIDES PREDICATE))
  (MAPC #'(LAMBDA (ELEMENT)
	    (COND ((AND (EQL PREDICATE (SECOND ELEMENT)) (EQL '- (CAR ELEMENT))) (RPLACD (THIRD ELEMENT) 'SYMMETRIC))))
	(DT-PREDICATE.GET '-SOTHERSIDES PREDICATE)))

(DEFUN DT=PREDICATE.OTHERSIDES.ADD.ASYMMETRY (PREDICATE)
  ;; EDITED:  30. 9. 1983
  ;; INPUT:   A PREDICATE SYMBOL
  ;; EFFECT:  THE ASYMMETRY RULE IS ADDED TO THE
  ;;          LINK OTHERSIDES DESCRIPTION
  ;; VALUE:   UNDEFINED
  (PROG ((PLUS (CAR DT*SIGN.PLUS.SYMBOLS)) (MINUS (CAR DT*SIGN.MINUS.SYMBOLS)))
    (DT-PREDICATE.PUT '+ROTHERSIDES PREDICATE
      (CONS (CONS PLUS (CONS PREDICATE '(NIL . ASYMMETRIC))) (DT-PREDICATE.GET '+ROTHERSIDES PREDICATE)))
    (DT-PREDICATE.PUT '+SOTHERSIDES PREDICATE
      (CONS (CONS MINUS (CONS PREDICATE '(POSITIVE . ASYMMETRIC))) (DT-PREDICATE.GET '+SOTHERSIDES PREDICATE)))
    (DT-PREDICATE.PUT '-SOTHERSIDES PREDICATE
      (CONS (CONS PLUS (CONS PREDICATE '(NEGATIVE . ASYMMETRIC))) (DT-PREDICATE.GET '-SOTHERSIDES PREDICATE)))
    (DT-PREDICATE.PUT '-TOTHERSIDES PREDICATE
      (CONS (CONS MINUS (CONS PREDICATE '(NIL . ASYMMETRIC))) (DT-PREDICATE.GET '-TOTHERSIDES PREDICATE)))))


(DEFMACRO DT-TAF.CREATE.FIRST (TAF)
  ;; EDITED: 16-FEB-83 16:39:12
  ;; INPUT:  A TERM ACCESS FUNCTION OR NIL
  ;; VALUE:  A NEW TERM ACCESS FUNCTION DENOTING THE
  ;;         FIRST SUBTERM OF THE TERM DENOTED BY TAF,
  ;;         OR A FCT DENOTING THE FIRST TERM OF A
  ;;         TERMLIST IF THE INPUT IS NIL.
  `(NREVERSE (CONS 1 (REVERSE ,TAF))))

(DEFMACRO DT-TAF.CREATE.SECOND (TAF)
  ;; EDITED: 16-FEB-83 16:39:12
  ;; INPUT:  A TERM ACCESS FUNCTION OR NIL
  ;; VALUE:  A NEW TERM ACCESS FUNCTION DENOTING THE
  ;;         SECOND SUBTERM OF THE TERM DENOTED BY TAF,
  ;;         OR A FCT DENOTING THE SECOND TERM OF A
  ;;         TERMLIST IF THE INPUT IS NIL.
  `(NREVERSE (CONS 2 (NREVERSE ,TAF))))

(DEFUN DT-TAF.CREATE.NEXT (TAF)
  ;; EDITED: 16-FEB-83 16:39:12
  ;; INPUT:  A TERM ACCESS FUNCTION DENOTING A SUBTERM
  ;;         OF A TERMLIST.
  ;; VALUE:  A NEW TERM ACCESS FUNCTION DENOTING THE
  ;;         NEXT SUBTERM OF THE TERMLIST AT THE SAME
  ;;         LEVEL.
  (SETQ TAF (COPY-TREE TAF))
  (incf (CAR (LAST TAF)))
  TAF)

(DEFMACRO DT-TAF.CREATE.LEFT NIL
  ;; EDITED: 16-FEB-83 16:39:12
  ;; INPUT:  NONE
  ;; VALUE:  A TERM ACCESS FUNCTION DENOTING THE FIRST
  ;;         TERM OF A TERMLIST.
  `(LIST 1))

(DEFMACRO DT-TAF.CREATE.RIGHT NIL
  ;; EDITED: 16-FEB-83 16:39:12
  ;; INPUT:  NONE
  ;; VALUE:  A TERM ACCESS FUNCTION DENOTING THE SECOND
  ;;         TERM OF A TERMLIST.
  `(LIST 2))

(DEFMACRO DT-TAF.IS.LEFT (TAF)
  ;; EDITED: 16-FEB-83 16:39:12
  ;; INPUT:  A TERM ACCESS FUNCTION
  ;; VALUE:  T, IF TAF DENOTES THE FIRST  TERM OF A
  ;;                           .
  `(EQUAL ,TAF '(1)))

(DEFMACRO DT-TAF.IS.RIGHT (TAF)
  ;; EDITED: 16-FEB-83 16:39:12
  ;; INPUT:  A TERM ACCESS FUNCTION
  ;; VALUE:  T, IF TAF DENOTES THE SECOND TERM OF A
  ;;         TERMLIST, ELSE NIL.
  `(EQUAL ,TAF '(2)))

#|(DEFMACRO DT-TAF.ACCESS.DEPTH (TAF)
  ;; EDITED: 3. 5. 1982   HJO
  ;; INPUT:  A TERM ACCESS FUNCTION
  ;; VALUE:  THE DEPTH OF THE GIVEN
  ;;         TERM ACCESS FUNCTION RELATIVE TO THE
  ;;         TOPLEVEL. A TOPLEVEL TERM ACCESS FUNCTION
  ;;         HAS ACCESS.DEPTH 0
  `(1- (LIST-LENGTH ,TAF)))|#

(DEFMACRO DT-TAF.TOPLEVEL (TAF)
 ;; EDITED AT 8-SEP-81 |09:08|)
 ;; INPUT:  A TERM ACCESS FUNCTION
  ;; EFFECT: RETURNS VALUE
  ;; VALUE:  TOPLEVEL TERM ACCESS FUNCTION, TO
  ;;         A TERM OF THE LITERAL
  `(LIST (CAR ,TAF)))

(DEFUN DT-TAF.TOPLEVEL.EQUALITY (TAF)
  ;; EDITED:  18. 8. 1982    KHB
  ;; INPUT:   A TERM ACCESS FUNCTION
  ;; VALUE:   THE (LEFT), (RIGHT) REPRESENTATION OF THE
  ;;          TOPLEVEL FUNCTION OF TAF.
  (COND ((EQL 1 (CAR TAF)) (DT-TAF.CREATE.LEFT)) ((EQL 2 (CAR TAF)) (DT-TAF.CREATE.RIGHT))
    (T (ERROR "BAD ACCESS FUNCTION IN DT-TAF.TOPLEVEL.EQUALITY: ~A" TAF))))

#|(DEFMACRO DT-TAF.NEXT.COARSE.STEP (TAF)
  ;; EDITED AT 28-OCT-81 |13:23|)
  ;; INPUT:  A TERM ACCESS FUNCTION
  ;; EFFECT: RETURNS VALUE
  ;; VALUE:  IF THE GIVEN FUNCTION IS A TOPLEVEL
  ;;         TERM ACCESS FUNCTION, THEN THE
  ;;         VALUE IS NIL, ELSE VALUE IS THE
  ;;         NEXT COARSE TERM ACCESS FUNCTION TO
  ;;         THE ONE LEVEL HIGHER TERM.
  `(NREVERSE (CDR (REVERSE ,TAF))))

(DEFUN DT-TAF.SOME.COARSE.STEPS (TAF STEPS)
  ;; EDITED  25. 10. 1982
  ;; INPUT:  A TERM ACCESS FUNCTION (TAF) AND A
  ;;         NATURAL NUMBER (STEPS)
  ;; EFFECT: RETURNS VALUE
  ;; VALUE:  TERM ACCESS FUNCTION OF THE STEP'TH
  ;;         SUPERTERM OF THE TERM GIVEN BY TAF.
  (COND ((< STEPS 1) TAF)
	(T (nreverse (nthcdr steps (reverse taf))))))|#

(DEFMACRO DT-TAF.DIFFERENT.SIDES (TAF1 TAF2)
  ;; EDITED:  11. 08. 1982    KHB
  ;; INPUT:   TWO TERM ACCESS FUNCTIONS
  ;; VALUE:   T, IF THE TERMS DETERMINED BY THE TWO
  ;;          TERM ACCESS FUNCTIONS ARE SUBTERMS OF
  ;;          DIFFERENT TOP-LEVEL TERMS  (ESPECIALLY
  ;;          THE TERM ACCESS FUNCTIONS POINT TO
  ;;          DIFFERENT SIDES OF AN EQUATION), ELSE
  ;;          THE VALUE IS NIL.
  `(NULL (EQL (CAR ,TAF1) (CAR ,TAF2))))

(DEFMACRO DT-TAF.ARE.EQUAL (TAF1 TAF2)
  ;; EDITED:  18.8.1982   KHB
  ;; INPUT:   TWO TERM ACCESS FUNCTIONS
  ;; VALUE:   T, IF THE TWO FUNCTION ARE EQUAL, I.E.
  ;;          DENOTE THE SAME SUBTERM OF A TERM,
  ;;          ELSE NIL.
  `(EQUAL ,TAF1 ,TAF2))

(DEFUN DT-TAF.DEEPER.OR.EQUAL (TAF1 TAF2)
  ;; EDITED: 16-FEB-83 18:10:57
  ;; INPUT:  TWO TERM ACCESS FUNCTIONS (TAF1 TAF2).
  ;;         TAF1 AND TAF2 DENOTE (SUB)TERMS T1 AND T2 OF
  ;;         A TERMLIST.
  ;; VALUE:  T, IF T1 IS EQUAL T2 OR T1 IS A SUBTERM OF
  ;;         T2 (FOR AN ARBITRARY TERMLIST), ELSE NIL
  (EQUAL (firstn TAF1 (LIST-LENGTH TAF2)) TAF2))

(DEFMACRO DT-TAF.COMPOSE.TWO.TAFS (TAF1 TAF2)
  ;; EDITED:  18.8.1982     KHB
  ;; INPUT:   TWO TERM ACCESS FUNCTIONS
  ;; VALUE:   THE COMPOSITION OF THE TWO FUNCTIONS.
  `(APPEND ,TAF1 (copy-list (CDR ,TAF2))))

(DEFUN DT-TAF.OTHERSIDE (TAF)
  ;; EDITED: 11.2.83
  ;; INPUT:  A TERM ACCESS FUNCTION (TAF)
  ;; VALUE:  IF TAF IS A TOPLEVEL FUNCTION ON THE FIRST
  ;;         OR SECOND ELEMENT THEN VALUE IS A TOPLEVEL
  ;;         FUNCTION DENOTING THE OTHER TERM
  ;;         (ESP. THE OTHERSIDE OF AN EQUATION),
  ;;         ELSE THE VALUE IS NIL.
  (COND ((DT-TAF.IS.LEFT TAF) (DT-TAF.CREATE.RIGHT))
	((DT-TAF.IS.RIGHT TAF) (DT-TAF.CREATE.LEFT))
	(T NIL)))

(DEFMACRO DT-TAF.ARGUMENT.POSITIONS (TAF)
  ;; EDITED:  6-AUG-83 10:46:26
  ;; INPUT:   A TERM ACCESS FUNCTION
  ;; VALUE:   THE ARGUMENT POSITIONS ACCESSED BY TAF
  `(COPY-TREE ,TAF))

(DEFMACRO DT-TAF.DEEPEST.ARGUMENT.NUMBER (TAF)
  ;; EDITED:  3-AUG-83 14:39:02
  ;; INPUT:   A TAF
  ;; VALUE:   THE NUMBER OF THE ARGUMENT POSITION OF THE
  ;;          TERM ACCESSED BY TAF WITHIN THE NEXT
  ;;          SUPERTERM.
  `(CAR (LAST ,TAF)))

(DEFVAR DT*UNI.CREATES.VARIABLES NIL)

(DEFVAR DT*SIGN.MINUS.SYMBOLS '(- --))

(DEFVAR DT*SIGN.PLUS.SYMBOLS '(+ ++))

(DEFMACRO DT-GETPROP (ADR IND)
  ;; EDITED: 28-OCT-83 17:11:53
  ;; INPUT:  A STORAGE ADDRESS AND A PROPERTY INDICATOR.
  ;; VALUE:  THE PROPERTY BELONGING TO THIS INDICATOR
  ;;         (LIKE LISP FUNCTION GETPROP)
  `(MEM-GETPROP ,ADR ,IND))

(DEFMACRO DT-PUTPROP (ADR IND PROPERTY)
  ;; EDITED: 28-OCT-83 17:17:39
  ;; INPUT:  A STORAGE ADDRESS,A PROPERTY INDICATOR
  ;;         AND AN S-EXPRESSION
  ;; EFFECT: LIKE LISP FUNCTION PUTPROP
  ;; VALUE:  UNDEFINED.
  `(MEM-PUTPROP ,ADR ,IND ,PROPERTY))

(DEFMACRO DT-GETPROPLIST (ADR)
  ;; EDITED: 28-OCT-83 16:57:56
  ;; INPUT:  A STORAGE ADDRESS
  ;; VALUE:  THE PROPERTY LIST (LIKE LISP FUNCTION
  ;;                            GETPROPLIST)
  `(MEM-GETPROPLIST ,ADR))

(DEFMACRO DT-SETPROPLIST (ADR PROPLIST)
  ;; EDITED: 28-OCT-83 17:02:33
  ;; INPUT:  A STORAGE ADDRESS AND A PROPERTY LIST
  ;; EFFECT: THE PROPERTY LIST IS STORED INTO
  ;;         THE PROPERTY CELL OF ADR
  ;;         (LIKE LISP FUNCTION SETPROPLIST)
  ;; VALUE:  UNDEFINED.
  `(MEM-SETPROPLIST ,ADR ,PROPLIST))

(DEFMACRO DT-REMPROP (ADR IND)
  ;; EDITED: 28-OCT-83 17:29:41
  ;; INPUT:  A STORAGE ADDRESS AND A PROPERTY INDICATOR
  ;; EFFECT: THE PROPERTY IS REMOVED FROM ADR'S
  ;;         PROPERTYLIST.
  ;;         (LIKE LISP FUNCTION REMPROP)
  ;; VALUE:  IND IF SUCH A PROPERTY EXISTS, ELSE NIL
  `(MEM-REMPROP ,ADR ,IND))

(DEFMACRO DT-REMPROPS (ADR INDS)
  ;; EDITED: 28-OCT-83 17:43:41
  ;; INPUT:  A STORAGE ADDRESS AND NIL OR A LIST
  ;;         OF PROPERTY INDICATORS.
  ;; EFFECT: IF IND = NIL, ALL PROPERTYS ARE
  ;;         REMOVED FROM THE PROPERTY LIST,
  ;;         ELSE THE PROPERTIES IN INDS ARE REMOVED
  ;;         (LIKE SERVICE FUNCTION REMPROPS).
  ;; VALUE:  UNDEFINED.
  `(MEM-REMPROPS ,ADR ,INDS))

(DEFMACRO DT-ADDPROP (ADR IND NEW FRONTFLAG)
  ;; EDITED: 28-OCT-83 17:58:52
  ;; INPUT:  A STRORAGE ADDRESS, A PROPERTY INDICATOR,
  ;;         AN S-EXPRESSION AND A BOOLEAN VALUE.
  ;; EFFECT: THE PROPERTY IND IS ASSUMED TO BE NIL
  ;;         OR A LIST.
  ;;         THE S-EXPRESSION NEW IS ADDED AT
  ;;         THE FRONT OF THE LIST (FRONTFLAG = T)
  ;;         OR AT THE END OF THE LIST (FRONTFLAG = NIL)
  ;;         (LIKE LISP FUNCTION ADDPROP.
  ;; VALUE:  UNDEFINED
  `(MEM-ADDPROP ,ADR ,IND ,NEW ,FRONTFLAG))

(DEFUN DT-RESET NIL
  ;; EDITED: 1-FEB-82 13:54:40
  ;; VALUE:  UNDEFINED
  ;; EFFECT: DELETES ALL VARIABLE, CONSTANT, FUNCTION
  ;;         AND PREDICATE SMBOLS AND SETS THE SYSTEM
  ;;         TO AN INITIAL STATE.
  (SETQ DT*UNI.CREATES.VARIABLES NIL)
  (SETQ DT*VARIABLE.BUFFER (BUFFER.CREATE 10))
  (SETQ DT*VARIABLE.COUNTER 0)
  (SETQ DT*CONSTANT.COUNTER 0)
  (SETQ DT*FUNCTION.COUNTER 0)
  (SETQ DT*FUNCTION.WITH.ARGUMENT.SYMMETRIES NIL)
  (SETQ DT*TRUE.PREDICATE NIL)
  (SETQ DT*FALSE.PREDICATE NIL)
  (SETQ DT*PREDICATE.COUNTER 0)
  (SETF (SYMBOL-PLIST 'DT*PREDICATE.WITH.ATTRIBUTES) NIL)
  (SETQ DT*EQUALITY.PREDICATES nil)
  (SETQ DT*NONEQUALITY.PREDICATES NIL)
  (SETQ DT*PREDICATE.ALL NIL)
  (SETQ DT*FUNCTION.ALL NIL)
  (SETQ DT*FUNCTION.ACTUAL.THEORIES NIL)
  (SETQ DT*CONSTANT.ALL NIL)
  (SETQ DT*SORT.NR 1)
  (DT=SORT.DELETE.ALL)
  (MAPC #'(LAMBDA (SCHEME)
		    (SETF (SYMBOL-VALUE SCHEME) NIL)
		    (SETF (SYMBOL-PLIST SCHEME) NIL))
	DT*ABBREVIATIONS)
  (SETQ DT*ABBREVIATIONS NIL)
  (MEM-RESET))

(DEFUN DT-ACCESS (TAF TERMLIST)
						; EDITED: "12-FEB-80 11:46:07"
						; VALUE:  SUBTERM OF TERMLIST ACCESSED BY FCT, OR NIL IF NO SUCH SUBTERM EXISTS 
  (if TAF
      (let (TERM)
	(SETQ TERM (nth (1- (CAR TAF)) TERMLIST))
	(MAPC #'(LAMBDA (DSS) (SETQ TERM (if (CONSP TERM) (NTH DSS TERM) NIL)))
	      (CDR TAF))
	TERM)
      NIL))

(DEFMACRO DT-TYPE (OBJECT)
						; EDITED: "28-NOV-80 14:30:43")
						; VALUE: EITHER OF THE ATOMS VARIABLE, CONSTANT, FUNCTION 
						;        OR PREDICATE. IF OBJECT IS NONE OF THESE OR HAS BEEN DELETED, NIL 
  `(MEM-TYPE ,OBJECT))

(DEFUN DT-PNAME (OBJECT)
  ;; EDITED: 14-MAR-83 12:08:23        NE
  ;; INPUT:  ARBITRARY S-EXPRESSION.
  ;; VALUE:  SAME EXPRESSION, WHERE ALL DT-OBJECTS ARE
  ;;         REPLACED BY THEIR PNAMES.
  (if (CONSP OBJECT)
      (CONS (DT-PNAME (CAR OBJECT)) (DT-PNAME (CDR OBJECT)))
      (CASE
	(DT-TYPE OBJECT)
	(PREDICATE (DT-PREDICATE.PNAME OBJECT))
	(FUNCTION (DT-FUNCTION.PNAME OBJECT))
	(CONSTANT (DT-CONSTANT.PNAME OBJECT))
	(VARIABLE (DT=VARIABLE.GETPNAME OBJECT))
	(OTHERWISE OBJECT))))

(DEFUN DT-REPLACE.TERM.IN.TERMLIST (TERM TAF TERMLIST)
  ;; EDITED: 17-FEB-83 09:59:59
  ;; INPUT:  A TERM, A TERM ACCESS FUNCTION (TAF) AND A
  ;;         TERMLIST.
  ;; VALUE:  IF TAF IS NOT NIL AND TAF DENOTES A SUBTERM
  ;;         IN TERMLIST, THEN THE INPUT TERMLIST, BUT
  ;;         THE SUBTERM DENOTED BY TAF REPLACED BY TERM,
  ;;         ELSE NIL.
  (COND
    (TAF
      (PROG (SUBTERM) (SETQ SUBTERM (NTHCDR (1- (CAR TAF)) TERMLIST))
        (MAPC #'(LAMBDA (DSS) (SETQ SUBTERM (NTHCDR DSS (CAR SUBTERM)))) (CDR TAF))
        (COND (SUBTERM (RPLACA SUBTERM TERM) (RETURN TERMLIST)) (T (RETURN NIL)))))
    (T NIL)))


(DEFUN DT-UPDATE.STRANGE.COMMONS (COMMON)
  ; INPUT: A COMMON-NAME OR 'ALL
  ; EFFECT: THE ACTUAL VALUE OF THE COMMON VARIABLE
  ;         IS COMPUTED.
  (COND
    ((MEMBER COMMON '(ALL DT*FUNCTION.ACTUAL.THEORIES))
     (MAPC #'(LAMBDA (FUNCTION)
	       (SETQ DT*FUNCTION.ACTUAL.THEORIES
		     (UNION (SYMBOL-VALUE 'DT*FUNCTION.ACTUAL.THEORIES) (DT=FUNCTION.GET 'ATTRIBUTES FUNCTION))))
	   DT*FUNCTION.ALL)))
  (COND ((MEMBER COMMON '(ALL DT*UNI.CREATES.VARIABLES)) NIL)))

(DEFMACRO DT-APPLY.TO.ALL.ADDR (FUNCTION)
 ; INPUT: A LAMBDA EXPRESSION
  ;  VALUE:UNDEFINED
  ; EFFECT: FUNCTION IS APPLIED TO ALL ADDRESSES
  `(DODOWN (RPTN (MEM-ALL.ADR)) (funcall ,FUNCTION (1+ RPTN))))

(DEFMACRO DT-UNI.CREATES.VARIABLES NIL
  ; EDITED: 27. 2. 1984
  ; VALUE:  T IF THE UNIFICATION ALGORITHM MAY
  ;         CREATE NEW VARIABLES
  ;         (ASSOCIATIVITY, POLYMORPHIC FUNCTIONS,
  ;          RULES ETC.), ELSE NIL
  `DT*UNI.CREATES.VARIABLES)



(DEFUN DT-SAVE (FILE)
						; INPUT:  A FILENAME OR NIL
						; VALUE:  IF FILE = NIL, AN S-EXPRESSION WHICH
						;         IF IT IS EVALUATED RESTORES THE VERY SAME
						;         DS-SYSTEM-STATE AS IT WAS WHEN DS-SAVE
						;         WAS CALLED
						;         ELSE NIL
						; EFFECT: IF FILE =/= NIL THE S-EXPRESSION IS
						;         WRITTEN ON FILE
						;         ELSE RETURNS VALUE
						; REMARK: THE FILE IS EXPECTED TO BE OPEN AND
						;         REMAINS SO
  (PROG
    ((EXPRESSION.COMMONVARS (CONS 'PROGN (MAPCAR #'(LAMBDA (VAR) `(setq ,var ',(symbol-value var)))
						 dt*commons.to.save)))
     (EXPRESSION.OTHERVARS (CONS 'PROGN (MAPCAR #'(LAMBDA (ATOM) `(setq ,atom ',(symbol-value atom)))
						DT*ABBREVIATIONS)))
     (EXPRESSION.COMMON.PROPLISTS (CONS 'PROGN (MAPCAR #'(LAMBDA (ATOM) (SAVE-PROPLIST ATOM '(dt*)))
						       '(DT*PREDICATE.WITH.ATTRIBUTES))))
     (EXPRESSION.OTHER.PROPLISTS (CONS 'PROGN (MAPCAR #'(LAMBDA (ATOM) (SAVE-PROPLIST ATOM '(dt*)))
						      (APPEND DT*SORT.ALL DT*ABBREVIATIONS)))))
    (COND
      ((NULL FILE)
       (RETURN
	 (LIST 'PROGN '(DT-RESET) (MEM-SAVE NIL)
	       EXPRESSION.COMMONVARS
	       EXPRESSION.OTHERVARS
	       EXPRESSION.COMMON.PROPLISTS
	       EXPRESSION.OTHER.PROPLISTS)))
      (T (PROGN (PRINC (LIST 'PROGN '(DT-RESET)) FILE)
		(TERPRI FILE))
	 (MEM-SAVE FILE)
	 (PRINT EXPRESSION.COMMONVARS FILE)
	 (PRINT eXPRESSION.OTHERVARS FILE)
	 (PRINT EXPRESSION.COMMON.PROPLISTS FILE)
	 (PRINT EXPRESSION.OTHER.PROPLISTS FILE)))))

(DEFUN DT-SAVE.SYMBOLS (FILE &OPTIONAL (INDENTATION 0))
						; edited: 28-aug-84 14:00:57  by cl
						; input : the code file (open for output)
						; effect: prints all constants, functions, and predi-
						;         cates to the code file in form of a list
						;         which, when evaluted, reinstalls these ob-
						;         jects in the 'dt' format.
						;         value of this evaluation is a list of dotted
						;         pairs ( old.address . new.address ).
						; value : undefined
  
  (FORMAT FILE "~&~vT(LET (NEW.ADDRESS)~%~10T(LIST "
	  INDENTATION (SETQ INDENTATION (+ INDENTATION 8)))
  
  (WHEN (DT-PREDICATE.ALL)			; predicates, they are first because they always exist 
    (LET ((PREDICATE (FIRST (DT-PREDICATE.ALL))))
      (IF (DT-PREDICATE.ATTRIBUTES PREDICATE)
	  (FORMAT FILE "(PROG1 (CONS ~4D (SETQ NEW.ADDRESS (DT-PREDICATE.CREATE \"~A\" '~A)))~%~
                                      ~vT(DT-PREDICATE.ADD.ATTRIBUTES NEW.ADDRESS '~A))"
		  PREDICATE (DT-PREDICATE.PNAME PREDICATE) (DT-PREDICATE.DOMAINSORTS PREDICATE)
		  (+ INDENTATION 7) (DT-PREDICATE.ATTRIBUTES PREDICATE))
	  (FORMAT FILE "(CONS ~4D (DT-PREDICATE.CREATE \"~A\" '~A))"
		  PREDICATE (DT-PREDICATE.PNAME PREDICATE) (DT-PREDICATE.DOMAINSORTS PREDICATE))))    
    (MAPC #'(LAMBDA (PREDICATE)
	      (IF (DT-PREDICATE.ATTRIBUTES PREDICATE)
		  (FORMAT FILE "~%~vT(PROG1 (CONS ~4D (SETQ NEW.ADDRESS (DT-PREDICATE.CREATE \"~A\" '~A)))~%~
                                      ~vT(DT-PREDICATE.ADD.ATTRIBUTES NEW.ADDRESS '~A))"
			  INDENTATION PREDICATE (DT-PREDICATE.PNAME PREDICATE) (DT-PREDICATE.DOMAINSORTS PREDICATE)
			  (+ INDENTATION 7) (DT-PREDICATE.ATTRIBUTES PREDICATE))
		  (FORMAT FILE "~%~vT(CONS ~4D (DT-PREDICATE.CREATE \"~A\" '~A))"
			  INDENTATION PREDICATE (DT-PREDICATE.PNAME PREDICATE) (DT-PREDICATE.DOMAINSORTS PREDICATE))))
	  (REST (DT-PREDICATE.ALL))))
						; constants
  (MAPC #'(LAMBDA (CONSTANT)
	    (FORMAT FILE "~%~vT(CONS ~4D (DT-CONSTANT.CREATE \"~(~A~)\" '~A))"
		    INDENTATION CONSTANT (DT-CONSTANT.PNAME CONSTANT) (DT-CONSTANT.SORT CONSTANT)))
	(DT-CONSTANT.ALL))
						; functions
  (MAPC #'(LAMBDA (FUNCTION)
	    (IF (DT-FUNCTION.ATTRIBUTES FUNCTION)
		(FORMAT FILE "~%~vT(PROG1 (CONS ~4D (SETQ NEW.ADDRESS (DT-FUNCTION.CREATE \"~(~A~)\" '~A '~A '~A)))~%~
                                       ~vT(DT-FUNCTION.ADD.ATTRIBUTES NEW.ADDRESS '~A))"
			INDENTATION FUNCTION (DT-FUNCTION.PNAME FUNCTION) (DT-FUNCTION.MAX.RANGE.SORT FUNCTION)
			(DT-FUNCTION.DOMAINSORTS FUNCTION) (DT-FUNCTION.SORTLIST FUNCTION)
			(+ INDENTATION 7) (DT-FUNCTION.ATTRIBUTES FUNCTION))
		(FORMAT FILE "~%~vT(CONS ~4D (DT-FUNCTION.CREATE \"~(~A~)\" '~A '~A '~A))"
			INDENTATION FUNCTION (DT-FUNCTION.PNAME FUNCTION) (DT-FUNCTION.MAX.RANGE.SORT FUNCTION)
			(DT-FUNCTION.DOMAINSORTS FUNCTION) (DT-FUNCTION.SORTLIST FUNCTION))))
	(DT-FUNCTION.ALL))
  
  (FORMAT FILE "))" ))

(DEFUN DT-PRINT.SYMBOLS (KINDS FILE LEFT.MARGIN RIGHT.MARGIN)
  ; EDITED: 18-OCT-84 13:50:55           BY CL
  ; INPUT : A LIST OF SYMBOL KINDS (SUBSET OF DT*SYMBOL.
  ;         KINDS) OR 'ALL (EQV NIL), THE LATTER MEANING
  ;         ALL SYMBOL KINDS (= DT*SYMBOL.KINDS),A FILE
  ;         NAME (OPEN FOR OUTPUT), AND TWO INTEGERS DE-
  ;         NOTING THE LEFTMOST AND RIGHTMOST COLOUMN TO
  ;         TO BE PRINTED IN, RESPECTIVELY.
  ; DEFAULT VALUES: KINDS        : DT*SYMBOL.KINDS
  ;                 FILE         : ACTUAL OUTPUT FILE
  ;                 LEFT MARGIN  :        1
  ;                 RIGHT MARGIN :       77
  ; EFFECT: PRINTS TABLES FOR THE GIVEN SYMBOL KINDS IN
  ;         A READABLE FORMAT.
  ; VALUE : UNDEFINED
  (PROG
    (SORT.ALL CONSTANT.ALL FUNCTION.ALL PREDICATE.ALL (MAX.SORT.NAME.LENGTH 0) (MAX.NAME.LENGTH 0) (MAX.DOMAIN.LENGTH 0)
     (MAX.RANGE.LENGTH 0) (MAX.ATTRIBUTES.LENGTH 0) (MAX.SINGLE.ATTRIBUTE.LENGTH 0) DOMAIN.TAB RANGE.TAB ATTRIBUTES.TAB
     RIGHT.TAB (TAB.SIGN "!"))
    (PROGN ; PREPARATION & SETTING OF DEFAULT VALUES FOR INPUT
      (COND ((OR (NULL KINDS) (EQL KINDS 'ALL)) (SETQ KINDS DT*SYMBOL.KINDS)))
      (MAPC
        #'(LAMBDA (KIND) (SET KIND (EVAL (INTERN (format nil "DT*~A" KIND) (find-package "MKRP")))))
        KINDS)
      (COND ((EQUAL SORT.ALL '(ANY)) (SETQ SORT.ALL NIL)))
      (SETQ LEFT.MARGIN (OR LEFT.MARGIN 1)) (SETQ RIGHT.MARGIN (OR RIGHT.MARGIN 77)))
    (PROGN ; CALCULATION OF MAXIMAL LENGTHS OF ENTRIES IN COLUMNS
      (COND ((NULL SORT.ALL)
	     (MAPC #'(LAMBDA (SORT)
		       (COND ((> (PRINT-LENGTH (DT-PNAME SORT) NIL) MAX.SORT.NAME.LENGTH)
			      (SETQ MAX.SORT.NAME.LENGTH (PRINT-LENGTH (DT-PNAME SORT) NIL)))))
		   DT*SORT.ALL))
	    (T (MAPC #'(LAMBDA (SORT)
			 (COND ((> (PRINT-LENGTH (DT-PNAME SORT) NIL) MAX.NAME.LENGTH)
				(SETQ MAX.NAME.LENGTH (PRINT-LENGTH (DT-PNAME SORT) NIL))))
			 (COND ((> (PRINT-LENGTH (DT-SORT.DIRECT.SUBSORTS SORT) NIL) MAX.DOMAIN.LENGTH)
				(SETQ MAX.DOMAIN.LENGTH (PRINT-LENGTH (DT-SORT.DIRECT.SUBSORTS SORT) NIL))))
			 (COND ((> (PRINT-LENGTH (DT-SORT.DIRECT.SUPERSORTS SORT) NIL) MAX.RANGE.LENGTH)
				(SETQ MAX.RANGE.LENGTH (PRINT-LENGTH (DT-SORT.DIRECT.SUPERSORTS SORT) NIL)))))
		     SORT.ALL)
	       (SETQ MAX.SORT.NAME.LENGTH MAX.NAME.LENGTH)))
      (MAPC #'(LAMBDA (CONSTANT)
		(COND ((> (PRINT-LENGTH (DT-CONSTANT.PNAME CONSTANT) NIL) MAX.NAME.LENGTH)
		       (SETQ MAX.NAME.LENGTH (PRINT-LENGTH (DT-CONSTANT.PNAME CONSTANT) NIL))))
		(COND ((> (PRINT-LENGTH (DT-CONSTANT.SORT CONSTANT) NIL) MAX.DOMAIN.LENGTH)
		       (SETQ MAX.DOMAIN.LENGTH (PRINT-LENGTH (DT-CONSTANT.SORT CONSTANT) NIL)))))
	    CONSTANT.ALL)
      (MAPC #'(LAMBDA (FUNCTION)
		(COND ((> (PRINT-LENGTH (DT-FUNCTION.PNAME FUNCTION) NIL) MAX.NAME.LENGTH)
		       (SETQ MAX.NAME.LENGTH (PRINT-LENGTH (DT-FUNCTION.PNAME FUNCTION) NIL))))
		(COND ((> (PRINT-LENGTH (DT-FUNCTION.DOMAINSORTS FUNCTION) NIL) MAX.DOMAIN.LENGTH)
		       (SETQ MAX.DOMAIN.LENGTH (PRINT-LENGTH (DT-FUNCTION.DOMAINSORTS FUNCTION) NIL))))
		(COND ((> (PRINT-LENGTH (DT-FUNCTION.MAX.RANGE.SORT FUNCTION) NIL) MAX.RANGE.LENGTH)
		       (SETQ MAX.RANGE.LENGTH (PRINT-LENGTH (DT-FUNCTION.MAX.RANGE.SORT FUNCTION) NIL))))
		(PROG ((ATTRIBUTES.LENGTH 0))
		      (MAPC #'(LAMBDA (ATTRIBUTE)
				(PROG ((ATTRIBUTE.LENGTH (PRINT-LENGTH ATTRIBUTE NIL)))
				      (SETQ ATTRIBUTES.LENGTH (+ ATTRIBUTES.LENGTH 1 ATTRIBUTE.LENGTH))
				      (COND ((> ATTRIBUTE.LENGTH MAX.SINGLE.ATTRIBUTE.LENGTH)
					     (SETQ MAX.SINGLE.ATTRIBUTE.LENGTH ATTRIBUTE.LENGTH)))))
			    (DT-FUNCTION.ATTRIBUTES FUNCTION))
		      (COND ((> ATTRIBUTES.LENGTH MAX.ATTRIBUTES.LENGTH) (SETQ MAX.ATTRIBUTES.LENGTH ATTRIBUTES.LENGTH)))))
	    FUNCTION.ALL)
      (MAPC #'(LAMBDA (PREDICATE)
		(COND ((> (PRINT-LENGTH (DT-PREDICATE.PNAME PREDICATE) NIL) MAX.NAME.LENGTH)
		       (SETQ MAX.NAME.LENGTH (PRINT-LENGTH (DT-PREDICATE.PNAME PREDICATE) NIL))))
		(COND ((> (PRINT-LENGTH (DT-PREDICATE.DOMAINSORTS PREDICATE) NIL) MAX.DOMAIN.LENGTH)
		       (SETQ MAX.DOMAIN.LENGTH (PRINT-LENGTH (DT-PREDICATE.DOMAINSORTS PREDICATE) NIL))))
		(PROG ((ATTRIBUTES.LENGTH 0))
		      (MAPC #'(LAMBDA (ATTRIBUTE)
				(PROG ((ATTRIBUTE.LENGTH (PRINT-LENGTH ATTRIBUTE NIL)))
				      (SETQ ATTRIBUTES.LENGTH (+ ATTRIBUTES.LENGTH 1 ATTRIBUTE.LENGTH))
				      (COND ((> ATTRIBUTE.LENGTH MAX.SINGLE.ATTRIBUTE.LENGTH)
					     (SETQ MAX.SINGLE.ATTRIBUTE.LENGTH ATTRIBUTE.LENGTH)))))
			    (DT-PREDICATE.ATTRIBUTES PREDICATE))
		      (COND ((> ATTRIBUTES.LENGTH MAX.ATTRIBUTES.LENGTH) (SETQ MAX.ATTRIBUTES.LENGTH ATTRIBUTES.LENGTH)))))
	    PREDICATE.ALL)
      (SETQ MAX.NAME.LENGTH (MAX (+ (1+ MAX.NAME.LENGTH) (PRINT-LENGTH TAB.SIGN NIL)) 6))
      (SETQ MAX.DOMAIN.LENGTH (MAX (+ (1+ MAX.DOMAIN.LENGTH) (PRINT-LENGTH TAB.SIGN NIL)) 10))
      (SETQ MAX.RANGE.LENGTH (MAX (+ (1+ MAX.RANGE.LENGTH) (PRINT-LENGTH TAB.SIGN NIL)) 12))
      (SETQ MAX.ATTRIBUTES.LENGTH (MAX (+ (1+ MAX.ATTRIBUTES.LENGTH) (PRINT-LENGTH TAB.SIGN NIL)) 12)))
    (PROGN ; CALCULATION OF TABULATOR POSIITONS
      (SETQ DOMAIN.TAB (+ LEFT.MARGIN 1 MAX.NAME.LENGTH))
      (SETQ RANGE.TAB (+ DOMAIN.TAB 1 MAX.DOMAIN.LENGTH)) (SETQ ATTRIBUTES.TAB (+ RANGE.TAB 1 MAX.RANGE.LENGTH))
      (SETQ RIGHT.TAB (+ ATTRIBUTES.TAB 1 MAX.ATTRIBUTES.LENGTH))
      (COND ((> RIGHT.TAB RIGHT.MARGIN)
	     ; I.E. IT DOESN'T FIT IN ONE LINE LIKE THIS.
	     (SETQ RIGHT.TAB (+ ATTRIBUTES.TAB 1 MAX.SINGLE.ATTRIBUTE.LENGTH))
	     (COND ((> RIGHT.TAB RIGHT.MARGIN)
		    ; I.E. IT STILL DOESN'T FIT
		    (SETQ RIGHT.TAB RIGHT.MARGIN) (SETQ ATTRIBUTES.TAB (- RIGHT.TAB (1+ MAX.SINGLE.ATTRIBUTE.LENGTH)))
		    (SETQ RANGE.TAB (- ATTRIBUTES.TAB (1+ MAX.SORT.NAME.LENGTH)))
		    (COND ((> (- RANGE.TAB DOMAIN.TAB) (1+ MAX.SORT.NAME.LENGTH))
			   ;     NOW EVERYTHING FITS IN ]
			   )
			  (T (ERROR "DT-PRINT.SYMBOLS : LINE TOO SHORT~A" NIL))))
		   (T (SETQ RIGHT.TAB RIGHT.MARGIN))))))
    (PROGN ; ACTUAL PRINTING OF THE SYMBOL TABLE :
      (TERPRI FILE)
      (COND
        (SORT.ALL (DODOWN (RPTN 3) (TERPRI FILE)) (FORMAT FILE "~T" LEFT.MARGIN) (PRINC "SORTS : " FILE) (FORMAT FILE "~T" LEFT.MARGIN)
		  (PRINC "======= " FILE) (TERPRI FILE) (TERPRI FILE)
		  (PRINTTAB
		    (LIST LEFT.MARGIN TAB.SIGN (LIST " NAME ") DOMAIN.TAB TAB.SIGN (LIST " SUBSORTS ") RANGE.TAB TAB.SIGN
			  (LIST " SUPERSORTS ") ATTRIBUTES.TAB TAB.SIGN)
		    FILE T)
		  (FORMAT FILE "~T" LEFT.MARGIN) (DODOWN (RPTN (- (1+ RIGHT.TAB) LEFT.MARGIN)) (PRINC "-" FILE))
		  (MAPC
		    #'(LAMBDA (SORT)
			(PRINTTAB
			  (LIST LEFT.MARGIN (CONCATENATE 'STRING (PRINC-TO-STRING TAB.SIGN) (PRINC-TO-STRING " ")) (LIST (DT-PNAME SORT))
				DOMAIN.TAB (CONCATENATE 'STRING (PRINC-TO-STRING TAB.SIGN) (PRINC-TO-STRING " ")) (DT-SORT.DIRECT.SUBSORTS SORT)
				RANGE.TAB (CONCATENATE 'STRING (PRINC-TO-STRING TAB.SIGN) (PRINC-TO-STRING " "))
				(DT-SORT.DIRECT.SUPERSORTS SORT) ATTRIBUTES.TAB TAB.SIGN)
			  FILE T))
		    SORT.ALL)
		  (FORMAT FILE "~T" LEFT.MARGIN) (DODOWN (RPTN (- (1+ RIGHT.TAB) LEFT.MARGIN)) (PRINC "-" FILE)) (TERPRI FILE)))
      (COND
        (CONSTANT.ALL (DODOWN (RPTN 3) (TERPRI FILE)) (FORMAT FILE "~T" LEFT.MARGIN) (PRINC "CONSTANTS : " FILE)
		      (FORMAT FILE "~T" LEFT.MARGIN) (PRINC "=========== " FILE) (TERPRI FILE) (TERPRI FILE)
		      (PRINTTAB
			(LIST LEFT.MARGIN TAB.SIGN (LIST " NAME ") DOMAIN.TAB TAB.SIGN (LIST " SORT ") RANGE.TAB TAB.SIGN (LIST " ")
			      ATTRIBUTES.TAB TAB.SIGN (LIST " ATTRIBUTES ") RIGHT.TAB TAB.SIGN)
			FILE T)
		      (FORMAT FILE "~T" LEFT.MARGIN) (DODOWN (RPTN (- (1+ RIGHT.TAB) LEFT.MARGIN)) (PRINC "-" FILE))
		      (MAPC
			#'(LAMBDA (CONSTANT)
			    (PRINTTAB
			      (LIST LEFT.MARGIN (CONCATENATE 'STRING (PRINC-TO-STRING TAB.SIGN) (PRINC-TO-STRING " "))
				    (LIST (DT-CONSTANT.PNAME CONSTANT)) DOMAIN.TAB
				    (format nil "~A ~A" TAB.SIGN (DT-CONSTANT.SORT CONSTANT)) RANGE.TAB
				    (format nil "~A " TAB.SIGN) (LIST " ") ATTRIBUTES.TAB
				    (format nil "~A " TAB.SIGN) (LIST " ") RIGHT.TAB TAB.SIGN)
			      FILE T))
			CONSTANT.ALL)
		      (FORMAT FILE "~T" LEFT.MARGIN) (DODOWN (RPTN (- (1+ RIGHT.TAB) LEFT.MARGIN)) (PRINC "-" FILE)) (TERPRI FILE)))
      (COND
        (FUNCTION.ALL (DODOWN (RPTN 3) (TERPRI FILE)) (FORMAT FILE "~T" LEFT.MARGIN) (PRINC "FUNCTIONS : " FILE)
		      (FORMAT FILE "~T" LEFT.MARGIN) (PRINC "=========== " FILE) (TERPRI FILE) (TERPRI FILE)
		      (PRINTTAB
			(LIST LEFT.MARGIN TAB.SIGN (LIST " NAME ") DOMAIN.TAB TAB.SIGN (LIST " DOMAIN ") RANGE.TAB TAB.SIGN
			      (LIST " RANGE ") ATTRIBUTES.TAB TAB.SIGN (LIST " ATTRIBUTES ") RIGHT.TAB TAB.SIGN)
			FILE T)
		      (FORMAT FILE "~T" LEFT.MARGIN) (DODOWN (RPTN (- (1+ RIGHT.TAB) LEFT.MARGIN)) (PRINC "-" FILE))
		      (MAPC
			#'(LAMBDA (FUNCTION)
			    (PRINTTAB
			      (LIST LEFT.MARGIN (CONCATENATE 'STRING (PRINC-TO-STRING TAB.SIGN) (PRINC-TO-STRING " "))
				    (LIST (DT-FUNCTION.PNAME FUNCTION)) DOMAIN.TAB
				    (CONCATENATE 'STRING (PRINC-TO-STRING TAB.SIGN) (PRINC-TO-STRING " "))
				    (DT-FUNCTION.DOMAINSORTS FUNCTION) RANGE.TAB
				    (CONCATENATE 'STRING (PRINC-TO-STRING TAB.SIGN) (PRINC-TO-STRING " "))
				    (LIST (DT-FUNCTION.MAX.RANGE.SORT FUNCTION)) ATTRIBUTES.TAB
				    (CONCATENATE 'STRING (PRINC-TO-STRING TAB.SIGN) (PRINC-TO-STRING " "))
				    (DT-FUNCTION.ATTRIBUTES FUNCTION) RIGHT.TAB
				    TAB.SIGN)
			      FILE T))
			FUNCTION.ALL)
		      (FORMAT FILE "~T" LEFT.MARGIN) (DODOWN (RPTN (- (1+ RIGHT.TAB) LEFT.MARGIN)) (PRINC "-" FILE)) (TERPRI FILE)))
      (COND
        (PREDICATE.ALL (DODOWN (RPTN 3) (TERPRI FILE)) (FORMAT FILE "~T" LEFT.MARGIN) (PRINC "PREDICATES : " FILE)
		       (FORMAT FILE "~T" LEFT.MARGIN) (PRINC "============ " FILE) (TERPRI FILE) (TERPRI FILE)
		       (PRINTTAB
			 (LIST LEFT.MARGIN TAB.SIGN (LIST " NAME ") DOMAIN.TAB TAB.SIGN (LIST " DOMAIN ")
			       RANGE.TAB TAB.SIGN (LIST " ")
			       ATTRIBUTES.TAB TAB.SIGN (LIST " ATTRIBUTES ") RIGHT.TAB TAB.SIGN)
			 FILE T)
		       (FORMAT FILE "~T" LEFT.MARGIN) (DODOWN (RPTN (- (1+ RIGHT.TAB) LEFT.MARGIN)) (PRINC "-" FILE))
		       (MAPC
			 #'(LAMBDA (PREDICATE)
			     (PRINTTAB
			       (LIST LEFT.MARGIN (CONCATENATE 'STRING (PRINC-TO-STRING TAB.SIGN) (PRINC-TO-STRING " "))
				     (LIST (DT-PREDICATE.PNAME PREDICATE)) DOMAIN.TAB
				     (CONCATENATE 'STRING (PRINC-TO-STRING TAB.SIGN) (PRINC-TO-STRING " "))
				     (DT-PREDICATE.DOMAINSORTS PREDICATE)
				     RANGE.TAB (CONCATENATE 'STRING (PRINC-TO-STRING TAB.SIGN) (PRINC-TO-STRING " "))
				     (LIST " ") ATTRIBUTES.TAB
				     (CONCATENATE 'STRING (PRINC-TO-STRING TAB.SIGN) (PRINC-TO-STRING " "))
				     (DT-PREDICATE.ATTRIBUTES PREDICATE)
				     RIGHT.TAB TAB.SIGN)
			       FILE T))
			 PREDICATE.ALL)
		       (FORMAT FILE "~T" LEFT.MARGIN) (DODOWN (RPTN (- (1+ RIGHT.TAB) LEFT.MARGIN)) (PRINC "-" FILE)) (TERPRI FILE)))
      (TERPRI FILE))))

(DEFVAR DT*SYMBOL.KINDS '(CONSTANT.ALL FUNCTION.ALL PREDICATE.ALL))

(DEFMACRO DT-GROUND.TERM.SORT (TERM) `(MEM-GET ,TERM 1))


(defun dt-set.difference (set1 set2)
  (mapc #'(lambda (object2) (dt-putprop object2 'dt*in t)) set2)
  (prog1 (mapcar-not-nil #'(lambda (object1) (if (dt-getprop object1 'dt*in) nil object1)) set1)
	 (mapc #'(lambda (object2) (dt-putprop object2 'dt*in nil)) set2)))


(defun dt-arity (object)
						; Edited:  15-FEB-1991 21:53
						; Authors: PRCKLN
						; Input:   A function, predicate, or constant
						; Effect:  -
						; Value:   The arity of function or predicate
						;          If constant 0
  (cond ((dt-function.is object) (dt-function.arity object))
	((dt-predicate.is object) (length (DT-PREDICATE.get 'DOMAINSORTS object)))
	((dt-constant.is object) 0)
	(t (error "~A is not function, predicate, or constant." object))))

