;; -*- Package: MKRP; Syntax: Common-Lisp; Mode: LISP -*-

(IN-PACKAGE "MARKGRAF-KARL" :use '("CL") :nicknames '("MKRP"))

(defvar cg*commons.to.save '(CG*RECOLOUR CG*OBJECTCLASSES CG*SAVESTACK CG*GRAPH.REGISTER CG*CHANGE.QUEUE.LITERALS))

(DEFVAR CG*RECOLOUR '((R . RD) (P . PD) (SI . SID) (RIW . RIWD)))

(DEFVAR CG*OBJECTCLASSES NIL)

(DEFVAR CG*SAVESTACK NIL)

(DEFVAR CG*GRAPH.REGISTER NIL)

(DEFUN CG-CREATE.EMPTY.GRAPH NIL
						; EDITED:  11-MAY-84 18:08:20           NE
						; INPUT:   NONE
						; EFFECT:  MAKES THE EMPTY GRAPH THE ACTUAL GRAPH.
						;          ANY OBJECTS BEEING PART OF THE ACTUAL
						;          GRAPH PRIOR TO THIS CALL ARE IRREVERSABLY
						;          DELETED.
						; VALUE:   UNDEFINED.
  (DS-RESET)
  (ord-reset)
  (SETQ CG*GRAPH.REGISTER NIL
	CG*SAVESTACK      NIL
	CG*OBJECTCLASSES  (CONS 'CLAUSES (DS-LINK.COLOURS.FOR 'ALL)))
  (CG=REPR_INIT))

(DEFUN CG-DISJOINTIFY NIL
						; EDITED: 18. 10. 83                 NE
						; INPUT:  NO ARGUMENTS.
						;         THE GRAPH MAY BE FIXED OR NOT. THE OBJECT
						;         LIST INVARIANT HOLDS, I.E. 'ALL AND 'REMOVED
						;         LISTS ARE DISJOINT, OTHERS MAY OVERLAP.
						; EFFECT: MAKES THE LISTS 'INSERTED 'CHANGED 'REMOVED
						;         DISJOINT BY REMOVING ALL INTERSECTING PARTS
						;         FROM 'CHANGED AND BY PHYSICALLY DELETING
						;         THE OBJECTS IN BOTH 'INSERTED AND 'REMOVED.
						; VALUE:  UNDEFINED.
  (MAPC #'(LAMBDA (CLASS)
	    (MAPC #'(LAMBDA (OBJECT) (DT-PUTPROP OBJECT 'CG*INSERTED T)) (CG=REPR_LIST CLASS 'INSERTED))
	    (MAPC #'(LAMBDA (OBJECT) (DT-PUTPROP OBJECT 'CG*REMOVED T)) (CG=REPR_LIST CLASS 'REMOVED))
	    (MAPC #'(LAMBDA (OBJECT)
		      (when (OR (DT-GETPROP OBJECT 'CG*INSERTED) (DT-GETPROP OBJECT 'CG*REMOVED))
			(CG=REPR_REMOVE CLASS 'CHANGED OBJECT)))
		  (CG=REPR_LIST CLASS 'CHANGED))
	    (MAPC #'(LAMBDA (OBJECT)
		      (DT-REMPROP OBJECT 'CG*REMOVED)
		      (when (DT-GETPROP OBJECT 'CG*INSERTED) 
			    (CG=REPR_REMOVE CLASS 'INSERTED OBJECT)
			    (CG=REPR_REMOVE CLASS 'REMOVED OBJECT)
			    (if (ds-clause.is object) (DS-CLAUSE.DELETE object) (DS-LINK.DELETE OBJECT))))
		  (CG=REPR_LIST CLASS 'REMOVED))
	    (MAPC #'(LAMBDA (OBJECT) (DT-REMPROP OBJECT 'CG*INSERTED)) (CG=REPR_LIST CLASS 'INSERTED)))
	CG*OBJECTCLASSES))

(DEFUN CG-FIX (&optional CLAUSERETAINFLAG)
						;EDITED:  1. 3. 1982   HJO
						;INPUT:   A BOOLEAN VALUE.
						;EFFECT:  ALL CHANGES TO THE GRAPH PERFORMED SINCE
						;         THE LAST CALL OF CG-FIX ARE NOW MADE
						;         IRREVERSABLE. REMOVED OBJECTS ARE
						;         PHYSICALLY DELETED. IF CLAUSERETAINFLAG = T
						;         CLAUSES ARE EXEMPTED FROM PHYSICAL DELETION
						;         BUT STILL DISAPPEAR FROM THE LISTS IN THIS
						;         MODULE]
						;RESET OF CHANGEQUEUE.
  (SETQ CG*CHANGE.QUEUE.LITERALS (LIST NIL))
						;DESTROY ANCESTOR//DESCENDANT PROPERTIES OF LINKS AND
						;LITERALS THAT HAVE GOT THIS PROPERTIES
  (MAPC #'(LAMBDA (COLOUR)
	    (MAPC #'(LAMBDA (LINK)
		      (MAPC #'(LAMBDA (ANCESTOR.LINK) (DT-REMPROP ANCESTOR.LINK 'CG*DESCENDANT.LINKS))
			    (CG=LINK_ANCESTOR.LINKS LINK))
		      (DT-REMPROP LINK 'CG*ANCESTOR.LINKS))
		  (CG=REPR_LIST COLOUR 'INSERTED)))
	(DS-LINK.COLOURS.FOR 'ALL))
  (MAPC #'(LAMBDA (CLAUSE)
	    (DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
		    (PROG ((LITNO (1+ RPTN)))
			  (MAPC #'(LAMBDA (ANCESTOR)
				    (let ((ANCESTOR.CLAUSE (CAR ANCESTOR)))
				      (DS-CLAUSE.ALL.LIT.REMPROP ANCESTOR.CLAUSE 'CG*DESCENDANTS)))
				(CG-CLAUSE_ANCESTOR.LITERALS CLAUSE LITNO))))
	    (DT-REMPROP CLAUSE 'CG*CREATOR.UNIFIER) (DS-CLAUSE.ALL.LIT.REMPROP CLAUSE 'CG*ANCESTORS))
	(CG=REPR_LIST 'CLAUSES 'INSERTED))
  (unless CLAUSERETAINFLAG			;DELETE REMOVED CLAUSES, RESET CLAUSE LISTS
    (MAPC #'DS-CLAUSE.DELETE (remove-duplicates (CG=REPR_LIST 'CLAUSES 'REMOVED))))
  (CG=REPR_RESET 'CLAUSES 'REMOVED) (CG=REPR_RESET 'CLAUSES 'INSERTED) (CG=REPR_RESET 'CLAUSES 'CHANGED)
  (PROGN					;DELETE REMOVED LINKS, RESET LINK LISTS
    (MAPC #'(LAMBDA (COLOUR)
	      (MAPC #'(lambda (x) (when (ds-link.is x) (DS-LINK.DELETE x)))
		    (CG=REPR_LIST COLOUR 'REMOVED)))
	  (DS-LINK.COLOURS.FOR 'ALL))
    (MAPC #'(LAMBDA (COLOUR)
	      (CG=REPR_RESET COLOUR 'INSERTED)
	      (CG=REPR_RESET COLOUR 'REMOVED)
	      (CG=REPR_RESET COLOUR 'CHANGED))
      (DS-LINK.COLOURS.FOR 'ALL))))

(DEFUN CG-PUSH NIL ; EDITED: 14-MAY-84 16:17:34        NE
						; INPUT:  NONE.
						; EFFECT: THE CURRENT STATE IS SAVED AND A NEW EMPTY
						;         GRAPH IS CREATED. THE SAVED STATE CAN BE
						;         RESTORED USING CG-POP.
						; VALUE:  UNDEFINED
  (SETQ CG*SAVESTACK (CONS (CG=SAVE.STACK T) CG*SAVESTACK))
  (CG=REPR_INIT)
  (MAPC #'(LAMBDA (PREDICATE)
	    (DT-PREDICATE.PUT.POSITIVE.OCCURRENCES PREDICATE NIL)
	    (DT-PREDICATE.PUT.NEGATIVE.OCCURRENCES PREDICATE NIL))
	(DT-PREDICATE.ALL)))

(DEFUN CG-POP NIL
  ; EDITED_ 21-MAR-83 13:31:27        NE
  ; INPUT:  NONE.
  ; EFFECT: RESTORES THE STATE MOST RECENTLY SAVED
  ;         WITH CG-PUSH.
  ; VALUE:  UNDEFINED.
  (EVAL (CAR CG*SAVESTACK)))

(DEFUN CG-STORE.GRAPH.REGISTER NIL ; EDITED: 14-MAY-84 16:18:04        MW
						; INPUT:  NONE.
						; EFFECT: THE ACTUAL GRAPH IS STORED AND A NEW EMPTY
						;         GRAPH CREATED.
						; VALUE:  UNDEF'D.
  (SETQ CG*GRAPH.REGISTER (CG=SAVE.STACK)) (CG=REPR_INIT)
  (MAPC #'(LAMBDA (PREDICATE)
	    (DT-PREDICATE.PUT.POSITIVE.OCCURRENCES PREDICATE NIL)
	    (DT-PREDICATE.PUT.NEGATIVE.OCCURRENCES PREDICATE NIL))
	(DT-PREDICATE.ALL)))

(DEFUN CG-RECALL.GRAPH.REGISTER NIL ; EDITED: 30-JAN-84 16:14:43        MW
						; INPUT:  NONE.
						; EFFECT: THE GRAPH STORED IS MADE THE ACTUAL GRAPH.
						; VALUE:  UNDEF'D.
  (EVAL CG*GRAPH.REGISTER))

(DEFUN CG-VIRTUAL.GRAPH (FILE)
						; EDITED:  1. 3. 1982   HJO
						; INPUT:   EITHER NIL OR THE NAME OF A FILE WHICH IS
						;          OPEN FOR OUTPUT.
						; EFFECT:  COMPUTES AN S-EXPRESSION REPRESENTING THE
						;          ACTUAL GRAPH SUCH THAT (EVAL S-EXPRESSION)
						;          CREATES THE VERY SAME GRAPH AS IT IS
						;          DURING INVOCATION OF THIS FUNCTION.
						;          IF FILE <> NIL, THE S-EXPRESSION IS WRITTEN
						;          ON FILE, OTHERWISE THE S-EXPRESSION IS THE
						;          VALUE OF THIS FUNCTION.
						; VALUE:   IF FILE = NIL THEN THE COMPUTED
						;          S-EXPRESSION, ELSE NIL.
  (PROG ((S-EXPRESSION (CG=SAVE)))
	(COND (FILE (DS-SAVE FILE) (PROGN (PRINC S-EXPRESSION FILE) (TERPRI FILE)))
	      (T (RETURN (LIST 'PROGN (DS-SAVE FILE) S-EXPRESSION))))))

(DEFUN CG=SAVE NIL
						; EDITED: 14-MAY-84 16:13:06        NE
						; INPUT:  NONE.
						; VALUE:  AN S-EXPRESSION WHICH, WHEN EVALUATED,
						;         RESTORES THE STATE OF THE CG-MODULE TO
						;         WHAT IT WAS DURING INVOCATION OF THIS
						;         FUNCTION.
						; REMARK: NO COPIES ARE SAVED, BUT THE VALUES
						;         THEMSELVES]
  (PROG (EXPRESSIONS.COMMONVARS EXPRESSION.OBJECTLISTS)
	(SETQ EXPRESSIONS.COMMONVARS	    
	      (MAPCAR #'(LAMBDA (VAR) (LIST 'setq VAR `',(SYMBOL-VALUE VAR)))
		      cg*commons.to.save))
	(SETQ EXPRESSION.OBJECTLISTS (CG=REPR_SAVE))
	(RETURN (CONS 'PROGN (NCONC1 EXPRESSIONS.COMMONVARS EXPRESSION.OBJECTLISTS)))))

(DEFUN CG=SAVE.STACK (&optional SAVESTACK)
						; EDITED: 14-MAY-84 16:13:06        NE
						; INPUT:  A BOOLEAN VALUE.
						; VALUE:  AN S-EXPRESSION WHICH, WHEN EVALUATED,
						;         RESTORES THE STATE OF THE CG-MODULE TO
						;         WHAT IT WAS DURING INVOCATION OF THIS
						;         FUNCTION.
						; REMARK: NO COPIES ARE SAVED, BUT THE VALUES
						;         THEMSELVES]
  (let (EXPRESSIONS.COMMONVARS EXPRESSION.OBJECTLISTS)
    (COND (SAVESTACK (SETQ EXPRESSIONS.COMMONVARS (LIST (LIST 'setq 'CG*SAVESTACK `',(SYMBOL-VALUE CG*SAVESTACK))))))
    (SETQ EXPRESSION.OBJECTLISTS (CG=REPR_SAVE))
    (CONS 'PROGN
	  (NCONC1
	    (NCONC EXPRESSIONS.COMMONVARS
		   (MAPCAN
		     #'(LAMBDA (PREDICATE)
			 (LIST (LIST 'DT-PREDICATE.PUT.POSITIVE.OCCURRENCES PREDICATE
				     (KWOTE (DT-PREDICATE.POSITIVE.OCCURRENCES PREDICATE)))
			       (LIST 'DT-PREDICATE.PUT.NEGATIVE.OCCURRENCES PREDICATE
				     (KWOTE (DT-PREDICATE.NEGATIVE.OCCURRENCES PREDICATE)))))
		     (DT-PREDICATE.ALL)))
	    EXPRESSION.OBJECTLISTS))))

;;; Handling sort literals

(defun cg=insert.epsilon (clause litno sign predicate)
						; Edited:  19-NOV-1991 23:11
						; Authors: PRCKLN
						; Input:   A clause a number of one of its literals,
						;          sign and predicate of this literal.
						; Effect:  Inserts the literal in the UPP lists
						;          if it is a positive sort literal.
						; Value:   Undefined.
  (when (and (opt-get.option sort_literals)
	     (ds-sign.is.positive sign)
	     (eql (dt-predicate.element) predicate))
    (upp-epsilon.literals.insert (list (list clause litno)))))

(defun cg=remove.epsilon (clause litno sign predicate)
						; Edited:  19-NOV-1991 23:11
						; Authors: PRCKLN
						; Input:   A clause a number of one of its literals,
						;          sign and predicate of this literal.
						; Effect:  Removes the literal from the UPP lists
						;          if it is a positive sort literal.
						; Value:   Undefined.
  (when (and (opt-get.option sort_literals)
	     (ds-sign.is.positive sign)
	     (eql (dt-predicate.element) predicate))
    (upp-epsilon.literals.delete (list (list clause litno)))))

(defun cg=replace.epsilon (clause litno sign predicate)
						; Edited:  19-NOV-1991 23:11
						; Authors: PRCKLN
						; Input:   A clause a number of one of its literals,
						;          sign and predicate of this literal.
						; Effect:  Removes the literal +1 from the UPP lists
						;          if it is a positive sort literal and inserts itself.
						; Value:   Undefined.
  (when (and (opt-get.option sort_literals)
	     (ds-sign.is.positive sign)
	     (eql (dt-predicate.element) predicate))
    (upp-epsilon.literals.delete (list (list clause (1+ litno))))
    (upp-epsilon.literals.insert (list (list clause litno)))))

(DEFMACRO CG-CLAUSES (CG.INDICATOR)
						; EDITED:  18. 3. 82        NE
						; INPUT:   INDICATOR IN (ALL, REMOVED, CHANGED,
						;          INSERTED)
						; VALUE:   LIST OF ALL CLAUSES WHICH BELONG TO THE
						;          ACTUAL GRAPH OR HAVE BEEN REMOVED, CHANGED
						;          OR INSERTED, RESPECTIVELY, SINCE THE LAST
						;          CALL OF CG-FIX.
  `(CG=REPR_LIST 'CLAUSES ',CG.INDICATOR))

(DEFMACRO CG-#CLAUSES (CG.INDICATOR)
						; EDITED: 21-MAR-83 13:10:37        NE
						; INPUT:  INDICATOR IN (ALL, REMOVED, INSERTED,
						;         CHANGED)
						; VALUE:  NUMBER OF CLAUSES BELONGING TO THE ACTUAL
						;         OR HAVING BEEN REMOVED, INSERTED, CHANGED
						;         SINCE THE LAST CALL OF CG-FIX.
  `(CG=REPR_LENGTH 'CLAUSES ',CG.INDICATOR))

(DEFUN CG-INSERT.CLAUSE (CLAUSE CREATOR.UNIFIER LIST.OF.ANCESTORS &optional REASON INFO)
						; EDITED:  1. 3. 1982   HJO
						; INPUT:   A CLAUSE ADDRESS AND A BOOLEAN VALUE.
						;          'CREATOR.UNIFIER' IS A DOTTED PAIR (LINK .
						;          UNIFIER), OPERATION ON IT GAVE 'CLAUSE'.
						;          LIST.OF.ANCESTORS CONTAINS AS ITH ELEMENT
						;          AN ASSOCIATION LIST OF THE FORM
						;          ((CL1 LITNO1 LITNO2)(CLK LITNOL LITNOM)..)
						;          REPRESENTING THE ANCESTORS OF THE ITH
						;          LITERAL IN CLAUSE, OR NIL.
						;          REASON IS AN ATOM GIVING A MNEMONIC EXPLA-
						;          NATION WHY THE INSERTION OCCURS. INFO IS A
						;          FURTHER SPECIFICATION IN A FORMAT DEPENDING
						;          ON REASON. BOTH REASON AND INFO ARE USED
						;          ONLY BY THE FUNCTIONS CALLED IN THE ADVISE
						;          OF CG-TRACE.
						; EFFECT:  CLAUSE BECOMES PART OF THE ACTUAL GRAPH.
						;          IF ANCESTORS <>NIL, THE ANCESTOR-DESCENDANT
						;          RELATION IS STORED BY LITERAL PROPERTIES.
						;          THE LISTS
						;          POSITIVE.OCCURRENCES AND NEGATIVE.OCCURRENCES
						;          OF THE PREDICATE SYMBOLS OCCURING IN CLAUSE
						;          ARE UPDATED ACCORDINGLY.
						; VALUE:   UNDEFINED.
						;
  info reason
						;
						; PUT CREATOR UNIFIER INTO PROPERTY LIST.
  (DT-PUTPROP CLAUSE 'CG*CREATOR.UNIFIER CREATOR.UNIFIER)
						; BUILT ANCESTOR AND DESCENDANT PROPERTIES FROM
						; LIST.OF.ANCESTORS:
  (when LIST.OF.ANCESTORS
    (let ((LITNO 0))
      (MAPC #'(LAMBDA (ANCESTORS)
		(SETQ LITNO (1+ LITNO))
		(DS-CLAUSE.LIT.PUTPROP CLAUSE LITNO 'CG*ANCESTORS ANCESTORS)
		(MAPC #'(LAMBDA (ANCESTOR) (CG=CLAUSE_PUT.DESCENDANT.LITERALS CLAUSE LITNO ANCESTOR))
		      ANCESTORS))
	    LIST.OF.ANCESTORS)))
  (PROGN					; update object lists:
    (CG=REPR_INSERT 'CLAUSES 'ALL CLAUSE)
    (CG=REPR_INSERT 'CLAUSES 'INSERTED CLAUSE))
  (let (POS.PREDICATES NEG.PREDICATES LITNOS)	; update predicate occurrence lists
    (ds-clause.do #'(lambda (litno)
		      (let ((PREDICATE (DS-CLAUSE.PREDICATE CLAUSE litno))
			    (SIGN (DS-CLAUSE.SIGN CLAUSE litno)))
			(cg=insert.epsilon clause litno sign predicate)
			(unless (OR (AND (DS-SIGN.IS.POSITIVE SIGN) (MEMBER PREDICATE POS.PREDICATES))
				    (AND (DS-SIGN.IS.NEGATIVE SIGN) (MEMBER PREDICATE NEG.PREDICATES)))
			  (COND
			    ((DS-SIGN.IS.POSITIVE SIGN)
			     (push PREDICATE POS.PREDICATES)
			     (SETQ LITNOS (LIST LITNO))
			     (DODOWN (RPTN (1- LITNO))
			       (COND
				 ((AND (DS-SIGN.IS.POSITIVE (DS-CLAUSE.SIGN CLAUSE (1+ rptn)))
				       (EQL PREDICATE (DS-CLAUSE.PREDICATE CLAUSE (1+ rptn))))
				  (push (1+ rptn) LITNOS))))		
			     (DT-PREDICATE.PUT.POSITIVE.OCCURRENCES
			       PREDICATE (CONS (CONS CLAUSE LITNOS)
					       (DT-PREDICATE.POSITIVE.OCCURRENCES PREDICATE))))
			    (T (push PREDICATE NEG.PREDICATES)
			       (SETQ LITNOS (LIST LITNO))
			       (DODOWN (RPTN (1- LITNO))
				 (COND
				   ((AND (DS-SIGN.IS.NEGATIVE (DS-CLAUSE.SIGN CLAUSE (1+ rptn)))
					 (EQL PREDICATE (DS-CLAUSE.PREDICATE CLAUSE (1+ rptn))))
				    (push (1+ rptn) LITNOS))))
			       (DT-PREDICATE.PUT.negative.OCCURRENCES
				 PREDICATE
				 (CONS (CONS CLAUSE LITNOS) (DT-PREDICATE.negative.OCCURRENCES PREDICATE))))))))
		  clause)))

(DEFMACRO CG-CLAUSE_ANCESTOR.LITERALS (CLAUSE LITNO)
						; EDITED: 23-JUN-83 12:56:56        MW
						; INPUT:  CLAUSE AND LITNO ADDRESS
						; VALUE:  ASSOC-LIST OF LITNO'TH ANCESTORS.
  `(DS-CLAUSE.LIT.GETPROP ,CLAUSE ,LITNO 'CG*ANCESTORS))

(DEFMACRO CG-CLAUSE_DESCENDANT.LITERALS (CLAUSE LITNO)
						; EDITED: 23-JUN-83 12:56:56        MW
						; INPUT:  CLAUSE AND LITNO ADDRESS
						; VALUE:  ASSOC-LIST OF LITNO'TH DESCENDANTS.
  `(DS-CLAUSE.LIT.GETPROP ,CLAUSE ,LITNO 'CG*DESCENDANTS))

(DEFMACRO CG-CLAUSE_CREATOR.UNIFIER (CLAUSE)
						; EDITED: 28-FEB-84 17:58:56
						; INPUT:  'CLAUSE' IS A CLAUSE.
						; EFFECT: -
						; VALUE:  DOTTED PAIR (LINK . UNIFIER), THE CREATOR
						;         UNIFIER OF 'CLAUSE' AND ITS LINK, STORED
						;         IN PROPERTY 'CG*CREATOR.UNIFIER'.
  `(DT-GETPROP ,CLAUSE 'CG*CREATOR.UNIFIER))

(DEFUN CG-REMOVE.CLAUSE (CLAUSE &optional REASON INFO)
						; EDITED:  1. 3. 1982   HJO
						; INPUT:   A CLAUSE OF THE ACTUAL GRAPH AND A
						;          BOOLEAN VALUE.
						;          REASON IS AN ATOM GIVING A MNEMONIC EXPLA-
						;          NATION WHY THE REMOVAL OCCURS. INFO IS A
						;          FURTHER SPECIFICATION IN A FORMAT DEPENDING
						;          ON REASON. BOTH REASON IND INFO ARE USED
						;          ONLY BY THE FUNCTIONS CALLED IN THE ADVISE
						;          OF CG-TRACE.
						; EFFECT:  CLAUSE IS REMOVED FROM THE ACTUAL GRAPH
						;          AND SO ARE ALL LINKS CONNECTED TO IT.
						;          THE LISTS POSITIVE.OCCURRENCES AND
						;          NEGATIVE.OCCURRENCES OF THE PREDICATE
						;          SYMBOLS OCCURING IN CLAUSE ARE UPDATED
						;          ACCORDINGLY.
						; VALUE:   LIST OF ALL OTHER CLAUSES THAT WERE
						;          CONNECTED TO ONE OF THE REMOVED RLINKS OR
						;          PLINKS OR IPLINKS.
  reason info
  (COND
    ((DISJOINTP (DS-CLAUSE.ATTRIBUTES CLAUSE) '(REFLEXIVITY IRREFLEXIVITY))
						; UPDATE ANCESTOR AND DESCENDANT PROPERTIES INSOFAR
						; AS A CHANGE IS CAUSED BY THE REMOVAL OF CLAUSE
     (DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
       (let ((LITNO (1+ RPTN)))
						; REMOVE DESCENDANTS OF ANCESTORS OF ALL LITERALS
						; OF CLAUSE:
	 (MAPC #'(LAMBDA (ANCESTOR)
		   (let ((ANCESTOR.CLAUSE (CAR ANCESTOR)))
		     (MAPC #'(LAMBDA (ANCESTOR.LITNO)
			       (DS-CLAUSE.LIT.PUTPROP
				 ANCESTOR.CLAUSE ANCESTOR.LITNO 'CG*DESCENDANTS
				 (REMASSOC CLAUSE (DS-CLAUSE.LIT.GETPROP ANCESTOR.CLAUSE ANCESTOR.LITNO 'CG*DESCENDANTS))))
			   (CDR ANCESTOR))))
	       (CG-CLAUSE_ANCESTOR.LITERALS CLAUSE LITNO))))
     (DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
       (let ((LITNO (1+ RPTN)))			; REMOVE ANCESTORS OF DESCENDANTS OF ALL LITERALS
						; OF CLAUSE:
	 (MAPC #'(LAMBDA (DESCENDANT)
		   (let ((DESCENDANT.CLAUSE (CAR DESCENDANT)))
		     (MAPC #'(LAMBDA (DESCENDANT.LITNO)
			       (DS-CLAUSE.LIT.PUTPROP
				 DESCENDANT.CLAUSE DESCENDANT.LITNO 'CG*ANCESTORS
				 (REMASSOC CLAUSE
					   (DS-CLAUSE.LIT.GETPROP DESCENDANT.CLAUSE DESCENDANT.LITNO 'CG*ANCESTORS))))
			   (CDR DESCENDANT))))
	       (CG-CLAUSE_DESCENDANT.LITERALS CLAUSE LITNO))))
						; REMOVE ANCESTORS AND DESCENDANTS OF CLAUSE
     (DS-CLAUSE.ALL.LIT.REMPROPS CLAUSE '(CG*ANCESTORS CG*DESCENDANTS))
						; UPDATE OBJECT LISTS
     (CG=REPR_REMOVE 'CLAUSES 'ALL CLAUSE)
     (CG=REPR_INSERT 'CLAUSES 'REMOVED CLAUSE)
     (ds-clause.do #'(lambda (litno)		; Update predicate occurrence lists
		       (let ((predicate (DS-CLAUSE.PREDICATE CLAUSE litno))
			     (sign (DS-CLAUSE.SIGN CLAUSE litno)))
			 (cg=remove.epsilon clause litno sign predicate)
			 (if (DS-SIGN.IS.POSITIVE sign)
			     (DT-PREDICATE.DELete.POSITIVE.OCCURRENCES clause PREDICATE)
			     (DT-PREDICATE.delete.NEGATIVE.OCCURRENCES clause PREDICATE))))
		   clause)))
  (MAPC #'(LAMBDA (LINK) (CG=REMOVE.LINK LINK 'PARENTREMOVAL CLAUSE))
	(DS-CLAUSE.ALL.LINKS (DS-LINK.COLOURS.FOR 'NOPURITY) CLAUSE))
  (DELETE CLAUSE
	  (REMOVE-DUPLICATES
	    (MAPCAN #'(LAMBDA (LINK) (CG=REMOVE.LINK LINK 'PARENTREMOVAL CLAUSE))
		    (DS-CLAUSE.ALL.LINKS (DS-LINK.COLOURS.FOR 'PURITY) CLAUSE)))))

(DEFUN CG-REPLACE.LITERAL (CLAUSE LITNO NEWSIGN NEWPREDICATE NEWTERMLIST &optional REASON INFO)
						; EDITED: 14-APR-83 16:44:58        NE
						; INPUT:   A CLAUSE OF THE ACTUAL GRAPH AND THE NUMBER
						;          OF A LITERAL OF THAT CLAUSE, A SIGN, A
						;          PREDICATE AND A LIST OF TERMS.
						;          REASON IS AN ATOM GIVING A MNEMONIC EXPLA-
						;          NATION WHY THE CHANGE  OCCURS. INFO IS A
						;          FURTHER SPECIFICATION IN A FORMAT DEPENDING
						;          ON REASON. BOTH REASON IND INFO ARE USED
						;          ONLY BY THE FUNCTIONS CALLED IN THE ADVISE
						;          OF CG-TRACE.
						; EFFECT:  THE TERMLIST OF THE LITNO-TH LITERAL OF
						;          CLAUSE IS REPLACED BY NEWTERMLIST.
						;          CLAUSE IS MARKED AS CHANGED.
						; VALUE:   UNDEFINED.
  reason info
  (unless (MEMBER CLAUSE (CG=REPR_LIST 'CLAUSES 'CHANGED))
    (CG=REPR_INSERT 'CLAUSES 'CHANGED CLAUSE))
  (ds-clause.lit.putprop clause litno 'cg*changed t)
  (DS-CLAUSE.REPLACE.LITERAL CLAUSE LITNO NEWSIGN NEWPREDICATE NEWTERMLIST))

(defun cg-clause.changed.p (clause litno)
  (ds-clause.lit.getprop clause litno 'cg*changed))

(DEFUN CG-REMOVE.LITERAL (CLAUSE LITNO SIBLINGLITNO &optional REASON INFO)
						; EDITED:  1. 3. 1982   HJO
						; INPUT:   A CLAUSE OF THE ACTUAL GRAPH AND THE NUMBER
						;          OF A LITERAL OF THAT CLAUSE AND A BOOLEAN
						;          VALUE.
						;          REASON IS AN ATOM GIVING A MNEMONIC EXPLA-
						;          NATION WHY THE REMOVAL OCCURS. INFO IS A
						;          FURTHER SPECIFICATION IN A FORMAT DEPENDING
						;          ON REASON. BOTH REASON IND INFO ARE USED
						;          ONLY BY THE FUNCTIONS CALLED IN THE ADVISE
						;          OF CG-TRACE.
						; EFFECT:  THE LITNO'TH LITERAL IS REMOVED FROM CLAUSE
						;          AND SO ARE ALL LINKS CONNECTED TO THAT
						;          LITERAL.
						;          CLAUSE IS MARKED AS CHANGED.
						;          THE LISTS
						;          POSITIVE.OCCURRENCES AND NEGATIVE.OCCURRENCES
						;          OF THE PREDICATE.SYMBOL  OCCURING IN
						;          IN THE LITNO'TH LITERAL OF CLAUSE ARE
						;          UPDATED.
						; VALUE:   LIST OF ALL OTHER CLAUSES THAT WERE
						;          CONNECTED TO ONE OF THE REMOVED RLINKS
						;          OR PLINKS OR IPLINKS.
  info reason
  (PROG (ANCESTORLIST DESCENDANTLIST)		; UPDATE ANCESTOR/DESCENDANT PROP.S AND RECALCULATE
						; LITERAL NUMBERS IF NECESSARY BY REPLACING LITNO BY
						; SIBLINGLITNO IF SIBLINGLITNO <> NIL
	(DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
	  (COND ((< (1+ RPTN) LITNO) (SETQ RPTN 0))
		(T (SETQ ANCESTORLIST   (CG=MERGE.RELATIVES (COPY-TREE (CG-CLAUSE_ANCESTOR.LITERALS CLAUSE (1+ RPTN)))
							    ANCESTORLIST)
			 DESCENDANTLIST (CG=MERGE.RELATIVES (COPY-TREE (CG-CLAUSE_DESCENDANT.LITERALS CLAUSE (1+ RPTN)))
							    DESCENDANTLIST)))))
	(MAPC #'(LAMBDA (ANCESTOR)
		  (let ((ANCESTOR.CLAUSE (CAR ANCESTOR))
			(ANCESTOR.LITNOS (CDR ANCESTOR))
			DESCENDANT DESCENDANTS)
		    (MAPC #'(LAMBDA (ANCESTOR.LITNO)
			      (SETQ DESCENDANTS (DS-CLAUSE.LIT.GETPROP ANCESTOR.CLAUSE ANCESTOR.LITNO 'CG*DESCENDANTS))
			      (when (and SIBLINGLITNO (MEMBER LITNO (ASSOC CLAUSE DESCENDANTS)))
				(SETQ DESCENDANTS (REMVALUESASSOC CLAUSE (list LITNO) DESCENDANTS))
				(SETQ DESCENDANTS (INSASSOC (LIST CLAUSE SIBLINGLITNO) DESCENDANTS)))
			      (SETQ DESCENDANT (ASSOC CLAUSE DESCENDANTS))
			      (RPLACD DESCENDANT (CG=RECALCULATE.LITNOS DESCENDANT LITNO))
			      (unless (CDR DESCENDANT) (SETQ DESCENDANTS (REMASSOC CLAUSE DESCENDANTS)))
			      (DS-CLAUSE.LIT.PUTPROP ANCESTOR.CLAUSE ANCESTOR.LITNO 'CG*DESCENDANTS DESCENDANTS))
			  ANCESTOR.LITNOS)))
	      ANCESTORLIST)
	(MAPC #'(LAMBDA (DESCENDANT)
		  (let ((DESCENDANT.CLAUSE (CAR DESCENDANT))
			(DESCENDANT.LITNOS (CDR DESCENDANT))
			ANCESTOR ANCESTORS)
		    (MAPC #'(LAMBDA (DESCENDANT.LITNO)
			      (SETQ ANCESTORS (DS-CLAUSE.LIT.GETPROP DESCENDANT.CLAUSE DESCENDANT.LITNO 'CG*ANCESTORS))
			      (when (and SIBLINGLITNO (MEMBER LITNO (ASSOC CLAUSE ANCESTORS)))
				(SETQ ANCESTORS (REMVALUESASSOC CLAUSE LITNO ANCESTORS))
				(SETQ ANCESTORS (INSASSOC (LIST CLAUSE SIBLINGLITNO) ANCESTORS)))
			      (SETQ ANCESTOR (ASSOC CLAUSE ANCESTORS)) (RPLACD ANCESTOR (CG=RECALCULATE.LITNOS ANCESTOR LITNO))
			      (unless (CDR ANCESTOR) (SETQ ANCESTORS (REMASSOC CLAUSE ANCESTORS)))
			      (DS-CLAUSE.LIT.PUTPROP DESCENDANT.CLAUSE DESCENDANT.LITNO 'CG*ANCESTORS ANCESTORS))
			  DESCENDANT.LITNOS)))
	      DESCENDANTLIST)
	(COND
	  (SIBLINGLITNO
	   (let ((SIBLING.ANCESTORS (DS-CLAUSE.LIT.GETPROP CLAUSE SIBLINGLITNO 'CG*ANCESTORS)))
	     (MAPC #'(LAMBDA (ANCESTOR) (SETQ SIBLING.ANCESTORS (INSASSOC ANCESTOR SIBLING.ANCESTORS)))
		   (CG-CLAUSE_ANCESTOR.LITERALS CLAUSE LITNO))
	     (DS-CLAUSE.LIT.PUTPROP CLAUSE SIBLINGLITNO 'CG*ANCESTORS SIBLING.ANCESTORS))
	   (let ((SIBLING.DESCENDANTS (DS-CLAUSE.LIT.GETPROP CLAUSE SIBLINGLITNO 'CG*DESCENDANTS)))
	     (MAPC #'(LAMBDA (DESCENDANT)
		       (SETQ SIBLING.DESCENDANTS (INSASSOC DESCENDANT SIBLING.DESCENDANTS)))
		   (CG-CLAUSE_DESCENDANT.LITERALS CLAUSE LITNO))
	     (DS-CLAUSE.LIT.PUTPROP CLAUSE SIBLINGLITNO 'CG*DESCENDANTS SIBLING.DESCENDANTS)))))
						; UPDATE PREDICATE OCCURrENCE   LISTS
  (let ((PREDICATE (DS-CLAUSE.PREDICATE CLAUSE LITNO))
	(SIGN (DS-CLAUSE.SIGN CLAUSE LITNO)))
    (cg=remove.epsilon clause litno sign predicate)
    (if (DS-SIGN.IS.POSITIVE SIGN)
	(DT-PREDICATE.delete.POSITIVE.OCCURRENCE clause litno predicate)
	(DT-PREDICATE.delete.negative.OCCURRENCE clause litno predicate))
    (DODOWN (RPTN (DS-CLAUSE.NOLIT CLAUSE))
      (COND ((EQL (1+ RPTN) LITNO) (SETQ RPTN -1))
	    (T (SETQ PREDICATE (DS-CLAUSE.PREDICATE CLAUSE (1+ RPTN)))
	       (cg=replace.epsilon clause rptn (DS-CLAUSE.SIGN CLAUSE (1+ RPTN)) predicate)
	       (RPLACA
		 (MEMBER (1+ RPTN)
			 (cASSOC CLAUSE
				 (COND ((DS-SIGN.IS.POSITIVE (DS-CLAUSE.SIGN CLAUSE (1+ RPTN)))
					(DT-PREDICATE.POSITIVE.OCCURRENCES PREDICATE))
				       (T (DT-PREDICATE.NEGATIVE.OCCURRENCES PREDICATE)))))
		 RPTN)))))
  (COND
    ((NOT (MEMBER CLAUSE (CG=REPR_LIST 'CLAUSES 'CHANGED)))
						; UPDATE OBJECT LISTS
     (CG=REPR_INSERT 'CLAUSES 'CHANGED CLAUSE)))
  (PROGN					; REMOVE NOPURITY LINKS FROM LITERAL TO BE REMOVED
    (MAPC #'(LAMBDA (LINK)
	      (CG=REMOVE.LINK LINK 'PARENTLITREMOVAL (CONS CLAUSE LITNO)))
	  (DS-CLAUSE.LINKS (DS-LINK.COLOURS.FOR 'NOPURITY) CLAUSE LITNO)))
  (let (RESULT)					; Remove purity links from literal to be removed
    (MAPC #'(LAMBDA (LINK)
	      (SETQ RESULT (NCONC (CG=REMOVE.LINK LINK 'PARENTLITREMOVAL (CONS CLAUSE LITNO)) RESULT)))
	  (DS-CLAUSE.LINKS (DS-LINK.COLOURS.FOR 'PURITY) CLAUSE LITNO))
    (DS-CLAUSE.REMOVE.LITERAL CLAUSE LITNO)      
    (DELETE CLAUSE (REMOVE-DUPLICATES RESULT))))

(DEFUN CG=MERGE.RELATIVES (RELATIVE1 RELATIVE2)
						; EDITED:  4-SEP-83 22:42:09        MW
						; INPUT:   RELATIVE1, RELATIVE2: ASSOCLISTS OF
						;          ANCESTOR OR DESCENDANT LITERALS WITH THEIR
						;          CLAUSE NUMBERS.
						; VALUE:   'UNION' OF THE TWO LISTS, AVOIDING MULTIPLE
						;          OCCURRENCES.
						; EFFECT:  DESTRUCTIVE ON INPUT PARAMETERS
  (MAPC #'(LAMBDA (ITEM1) (SETQ RELATIVE2 (INSASSOC ITEM1 RELATIVE2))) RELATIVE1) RELATIVE2)

(DEFUN CG=RECALCULATE.LITNOS (CENDANT LITNO)
						; EDITED:  4-SEP-83 16:39:00        MW
						; INPUT:  CENDANT: AN ANCESTOR OR DESCENDANT: AN
						;         ELEMENT OF AN ASSOCLIST OF ANCESTORS OR
						;         DESCENDANTS. IT HAS THE FORM
						;         (<CLAUSENO> 1 2 ... <CLAUSE.NOLIT>).
						; EFFECT: RECALCULATE LITERAL NUMBERS IN CENDANT
						;         IN ACCORDANCE WITH THE REMOVAL OF LITNO.
						; VALUE:  NEW CDR OF CENDANT.
  (DREMAP (CDR CENDANT) NIL
	  #'(LAMBDA (TAIL.OF.LITNOS)
	      (let ((CENDANT.LITNO (CAR TAIL.OF.LITNOS)))
		(COND ((> CENDANT.LITNO LITNO) (RPLACA TAIL.OF.LITNOS (1- CENDANT.LITNO)) NIL) ((EQL CENDANT.LITNO LITNO)))))))

(DEFUN CG=CLAUSE_PUT.DESCENDANT.LITERALS (CLAUSE LITNO ANCESTOR)
						; EDITED: 17-MAR-84 22:17:40
						; INPUT:  'LITNO' IS THE NUMBER OF A LITERAL OF CLAUSE
						;         'CLAUSE'. 'ANCESTOR' DESCRIBES ALL ITS
						;         ANCESTORLITERALS IN ONE CLAUSE (CLAUSE
						;         LITNO1 ... LITNOK).
						; EFFECT: IN ALL ANCESTORLITERALS ('ANCESTOR')
						;         DESCENDANTS ARE UPDATED ACCORDING TO
						;         'ANCESTOR'.
						; VALUE:  UNDEFINED.
  clause litno
  (let ((ANCESTOR.CLAUSE (first ANCESTOR)))
    (MAPC #'(LAMBDA (ANCESTOR.LITNO)
	      (DS-CLAUSE.LIT.PUTPROP
		ANCESTOR.CLAUSE ANCESTOR.LITNO 'CG*DESCENDANTS
		(INSASSOC (LIST CLAUSE LITNO) (DS-CLAUSE.LIT.GETPROP ANCESTOR.CLAUSE ANCESTOR.LITNO 'CG*DESCENDANTS))))
	  (CDR ANCESTOR))))

(DEFMACRO CG-LINKS (COLOUR INDICATOR)
						; EDITED: 18-MAR-83 18:59:54        NE
						; INPUT:  COLOUR  IS EITHER DIRECTLY OR AFTER
						;         BEEING EVALUATED
						;         A LINK COLOUR OR A LIST OF LINK COLOURS.
						;         INDICATOR IN (ALL REMOVED CHANGED INSERTED
						; VALUE:  LIST OF ALL LINKS OF THE RESPECTIVE COLOUR
						;         (OR LIST OF COLOURS (WORKES LIKE APPEND]))
						;         WHICH ARE IN THE ACTUAL GRAPH OR HAVE
						;         BEEN REMOVED, CHANGED OR INSERTERD,
						;         RESPECTIVELY, SINCE THE LAST CALL OF CG-FIX.
						;
						; NOTICE: THE COMPUTED MACRO EVALUATES THE COLOUR
						;         ARGUMENT AT COMPILE TIME, IF THE VALUE IS
						;         AN ADMISSIBLE LINK COLOUR OR A LIST OF
						;         LINK COLOURS
  (cond ((and (consp colour)
	      (macro-function (first colour)))	; colour is itself a macro call, which may now be expanded
	 `(cg-links ,(macroexpand-1 colour) ,indicator))
	((and (atom colour)
	      (member colour (ds-link.colours.for 'all)))
	 `(cg=repr_list ',colour ',indicator))
	((and (listp colour)
	      (eq (car colour) 'quote)
	      (member (cadr colour) (ds-link.colours.for 'all)))
	 `(cg=repr_list ,colour ',indicator))
	((and (listp colour)
	      (member (car colour) (ds-link.colours.for 'all)))
	 `(mapcan #'(lambda (co)
		      (copy-list (cg=repr_list co ',indicator)))
		  ',colour))
	((and (listp colour)
	      (eq (car colour) 'quote)
	      `(nconc ,@(mapcar #'(lambda (co)
				    `(copy-list (cg=repr_list ',co ',indicator)))
				(second colour)))))
	((member indicator '(ALL REMOVED CHANGED INSERTED))
	 `(let ((colour ,colour))
	    (if (atom colour)
		(cg=repr_list colour ',indicator)
		(mapcan #'(lambda (co)
			    (copy-list (cg=repr_list co ',indicator)))
			colour))))
	(t `(let ((colour ,colour))
	      (if (atom colour)
		  (cg=repr_list colour ,indicator)
		  (mapcan #'(lambda (co)
			      (copy-list (cg=repr_list co ,indicator)))
			  colour))))))

(DEFMACRO CG-#LINKS (COLOUR INDICATOR)		; edited: 21-mar-83 13:15:43        ne
						; input:  colour  is either directly or after evaluation
						;         a link colour or a list of link colours.
						;         indicator in (all, removed, changed, inserted).
						; value:  if a single colour is given, the number of
						;         links of that colour belonging to the actual
						;         graph or having been removed, changed, or inserted.
						;         if a list of colours is given, a list of the respective numbers.
						; remark: the computed macro evaluates the colour
						;         argument at compile time, if admissible.
  (cond ((and (consp colour)
	      (macro-function (first colour)))	; colour is itself a macro call, which may now be expanded
	 `(cg-#links ,(macroexpand-1 colour) ,indicator))
	((and (atom colour)
	      (member colour (ds-link.colours.for 'all)))	; a link colour as in (CG-#LINKS R ALL)
	 `(cg=repr_length ',colour ',indicator))
	((and (listp colour)
	      (eq (car colour) 'quote)
	      (member (cadr colour) (ds-link.colours.for 'all)))	; a quoted link colour as in (CG-#LINKS 'RIW ALL)
	 `(cg=repr_length ,colour ',indicator))
	((and (listp colour)
	      (member (car colour) (ds-link.colours.for 'all)))	; a list of link colours as in (CG-#LINKS (R RIW) ALL)
	 `(mapcar #'(lambda (co) (cg=repr_length co ',indicator))
		  ',colour))
	((and (listp colour)
	      (eq (car colour) 'quote))		; a quoted list of link colours as in (CG-#LINKS '(R RIW) ALL)
	 `(list ,@(mapcar #'(lambda (co) `(cg=repr_length ',co ',indicator))
			  (second colour))))
	((member indicator '(ALL REMOVED CHANGED INSERTED))
	 `(let ((colour ,colour))
	    (if (atom colour)
		(cg=repr_length colour ',indicator)
		(mapcar #'(lambda (co)
			    (cg=repr_length co ',indicator))
			colour))))
	(t `(let ((colour ,colour))		; else, to be computed at run time, as in (CG-#LINKS (foo x) ALL)
	      (if (atom colour)
		  (cg=repr_length colour ',indicator)
		  (mapcar #'(lambda (co) (cg=repr_length co ',indicator))
			  colour))))))

(DEFMACRO CG-INSERT.LINK (LINK ANCESTORLINKS &optional REASON INFO)
						; EDITED:  1. 3. 1982   HJO
						; INPUT:   A LINK ADDRESS
						;          ANCESTORLINKS IS A LIST OF ANCESTORLINKS
						;          REASON IS AN ATOM GIVING A MNEMONIC EXPLA-
						;          NATION WHY THE INSERTION OCCURS. INFO IS A
						;          FURTHER SPECIFICATION IN A FORMAT DEPENDING
						;          ON REASON. BOTH REASON AND INFO ARE USED
						;          ONLY BY THE FUNCTIONS CALLED IN THE ADVISE
						; EFFECT:  LINK IS CONNECTED TO ITS PARENT CLAUSES,
						;          WHICH MUST BE PART OF THE ACTUAL GRAPH,
						;          AND BECOMES PART OF THE ACTUAL GRAPH
						;          ITSELF.
						;          IF ANCESTORLINKS <>NIL, THE ANCESTOR-DES=
						;          CENDANT RELATION IS UPDATED FOR THE LINKS.
						; VALUE:   UNDEFINED.
  `(CG=INSERT.LINK ,LINK ,ANCESTORLINKS ,REASON ,INFO))

(DEFMACRO CG-LINK_ANCESTOR.LINKS (LINK)
						; EDITED: 23-JUN-83 14:33:04        MW
						; INPUT:  A LINK ADDRESS
						; VALUE:  LIST OF ANCESTOR LINKS OF LINK.
  `(CG=LINK_ANCESTOR.LINKS ,LINK))

(DEFMACRO CG-LINK_DESCENDANT.LINKS (LINK)
						; EDITED: 23-JUN-83 14:35:00        MW
						; INPUT:  A LINK ADDRESS
						; VALUE:  LIST OF DESCENDANT LINKS OF LINK.
  `(DT-GETPROP ,LINK 'CG*DESCENDANT.LINKS))

(DEFun CG-REMOVE.LINK (LINK &optional REASON INFO)
						; EDITED:  1. 3. 1982    HJO
						; INPUT:   A LINK OF THE ACTUAL GRAPH
						;          REASON IS AN ATOM GIVING A MNEMONIC EXPLA-
						;          NATION WHY THE REMOVAL OCCURS. INFO IS A
						;          FURTHER SPECIFICATION IN A FORMAT DEPENDING
						;          ON REASON. BOTH REASON IND INFO ARE USED
						;          ONLY BY THE FUNCTIONS CALLED IN THE ADVISE
						;          OF CG-TRACE.
						; EFFECT:  LINK IS REMOVED FROM THE ACTUAL GRAPH.
						; VALUE:   IF LINK HAS COLOUR R OR P, THEN A LIST OF
						;          THE PARENT CLAUSES OF LINK, ELSE NIL.
  (CG=REMOVE.LINK LINK REASON INFO))

(DEFMACRO CG-INSERT.UNIFIER (LINK UNIFIER &optional REASON INFO)
						; EDITED: 28-FEB-84 18:01:57
						; INPUT:  'LINK' IS A LINK, 'UNIFIER' A SUBSTITUTION,
						;         'REASON, AND 'INFO' ARE INFORMATIONS, USED
						;         IN CG-TRACE ONLY.
						; EFFECT: ADDS 'UNIFIER' TO THE UNIFIERS OF 'LINK'
						;         AND INSERTS 'LINK' INTO LINKS CHANGED.
  `(CG=INSERT.UNIFIER ,LINK ,UNIFIER ,REASON ,INFO))

(DEFMACRO CG-REMOVE.UNIFIER (LINK UNIFIER &optional RECOLOURFLAG REASON INFO)
						; EDITED:  7-DEC-83 14:22:35                      NE
						; INPUT:   A LINK OF THE ACTUAL GRAPH WITH OPERATIONAL
						;          COLOUR, A UNIFIER OF LINK AND A BOOLEAN
						;          VALUE.
						;          REASON IS AN ATOM GIVING A MNEMONIC EXPLA-
						;          NATION WHY THE REMOVAL OCCURS. INFO IS A
						;          FURTHER SPECIFICATION IN A FORMAT DEPENDING
						;          ON REASON. BOTH REASON IND INFO ARE USED
						;          ONLY BY THE FUNCTIONS CALLED IN THE ADVISE
						;          OF CG-TRACE.
						; EFFECT:  UNIFIER IS REMOVED FROM THE UNIFIER LIST OF
						;          LINK. IF ANY OTHER UNIFIERS REMAIN, LINK IS
						;          MARKED AS CHANGED, OTHERWISE REMOVED FROM
						;          THE ACTUAL GRAPH. IF RECOLOURFLAG = T
						;          AND LINK HAS AN APPROPRIATE COLOUR, IT IS
						;          RECOLOURED INSTEAD OF REMOVED.
						; VALUE:   IF LINK WAS REMOVED//RECOLOURED AND HAS
						;          A PURITY COLOUR, THEN A LIST OF
						;          THE PARENT CLAUSES OF LINK, ELSE NIL.
  `(CG=REMOVE.UNIFIER ,LINK ,UNIFIER ,RECOLOURFLAG ,REASON ,INFO))

(DEFUN CG-INHIBIT.UNIFIER (LINK UNIFIER &optional REASON INFO)
						; EDITED: 14-APR-84 01:50:32                    AP
						; INPUT:  'UNIFIER' IS A UNIFIER OF LINK 'LINK'.
						;         'REASON' AND 'INFO' ARE ONLY USED BY ADVISE
						;         FOR TRACE.
						; EFFECT: ONLY IF 'LINK' HAS ONE OF THE COLOURS WITH
						;         COMPONENT 'LABEL', E.G. R, P, OR SI.
						;         IF THERE EXISTS A PARALLEL LINK TO LINK
						;         'LINK' WITH SAME RULE, PARENTLITERALS,
						;         (PARENTTERMS,) COLOUR, AND DIRECTION, THAT
						;         IS LABELED WITH 'INHIBITED', 'UNIFIER' IS
						;         REMOVED FROM LINK 'LINK' AND INSERTED INTO
						;         UNIFIERLIST OF THE PARALLEL LINK.
						;         IF NO SUCH LINK EXISTS AND 'LINK' HAS ONLY
						;         ONE UNIFIER, 'LINK' WILL BE LABELED WITH
						;         'INHIBITED'.
						;         IF IT HAS MORE UNIFIERS, 'UNIFIER' IS
						;         REMOVED FROM THE LIST AND A NEW LINK WITH
						;         THIS UNIFIER,THE SAME ANCESTORS,
						;         DESCENDANTS, AND LABEL 'INHIBITED' WILL BE
						;         GENERATED.
						; VALUE:  NEW LINK IF ONE HAS BEEN GENERATED, NIL
						;         ELSE.
  info
  (PROG ((NEW.LINK NIL) (COLOUR (DS-LINK.COLOUR LINK)))
	(COND
	  ((AND (MEMBER COLOUR (DS-LINK.COLOURS.WITH 'LABEL)) (NOT (DS-LINK.IS.MARKED INHIBITED LINK)))
	   (PROG
	     ((UNIFIERS (DS-LINK.UNIFIERS LINK)) (RULE (DS-LINK.RULE LINK)) (POSPAR (DS-LINK.POSPAR LINK))
	      (NEGPAR (COND ((MEMBER COLOUR (DS-LINK.COLOURS.FOR 'PURITY)) (DS-LINK.NEGPAR LINK))))
	      (POSLITNO (DS-LINK.POSLITNO LINK))
	      (NEGLITNO (DS-LINK.NEGLITNO LINK)) (POSFCT NIL) (NEGFCT NIL) (OTHERLINKS NIL) (MARKED.LINK NIL))
	     (COND
	       ((MEMBER COLOUR (DS-LINK.COLOURS.FOR 'PARAMODULATION)) (SETQ POSFCT (DS-LINK.POSFCT LINK))
		(SETQ NEGFCT (DS-LINK.NEGFCT LINK))))	; SEARCHING PARALLEL LINKS.
						; Repairing the problem that after inhibition with following deletion of
						; the unifier will be removed again: Don' t delete the link.
	     #|(MAPC
	       (FUNCTION
		 (LAMBDA (OTHERLINK)
		   (COND
		     ((AND (NEQ LINK OTHERLINK)
			   (EQL RULE (DS-LINK.RULE OTHERLINK))
			   (EQL POSPAR (DS-LINK.POSPAR OTHERLINK))
			   (EQL NEGPAR (COND ((MEMBER (DS-LINK.COLOUR OTHERLINK)
						      (DS-LINK.COLOURS.FOR 'PURITY))
					      (DS-LINK.NEGPAR OTHERLINK))))
			   (EQL POSLITNO (DS-LINK.POSLITNO OTHERLINK))
			   (EQL NEGLITNO (DS-LINK.NEGLITNO OTHERLINK)) (EQL COLOUR (DS-LINK.COLOUR OTHERLINK))
			   (OR (NOT (MEMBER (DS-LINK.COLOUR OTHERLINK) (DS-LINK.COLOURS.FOR 'PARAMODULATION)))
			       (AND (EQUAL POSFCT (DS-LINK.POSFCT OTHERLINK)) (EQUAL NEGFCT (DS-LINK.NEGFCT OTHERLINK)))))
                      (SETQ OTHERLINKS (CONS OTHERLINK OTHERLINKS)))))
                      (DS-CLAUSE.LINKS COLOUR POSPAR POSLITNO))|#
	     ;; IS A PARALLEL LINK INHIBITED?
	     (MEMBER-IF
	       (FUNCTION
		 (LAMBDA (CANDIDATE.MARK)
		   (COND ((DS-LINK.IS.MARKED INHIBITED CANDIDATE.MARK) (SETQ MARKED.LINK CANDIDATE.MARK)))))
	       OTHERLINKS)
	     (COND
	       ((AND (CDR UNIFIERS) (NOT MARKED.LINK))
						; GENERATION OF A NEW LINK IF NO PARALLEL LINK EXISTS
						; AND 'LINK' HAS MORE THAN ONE UNIFIER.
		(CG=INSERT.LINK
		  (SETQ NEW.LINK
			(DS-LINK.CREATE COLOUR (LIST UNIFIER) POSPAR POSLITNO NEGPAR NEGLITNO POSFCT NEGFCT RULE))
		  (CG=LINK_ANCESTOR.LINKS LINK) 'INHIBITION REASON)
		(CG=REMOVE.UNIFIER LINK UNIFIER NIL 'INHIBITION REASON)
		(DS-LINK.MARK INHIBITED NEW.LINK)
		(CG=REPR_INSERT (DS-LINK.COLOUR NEW.LINK) 'CHANGED NEW.LINK))
	       (MARKED.LINK			; THERE EXISTS AN ADEQUATE PARALLEL LINK.
		(CG=INSERT.UNIFIER MARKED.LINK UNIFIER 'INHIBITION REASON)
		(CG=REMOVE.UNIFIER LINK UNIFIER NIL 'INHIBITION REASON))
	       (T				; 'LINK' IS INHIBITED IF NO PARALLEL LINK EXISTS AND
						; IT HAS ONLY ONE UNIFIER.
		(DS-LINK.MARK INHIBITED LINK)
		(COND
		  ((NOT (MEMBER LINK (CG=REPR_LIST (DS-LINK.COLOUR LINK) 'CHANGED)))
		   (CG=REPR_INSERT (DS-LINK.COLOUR LINK) 'CHANGED LINK))))))))
						; RETURN NEWLY GENERATED LINK.
						(RETURN NEW.LINK)))
  
(DEFUN CG=INSERT.LINK (LINK ANCESTORLINKS &optional REASON INFO)
						; EDITED:  1. 3. 1982   HJO
						; INPUT:   A LINK ADDRESS
						;          ANCESTORLINKS IS A LIST OF ANCESTORLINKS
						;          REASON IS AN ATOM GIVING A MNEMONIC EXPLA-
						;          NATION WHY THE INSERTION OCCURS. INFO IS A
						;          FURTHER SPECIFICATION IN A FORMAT DEPENDING
						;          ON REASON. BOTH REASON AND INFO ARE USED
						;          ONLY BY THE FUNCTIONS CALLED IN THE ADVISE
						; EFFECT:  LINK IS CONNECTED TO ITS PARENT CLAUSES,
						;          WHICH MUST BE PART OF THE ACTUAL GRAPH,
						;          AND BECOMES PART OF THE ACTUAL GRAPH
						;          ITSELF.
						;          IF ANCESTORLINKS <>NIL, THE ANCESTOR-DES=
						;          CENDANT RELATION IS UPDATED FOR THE LINKS.
						; VALUE:   UNDEFINED.
  info reason 
  (COND (ANCESTORLINKS (MAPC #'(LAMBDA (ANCESTOR.LINK)
				 (DT-PUTPROP ANCESTOR.LINK 'CG*DESCENDANT.LINKS
					     (INS LINK (DT-GETPROP ANCESTOR.LINK 'CG*DESCENDANT.LINKS))))
			     ANCESTORLINKS)
		       (DT-PUTPROP LINK 'CG*ANCESTOR.LINKS ANCESTORLINKS)))
  (let ((COLOUR (DS-LINK.COLOUR LINK)))
    (CG=REPR_INSERT COLOUR 'ALL LINK)
    (CG=REPR_INSERT COLOUR 'INSERTED LINK))
  (DS-LINK.CONNECT LINK))


  
(DEFUN CG=INSERT.UNIFIER (LINK UNIFIER REASON INFO)
    ; EDITED: 28-FEB-84 18:01:57
    ; INPUT:  'LINK' IS A LINK, 'UNIFIER' A SUBSTITUTION,
    ;         'REASON, AND 'INFO' ARE INFORMATIONS, USED
    ;         IN CG-TRACE ONLY.
    ; EFFECT: ADDS 'UNIFIER' TO THE UNIFIERS OF 'LINK'
    ;         AND INSERTS 'LINK' INTO LINKS CHANGED.
    reason info
    (DS-LINK.PUTUNIFIERS LINK (CONS UNIFIER (DS-LINK.UNIFIERS LINK)))
    (COND
      ((NOT (MEMBER LINK (CG=REPR_LIST (DS-LINK.COLOUR LINK) 'CHANGED))) (CG=REPR_INSERT (DS-LINK.COLOUR LINK) 'CHANGED LINK))))
  
(DEFUN CG=REMOVE.LINK (LINK REASON INFO)
						; EDITED:  1. 3. 1982    HJO
						; INPUT:   A LINK OF THE ACTUAL GRAPH
						;          REASON IS AN ATOM GIVING A MNEMONIC EXPLA-
						;          NATION WHY THE REMOVAL OCCURS. INFO IS A
						;          FURTHER SPECIFICATION IN A FORMAT DEPENDING
						;          ON REASON. BOTH REASON IND INFO ARE USED
						;          ONLY BY THE FUNCTIONS CALLED IN THE ADVISE
						;          OF CG-TRACE.
						; EFFECT:  LINK IS REMOVED FROM THE ACTUAL GRAPH.
						;          THE LINK PROPERTIES ANCESTOR-//DESCENDANT-
						;          LINKS ARE UPDATED IF THEY EXISTED BEFORE.
						; VALUE:   IF LINK HAS COLOUR R,P OR IP, THEN A LIST
						;          OF THE PARENT CLAUSES OF LINK, ELSE NIL.
  info reason
  (MAPC #'(LAMBDA (ANCESTOR.LINK)
	    (DT-PUTPROP ANCESTOR.LINK 'CG*DESCENDANT.LINKS (DELETE LINK (DT-GETPROP ANCESTOR.LINK 'CG*DESCENDANT.LINKS))))
	(CG=LINK_ANCESTOR.LINKS LINK))
  (MAPC #'(LAMBDA (DESCENDANT.LINK)
	    (DT-PUTPROP DESCENDANT.LINK 'CG*ANCESTOR.LINKS (DELETE LINK (DT-GETPROP DESCENDANT.LINK 'CG*ANCESTOR.LINKS))))
	(CG-LINK_DESCENDANT.LINKS LINK))
  (DT-REMPROP LINK 'CG*ANCESTOR.LINKS)
  (DT-REMPROP LINK 'CG*DESCENDANT.LINKS)
  (mapc #'(lambda (var) (dt-variable.delete var)) (second (dt-getprop link 'cg*result)))
  (dt-remprop link 'cg*result)
  ;(if (= 13886 link) (break "Link remove"))
  (let ((COLOUR (DS-LINK.COLOUR LINK)))
    (prog1 (if (MEMBER COLOUR (DS-LINK.COLOURS.FOR 'PURITY))
	       (LIST (DS-LINK.NEGPAR LINK) (DS-LINK.POSPAR LINK))
	       nil)
	   (CG=REPR_REMOVE COLOUR 'ALL LINK)
	   (CG=REPR_INSERT COLOUR 'REMOVED LINK)
	   (DS-LINK.DISCONNECT LINK))))
  
(DEFUN CG=REMOVE.UNIFIER (LINK UNIFIER RECOLOURFLAG REASON INFO)
  ; EDITED:  7-DEC-83 14:22:35                      NE
  ; INPUT:   A LINK OF THE ACTUAL GRAPH WITH OPERATIONAL
  ;          COLOUR, A UNIFIER OF LINK AND A BOOLEAN
  ;          VALUE.
  ;          REASON IS AN ATOM GIVING A MNEMONIC EXPLA-
  ;          NATION WHY THE REMOVAL OCCURS. INFO IS A
  ;          FURTHER SPECIFICATION IN A FORMAT DEPENDING
  ;          ON REASON. BOTH REASON IND INFO ARE USED
  ;          ONLY BY THE FUNCTIONS CALLED IN THE ADVISE
  ;          OF CG-TRACE.
  ; EFFECT:  UNIFIER IS REMOVED FROM THE UNIFIER LIST OF
  ;          LINK. IF ANY OTHER UNIFIERS REMAIN, LINK IS
  ;          MARKED AS CHANGED, OTHERWISE REMOVED FROM
  ;          THE ACTUAL GRAPH. IF RECOLOURFLAG = T
  ;          AND LINK HAS AN APPROPRIATE COLOUR, IT IS
  ;          RECOLOURED INSTEAD OF REMOVED.
  ; VALUE:   IF LINK WAS REMOVED//RECOLOURED AND HAS
  ;          A PURITY COLOUR, THEN A LIST OF
  ;          THE PARENT CLAUSES OF LINK, ELSE NIL.
  reason info
  (let* ((COLOUR (DS-LINK.COLOUR LINK))
	 (RULE (DS-LINK.RULE LINK))
	 (POSLITNO (DS-LINK.POSLITNO LINK))
	 (NEGPAR (if (MEMBER COLOUR (DS-LINK.COLOURS.WITH 'NEGPAR))
		     (DS-LINK.NEGPAR LINK)
		     (DS-LINK.POSPAR LINK)))
	 RECOLOUR OTHERLINKS NEW.LINK)
    (when (AND RECOLOURFLAG (SETQ RECOLOUR (CdR (ASSOC COLOUR CG*RECOLOUR))))
      (SETQ OTHERLINKS
	    (INTERSECTION (DS-CLAUSE.LINKS RECOLOUR (DS-LINK.POSPAR LINK) POSLITNO)
			  (DS-CLAUSE.LINKS RECOLOUR NEGPAR (DS-LINK.NEGLITNO LINK))))
      (COND ((AND OTHERLINKS (SETQ OTHERLINKS (MEMBER-IF #'(LAMBDA (OTHERLINK)
							     (AND (EQL RULE (DS-LINK.RULE OTHERLINK))
								  (EQL POSLITNO (DS-LINK.POSLITNO OTHERLINK))))
							 OTHERLINKS)))
	     (DS-LINK.PUTUNIFIERS (CAR OTHERLINKS) (CONS UNIFIER (DS-LINK.UNIFIERS (CAR OTHERLINKS)))))
	    (T (SETQ NEW.LINK
		     (DS-LINK.CREATE RECOLOUR (LIST UNIFIER) (DS-LINK.POSPAR LINK) POSLITNO NEGPAR (DS-LINK.NEGLITNO LINK)
				     (COND ((MEMBER COLOUR (DS-LINK.COLOURS.WITH 'POSFCT)) (DS-LINK.POSFCT LINK)))
				     (COND ((MEMBER COLOUR (DS-LINK.COLOURS.WITH 'NEGFCT)) (DS-LINK.NEGFCT LINK))) RULE))
	       (nar-narrow.info.put new.link (nar-narrow.info link))
	       (DS-LINK.CONNECT NEW.LINK)
	       (CG=REPR_INSERT RECOLOUR 'ALL NEW.LINK)
	       (CG=REPR_INSERT RECOLOUR 'INSERTED NEW.LINK))))
    (DS-LINK.REMOVE.UNIFIER LINK UNIFIER)
    (COND ((NULL (DS-LINK.UNIFIERS LINK)) (CG=REMOVE.LINK LINK 'UNIFIERREMOVAL NIL))
	  ((NOT (MEMBER LINK (CG=REPR_LIST COLOUR 'CHANGED))) (CG=REPR_INSERT COLOUR 'CHANGED LINK) NIL))))
  
(DEFMACRO CG=LINK_ANCESTOR.LINKS (LINK)
  ; EDITED: 23-JUN-83 14:33:04        MW
  ; INPUT:  A LINK ADDRESS
  ; VALUE:  LIST OF ANCESTOR LINKS OF LINK.
  `(DT-GETPROP ,LINK 'CG*ANCESTOR.LINKS))
  
(DEFMACRO CG=REPR_EMPTY.OBJECTLIST NIL
  ; EDITED:  4-MAY-83 14:48:50        NE
  ; VALUE:  REPRESENTATION OF AN EMPTY OBJECT LIST.
  `(PROG ((RESULT (LIST 0 0))) (RPLACA RESULT (CDR RESULT)) (RETURN RESULT)))
  
(DEFMACRO CG=REPR_RESET (OBJECTCLASS INDICATOR)
  ; EDITED: 18-MAR-83 14:30:51        NE
  ; INPUT:  TWO LITERAL ATOMS
  ; EFFECT: INITIALIZES THE OBJECT LIST REFERENCED BY
  ;         THE TWO ATOMS WITH THE EMPTY LIST.
  ; VALUE:  UNDEFINED.
  `(CG=REPR_PUT ,OBJECTCLASS ,INDICATOR (CG=REPR_EMPTY.OBJECTLIST)))
  
(DEFMACRO CG=REPR_INSERT (&OPTIONAL OBJECTCLASS INDICATOR OBJECT)
  ; EDITED: 18-MAR-83 14:30:54        NE
  ; INPUT:  TWO LITERAL ATOMS AND AN S-EXPRESSION.
  ; EFFECT: INSERTS OBJECT INTO THE OBJECT LIST
  ;         REFERENCED BY THE TWO ATOMS.
  ; VALUE:  UNDEFINED.
  `(let ((OBJECTLIST (CG=REPR_GET ,OBJECTCLASS ,INDICATOR))
	 (NEW.CELL (LIST ,OBJECT)))
     (incf (second objectlist))
     (RPLACD (CAR OBJECTLIST) NEW.CELL)
     (RPLACA OBJECTLIST NEW.CELL)))
  
(DEFMACRO CG=REPR_REMOVE (&OPTIONAL OBJECTCLASS INDICATOR OBJECT)
						; EDITED: 18-MAR-83 14:30:59        NE
						; INPUT:  TWO LITERAL ATOMS AND AN S-EXPRESSION.
						; EFFECT: REMOVES THE FIRST OCCURRENCE OF OBJECT FROM
						;         THE OBJECT LIST REFERENCED BY THE TWO ATOMS.
						; VALUE:  UNDEFINED.
  `(let ((OBJ ,OBJECT)
	 (OBJECTLIST (CG=REPR_GET ,OBJECTCLASS ,INDICATOR))
	 POINTER1 POINTER2)
     (SETQ POINTER1 (CDR OBJECTLIST))
     (SETQ POINTER2 (CDR POINTER1))
						;INVARIANT: POINTER2=CDR(POINTER1) AND ALL ELEMENTS
						;           BEFORE THE POINTER2-TH ARE NEQ TO OBJECT.
     (WHILE (AND POINTER2
		 (NEQ OBJ (CAR POINTER2))
		 (SETQ POINTER1 POINTER2)
		 (setq POINTER2 (CDR POINTER2))))
     (COND (POINTER2
	    (COND ((EQ POINTER2 (CAR OBJECTLIST))
						; LAST CELL, UPDATE CAR OBJECTLIST
		   (RPLACA OBJECTLIST POINTER1)))
	    (RPLACD POINTER1 (CDR POINTER2))
	    (decf (second OBJECTLIST))))))
  
(DEFMACRO CG=REPR_LIST (&OPTIONAL OBJECTCLASS INDICATOR)
  ; EDITED: 18-MAR-83 14:47:11        NE
  ; INPUT:  TWO LITERAL ATOMS REFERENCING AN OBJECT LIST
  ; VALUE:  LIST OF ALL OBJECTS STORED IN THE OBJECT
  ;         LIST.
  `(CDDR (CG=REPR_GET ,OBJECTCLASS ,INDICATOR)))
  
(DEFMACRO CG=REPR_LENGTH (&OPTIONAL OBJECTCLASS INDICATOR)
  ; EDITED: 18-MAR-83 14:47:12        NE
  ; INPUT:  TWO LITERAL ATOMS REFERENCING AN OBJECT LIST
  ; VALUE:  NUMBER OF OBJECTS STORED IN THE OBJECT LIST.
  `(CADR (CG=REPR_GEt ,OBJECTCLASS ,INDICATOR)))
  
(DEFUN CG=REPR_SAVE NIL
  ; EDITED:  4-MAY-83 14:52:23        NE
  ; INPUT:  NONE.
  ; VALUE:  AN S-EXPRESSION WHICH, WHEN EVALUATED,
  ;         RESTORES THE OBJECT LISTS OF ALL OBJECT
  ;         CLASSES.
  ; NOTE:   SAVES NO COPIES, BUT THE ACTUAL LISTS]
  (SUBPAIR (LIST (SYMBOL-PLIST 'CG*OBJECTS.ALL)
		 (SYMBOL-PLIST 'CG*OBJECTS.INSERTED)
		 (SYMBOL-PLIST 'CG*OBJECTS.CHANGED)
		 (SYMBOL-PLIST 'CG*OBJECTS.REMOVED))
	   '(***ALL.PROPLIST*** ***INSERTED.PROPLIST*** ***CHANGED.PROPLIST*** ***REMOVED.PROPLIST***)
	   '(PROG
	      ((ALL.PROPLIST '***ALL.PROPLIST***)
	       (INSERTED.PROPLIST '***INSERTED.PROPLIST***)
	       (CHANGED.PROPLIST '***CHANGED.PROPLIST***)
	       (REMOVED.PROPLIST '***REMOVED.PROPLIST***))
	      (SMAPC #'(LAMBDA (OBJECTLIST) (RPLACA OBJECTLIST (LAST OBJECTLIST))) (FUNCTION CDDR) (CDR ALL.PROPLIST))
	      (SETF (SYMBOL-PLIST 'CG*OBJECTS.ALL) ALL.PROPLIST)
	      (SMAPC #'(LAMBDA (OBJECTLIST) (RPLACA OBJECTLIST (LAST OBJECTLIST))) (FUNCTION CDDR) (CDR INSERTED.PROPLIST))
	      (SETF (SYMBOL-PLIST 'CG*OBJECTS.INSERTED) INSERTED.PROPLIST)
	      (SMAPC #'(LAMBDA (OBJECTLIST) (RPLACA OBJECTLIST (LAST OBJECTLIST))) (FUNCTION CDDR) (CDR CHANGED.PROPLIST))
	      (SETF (SYMBOL-PLIST 'CG*OBJECTS.CHANGED) CHANGED.PROPLIST)
	      (SMAPC #'(LAMBDA (OBJECTLIST) (RPLACA OBJECTLIST (LAST OBJECTLIST))) (FUNCTION CDDR) (CDR REMOVED.PROPLIST))
	      (SETF (SYMBOL-PLIST 'CG*OBJECTS.REMOVED) REMOVED.PROPLIST))))
  
(DEFUN CG=REPR_INIT NIL
  ; EDITED_  4-MAY-83 14:41:37        NE
  ; INPUT:  CG*OBJECTCLASSES IS A LIST OF ALL ADMISSIBLE
  ;         OBJECT CLASSES. FOR EFFICIENCY REASONS THEY
  ;         SHOULD BE ORDERED BY DECREASING ACCESS
  ;         FREQUENCY.
  ; EFFECT: ALL OBJECT LISTS REFERENCED BY ANY OF THE
  ;         GIVEN OBJECTCLASSES ARE RESET.
  ; VALUE:  UNDEFINED.
  (MAPC #'(LAMBDA (INDICATOR)
	    (SETF (SYMBOL-PLIST INDICATOR)
		  (MAPCAN #'(LAMBDA (OBJECTCLASS) (LIST OBJECTCLASS (CG=REPR_EMPTY.OBJECTLIST))) CG*OBJECTCLASSES)))
	'(CG*OBJECTS.ALL CG*OBJECTS.INSERTED CG*OBJECTS.CHANGED CG*OBJECTS.REMOVED)))
  
(DEFMACRO CG=REPR_GET (&OPTIONAL OBJECTCLASS INDICATOR)
						; EDITED: 18-MAR-83 14:47:12        NE
						; INPUT:  TWO LITERAL ATOMS REFERENCING AN OBJECT LIST
						; VALUE:  THE ENTIRE OBJECT LIST REFERENCED BY THE
						;         ATOMS.
  (cond ((constantp indicator)
	 `(GET ',(INTERN (CONCAtenate 'string  "CG*OBJECTS." (string (cadr INDICATOR)))
			 (find-package "MKRP")) ,OBJECTCLASS))
	(t `(get (intern (concatenate 'string "CG*OBJECTS." ,(string indicator))
			 (find-package "MKRP")) ,objectclass))))
  
						; (CG=repr_get 'R 'removed)
  
  
(DEFMACRO CG=REPR_PUT (&OPTIONAL OBJECTCLASS INDICATOR OBJECTLIST)
  ; EDITED: 18-MAR-83 14:47:12        NE
  ; INPUT:  TWO LITERAL ATOMS AND   AN OBJECT LIST
  ; EFFECT: HENCEFORTH THE ATOMS REFERENCE THE GIVEN
  ;         OBJECT LIST.
  ; VALUE:  UNDEFINED.
  (cond ((and (listp indicator) (eq (car indicator) 'quote))
	 `(setf (get ',(INTERN (CONCAtenate 'string  "CG*OBJECTS." (string (cadr INDICATOR)))
			       (find-package "MKRP")) ,OBJECTCLASS) ,OBJECTLIST))
	(t `(setf (get (intern (concatenate 'string  "CG*OBJECTS." ,(string indicator))
			       (find-package "MKRP")) ,objectclass) ,objectlist))))
  
(DEFVAR CG*CHANGE.QUEUE.LITERALS (LIST NIL))
  
(DEFMACRO CG-CHANGE.QUEUE_APPEND (ELEMENT)
  ; EDITED: 17-MAR-84 22:29:42
  ; INPUT:  'ELEMENT' HAS ONE OF THE XI FORMATS IN
  ;         'COMS'.
  ; EFFECT: 'ELEMENT' IS INSERTED AT THE END OF
  ;         CHANGEQUEUE.
  ; VALUE:  NEW CHANGEQUEUE.
  `(CAR (QCONC1 CG*CHANGE.QUEUE.LITERALS ,ELEMENT)))
  
(DEFMACRO CG-CHANGE.QUEUE_TOP NIL
  ; EDITED: 17-MAR-84 22:33:42
  ; INPUT:  -
  ; EFFECT: -
  ; VALUE:  THE FIRST ELEMENT OF CHANGEQUEUE.
  `(CAAR CG*CHANGE.QUEUE.LITERALS))
  
(DEFMACRO CG-CHANGE.QUEUE_POP NIL
  ; EDITED: 17-MAR-84 22:33:42
  ; INPUT:  -
  ; EFFECT: THE FIRST ELEMENT OF CHANGEQUEUE WILL BE
  ;         REMOVED.
  ; VALUE:  THE REMOVED ELEMENT.
  `(PROG1 (CAAR CG*CHANGE.QUEUE.LITERALS) (QDELETE-NTH CG*CHANGE.QUEUE.LITERALS 1)))
  
(DEFMACRO CG-CHANGE.QUEUE NIL ; EDITED: 17-MAR-84 22:38:61
  ; INPUT:  -
  ; EFFECT: -
  ; VALUE:  THE WHOLE CHANGEQUEUE.
  `(CAR CG*CHANGE.QUEUE.LITERALS))
  
(defvar cg*position 0)
  
(DEFmacro CG-DUMP (CG.FILE CG.COMMANDS &optional (begin.position 0))
  "The actual graph is printed according to
             the commands as follows:
             (MESSAGE S1 ... SN)
             The SI are evaluated and their values
             printed into one line separated by blanks.
             (SEPARATOR)
             If this command occurs, the dump is enclosed
             between a heading and a footing line.
             (CLAUSES C FORMAT)
             C is either one of the atoms ALL, INSERTED,
             REMOVED, CHANGED, or evaluates to a list of
             clauses. FORMAT is in (I L LR N A) meaning
             the formats IMPLEMENTATIONAL, LOGICAL,
             LOGICAL with variable renaming, pnames, or
             addresses. Other INPUTS ARE EVALUATED.
             IF C IS ONE OF THE GIVEN ATOMS, THE RESPEC-
             TIVE CLAUSES OF THE ACTUAL GRAPH ARE COMPU-
             TED, OTHERWISE C IS EVALUATED.
             THE RESULTING CLAUSES ARE PRINTED IN THE
             GIVEN FORMAT.
             (LINKS L COLOURS FORMAT)
             L IS EITHER ONE OF THE ATOMS ALL, INSERTED,
             REMOVED, CHANGED, OR EVALUATES TO A LIST OF
             LINKS. COLOURS EVALUATES TO A LIST OF LINK
             COLOURS. COLOURS=NIL MEANS ALL COLOURS.
             FORMAT IS IN (I A) MEANING THE FORMATS
             IMPLEMENTATIONAL OR ADDRESS LIST.
             OTHER INPUTS ARE EVALUATED.
             IF L IS ONE OF THE GIVEN ATOMS, THE RESPEC-
             TIVE LINKS OF THE ACTUAL GRAPH ARE COMPUTED,
             OTHERWISE L IS EVALUATED.
             THE RESULTING LINKS ARE PRINTED CLUSTERED
             BY THE GIVEN COLOURS IN THE GIVEN FORMAT."
    `(cg==dump ,CG.FILE ,CG.COMMANDS ,begin.position))
  
(defun cg==dump (CG.FILE CG.COMMANDS begin.position)
  ; EDITED:  9-MAR-83 19:02:35        NE
  ; INPUT:  A FILE NAME AND ARBITRARILY MANY COMMANDS
  ;         AS SPECIFIED UNDER EFFECTS.
  ; EFFECT: THE ACTUAL GRAPH IS PRINTED ACCORDING TO
  ;         THE COMMANDS AS FOLLOWS:
  ;         (MESSAGE S1 ... SN)
  ;         THE SI ARE EVALUATED AND THEIR VALUES
  ;         PRINTED INTO ONE LINE SEPARATED BY BLANKS.
  ;         (SEPARATOR)
  ;         IF THIS COMMAND OCCURS, THE DUMP IS ENCLOSED
  ;         BETWEEN A HEADING AND A FOOTING LINE.
  ;         (CLAUSES C FORMAT)
  ;         C IS EITHER ONE OF THE ATOMS ALL, INSERTED,
  ;         REMOVED, CHANGED, OR EVALUATES TO A LIST OF
  ;         CLAUSES. FORMAT IS IN (I L LR N A) MEANING
  ;         THE FORMATS IMPLEMENTATIONAL, LOGICAL,
  ;         LOGICAL WITH VARIABLE RENAMING, PNAMES, OR
  ;         ADDRESSES. OTHER INPUTS ARE EVALUATED.
  ;         IF C IS ONE OF THE GIVEN ATOMS, THE RESPEC-
  ;         TIVE CLAUSES OF THE ACTUAL GRAPH ARE COMPU-
  ;         TED, OTHERWISE C IS EVALUATED.
  ;         THE RESULTING CLAUSES ARE PRINTED IN THE
  ;         GIVEN FORMAT.
  ;         (LINKS L COLOURS FORMAT)
  ;         L IS EITHER ONE OF THE ATOMS ALL, INSERTED,
  ;         REMOVED, CHANGED, OR EVALUATES TO A LIST OF
  ;         LINKS. COLOURS EVALUATES TO A LIST OF LINK
  ;         COLOURS. COLOURS=NIL MEANS ALL COLOURS.
  ;         FORMAT IS IN (I A) MEANING THE FORMATS
  ;         IMPLEMENTATIONAL OR ADDRESS LIST.
  ;         OTHER INPUTS ARE EVALUATED.
  ;         IF L IS ONE OF THE GIVEN ATOMS, THE RESPEC-
  ;         TIVE LINKS OF THE ACTUAL GRAPH ARE COMPUTED,
  ;         OTHERWISE L IS EVALUATED.
  ;         THE RESULTING LINKS ARE PRINTED CLUSTERED
  ;         BY THE GIVEN COLOURS IN THE GIVEN FORMAT.
  ; VALUE:  UNDEFINED.
  (let (SEPARATORFLAG)    
    (setq cg*position begin.position)
    (setq cg.commands (delete nil cg.commands))
    (MAPC #'(LAMBDA (COMMAND)
	      (unless (MEMBER (CAR COMMAND) '(MESSAGE SEPARATOR CLAUSES LINKS))
		(cerror "ignore" "Wrong command: ~A" command) nil))
	  CG.COMMANDS)
    (SETQ CG.COMMANDS
	  (MAPCAN
	    #'(LAMBDA (COMMAND)
		(PROG ((COM (CAR COMMAND)) (ARGUMENTS (CDR COMMAND)) NEWCOMMANDS)
		      (CASE COM
			(MESSAGE (SETQ NEWCOMMANDS (LIST (CONS 'MESSAGE ARGUMENTS))))
			(SEPARATOR (SETQ SEPARATORFLAG T) (SETQ NEWCOMMANDS NIL))
			(CLAUSES
			  (let ((C (CAR ARGUMENTS))
				(FORMAT (SECOND ARGUMENTS)) HEADER)
			    (COND ((NOT (MEMBER FORMAT '(I L LR N A))) (SETQ FORMAT (EVAL FORMAT))))
			    (CASE C
			      (ALL (SETQ C (CG-CLAUSES ALL)) (SETQ HEADER "ACTUAL"))
			      (INSERTED (SETQ C (CG-CLAUSES INSERTED)) (SETQ HEADER "INSERTED"))
			      (REMOVED (SETQ C (CG-CLAUSES REMOVED)) (SETQ HEADER "REMOVED"))
			      (CHANGED (SETQ C (CG-CLAUSES CHANGED)) (SETQ HEADER "CHANGED"))
			      (OTHERWISE (PROGN (SETQ HEADER ""))))
			    (SETQ NEWCOMMANDS (LIST (LIST 'MESSAGE "  " HEADER "CLAUSES:") '(MESSAGE "")
						    (LIST 'CLAUSES C FORMAT)))))
			(LINKS
			  (PROG
			    ((L (CAR ARGUMENTS))
			     (COLOURS (SECOND ARGUMENTS))
			     (FORMAT (THIRD ARGUMENTS))
			     HEADER)
			    (COND ((NULL COLOURS) (SETQ COLOURS (DS-LINK.COLOURS.FOR 'ALL)))
				  ((AND (ATOM COLOURS) (MEMBER COLOURS (DS-LINK.COLOURS.FOR 'ALL)))
				   (SETQ COLOURS (LIST COLOURS)))
				  ((AND (CONSP COLOURS) (MEMBER (CAR COLOURS) (DS-LINK.COLOURS.FOR 'ALL))) NIL))
			    (COND ((NOT (MEMBER FORMAT '(I A))) (SETQ FORMAT (EVAL FORMAT))))
			    (CASE L
			      (ALL (SETQ L (CG-LINKS COLOURS 'ALL)) (SETQ HEADER "ACTUAL"))
			      (INSERTED (SETQ L (CG-LINKS COLOURS 'INSERTED)) (SETQ HEADER "INSERTED"))
			      (REMOVED (SETQ L (CG-LINKS COLOURS 'REMOVED)) (SETQ HEADER "REMOVED"))
			      (CHANGED (SETQ L (CG-LINKS COLOURS 'CHANGED)) (SETQ HEADER "CHANGED"))
			      (OTHERWISE (PROGN (SETQ HEADER ""))))
			    (SETQ NEWCOMMANDS
				  (LIST (LIST 'MESSAGE "  " HEADER "LINKS:") '(MESSAGE "") (LIST 'LINKS L COLOURS FORMAT)))))
			(OTHERWISE (SETQ NEWCOMMANDS NIL)))
		      (RETURN NEWCOMMANDS)))
	    CG.COMMANDS))
    (CG=DUMP CG.FILE CG.COMMANDS SEPARATORFLAG)))
  
(DEFUN CG=DUMP (FILE COMMANDS SEPARATORFLAG)
						; EDITED:  9-MAR-83 18:26:02        NE
						; INPUT:  A FILE NAME AND A LIST OF COMMANDS AS
						;         SPECIFIED UNDER EFFECT, AND A BOOLEAN VALUE.
						; EFFECT: THE ACTUAL GRAPH IS PRINTED ON FILE ACCORD-
						;         ING TO THE COMMANDS. IF FILE IS OPEN, IT
						;         REMAINS OPEN AFTER THE DUMP, OTHERWISE IT
						;         IS OPENED AT THE BEGINNING AND CLOSED AT
						;         THE END OF THE DUMP.
						;         IF SEPERATORFLAG=T, THE DUMP IS ENCLOSED
						;         BETWEEN A HEADING AND A FOOTING LINE.
						;         COMMANDS AND THEIR EFFECTS:
						;         (MESSAGE S1 ... SN)
						;         THE SI ARE PRINTED INTO ONE LINE SEPARATED
						;         BY BLANKS.
						;         (CLAUSES C FORMAT)
						;         C IS A LIST OF CLAUSES, FORMAT AS DESCRIBED
						;         IN CG-DUMP. THE CLAUSES ARE PRINTED IN THE
						;         GIVEN FORMAT.
						;         (LINKS L COLOURS FORMAT)
						;         L IS A LIST OF LINKS, COLOURS AND FORMAT AS
						;         DESCRIBED IN CG-DUMP.
						;         THE LINKS ARE PRINTED CLUSTERED BY THE GIVEN
						;         COLOURS.
						; VALUE:  UNDEFINED.
  (when (eq t file) (setq file *terminal-io*))
  (let ((CLOSEDFLAG (NOT (mkrp-OUTSTREAMP FILE))) LINELENGTH.BEFORE)
    (COND (CLOSEDFLAG (SETQ FILE (mkrp-OPENOUT FILE nil))
		      (SETQ LINELENGTH.BEFORE (LINELENGTH 117 FILE))))
    (when SEPARATORFLAG				; HEADER
      (format FILE "~%***** CG-DUMP START ~A " (DATE))
      (format FILE "~%~%"))
    (PROGN					; INTERPRET COMMANDS
      (MAPC #'(LAMBDA (COMMAND)
		(CASE (CAR COMMAND)
		  (MESSAGE (fresh-line file) (MAPPRINT (CDR COMMAND) NIL NIL " " 'PRINC FILE) (TERPRI FILE))
		  (CLAUSES
		    (let ((CLAUSES (SECOND COMMAND))
			  (FORMAT (THIRD COMMAND)))
		      (CG=PR_CLAUSES CLAUSES FILE FORMAT)
		      (TERPRI FILE)))
		  (LINKS
		    (PROG ((LINKS (SECOND COMMAND))
			   (COLOURS (THIRD COMMAND))
			   (FORMAT (FOURTH COMMAND)))
			  (CG=PR_LINKS LINKS FILE COLOURS FORMAT)
			  (TERPRI FILE)))
		  (OTHERWISE
		    (format file "????? Command unknown for CG=DUMP: ~A~%" COMMAND))))
	    COMMANDS))
    (COND
      (SEPARATORFLAG				; FOOTING
       (PRINC "***** CG-DUMP END   " FILE) (PRINC (DATE) FILE)
       (TERPRI FILE) (TERPRI FILE)))
    (COND (CLOSEDFLAG (LINELENGTH LINELENGTH.BEFORE FILE) (CLOSEFILE FILE)))))
  
(defmacro CG-DUMP.SHORT (CG.FILE CG.FORMAT.REMOVED CG.FORMAT.INSERTED CG.FORMAT.CHANGED CG.FORMAT.ALL
			 &optional (begin.position 0))
    `(cg==dump.short ,CG.FILE ',CG.FORMAT.REMOVED ',CG.FORMAT.INSERTED ',CG.FORMAT.CHANGED ',CG.FORMAT.ALL ',begin.position))
  
(DEFUN CG==dUMP.SHORT (CG.FILE CG.FORMAT.REMOVED CG.FORMAT.INSERTED CG.FORMAT.CHANGED CG.FORMAT.ALL begin.position)
    ; EDITED:  5-MAY-83 14:17:39        NE
    ; INPUT:  NAME OF AN OPEN FILE AND FOUR ATOMS.
    ; EFFECT: THE RESPECTIVE CLAUSE LISTS ARE PRINTED IN
    ;         THE GIVEN FORMATS, PROVIDED THEY ARE MEMBERS
    ;         OF (I L LR N A).
    ; VALUE:  UNDEFINED.
    (setq cg*position (+ 10 begin.position))
    (CG=DUMP.SHORT CG.FILE "Removed:  " CG.FORMAT.REMOVED (CG-CLAUSES REMOVED))
    (setq cg*position (+ 10 begin.position))
    (CG=DUMP.SHORT CG.FILE "Inserted: " CG.FORMAT.INSERTED (CG-CLAUSES INSERTED))
    (setq cg*position (+ 10 begin.position))
    (CG=DUMP.SHORT CG.FILE "Changed:  " CG.FORMAT.CHANGED (CG-CLAUSES CHANGED))
    (setq cg*position (+ 10 begin.position))
    (CG=DUMP.SHORT CG.FILE "All:      " CG.FORMAT.ALL (CG-CLAUSES ALL)))
  
(DEFUN CG=DUMP.SHORT (FILE HEADER FORMAT CLAUSELIST)
    ; EDITED:  5-MAY-83 14:30:59        NE
    ; INPUT:  NAME OF AN OPEN FILE, A STRING, AN ATOM,
    ;         AND A LIST OF CLAUSES.
    ; EFFECT: IF CLAUSELIST IS NOT NIL AND FORMAT IS IN
    ;         (I L LR N A), THE HEADER IS PRINTED ON FILE
    ;         FOLLOWED BY THE CLAUSES IN THE GIVEN FORMAT.
    ; VALUE:  UNDEFINED.
    (when (AND CLAUSELIST (MEMBER FORMAT '(I L LR N A)))
      (PRINC HEADER FILE) (CG=PR_CLAUSES CLAUSELIST FILE FORMAT)))
  
(DEFUN CG=PR_CLAUSES (CLAUSES FILE FORMAT)
									      ; EDITED:  9-MAR-83 17:57:14        NE
									      ; INPUT:  A LIST OF CLAUSE ADDRESSES, A FILE NAME,
									      ;         AND EITHER OF THE ATOMS I,L,LR,N,A.
									      ; REMARK: OTHER VALUES OF FORMAT ARE CONVERTED TO LR.
									      ; EFFECT: CLAUSES ARE PRINTED ON FILE IN IMPLEMENTA-
									      ;         TIONAL FORM, LOGICAL FORM, LOGICAL FORM
									      ;         WITH VARIABLE RENAMING, AS A LIST OF PNAMES
									      ;         OR A LIST OF ADDRESSES, DEPENDING ON FORMAT.
									      ; VALUE:  UNDEFINED.
  (COND ((NOT (MEMBER FORMAT '(I L LR N A)))				      ; USE LR AS DEFAULT
	 (SETQ FORMAT 'LR)))
  (PROG ((FIRSTPOS (if (< cg*position 6) 6 cg*position)))
	(CASE FORMAT
	  (I (MAPC #'(LAMBDA (CLAUSE) (CG=PR_CLAUSE CLAUSE FIRSTPOS FILE)) CLAUSES))
	  (L (MAPC #'(LAMBDA (CLAUSE) (CG=PR_CLAUSE.SHORT CLAUSE FIRSTPOS FILE NIL)) CLAUSES))
	  (LR (MAPC #'(LAMBDA (CLAUSE) (CG=PR_CLAUSE.SHORT CLAUSE FIRSTPOS FILE T)) CLAUSES))
	  (N (CG=PR_LIST (DS-PNAME CLAUSES) FIRSTPOS FILE) (TERPRI FILE))
	  (A (CG=PR_LIST CLAUSES FIRSTPOS FILE)
	     (TERPRI FILE))
	  (OTHERWISE NIL))))
  
(DEFUN CG=PR_CLAUSE (CLAUSE FIRSTPOS FILE)
    ; EDITED: 02-MAR-83 15:01:06        NE
    ; INPUT:  A CLAUSE ADDRESS, A NATURAL NUMBER, AND A
    ;         FILE NAME.
    ; EFFECT: THE COMPONENTS OF CLAUSE ARE PRINTED ON FILE
    ;         IN A READABLE FORMAT, INDENTED TO FIRSTPOS.
    ; VALUE:  UNDEFINED.
    (PROG
      ((PNAME (DS-CLAUSE.PNAME CLAUSE))
       (width.clause.and.pname 10)
       (NOLIT (DS-CLAUSE.NOLIT CLAUSE))
       (width.nolit 2)
       (DEPTH (DS-CLAUSE.DEPTH CLAUSE))
       (width.depth 2)
       (PARENTS (DS-CLAUSE.PARENTS CLAUSE))
       (width.parents 10)
       (VARIABLES (DS-CLAUSE.VARIABLES CLAUSE))
       (POS.VARIABLES 62)
       (RENAMING (DS-CLAUSE.RENAMING CLAUSE))
       (PROPLIST (DT-GETPROPLIST CLAUSE))
       (ATTRIBUTES (DS-CLAUSE.ATTRIBUTES CLAUSE))
       (POT.FALSE.LITNOS (DS-CLAUSE.POTENTIALLY.FALSE.LITNOS CLAUSE))
       (LINKLIST NIL)
       (POT.TRUE.LITNOS (DS-CLAUSE.POTENTIALLY.TRUE.LITNOS CLAUSE))
       (pos.relative (+ 11 firstpos))
       (pos.linklist (+ firstpos 11))
       (POS.LITERALS (+ firstpos 11)))
      (let ((output (format nil "~VA~vA Nolit:~V@A Depth:~V@A Parents:~VA Variables: "
			    (- firstpos cg*position) ""	; Leading blanks
			    width.clause.and.pname (format nil "~A<~A>" pname clause)	; NAME 
			    width.nolit nolit	; NOLIT
			    width.depth depth	; DEPTH
			    width.parents (DS-PNAME PARENTS))))	; Parents
	(princ output file)
	(incf cg*position (length output)))
      (CG=PR_VARIABLES VARIABLES pos.variables FILE NIL)
      (setq cg*position 0)
      (when renaming				; Renaming
	(let ((output (format nil "~%~VARenaming:   " pos.relative "")))
	  (princ output file)
	  (incf cg*position (length output))
	  (CG=PR_LIST (DS-PNAME RENAMING) cg*position FILE)))
      (when proplist				; Proplist
	(let ((output (format nil "~%~VAProperties: " pos.relative "")))
	  (princ output file)
	  (setq cg*position (1- (length output)))
	  (CG=PR_LIST PROPLIST cg*position FILE)))
      (when attributes				; Attributes
	(let ((output (format nil "~%~VAAttributes: " pos.relative "")))
	  (princ output file)
	  (setq cg*position (1- (length output)))
	  (CG=PR_LIST ATTRIBUTES cg*position FILE)))
      (when pot.false.litnos			; pot.false.litnos
	(let ((output (format nil "~%~VANumbers of potentially false literals: " pos.relative "")))
	  (princ output file)
	  (setq cg*position (1- (length output)))
	  (CG=PR_LIST pot.false.litnos cg*position FILE)))
      (when pot.true.litnos			; pot.true.litnos
	(let ((output (format nil "~%~VANumbers of potentially true literals: " pos.relative "")))
	  (princ output file)
	  (setq cg*position (1- (length output)))
	  (CG=PR_LIST pot.true.litnos cg*position FILE)))
      (setq cg*position 0)
      (fresh-line FILE)				; Link lists
      (MAPC #'(LAMBDA (COLOUR)
		(SETQ LINKLIST (DS-CLAUSE.ALL.LINKS COLOUR CLAUSE))
		(COND
		  (LINKLIST
		   (COND
		     ((< cg*position POS.LINKLIST)	; FIRST LINK LIST TO BE PRINTED 
		      (FORMAT FILE "~T" POS.LINKLIST))
		     ((EQL cg*position POS.LINKLIST)	; POSITION OK
		      )
		     ((NOT (> (+ cg*position 3 (PRINT-LENGTH COLOUR NIL) (PRINT-LENGTH LINKLIST NIL)) 116))
		      ; OUTPUT FITS INTO SAME LINE 
		      (SPACES 3 FILE))
		     (T (FORMAT FILE "~T" POS.LINKLIST)))
		   (PRINC COLOUR FILE) (PRINC ": " FILE)
		   (CG=PR_LIST LINKLIST cg*position FILE))))
	    (DS-LINK.COLOURS.FOR 'CLAUSE))
      (DOtimes (litno NOLIT)			; LITERALS
	(fresh-line file)
	(setq cg*position 0)
	(CG=PR_LITERAL CLAUSE (1+ LITNO) POS.LITERALS FILE))
      (fresh-line FILE)
      (setq cg*position 0)))
  
  
(DEFUN CG=PR_LITERAL (CLAUSE LITNO FIRSTPOS FILE)
    ; EDITED: 25-FEB-83 17:37:25        NE
    ; INPUT:  A CLAUSE ADDRESS, THE NUMBER OF A LITERAL IN
    ;         CLAUSE, A NATURAL NUMBER, AND A FILE NAME.
    ; EFFECT: THE COMPONENTS OF THE RESPECTIVE LITERAL ARE
    ;         PRINTED ON FILE IN A READABLE FORMAT. THE
    ;         OUTPUT IS INDENTED TO POSITION FIRSTPOS.
    ; VALUE:  UNDEFINED.
    (PROG
      ((SIGN (DS-CLAUSE.SIGN CLAUSE LITNO))
       (PREDICATE (DS-CLAUSE.PREDICATE CLAUSE LITNO))
       (TERMLIST (DT-PNAME (DS-CLAUSE.TERMLIST CLAUSE LITNO)))
       (VARIABLES (DS-CLAUSE.LIT.VARIABLES CLAUSE LITNO))
       (PROPLIST (DS-CLAUSE.LIT.GETPROPLIST CLAUSE LITNO))
       (LINKLIST NIL)
       (POS.RELATIVE (+ FIRSTPOS 3))
       (POS.LINKLIST (+ FIRSTPOS 3))
       (LASTPOS 116))
      (LET ((OUTPUT (FORMAT NIL "~vA~D  "
			    (- FIRSTPOS CG*POSITION) "" LITNO)))	; LITERAL NUMBER
	(PRINC OUTPUT FILE)
	(INCF CG*POSITION (LENGTH OUTPUT)))
      (pp-print.literal (list SIGN (cons predicate termlist)) file :current.pos cg*position)
      (when (ds-clause.lit.is.max clause litno) (princ "[*]"))
      (setq cg*position 0)
      (terpri file)
      (when variables
	(LET ((OUTPUT (FORMAT NIL "~vAVariables: " pos.relative "")))
	  (PRINC OUTPUT FILE)
	  (setq CG*POSITION (1- (LENGTH OUTPUT))))
	(CG=PR_VARIABLES VARIABLES CG*POSITION FILE NIL))
      (when (and variables proplist) (terpri file) (setq cg*position 0))
      (WHEN PROPLIST				; Proplist
	(LET ((OUTPUT (FORMAT NIL "~VAProperties: " POS.RELATIVE "")))
	  (PRINC OUTPUT FILE)
	  (setq CG*POSITION (1- (LENGTH OUTPUT)))
	  (CG=PR_LIST PROPLIST CG*POSITION FILE)))
      (PROGN ; LINKLISTS
	(MAPC #'(LAMBDA (COLOUR)
		  (SETQ LINKLIST (DS-CLAUSE.LINKS COLOUR CLAUSE LITNO))
		  (WHEN LINKLIST
		    (COND
		      ((< CG*POSITION POS.LINKLIST)
		       ; FIRST LINK LIST TO BE PRINTED
		       (FORMAT FILE "~VA" (- POS.LINKLIST cg*position) "")
		       (SETQ CG*POSITION POS.LINKLIST))
		      ((EQL CG*POSITION POS.LINKLIST))	; POSITION OK
		      ((NOT (> (+ CG*POSITION 3 (PRINT-LENGTH COLOUR NIL) (PRINT-LENGTH LINKLIST NIL)) LASTPOS))
		       ; OUTPUT FITS INTO SAME LINE 
		       (SPACES 3 FILE))
		      (T (FORMAT FILE "~%~VA" POS.LINKLIST "")
			 (SETQ CG*POSITION POS.LINKLIST)))
		    (LET ((OUTPUT (FORMAT NIL "~A: " COLOUR)))
		      (PRINC OUTPUT FILE)
		      (INCF CG*POSITION (LENGTH OUTPUT))
		      (CG=PR_LIST LINKLIST CG*POSITION FILE))))
	      (DS-LINK.COLOURS.FOR 'LITERALS)))))
  
(defun cg=pr_clause.short (clause firstpos file variable.rename.flag)
						; edited: 2-mar-83 15:09:06  ne
						; input:  a clause address, a natural number, a file name, and a boolean value.
						; effect: prints the clause in a short readable format on file,
						;         indented to firstpos.
						; value:  undefined.
  (let ((width.empty (- firstpos cg*position))
	(variables (ds-clause.variables clause))
	variable.printnames lit.pos)
    (progn					; pname
      (let ((output (if  variable.rename.flag
			 (format nil "~va~a: " width.empty "" (ds-clause.pname clause))
			 (format nil "~va~a<~A>: " width.empty "" (ds-clause.pname clause) clause))))
	(princ output file)
	(incf cg*position (length output))))
    (setq lit.pos (+ cg*position 2))
    (when variables				; variables
      (format file "All ")
      (incf cg*position 4)
      (setq variable.printnames (cg=pr_variables variables cg*position file variable.rename.flag))
      (spaces 2 file)
      (incf cg*position 2))
    (progn (if (> (length variables) 5)		; Literals
	       (format file "~%~vA" lit.pos "")
	       (setq lit.pos cg*position))
	   (pp-print.literals (cg=pr_list.literals clause variables variable.printnames) file :current.pos lit.pos)))
  (terpri file)
  (setq cg*position 0))
  
(defun cg=pr_list.literals (clause variables variable.printnames)
  (do ((litno (ds-clause.nolit clause) (1- litno))
       (literals nil (cons (list (DS-CLAUSE.SIGN CLAUSE LITNO)
				 (cons (dt-pname (DS-CLAUSE.PREDICATE CLAUSE LITNO))
				       (DS-PNAME (SUBPAIR VARIABLE.PrintNAMES VARIABLES
							  (DS-CLAUSE.TERMLIST CLAUSE LITNO)))))
			   literals)))
      ((zerop litno) literals)))
  
(DEFUN CG=PR_LINKS (LINKS FILE COLOURS FORMAT)
						; EDITED:  9-MAR-83 18:06:42        NE                
						; INPUT:  A LIST OF LINK ADDRESSES, A FILE NAME, A    
						;         LIST OF LINK COLOURS, E MEMBER OF (I A).    
						; REMARK: OTHER VALUES OF COLOURS ARE CONVERTED TO    
						;         A LIST OF ALL POSSIBLE LINK COLOURS.        
						;         OTHER FORMATS ARE CONVERTED TO A.           
						; EFFECT: THE LINKS ARE CLUSTERED BY COLOURS AND      
						;         PRINTED ON FILE IN THE GIVEN FORMAT.        
						; VALUE:  UNDEFINED.                                  
  (let (ASSOCLIST POINTER (FIRSTPOS.I 10) (FIRSTPOS.A 17))
    (unless (AND (LISTP COLOURS)
		 (SUPERSET (DS-LINK.COLOURS.FOR 'ALL) COLOURS)) ; USE ALL LINK COLOURS AS DEFAULT
      (SETQ COLOURS (DS-LINK.COLOURS.FOR 'ALL)))
    (unless (MEMBer FORMAT '(I A))		; USE A AS DEFAULT
      (SETQ FORMAT 'A))
    (SETQ ASSOCLIST (MAPCAR #'(LAMBDA (COLOUR) (LIST COLOUR)) COLOURS))
    (MAPC #'(LAMBDA (LINK)
	      (SETQ POINTER (ASSOC (DS-LINK.COLOUR LINK) ASSOCLIST))
	      (when POINTER (NCONC1 POINTER LINK)))
	  LINKS)
    (MAPC #'(LAMBDA (COLOUR.LINKS)
	      (when (CDR COLOUR.LINKS)
		(let ((string (format nil "      ~A-LINKS" (CAR COLOUR.LINKS))))
		  (format file "~%~A" string)
		  (case FORMAT
		    (I (TERPRI FILE)
		       (MAPC #'(LAMBDA (LINK)
				 (CG=PR_LINK LINK FIRSTPOS.I FILE)
				 (terpri file))
			     (CDR COLOUR.LINKS)))
		    (A (setq cg*position (length string))
		       (CG=PR_LIST (CDR COLOUR.LINKS) FIRSTPOS.A FILE)
		       (TERPRI FILE))))))
	  ASSOCLIST)))
  
(DEFUN CG=PR_LINK (LINK FIRSTPOS FILE)
  ; EDITED:  7-MAR-83 12:56:41        NE
  ; INPUT:  A LINK ADDRESS, ANATURAL NUMBER, AND A FILE
  ;         NAME.
  ; EFFECT: THE COMPONENTS OF LINK ARE PRINTED ON FILE
  ;         IN A READABLE FORMAT, INDENTED TO FIRSTPOS.
  ; VALUE:  UNDEFINED.
  (let*
    ((COLOUR (DS-LINK.COLOUR LINK)) (POS.COLOUR (+ FIRSTPOS 10)) (POS.LABEL (+ FIRSTPOS 15)) (POS.ORIENTED (+ FIRSTPOS 25))
     (pos.rule (+ firstpos 10)) (LASTPOS (- (LINELENGTH NIL FILE) 1)) 
     (LABEL (DS-LINK.LABEL LINK))
     (POSPAR (DS-LINK.POSPAR LINK))
     (POSLITNO (DS-LINK.POSLITNO LINK))
     (POSFCT (DS-LINK.POSFCT LINK))
     (NEGPAR (DS-LINK.NEGPAR LINK))
     (NEGLITNO (DS-LINK.NEGLITNO LINK))
     (NEGFCT (DS-LINK.NEGFCT LINK))
     (PROPLIST (DT-GETPROPLIST LINK))
     (UNIFIERS (DS-LINK.UNIFIERS LINK))
     (RULE (DS-LINK.RULE LINK))
     (ORIENTED (COND ((AND RULE (DS-RULE.ORIENTED RULE COLOUR)) 'ORIENTED))))
    (format file "~vA~A" firstpos "" link)
    (setq cg*position (+ firstpos (print-length link))) ; LINK NAME 
    (PROGN ; COLOUR AND LABEL
      (format file "~vA" (- pos.colour cg*position) "")
      (setq cg*position pos.colour)
      (PRINC COLOUR FILE)
      (COND (LABEL (FORMAT FILE "~vT" POS.LABEL) (PRINC LABEL FILE))))
    (PROGN ; 'ORIENTED' IF LINK IS NECESSARILY ORIENTED.
      (COND (ORIENTED (FORMAT FILE "~vT" POS.ORIENTED) (PRINC ORIENTED FILE))))
    (COND ((OR negPAR negLITNO negFCT)
	   (let ((output (format nil " NEG: ~A[~A] ~@[~A~]"		      ; NEGATIVE SIDE
				 (COND (NEGPAR (DS-CLAUSE.PNAME NEGPAR))) (COND (NEGLITNO NEGLITNO)) (COND (NEGFCT NEGFCT))))) 
	     (princ output file)
	     (incf cg*position (length output)))))
    (COND ((OR POSPAR POSLITNO POSFCT)
	   (let ((output (format nil " POS: ~A[~A] ~@[~A~]"		      ; positive SIDE
				 (COND (pospar (DS-CLAUSE.PNAME posPAR)))
				 (COND (posLITNO posLITNO))
				 (COND (posFCT posFcT))))) 
	     (princ output file)
	     (incf cg*position (length output)))))
    (COND								      ; property list
      (PROPLIST (format file "~%~vAProperties: " pos.colour "")
		(setq cg*position (+ pos.colour 12))
		(princ PROPLIST FILE)
		(incf cg*position (print-length proplist))))
    (when UNIFIERS							      ;Unifiers
      (SETQ UNIFIERS (DS-PNAME (mapcar #'(lambda (uni) (cons "" uni)) UNIFIERS)))
									      ; now a unifier is a term with empty function symbol
      (let ((lengths.of.unifiers (mapcar #'PP-term.length.one.line unifiers)))
	(cond ((<= (+ cg*position 10 (reduce #'+ lengths.of.unifiers) 1 (length unifiers)) lastpos)
	       (format file "Unifiers: ")
	       (mapprint unifiers nil nil " " #'pp-print.term.one.line file))
	      (T (format file "~%~vAUnifiers: " pos.colour "")
		 (setq cg*position (+ pos.colour 10))
		 (cond ((<= (+ cg*position (reduce #'+ lengths.of.unifiers) 1 (length unifiers)) lastpos)
			(mapprint unifiers nil nil " " #'pp-print.term.one.line file))
		       ((<= (+ cg*position (apply #'max lengths.of.unifiers) 1) lastpos)
			(mapprint unifiers nil nil (format nil "~%~vA" cg*position) #'pp-print.term.one.line file))
		       (T (mapc #'(lambda (unifier length)
				    (if (<= (+ cg*position length) lastpos)
					(pp-print.term.one.line unifier file)
					(progn (princ "(" file)
					       (smapl #'(lambda (rest.unifier)
							  (let ((var (first rest.unifier)) (term (second rest.unifier)))
							    (format file "~A " var)
							    (pp-print.term term file
									   :current.pos (+ cg*position (length var) 2)
									   :right.pos lastpos)
							    (when (cddr rest.unifier)
							      (format file "~%~vA" (1+ cg*position) ""))))
						      #'cddr
						      (cdr unifier))	      ; cdr unifier = actual unifier
					       (princ ")" file)))
				    (format nil "~%~vA" cg*position ""))
				unifiers lengths.of.unifiers)))))
	(setq cg*position lastpos)))
    (COND								      ; Rule
      (RULE (FORMAT FILE "~%~VA" pos.rule "")
	    (PRINC "RULE: " FILE)
	    (PRINC RULE FILE)
	    (COND ((CONSP RULE) (PRINC " / " FILE)
		   (setq cg*position (+ pos.rule 6 (print-length rule)))
		   (CG=PR_CLAUSE.SHORT (CAR RULE) 1 FILE NIL)))))))
  
(DEFUN CG=PR_VARIABLES (VARIABLES FIRSTPOS FILE RENAMEFLAG)
  ; EDITED: 25-FEB-83 17:04:48        NE
  ; INPUT:  A LIST OF VARIABLES, A NATURAL NUMBER, A
  ;         FILE NAME, AND A BOOLEAN VALUE.
  ; EFFECT: PRINTS THE VARIABLES ON FILE CLUSTERED
  ;         BY THEIR SORTS AND INDENTED TO FIRSTPOS.
  ;         IF RENAMEFLAG=T, UP TO  6 VARIABLES ARE
  ;         RENAMED WITH STANDARD IDENTIFIERS.
  ; VALUE:  LIST OF THE PRINT NAMES ACTUALLY USED FOR
  ;         OUTPUT IN THE SAME ORDER AS VARIABLES.
  (let ((PNAMES (MAPCAR #'(lambda (x) (DT-PNAME x)) VARIABLES))
	(ASSOCLIST (MAPCAR #'(lambda (x) (if (dt-variable.is x) (DT-VARIABLE.SORT x))) VARIABLES)))
    (SETQ ASSOCLIST (MAPCAR #'(LAMBDA (SORT) (LIST SORT))
			    (remove-duplicates ASSOCLIST)))
    (MAPC #'(LAMBDA (x) (NCONC1 (ASSOC (if (dt-variable.is x) (DT-VARIABLE.SORT x)) ASSOCLIST) x))
	  VARIABLES)
    (COND
      (RENAMEFLAG
       (SETQ PNAMES
	     (CASE (LIST-LENGTH PNAMES)
	       (0 NIL)
	       (1 (LIST "X"))
	       (2 (LIST "X" "Y"))
	       (3 (LIST "X" "Y" "Z"))
	       (4 (LIST "W" "X" "Y" "Z"))
	       (5 (LIST "V" "W" "X" "Y" "Z"))
	       (6 (LIST "U" "V" "W" "X" "Y" "Z"))
	       (OTHERWISE (APPEND '("U" "V" "W" "X" "Y" "Z") (CDDR (CDDDDR PNAMES))))))))
    (setq assoclist (copy-tree assoclist))
    (MAPC #'(LAMBDA (V P) (NSUBST P V ASSOCLIST))
	  VARIABLES PNAMES)
    (spaces (- cg*position firstpos) file)
    (incf cg*position (- cg*position firstpos))
    (MAPC #'(LAMBDA (SORT.VARIABLES)
	      (let ((output (format nil "~A~{,~A~}:~A " (second sort.variables) (cddr sort.variables)
				    (if (opt-get.option sort_literals)
					(with-output-to-string (st)
					  (pp-print.term.one.line (dt-pname (first SORT.VARIABLES)) st :current.pos 0))
					(CAR SORT.VARIABLES)))))
		(princ output file)
		(incf cg*position (length output))))
	  ASSOCLIST)
    PNAMES))
  
  
(DEFUN CG=PR_LIST (LIST FIRSTPOS FILE)
									      ; EDITED: 25-FEB-83 16:03:29        NE
									      ; INPUT:  A LIST (OR NIL), A NATURAL NUMBER, AND
									      ;         A FILE NAME.
									      ; EFFECT: THE ELEMENTS OF LIST ARE PRINTED ON FILE
									      ;         INDENTED TO FIRSTPOS.
									      ; VALUE:  UNDEFINED.
  (let ((LASTPOS 116))
    (cond ((< cg*position firstpos) (spaces (- firstpos cg*position) file))
	  ((> cg*position firstpos) (format file "~%~VA" firstpos "")))
    (setq cg*position firstpos)
    (MAPC #'(LAMBDA (ELEM)
	      (if (< LASTPOS (+ cg*position (PRINT-LENGTH ELEM NIL)))
		  (let ((output (FORMAT nil "~%~vA~A" FIRSTPOS "" elem)))
		    (princ output file)
		    (setq cg*position (1- (length output))))
		  (let ((output (FORMAT nil "~A" elem)))
		    (princ output file)
		    (incf cg*position (length output))))
	      (when (< cg*position Lastpos) (SPACES 1 FILE) (incf cg*position)))
	  LIST)))