(*  Title: 	ZF/zf.thy
    Author: 	Lawrence C Paulson and Martin D Coen, CU Computer Laboratory
    Copyright   1993  University of Cambridge

Zermelo-Fraenkel Set Theory 
*)

ZF = FOL +

types 	i 0
	ulist 0
	tuple 0

arities i :: term

consts	
    "0"         :: "i"     ("0")           (*the empty set*)
    Pow         :: "i=>i"                  (*power sets*)
    Inf         :: "i"                     (*infinite set*)
    (** bounded quantifiers **)
    "@Ball"     :: "[id,i,o]=>o"           ("(3ALL _:_./ _)" 10)
    "@Bex"      :: "[id,i,o]=>o"           ("(3EX _:_./ _)" 10)
    Ball        :: "[i, i=>o]=>o"
    Bex         :: "[i, i=>o]=>o"
    (** general union and intersection **)
    "@INTER"    :: "[id,i,i]=>i"           ("(3INT _:_./ _)" 10)
    "@UNION"    :: "[id,i,i]=>i"           ("(3UN _:_./ _)" 10)
    Union,Inter :: "i=>i"
    (** Variations on Replacement **)
    "@Replace"  :: "[id,id,i,o]=>i"        ("(1{_ ./ _: _, _})")
    "@RepFun"   :: "[i,id,i]=>i"           ("(1{_ ./ _: _})")
    "@Collect"  :: "[id,i,o]=>i"           ("(1{_: _ ./ _})")
    PrimReplace :: "[i, [i,i]=>o] => i"
    Replace     :: "[i, [i,i]=>o] => i"
    RepFun      :: "[i, i=>i] => i"
    Collect     :: "[i, i=>o] => i"
    (** Descriptions **)
    "@THE"      :: "[id,o]=>i"             ("(3THE _./ _)" 10)
    The         :: "[i=>o]=>i"
    if          :: "[o,i,i]=>i"
    (** finite sets **)
    "@Enum"     :: "ulist=>i"              ("{_}")
    "@Sing"     :: "i=>ulist"              ("_")
    "@Ulist"    :: "[i,ulist]=>ulist"      ("_,/_")
    Upair,cons  :: "[i,i]=>i"
    succ        :: "i=>i"
    (** ordered pairing and n-tuples **)
    "@PairA"    :: "[i,tuple]=>i"          ("(1<_,/_>)")
    ""          :: "i=>tuple"              ("_")
    "@PairB"    :: "[i,tuple]=>tuple"      ("_,/_")
    Pair        :: "[i,i]=>i"
    split       :: "[i, [i,i]=>i] => i"
    fst,snd     :: "i=>i"
    (** Sigma and Pi operators **)
    "@PROD"     :: "[id,i,i]=>i"           ("(3PROD _:_./ _)" 10)
    "@SUM"      :: "[id,i,i]=>i"           ("(3SUM _:_./ _)" 10)
    "@lam"      :: "[id,i,i]=>i"           ("(3lam _:_./ _)" 10)
    Pi,Sigma    :: "[i,i=>i]=>i"
    (** relations and functions **)
    domain      :: "i=>i"
    range       :: "i=>i"
    field       :: "i=>i"
    converse    :: "i=>i"
    Lambda      :: "[i, i=>i]=>i"
    restrict    :: "[i, i] =>i"

(** Infixes in order of decreasing precedence **)
    "``"	:: "[i,i]=>i"  (infixl 90)  (*image*)
    "-``"	:: "[i,i]=>i"  (infixl 90)  (*inverse image*)
    "`"		:: "[i,i]=>i"  (infixl 90)  (*function application*)
(*Except for their translations, * and -> are right-associating infixes*)
    " *"	:: "[i,i]=>i"  ("(_ */ _)" [81,80] 80)  (*Cartesian product*)
    "Int"	:: "[i,i]=>i"  (infixl 70)  (*binary intersection*)
    "Un"	:: "[i,i]=>i"  (infixl 65)  (*binary union*)
    "-" 	:: "[i,i]=>i"  (infixl 65)  (*set difference*)
    " ->"	:: "[i,i]=>i"  ("(_ ->/ _)" [61,60] 60)  (*function space*)
    "<="	:: "[i,i]=>o"  (infixl 50)  (*subset relation*)
    ":" 	:: "[i,i]=>o"  (infixl 50)  (*membership relation*)

rules

 (* Bounded Quantifiers *)
Ball_def        "Ball(A,P) == ALL x. x:A --> P(x)"
Bex_def 	"Bex(A,P) == EX x. x:A & P(x)"
subset_def      "A <= B == ALL x:A. x:B"

 (* ZF axioms -- see Suppes p.238
    Axioms for Union, Pow and Replace state existence only,
        uniqueness is derivable using extensionality.  *)

extension       "A = B <-> A <= B & B <= A"
union_iff       "A : Union(C) <-> (EX B:C. A:B)"
power_set       "A : Pow(B) <-> A <= B"
succ_def        "succ(i) == cons(i,i)"

 (*We may name this set, though it is not uniquely defined. *)
infinity        "0:Inf & (ALL y:Inf. succ(y): Inf)"

 (*This formulation facilitates case analysis on A. *)
foundation      "A=0 | (EX x:A. ALL y:x. ~ y:A)"

 (* Schema axiom since predicate P is a higher-order variable *)
replacement     "(ALL x:A. ALL y z. P(x,y) & P(x,z) --> y=z) ==> \
\                        b : PrimReplace(A,P) <-> (EX x:A. P(x,b))"

 (* Derived form of replacement, restricting P to its functional part.
    The resulting set (for functional P) is the same as with
    PrimReplace, but the rules are simpler. *)
Replace_def	"Replace(A,P) == PrimReplace(A, %x y. (EX!z.P(x,z)) & P(x,y))"

 (* Functional form of replacement -- analgous to ML's map functional *)
RepFun_def      "RepFun(A,f) == {y . x:A, y=f(x)}"

 (* Separation and Pairing can be derived from the Replacement
    and Powerset Axioms using the following definitions.  *)

Collect_def     "Collect(A,P) == {y . x:A, x=y & P(x)}"

 (*Unordered pairs (Upair) express binary union/intersection and cons;
   set enumerations translate as {a,...,z} = cons(a,...,cons(z,0)...)  *)
Upair_def   "Upair(a,b) == {y. x:Pow(Pow(0)), (x=0 & y=a) | (x=Pow(0) & y=b)}"
cons_def    "cons(a,A) == Upair(a,a) Un A"

 (* Difference, general intersection, binary union and small intersection *)

Diff_def        "A - B    == { x:A . ~(x:B) }"
Inter_def       "Inter(A) == { x:Union(A) . ALL y:A. x:y}"
Un_def  	"A Un  B  == Union(Upair(A,B))"
Int_def 	"A Int B  == Inter(Upair(A,B))"

 (* Definite descriptions -- via Replace over the set "1" *)

the_def 	"The(P)    == Union({y . x:{0}, P(y)})"
if_def  	"if(P,a,b) == THE z. P & z=a | ~P & z=b"

 (* Ordered pairs and disjoint union of a family of sets *)

 (* this "symmetric" definition works better than {{a}, {a,b}} *)
Pair_def        "<a,b>  == {{a,a}, {a,b}}"
split_def       "split(p,c) == THE y. EX a b. p=<a,b> & y=c(a,b)"
fst_def 	"fst(p) == split(p, %x y.x)"
snd_def 	"snd(p) == split(p, %x y.y)"
Sigma_def       "Sigma(A,B) == UN x:A. UN y:B(x). {<x,y>}"

 (* Operations on relations *)

(*converse of relation r, inverse of function*)
converse_def	"converse(r) == {z. w:r, EX x y. w=<x,y> & z=<y,x>}"

domain_def      "domain(r) == {x. w:r, EX y. w=<x,y>}"
range_def       "range(r) == domain(converse(r))"
field_def       "field(r) == domain(r) Un range(r)"
image_def       "r `` A  == {y : range(r) . EX x:A. <x,y> : r}"
vimage_def      "r -`` A == converse(r)``A"

 (* Abstraction, application and Cartesian product of a family of sets *)

lam_def 	"Lambda(A,b) == {<x,b(x)> . x:A}"
apply_def       "f`a == THE y. <a,y> : f"
Pi_def  	"Pi(A,B)  == {f: Pow(Sigma(A,B)). ALL x:A. EX! y. <x,y>: f}"

  (* Restrict the function f to the domain A *)
restrict_def    "restrict(f,A) == lam x:A.f`x"

end

ML

(** Ordered tuples -- change the type of Pair from tuple to i 
    @PairA is for the outermost tuple and ensures that <...> are printed
    @PairB is for inner tuples nested to the right. **)

(*Both @PairA and @PairB translate to Pair*)
fun pairA_tr[x, y] = Const("Pair", dummyT) $ x $ y;
fun pairB_tr[x, y] = Const("Pair", dummyT) $ x $ y;

(*Pair translates to either @PairA or @PairB*)
fun pairB_tr' (Const("Pair",_) $ x $ y) = 
	Const("@PairB", dummyT) $ x $ pairB_tr' y
  | pairB_tr' x = x;

fun pair_tr'[x,y] = Const("@PairA", dummyT) $ x $ pairB_tr' y;

(*** finite sets ***)

val mtset = Const("0", dummyT);
val cons = Const("cons", dummyT);

(* enumeration of finite set elements *)
fun make_finset (Const("@Sing",_)$e) = cons $ e $ mtset
  | make_finset (Const("@Ulist",_)$e$l) = cons $ e $ make_finset l;

fun enum_tr[elts] = make_finset elts;

(*Return the elements of a finite set, raise exception if ill-formed.*)
fun dest_finset (Const("cons",_) $ e $ Const("0",_)) = 
	Const("@Sing",dummyT) $ e
  | dest_finset (Const("cons",_) $ e $ l) = 
	Const("@Ulist", dummyT) $ e $ dest_finset l
  | dest_finset (Const("0",_)) = mtset
  | dest_finset _ = raise Match;

fun enum_tr'[x,y] = Const("@Enum", dummyT) $ dest_finset(cons$x$y);

(** Replacement  { y . x:A, P[x,y] } = Replace(A, %x y.P[x,y]) **)
fun rep_tr[Free(y,_), Free(x,T), A, Pxy] =
    Const("Replace", dummyT) $ A $ absfree(x, T, absfree(y, T, Pxy));

fun rep_tr'[A, Abs(x,Tx, Abs(y,Ty,P))] =
    let val (y',Py)  = variant_abs(y,Ty,P)
        val (x',Pxy) = variant_abs(x,Tx,Py)
    in Const("@Replace",dummyT) $ Free(y',Ty) $ Free(x',Tx) $ A $ Pxy end;

(** RepFun's concrete syntax -- note argument order! **)

fun repfun_tr [f, Free(id,T), A] = 
    Const("RepFun",dummyT) $ A $ absfree(id,T,f);

fun repfun_tr' [A, Abs(id,T,f)] =
    let val (id',f') = variant_abs(id,T,f)
    in Const("@RepFun",dummyT) $ f' $ Free(id',T) $ A end;

(** Union/Inter of a family of sets -- involves RepFun **)
fun unint_tr q [Free(id,T),A,B] = 
    Const(q,dummyT) $ (Const("RepFun",dummyT) $ A $ absfree(id,T,B));

fun unint_tr' q [Const("RepFun",_) $ A $ Abs(id,T,B)] =
    let val (id',B') = variant_abs(id,T,B)
    in Const(q,dummyT) $ Free(id',T) $ A $ B' end;

(** For quantifications of the form %x:A.P(x) **)
fun qnt_tr q [Free(id,T), A, P] = Const(q,dummyT)$ A $absfree(id,T,P);

fun qnt_tr' q [A, Abs(id,T,P)] =
    let val (id',P') = variant_abs(id,T,P)
    in Const(q,dummyT) $ Free(id',T) $ A $ P' end;

(** 'Dependent' type operators **)
fun ndependent_tr q [A,B] = 
    Const(q,dummyT) $ A $ Abs("x", dummyT, incr_boundvars 1 B);

(*If there is a dependence then use quantifier q; else use r. *)
fun dependent_tr' (q,r) [A, Abs(id,T,B)] =
    if  0 mem (loose_bnos B) then 
      let val (id',B') = variant_abs(id,T,B)
      in Const(q,dummyT) $ Free(id',T) $ A $ B' end
    else Const(r,dummyT) $ A $ B;

val parse_translation =
    [("@PairA",		pairA_tr),
     ("@PairB",		pairB_tr),
     ("@Enum",		enum_tr),
     ("@Replace",	rep_tr),
     ("@RepFun",	repfun_tr),
     mk_binder_tr("@THE","The"),
     ("@INTER",		unint_tr "Inter"),
     ("@UNION",		unint_tr "Union"),
     (" ->",		ndependent_tr "Pi"),
     (" *",		ndependent_tr "Sigma"),
     ("@PROD",		qnt_tr "Pi"),
     ("@SUM",		qnt_tr "Sigma"),
     ("@Collect",	qnt_tr "Collect"),
     ("@Ball",		qnt_tr "Ball"),
     ("@Bex",		qnt_tr "Bex"),
     ("@lam",		qnt_tr "Lambda")]
and print_translation =
    [("Pair",		pair_tr'),
     ("cons",		enum_tr'),
     ("Replace",	rep_tr'),
     ("RepFun",		repfun_tr'),
     mk_binder_tr'("The","@THE"),
     ("Inter",		unint_tr' "@INTER"),
     ("Union",		unint_tr' "@UNION"),
     ("Pi",		dependent_tr' ("@PROD"," ->")),
     ("Sigma",		dependent_tr' ("@SUM"," *")),
     ("Collect",	qnt_tr' "@Collect"),
     ("Ball",		qnt_tr' "@Ball"),
     ("Bex",		qnt_tr' "@Bex"),
     ("Lambda",		qnt_tr' "@lam")];

