(*  Title: 	ZF/list.ML
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1992  University of Cambridge

For list.thy.  Lists in Zermelo-Fraenkel Set Theory 
*)

open List;

(** Monotonicity and unfolding of the function **)

goal List.thy "bnd_mono(univ(A), %X. {0} Un A*X)";
by (rtac bnd_monoI 1);
by (REPEAT (ares_tac [subset_refl, Un_mono, product_mono] 2));
by (REPEAT (resolve_tac [subset_refl, zero_in_univ RS singleton_subsetI, 
		         A_subset_univ, Un_least, product_subset_univ] 1));
val list_bnd_mono = result();

(** These lemmas justify using "list" in other recursive type definitions **)

val [prem] = goalw List.thy [list_def] "A<=B ==> list(A) <= list(B)";
by (rtac lfp_mono 1);
by (REPEAT (resolve_tac [list_bnd_mono, prem, 
			 subset_refl, Un_mono, product_mono] 1));
val list_mono = result();

(*Easily provable by induction also*)
goalw List.thy [list_def] "list(univ(A)) <= univ(A)";
by (rtac lfp_lowerbound 1);
by (rtac (A_subset_univ RS univ_mono) 2);
by (REPEAT (resolve_tac [subset_refl, zero_in_univ RS singleton_subsetI,
			 Un_least, product_subset_univ] 1));
val list_univ = result();

val list_subset_univ = standard
    (list_mono RS (list_univ RSN (2,subset_trans)));


(** Type checking of 0 (as nil) and <-,-> (as cons) **)

(* list(A) = {0} Un (A * list(A)) *)
val list_unfold = list_bnd_mono RS (list_def RS def_Tarski);

goal List.thy "0 : list(A)";
by (rtac (list_unfold RS ssubst) 1);
by (rtac (singletonI RS UnI1) 1);
val list_0I = result();

val prems = goal List.thy "[| a: A;  l: list(A) |] ==> <a,l> : list(A)";
by (rtac (list_unfold RS ssubst) 1);
by (REPEAT (resolve_tac (prems @ [SigmaI, UnI2]) 1));
val list_PairI = result();

(** Structural induction on lists **)

val major::prems = goal List.thy
    "[| l: list(A);  \
\       P(0);        \
\       !!x y. [| x: A;  y: list(A);  P(y) |] ==> P(<x,y>) \
\    |] ==> P(l)";
by (rtac (major RS (list_def RS def_induct)) 1);
by (rtac list_bnd_mono 1);
by (fast_tac (ZF_cs addIs prems) 1);
val list_induct = result();

(*Perform induction on l, then prove the major premise using prems. *)
fun list_ind_tac a prems i = 
    EVERY [res_inst_tac [("l",a)] list_induct i,
	   rename_last_tac a ["1"] (i+2),
	   ares_tac prems i];

val [prem] = goal List.thy
    "l: list(A) ==> l=0 | (EX x:A. EX xs: list(A). l=<x,xs>)";
by (rtac (prem RS list_induct) 1);
by (DEPTH_SOLVE (ares_tac [refl, disjI1, disjI2, bexI] 1));
val list_cases = result();

(** list_case **)

goalw List.thy [list_case_def] "list_case(0,c,h) = c";
by (fast_tac (ZF_cs addIs [the_equality]) 1);
val list_case_0 = result();

goalw List.thy [list_case_def] "list_case(<a,l>, c, h) = h(a,l)";
by (fast_tac (ZF_cs addIs [the_equality]) 1);
val list_case_Pair = result();

val major::prems = goal List.thy
    "[| l: list(A);    \
\       c: C(0);       \
\       !!x y. [| x: A;  y: list(A) |] ==> h(x,y): C(<x,y>)  \
\    |] ==> list_case(l,c,h) : C(l)";
by (rtac (major RS list_induct) 1);
by (ALLGOALS (ASM_SIMP_TAC (ZF_ss addrews
			    ([list_case_0,list_case_Pair]@prems))));
val list_case_type = result();


(** list_rec -- by Vset recursion **)

(*Used to verify list_rec*)
val list_rec_ss = ZF_ss 
      addcongs (mk_typed_congs List.thy [("h", "[i,i,i]=>i")])
      addrews [list_case_0, list_case_Pair];

goal List.thy "list_rec(0,c,h) = c";
by (rtac (list_rec_def RS def_Vrec RS trans) 1);
by (SIMP_TAC list_rec_ss 1);
val list_rec_0 = result();

goal List.thy "list_rec(<a,l>, c, h) = h(a, l, list_rec(l,c,h))";
by (rtac (list_rec_def RS def_Vrec RS trans) 1);
by (SIMP_TAC (list_rec_ss addrews [Vset_rankI, rank_pair2]) 1);
val list_rec_Pair = result();

(*Type checking -- proved by induction, as usual*)
val prems = goal List.thy
    "[| l: list(A);    \
\       c: C(0);       \
\       !!x y r. [| x: A;  y: list(A);  r: C(y) |] ==> h(x,y,r): C(<x,y>)  \
\    |] ==> list_rec(l,c,h) : C(l)";
by (list_ind_tac "l" prems 1);
by (ALLGOALS (ASM_SIMP_TAC
	      (ZF_ss addrews (prems@[list_rec_0,list_rec_Pair]))));
val list_rec_type = result();

(** Versions for use with definitions **)

val [rew] = goal List.thy
    "[| !!l. j(l)==list_rec(l, c, h) |] ==> j(0) = c";
by (rewtac rew);
by (rtac list_rec_0 1);
val def_list_rec_0 = result();

val [rew] = goal List.thy
    "[| !!l. j(l)==list_rec(l, c, h) |] ==> j(<a,l>) = h(a,l,j(l))";
by (rewtac rew);
by (rtac list_rec_Pair 1);
val def_list_rec_Pair = result();

(*should be a list not a pair!*)
fun list_recs def = 
      (standard (def RS def_list_rec_0),
       standard (def RS def_list_rec_Pair));

(** map **)

val (map_0,map_Pair) = list_recs map_def;

val prems = goalw List.thy [map_def] 
    "[| l: list(A);  !!x. x: A ==> h(x): B |] ==> map(h,l) : list(B)";
by (REPEAT (ares_tac (prems@[list_rec_type,list_0I, list_PairI]) 1));
val map_type = result();

val [major] = goal List.thy "l: list(A) ==> map(h,l) : list({h(u). u:A})";
by (rtac (major RS map_type) 1);
by (etac RepFunI 1);
val map_type2 = result();

(** length **)

val (length_0,length_Pair) = list_recs length_def;

val prems = goalw List.thy [length_def] 
    "l: list(A) ==> length(l) : nat";
by (REPEAT (ares_tac (prems @ [list_rec_type, nat_0I, nat_succI]) 1));
val length_type = result();

(** app **)

val (app_0,app_Pair) = list_recs app_def;

val prems = goalw List.thy [app_def] 
    "[| xs: list(A);  ys: list(A) |] ==> xs@ys : list(A)";
by (REPEAT (ares_tac (prems @ [list_rec_type, list_PairI]) 1));
val app_type = result();

(** rev **)

val (rev_0,rev_Pair) = list_recs rev_def;

val prems = goalw List.thy [rev_def] 
    "xs: list(A) ==> rev(xs) : list(A)";
by (REPEAT (ares_tac 
	    (prems @ [list_rec_type, list_0I, list_PairI, app_type]) 1));
val rev_type = result();


(** flat **)

val (flat_0,flat_Pair) = list_recs flat_def;

val prems = goalw List.thy [flat_def] 
    "ls: list(list(A)) ==> flat(ls) : list(A)";
by (REPEAT (ares_tac 
	    (prems @ [list_rec_type, list_0I, list_PairI, app_type]) 1));
val flat_type = result();


(** list_add **)

val (list_add_0,list_add_Pair) = list_recs list_add_def;

val prems = goalw List.thy [list_add_def] 
    "xs: list(nat) ==> list_add(xs) : nat";
by (REPEAT (ares_tac (prems @ [list_rec_type, nat_0I, add_type]) 1));
val list_add_type = result();

(** List simplification **)

val list_typechecks =
      [list_0I, list_PairI, list_case_type, list_rec_type,
       map_type, map_type2, app_type, length_type, rev_type, flat_type,
       list_add_type];

val list_congs = mk_congs List.thy
    ["list_case","list_rec","map","op @","length","rev","flat","list_add"];

val list_ss = arith_ss 
    addcongs list_congs
    addrews([list_case_0, list_case_Pair, 
	     list_rec_0, list_rec_Pair, 
	     map_0, map_Pair,
	     app_0, app_Pair,
	     length_0, length_Pair,
	     rev_0, rev_Pair,
	     flat_0, flat_Pair,
	     list_add_0, list_add_Pair] @ list_typechecks);

(*** theorems about map ***)

val prems = goal List.thy
    "l: list(A) ==> map(%u.u, l) = l";
by (list_ind_tac "l" prems 1);
by (ALLGOALS (ASM_SIMP_TAC list_ss));
val map_ident = result();

val prems = goal List.thy
    "l: list(A) ==> map(h, map(j,l)) = map(%u.h(j(u)), l)";
by (list_ind_tac "l" prems 1);
by (ALLGOALS (ASM_SIMP_TAC list_ss));
val map_compose = result();

val prems = goal List.thy
    "xs: list(A) ==> map(h, xs@ys) = map(h,xs) @ map(h,ys)";
by (list_ind_tac "xs" prems 1);
by (ALLGOALS (ASM_SIMP_TAC list_ss));
val map_app_distrib = result();

val prems = goal List.thy
    "ls: list(list(A)) ==> map(h, flat(ls)) = flat(map(map(h),ls))";
by (list_ind_tac "ls" prems 1);
by (ALLGOALS (ASM_SIMP_TAC (list_ss addrews [map_app_distrib])));
val map_flat = result();

val prems = goal List.thy
    "l: list(A) ==> \
\    list_rec(map(h,l), c, d) = \
\    list_rec(l, c, %x xs r. d(h(x), map(h,xs), r))";
by (list_ind_tac "l" prems 1);
by (ALLGOALS 
    (ASM_SIMP_TAC 
     (list_ss addcongs (mk_typed_congs List.thy [("d", "[i,i,i]=>i")]))));
val list_rec_map = result();

(** theorems about list(Collect(A,P)) -- used in ex/term.ML **)

(* c : list(Collect(B,P)) ==> c : list(B) *)
val list_CollectD = standard (Collect_subset RS list_mono RS subsetD);

val prems = goal List.thy
    "l: list({x:A. h(x)=j(x)}) ==> map(h,l) = map(j,l)";
by (list_ind_tac "l" prems 1);
by (ALLGOALS (ASM_SIMP_TAC list_ss));
val map_list_Collect = result();

(*** theorems about length ***)

val prems = goal List.thy
    "xs: list(A) ==> length(map(h,xs)) = length(xs)";
by (list_ind_tac "xs" prems 1);
by (ALLGOALS (ASM_SIMP_TAC list_ss));
val length_map = result();

val prems = goal List.thy
    "xs: list(A) ==> length(xs@ys) = length(xs) #+ length(ys)";
by (list_ind_tac "xs" prems 1);
by (ALLGOALS (ASM_SIMP_TAC list_ss));
val length_app = result();

(* [| m: nat; n: nat |] ==> m #+ succ(n) = succ(n) #+ m 
   Used for rewriting below*)
val add_commute_succ = nat_succI RSN (2,add_commute);

val prems = goal List.thy
    "xs: list(A) ==> length(rev(xs)) = length(xs)";
by (list_ind_tac "xs" prems 1);
by (ALLGOALS (ASM_SIMP_TAC (list_ss addrews [length_app, add_commute_succ])));
val length_rev = result();

val prems = goal List.thy
    "ls: list(list(A)) ==> length(flat(ls)) = list_add(map(length,ls))";
by (list_ind_tac "ls" prems 1);
by (ALLGOALS (ASM_SIMP_TAC (list_ss addrews [length_app])));
val length_flat = result();

(*** theorems about app ***)

val [major] = goal List.thy "xs: list(A) ==> xs@0=xs";
by (rtac (major RS list_induct) 1);
by (ALLGOALS (ASM_SIMP_TAC list_ss));
val app_right_0 = result();

val prems = goal List.thy
    "xs: list(A) ==> (xs@ys)@zs = xs@(ys@zs)";
by (list_ind_tac "xs" prems 1);
by (ALLGOALS (ASM_SIMP_TAC list_ss));
val app_assoc = result();

val prems = goal List.thy
    "ls: list(list(A)) ==> flat(ls@ms) = flat(ls)@flat(ms)";
by (list_ind_tac "ls" prems 1);
by (ALLGOALS (ASM_SIMP_TAC (list_ss addrews [app_assoc])));
val flat_app_distrib = result();

(*** theorems about rev ***)

val prems = goal List.thy
    "l: list(A) ==> rev(map(h,l)) = map(h,rev(l))";
by (list_ind_tac "l" prems 1);
by (ALLGOALS (ASM_SIMP_TAC (list_ss addrews [map_app_distrib])));
val rev_map_distrib = result();

(*Simplifier needs the premises as assumptions because rewriting will not
  instantiate the variable ?A in the rules' typing conditions; note that
  rev_type does not instantiate ?A.  Only the premises do.
*)
val prems = goal List.thy
    "[| xs: list(A);  ys: list(A) |] ==> rev(xs@ys) = rev(ys)@rev(xs)";
by (cut_facts_tac prems 1);
by (etac list_induct 1);
by (ALLGOALS (ASM_SIMP_TAC (list_ss addrews [app_right_0,app_assoc])));
val rev_app_distrib = result();

val prems = goal List.thy
    "l: list(A) ==> rev(rev(l))=l";
by (list_ind_tac "l" prems 1);
by (ALLGOALS (ASM_SIMP_TAC (list_ss addrews [rev_app_distrib])));
val rev_rev_ident = result();

val prems = goal List.thy
    "ls: list(list(A)) ==> rev(flat(ls)) = flat(map(rev,rev(ls)))";
by (list_ind_tac "ls" prems 1);
by (ALLGOALS (ASM_SIMP_TAC (list_ss addrews 
         [map_app_distrib, flat_app_distrib, rev_app_distrib, app_right_0])));
val rev_flat = result();


(*** theorems about list_add ***)

val prems = goal List.thy
    "[| xs: list(nat);  ys: list(nat) |] ==> \
\    list_add(xs@ys) = list_add(ys) #+ list_add(xs)";
by (cut_facts_tac prems 1);
by (list_ind_tac "xs" prems 1);
by (ALLGOALS 
    (ASM_SIMP_TAC (list_ss addrews [add_0_right, add_assoc RS sym])));
by (resolve_tac arith_congs 1);
by (REPEAT (ares_tac [refl, list_add_type, add_commute] 1));
val list_add_app = result();

val prems = goal List.thy
    "l: list(nat) ==> list_add(rev(l)) = list_add(l)";
by (list_ind_tac "l" prems 1);
by (ALLGOALS
    (ASM_SIMP_TAC (list_ss addrews [list_add_app, add_0_right])));
val list_add_rev = result();

val prems = goal List.thy
    "ls: list(list(nat)) ==> list_add(flat(ls)) = list_add(map(list_add,ls))";
by (list_ind_tac "ls" prems 1);
by (ALLGOALS (ASM_SIMP_TAC (list_ss addrews [list_add_app])));
by (REPEAT (ares_tac [refl, list_add_type, map_type, add_commute] 1));
val list_add_flat = result();
