(*  Title: 	ZF/ex/term
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1992  University of Cambridge

Terms over a given alphabet -- function applications; illustrates list functor
  (essentially the same type as in Trees & Forests)
*)

writeln"File ZF/ex/term.";

open Term;

(*** Monotonicity and unfolding of the function ***)

goal Term.thy "bnd_mono(univ(A), %X. A * list(X))";
by (rtac bnd_monoI 1);
by (REPEAT (ares_tac [subset_refl, list_mono, product_mono] 2));
by (REPEAT (ares_tac [subset_refl, A_subset_univ, product_subset_univ, 
		      list_subset_univ] 1));
val term_bnd_mono = result();

(** These lemmas justify using "term" in other recursive type definitions **)

val [prem] = goalw Term.thy [term_def] "A<=B ==> term(A) <= term(B)";
by (rtac lfp_mono 1);
by (REPEAT (resolve_tac [term_bnd_mono, prem, subset_refl, product_mono] 1));
val term_mono = result();

goalw Term.thy [term_def] "term(univ(A)) <= univ(A)";
by (rtac lfp_lowerbound 1);
by (rtac (A_subset_univ RS univ_mono) 2);
by (REPEAT (resolve_tac [subset_refl, list_univ, product_subset_univ] 1));
val term_univ = result();

val term_subset_univ = standard
    (term_mono RS (term_univ RSN (2,subset_trans)));

(** Type checking of <-,-> (as term constructor) **)

(* term(A) = A * list(term(A)) *)
val term_unfold = term_bnd_mono RS (term_def RS def_Tarski);

val prems = goal Term.thy "[| a: A;  l: list(term(A)) |] ==> <a,l> : term(A)";
by (rtac (term_unfold RS ssubst) 1);
by (REPEAT (resolve_tac (prems @ [SigmaI]) 1));
val termI = result();

(** Structural Induction **)

(*Induction on term(A) *)
val major::prems = goal Term.thy
    "[| t: term(A);  \
\       !!x zs. [| x: A;  zs: list({z: term(A). P(z)}) |] ==> P(<x,zs>)  \
\    |] ==> P(t)";
by (rtac (major RS (term_def RS def_induct)) 1);
by (rtac term_bnd_mono 1);
by (fast_tac (ZF_cs addIs prems) 1);
val term_induct = result();

(*Induction on term(A) followed by induction on List *)
val major::prems = goal Term.thy
    "[| t: term(A);  \
\       !!x.      [| x: A |] ==> P(<x,0>);  \
\       !!x z zs. [| x: A;  z: term(A);  zs: list(term(A));  P(<x,zs>)  \
\                 |] ==> P(<x, <z,zs>>)  \
\    |] ==> P(t)";
by (rtac (major RS term_induct) 1);
by (etac list_induct 1);
by (etac CollectE 2);
by (REPEAT (ares_tac (prems@[list_CollectD]) 1));
val term_induct2 = result();

(*Induction on term(A) to prove an equation*)
val major::prems = goal Term.thy
    "[| t: term(A);  \
\       !!x zs. [| x: A;  zs: list(term(A));  map(f,zs) = map(g,zs) |] ==> \
\               f(<x,zs>) = g(<x,zs>)  \
\    |] ==> f(t)=g(t)";
by (rtac (major RS term_induct) 1);
by (resolve_tac prems 1);
by (REPEAT (eresolve_tac [asm_rl, map_list_Collect, list_CollectD] 1));
val term_induct_eqn = result();


(*** term_rec -- by Vset recursion ***)

(*Lemma: map works correctly on the underlying list of terms*)
val [major,ordi] = goal List.thy
    "[| l: list(A);  Ord(i) |] ==>  \
\    rank(l): i --> map(%z. (lam x:Vset(i).h(x)) ` z, l) = map(h,l)";
by (rtac (major RS list_induct) 1);
by (SIMP_TAC list_ss 1);
by (rtac impI 1);
by (forward_tac [rank_pair1 RS Ord_trans] 1);
by (dtac (rank_pair2 RS Ord_trans) 2);
by (ALLGOALS (ASM_SIMP_TAC (list_ss addrews [ordi, VsetI])));
val map_lemma = result();

(*Typing premise is necessary to invoke map_lemma*)
val [prem] = goal Term.thy
    "ts: list(A) ==> \
\    term_rec(<a,ts>, d) = d(a, ts, map (%z. term_rec(z,d), ts))";
by (rtac (term_rec_def RS def_Vrec RS trans) 1);
val term_rec_ss = ZF_ss 
      addcongs (mk_typed_congs Term.thy [("d", "[i,i,i]=>i")])
      addrews [Ord_rank, rank_pair2, prem RS map_lemma];
by (SIMP_TAC term_rec_ss 1);
val term_rec = result();

(*Let z = [z1,...,zn].  Then list_rec(zs,...) = C(z1)*...*C(zn)*{0},
  which is a generalization of list(C)*)
val major::prems = goal Term.thy
    "[| t: term(A);					     \
\       !!x zs r. [| x: A;  zs: list(term(A));  \
\                    r: list_rec(zs, {0}, %z us v. C(z)*v) |]  \
\                 ==> d(x, zs, r): C(<x,zs>)  		     \
\    |] ==> term_rec(t,d) : C(t)";
by (rtac (major RS term_induct) 1);
by (forward_tac [list_CollectD] 1);
by (rtac (term_rec RS ssubst) 1);
by (REPEAT (ares_tac prems 1));
by (etac list_induct 1);
by (ALLGOALS (ASM_SIMP_TAC (list_ss addrews [term_rec])));
val term_rec_type = result();

val [rew,tslist] = goal Term.thy
    "[| !!t. j(t)==term_rec(t,d);  ts: list(A) |] ==> \
\    j(<a,ts>) = d(a, ts, map(%Z.j(Z), ts))";
by (rewtac rew);
by (rtac (tslist RS term_rec) 1);
val def_term_rec = result();

(*Could be used with term_rec_type to prove term_rec_simple_type...*)
val prems = goal List.thy
    "l: list(A) ==> list_rec(l, {0}, %z zs r. C(z)*r) <= list(UN z:A. C(z))";
by (list_ind_tac "l" prems 1);
by (ALLGOALS (SIMP_TAC list_ss));
by (fast_tac (ZF_cs addSIs [list_0I]) 1);
by (fast_tac (ZF_cs addSIs [list_PairI]) 1);
val list_rec_subset_list_UN = result();

(*...but this direct induction is simpler!*)
val major::prems = goal Term.thy
    "[| t: term(A);					     \
\       !!x zs r. [| x: A;  zs: list(term(A));  r: list(C) |]  \
\                 ==> d(x, zs, r): C  		     \
\    |] ==> term_rec(t,d) : C";
by (rtac (major RS term_induct) 1);
by (forward_tac [list_CollectD] 1);
by (rtac (term_rec RS ssubst) 1);
by (REPEAT (ares_tac prems 1));
by (etac list_induct 1);
by (ALLGOALS (ASM_SIMP_TAC (list_ss addrews [term_rec])));
val term_rec_simple_type = result();


(** term_map **)

val term_map = standard (term_map_def RS def_term_rec);

val prems = goalw Term.thy [term_map_def]
    "[| t: term(A);  !!x. x: A ==> f(x): B |] ==> term_map(f,t) : term(B)";
by (REPEAT (ares_tac ([term_rec_simple_type, termI] @ prems) 1));
val term_map_type = result();

val [major] = goal Term.thy "t: term(A) ==> term_map(f,t) : term({f(u). u:A})";
by (rtac (major RS term_map_type) 1);
by (etac RepFunI 1);
val term_map_type2 = result();


(** term_size **)

val term_size = standard (term_size_def RS def_term_rec);

val prems = goalw Term.thy [term_size_def]
    "t: term(A) ==> term_size(t) : nat";
by (REPEAT (ares_tac ([term_rec_simple_type, list_add_type, 
		       nat_succI] @ prems) 1));
val term_size_type = result();


(** reflect **)

val reflect = standard (reflect_def RS def_term_rec);

val prems = goalw Term.thy [reflect_def]
    "t: term(A) ==> reflect(t) : term(A)";
by (REPEAT (ares_tac ([term_rec_simple_type,rev_type,termI] @ prems) 1));
val reflect_type = result();

(** preorder **)

val preorder = standard (preorder_def RS def_term_rec);

val prems = goalw Term.thy [preorder_def]
    "t: term(A) ==> preorder(t) : list(A)";
by (REPEAT (ares_tac ([term_rec_simple_type,list_PairI,flat_type] @ prems) 1));
val preorder_type = result();


(** Term simplification **)

val term_typechecks =
    [termI, term_map_type, term_map_type2, term_size_type, reflect_type, 
     preorder_type];

(*map_type2 and term_map_type2 instantiate variables*)
val term_ss = list_ss 
      addcongs (mk_congs Term.thy ["term_rec","term_map","term_size",
				   "reflect","preorder"])
      addrews [term_rec, term_map, term_size, reflect,
	       preorder]
      setauto type_auto_tac (list_typechecks@term_typechecks);


(** theorems about term_map **)

val [major] = goal Term.thy
    "t: term(A) ==> term_map(%u.u, t) = t";
by (rtac (major RS term_induct_eqn) 1);
by (ASM_SIMP_TAC (term_ss addrews [map_ident]) 1);
val term_map_ident = result();

val [major] = goal Term.thy
    "t: term(A) ==> term_map(f, term_map(g,t)) = term_map(%u.f(g(u)), t)";
by (rtac (major RS term_induct_eqn) 1);
by (ASM_SIMP_TAC (term_ss addrews [map_compose]) 1);
val term_map_compose = result();

val [major] = goal Term.thy
    "t: term(A) ==> term_map(f, reflect(t)) = reflect(term_map(f,t))";
by (rtac (major RS term_induct_eqn) 1);
by (ASM_SIMP_TAC (term_ss addrews [rev_map_distrib RS sym, map_compose]) 1);
val term_map_reflect = result();


(** theorems about term_size **)

val [major] = goal Term.thy
    "t: term(A) ==> term_size(term_map(f,t)) = term_size(t)";
by (rtac (major RS term_induct_eqn) 1);
by (ASM_SIMP_TAC (term_ss addrews [map_compose]) 1);
val term_size_term_map = result();

val [major] = goal Term.thy
    "t: term(A) ==> term_size(reflect(t)) = term_size(t)";
by (rtac (major RS term_induct_eqn) 1);
by (ASM_SIMP_TAC (term_ss addrews [rev_map_distrib RS sym, map_compose,
				   list_add_rev]) 1);
val term_size_reflect = result();

val [major] = goal Term.thy
    "t: term(A) ==> term_size(t) = length(preorder(t))";
by (rtac (major RS term_induct_eqn) 1);
by (ASM_SIMP_TAC (term_ss addrews [length_flat, map_compose]) 1);
val term_size_length = result();


(** theorems about reflect **)

val [major] = goal Term.thy
    "t: term(A) ==> reflect(reflect(t)) = t";
by (rtac (major RS term_induct_eqn) 1);
by (ASM_SIMP_TAC (term_ss addrews [rev_map_distrib, map_compose,
				   map_ident, rev_rev_ident]) 1);
val reflect_reflect_ident = result();


(** theorems about preorder **)

val [major] = goal Term.thy
    "t: term(A) ==> preorder(term_map(f,t)) = map(f, preorder(t))";
by (rtac (major RS term_induct_eqn) 1);
by (ASM_SIMP_TAC (term_ss addrews [map_compose, map_flat]) 1);
val preorder_term_map = result();

(** preorder(reflect(t)) = rev(postorder(t)) **)

writeln"Reached end of file.";
