(*  Title: 	ZF/ex/sexp.ML
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1992  University of Cambridge

For sexp.thy.  S-expressions, i.e. general binary trees
*)

open Sexp;

(** Monotonicity and unfolding of the function **)

val prems = goalw Sexp.thy [Atom_def] "a: A ==> Atom(a) : univ(A)";
by (rtac (zero_in_univ RS doubleton_in_univ) 1);
by (rtac (A_subset_univ RS subsetD) 1);
by (resolve_tac prems 1);
val Atom_in_univ = result();

goal Sexp.thy "bnd_mono(univ(A), %X. RepFun(A,Atom) Un X*X)";
by (rtac bnd_monoI 1);
by (REPEAT (ares_tac [subset_refl, RepFun_mono, Un_mono, product_mono] 2));
by (REPEAT (ares_tac [subset_refl, product_subset_univ, 
		      RepFun_subset, Atom_in_univ, Un_least] 1));
val sexp_bnd_mono = result();

(* sexp(A) = RepFun(A,Atom) Un sexp(A)*sexp(A) *)
val sexp_unfold = sexp_bnd_mono RS (sexp_def RS def_Tarski);


(** Type checking of "Atom" and Pair **)

val prems = goal Sexp.thy "a: A ==> Atom(a) : sexp(A)";
by (rtac (sexp_unfold RS ssubst) 1);
by (rtac (RepFunI RS UnI1) 1);
by (resolve_tac prems 1);
val sexp_AtomI = result();

val prems = goal Sexp.thy
    "[| M: sexp(A);  N: sexp(A) |] ==> <M,N> : sexp(A)";
by (rtac (sexp_unfold RS ssubst) 1);
by (rtac (SigmaI RS UnI2) 1);
by (REPEAT (resolve_tac prems 1));
val sexp_PairI = result();


(** Structural induction on sexps **)
val major::prems = goal Sexp.thy
  "[| M: sexp(A);  \
\     !!x. x: A ==> P(Atom(x));        \
\     !!N1 N2. [| N1: sexp(A);  N2: sexp(A);  P(N1);  P(N2) |] ==> P(<N1,N2>) \
\  |] ==> P(M)";
by (rtac (major RS (sexp_def RS def_induct)) 1);
by (rtac sexp_bnd_mono 1);
by (fast_tac (ZF_cs addIs prems) 1);
val sexp_induct = result();

(*Perform induction on M, then prove the major premise using prems. *)
fun sexp_ind_tac a prems i = 
    EVERY [res_inst_tac [("M",a)] sexp_induct i,
	   rename_last_tac a ["1","2"] (i+2),
	   ares_tac prems i];


(** Injectivity properties **)

val [major] = goalw Sexp.thy [Atom_def,Pair_def]
    "Atom(a) = <M,N> ==> P";
by (cut_facts_tac [consI1 RS (major RS equalityD1 RS subsetD)] 1);
by (fast_tac (ZF_cs addEs [sym RS equals0D]) 1);
val Atom_neq_Pair = result();

val [major] = goalw Sexp.thy [Atom_def] "Atom(a) = Atom(b) ==>  a=b";
by (rtac (major RS equalityE) 1);
by (fast_tac ZF_cs 1);
val Atom_inject = result();

val [prem] = goal Sexp.thy "<M,N> : sexp(A) ==> M : sexp(A) & N : sexp(A)";
by (rtac (prem RS setup_induction) 1);
by (etac sexp_induct 1);
by (fast_tac (ZF_cs addEs [sym RS Atom_neq_Pair]) 1);
by (fast_tac (ZF_cs addEs [Pair_inject]) 1);
val sexp_PairD = result();

val sexp_PairD1 = sexp_PairD RS conjunct1
and sexp_PairD2 = sexp_PairD RS conjunct2
and sexp_PairE  = sexp_PairD RS conjE;

(** sexp_case **)

goalw Sexp.thy [sexp_case_def] "sexp_case(Atom(a),c,d) = c(a)";
by (fast_tac (ZF_cs addIs [the_equality] 
	            addEs [make_elim Atom_inject, Atom_neq_Pair]) 1);
val sexp_case_Atom = result();

goalw Sexp.thy [sexp_case_def] "sexp_case(<M,N>, c, d) = d(M,N)";
by (fast_tac (ZF_cs addIs [the_equality] 
	            addEs [Pair_inject, sym RS Atom_neq_Pair]) 1);
val sexp_case_Pair = result();

(*Type checking rules are perhaps unnecessary*)
val major::prems = goal Sexp.thy
    "[| M: sexp(A);  \
\       !!x. x: A ==> c(x) : C(Atom(x));        \
\       !!x y. [| x: sexp(A);  y: sexp(A) |] ==> d(x,y) : C(<x,y>) \
\    |] ==> sexp_case(M,c,d) : C(M)";
by (rtac (major RS sexp_induct) 1);
by (rtac (sexp_case_Pair RS ssubst) 2);
by (rtac (sexp_case_Atom RS ssubst) 1);
by (REPEAT (ares_tac prems 1));
val sexp_case_type = result();


(*** sexp_rec -- by Vset recursion ***)

(*Used just to verify sexp_rec*)
val sexp_rec_ss = ZF_ss 
      addcongs (mk_typed_congs Sexp.thy [("h", "[i,i,i,i]=>i")])
      addrews [sexp_case_Atom, sexp_case_Pair];

(** conversion rules **)

goal Sexp.thy "sexp_rec(Atom(a), c, h) = c(a)";
by (rtac (sexp_rec_def RS def_Vrec RS trans) 1);
by (SIMP_TAC sexp_rec_ss 1);
val sexp_rec_Atom = result();

goal Sexp.thy
    "sexp_rec(<M,N>, c, h) = h(M, N, sexp_rec(M,c,h), sexp_rec(N,c,h))";
by (rtac (sexp_rec_def RS def_Vrec RS trans) 1);
by (SIMP_TAC (sexp_rec_ss addrews [Vset_rankI, rank_pair1, rank_pair2]) 1);
val sexp_rec_Pair = result();

val sexp_ss = arith_ss addrews
  [sexp_case_Atom,sexp_case_Pair,
   sexp_rec_Atom,sexp_rec_Pair,
   sexp_AtomI, sexp_PairI];

(*Type checking.  This proof is vastly simpler than using wfrec_type*)
val prems = goal Sexp.thy
 "[| M: sexp(A);    \
\    !!x. x: A ==> c(x): C(Atom(x));       \
\    !!N1 N2 r1 r2. [| N1: sexp(A);  N2: sexp(A);  r1: C(N1);  r2: C(N2) \
\        	    |] ==> h(N1,N2,r1,r2): C(<N1,N2>) \
\ |] ==> sexp_rec(M,c,h) : C(M)";
by (sexp_ind_tac "M" prems 1);
by (ALLGOALS (ASM_SIMP_TAC (sexp_ss addrews prems)));
val sexp_rec_type = result();
