(*  Title: 	ZF/ex/prop-log.ML
    Author: 	Tobias Nipkow & Lawrence C Paulson
    Copyright   1992  University of Cambridge

For ex/prop-log.thy.  Inductive definition of propositional logic.
Soundness and completeness w.r.t. truth-tables.

Prove: If H|=p then G|=p where G:Fin(H)
*)

open PL;

val pl_defs = [pl_def, false_def, var_def, imp_def, is_true_def, hyps_def];
val rule_defs = [axK_def, axS_def, axDN_def, ruleMP_def];

(** Introduction rules for pl **)

goalw PL.thy pl_defs "false: pl";
by (rtac (nat_0I RS sexp_AtomI) 1);
val pl_falseI = result();

val [prem] = goalw PL.thy pl_defs "x: nat ==> #x: pl";
by (rtac (prem RS nat_succI RS sexp_AtomI) 1);
val pl_varI = result();

val prems = goalw PL.thy pl_defs
 "[| p: pl;  q: pl |] ==> p=>q : pl";
by (REPEAT (resolve_tac (prems@[sexp_PairI]) 1));
val pl_impI = result();

(** Injectivity properties **)

val [major] = goalw PL.thy pl_defs "(#v = p=>q) ==> R";
by (rtac (major RS Atom_neq_Pair) 1);
val var_neq_imp = result();

val [major] = goalw PL.thy pl_defs "#v = #w ==> v=w";
by (rtac (major RS Atom_inject RS succ_inject) 1);
val var_inject = result();

val major::prems = goalw PL.thy pl_defs
    "[| p=>q = p'=>q';  [| p=p'; q=q' |] ==> R |] ==> R";
by (rtac (major RS Pair_inject) 1);
by (REPEAT (ares_tac prems 1));
val imp_inject = result();


(** Type-checking rules **)

val [prem] = goalw PL.thy pl_defs "p=>q : pl ==> p:pl & q:pl";
by (rtac (prem RS sexp_PairD) 1);
val pl_impD = result();

val pl_impD1 = pl_impD RS conjunct1
and pl_impD2 = pl_impD RS conjunct2
and pl_impE  = pl_impD RS conjE;


(** Monotonicity and unfolding of the function **)

goalw PL.thy rule_defs
    "bnd_mono(pl, %X. (H Int pl) Un axK Un axS Un axDN Un ruleMP(X))";
by (rtac bnd_monoI 1);
by (REPEAT (ares_tac [subset_refl, Un_mono, UN_mono, Collect_mono] 2
     ORELSE etac subset_trans 2));
by (REPEAT (ares_tac [Int_lower2, Un_least, UN_least, singleton_subsetI, 
		      Collect_subset, pl_falseI, pl_varI, pl_impI] 1));
val thms_bnd_mono = result();

val [prem] = goalw PL.thy [thms_def] "G<=H ==> thms(G) <= thms(H)";
by (rtac lfp_mono 1);
by (REPEAT (resolve_tac [thms_bnd_mono, prem, 
			 subset_refl, Un_mono, Int_mono] 1));
val thms_mono = result();

goalw PL.thy (thms_def::rule_defs) "thms(H) <= pl";
by (rtac lfp_lowerbound 1);
by (rtac subset_refl 2);
by (REPEAT (ares_tac [Int_lower2, Un_least, UN_least, singleton_subsetI, 
		      Collect_subset, pl_falseI, pl_varI, pl_impI] 1));
val thms_subset_pl = result();

(*Theorems have the correct type! Only  use if you already have H|-p! *)
val [prem] = goalw PL.thy [conseq_def] "H |- p ==> p: pl";
by (rtac (prem RS (thms_subset_pl RS subsetD)) 1);
val conseq_in_pl = result();

val pl_typechecks = [pl_falseI, pl_varI, pl_impI];

(** Introduction rules for the consequence relation **)

(* thms(H) = (H Int pl) Un axK Un axS Un ruleMP(thms(H)) *)
val thms_unfold = thms_bnd_mono RS (thms_def RS def_Tarski);

(*Proof by hypothesis*)
val prems = goalw PL.thy [conseq_def]
    "[| p:H;  p:pl |] ==> H |- p";
by (rtac (thms_unfold RS ssubst) 1);
by (fast_tac (ZF_cs addSIs prems) 1);
val conseq_H = result();

(*Proof by axiom K*)
val prems = goalw PL.thy [conseq_def]
    "[| p:pl;  q:pl |] ==> H |- p=>q=>p";
by (rtac (thms_unfold RS ssubst) 1);
by (rewtac axK_def);
by (fast_tac (ZF_cs addSIs prems) 1);
val conseq_K = result();

(*Proof by axiom S*)
val prems = goalw PL.thy [conseq_def]
    "[| p:pl;  q:pl;  r:pl |] ==> H |- (p=>q=>r) => (p=>q) => p => r";
by (rtac (thms_unfold RS ssubst) 1);
by (rewtac axS_def);
by (fast_tac (ZF_cs addSIs prems) 1);
val conseq_S = result();

(*Proof by axiom DN (double negation) *)
val prems = goalw PL.thy [conseq_def]
    "p:pl ==> H |- ((p=>false) => false) => p";
by (rtac (thms_unfold RS ssubst) 1);
by (rewtac axDN_def);
by (fast_tac (ZF_cs addSIs prems) 1);
val conseq_DN = result();

(*Proof by rule MP (Modus Ponens) *)
val [prempq,premp] = goalw PL.thy [conseq_def]
    "[| H |- p=>q;  H |- p |] ==> H |- q";
by (rtac (thms_unfold RS ssubst) 1);
by (rewtac ruleMP_def);
(*Insert the facts p:pl and q:pl*)
val thms_in_pl = thms_subset_pl RS subsetD;
by (rtac (prempq RS thms_in_pl RS pl_impE) 1);
by (fast_tac (ZF_cs addSIs [premp,prempq]) 1);
val conseq_MP = result();

(*Rule is called I for Identity Combinator, not for Introduction*)
val prems = goal PL.thy "p:pl ==> H |- p=>p";
by (rtac (conseq_S RS conseq_MP RS conseq_MP) 1);
by (rtac conseq_K 5);
by (rtac conseq_K 4);
by (REPEAT (resolve_tac (prems@[pl_impI]) 1));
val conseq_I = result();

(** Weakening, left and right **)

(*This order of premises is convenient with RS*)
val prems = goalw PL.thy [conseq_def] "[| G<=H;  G |- p |] ==> H |- p";
by (rtac (thms_mono RS subsetD) 1);
by (REPEAT (resolve_tac prems 1));
val weaken_left = result();

(* H |- p ==> cons(a,H) |- p *)
val weaken_left_cons = subset_consI RS weaken_left;

val weaken_left_Un1  =    Un_upper1 RS weaken_left;
val weaken_left_Un2  =    Un_upper2 RS weaken_left;

val prems = goal PL.thy "[| H |- q;  p:pl |] ==> H |- p=>q";
by (rtac (conseq_K RS conseq_MP) 1);
by (REPEAT (resolve_tac (prems@[conseq_in_pl]) 1));
val weaken_right = result();

(** Rule induction for H|-p **)

(*Careful unfolding/folding to avoid a big expansion*)
val major::prems = goalw PL.thy [conseq_def]
  "[| H |- a;							\
\     !!x. [| x:pl; x:H |] ==> P(x);				\
\     !!x y. [| x:pl; y:pl |] ==> P(x=>y=>x);			\
\     !!x y z. [| x:pl; y:pl; z:pl |] ==> 			\
\              P((x=>y=>z)=>(x=>y)=>x=>z);			\
\     !!x. x:pl ==> P(((x=>false)=>false)=>x);			\
\     !!x y. [| H |- x=>y;  H |- x;  P(x=>y);  P(x) |] ==> P(y) \
\  |] ==> P(a)";
by (rtac (major RS (thms_def RS def_induct)) 1);
by (rtac thms_bnd_mono 1);
by (rewrite_tac rule_defs);
by (fast_tac (ZF_cs addIs prems addSEs [cons_subsetE]) 1);
val conseq_induct = result();

(*The deduction theorem*)
val [major,minor] = goal PL.thy "[| cons(p,H) |- q;  p:pl |] ==>  H |- p=>q";
by (rtac (major RS conseq_induct) 1);
by (fast_tac (ZF_cs addIs [minor, conseq_I, conseq_H RS weaken_right]) 1);
by (fast_tac (ZF_cs addIs [minor, conseq_K RS weaken_right]) 1);
by (fast_tac (ZF_cs addIs [minor, conseq_S RS weaken_right]) 1);
by (fast_tac (ZF_cs addIs [minor, conseq_DN RS weaken_right]) 1);
by (fast_tac (ZF_cs addIs [minor, conseq_S RS conseq_MP RS conseq_MP]
                    addSEs [conseq_in_pl, conseq_in_pl RS pl_impD2]) 1);
val deduction = result();


(*The cut rule*)
val prems = goal PL.thy "[| H|-p;  cons(p,H) |- q |] ==>  H |- q";
by (rtac (deduction RS conseq_MP) 1);
by (REPEAT (resolve_tac (prems@[conseq_in_pl]) 1));
val cut = result();

val prems = goal PL.thy "[| H |- false; p:pl |] ==> H |- p";
by (rtac (conseq_DN RS conseq_MP) 1);
by (rtac weaken_right 2);
by (REPEAT (resolve_tac (prems@pl_typechecks@[consI1]) 1));
val conseq_falseE = result();

(* [| H |- p=>false;  H |- p;  q: pl |] ==> H |- q *)
val conseq_notE = standard (conseq_MP RS conseq_falseE);

(** The function is_true **)

goalw PL.thy pl_defs "is_true(false,t) <-> False";
by (SIMP_TAC (sexp_ss addrews [one_not_0 RS not_sym]) 1);
val is_true_false = result();

goalw PL.thy pl_defs "is_true(#v,t) <-> v:t";
by (SIMP_CASE_TAC (sexp_ss addrews [one_not_0 RS not_sym]) 1);
val is_true_var = result();

goalw PL.thy pl_defs "is_true(p=>q,t) <-> (is_true(p,t)-->is_true(q,t))";
by (SIMP_CASE_TAC sexp_ss 1);
val is_true_imp = result();

(** The function hyps **)

goalw PL.thy pl_defs "hyps(false,t) = 0";
by (SIMP_TAC sexp_ss 1);
val hyps_false = result();

goalw PL.thy pl_defs "hyps(#v,t) = {if(v:t, #v, #v=>false)}";
by (SIMP_TAC sexp_ss 1);
val hyps_var = result();

goalw PL.thy pl_defs "hyps(p=>q,t) = hyps(p,t) Un hyps(q,t)";
by (SIMP_TAC sexp_ss 1);
val hyps_imp = result();

val pl_ss = sexp_ss 
    addcongs (mk_congs PL.thy 
	      ["Fin","var","op =>","op |-","op |=","is_true","hyps"])
    addrews (pl_typechecks @
	     [is_true_false, is_true_var, is_true_imp,
	      hyps_false, hyps_var, hyps_imp]);


(*Soundness of the rules wrt truth-table semantics*)
val [major] = goalw PL.thy [sat_def] "H |- p ==> H |= p";
by (rtac (major RS conseq_induct) 1);
by (ALLGOALS (ASM_SIMP_TAC pl_ss));
by (fast_tac (ZF_cs addSDs [is_true_imp RS iffD1 RS mp]) 1);
val soundness = result();

(** Structural induction on pl **)

val major::prems = goalw PL.thy pl_defs
  "[| q: pl;  				\
\     P(false);				\
\     !!v. v:nat ==> P(#v);        \
\     !!q1 q2. [| q1: pl;  q2: pl;  P(q1);  P(q2) |] ==> P(q1=>q2) \
\  |] ==> P(q)";
by (rtac (major RS sexp_induct) 1);
by (etac nat_induct 1);
by (REPEAT (ares_tac prems 1));
val pl_induct = result();

(*** Towards the completeness proof ***)

val [premf,premq] = goal PL.thy "[| H |- p=>false; q: pl |] ==> H |- p=>q";
by (rtac (premf RS conseq_in_pl RS pl_impE) 1);
by (rtac deduction 1);
by (rtac (premf RS weaken_left_cons RS conseq_notE) 1);
by (REPEAT (ares_tac [premq, consI1, conseq_H] 1));
val false_imp = result();

val [premp,premq] = goal PL.thy
    "[| H |- p;  H |- q=>false |] ==> H |- (p=>q)=>false";
by (cut_facts_tac ([premp,premq] RL [conseq_in_pl]) 1);
by (etac pl_impE 1);
by (rtac deduction 1);
by (rtac (premq RS weaken_left_cons RS conseq_MP) 1);
by (rtac (consI1 RS conseq_H RS conseq_MP) 1);
by (rtac (premp RS weaken_left_cons) 2);
by (REPEAT (ares_tac pl_typechecks 1));
val imp_false = result();

(*This formulation is required for strong induction hypotheses*)
val [major] = goal PL.thy 
    "p: pl ==> hyps(p,t) |- if(is_true(p,t), p, p=>false)";
by (rtac (expand_if RS iffD2) 1);
by (rtac (major RS pl_induct) 1);
by (ALLGOALS (ASM_SIMP_TAC (pl_ss addrews [conseq_I, conseq_H])));
by (fast_tac (ZF_cs addIs [weaken_left_Un1, weaken_left_Un2, 
			   weaken_right, imp_false]
                    addSEs [false_imp]) 1);
val hyps_conseq_if = result();

(*Key lemma for completeness; yields a set of assumptions satisfying p*)
val [premp,sat] = goalw PL.thy [sat_def]
    "[| p: pl;  0 |= p |] ==> hyps(p,t) |- p";
by (rtac (sat RS spec RS mp RS if_P RS subst) 1 THEN
    rtac (premp RS hyps_conseq_if) 2);
by (fast_tac ZF_cs 1);
val sat_conseq_p = result();

(*For proving certain theorems in our new propositional logic*)
val conseq_cs = 
    ZF_cs addSIs [pl_falseI, pl_varI, pl_impI, deduction]
          addIs [conseq_in_pl, conseq_H, conseq_H RS conseq_MP];

(*The excluded middle in the form of an elimination rule*)
val prems = goal PL.thy
    "[| p: pl;  q: pl |] ==> H |- (p=>q) => ((p=>false)=>q) => q";
by (rtac (deduction RS deduction) 1);
by (rtac (conseq_DN RS conseq_MP) 1);
by (ALLGOALS (best_tac (conseq_cs addSIs prems)));
val conseq_excluded_middle = result();

(*Hard to prove directly because it requires cuts*)
val prems = goal PL.thy
    "[| cons(p,H) |- q;  cons(p=>false,H) |- q;  p: pl |] ==> H |- q";
by (rtac (conseq_excluded_middle RS conseq_MP RS conseq_MP) 1);
by (REPEAT (resolve_tac (prems@pl_typechecks@[deduction,conseq_in_pl]) 1));
val conseq_excluded_middle_rule = result();

(*** Completeness -- lemmas for reducing the set of assumptions ***)

(*For the case hyps(p,t)-cons(#v,Y) |- p;
  we also have hyps(p,t)-{#v} <= hyps(p, t-{v}) *)
val [major] = goal PL.thy
    "p: pl ==> hyps(p, t-{v}) <= cons(#v=>false, hyps(p,t)-{#v})";
by (rtac (major RS pl_induct) 1);
by (SIMP_TAC pl_ss 1);
by (ASM_SIMP_CASE_TAC pl_ss 1);
by (fast_tac (ZF_cs addSEs [sym RS var_neq_imp] addSDs [var_inject]) 1);
by (ASM_SIMP_TAC pl_ss 1);
by (fast_tac ZF_cs 1);
val hyps_Diff = result();

(*For the case hyps(p,t)-cons(#v => false,Y) |- p;
  we also have hyps(p,t)-{#v=>false} <= hyps(p, cons(v,t)) *)
val [major] = goal PL.thy
    "p: pl ==> hyps(p, cons(v,t)) <= cons(#v, hyps(p,t)-{#v=>false})";
by (rtac (major RS pl_induct) 1);
by (SIMP_TAC pl_ss 1);
by (ASM_SIMP_CASE_TAC pl_ss 1);
by (fast_tac (ZF_cs addSEs [var_neq_imp, imp_inject] addSDs [var_inject]) 1);
by (ASM_SIMP_TAC pl_ss 1);
by (fast_tac ZF_cs 1);
val hyps_cons = result();

(** Two lemmas for use with weaken_left **)

goal ZF.thy "B-C <= cons(a, B-cons(a,C))";
by (fast_tac ZF_cs 1);
val cons_Diff_same = result();

goal ZF.thy "cons(a, B-{c}) - D <= cons(a, B-cons(c,D))";
by (fast_tac ZF_cs 1);
val cons_Diff_subset2 = result();

(*The set hyps(p,t) is finite, and elements have the form #v or #v=>false;
 could probably prove the stronger hyps(p,t) : Fin(hyps(p,0) Un hyps(p,nat))*)
val [major] = goal PL.thy
    "p: pl ==> hyps(p,t) : Fin(UN v:nat. {#v, #v=>false})";
by (rtac (major RS pl_induct) 1);
by (ASM_SIMP_CASE_TAC (pl_ss addrews [Fin_0I, Fin_consI, UN_I]) 2);
by (ALLGOALS (ASM_SIMP_TAC (pl_ss addrews [Un_0, Fin_0I, Fin_UnI])));
val hyps_finite = result();

val Diff_weaken_left = subset_refl RSN (2, Diff_mono) RS weaken_left;

(*Induction on the finite set of assumptions hyps(p,t0).
  We may repeatedly subtract assumptions until none are left!*)
val [premp,sat] = goal PL.thy
    "[| p: pl;  0 |= p |] ==> ALL t. hyps(p,t) - hyps(p,t0) |- p";
by (rtac (premp RS hyps_finite RS Fin_induct) 1);
by (SIMP_TAC (pl_ss addrews [premp, sat, sat_conseq_p, Diff_0]) 1);
by (safe_tac ZF_cs);
(*Case hyps(p,t)-cons(#v,Y) |- p *)
by (rtac conseq_excluded_middle_rule 1);
by (etac pl_varI 3);
by (rtac (cons_Diff_same RS weaken_left) 1);
by (etac spec 1);
by (rtac (cons_Diff_subset2 RS weaken_left) 1);
by (rtac (premp RS hyps_Diff RS Diff_weaken_left) 1);
by (etac spec 1);
(*Case hyps(p,t)-cons(#v => false,Y) |- p *)
by (rtac conseq_excluded_middle_rule 1);
by (etac pl_varI 3);
by (rtac (cons_Diff_same RS weaken_left) 2);
by (etac spec 2);
by (rtac (cons_Diff_subset2 RS weaken_left) 1);
by (rtac (premp RS hyps_cons RS Diff_weaken_left) 1);
by (etac spec 1);
val completeness_0_lemma = result();

(*The base case for completeness*)
val [premp,sat] = goal PL.thy "[| p: pl;  0 |= p |] ==> 0 |- p";
by (rtac (Diff_cancel RS subst) 1);
by (rtac (sat RS (premp RS completeness_0_lemma RS spec)) 1);
val completeness_0 = result();

(*A semantic analogue of the Deduction Theorem*)
val [sat] = goalw PL.thy [sat_def] "[| cons(p,H) |= q |] ==> H |= p=>q";
by (SIMP_TAC pl_ss 1);
by (cfast_tac [sat] 1);
val sat_imp = result();

val [finite] = goal PL.thy "H: Fin(pl) ==> ALL p:pl. H |= p --> H |- p";
by (rtac (finite RS Fin_induct) 1);
by (safe_tac (ZF_cs addSIs [completeness_0]));
by (rtac (weaken_left_cons RS conseq_MP) 1);
by (fast_tac (ZF_cs addSIs [sat_imp,pl_impI]) 1);
by (fast_tac conseq_cs 1);
val completeness_lemma = result();

val completeness = completeness_lemma RS bspec RS mp;

val [finite] = goal PL.thy "H: Fin(pl) ==> H |- p <-> H |= p & p:pl";
by (fast_tac (ZF_cs addSEs [soundness, finite RS completeness, 
			    conseq_in_pl]) 1);
val conseq_iff = result();

writeln"Reached end of file.";


