(*  Title: 	ZF/bt.ML
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1992  University of Cambridge

For bt.thy.  Binary trees
*)

open BT;

(** Monotonicity and unfolding of the function **)

goal BT.thy "bnd_mono(univ(A), %X. {0} Un A*X*X)";
by (rtac bnd_monoI 1);
by (REPEAT (ares_tac [subset_refl, Un_mono, product_mono] 2));
by (REPEAT (resolve_tac [subset_refl, zero_in_univ RS singleton_subsetI, 
		         A_subset_univ, Un_least, product_subset_univ] 1));
val bt_bnd_mono = result();

(** These lemmas justify using "bt" in other recursive type definitions **)

val [prem] = goalw BT.thy [bt_def] "A<=B ==> bt(A) <= bt(B)";
by (rtac lfp_mono 1);
by (REPEAT (resolve_tac [bt_bnd_mono, prem, 
			 subset_refl, Un_mono, product_mono] 1));
val bt_mono = result();

goalw BT.thy [bt_def] "bt(univ(A)) <= univ(A)";
by (rtac lfp_lowerbound 1);
by (rtac (A_subset_univ RS univ_mono) 2);
by (REPEAT (resolve_tac [subset_refl, zero_in_univ RS singleton_subsetI,
			 Un_least, product_subset_univ] 1));
val bt_univ = result();

val bt_subset_univ = standard (bt_mono RS (bt_univ RSN (2,subset_trans)));


(** Type checking of 0 (as nil) and <-,-,-> (as branching) **)

(* bt(A) = {0} Un (A * bt(A) * bt(A)) *)
val bt_unfold = bt_bnd_mono RS (bt_def RS def_Tarski);

goal BT.thy "0 : bt(A)";
by (rtac (bt_unfold RS ssubst) 1);
by (rtac (singletonI RS UnI1) 1);
val bt_0I = result();

val prems = goal BT.thy "[| a: A;  l: bt(A);  r: bt(A) |] ==> <a,l,r> : bt(A)";
by (rtac (bt_unfold RS ssubst) 1);
by (REPEAT (resolve_tac (prems @ [SigmaI, UnI2]) 1));
val bt_BrI = result();

(** Structural induction on binary trees **)

val major::prems = goal BT.thy
    "[| t: bt(A);  \
\       P(0);        \
\       !!x y z. [| x:A;  y:bt(A);  z:bt(A);  P(y);  P(z) |] ==> P(<x,y,z>) \
\    |] ==> P(t)";
by (rtac (major RS (bt_def RS def_induct)) 1);
by (rtac bt_bnd_mono 1);
by (fast_tac (ZF_cs addIs prems) 1);
val bt_induct = result();

(*Perform induction on l, then prove the major premise using prems. *)
fun bt_ind_tac a prems i = 
    EVERY [res_inst_tac [("t",a)] bt_induct i,
	   rename_last_tac a ["1","2"] (i+2),
	   ares_tac prems i];

(** bt_case **)

goalw BT.thy [bt_case_def] "bt_case(0,c,h) = c";
by (fast_tac (ZF_cs addIs [the_equality]) 1);
val bt_case_0 = result();

goalw BT.thy [bt_case_def] "bt_case(<a,l,r>, c, h) = h(a,l,r)";
by (fast_tac (ZF_cs addIs [the_equality]) 1);
val bt_case_Br = result();

val major::prems = goal BT.thy
    "[| t: bt(A);    \
\       c: C(0);       \
\       !!x y z. [| x: A;  y: bt(A);  y: bt(A) |] ==> h(x,y,z): C(<x,y,z>)  \
\    |] ==> bt_case(t,c,h) : C(t)";
by (rtac (major RS bt_induct) 1);
by (ALLGOALS (ASM_SIMP_TAC (ZF_ss addrews
			    ([bt_case_0,bt_case_Br]@prems))));
val bt_case_type = result();


(** bt_rec -- by Vset recursion **)

(*Used to verify bt_rec*)
val bt_rec_ss = ZF_ss 
      addcongs (mk_typed_congs BT.thy [("h", "[i,i,i,i,i]=>i")])
      addrews [bt_case_0, bt_case_Br];

goal BT.thy "bt_rec(0,c,h) = c";
by (rtac (bt_rec_def RS def_Vrec RS trans) 1);
by (SIMP_TAC bt_rec_ss 1);
val bt_rec_0 = result();

goal BT.thy "bt_rec(<a,l,r>, c, h) = h(a, l, r, bt_rec(l,c,h), bt_rec(r,c,h))";
by (rtac (bt_rec_def RS def_Vrec RS trans) 1);
by (SIMP_TAC (bt_rec_ss addrews [Vset_rankI, rank_pair2 RSN (2,rank_trans),
				 rank_pair1, rank_pair2]) 1);
val bt_rec_Br = result();

(*Type checking -- proved by induction, as usual*)
val prems = goal BT.thy
    "[| t: bt(A);    \
\       c: C(0);       \
\       !!x y z r s. [| x:A;  y:bt(A);  z:bt(A);  r:C(y);  s:C(z) |] ==> \
\		     h(x,y,z,r,s): C(<x,y,z>)  \
\    |] ==> bt_rec(t,c,h) : C(t)";
by (bt_ind_tac "t" prems 1);
by (ALLGOALS (ASM_SIMP_TAC (ZF_ss addrews
			    (prems@[bt_rec_0,bt_rec_Br]))));
val bt_rec_type = result();

(** Versions for use with definitions **)

val [rew] = goal BT.thy "[| !!t. j(t)==bt_rec(t, c, h) |] ==> j(0) = c";
by (rewtac rew);
by (rtac bt_rec_0 1);
val def_bt_rec_0 = result();

val [rew] = goal BT.thy
    "[| !!t. j(t)==bt_rec(t, c, h) |] ==> j(<a,l,r>) = h(a,l,r,j(l),j(r))";
by (rewtac rew);
by (rtac bt_rec_Br 1);
val def_bt_rec_Br = result();

fun bt_recs def = 
      (standard (def RS def_bt_rec_0),
       standard (def RS def_bt_rec_Br));


(** n_nodes **)

val (n_nodes_0,n_nodes_Br) = bt_recs n_nodes_def;

val prems = goalw BT.thy [n_nodes_def] 
    "xs: bt(A) ==> n_nodes(xs) : nat";
by (REPEAT (ares_tac (prems @ [bt_rec_type, nat_0I, nat_succI, add_type]) 1));
val n_nodes_type = result();


(** n_leaves **)

val (n_leaves_0,n_leaves_Br) = bt_recs n_leaves_def;

val prems = goalw BT.thy [n_leaves_def] 
    "xs: bt(A) ==> n_leaves(xs) : nat";
by (REPEAT (ares_tac (prems @ [bt_rec_type, nat_0I, nat_succI, add_type]) 1));
val n_leaves_type = result();

(** bt_reflect **)

val (bt_reflect_0, bt_reflect_Br) = bt_recs bt_reflect_def;

val prems = goalw BT.thy [bt_reflect_def] 
    "xs: bt(A) ==> bt_reflect(xs) : bt(A)";
by (REPEAT (ares_tac (prems @ [bt_rec_type, bt_0I, bt_BrI]) 1));
val bt_reflect_type = result();


(** BT simplification **)


val bt_typechecks =
      [bt_0I, bt_BrI, bt_case_type, bt_rec_type,
       n_nodes_type, n_leaves_type, bt_reflect_type];

val bt_congs = mk_congs BT.thy
    ["bt_case","bt_rec","n_nodes","n_leaves","bt_reflect"];

val bt_ss = arith_ss 
    addcongs bt_congs
    addrews([bt_case_0, bt_case_Br, 
	     bt_rec_0, bt_rec_Br, 
	     n_nodes_0, n_nodes_Br,
	     n_leaves_0, n_leaves_Br,
	     bt_reflect_0, bt_reflect_Br] @ bt_typechecks);



(*** theorems about n_leaves ***)

val prems = goal BT.thy
    "t: bt(A) ==> n_leaves(bt_reflect(t)) = n_leaves(t)";
by (bt_ind_tac "t" prems 1);
by (ALLGOALS (ASM_SIMP_TAC bt_ss));
by (REPEAT (ares_tac [add_commute, n_leaves_type] 1));
val n_leaves_reflect = result();

val prems = goal BT.thy
    "t: bt(A) ==> n_leaves(t) = succ(n_nodes(t))";
by (bt_ind_tac "t" prems 1);
by (ALLGOALS (ASM_SIMP_TAC (bt_ss addrews [add_succ_right])));
val n_leaves_nodes = result();

(*** theorems about bt_reflect ***)

val prems = goal BT.thy
    "t: bt(A) ==> bt_reflect(bt_reflect(t))=t";
by (bt_ind_tac "t" prems 1);
by (ALLGOALS (ASM_SIMP_TAC bt_ss));
val bt_reflect_bt_reflect_ident = result();


