(*  Title: 	ZF/ex/bin.ML
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1993  University of Cambridge

For bin.thy.  Arithmetic on binary integers.
*)

open Bin;

(** Monotonicity and unfolding of the function **)

goal Bin.thy "bnd_mono(univ(0), %X. bool + X*bool)";
by (rtac bnd_monoI 1);
by (REPEAT (ares_tac [subset_refl, sum_mono, product_mono] 2));
by (REPEAT (resolve_tac [subset_refl, bool_subset_univ,
		         sum_subset_univ, product_subset_univ] 1));
val bin_bnd_mono = result();


(** Type checking of Plus, Minus, $$ **)

(* bin(A) = {0} Un (A * bin(A)) *)
val bin_unfold = bin_bnd_mono RS (bin_def RS def_Tarski);

goalw Bin.thy [Plus_def] "Plus : bin";
by (rtac (bin_unfold RS ssubst) 1);
by (rtac (bool_1I RS sum_InlI) 1);
val bin_PlusI = result();

goalw Bin.thy [Minus_def] "Minus : bin";
by (rtac (bin_unfold RS ssubst) 1);
by (rtac (bool_0I RS sum_InlI) 1);
val bin_MinusI = result();

val prems = goalw Bin.thy [bcons_def]
    "[| w: bin; b: bool |] ==> w$$b : bin";
by (rtac (bin_unfold RS ssubst) 1);
by (REPEAT (resolve_tac (prems @ [SigmaI RS sum_InrI]) 1));
val bin_bconsI = result();

(** Structural induction on bins **)

val major::prems = goalw Bin.thy[Plus_def,Minus_def,bcons_def]
    "[| w: bin;	\
\       P(Plus);	\
\       P(Minus);	\
\       !!w x. [| w: bin;  x: bool;  P(w) |] ==> P(w$$x)	\
\    |] ==> P(w)";
by (rtac (major RS (bin_def RS def_induct)) 1);
by (rtac bin_bnd_mono 1);
by (fast_tac (ZF_cs addIs (prems@[bool_1I,bool_0I]) addSEs [sumE,boolE]) 1);
val bin_induct = result();

(*Perform induction on l, then prove the major premise using prems. *)
fun bin_ind_tac a prems i = 
    EVERY [res_inst_tac [("w",a)] bin_induct i,
	   rename_last_tac a ["1"] (i+3),
	   ares_tac prems i];

(** bin_rec -- by Vset recursion **)

(*Used to verify bin_rec*)
val bin_rec_ss = ZF_ss 
      addcongs (mk_typed_congs Bin.thy [("h", "[i,i,i]=>i")])
      addrews [case_Inl,case_Inr,cond_1,cond_0];

goalw Bin.thy [Plus_def] "bin_rec(Plus,a,b,h) = a";
by (rtac (bin_rec_def RS def_Vrec RS trans) 1);
by (SIMP_TAC bin_rec_ss 1);
val bin_rec_Plus = result();

goalw Bin.thy [Minus_def] "bin_rec(Minus,a,b,h) = b";
by (rtac (bin_rec_def RS def_Vrec RS trans) 1);
by (SIMP_TAC bin_rec_ss 1);
val bin_rec_Minus = result();

goalw Bin.thy [bcons_def]
    "bin_rec(w$$x,a,b,h) = h(w, x, bin_rec(w,a,b,h))";
by (rtac (bin_rec_def RS def_Vrec RS trans) 1);
by (SIMP_TAC (bin_rec_ss addrews [Vset_rankI, rank_pair1, 
				  rank_Inr RSN (2,rank_trans)]) 1);
val bin_rec_bcons = result();

(*Type checking*)
val prems = goal Bin.thy
    "[| w: bin;    \
\       a: C(Plus);   b: C(Minus);       \
\       !!w x r. [| w: bin;  x: bool;  r: C(w) |] ==> h(w,x,r): C(w$$x)  \
\    |] ==> bin_rec(w,a,b,h) : C(w)";
by (bin_ind_tac "w" prems 1);
by (ALLGOALS 
    (ASM_SIMP_TAC (ZF_ss addrews (prems@[bin_rec_Plus,bin_rec_Minus,
					 bin_rec_bcons]))));
val bin_rec_type = result();

(** Versions for use with definitions **)

val [rew] = goal Bin.thy
    "[| !!w. j(w)==bin_rec(w,a,b,h) |] ==> j(Plus) = a";
by (rewtac rew);
by (rtac bin_rec_Plus 1);
val def_bin_rec_Plus = result();

val [rew] = goal Bin.thy
    "[| !!w. j(w)==bin_rec(w,a,b,h) |] ==> j(Minus) = b";
by (rewtac rew);
by (rtac bin_rec_Minus 1);
val def_bin_rec_Minus = result();

val [rew] = goal Bin.thy
    "[| !!w. j(w)==bin_rec(w,a,b,h) |] ==> j(w$$x) = h(w,x,j(w))";
by (rewtac rew);
by (rtac bin_rec_bcons 1);
val def_bin_rec_bcons = result();

fun bin_recs def = 
      [standard (def RS def_bin_rec_Plus),
       standard (def RS def_bin_rec_Minus),
       standard (def RS def_bin_rec_bcons)];

(** Type checking **)

val bin_typechecks0 =
    [bin_PlusI, bin_MinusI, bin_bconsI, bin_rec_type];

val prems = goalw Bin.thy [integ_of_bin_def]
    "w: bin ==> integ_of_bin(w) : integ";
by (typechk_tac (prems@[bool_subset_nat RS subsetD]@
		 bin_typechecks0@integ_typechecks@nat_typechecks));
val integ_of_bin_type = result();

val prems = goalw Bin.thy [bin_succ_def]
    "w: bin ==> bin_succ(w) : bin";
by (typechk_tac (prems@bin_typechecks0@bool_typechecks));
val bin_succ_type = result();

val prems = goalw Bin.thy [bin_pred_def]
    "w: bin ==> bin_pred(w) : bin";
by (typechk_tac (prems@bin_typechecks0@bool_typechecks));
val bin_pred_type = result();

val prems = goalw Bin.thy [bin_minus_def]
    "w: bin ==> bin_minus(w) : bin";
by (typechk_tac (prems@[bin_pred_type]@bin_typechecks0@bool_typechecks));
val bin_minus_type = result();

val prems = goalw Bin.thy [bin_add_def]
    "[| v: bin; w: bin |] ==> bin_add(v,w) : bin";
by (typechk_tac (prems@[bin_succ_type,bin_pred_type]@bin_typechecks0@
		 bool_typechecks@ZF_typechecks));
val bin_add_type = result();

val prems = goalw Bin.thy [bin_mult_def]
    "[| v: bin; w: bin |] ==> bin_mult(v,w) : bin";
by (typechk_tac (prems@[bin_minus_type,bin_add_type]@bin_typechecks0@
		 bool_typechecks));
val bin_mult_type = result();

val bin_typechecks = bin_typechecks0 @
    [integ_of_bin_type, bin_succ_type, bin_pred_type, 
     bin_minus_type, bin_add_type, bin_mult_type];

val bin_congs = mk_congs Bin.thy
    ["bin_rec","op $$","integ_of_bin","bin_succ","bin_pred",
     "bin_minus","bin_add","bin_mult"];

val bin_ss = integ_ss 
    addcongs (bin_congs@bool_congs)
    addrews([bool_1I, bool_0I,
	     bin_rec_Plus, bin_rec_Minus, bin_rec_bcons] @ 
	     bin_recs integ_of_bin_def @ bool_rews @ bin_typechecks);

val typechecks = bin_typechecks @ integ_typechecks @ nat_typechecks @
                 [bool_subset_nat RS subsetD];

(**** The carry/borrow functions, bin_succ and bin_pred ****)

(** Lemmas **)

val prems = goal Integ.thy 
    "[| z $+ v = z' $+ v';  \
\       z: integ; z': integ;  v: integ; v': integ;  w: integ |]   \
\    ==> z $+ (v $+ w) = z' $+ (v' $+ w)";
by (ASM_SIMP_TAC (integ_ss addrews ([zadd_assoc RS sym] @ prems)) 1);
val zadd_assoc_cong = result();

val prems = goal Integ.thy 
    "[| z: integ;  v: integ;  w: integ |]   \
\    ==> z $+ (v $+ w) = v $+ (z $+ w)";
by (REPEAT (resolve_tac ([zadd_commute RS zadd_assoc_cong] @ prems) 1));
val zadd_assoc_swap = result();

val [zadd_cong] = mk_congs Integ.thy ["op $+"];

val zadd_kill = (refl RS zadd_cong);
val zadd_assoc_swap_kill = zadd_kill RSN (4, zadd_assoc_swap RS trans);

(*Pushes 'constants' of the form $#m to the right -- LOOPS if two!*)
val zadd_assoc_znat = standard (znat_type RS zadd_assoc_swap);

val prems = goal Integ.thy 
    "[| z: integ;  w: integ |]   \
\    ==> w $+ (z $+ (w $+ z)) = w $+ (w $+ (z $+ z))";
by (REPEAT (resolve_tac (prems@[zadd_kill, zadd_assoc_swap]) 1));
val zadd_swap_pairs = result();


val carry_ss = bin_ss addrews 
               (bin_recs bin_succ_def @ bin_recs bin_pred_def);

val prems = goal Bin.thy
    "w: bin ==> integ_of_bin(bin_succ(w)) = $#1 $+ integ_of_bin(w)";
by (bin_ind_tac "w" prems 1);
by (SIMP_TAC (carry_ss addrews [zadd_0_right]) 1);
by (SIMP_TAC (carry_ss addrews [zadd_zminus_inverse]) 1);
by (etac boolE 1);
by (ALLGOALS (ASM_SIMP_TAC (carry_ss addrews [zadd_assoc])));
by (REPEAT (ares_tac (zadd_swap_pairs::typechecks) 1));
val integ_of_bin_succ = result();

val prems = goal Bin.thy
    "w: bin ==> \
\    integ_of_bin(bin_pred(w)) = $~ ($#1) $+ integ_of_bin(w)";
by (bin_ind_tac "w" prems 1);
by (SIMP_TAC (carry_ss addrews [zadd_0_right]) 1);
by (SIMP_TAC (carry_ss addrews [zadd_zminus_inverse]) 1);
by (etac boolE 1);
by (ALLGOALS 
    (ASM_SIMP_TAC 
     (carry_ss addrews [zadd_assoc RS sym,
			zadd_zminus_inverse, zadd_zminus_inverse2])));
by (REPEAT (ares_tac ([zadd_commute, zadd_cong, refl]@typechecks) 1));
val integ_of_bin_pred = result();

(*These two results replace the definitions of bin_succ and bin_pred*)


(*** bin_minus: (unary!) negation of binary integers ***)

val bin_minus_ss =
    bin_ss addrews (bin_recs bin_minus_def @
		    [integ_of_bin_succ, integ_of_bin_pred]);

val prems = goal Bin.thy
    "w: bin ==> integ_of_bin(bin_minus(w)) = $~ integ_of_bin(w)";
by (bin_ind_tac "w" prems 1);
by (SIMP_TAC (bin_minus_ss addrews [zminus_0]) 1);
by (SIMP_TAC (bin_minus_ss addrews [zadd_0_right]) 1);
by (etac boolE 1);
by (ALLGOALS 
    (ASM_SIMP_TAC (bin_minus_ss addrews [zminus_zadd_distrib, zadd_assoc])));
val integ_of_bin_minus = result();


(*** bin_add: binary addition ***)

val prems = goalw Bin.thy [bin_add_def]
    "w: bin ==> bin_add(Plus,w) = w";
by (SIMP_TAC (bin_ss addrews prems) 1);
val bin_add_Plus = result();

val prems = goalw Bin.thy [bin_add_def]
    "w: bin ==> bin_add(Minus,w) = bin_pred(w)";
by (SIMP_TAC (bin_ss addrews prems) 1);
val bin_add_Minus = result();

goalw Bin.thy [bin_add_def]
    "bin_add(v$$x,Plus) = v$$x";
by (SIMP_TAC bin_ss 1);
val bin_add_bcons_Plus = result();

goalw Bin.thy [bin_add_def]
    "bin_add(v$$x,Minus) = bin_pred(v$$x)";
by (SIMP_TAC bin_ss 1);
val bin_add_bcons_Minus = result();

val prems = goalw Bin.thy [bin_add_def]
    "[| w: bin;  y: bool |] ==> \
\    bin_add(v$$x, w$$y) = \
\    bin_add(v, cond(x and y, bin_succ(w), w)) $$ (x xor y)";
by (SIMP_TAC (bin_ss addrews prems) 1);
val bin_add_bcons_bcons = result();

val bin_add_rews = [bin_add_Plus, bin_add_Minus, bin_add_bcons_Plus,
		    bin_add_bcons_Minus, bin_add_bcons_bcons,
		    integ_of_bin_succ, integ_of_bin_pred];

val bin_add_ss = bin_ss addrews ([bool_subset_nat RS subsetD] @ bin_add_rews);

val prems = goal Bin.thy
    "v: bin ==> \
\    ALL w: bin. integ_of_bin(bin_add(v,w)) = \
\                   integ_of_bin(v) $+ integ_of_bin(w)";
by (bin_ind_tac "v" prems 1);
by (SIMP_TAC bin_add_ss 1);
by (SIMP_TAC bin_add_ss 1);
by (rtac ballI 1);
by (bin_ind_tac "wa" [] 1);
by (ASM_SIMP_TAC (bin_add_ss addrews [zadd_0_right]) 1);
by (ASM_SIMP_TAC bin_add_ss 1);
by (REPEAT (ares_tac (zadd_commute::typechecks) 1));
by (etac boolE 1);
by (ASM_SIMP_TAC (bin_add_ss addrews [zadd_assoc, zadd_swap_pairs]) 2);
by (REPEAT (ares_tac ([refl, zadd_kill, zadd_assoc_swap_kill]@typechecks) 2));
by (etac boolE 1);
by (ALLGOALS (ASM_SIMP_TAC (bin_add_ss addrews [zadd_assoc,zadd_swap_pairs])));
by (REPEAT (ares_tac ([refl, zadd_kill, zadd_assoc_swap_kill RS sym]@
		      typechecks) 1));
val integ_of_bin_add_lemma = result();

val integ_of_bin_add = integ_of_bin_add_lemma RS bspec;


(*** bin_add: binary multiplication ***)

val bin_mult_ss =
    bin_ss addrews (bin_recs bin_mult_def @ 
		       [integ_of_bin_minus, integ_of_bin_add]);


val major::prems = goal Bin.thy
    "[| v: bin; w: bin |] ==>	\
\    integ_of_bin(bin_mult(v,w)) = \
\    integ_of_bin(v) $* integ_of_bin(w)";
by (cut_facts_tac prems 1);
by (bin_ind_tac "v" [major] 1);
by (SIMP_TAC (bin_mult_ss addrews [zmult_0]) 1);
by (SIMP_TAC (bin_mult_ss addrews [zmult_1,zmult_zminus]) 1);
by (etac boolE 1);
by (ASM_SIMP_TAC (bin_mult_ss addrews [zadd_zmult_distrib]) 2);
by (ASM_SIMP_TAC 
    (bin_mult_ss addrews [zadd_zmult_distrib, zmult_1, zadd_assoc]) 1);
by (REPEAT (ares_tac ([zadd_commute, zadd_assoc_swap_kill RS sym]@
		      typechecks) 1));
val integ_of_bin_mult = result();

(**** Computations ****)

(** extra rules for bin_succ, bin_pred **)

val [bin_succ_Plus, bin_succ_Minus, _] = bin_recs bin_succ_def;
val [bin_pred_Plus, bin_pred_Minus, _] = bin_recs bin_pred_def;

goal Bin.thy "bin_succ(w$$1) = bin_succ(w) $$ 0";
by (SIMP_TAC carry_ss 1);
val bin_succ_bcons1 = result();

goal Bin.thy "bin_succ(w$$0) = w$$1";
by (SIMP_TAC carry_ss 1);
val bin_succ_bcons0 = result();

goal Bin.thy "bin_pred(w$$1) = w$$0";
by (SIMP_TAC carry_ss 1);
val bin_pred_bcons1 = result();

goal Bin.thy "bin_pred(w$$0) = bin_pred(w) $$ 1";
by (SIMP_TAC carry_ss 1);
val bin_pred_bcons0 = result();

(** extra rules for bin_minus **)

val [bin_minus_Plus, bin_minus_Minus, _] = bin_recs bin_minus_def;

goal Bin.thy "bin_minus(w$$1) = bin_pred(bin_minus(w) $$ 0)";
by (SIMP_TAC bin_minus_ss 1);
val bin_minus_bcons1 = result();

goal Bin.thy "bin_minus(w$$0) = bin_minus(w) $$ 0";
by (SIMP_TAC bin_minus_ss 1);
val bin_minus_bcons0 = result();

(** extra rules for bin_add **)

val prems = goal Bin.thy 
    "w: bin ==> bin_add(v$$1, w$$1) = bin_add(v, bin_succ(w)) $$ 0";
by (SIMP_TAC (bin_add_ss addrews prems) 1);
val bin_add_bcons_bcons11 = result();

val prems = goal Bin.thy 
    "w: bin ==> bin_add(v$$1, w$$0) = bin_add(v,w) $$ 1";
by (SIMP_TAC (bin_add_ss addrews prems) 1);
val bin_add_bcons_bcons10 = result();

val prems = goal Bin.thy 
    "[| w: bin;  y: bool |] ==> bin_add(v$$0, w$$y) = bin_add(v,w) $$ y";
by (SIMP_TAC (bin_add_ss addrews prems) 1);
val bin_add_bcons_bcons0 = result();

(** extra rules for bin_mult **)

val [bin_mult_Plus, bin_mult_Minus, _] = bin_recs bin_mult_def;

goal Bin.thy "bin_mult(v$$1, w) = bin_add(bin_mult(v,w)$$0, w)";
by (SIMP_TAC bin_mult_ss 1);
val bin_mult_bcons1 = result();

goal Bin.thy "bin_mult(v$$0, w) = bin_mult(v,w)$$0";
by (SIMP_TAC bin_mult_ss 1);
val bin_mult_bcons0 = result();


(*** The computation simpset ***)

val bin_comp_ss = carry_ss addrews
    [bin_add_Plus, bin_add_Minus, bin_add_bcons_Plus, bin_add_bcons_Minus, 
     bin_add_bcons_bcons0, bin_add_bcons_bcons10, bin_add_bcons_bcons11]
    setauto (type_auto_tac bin_typechecks0);

val bin_comp_ss = integ_ss 
    addcongs bin_congs
    addrews [bin_succ_Plus, bin_succ_Minus,
	     bin_succ_bcons1, bin_succ_bcons0,
	     bin_pred_Plus, bin_pred_Minus,
	     bin_pred_bcons1, bin_pred_bcons0,
	     bin_minus_Plus, bin_minus_Minus,
	     bin_minus_bcons1, bin_minus_bcons0,
	     bin_add_Plus, bin_add_Minus, bin_add_bcons_Plus, 
	     bin_add_bcons_Minus, bin_add_bcons_bcons0, 
	     bin_add_bcons_bcons10, bin_add_bcons_bcons11,
	     bin_mult_Plus, bin_mult_Minus,
	     bin_mult_bcons1, bin_mult_bcons0]
    setauto (type_auto_tac ([bool_1I, bool_0I] @ bin_typechecks0));

proof_timing := true;
(*All runtimes below are on a SPARCserver 10*)

(* 13+19 = 32 *)
goal Bin.thy "bin_add(Plus$$1$$1$$0$$1, Plus$$1$$0$$0$$1$$1) = 0";
by (SIMP_TAC bin_comp_ss 1);

(* 1234+5678 = 6912 *)
goal Bin.thy
    "bin_add(Plus$$1$$0$$0$$1$$1$$0$$1$$0$$0$$1$$0, \
\	     Plus$$1$$0$$1$$1$$0$$0$$0$$1$$0$$1$$1$$1$$0) = 0";
by (SIMP_TAC bin_comp_ss 1);
(*8.9 secs*)

(* 1359-2468 = ~1109 *)
goal Bin.thy
    "bin_add(Plus$$1$$0$$1$$0$$1$$0$$0$$1$$1$$1$$1, \
\	     Minus$$0$$1$$1$$0$$0$$1$$0$$1$$1$$1$$0$$0) = 0";
by (SIMP_TAC bin_comp_ss 1);
(*7.4 secs*)

(* 93746-46375 = 47371 *)
goal Bin.thy
    "bin_add(Plus$$1$$0$$1$$1$$0$$1$$1$$1$$0$$0$$0$$1$$1$$0$$0$$1$$0, \
\	     Minus$$0$$1$$0$$0$$1$$0$$1$$0$$1$$1$$0$$1$$1$$0$$0$$1) = 0";
by (SIMP_TAC bin_comp_ss 1);
(*13.7 secs*)

(* negation of 65745 *)
goal Bin.thy
    "bin_minus(Plus$$1$$0$$0$$0$$0$$0$$0$$0$$0$$1$$1$$0$$1$$0$$0$$0$$1) = 0";
by (SIMP_TAC bin_comp_ss 1);
(*2.8 secs*)

(* negation of ~54321 *)
goal Bin.thy
    "bin_minus(Minus$$0$$0$$1$$0$$1$$0$$1$$1$$1$$1$$0$$0$$1$$1$$1$$1) = 0";
by (SIMP_TAC bin_comp_ss 1);
(*3.3 secs*)

(* 13*19 = 247 *)
goal Bin.thy "bin_mult(Plus$$1$$1$$0$$1, Plus$$1$$0$$0$$1$$1) = 0";
by (SIMP_TAC bin_comp_ss 1);
(*4.4 secs*)

(* ~84 * 51 = ~4284 *)
goal Bin.thy "bin_mult(Minus$$0$$1$$0$$1$$1$$0$$0, Plus$$1$$1$$0$$0$$1$$1) =0";
by (SIMP_TAC bin_comp_ss 1);
(*9.2 secs*)

(***************** TOO SLOW TO INCLUDE IN TEST RUNS
    (* 255*255 = 65025;  the worst case for 8-bit operands *)
    goal Bin.thy
	"bin_mult(Plus$$1$$1$$1$$1$$1$$1$$1$$1, \
    \             Plus$$1$$1$$1$$1$$1$$1$$1$$1) = 0";
    by (SIMP_TAC bin_comp_ss 1);
    (*38.4 secs*)

    (* 1359 * ~2468 = ~3354012 *)
    goal Bin.thy
	"bin_mult(Plus$$1$$0$$1$$0$$1$$0$$0$$1$$1$$1$$1, \
    \	      Minus$$0$$1$$1$$0$$0$$1$$0$$1$$1$$1$$0$$0) = 0";
    by (SIMP_TAC bin_comp_ss 1);
    (*54.8 secs*)
****************)
