(*  Title: 	tctical
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1989  University of Cambridge

Tacticals
Derived rules and other operations on theorems and theories
*)

infix 1 THEN THEN' THEN_BEST_FIRST;
infix 0 ORELSE APPEND INTLEAVE ORELSE' APPEND' INTLEAVE';
infix 0 RS RSN RL RLN COMP;


signature TACTICAL =
  sig
  structure Thm : THM
  local open Thm  in
  datatype tactic = Tactic of thm -> thm Sequence.seq
  val all_tac: tactic
  val ALLGOALS: (int -> tactic) -> tactic   
  val APPEND: tactic * tactic -> tactic
  val APPEND': ('a -> tactic) * ('a -> tactic) -> 'a -> tactic
  val assume_ax: theory -> string -> thm
  val BEST_FIRST: (thm -> bool) * (thm -> int) -> tactic -> tactic
  val BREADTH_FIRST: (thm -> bool) -> tactic -> tactic
  val CHANGED: tactic -> tactic
  val COMP: thm * thm -> thm
  val compose: thm * int * thm -> thm list
  val COND: (thm -> bool) -> tactic -> tactic -> tactic   
  val cterm_instantiate: (Sign.cterm*Sign.cterm)list -> thm -> thm
  val DEPTH_FIRST: (thm -> bool) -> tactic -> tactic
  val DEPTH_SOLVE: tactic -> tactic
  val DEPTH_SOLVE_1: tactic -> tactic
  val DETERM: tactic -> tactic
  val eq_sg: Sign.sg * Sign.sg -> bool
  val eq_thm: thm * thm -> bool
  val eq_thm_sg: thm * thm -> bool
  val EVERY: tactic list -> tactic   
  val EVERY': ('a -> tactic) list -> 'a -> tactic
  val EVERY1: (int -> tactic) list -> tactic
  val FILTER: (thm -> bool) -> tactic -> tactic
  val FIRST: tactic list -> tactic   
  val FIRST': ('a -> tactic) list -> 'a -> tactic
  val FIRST1: (int -> tactic) list -> tactic
  val FIRSTGOAL: (int -> tactic) -> tactic
  val forall_intr_list: Sign.cterm list -> thm -> thm
  val forall_intr_frees: thm -> thm
  val forall_elim_list: Sign.cterm list -> thm -> thm
  val forall_elim_var: int -> thm -> thm
  val forall_elim_vars: int -> thm -> thm
  val goals_limit: int ref
  val has_fewer_prems: int -> thm -> bool   
  val IF_UNSOLVED: tactic -> tactic
  val implies_elim_list: thm -> thm list -> thm
  val implies_intr_list: Sign.cterm list -> thm -> thm
  val INTLEAVE: tactic * tactic -> tactic
  val INTLEAVE': ('a -> tactic) * ('a -> tactic) -> 'a -> tactic
  val METAHYPS: (thm list -> tactic) -> int -> tactic
  val no_tac: tactic
  val ORELSE: tactic * tactic -> tactic
  val ORELSE': ('a -> tactic) * ('a -> tactic) -> 'a -> tactic
  val pause_tac: tactic
  val print_goals: int -> thm -> unit
  val print_thm: thm -> unit
  val print_tac: tactic
  val prth: thm -> thm
  val prthq: thm Sequence.seq -> thm Sequence.seq
  val prths: thm list -> thm list
  val read_instantiate: (string*string)list -> thm -> thm
  val read_instantiate_sg: Sign.sg -> (string*string)list -> thm -> thm
  val REPEAT1: tactic -> tactic
  val REPEAT: tactic -> tactic
  val REPEAT_DETERM: tactic -> tactic
  val REPEAT_FIRST: (int -> tactic) -> tactic
  val REPEAT_SOME: (int -> tactic) -> tactic
  val RS: thm * thm -> thm
  val RSN: thm * (int * thm) -> thm
  val RL: thm list * thm list -> thm list
  val RLN: thm list * (int * thm list) -> thm list
  val SELECT_GOAL: tactic -> int -> tactic
  val show_hyps: bool ref
  val size_of_thm: thm -> int
  val SOMEGOAL: (int -> tactic) -> tactic   
  val standard: thm -> thm
  val STATE: (thm -> tactic) -> tactic
  val string_of_thm: thm -> string
  val strip_context: term -> (string * typ) list * term list * term
  val SUBGOAL: ((term*int) -> tactic) -> int -> tactic
  val tapply: tactic * thm -> thm Sequence.seq
  val THEN: tactic * tactic -> tactic
  val THEN': ('a -> tactic) * ('a -> tactic) -> 'a -> tactic
  val THEN_BEST_FIRST: tactic * ((thm->bool) * (thm->int) * tactic) -> tactic
  val traced_tac: (thm -> (thm * thm Sequence.seq) option) -> tactic
  val tracify: bool ref -> tactic -> thm -> thm Sequence.seq
  val trace_BEST_FIRST: bool ref
  val trace_DEPTH_FIRST: bool ref
  val trace_REPEAT: bool ref
  val TRY: tactic -> tactic
  val TRYALL: (int -> tactic) -> tactic   
  val types_sorts: thm -> (indexname-> typ option) * (indexname-> sort option)
  val zero_var_indexes: thm -> thm
  end
  end;


functor TacticalFun (structure Logic: LOGIC and Thm: THM) : TACTICAL = 
struct
structure Thm = Thm;
structure Sequence = Thm.Sequence;
structure Sign = Thm.Sign;
structure Type = Sign.Type;
structure Pretty = Sign.Syntax.Pretty
local open Thm
in

(**** More derived rules and operations on theorems ****)

(*** Find the type (sort) associated with a (T)Var or (T)Free in a term 
     Used for establishing default types (of variables) and sorts (of
     type variables) when reading another term.
     Index -1 indicates that a (T)Free rather than a (T)Var is wanted.
***)

fun types_sorts thm =
    let val {prop,hyps,...} = rep_thm thm;
	val big = list_comb(prop,hyps); (* bogus term! *)
	val vars = map dest_Var (Logic.vars big);
	val frees = map dest_Free (Logic.frees big);
	val tvars = add_term_tvars(big,[]);
	val tfrees = add_term_tfrees(big,[]);
	fun typ(a,i) = if i<0 then assoc(frees,a) else assoc(vars,(a,i));
	fun sort(a,i) = if i<0 then assoc(tfrees,a) else assoc(tvars,(a,i));
    in (typ,sort) end;

(** Standardization of rules **)

(*Generalization over a list of variables, IGNORING bad ones*)
fun forall_intr_list [] th = th
  | forall_intr_list (y::ys) th =
	let val gth = forall_intr_list ys th
	in  forall_intr y gth   handle THM _ =>  gth  end;

(*Generalization over all suitable Free variables*)
fun forall_intr_frees th =
    let val {prop,sign,...} = rep_thm th
    in  forall_intr_list
         (map (Sign.cterm_of sign) (sort Logic.atless (Logic.frees prop))) 
         th
    end;

(*Replace outermost quantified variable by Var of given index.
    Could clash with Vars already present.*)
fun forall_elim_var i th = 
    let val {prop,sign,...} = rep_thm th
    in case prop of
	  Const("all",_) $ Abs(a,T,_) =>
	      forall_elim (Sign.cterm_of sign (Var((a,i), T)))  th
	| _ => raise THM("forall_elim_var", i, [th])
    end;

(*Repeat forall_elim_var until all outer quantifiers are removed*)
fun forall_elim_vars i th = 
    forall_elim_vars i (forall_elim_var i th)
	handle THM _ => th;

(*Specialization over a list of cterms*)
fun forall_elim_list cts th = foldr (uncurry forall_elim) (rev cts, th);

(* maps [A1,...,An], B   to   [| A1;...;An |] ==> B  *)
fun implies_intr_list cAs th = foldr (uncurry implies_intr) (cAs,th);

(* maps [| A1;...;An |] ==> B and [A1,...,An]   to   B *)
fun implies_elim_list impth ths = foldl (uncurry implies_elim) (impth,ths);

(*Reset Var indexes to zero, renaming to preserve distinctness*)
fun zero_var_indexes th = 
    let val {prop,sign,...} = rep_thm th;
        val vars = Logic.vars prop
        val bs = foldl add_new_id ([], map (fn Var((a,_),_)=>a) vars)
	val inrs = add_term_tvars(prop,[]);
	val nms' = rev(foldl add_new_id ([], map (#1 o #1) inrs));
	val tye = map (fn ((v,rs),a) => (v, TVar((a,0),rs))) (inrs ~~ nms')
	val ctye = map (fn (v,T) => (v,Sign.ctyp_of sign T)) tye;
	fun varpairs([],[]) = []
	  | varpairs((var as Var(v,T)) :: vars, b::bs) =
		let val T' = Type.inst_typ_tvars(#tsig(Sign.rep_sg sign),tye) T
		in (Sign.cterm_of sign (Var(v,T')),
		    Sign.cterm_of sign (Var((b,0),T'))) :: varpairs(vars,bs)
		end
	  | varpairs _ = raise TERM("varpairs", []);
    in instantiate (ctye, varpairs(vars,rev bs)) th end;


(*Standard form of object-rule: no hypotheses, Frees, or outer quantifiers;
    all generality expressed by Vars having index 0.*)
fun standard th =
    let val {maxidx,...} = rep_thm th
    in  varifyT (zero_var_indexes (forall_elim_vars(maxidx+1) 
                         (forall_intr_frees(implies_intr_hyps th))))
    end;

(*Assume a new formula, read following the same conventions as axioms. 
  Generalizes over Free variables,
  creates the assumption, and then strips quantifiers.
  Example is [| ALL x:?A. ?P(x) |] ==> [| ?P(?a) |]
	     [ !(A,P,a)[| ALL x:A. P(x) |] ==> [| P(a) |] ]    *)
fun assume_ax thy sP =
    let val sign = sign_of thy;
	val prop = Logic.close_form (Sign.term_of (Sign.read_cterm sign
			 (sP, propT)))
    in forall_elim_vars 0 (assume (Sign.cterm_of sign prop))  end;

(*Resolution: exactly one resolvent must be produced.*) 
fun tha RSN (i,thb) =
  case Sequence.chop (2, biresolution false [(false,tha)] i thb) of
      ([th],_) => th
    | ([],_)   => raise THM("RSN: no unifiers", i, [tha,thb])
    |      _   => raise THM("RSN: multiple unifiers", i, [tha,thb]);

(*resolution: P==>Q, Q==>R gives P==>R. *)
fun tha RS thb = tha RSN (1,thb);

(*For joining lists of rules*)
fun thas RLN (i,thbs) = 
  let val resolve = biresolution false (map (pair false) thas) i
      fun resb thb = Sequence.list_of_s (resolve thb) handle THM _ => []
  in  flat (map resb thbs)  end;

fun thas RL thbs = thas RLN (1,thbs);

(*compose Q and [...,Qi,Q(i+1),...]==>R to [...,Q(i+1),...]==>R 
  with no lifting or renaming!  Q may contain ==> or meta-quants
  ALWAYS deletes premise i *)
fun compose(tha,i,thb) = 
    Sequence.list_of_s (bicompose false (false,tha,0) i thb);

(*compose Q and [Q1,Q2,...,Qk]==>R to [Q2,...,Qk]==>R getting unique result*)
fun tha COMP thb =
    case compose(tha,1,thb) of
        [th] => th  
      | _ =>   raise THM("COMP", 1, [tha,thb]);

(*Instantiate theorem th, reading instantiations under signature sg*)
fun read_instantiate_sg sg sinsts th =
    let val ts = types_sorts th;
        val instpair = Sign.read_insts sg ts ts sinsts
    in  instantiate instpair th  end;

(*Instantiate theorem th, reading instantiations under theory of th*)
fun read_instantiate sinsts th =
    read_instantiate_sg (#sign (rep_thm th)) sinsts th;


(*Left-to-right replacements: tpairs = [...,(vi,ti),...].
  Instantiates distinct Vars by terms, inferring type instantiations. *)
local
  fun add_types ((ct,cu), (sign,tye)) =
    let val {sign=signt, t=t, T= T, ...} = Sign.rep_cterm ct
        and {sign=signu, t=u, T= U, ...} = Sign.rep_cterm cu
        val sign' = Sign.merge(sign, Sign.merge(signt, signu))
	val tye' = Type.unify (#tsig(Sign.rep_sg sign')) ((T,U), tye)
	  handle Type.TUNIFY => raise TYPE("add_types", [T,U], [t,u])
    in  (sign', tye')  end;
in
fun cterm_instantiate ctpairs0 th = 
  let val (sign,tye) =
		foldr add_types (ctpairs0, (#sign(rep_thm th),[]))
      val tsig = #tsig(Sign.rep_sg sign);
      fun instT(ct,cu) = let val inst = Type.inst_term_tvars(tsig,tye)
			 in (Sign.cfun inst ct, Sign.cfun inst cu) end
      fun ctyp2 (ix,T) = (ix, Sign.ctyp_of sign T)
  in  instantiate (map ctyp2 tye, map instT ctpairs0) th  end
  handle TERM _ => 
           raise THM("cterm_instantiate: incompatible signatures",0,[th])
       | TYPE _ => raise THM("cterm_instantiate: types", 0, [th])
end;


(*** Printing of theorems ***)

(*If false, hypotheses are printed as dots*)
val show_hyps = ref true;

fun pretty_thm th =
let val {sign, hyps, prop,...} = rep_thm th
    val hsymbs = if null hyps then []
		 else if !show_hyps then
		      [Pretty.brk 2,
		       Pretty.lst("[","]") (map (Sign.pretty_term sign) hyps)]
		 else Pretty.str" [" :: map (fn _ => Pretty.str".") hyps @
		      [Pretty.str"]"];
in Pretty.blk(0, Sign.pretty_term sign prop :: hsymbs) end;

val string_of_thm = Pretty.string_of o pretty_thm;

(** Top-level commands for printing theorems **)
val print_thm = writeln o string_of_thm;

fun prth th = (print_thm th; th);

(*Print and return a sequence of theorems, separated by blank lines. *)
fun prthq thseq =
    (Sequence.prints (fn _ => print_thm) 100000 thseq;
     thseq);

(*Print and return a list of theorems, separated by blank lines. *)
fun prths ths = (print_list_ln print_thm ths; ths);


(** Print thm A1,...,An/B in "goal style" -- premises as numbered subgoals **)

fun prettyprints es = writeln(Pretty.string_of(Pretty.blk(0,es)));

fun print_goals maxgoals th : unit =
let val {sign, hyps, prop,...} = rep_thm th;
    fun printgoals (_, []) = ()
      | printgoals (n, A::As) =
	let val prettyn = Pretty.str(" " ^ string_of_int n ^ ". ");
	    val prettyA = Sign.pretty_term sign A
	in prettyprints[prettyn,prettyA]; 
           printgoals (n+1,As) 
        end;
    fun prettypair(t,u) =
        Pretty.blk(0, [Sign.pretty_term sign t, Pretty.str" =?=", Pretty.brk 1,
		       Sign.pretty_term sign u]);
    fun printff [] = ()
      | printff tpairs =
	 writeln("\nFlex-flex pairs:\n" ^
		 Pretty.string_of(Pretty.lst("","") (map prettypair tpairs)))
    val (tpairs,As,B) = Logic.strip_horn(prop);
    val ngoals = length As
in 
   writeln (Sign.string_of_term sign B);
   if ngoals=0 then writeln"No subgoals!"
   else if ngoals>maxgoals 
        then (printgoals (1, front(maxgoals,As));
	      writeln("A total of " ^ string_of_int ngoals ^ " subgoals..."))
        else printgoals (1, As);
   printff tpairs
end;


(**** Tactics ****)

(*A tactic maps a proof tree to a sequence of proof trees:
    if length of sequence = 0 then the tactic does not apply;
    if length > 1 then backtracking on the alternatives can occur.*)

datatype tactic = Tactic of thm -> thm Sequence.seq;

fun tapply(Tactic tf, state) = tf (state);

(*Makes a tactic from one that uses the components of the state.*)
fun STATE tacfun = Tactic (fn state => tapply(tacfun state, state));


(*** LCF-style tacticals ***)

(*the tactical THEN performs one tactic followed by another*)
fun (Tactic tf1)  THEN  (Tactic tf2) = 
  Tactic (fn state => Sequence.flats (Sequence.maps tf2 (tf1 state)));


(*The tactical ORELSE uses the first tactic that returns a nonempty sequence.
  Like in LCF, ORELSE commits to either tac1 or tac2 immediately.
  Does not backtrack to tac2 if tac1 was initially chosen. *)
fun (Tactic tf1)  ORELSE  (Tactic tf2) = 
  Tactic (fn state =>  
    case Sequence.pull(tf1 state) of
	None       => tf2 state
      | sequencecell => Sequence.seqof(fn()=> sequencecell));


(*The tactical APPEND combines the results of two tactics.
  Like ORELSE, but allows backtracking on both tac1 and tac2.
  The tactic tac2 is not applied until needed.*)
fun (Tactic tf1)  APPEND  (Tactic tf2) = 
  Tactic (fn state =>  Sequence.append(tf1 state,
                          Sequence.seqof(fn()=> Sequence.pull (tf2 state))));

(*Like APPEND, but interleaves results of tac1 and tac2.*)
fun (Tactic tf1)  INTLEAVE  (Tactic tf2) = 
  Tactic (fn state =>  Sequence.interleave(tf1 state,
                          Sequence.seqof(fn()=> Sequence.pull (tf2 state))));

(*Versions for combining tactic-valued functions, as in
     SOMEGOAL (resolve_tac rls THEN' assume_tac) *)
fun tac1 THEN' tac2 = fn x => tac1 x THEN tac2 x;
fun tac1 ORELSE' tac2 = fn x => tac1 x ORELSE tac2 x;
fun tac1 APPEND' tac2 = fn x => tac1 x APPEND tac2 x;
fun tac1 INTLEAVE' tac2 = fn x => tac1 x INTLEAVE tac2 x;

(*passes all proofs through unchanged;  identity of THEN*)
val all_tac = Tactic (fn state => Sequence.single state);

(*passes no proofs through;  identity of ORELSE and APPEND*)
val no_tac  = Tactic (fn state => Sequence.null);


(*Make a tactic deterministic by chopping the tail of the proof sequence*)
fun DETERM (Tactic tf) = Tactic (fn state => 
      case Sequence.pull (tf state) of
	      None => Sequence.null
            | Some(x,_) => Sequence.cons(x, Sequence.null));


(*Conditional tactical: testfun controls which tactic to use next.
  Beware: due to eager evaluation, both thentac and elsetac are evaluated.*)
fun COND testfun (Tactic thenf) (Tactic elsef) = Tactic (fn prf =>
    if testfun prf then  thenf prf   else  elsef prf);

(*Do the tactic or else do nothing*)
fun TRY tac = tac ORELSE all_tac;


(*** List-oriented tactics ***)

(* EVERY [tac1,...,tacn]   equals    tac1 THEN ... THEN tacn   *)
fun EVERY tacs = foldr (op THEN) (tacs, all_tac);

(* EVERY' [tf1,...,tfn] i  equals    tf1 i THEN ... THEN tfn i   *)
fun EVERY' tfs = foldr (op THEN') (tfs, K all_tac);

(*Apply every tactic to 1*)
fun EVERY1 tfs = EVERY' tfs 1;

(* FIRST [tac1,...,tacn]   equals    tac1 ORELSE ... ORELSE tacn   *)
fun FIRST tacs = foldr (op ORELSE) (tacs, no_tac);

(* FIRST' [tf1,...,tfn] i  equals    tf1 i ORELSE ... ORELSE tfn i   *)
fun FIRST' tfs = foldr (op ORELSE') (tfs, K no_tac);

(*Apply first tactic to 1*)
fun FIRST1 tfs = FIRST' tfs 1;


(*** Tracing tactics ***)

(*Max number of goals to print -- set by user*)
val goals_limit = ref 10;

(*Print the current proof state and pass it on.*)
val print_tac = Tactic (fn state => 
  (print_goals (!goals_limit) state;   Sequence.single state));

(*Pause until a line is typed -- if non-empty then fail. *)
val pause_tac = Tactic (fn state => 
  (prs"** Press RETURN to continue: ";
   if input(std_in,1) = "\n" then Sequence.single state
   else (prs"Goodbye\n";  Sequence.null)));

exception TRACE_EXIT of thm
and TRACE_QUIT;

(*Handle all tracing commands for current state and tactic *)
fun exec_trace_command flag (tf, state) = 
   case input_line(std_in) of
       "\n" => tf state
     | "f\n" => Sequence.null
     | "o\n" => (flag:=false; tf state)
     | "x\n" => (prs"Exiting now\n";  raise (TRACE_EXIT state))
     | "quit\n" => raise TRACE_QUIT
     | _     => (prs
"Type RETURN to continue or...\n\
\     f    - to fail here\n\
\     o    - to switch tracing off\n\
\     x    - to exit at this point\n\
\     quit - to abort this tracing run\n\
\** Well? "     ;  exec_trace_command flag (tf, state));


(*Extract from a tactic, a thm->thm seq function that handles tracing*)
fun tracify flag (Tactic tf) state =
  if !flag then (print_goals (!goals_limit) state;  
		 prs"** Press RETURN to continue: ";
		 exec_trace_command flag (tf,state))
  else tf state;

(*Create a tactic whose outcome is given by seqf, handling TRACE_EXIT*)
fun traced_tac seqf = Tactic (fn st =>
    Sequence.seqof (fn()=> seqf st
		           handle TRACE_EXIT st' => Some(st', Sequence.null)));


(*Tracing flags*)
val trace_REPEAT= ref false
and trace_DEPTH_FIRST = ref false
and trace_BEST_FIRST = ref false;

(*Deterministic REPEAT: only retains the first outcome; 
  uses less space than REPEAT; tail recursive*)
fun REPEAT_DETERM tac = 
  let val tf = tracify trace_REPEAT tac
      fun drep st =
        case Sequence.pull(tf st) of
  	    None       => Some(st, Sequence.null)
          | Some(st',_) => drep st'
  in  traced_tac drep  end;

(*General REPEAT: maintains a stack of alternatives; tail recursive*)
fun REPEAT tac = 
  let val tf = tracify trace_REPEAT tac
      fun rep qs st = 
	case Sequence.pull(tf st) of
  	    None       => Some(st, Sequence.seqof(fn()=> repq qs))
          | Some(st',q) => rep (q::qs) st'
      and repq [] = None
        | repq(q::qs) = case Sequence.pull q of
  	    None       => repq qs
          | Some(st,q) => rep (q::qs) st
  in  traced_tac (rep [])  end;

(*Repeat 1 or more times*)
fun REPEAT1 tac = tac THEN REPEAT tac;


(** Search tacticals **)

(*Seaarches "satp" reports proof tree as satisfied*)
fun DEPTH_FIRST satp tac = 
 let val tf = tracify trace_DEPTH_FIRST tac
     fun depth [] = None
       | depth(q::qs) =
	  case Sequence.pull q of
	      None         => depth qs
	    | Some(st,stq) => 
		if satp st then Some(st, Sequence.seqof(fn()=> depth(stq::qs)))
		else depth (tf st :: stq :: qs)
  in  traced_tac (fn st => depth([Sequence.single st]))  end;


(*Predicate: Does the rule have fewer than n premises?*)
fun has_fewer_prems n rule = (nprems_of rule < n);

(*Apply a tactic if subgoals remain, else do nothing.*)
val IF_UNSOLVED = COND (has_fewer_prems 1) all_tac;

(*Tactical to reduce the number of premises by 1.
  If no subgoals then it must fail! *)
fun DEPTH_SOLVE_1 tac = STATE
 (fn state => 
    (case nprems_of state of
	0 => no_tac
      | n => DEPTH_FIRST (has_fewer_prems n) tac));

(*Uses depth-first search to solve ALL subgoals*)
val DEPTH_SOLVE = DEPTH_FIRST (has_fewer_prems 1);

(** theorem equality test is exported and used by BEST_FIRST **)

(*equality of signatures means exact identity -- by ref equality*)
fun eq_sg (sg1,sg2) = (#stamps(Sign.rep_sg sg1) = #stamps(Sign.rep_sg sg2));

(*equality of theorems uses equality of signatures and 
  the a-convertible test for terms*)
fun eq_thm (th1,th2) = 
    let val {sign=sg1, hyps=hyps1, prop=prop1, ...} = rep_thm th1
	and {sign=sg2, hyps=hyps2, prop=prop2, ...} = rep_thm th2
    in  eq_sg (sg1,sg2) andalso 
        aconvs(hyps1,hyps2) andalso 
        prop1 aconv prop2  
    end;

(*Do the two theorems have the same signature?*)
fun eq_thm_sg (th1,th2) = eq_sg(#sign(rep_thm th1), #sign(rep_thm th2));

(*Useful "distance" function for BEST_FIRST*)
val size_of_thm = size_of_term o #prop o rep_thm;


(*** Best-first search ***)

(*Insertion into priority queue of states *)
fun insert (nth: int*thm, []) = [nth]
  | insert ((m,th), (n,th')::nths) = 
      if  n<m then (n,th') :: insert ((m,th), nths)
      else if  n=m andalso eq_thm(th,th')
              then (n,th')::nths
              else (m,th)::(n,th')::nths;

(*For creating output sequence*)
fun some_of_list []     = None
  | some_of_list (x::l) = Some (x, Sequence.seqof (fn () => some_of_list l));


(* Best-first search for a state that satisfies satp (incl initial state)
  Function sizef estimates size of problem remaining (smaller means better).
  tactic tf0 sets up the initial priority queue, which is searched by tac. *)
fun (Tactic tf0) THEN_BEST_FIRST (satp, sizef, tac) = 
  let val tf = tracify trace_BEST_FIRST tac
      fun pairsize th = (sizef th, th);
      fun bfs (news,nprfs) =
	   (case  partition satp news  of
		([],nonsats) => next(foldr insert
					(map pairsize nonsats, nprfs)) 
	      | (sats,_)  => some_of_list sats)
      and next [] = None
        | next ((n,prf)::nprfs) =
	    (if !trace_BEST_FIRST 
	       then writeln("state size = " ^ string_of_int n ^ 
		         "  queue length =" ^ string_of_int (length nprfs))  
               else ();
	     bfs (Sequence.list_of_s (tf prf), nprfs))
      fun tf st = bfs (Sequence.list_of_s (tf0 st),  [])
  in traced_tac tf end;

(*Ordinary best-first search, with no initial tactic*)
fun BEST_FIRST (satp,sizef) tac = all_tac THEN_BEST_FIRST (satp,sizef,tac);

(*Breadth-first search to satisfy satpred (including initial state) 
  SLOW -- SHOULD NOT USE APPEND!*)
fun BREADTH_FIRST satpred (Tactic tf) = 
  let val tacf = Sequence.list_of_s o tf;
      fun bfs prfs =
	 (case  partition satpred prfs  of
	      ([],[]) => []
	    | ([],nonsats) => 
		  (prs("breadth=" ^ string_of_int(length nonsats) ^ "\n");
		   bfs (flat (map tacf nonsats)))
	    | (sats,_)  => sats)
  in Tactic (fn state => Sequence.s_of_list (bfs [state])) end;


(** Filtering tacticals **)

(*Returns all states satisfying the predicate*)
fun FILTER pred (Tactic tf) = Tactic
      (fn state => Sequence.filters pred (tf state));

(*Returns all changed states*)
fun CHANGED (Tactic tf)  = 
  Tactic (fn state => 
    let fun diff st = not (eq_thm(state,st))
    in  Sequence.filters diff (tf state)
    end );


(*** Tacticals based on subgoal numbering ***)

(*For n subgoals, performs tf(n) THEN ... THEN tf(1) 
  Essential to work backwards since tf(i) may add/delete subgoals at i. *)
fun ALLGOALS tf = 
  let fun tac 0 = all_tac
	| tac n = tf(n) THEN tac(n-1)
  in  Tactic(fn state => tapply(tac(nprems_of state), state))  end;

(*For n subgoals, performs tf(n) ORELSE ... ORELSE tf(1)  *)
fun SOMEGOAL tf = 
  let fun tac 0 = no_tac
	| tac n = tf(n) ORELSE tac(n-1)
  in  Tactic(fn state => tapply(tac(nprems_of state), state))  end;

(*For n subgoals, performs tf(1) ORELSE ... ORELSE tf(n).
  More appropriate than SOMEGOAL in some cases.*)
fun FIRSTGOAL tf = 
  let fun tac (i,n) = if i>n then no_tac else  tf(i) ORELSE tac (i+1,n)
  in  Tactic(fn state => tapply(tac(1, nprems_of state), state))  end;

(*Repeatedly solve some using tf. *)
fun REPEAT_SOME tf = REPEAT1 (SOMEGOAL (REPEAT1 o tf));

(*Repeatedly solve the first possible subgoal using tf. *)
fun REPEAT_FIRST tf = REPEAT1 (FIRSTGOAL (REPEAT1 o tf));

(*For n subgoals, tries to apply tf to n,...1  *)
fun TRYALL tf = ALLGOALS (TRY o tf);


(*Make a tactic for subgoal i, if there is one.  *)
fun SUBGOAL goalfun i = Tactic(fn state =>
  case nth_tail(i-1, prems_of state) of
      [] => Sequence.null
    | prem::_ => tapply(goalfun (prem,i), state));

(*Tactical for restricting the effect of a tactic to subgoal i.
  Works by making a new state from subgoal i, applying tf to it, and
  composing the resulting metathm with the original state.
  The "main goal" of the new state will not be atomic, some tactics may fail!
  DOES NOT work if tactic affects the main goal other than by instantiation.*)

(* (!!x. ?V) ==> ?V ;  used by protect_subgoal.*)
val dummy_quant_rl = 
  standard (forall_elim_var 0 (assume 
                  (Sign.read_cterm Sign.pure ("!!x. PROP V",propT))));

(* Prevent the subgoal's assumptions from becoming additional subgoals in the
   new proof state by enclosing them by a universal quantification *)
fun protect_subgoal state i =
  case Sequence.chop (1, bicompose false (false,dummy_quant_rl,1) i state)
  of
      ([state'],_) => state'
    | _ => error"SELECT_GOAL -- impossible error???";

(*Does the work of SELECT_GOAL. *)
fun select (Tactic tf) state i =
  let val prem::_ = nth_tail(i-1, prems_of state)
      val st0 = trivial (Sign.cterm_of (#sign(rep_thm state)) prem);
      fun next st = bicompose false (false, st, nprems_of st) i state
  in  Sequence.flats (Sequence.maps next (tf st0))
  end;

fun SELECT_GOAL tac i = Tactic (fn state =>
  case nth_tail(i-1, prems_of state) of
      [] => Sequence.null
    | (Const("==>",_) $ _ $ _) :: _ => select tac (protect_subgoal state i) i
    | prem :: _ => select tac state i );



(*Strips assumptions in goal yielding  ( [x1,...,xm], [H1,...,Hn], B )
    H1,...,Hn are the hypotheses;  x1...xm are variants of the parameters. 
  Main difference from strip_assums concerns parameters: 
    it replaces the bound variables by free variables.  *)
fun strip_context_aux (params, Hs, Const("==>", _) $ H $ B) = 
	strip_context_aux (params, H::Hs, B)
  | strip_context_aux (params, Hs, Const("all",_)$Abs(a,T,t)) =
        let val (b,u) = variant_abs(a,T,t)
	in  strip_context_aux ((b,T)::params, Hs, u)  end
  | strip_context_aux (params, Hs, B) = (rev params, rev Hs, B);

fun strip_context A = strip_context_aux ([],[],A);


(**** METAHYPS -- tactical for using hypotheses as meta-level assumptions
       METAHYPS (fn prems => tac (prems)) i

converts subgoal i, of the form !!x1...xm. [| A1;...;An] ==> A into a new
proof state A==>A, supplying A1,...,An as meta-level assumptions (in
"prems").  The parameters x1,...,xm become free variables.  If the
resulting proof state is [| B1;...;Bk] ==> C (possibly assuming A1,...,An)
then it is lifted back into the original context, yielding k subgoals.

DOES NOT WORK if the context or B1,...,Bk contain unknowns.  New unknowns in 
[| B1;...;Bk] ==> C should be (BUT AREN'T) lifted over x1,...,xm.
****)

local open Logic 
in

fun metahyps_aux_tac tacf (prem,i) = Tactic (fn state =>
  let val cterm = Sign.cterm_of (#sign(rep_thm state))
      val (params,hyps,concl) = strip_context prem
      val cparams = map (cterm o Free) params
      and chyps = map cterm hyps
      val hypths = map assume chyps
      (*function to embed B in the original context of params and hyps*)
      fun embed B = list_all_free (params, list_implies (hyps, B))
      (*function to strip the context using elimination rules*)
      fun elim Bhyp = implies_elim_list (forall_elim_list cparams Bhyp) hypths
      (*A form of lifting that discharges assumptions.*)
      fun relift st = 
	let val emBs = map (cterm o embed) (prems_of st)
	    val Cth =  implies_elim_list st (map (elim o assume) emBs)
	in  (*discharge assumptions from state in same order*)
	    implies_intr_list emBs
	      (forall_intr_list cparams (implies_intr_list chyps Cth))
	end
      val subprems = map (forall_elim_vars 0) hypths
      and st0 = trivial (cterm concl)
      (*function to replace the current subgoal*)
      fun next st = bicompose false (false, relift st, nprems_of st)
	            i state
  in  Sequence.flats (Sequence.maps next (tapply(tacf subprems, st0)))
  end);

end;


fun METAHYPS tacf = SUBGOAL (metahyps_aux_tac tacf);

end;
end;
