(*  Title: 	stringtree
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

Data structure for indexing of items by string lists
*)


signature STRINGTREE = 
  sig
  type 'a tree
  exception DELETE and INSERT
  val below: 'a tree -> 'a list
  val compat: 'a tree * string list -> 'a list
  val delete: (string list * 'a) * 'b tree * ('a*'b -> bool) -> 'b tree
  val insert: (string list * 'a) * 'a tree * ('a*'a -> bool) -> 'a tree
  val lookup: 'a tree * string list -> 'a list
  val empty: 'a tree
  end;


functor StringtreeFun () : STRINGTREE = 
struct

(*Trees indexed by string lists: each arc is labelled by a string
  Each node contains a list of items, and arcs to children.
  Keys in the association list (alist) are stored in ascending order.
  Empty string addresses entire tree 
  Vital to return items the proper order:
    Items stored deeper are more specific and must preceed items stored
    above them;  order must be preserved in items stored at same level.
*)
datatype 'a tree = Stree of 'a list * (string * 'a tree) list;

val empty = Stree([],[]);

(*duplicate item in the tree -- "insert"*)
exception INSERT;

(*Adds item x to the list at the node addressed by the keys;
  creates node if not already present.*)
fun insert (([],        x), Stree(xs,alist), eq) = 
      if gen_mem eq (x,xs) then  raise INSERT
      else Stree(x::xs, alist)
  | insert ((key::keys, x), Stree(xs,alist), eq) =
      let fun newpair tr = (key, insert((keys,x), tr, eq)) 
	  fun inslist [] = [ newpair empty ]
	    | inslist((keyi: string, tri)::alist) =
		if key=keyi then newpair tri :: alist
		else if key<keyi then (*absent, insert in alist*)
		    newpair empty :: (keyi,tri) :: alist
		else (keyi,tri) :: inslist alist
      in  Stree(xs, inslist alist)  end;

(*missing item in the tree -- "delete"*)
exception DELETE;

(*add new (key,tr) pair to the alist provided tr is nonempty*)
fun conspair((key, Stree([],[])), alist) = alist
  | conspair((key, tr), alist)           = (key, tr) :: alist;

(*Deletes item x from the list at the node addressed by the keys.
  Use "eq" for the equality test.  Raises DELETE if absent. 
  Collapses the tree if possible.*)
fun delete ((keys0: string list, x), Stree(xs,alist), eq) = 
 let fun del ys = if gen_mem eq (x,ys) then gen_rem eq (ys,x)
                  else raise DELETE
 in case keys0 of
      [] => Stree(del xs, alist)
    | (key::keys) =>
	let fun newpair tr = (key, delete((keys,x), tr, eq)) 
	    fun dellist [] = raise DELETE
	      | dellist((keyi: string, tri)::alist) =
		  if key=keyi then 
		      conspair(newpair tri, alist)
		  else if key<keyi then (*absent*)
		      raise DELETE
		  else (keyi,tri) :: dellist alist
	in  Stree(xs, dellist alist)  end
 end;

(*Return the list of items at the given node, [] if no such node*)
fun lookup (Stree(xs,alist), []) = xs
  | lookup (Stree(xs,alist), key::keys) =
       (case assoc(alist,key) of 
	   None =>  []
	 | Some tr' => lookup(tr',keys));

(*Return the list of all items in the tree*)
fun below (Stree(xs,alist)) =
      let fun bel [] = []
	    | bel ((_,tr)::alist) = below tr  @  bel alist
      in  bel alist @ xs  end;

(*Return all items with compatible addresses:
  those where one address is a prefix of the other*)
fun compat (tr  ,  []) = below tr
  | compat (Stree(xs,alist), key::keys) = 
       (case assoc(alist,key) of 
	   None =>  xs
	 | Some tr' => compat(tr',keys) @ xs);

end;
