(*  Title: 	logic
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   Cambridge University 1992

Supporting code for defining the abstract type "thm"
*)

infix occs;

signature LOGIC = 
  sig
  structure Type: TYPE
  exception TMATCH;      (*failure of pattern-matching*)
  val add_term_frees: term * term list -> term list
  val add_term_vars: term * term list -> term list
  val assum_pairs: term -> (term*term)list
  val atless: term * term -> bool
  val auto_rename: bool ref   
  val close_form: term -> term   
  val count_prems: term * int -> int
  val dest_equals: term -> term * term
  val dest_implies: term -> term * term
  val eta_contract: term -> term
  val flatten_params: int -> term -> term
  val frees: term -> term list
  val incr_indexes: typ list * int -> term -> term
  val insert_aterm: term * term list -> term list
  val lift_fns: term * int -> (term -> term) * (term -> term)
  val list_flexpairs: (term*term)list * term -> term
  val list_implies: term list * term -> term
  val list_rename_params: string list * term -> term
  val matches: Type.type_sig -> term*term -> bool
  val mk_equals: term * term -> term
  val mk_implies: term * term -> term
  val occs: term * term -> bool
  val rule_of: (term*term)list * term list * term -> term
  val set_rename_prefix: string -> unit   
  val skip_flexpairs: term -> term
  val strip_assums_concl: term -> term
  val strip_assums_hyp: term -> term list
  val strip_flexpairs: term -> (term*term)list * term
  val strip_horn: term -> (term*term)list * term list * term
  val strip_imp_concl: term -> term
  val strip_imp_prems: term -> term list
  val strip_params: term -> (string * typ) list
  val strip_prems: int * term list * term -> term list * term
  val rewritec_bottom: (term*term) list -> {tsig: Type.type_sig, nextv: int} -> term -> term
  val rewritec_top: (term*term) list -> {tsig: Type.type_sig, nextv: int} -> term -> term
  val term_match: Type.type_sig -> term * term -> (indexname*typ)list * (term*term)list
  val varify: term -> term  
  val vars: term -> term list
  end;

functor LogicFun (Unify: UNIFY) : LOGIC  = 
struct
structure Type = Unify.Sign.Type;

(*** Abstract syntax operations on the meta-connectives ***)

(** equality **)

(*Make an equality.  DOES NOT CHECK TYPE OF u1 *)
fun mk_equals(t,u) = equals(type_of t) $ t $ u;

fun dest_equals (Const("==",_) $ t $ u)  =  (t,u)
  | dest_equals t = raise TERM("dest_equals", [t]);

(** implies **)

fun mk_implies(A,B) = implies $ A $ B;

fun dest_implies (Const("==>",_) $ A $ B)  =  (A,B)
  | dest_implies A = raise TERM("dest_implies", [A]);

(** nested implications **)

(* [A1,...,An], B  goes to  A1==>...An==>B  *)
fun list_implies ([], B) = B : term
  | list_implies (A::AS, B) = implies $ A $ list_implies(AS,B);

(* A1==>...An==>B  goes to  [A1,...,An], where B is not an implication *)
fun strip_imp_prems (Const("==>", _) $ A $ B) = A :: strip_imp_prems B
  | strip_imp_prems _ = [];

(* A1==>...An==>B  goes to B, where B is not an implication *)
fun strip_imp_concl (Const("==>", _) $ A $ B) = strip_imp_concl B
  | strip_imp_concl A = A : term;

(*Strip and return premises: (i, [], A1==>...Ai==>B)
    goes to   ([Ai, A(i-1),...,A1] , B) 	(REVERSED) 
  if  i<0 or else i too big then raises  TERM*)
fun strip_prems (0, As, B) = (As, B) 
  | strip_prems (i, As, Const("==>", _) $ A $ B) = 
	strip_prems (i-1, A::As, B)
  | strip_prems (_, As, A) = raise TERM("strip_prems", A::As);

(*Count premises -- quicker than (length ostrip_prems) *)
fun count_prems (Const("==>", _) $ A $ B, n) = count_prems (B,n+1)
  | count_prems (_,n) = n;

(** flex-flex constraints **)

(*make flexflex antecedents: ( [(a1,b1),...,(an,bn)] , C )
    goes to (a1=?=b1) ==>...(an=?=bn)==>C *)
fun flexpair T = Const("=?=", T-->T-->propT);
fun mk_flexpair(t,u) = flexpair(type_of t) $ t $ u;

fun list_flexpairs ([], A) = A
  | list_flexpairs ((t,u)::pairs, A) =
	implies $ (mk_flexpair(t,u)) $ list_flexpairs(pairs,A);

(*Make the object-rule tpairs==>As==>B   *)
fun rule_of (tpairs, As, B) = list_flexpairs(tpairs, list_implies(As, B));

(*Remove and return flexflex pairs: 
    (a1=?=b1)==>...(an=?=bn)==>C  to  ( [(a1,b1),...,(an,bn)] , C )	
  [Tail recursive in order to return a pair of results] *)
fun strip_flex_aux (pairs, Const("==>", _) $ (Const("=?=",_)$t$u) $ C) =
        strip_flex_aux ((t,u)::pairs, C)
  | strip_flex_aux (pairs,C) = (rev pairs, C);

fun strip_flexpairs A = strip_flex_aux([], A);

(*Discard flexflex pairs*)
fun skip_flexpairs (Const("==>", _) $ (Const("=?=",_)$_$_) $ C) =
	skip_flexpairs C
  | skip_flexpairs C = C;

(*strip a proof state (Horn clause): 
   (a1==b1)==>...(am==bm)==>B1==>...Bn==>C
    goes to   ( [(a1,b1),...,(am,bm)] , [B1,...,Bn] , C)    *)
fun strip_horn A =
  let val (tpairs,horn) = strip_flexpairs A 
  in  (tpairs, strip_imp_prems horn, strip_imp_concl horn)   end;


(*** Low-level term operations ***)

(*Does t occur in u?  Or is alpha-convertible to u?
  The term t must contain no loose bound variables*)
fun t occs u = (t aconv u) orelse 
      (case u of
          Abs(_,_,body) => t occs body
	| f$t' => t occs f  orelse  t occs t'
	| _ => false);

(*a partial ordering (not reflexive) for atomic terms*)
fun atless (Const (a,_), Const (b,_))  =  a<b
  | atless (Free (a,_), Free (b,_)) =  a<b
  | atless (Var(v,_), Var(w,_))  =  xless(v,w)
  | atless (Bound i, Bound j)  =   i<j
  | atless _  =  false;

(*insert atomic term into partially sorted list, suppressing duplicates (?)*)
fun insert_aterm (t,us) =
  let fun inserta [] = [t]
        | inserta (us as u::us') = 
	      if atless(t,u) then t::us
	      else if t=u then us (*duplicate*)
	      else u :: inserta(us')
  in  inserta us  end;

(*Accumulates the Vars in the term, suppressing duplicates*)
fun add_term_vars (t, vars: term list) = case t of
    Var   _ => insert_aterm(t,vars)
  | Abs (_,_,body) => add_term_vars(body,vars)
  | f$t =>  add_term_vars (f, add_term_vars(t, vars))
  | _ => vars;

fun vars t = add_term_vars(t,[]);

(*Accumulates the Frees in the term, suppressing duplicates*)
fun add_term_frees (t, frees: term list) = case t of
    Free   _ => insert_aterm(t,frees)
  | Abs (_,_,body) => add_term_frees(body,frees)
  | f$t =>  add_term_frees (f, add_term_frees(t, frees))
  | _ => frees;

fun frees t = add_term_frees(t,[]);

(*Close up a formula over all free variables by quantification*)
fun close_form A =
    list_all_free (map dest_Free (sort atless (frees A)),   
		   A);


(*** Specialized operations for resolution... ***)

(*For all variables in the term, increment indexnames and lift over the Us
    result is ?Gidx(B.(lev+n-1),...,B.lev) where lev is abstraction level *)
fun incr_indexes (Us: typ list, inc:int) t = 
  let fun incr (Var ((a,i), T), lev) = 
		Unify.combound (Var((a, i+inc), Us---> incr_tvar inc T),
				lev, length Us)
	| incr (Abs (a,T,body), lev) =
		Abs (a, incr_tvar inc T, incr(body,lev+1))
	| incr (Const(a,T),_) = Const(a, incr_tvar inc T)
	| incr (Free(a,T),_) = Free(a, incr_tvar inc T)
	| incr (f$t, lev) = incr(f,lev) $ incr(t,lev)
	| incr (t,lev) = t
  in  incr(t,0)  end;

(*Make lifting functions from subgoal and increment.
    lift_abs operates on tpairs (unification constraints)
    lift_all operates on propositions     *)
fun lift_fns (B,inc) =
  let fun lift_abs (Us, Const("==>", _) $ _ $ B) u = lift_abs (Us,B) u
	| lift_abs (Us, Const("all",_)$Abs(a,T,t)) u =
	      Abs(a, T, lift_abs (T::Us, t) u)
	| lift_abs (Us, _) u = incr_indexes(rev Us, inc) u
      fun lift_all (Us, Const("==>", _) $ A $ B) u =
	      implies $ A $ lift_all (Us,B) u
	| lift_all (Us, Const("all",_)$Abs(a,T,t)) u = 
	      all T $ Abs(a, T, lift_all (T::Us,t) u)
	| lift_all (Us, _) u = incr_indexes(rev Us, inc) u;
  in  (lift_abs([],B), lift_all([],B))  end;

(*Strips assumptions in goal, yielding list of hypotheses.   *)
fun strip_assums_hyp (Const("==>", _) $ H $ B) = H :: strip_assums_hyp B
  | strip_assums_hyp (Const("all",_)$Abs(a,T,t)) = strip_assums_hyp t
  | strip_assums_hyp B = [];

(*Strips assumptions in goal, yielding conclusion.   *)
fun strip_assums_concl (Const("==>", _) $ H $ B) = strip_assums_concl B
  | strip_assums_concl (Const("all",_)$Abs(a,T,t)) = strip_assums_concl t
  | strip_assums_concl B = B;

(*Make a list of all the parameters in a subgoal, even if nested*)
fun strip_params (Const("==>", _) $ H $ B) = strip_params B
  | strip_params (Const("all",_)$Abs(a,T,t)) = (a,T) :: strip_params t
  | strip_params B = [];

(*Removes the parameters from a subgoal and renumber bvars in hypotheses,
    where j is the total number of parameters (precomputed) 
  If n>0 then deletes assumption n. *)
fun remove_params j n A = 
    if j=0 andalso n<=0 then A  (*nothing left to do...*)
    else case A of
        Const("==>", _) $ H $ B => 
	  if n=1 then                           (remove_params j (n-1) B)
	  else implies $ (incr_boundvars j H) $ (remove_params j (n-1) B)
      | Const("all",_)$Abs(a,T,t) => remove_params (j-1) n t
      | _ => if n>0 then raise TERM("remove_params", [A])
             else A;

(** Auto-renaming of parameters in subgoals **)

val auto_rename = ref false
and rename_prefix = ref "ka";

(*rename_prefix is not exported; it is set by this function.*)
fun set_rename_prefix a =
    if a<>"" andalso forall is_letter (explode a)
    then  (rename_prefix := a;  auto_rename := true)
    else  error"rename prefix must be nonempty and consist of letters";

(*Makes parameters in a goal have distinctive names (not guaranteed unique!)
  A name clash could cause the printer to rename bound vars;
    then res_inst_tac would not work properly.*)
fun rename_vars (a, []) = []
  | rename_vars (a, (_,T)::vars) =
        (a,T) :: rename_vars (bump_string a, vars);

(*Move all parameters to the front of the subgoal, renaming them apart;
  if n>0 then deletes assumption n. *)
fun flatten_params n A =
    let val params = strip_params A;
	val vars = if !auto_rename 
		   then rename_vars (!rename_prefix, params)
		   else variantlist(map #1 params,[]) ~~ map #2 params
    in  list_all (vars, remove_params (length vars) n A)
    end;

(*Makes parameters in a goal have the names supplied by the list cs.*)
fun list_rename_params (cs, Const("==>", _) $ A $ B) =
      implies $ A $ list_rename_params (cs, B)
  | list_rename_params (c::cs, Const("all",_)$Abs(_,T,t)) = 
      all T $ Abs(c, T, list_rename_params (cs, t))
  | list_rename_params (cs, B) = B;

(*Strips assumptions in goal yielding  ( [Hn,...,H1], [xm,...,x1], B )
  where H1,...,Hn are the hypotheses and x1...xm are the parameters.   *)
fun strip_assums_aux (Hs, params, Const("==>", _) $ H $ B) = 
	strip_assums_aux (H::Hs, params, B)
  | strip_assums_aux (Hs, params, Const("all",_)$Abs(a,T,t)) =
	strip_assums_aux (Hs, (a,T)::params, t)
  | strip_assums_aux (Hs, params, B) = (Hs, params, B);

fun strip_assums A = strip_assums_aux ([],[],A);


(*Produces disagreement pairs, one for each assumption proof, in order.
  A is the first premise of the lifted rule, and thus has the form
    H1 ==> ... Hk ==> B   and the pairs are (H1,B),...,(Hk,B) *)
fun assum_pairs A =
  let val (Hs, params, B) = strip_assums A
      val D = Unify.rlist_abs(params, B)
      fun pairrev ([],pairs) = pairs  
        | pairrev (H::Hs,pairs) = 
	    pairrev(Hs, (Unify.rlist_abs(params,H), D) :: pairs)
  in  pairrev (Hs,[])   (*WAS:  map pair (rev Hs)  *)
  end;


(*Converts Frees to Vars and TFrees to TVars so that axioms can be written
  without (?) everywhere*)
fun varify (Const(a,T)) = Const(a, Type.varifyT T)
  | varify (Free(a,T)) = Var((a,0), Type.varifyT T)
  | varify (Var(ixn,T)) = Var(ixn, Type.varifyT T)
  | varify (Abs (a,T,body)) = Abs (a, Type.varifyT T, varify body)
  | varify (f$t) = varify f $ varify t
  | varify t = t;


(*Perform eta-contractions upon a term*)
fun eta_contract (Abs(a,T,body)) = 
      (case eta_contract body  of
        body' as (f $ Bound i) => 
	  if i=0 andalso not (0 mem loose_bnos f) then incr_boundvars ~1 f 
	  else Abs(a,T,body')
      | body' => Abs(a,T,body'))
  | eta_contract(f$t) = eta_contract f $ eta_contract t
  | eta_contract t = t;


(*** Meta-level rewriting 
     uses conversions, omitting proofs for efficiency.  See
	L C Paulson, A higher-order implementation of rewriting,
	Science of Computer Programming 3 (1983), pages 119-149. ***)

infix thenc;

(*A "crec" contains data needed during conversion: type signature 
  and next var index (for rewriting lambda abstractions) *)
type convrec = {tsig: Type.type_sig, nextv: int};

(*A conversion is applied to a "crec" and a term.*)
type conv = convrec -> term -> term;

exception TMATCH;      (*failure of pattern-matching*)

(*First-order matching;  term_match tsig (pattern, object)
    returns a (tyvar,typ)list and (var,term)list.
  The pattern and object may have variables in common.
  Instantiation does not affect the object, so matching ?a with ?a+1 works.
  A Const does not match a Free of the same name! 
  Does not notice eta-equality, thus f does not match %(x)f(x)  *)
fun term_match tsig (pat,obj) =
let fun typ_match args = (Type.typ_match tsig args)
			 handle Type.TYPE_MATCH => raise TMATCH;
    fun tmatch (tyinsts,insts) = fn
	(v as Var(_,T), t)  =>
	  if null (loose_bnos t)
	  then case tassoc(insts,v) of
		  None => (typ_match (tyinsts, (T,fastype_of([],t))), 
			   (v,t)::insts)
		| Some u => if t aconv u then (tyinsts,insts) else raise TMATCH
	  else raise TMATCH
      | (Free (a,T), Free (b,U)) =>
	  if  a=b  then (typ_match (tyinsts,(T,U)), insts)  else raise TMATCH
      | (Const (a,T), Const (b,U))  =>
	  if  a=b  then (typ_match (tyinsts,(T,U)), insts)  else raise TMATCH
      | (Bound i, Bound j)  =>
          if  i=j  then  (tyinsts,insts)  else raise TMATCH
      | (Abs(_,T,t), Abs(_,U,u))  =>
	  tmatch (typ_match (tyinsts,(T,U)),insts) (t,u)
      | (f$t, g$u) => tmatch (tmatch (tyinsts,insts) (f,g)) (t, u)
      | _ => raise TMATCH
    val (tyinsts,insts) = tmatch([],[]) (eta_contract pat, eta_contract obj)
    val insttype = Type.inst_term_tvars (tsig,tyinsts)
in (tyinsts, map (fn(v,t) => (insttype v, t)) insts) end;

(*Predicate: does the pattern match the object?*)
fun matches tsig (pat,obj) = (term_match tsig (pat,obj); true)
    handle TMATCH => false;

(*Conversion to apply the rewrites (a list of pairs) to a term*)
fun rewritec pairs {tsig,nextv} =
  let fun rew (lhs,rhs) t =
	let val (vTs,insts) = term_match tsig (lhs, t)
	in  subst_atomic insts (Type.inst_term_tvars (tsig,vTs) rhs)  end
      fun rewl [] t = raise TMATCH
	| rewl ((lhs,rhs)::pairs) t =
	    rew (lhs,rhs) t handle TMATCH => rewl pairs t;
  in rewl pairs end;

(*Apply the conversion cv to all top-level subterms of the term.*)
fun subc cv {tsig,nextv} (Abs (a,T,body)) = 
      let val v = Var((".subc.",nextv),T)
          val body' = cv {tsig=tsig, nextv=nextv+1}
			 (subst_bounds([v],body))
      in  Abs (a, T, abstract_over (v, body'))  end
  | subc cv crec (t$u) = cv crec t $ cv crec u
  | subc cv crec t = t;

(* apply two conversions in succession, fail if either does*)
fun (cv1 thenc cv2) crec t = cv2 crec (cv1 crec t);

(*Apply cv or do nothing -- we don't need orelsec or allc*)
fun tryc cv crec t = cv crec t  handle TMATCH => t;

(*Apply cv repeatedly -- the abstraction over crec is essential!*)
fun repeatc cv crec = (tryc (cv  thenc  repeatc cv)) crec;

(*Try conversions at top level before descending.*)
fun topc cv x =
   (repeatc cv  thenc
    subc (topc cv)  thenc
    tryc (cv thenc topc cv)) x;

(*rewrite a term top-down using a list of theorems *)
fun rewritec_top pairs = topc (rewritec pairs);

(*Bottom-up rewriting (formerly called "redepth"). *)
fun bottomc cv x =
   (subc (bottomc cv)  thenc
    tryc (cv thenc bottomc cv)) x;

(*rewrite a term bottom-up using a list of theorems *)
fun rewritec_bottom pairs = bottomc (rewritec pairs);

end;
