(*  Title:      Pure/SYNTAX/type_ext
    Author:     Tobias Nipkow
    Copyright   1993  University of Cambridge

The concrete syntax of types
*)

signature TYPE_EXT0 =
sig
  val typ_of_term: (indexname -> sort) -> term -> typ
end;

signature TYPE_EXT =
sig
  structure Extension:EXTENSION
  val constrain: string list * bool * term * typ -> term
  val tapplC: term
  val term_of_typ: string list * bool -> typ -> term
  val typesC: term
  val type_ext: Extension.ext
  include TYPE_EXT0
end;

functor TypeExtFun(structure Extension:EXTENSION and Lexicon:LEXICON)
	: TYPE_EXT =
struct

structure Extension = Extension;

local open Extension in

(* TRANSLATION BETWEEN TYPES AND THEIR TERM REPRESENTATION *)

val funS = "_=>_";
val funC = Const(funS,dummyT);

val tapplS = "_()type";
val tapplC = Const(tapplS,dummyT);

val typesS = "_,type";
val typesC = Const(typesS,dummyT);


fun sort_of_classes(Free(s,_)) = [s]
  | sort_of_classes(_ $ Free(s,_) $ cls) = s :: sort_of_classes cls;

fun sort_of_term(Const _) = []
  | sort_of_term(Free(s,_)) = [s]
  | sort_of_term(_ $ cls) = sort_of_classes cls;

fun sort_err(a) = error("Inconsistent sort constraints for type variable "^a);

fun typ(t as Free(a,_),mm) =
	    (if Lexicon.is_identifier a then Type(a,[]) else TFree(a,[]),mm)
  | typ(Var(v,_),mm) = (TVar(v,[]),mm)
  | typ(Const("_of sort",_)$Free(a,_)$st,(fm,vm)) =
    let val Ss = sort_of_term st and T = TFree(a,[])
    in case assoc(fm,a) of
	 None => (T,((a,Ss)::fm,vm))
       | Some(Ss') => if Ss=Ss' then (T,(fm,vm)) else sort_err(a)
    end
  | typ(Const("_of sort",_)$Var(v,_)$st,(fm,vm)) =
    let val Ss = sort_of_term st and T = TVar(v,[])
    in case assoc(vm,v) of
	 None => (T,(fm,(v,Ss)::vm))
       | Some(Ss') => if Ss=Ss' then (T,(fm,vm))
		      else sort_err(Lexicon.string_of_vname v)
    end
  | typ(_$args$Free(a,_),mm) =
    let val (Ts,mm') = typs(args,mm) in (Type(a,Ts), mm') end

and typs(Const("_,type",_)$ty$tys,mm) =
    let val (T,mm') = typ(ty,mm);
	val (Ts,mm'') = typs(tys,mm')
    in (T::Ts,mm'') end
  | typs(x) = let val (T,mm) = typ(x) in ([T],mm) end;

fun typ_of_term defaultS t =
  let val (T0,(fm,vm)) = typ (t,([],[]));
      fun vsort(ixn) =
	    case assoc(vm,ixn) of None => defaultS ixn | Some(S) => S;
      fun fsort(a) =
	    case assoc(fm,a) of None => defaultS(a,~1) | Some(S) => S;
      fun addS(Type(a,Ts)) = Type(a, map addS Ts)
	| addS(TVar(v,_)) = TVar(v, vsort v)
	| addS(TFree(a,_)) = TFree(a, fsort a)
  in addS T0 end;


val emptysortS = "_emptysort";
val emptysortC = Const(emptysortS,dummyT);

val sortconsS = "_sortcons";
val sortconsC = Const(sortconsS,dummyT);

val classesS = "_classes";
val classesC = Const(classesS,dummyT);

fun term_of_classes(S::[]) = Free(S,dummyT)
  | term_of_classes(S::Ss) = sortconsC $ Free(S,dummyT) $ term_of_classes Ss;

fun term_of_sort([]) = emptysortC
  | term_of_sort(S::[]) = Free(S,dummyT)
  | term_of_sort(Ss) = classesC $ term_of_classes Ss;

val ofsortS = "_of sort";
val ofsortC = Const(ofsortS,dummyT);

fun term_of_typ (tmixfixs,showsorts) =
let fun sort(t,Ss) =
	   if showsorts then ofsortC $ t $ term_of_sort Ss else t;
    fun typ(Type(a, [])) = Free(a,dummyT)
      | typ(Type(a, Ts)) =
	  if a mem tmixfixs then list_comb(Const(a,dummyT), map typ Ts)
	  else tapplC $ typs Ts $ Free(a,dummyT)
      | typ(TFree(a,Ss)) = sort(Free(a,dummyT),Ss)
      | typ(TVar(v,Ss)) = sort(Var(v,dummyT),Ss)
    and typs(T::Ts) = let val t = typ T
	in if Ts=[] then t else typesC $ t $ typs(Ts) end
in typ end;

fun constrain(fixs,showsorts,t,T) =
    Const(constrainC,dummyT)$ t $ term_of_typ(fixs,showsorts) T;

val typesT = Type("types",[]);

fun fun_tr[t1,t2] = tapplC$(typesC$t1$t2)$Free("fun",dummyT);

fun fun_tr'[_$t1$t2, Free("fun",_)] = funC$t1$t2;

fun bracket_tr[dom,cod] =
let fun tr(Const("_,type",_)$T$Ts) = fun_tr[T,tr Ts]
      | tr(T) = fun_tr[T,cod]
in tr dom end;

val sortT = Type("_SORT",[]);
val classesT = Type("_CLASSES",[]);

val type_ext = Ext{roots = [Extension.logic,"type"],
mfix=
[(* meta-types: *)
 Mfix("_",   idT      --> sortT, "",         [], max_pri),
 Mfix("{}",               sortT, emptysortS, [], max_pri),
 Mfix("{_}", classesT --> sortT, classesS,   [], max_pri),
 Mfix("_",    idT            --> classesT, "",        [], max_pri),
 Mfix("_,_", [idT,classesT] ---> classesT, sortconsS, [], max_pri),
 Mfix("_", tfreeT --> typeT, "",[],max_pri),
 Mfix("_", tvarT  --> typeT, "",[],max_pri),
 Mfix("_", idT    --> typeT, "",[],max_pri),
 Mfix("_::_", [tfreeT,sortT] ---> typeT, ofsortS,[max_pri,0],max_pri),
 Mfix("_::_", [tvarT,sortT]  ---> typeT, ofsortS,[max_pri,0],max_pri),
 Mfix("_ _", [typeT,idT] ---> typeT, tapplS, [max_pri,0], max_pri),
 Mfix("((1'(_'))_)", [typesT,idT] ---> typeT, tapplS,[],max_pri),
 Mfix("_", typeT --> typesT, "",[],max_pri),
 Mfix("_,/_", [typeT,typesT] ---> typesT, typesS,[],max_pri),
 Mfix("(_/ => _)", [typeT,typeT] ---> typeT, funS, [1,0], 0),
 (* an abbriviation: *)
 Mfix("([_]/ => _)", [typesT,typeT] ---> typeT, "_[]=>", [0,0], 0)
],

parse_translation = [(funS,fun_tr), ("_[]=>",bracket_tr)],
print_translation = [(tapplS,fun_tr')],
tmixfixs = []
};

end;

end;
