(*  Title:      Sextension
    Author:     Tobias Nipkow

Syntax extensions: mixfix declarations, infixes, binders, and the Pure syntax.
*)

signature SEXTENSION0 =
sig
  datatype mixfix = Mixfix of string * string * string * int list * int
		  | Delimfix of string * string * string
		  | Infixl of string * string * int
		  | Infixr of string * string * int
		  | Binder of string * string * string * int*int
		  | TInfixl of string * string * int
		  | TInfixr of string * string * int
  datatype sext = Sext of
	{mixfix: mixfix list,
	 parse_translation: (string * (term list -> term)) list,
	 print_translation: (string * (term list -> term)) list}
  val mk_binder_tr : string * string -> string * (term list -> term)
  val mk_binder_tr': string * string -> string * (term list -> term)
  val max_pri: int
end;

signature SEXTENSION1 =
sig
  include SEXTENSION0
  val pure_sext: sext
  val syntax_types: string list;
  val constants : sext -> (string list * string) list
  val constrainAbsC: string
end;

signature SEXTENSION =
sig
  include SEXTENSION1
  structure Extension:EXTENSION
  val abs_tr': term -> term
  val appl_tr': term * term list -> term
  val ext_of_sext: string list * sext * (string->typ) -> Extension.ext
end;

functor SExtensionFun(TypeExt:TYPE_EXT) : SEXTENSION =
struct

structure Extension = TypeExt.Extension

open Extension;

val islist_const' = "_Id list";
val idListConst = Const(islist_const',dummyT);

val constrainAbsC = "_constrainAbs";
val constrain = Const(constrainAbsC,dummyT);

fun list_to_bt const = let fun bt[e]= e | bt(e::l)= const$e$bt l in bt end;

fun abs_list_tr mk = fn [idts,body] =>
let fun tr(Free(id,T), t) = mk(absfree(id,T,t))
      | tr(Const("_$_",_) $ Free(id,T) $ tT, t) =
		mk(constrain $ absfree(id,T,t) $ tT)
      | tr(_ $ idt $ idts, t) = tr(idt,tr(idts,t));
in tr(idts,body) end;

fun abs_list_tr'(const,varsf,bodyf) = fn t =>
let val vars = varsf t and body = bodyf t;
    val newFvars = map Free (rename_wrt_term body vars)
    val newbody = subst_bounds(newFvars,body)
in const $ list_to_bt idListConst (rev newFvars) $ newbody end;

datatype mixfix = Mixfix of string * string * string * int list * int
		| Delimfix of string * string * string
		| Infixl of string * string * int
		| Infixr of string * string * int
		| Binder of string * string * string * int * int
		| TInfixl of string * string * int
		| TInfixr of string * string * int

datatype sext = Sext of
	{mixfix: mixfix list,
	 parse_translation: (string * (term list -> term)) list,
	 print_translation: (string * (term list -> term)) list}


val clean =
let fun q("'"::c::cs) = c^q(cs)
      | q(c::cs) = c^q(cs)
      | q([]) = ""
in q o explode end;

val opn = "op ";

fun const(Delimfix(_,ty,c)) = ([c],ty)
  | const(Mixfix(_,ty,c,_,_)) = ([c],ty)
  | const(Infixl(c,ty,_)) = ([opn ^ clean c],ty)
  | const(Infixr(c,ty,_)) = ([opn ^ clean c],ty)
  | const(Binder(_,ty,c,_,_)) = ([c],ty)
  | const _ = ([""],"") (* is filtered out in "constants" *);

fun constants(Sext{mixfix,...}) =
	distinct(filter_out (fn (l,_) => l=[""]) (map const mixfix));

fun add_tmixfix(l,mixfix) = case mixfix of
        TInfixl(_,c,_) => c::l
      | TInfixr(_,c,_) => c::l
      | _            => l;

fun mk_tinfix_tr c =
    (c,fn [t1,t2] => TypeExt.tapplC $ (TypeExt.typesC$t1$t2) $ Free(c,dummyT));


fun binder(Binder(sy,_,name,_,_)) = [(sy, name)]
  | binder _ = [];

(** Parse/print translations for binders **)

fun mk_binder_tr(sy,name) = (sy, abs_list_tr(fn t => Const(name,dummyT)$t))

fun mk_binder_tr'(name,sy) =
let val syC = Const(sy,dummyT) and nameC = Const(name,dummyT)
    fun tr' t = abs_list_tr'(syC,strip_qnt_vars name,strip_qnt_body name)
			    (nameC$t)
in (name, fn (t::ts) => list_comb(tr' t, ts)) end;

fun mfix readT =
  let val infixT = [typeT,typeT]--->typeT;
      fun binderT(ty) = case readT(ty) of
	      Type("fun",[Type("fun",[_,T2]),T3]) =>
		[Type("idts",[]),T2] ---> T3
	    | _ => error(ty ^ " is not a valid binder type.");
  in fn Mixfix(sy,ty,c,pl,p) => [Mfix(sy,readT ty,c,pl,p)]
      | Delimfix(sy,ty,c) => [Mfix(sy,readT ty,c,[],max_pri)]
      | Infixl(sy,ty,p) =>
	let val T = readT ty and c = opn^sy and c' = opn ^ clean sy
	in [Mfix(c,T,c',[],max_pri),
	    Mfix("(_ "^sy^"/ _)",T,c',[p,p+1],p)] end
      | Infixr(sy,ty,p) =>
	let val T = readT ty and c = opn^sy and c' = opn ^ clean sy
	in [Mfix(c,T,c',[],max_pri),
	    Mfix("(_ "^sy^"/ _)",T,c',[p+1,p],p)] end
      | Binder(sy,ty,_,p,q) =>
	[Mfix("(3"^sy^"_./ _)", binderT(ty), sy, [0,p], q)]
      | TInfixl(s,c,p) =>
	[Mfix("(_ "^s^"/ _)", infixT, c, [p,p+1], p)]
      | TInfixr(s,c,p) =>
	[Mfix("(_ "^s^"/ _)", infixT, c, [p+1,p], p)]
  end;

fun ext_of_sext(roots,Sext{mixfix,parse_translation,print_translation},readT) =
let val bs = flat (map binder mixfix);
    val bparses = map mk_binder_tr bs;
    val bprints = map (mk_binder_tr' o (fn (x,y) => (y,x))) bs;
    val tmixfixs = foldl add_tmixfix ([],mixfix);
    val tparses = map mk_tinfix_tr tmixfixs;
in Ext{roots= roots, mfix= flat(map (mfix readT) mixfix),
       parse_translation= tparses @ bparses @ parse_translation,
       print_translation= bprints @ print_translation,
       tmixfixs= tmixfixs}
end;

(* PURE SEXTENSION *)

val abs_const' = "_%";

val args_const' = "_Args";

(* TRANSLATIONS *)

fun args_tr(Const("_Args",_)$arg$args,f) = args_tr(args,f$arg) |
    args_tr(arg,f) = f$arg;

fun app_tr [f,args] = args_tr(args,f);

fun appl_tr'(f,args) = Const(appl_const',dummyT) $ f $
			(list_to_bt (Const(args_const',dummyT)) args)


(* Eta-contraction before printing an abstraction *)

(*Perform (partial) eta-contractions upon a term*)
fun eta_abs(Abs(a,T,t)) =
      (case eta_abs t of
         t' as (f $ u) =>
            (case eta_abs u of
               Bound 0 => if not (0 mem loose_bnos f) then incr_boundvars ~1 f 
	                  else Abs(a,T,t')
             | _ => Abs(a,T,t'))
       | t' => Abs(a,T,t'))
  | eta_abs t = t;

fun abs_tr' t = case eta_abs t of
      t' as (Abs _) => abs_list_tr' (Const(abs_const',dummyT),
			             strip_abs_vars, strip_abs_body) t'
     |t' => t';

fun semi_impl_tr [asms,concl] =
    let fun tr(Const("_asms",_)$a$al) = Const("==>",dummyT)$a$(tr al)
	  | tr(Const("_;",_)$asms) = tr asms
	  | tr(_) = concl
    in tr asms end;

fun semi_intr(asm,concl) =
let val pref = Const("_asms",dummyT) $ asm
in case concl of
	Const("==>",_)$asm'$concl' =>
	let val (asms,concl) = semi_intr(asm',concl')
	in (pref$(Const("_;",dummyT)$asms),concl) end
      | _ => (pref$Const("_eoasms",dummyT),concl)
end;

fun meta_impl_tr'[asm, concls as Const("==>",_)$_$_] =
	let val (asms,concl) = semi_intr(asm,concls)
	in Const("_;==>",dummyT)$asms$concl end
  | meta_impl_tr'[asm,concl] = Const("_==>",dummyT)$asm$concl;

fun meta_impl_tr[asm,concl] = Const("==>",dummyT)$asm$concl;

(* SYNTAX *)

val pure_sext = Sext{
mixfix=
[Mixfix("(3%_./ _)", "[idts, 'a] => ('b=>'a)", abs_const', [0],0),
 Delimfix("_","'a =>"^args,""),
 Delimfix("_,/_", "['a, "^args^"] => "^args, args_const'),
 Delimfix("_", "id => idt", ""),
 Mixfix("_::_", "[id, type] => idt", "_$_", [0,0], 0),
 Delimfix("'(_')", "idt => idt", ""),
 Delimfix("_", "idt => idts", ""),
 Mixfix("_/ _", "[idt, idts] => idts", islist_const', [1,0], 0),

 Delimfix("_", "id => aprop", ""),
 Delimfix("_", "var => aprop", ""),
 Mixfix("_'(_')","[('b=>'a),"^args^"]=>aprop", appl_const',[max_pri,0],0),
 Delimfix("PROP _", "aprop => prop", ""),

 Delimfix("__", "[prop, asms'] => asms", "_asms"),
 Delimfix("", "asms'", "_eoasms"),
 Delimfix(";/ _", "asms => asms'", "_;"),

 Mixfix("((3[| _ |]) ==>/ _)", "[asms, prop] => prop", "_;==>", [0,1], 1),

 Mixfix("(_ ==/ _)", "['a::logic, 'a] => prop", "==", [3,2], 2),
 Mixfix("(_ =?=/ _)", "['a::logic, 'a] => prop", "=?=", [3,2], 2),
 Mixfix("(_ ==>/ _)", "[prop, prop] => prop", "_==>", [2,1], 1),
 Binder("!!", "('a::logic => prop) => prop", "all", 0, 0)],
parse_translation =
[(appl_const',app_tr), (abs_const', abs_list_tr I),
 ("_==>",meta_impl_tr), ("_;==>",semi_impl_tr)],
print_translation = [("==>",meta_impl_tr')]
};

val syntax_types =
[logic, "aprop", args, "asms", "asms'", id, "idt", "idts",
 tfree, tvar, "type", var];

end;
