(*  Title: 	HOL/sexp
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1992  University of Cambridge

For sexp.thy.  S-expressions.
*)

open Sexp;

(** apfst -- can be used in similar type definitions **)

goalw Sexp.thy [apfst_def] "apfst(f,<a,b>) = <f(a),b>";
by (stac fst 1);
by (stac snd 1);
by (rtac refl 1);
val apfstI = result();

val [major,minor] = goalw Sexp.thy [apfst_def] 
    "[| q = apfst(f,p);  !!x y. [| p = <x,y>;  q = <f(x),y> |] ==> R \
\    |] ==> R";
by (rtac (surjective_pairing RS minor) 1);
by (rtac major 1);
val apfstE = result();

(** Push -- an injection, analogous to Cons on lists... *)

val [major] = goalw Sexp.thy [Push_def] "Push(x,f)=Push(y,g) ==> x=y";
by (rtac (major RS fun_cong RS box_equals) 1);
by (rtac nat_case_0 1);
by (rtac nat_case_0 1);
val Push_inject1 = result();

val [major] = goalw Sexp.thy [Push_def] "Push(x,f)=Push(y,g) ==> f=g";
by (rtac (major RS fun_cong RS ext RS box_equals) 1);
by (rtac (nat_case_Suc RS ext) 1);
by (rtac (nat_case_Suc RS ext) 1);
val Push_inject2 = result();

val major::prems = goal Sexp.thy
    "[| Push(x,f)=Push(y,g);  [| x=y;  f=g |] ==> P \
\    |] ==> P";
by (resolve_tac prems 1);
by (rtac (major RS Push_inject1) 1);
by (rtac (major RS Push_inject2) 1);
val Push_inject = result();

val [major] = goalw Sexp.thy [Push_def] "Push(Suc(k),f)=(%z.0) ==> P";
by (rtac (major RS fun_cong RS box_equals RS Suc_neq_Zero) 1);
by (rtac nat_case_0 1);
by (rtac refl 1);
val PushSuc_neq_K0 = result();

(** the sexp functional **)

goal Sexp.thy
    "mono(%Z. range(Atom_Rep) Un (UN i:Z. UN j:Z. {Scons_Rep(i,j)}))";
by (REPEAT (ares_tac [monoI, subset_refl, UN_mono, Un_mono] 1));
val Sexp_fun_mono = result();

val Sexp_unfold = Sexp_fun_mono RS (Sexp_def RS def_Tarski);

(** Atom_Rep and Scons_Rep: Representations of the constructors **)

goal Sexp.thy "Atom_Rep(a) : Sexp";
by (rtac (Sexp_unfold RS ssubst) 1);
by (rtac (rangeI RS UnI1) 1);
val Atom_RepI = result();

val prems = goal Sexp.thy
    "[| i: Sexp;  j: Sexp |] ==> Scons_Rep(i,j) : Sexp";
by (rtac (Sexp_unfold RS ssubst) 1);
by (rtac UnI2 1);
by (DEPTH_SOLVE (resolve_tac (prems@[refl,UN_I,singletonI]) 1));
val Scons_RepI = result();

(** Induction **)

val major::prems = goal Sexp.thy 
    "[| ii: Sexp;  !!a. P(Atom_Rep(a));   \
\       !!i j. [| i: Sexp; j: Sexp; P(i); P(j) |] ==> \
\              P(Scons_Rep(i,j)) \
\    |]  ==> P(ii)";
by (rtac (major RS (Sexp_def RS def_induct)) 1);
by (rtac Sexp_fun_mono 1);
by (fast_tac (set_cs addIs prems) 1);
val Sexp_induct = result();

val prems = goalw Sexp.thy [Atom_def,Scons_def]
    "[| !!a. P(Atom(a));   \
\       !!M N. [| P(M); P(N) |] ==> P(M.N) |]  ==> P(MM)";
by (rtac (Rep_Sexp_inverse RS subst) 1);   (*types force good instantiation*)
by (rtac (Rep_Sexp RS Sexp_induct) 1);
by (REPEAT (ares_tac prems 1
     ORELSE eresolve_tac [Abs_Sexp_inverse RS subst] 1));
val sexp_induct = result();

(*Perform induction on N. *)
fun sexp_ind_tac a i = 
    EVERY [res_inst_tac [("MM",a)] sexp_induct i,
	   rename_last_tac a ["1","2"] (i+1)];


(*** Isomorphisms ***)

goal Sexp.thy "inj(Rep_Sexp)";
by (rtac inj_inverseI 1);
by (rtac Rep_Sexp_inverse 1);
val inj_Rep_Sexp = result();

goal Sexp.thy "inj_onto(Abs_Sexp,Sexp)";
by (rtac inj_onto_inverseI 1);
by (etac Abs_Sexp_inverse 1);
val inj_onto_Abs_Sexp = result();

(*** Distinctness of constructors ***)

goalw Sexp.thy [Atom_Rep_def,Scons_Rep_def] "~ (Scons_Rep(i,j) = Atom_Rep(a))";
by (rtac notI 1);
by (etac (equalityD2 RS subsetD RS UnE) 1);
by (rtac singletonI 1);
by (REPEAT (eresolve_tac [imageE, apfstE, Pair_inject, 
			  sym RS PushSuc_neq_K0] 1));
val Scons_Rep_not_Atom_Rep = result();

goalw Sexp.thy [Atom_def,Scons_def] "~ ((M.N) = Atom(a))";
by (rtac (inj_onto_Abs_Sexp RS inj_onto_contraD) 1);
by (rtac Scons_Rep_not_Atom_Rep 1);
by (REPEAT (resolve_tac [Atom_RepI, Rep_Sexp, Scons_RepI] 1));
val Scons_not_Atom = result();
val Atom_not_Scons = standard (Scons_not_Atom RS not_sym);

val Scons_neq_Atom = standard (Scons_not_Atom RS notE);
val Atom_neq_Scons = sym RS Scons_neq_Atom;


(** Injectiveness of Atom **)

val [major] = goalw Sexp.thy [Atom_Rep_def]
    "Atom_Rep(a) = Atom_Rep(b) ==> a=b";
by (rtac (major RS equalityD1 RS subsetD RS singletonD RS Pair_inject) 1);
by (rtac singletonI 1);
by (assume_tac 1);
val Atom_Rep_inject = result();

goalw Sexp.thy [Atom_def] "inj(Atom)";
by (rtac injI 1);
by (etac (inj_onto_Abs_Sexp RS inj_ontoD RS Atom_Rep_inject) 1);
by (REPEAT (resolve_tac [Rep_Sexp, Atom_RepI] 1));
val inj_Atom = result();
val Atom_inject = inj_Atom RS injD;


(** Injectiveness of Scons **)

val [major,minor] = goalw Sexp.thy [Scons_Rep_def]
    "[| Scons_Rep(i,j)<=Scons_Rep(i',j');  [| i<=i';  j<=j' |] ==> P \
\    |] ==> P";
by (rtac minor 1);
by (ALLGOALS (EVERY'
     [rtac subsetI,
      rtac (major RS subsetD RS UnE),
      resolve_tac [UnI1,UnI2],      etac imageI]));
by (REPEAT (hyp_subst_tac 1
     ORELSE eresolve_tac [asm_rl, sym RS PushSuc_neq_K0, make_elim Suc_inject,
			  Suc_neq_Zero, Zero_neq_Suc, 
			  imageE, apfstE, sym RS apfstE, 
			  Pair_inject, Push_inject] 1));
val Scons_Rep_inj_lemma = result();

val major::prems = goal Sexp.thy
    "[| Scons_Rep(i,j)=Scons_Rep(i',j');  [| i=i';  j=j' |] ==> P \
\    |] ==> P";
by (rtac (major RS equalityE) 1);
by (resolve_tac prems 1);
by (REPEAT (ares_tac [equalityI] 1
     ORELSE etac Scons_Rep_inj_lemma 1));
val Scons_Rep_inject = result();

val [major,minor] = goalw Sexp.thy [Scons_def]
    "[| (M.N)=(M'.N');  [| M=M';  N=N' |] ==> P \
\    |] ==> P";
by (rtac (inj_onto_Abs_Sexp RS inj_ontoD RS Scons_Rep_inject) 1);
by (rtac major 1);
by (REPEAT (resolve_tac [Rep_Sexp, Scons_RepI] 1));
by (rtac minor 1);
by (REPEAT (etac (inj_Rep_Sexp RS injD) 1));
val Scons_inject = result();

(** Some rewrite rules **)

goal Sexp.thy "(M.N = M'.N') = (M=M' & N=N')";
by (REPEAT (ares_tac [refl,iffI,conjI] 1
     ORELSE eresolve_tac [conjE, Scons_inject, ssubst] 1));
val Scons_Scons_eq = result();

val sexp_ss = 
    HOL_ss addcongs (mk_congs Sexp.thy ["op ."])
           addrews [Scons_not_Atom, Atom_not_Scons, Scons_Scons_eq];

goal Sexp.thy "!M. ~(N=(M.N))";
by (sexp_ind_tac "N" 1);
by (rtac (Atom_not_Scons RS allI) 1);
by (ASM_SIMP_TAC sexp_ss 1);
val n_not_Scons_N = result();

(** sexp_case **)

goalw Sexp.thy [sexp_case_def] "sexp_case(Atom(a),c,d) = c(a)";
by (fast_tac (HOL_cs addIs [select_equality] 
	             addEs [make_elim Atom_inject, Atom_neq_Scons]) 1);
val sexp_case_Atom = result();

goalw Sexp.thy [sexp_case_def] "sexp_case(M.N, c, d) = d(M,N)";
by (fast_tac (HOL_cs addIs [select_equality] 
	             addEs [Scons_inject, Scons_neq_Atom]) 1);
val sexp_case_Scons = result();


(** Introduction rules for 'pred_sexp' **)

goalw Sexp.thy [pred_sexp_def] "<M, M.N> : pred_sexp";
by (fast_tac set_cs 1);
val pred_sexpI1 = result();

goalw Sexp.thy [pred_sexp_def] "<N, M.N> : pred_sexp";
by (fast_tac set_cs 1);
val pred_sexpI2 = result();

(*Combinations involving transitivity and the rules above*)
val pred_sexp_t1 = pred_sexpI1 RS r_into_trancl
and pred_sexp_t2 = pred_sexpI2 RS r_into_trancl;

val pred_sexp_trans1 = pred_sexp_t1 RSN (2, trans_trancl RS transD)
and pred_sexp_trans2 = pred_sexp_t2 RSN (2, trans_trancl RS transD);

fun pred_sexp_tac i =
    DEPTH_SOLVE_1 (ares_tac [pred_sexp_t1, pred_sexp_t2,
			     pred_sexp_trans1, pred_sexp_trans2] i);
				     

val major::prems = goalw Sexp.thy [pred_sexp_def]
    "[| p : pred_sexp;  \
\       !!M N. [| p = <M, M.N> |] ==> R; \
\       !!M N. [| p = <N, M.N> |] ==> R  \
\    |] ==> R";
by (rtac (major RS CollectE) 1);
by (REPEAT (eresolve_tac ([asm_rl,exE,disjE]@prems) 1));
val pred_sexpE = result();

goalw Sexp.thy [wf_def] "wf(pred_sexp)";
by (strip_tac 1);
by (sexp_ind_tac "x" 1);
by (fast_tac (HOL_cs addSEs [mp, pred_sexpE, Pair_inject, Scons_inject]) 2);
by (fast_tac (HOL_cs addSEs [mp, pred_sexpE, Pair_inject, Atom_neq_Scons]) 1);
val wf_pred_sexp = result();


(*** sexp_rec -- by wf recursion on pred_sexp ***)

(** conversion rules **)

val sexp_rec_unfold = wf_pred_sexp RS (sexp_rec_def RS def_wfrec);

goal Sexp.thy "sexp_rec(Atom(a), c, h) = c(a)";
by (stac sexp_rec_unfold 1);
by (rtac sexp_case_Atom 1);
val sexp_rec_Atom = result();

goal Sexp.thy
    "sexp_rec(M.N, c, h) = h(M, N, sexp_rec(M,c,h), sexp_rec(N,c,h))";
by (rtac (sexp_rec_unfold RS trans) 1);
by (rtac (sexp_case_Scons RS trans) 1);
by (SIMP_TAC (HOL_ss 
      addcongs (mk_typed_congs Sexp.thy [("h", "[?'a, ?'a, ?'b, ?'b]=> ?'b")])
      addrews [pred_sexpI1, pred_sexpI2, cut_apply]) 1);
val sexp_rec_Scons = result();

val sexp_ss = HOL_ss addrews
  [sexp_case_Atom, sexp_case_Scons,
   sexp_rec_Atom, sexp_rec_Scons];
