(*  Title: 	HOL/ex/simult.ML
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

For simult.thy.

Primitives for simultaneous recursive type definitions
  includes worked example of trees & forests

This is essentially the same data structure that on ex/term.ML, which is
simpler because it uses List as a new type former.  The approach in this
file may be superior for other simultaneous recursions.
*)

open Simult;

(*** General rules for Part ***)

val prems = goalw Simult.thy [Part_def] "h(a) : A ==> h(a) : Part(A,h)";
by (cfast_tac prems 1);
val PartI = result();

val major::prems = goalw Simult.thy [Part_def]
    "[| a : Part(A,h);  !!z. [| a : A;  a=h(z) |] ==> P  \
\    |] ==> P";
by (rtac (major RS CollectE) 1);
by (etac conjE 1);
by (etac exE 1);
by (REPEAT (ares_tac prems 1));
val PartE = result();

val [prem] = goal Simult.thy "A<=B ==> Part(A,h) <= Part(B,h)";
by (fast_tac (set_cs addSIs [PartI, prem RS subsetD] addSEs [PartE]) 1);
val Part_mono = result();


(*** Monotonicity and unfolding of the function ***)

goal Simult.thy "mono(%Z.  A <*> Part(Z,IN1) \
\                      <+> ({NUMB(0)} <+> Part(Z,IN0) <*> Part(Z,IN1)))";
by (REPEAT (ares_tac [monoI, subset_refl, usum_mono, uprod_mono,
		      Part_mono] 1));
val TF_fun_mono = result();

val TF_unfold = TF_fun_mono RS (TF_def RS def_Tarski);


(** Elimination -- structural induction on the set TF **)

val TF_Rep_defs = [TCONS_def,FNIL_def,FCONS_def];

val major::prems = goalw Simult.thy TF_Rep_defs
 "[| i: TF(A);  \
\    !!M N. [| M: A;  N: Part(TF(A),IN1);  R(N) |] ==> R(TCONS(M,N));	\
\    R(FNIL);        		\
\    !!M N. [| M:  Part(TF(A),IN0);  N: Part(TF(A),IN1);  R(M);  R(N) \
\            |] ==> R(FCONS(M,N))    \
\    |] ==> R(i)";
by (rtac (major RS (TF_def RS def_induct)) 1);
by (rtac TF_fun_mono 1);
by (fast_tac (set_cs addIs (prems@[PartI])
		       addEs [usumE, uprodE, PartE]) 1);
val TF_induct = result();

(*This lemma replaces a use of subgoal_tac to prove tree_forest_induct*)
val prems = goalw Simult.thy [Part_def]
 "! M: TF(A). (M: Part(TF(A),IN0) --> P(M)) & (M: Part(TF(A),IN1) --> Q(M)) \
\ ==> (! M: Part(TF(A),IN0). P(M)) & (! M: Part(TF(A),IN1). Q(M))";
by (cfast_tac prems 1);
val TF_induct_lemma = result();

val uplus_cs = set_cs addSIs [PartI]
		      addSDs [IN0_inject, IN1_inject]
		      addSEs [IN0_neq_IN1, IN1_neq_IN0, PartE];

(*Could prove  ~ TCONS(M,N) : Part(TF(A),IN1)  etc. *)

(*Induction on TF with separate predicates P, Q*)
val prems = goalw Simult.thy TF_Rep_defs
    "[| !!M N. [| M: A;  N: Part(TF(A),IN1);  Q(N) |] ==> P(TCONS(M,N)); \
\       Q(FNIL);        \
\       !!M N. [| M:  Part(TF(A),IN0);  N: Part(TF(A),IN1);  P(M);  Q(N) \
\               |] ==> Q(FCONS(M,N))     \
\    |] ==> (! M: Part(TF(A),IN0). P(M)) & (! N: Part(TF(A),IN1). Q(N))";
by (rtac (ballI RS TF_induct_lemma) 1);
by (etac TF_induct 1);
bws TF_Rep_defs;
by (ALLGOALS (fast_tac (uplus_cs addIs prems)));
(*29 secs??*)
val Tree_Forest_induct = result();

(*Induction for the abstract types 'a tree, 'a forest*)
val prems = goalw Simult.thy [Tcons_def,Fnil_def,Fcons_def]
    "[| !!x ts. Q(ts) ==> P(Tcons(x,ts));     \
\	Q(Fnil);        \
\       !!t ts. [| P(t);  Q(ts) |] ==> Q(Fcons(t,ts))    \
\    |] ==> (! t. P(t)) & (! ts. Q(ts))";
by (res_inst_tac [("P1","%z.P(Abs_Tree(z))"),
		  ("Q1","%z.Q(Abs_Forest(z))")] 
    (Tree_Forest_induct RS conjE) 1);
(*Instantiates ?A1 to range(ATOM). *)
by (fast_tac (set_cs addSEs [Rep_Tree_inverse RS subst, 
			     Rep_Forest_inverse RS subst] 
	             addSIs [Rep_Tree,Rep_Forest]) 4);
(*Cannot use simplifier: the rewrites work in the wrong direction!*)
by (ALLGOALS (fast_tac (set_cs addSEs [Abs_Tree_inverse RS subst,
                          Abs_Forest_inverse RS subst] 
	             addSIs prems)));
val tree_forest_induct = result();



(*** Isomorphisms ***)

goal Simult.thy "inj(Rep_Tree)";
by (rtac inj_inverseI 1);
by (rtac Rep_Tree_inverse 1);
val inj_Rep_Tree = result();

goal Simult.thy "inj_onto(Abs_Tree,Part(TF(range(ATOM)),IN0))";
by (rtac inj_onto_inverseI 1);
by (etac Abs_Tree_inverse 1);
val inj_onto_Abs_Tree = result();

goal Simult.thy "inj(Rep_Forest)";
by (rtac inj_inverseI 1);
by (rtac Rep_Forest_inverse 1);
val inj_Rep_Forest = result();

goal Simult.thy "inj_onto(Abs_Forest,Part(TF(range(ATOM)),IN1))";
by (rtac inj_onto_inverseI 1);
by (etac Abs_Forest_inverse 1);
val inj_onto_Abs_Forest = result();

(** Introduction rules for constructors **)

(* c : A <*> Part(TF(A),IN1) 
        <+> {NUMB(0)} <+> Part(TF(A),IN0) <*> Part(TF(A),IN1) ==> c : TF(A) *)
val TF_I = TF_unfold RS equalityD2 RS subsetD;

(*For reasoning about the representation*)
val TF_Rep_cs = uplus_cs addIs [TF_I, uprodI, usum_IN0I, usum_IN1I]
	                 addSEs [Scons_inject];

val prems = goalw Simult.thy [TCONS_def]
    "[| a: A;  M: Part(TF(A),IN1) |] ==> TCONS(a,M) : Part(TF(A),IN0)";
by (fast_tac (TF_Rep_cs addIs prems) 1);
val TCONS_I = result();

(* FNIL is a TF(A) -- this also justifies the type definition*)
goalw Simult.thy [FNIL_def] "FNIL: Part(TF(A),IN1)";
by (fast_tac TF_Rep_cs 1);
val FNIL_I = result();

val prems = goalw Simult.thy [FCONS_def]
    "[| M: Part(TF(A),IN0);  N: Part(TF(A),IN1) |] ==> \
\    FCONS(M,N) : Part(TF(A),IN1)";
by (fast_tac (TF_Rep_cs addIs prems) 1);
val FCONS_I = result();

(** Injectiveness of TCONS and FCONS **)

goalw Simult.thy [TCONS_def] "(TCONS(K,M)=TCONS(L,N)) = (K=L & M=N)";
by (fast_tac TF_Rep_cs 1);
val TCONS_TCONS_eq = result();
val TCONS_inject = standard (TCONS_TCONS_eq RS iffD1 RS conjE);

goalw Simult.thy [FCONS_def] "(FCONS(K,M)=FCONS(L,N)) = (K=L & M=N)";
by (fast_tac TF_Rep_cs 1);
val FCONS_FCONS_eq = result();
val FCONS_inject = standard (FCONS_FCONS_eq RS iffD1 RS conjE);

(** Distinctness of TCONS, FNIL and FCONS **)

goalw Simult.thy TF_Rep_defs "~ TCONS(M,N) = FNIL";
by (fast_tac TF_Rep_cs 1);
val TCONS_not_FNIL = result();
val FNIL_not_TCONS = standard (TCONS_not_FNIL RS not_sym);

val TCONS_neq_FNIL = standard (TCONS_not_FNIL RS notE);
val FNIL_neq_TCONS = sym RS TCONS_neq_FNIL;

goalw Simult.thy TF_Rep_defs "~ FCONS(M,N) = FNIL";
by (fast_tac TF_Rep_cs 1);
val FCONS_not_FNIL = result();
val FNIL_not_FCONS = standard (FCONS_not_FNIL RS not_sym);

val FCONS_neq_FNIL = standard (FCONS_not_FNIL RS notE);
val FNIL_neq_FCONS = sym RS FCONS_neq_FNIL;

goalw Simult.thy TF_Rep_defs "~ TCONS(M,N) = FCONS(K,L)";
by (fast_tac TF_Rep_cs 1);
val TCONS_not_FCONS = result();
val FCONS_not_TCONS = standard (TCONS_not_FCONS RS not_sym);

val TCONS_neq_FCONS = standard (TCONS_not_FCONS RS notE);
val FCONS_neq_TCONS = sym RS TCONS_neq_FCONS;

(*???? Too many derived rules ????
  Automatically generate symmetric forms?  Always expand TF_Rep_defs? *)

(** Injectiveness of Tcons and Fcons **)

(*For reasoning about abstract constructors*)
val TF_cs = set_cs addSIs [Rep_Tree, Rep_Forest, TCONS_I, FNIL_I, FCONS_I]
	           addSEs [TCONS_inject, FCONS_inject,
			   TCONS_neq_FNIL, FNIL_neq_TCONS,
			   FCONS_neq_FNIL, FNIL_neq_FCONS,
			   TCONS_neq_FCONS, FCONS_neq_TCONS]
		   addSDs [inj_onto_Abs_Tree RS inj_ontoD,
			   inj_onto_Abs_Forest RS inj_ontoD,
			   inj_Rep_Tree RS injD, inj_Rep_Forest RS injD,
			   ATOM_inject];

goalw Simult.thy [Tcons_def] "(Tcons(x,xs)=Tcons(y,ys)) = (x=y & xs=ys)";
by (fast_tac TF_cs 1);
val Tcons_Tcons_eq = result();
val Tcons_inject = standard (Tcons_Tcons_eq RS iffD1 RS conjE);

goalw Simult.thy [Fcons_def,Fnil_def] "~ Fcons(x,xs) = Fnil";
by (fast_tac TF_cs 1);
val Fcons_not_Fnil = result();

val Fcons_neq_Fnil = standard (Fcons_not_Fnil RS notE);;
val Fnil_neq_Fcons = sym RS Fcons_neq_Fnil;


(** Injectiveness of Fcons **)

goalw Simult.thy [Fcons_def] "(Fcons(x,xs)=Fcons(y,ys)) = (x=y & xs=ys)";
by (fast_tac TF_cs 1);
val Fcons_Fcons_eq = result();
val Fcons_inject = standard (Fcons_Fcons_eq RS iffD1 RS conjE);


(*** TF_rec -- by wf recursion on pred_sexp ***)

val TF_rec_unfold =
    wf_pred_sexp RS wf_trancl RS (TF_rec_def RS def_wfrec);

(** conversion rules **)

goalw Simult.thy [IN0_def,TCONS_def] "<N, TCONS(M,N)> : pred_sexp^+";
by (pred_sexp_tac 1);
val pred_sexp_TCONS = result();

goalw Simult.thy [IN1_def,FCONS_def] "<M, FCONS(M,N)> : pred_sexp^+";
by (pred_sexp_tac 1);
val pred_sexp_FCONS1 = result();

goalw Simult.thy [IN1_def,FCONS_def] "<N, FCONS(M,N)> : pred_sexp^+";
by (pred_sexp_tac 1);
val pred_sexp_FCONS2 = result();

goal Simult.thy
    "TF_rec(TCONS(M,N),b,c,d) = b(M, N, TF_rec(N,b,c,d))";
by (rtac (TF_rec_unfold RS trans) 1);
by (rtac (select_equality RS trans) 1);
by (fast_tac HOL_cs 1);		(*verifies that TCONS(M,N) = TCONS(M,N) *)
by (fast_tac TF_cs 1);
by (SIMP_TAC (HOL_ss 
      addcongs (mk_typed_congs Simult.thy [("b", "[?'x, ?'x, ?'y]=>?'y")])
      addrews [pred_sexp_TCONS, cut_apply]) 1);
val TF_rec_TCONS = result();

goal Simult.thy "TF_rec(FNIL,b,c,d) = c";
by (rtac (TF_rec_unfold RS trans) 1);
by (fast_tac (TF_cs addIs [select_equality]) 1);
val TF_rec_FNIL = result();

goal Simult.thy
    "TF_rec(FCONS(M,N),b,c,d) = d(M, N, TF_rec(M,b,c,d), TF_rec(N,b,c,d))";
by (rtac (TF_rec_unfold RS trans) 1);
by (rtac (select_equality RS trans) 1);
by (fast_tac HOL_cs 1);		(*verifies that FCONS(M,N) = FCONS(M,N) *)
by (fast_tac TF_cs 1);
by (SIMP_TAC (HOL_ss 
      addcongs (mk_typed_congs Simult.thy [("d", "[?'x, ?'x, ?'y, ?'y]=>?'y")])
      addrews [pred_sexp_FCONS1, pred_sexp_FCONS2, cut_apply]) 1);
val TF_rec_FCONS = result();


(*** tree_rec, forest_rec -- by TF_rec ***)

val tf_rec_ss = 
 HOL_ss addcongs (mk_congs Simult.thy ["TF_rec"]@
		  mk_typed_congs Simult.thy
		     [("b", "[?'a,?'a forest,?'b]=>?'b"),
		      ("d", "[?'a tree,?'a forest,?'b,?'b]=>?'b")])
        addrews [TF_rec_TCONS, TF_rec_FNIL, TF_rec_FCONS,
		 TCONS_I, FNIL_I, FCONS_I, Rep_Tree, Rep_Forest, rangeI, 
		 Rep_Tree_inverse, Rep_Forest_inverse,
		 Abs_Tree_inverse, Abs_Forest_inverse,
		 inj_ATOM, Inv_f_f];

goalw Simult.thy [tree_rec_def, forest_rec_def, Tcons_def]
    "tree_rec(Tcons(a,tf),b,c,d) = b(a,tf,forest_rec(tf,b,c,d))";
by (SIMP_TAC tf_rec_ss 1);
val tree_rec_Tcons = result();

goalw Simult.thy [forest_rec_def, Fnil_def] "forest_rec(Fnil,b,c,d) = c";
by (SIMP_TAC tf_rec_ss 1);
val forest_rec_Fnil = result();

goalw Simult.thy [tree_rec_def, forest_rec_def, Fcons_def]
    "forest_rec(Fcons(t,tf),b,c,d) = \
\    d(t,tf,tree_rec(t,b,c,d), forest_rec(tf,b,c,d))";
by (SIMP_TAC tf_rec_ss 1);
val forest_rec_Cons = result();
