(*  Title: 	HOL/arith.ML
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1993  University of Cambridge

For HOL/arith.thy.

Proofs about elementary arithmetic: addition, multiplication, etc.
Tests definitions and simplifier.
*)

open Arith;

(*** Basic rewrite and congruence rules for the arithmetic operators ***)

val [add_0,add_Suc] = nat_recs add_def; 
val [mult_0,mult_Suc] = nat_recs mult_def; 

(** Difference **)

val diff_0 = diff_def RS def_nat_rec_0;

val diff_0_eq_0 = prove_goalw Arith.thy [diff_def] "0 - n = 0"
 (fn _ => [ (nat_ind_tac "n" 1), (ALLGOALS (ASM_SIMP_TAC nat_ss)) ]);

(*Must simplify BEFORE the induction!!  (Else we get a critical pair)
  Suc(m) - Suc(n)   rewrites to   pred(Suc(m) - n)  *)
val diff_Suc_Suc = prove_goalw Arith.thy [diff_def] "Suc(m) - Suc(n) = m - n"
 (fn _ =>
  [ (ASM_SIMP_TAC nat_ss 1),
    (nat_ind_tac "n" 1),
    (ALLGOALS (ASM_SIMP_TAC nat_ss)) ]);

(*** Simplification over add, mult, diff ***)

val arith_rews = [add_0, add_Suc,
		  mult_0, mult_Suc,
		  diff_0, diff_0_eq_0, diff_Suc_Suc];

val arith_congs = mk_congs Arith.thy ["op +", "op -", "op *"];

val arith_ss = nat_ss addcongs arith_congs addrews arith_rews;


(**** Inductive properties of the operators ****)

(*** Addition ***)

val add_0_right = prove_goal Arith.thy "m + 0 = m"
 (fn _ => [ (nat_ind_tac "m" 1), (ALLGOALS (ASM_SIMP_TAC arith_ss))  ]);

val add_Suc_right = prove_goal Arith.thy "m + Suc(n) = Suc(m+n)"
 (fn _ => [ (nat_ind_tac "m" 1), (ALLGOALS (ASM_SIMP_TAC arith_ss))  ]);

(*Associative law for addition*)
val add_assoc = prove_goal Arith.thy "(m + n) + k = m + (n + k)"
 (fn _ => [ (nat_ind_tac "m" 1), (ALLGOALS (ASM_SIMP_TAC arith_ss)) ]);

(*Commutative law for addition.  Must simplify after first induction!
  Orientation of rewrites is delicate*)  
val add_commute = prove_goal Arith.thy "m + n = n + m"
 (fn _ =>
  [ (nat_ind_tac "m" 1),
    (ALLGOALS (ASM_SIMP_TAC (arith_ss addrews [add_0_right,add_Suc_right])))]);

(*** Multiplication ***)

(*right annihilation in product*)
val mult_0_right = prove_goal Arith.thy "m * 0 = 0"
 (fn _ => [ (nat_ind_tac "m" 1), (ALLGOALS (ASM_SIMP_TAC arith_ss))  ]);

(*right Sucessor law for multiplication*)
val mult_Suc_right = prove_goal Arith.thy 
    "m * Suc(n) = m + (m * n)"
 (fn _ =>
  [ (nat_ind_tac "m" 1),
    (ALLGOALS (ASM_SIMP_TAC (arith_ss addrews [add_assoc RS sym]))),
       (*The final goal requires the commutative law for addition*)
    (REPEAT (ares_tac ([refl,add_commute]@arith_congs) 1))  ]);

(*Commutative law for multiplication*)
val mult_commute = prove_goal Arith.thy "m * n = n * m"
 (fn _ => 
  [ (nat_ind_tac "m" 1), 
    (ALLGOALS (ASM_SIMP_TAC 
	       (arith_ss addrews [mult_0_right,mult_Suc_right]))) ]);

(*addition distributes over multiplication*)
val add_mult_distrib = prove_goal Arith.thy "(m + n) * k = (m * k) + (n * k)"
 (fn _ =>
  [ (nat_ind_tac "m" 1),
    (ALLGOALS (ASM_SIMP_TAC (arith_ss addrews [add_assoc RS sym]))) ]);

(*Associative law for multiplication*)
val mult_assoc = prove_goal Arith.thy "(m * n) * k = m * (n * k)"
  (fn _ => 
   [ (nat_ind_tac "m" 1),
     (ALLGOALS (ASM_SIMP_TAC (arith_ss addrews [add_mult_distrib]))) ]);


(*** Difference ***)

val diff_self_eq_0 = prove_goal Arith.thy "m - m = 0"
 (fn _ => [ (nat_ind_tac "m" 1), (ALLGOALS (ASM_SIMP_TAC arith_ss)) ]);

(*Addition is the inverse of subtraction: if n<=m then n+(m-n) = m. *)
val [prem] = goal Arith.thy "[| ~ m<n |] ==> n + (m-n) = m";
by (rtac (prem RS rev_mp) 1);
by (res_inst_tac [("m","m"),("n","n")] diff_induct 1);
by (ALLGOALS (ASM_SIMP_TAC arith_ss));
val add_diff_inverse = result();


(*** Remainder ***)

goal Arith.thy "m - n < Suc(m)";
by (res_inst_tac [("m","m"),("n","n")] diff_induct 1);
by (etac less_SucE 3);
by (ALLGOALS (ASM_SIMP_TAC (arith_ss addrews [less_SucI])));
val diff_less_Suc = result();

(*In ordinary notation: if 0<n and n<=m then m-n < m *)
val prems = goal Arith.thy "[| 0<n; ~ m<n |] ==> m - n < m";
by (subgoal_tac "0<n --> ~ m<n --> m - n < m" 1);
by (cfast_tac prems 1);
by (res_inst_tac [("m","m"),("n","n")] diff_induct 1);
by (ALLGOALS
    (ASM_SIMP_TAC (nat_ss addrews (prems@[diff_less_Suc,diff_Suc_Suc]))));
val quorem_termination = result();

val wf_less_trans = wf_pred_nat RS wf_trancl RSN (2, def_wfrec RS trans);

goalw Nat.thy [less_def] "<m,n> : pred_nat^+ = (m<n)";
by (rtac refl 1);
val less_eq = result();

val quorem_ss = nat_ss addrews [quorem_termination, cut_apply, less_eq];

val prems = goal Arith.thy "m<n ==> m//n = m";
by (rtac (mod_def RS wf_less_trans) 1);
by (SIMP_TAC (nat_ss addrews prems) 1);
val mod_less = result();

val prems = goal Arith.thy "[| 0<n;  ~m<n |] ==> m//n = (m-n)//n";
by (rtac (mod_def RS wf_less_trans) 1);
by (SIMP_TAC (quorem_ss addrews prems) 1);
val mod_geq = result();


(*** Quotient ***)

val prems = goal Arith.thy "m<n ==> m/n = 0";
by (rtac (quo_def RS wf_less_trans) 1);
by (SIMP_TAC (nat_ss addrews prems) 1);
val quo_less = result();

val prems = goal Arith.thy "[| 0<n;  ~m<n |] ==> m/n = Suc((m-n)/n)";
by (rtac (quo_def RS wf_less_trans) 1);
by (SIMP_TAC (quorem_ss addrews prems) 1);
val quo_geq = result();

(*Main Result about quotient and remainder.*)
val prems = goal Arith.thy "[| 0<n |] ==> (m/n)*n + m//n = m";
by (res_inst_tac [("n","m")] less_induct 1);
by (rename_tac "k" 1);    (*Variable name used in line below*)
by (res_inst_tac [("Q","k<n")] (excluded_middle RS disjE) 1);
by (ALLGOALS (ASM_SIMP_TAC (arith_ss addrews (prems @
        [mod_less, mod_geq, quo_less, quo_geq,
	 add_assoc, add_diff_inverse, quorem_termination]))));
val mod_quo_equality = result();


(*** More results about difference ***)

val [prem] = goal Arith.thy "m < Suc(n) ==> m-n = 0";
by (rtac (prem RS rev_mp) 1);
by (res_inst_tac [("m","m"),("n","n")] diff_induct 1);
by (SIMP_TAC arith_ss 1);
by (SIMP_TAC arith_ss 1);
by (ASM_SIMP_TAC arith_ss 1);
val less_imp_diff_is_0 = result();

val prems = goal Arith.thy "m-n = 0  -->  n-m = 0  -->  m=n";
by (res_inst_tac [("m","m"),("n","n")] diff_induct 1);
by (ALLGOALS (SIMP_TAC arith_ss));
val diffs0_imp_equal_lemma = result();

(*  [| m-n = 0;  n-m = 0 |] ==> m=n  *)
val diffs0_imp_equal = standard (diffs0_imp_equal_lemma RS mp RS mp);

val [prem] = goal Arith.thy "m<n ==> 0<n-m";
by (rtac (prem RS rev_mp) 1);
by (res_inst_tac [("m","m"),("n","n")] diff_induct 1);
by (SIMP_TAC arith_ss 1);
by (SIMP_TAC arith_ss 1);
by (ASM_SIMP_TAC arith_ss 1);
val less_imp_diff_positive = result();

val [prem] = goal Arith.thy "n < Suc(m) ==> Suc(m)-n = Suc(m-n)";
by (rtac (prem RS rev_mp) 1);
by (res_inst_tac [("m","m"),("n","n")] diff_induct 1);
by (SIMP_TAC arith_ss 1);
by (SIMP_TAC arith_ss 1);
by (ASM_SIMP_TAC arith_ss 1);
val Suc_diff_n = result();

goal Arith.thy "Suc(m)-n = if(m<n, 0, Suc(m-n))";
by (SIMP_CASE_TAC (arith_ss addrews [less_imp_diff_is_0, 
				     not_less_eq, Suc_diff_n]) 1);
val if_Suc_diff_n = result();

val ss1 = arith_ss addcongs [read_instantiate [("f","P")] arg_cong];

goal Arith.thy "P(k) --> (!n. P(Suc(n))--> P(n)) --> P(k-i)";
by (res_inst_tac [("m","k"),("n","i")] diff_induct 1);
by (ALLGOALS (strip_tac THEN' SIMP_TAC ss1));
by (fast_tac HOL_cs 1);
val zero_induct_lemma = result();

val prems = goal Arith.thy "[| P(k);  !!n. P(Suc(n)) ==> P(n) |] ==> P(0)";
by (rtac (diff_self_eq_0 RS subst) 1);
by (rtac (zero_induct_lemma RS mp RS mp) 1);
by (REPEAT (ares_tac ([impI,allI]@prems) 1));
val zero_induct = result();

(*13 July 1992: loaded in 105.7s*)

(**** Additional theorems about "less than" ****)

goal Arith.thy "n <= (m + n)";
by (nat_ind_tac "m" 1);
by (ALLGOALS (ASM_SIMP_TAC arith_ss));
by (rtac le_refl 1);
by (etac le_trans 1);
by (rtac (lessI RS less_imp_le) 1);
val le_add2 = result();

goal Arith.thy "m <= (m + n)";
by (stac add_commute 1);
br le_add2 1;
val le_add1 = result();

val less_add_Suc1 = standard (lessI RS (le_add1 RS le_less_trans));
val less_add_Suc2 = standard (lessI RS (le_add2 RS le_less_trans));

