;;; isa-thy-mode.el - Mode for Isabelle theory files.
;;;
;;; Author:  David Aspinall <da@dcs.ed.ac.uk>
;;;
;;; $Id: isa-thy-mode.el,v 1.4 1994/03/04 11:34:43 da Exp $
;;;


(require 'isa-load)
(require 'isa-menus)


;;; ========== Theory File Mode User Options ==========

(defvar isa-thy-heading-indent 0
  "*Indentation for section headings.")

(defvar isa-thy-indent-level 2
  "*Indentation level for Isabelle theory files.")

(defvar isa-thy-indent-factors
  '(("top" . 0) ("classes" . 2) ("types" . 2) ("arities" . 2))
  "*Multiples of isa-thy-indent-level to use in particular sections.")

(defvar isa-thy-rigid-backslash-left 0
  "*Controls the positioning of string-continuation backslashes.
If non-nil, it should be a column number which backslashes on the
continuation line will be indented to.
If nil, backslashes will be indented to appear under the
corresponding double-quote.")

(defvar isa-thy-rigid-backslash-right "   "
  "*Controls the positioning of string-continuation backslashes.
If non-nil, it may be a column number which backslashes on the
extended line will padded to or a string of white-space 
characters which will be inserted before the backslash.
If nil, backslashes will be appear at the end of the line
being extended.")

(defvar isa-thy-indent-strings t
  "*If non-nil, indent inside strings.
You may wish to disable indenting inside strings if your logic uses
any of the usual bracket characters in unusual ways.")

(defvar isa-thy-use-sml-mode t
  "*If non-nil, invoke sml-mode inside \"ML\" section of theory files.")



;;; ========== Code ==========


(defvar isa-thy-mode-map nil)

(defvar isa-thy-mode-syntax-table nil)		; Shared below.

(if isa-thy-mode-syntax-table	
    nil		
  ;; This is like sml-mode, except:
  ;;   _ ' are symbol-constituents, not word-constituents
  ;;   .   is a symbol-constituent
  ;;   "   is a paired delimiter
  (setq isa-thy-mode-syntax-table (make-syntax-table))
  (modify-syntax-entry ?\( "()1 " isa-thy-mode-syntax-table)
  (modify-syntax-entry ?\) ")(4 " isa-thy-mode-syntax-table)
  (modify-syntax-entry ?\\ "\\   " isa-thy-mode-syntax-table)
  (modify-syntax-entry ?*  ". 23" isa-thy-mode-syntax-table)
  (modify-syntax-entry ?_  "_   " isa-thy-mode-syntax-table)
  (modify-syntax-entry ?\' "_   " isa-thy-mode-syntax-table)
  (modify-syntax-entry ?\" "$   " isa-thy-mode-syntax-table)
  (modify-syntax-entry ?.  "_   " isa-thy-mode-syntax-table))

(or isa-thy-mode-map
    (let ((map (make-sparse-keymap)))
      (define-key map "\C-c\C-n" 'isa-thy-goto-next-section)
      (define-key map "\C-c\C-p" 'isa-thy-goto-previous-section)
      (define-key map '(control up) 'isa-thy-goto-previous-section)
      (define-key map '(control down) 'isa-thy-goto-next-section)
      (define-key map "\C-c\C-c" 'isa-thy-minor-sml-mode)
      (define-key map "\C-c\C-t" 'isa-thy-insert-template)
      (define-key map "\C-M" 'newline-and-indent)
      (define-key map "\C-k" 'isa-thy-kill-line)
      (setq isa-thy-mode-map map)))

;; This only works if you visit the theory file first.
;; (which is probably the right thing?)

(add-hook 'sml-mode-hook 'isa-menus)

(defun isa-thy-mode ()
  "Major mode for editing Isabelle theory files.
\\<isa-thy-mode-map>
\\[isa-thy-goto-next-section]\t Skips to the next section.
\\[isa-thy-goto-previous-section]\t Skips to the previous section.

\\[indent-for-tab-command]\t Indents the current line.

\\[isa-thy-insert-template]\t Inserts a template for the file or current section.

The indentation function automatically inserts and aligns backslashes in 
strings which split over multiple lines.

If isa-thy-use-sml-mode is non-nil, \\<isa-thy-mode-map>\\[isa-thy-minor-sml-mode] \
invokes sml-mode as a minor mode 
in the ML section.  This is done automatically by \
\\[indent-for-tab-command].

The style of indentation for theory files is controlled by these variables:
  isa-thy-heading-indent 
  isa-thy-indent-level
  isa-thy-indent-factors
  isa-thy-rigid-backslash-left
  isa-thy-rigid-backslash-right
  isa-thy-indent-string
- see individual variable documentation for details."
  (interactive)
  (kill-all-local-variables)
  (setq major-mode 'isa-thy-mode)
  (setq mode-name "Theory")
  (use-local-map isa-thy-mode-map)
  (isa-menus)				                ; Add "isabelle" menu.
  (set-syntax-table isa-thy-mode-syntax-table)
  (make-local-variable 'indent-line-function)
  (setq indent-line-function 'isa-thy-indent-line)
  (make-local-variable 'comment-start)			; Following lines as in sml-mode
  (setq comment-start "(* ")				; .
  (make-local-variable 'comment-end)			; .
  (setq comment-end " *)")				; .
  (setq comment-start-skip "(\\*+[ \t]?")		; .
  (force-mode-line-update)
  (run-hooks 'isa-thy-mode-hook))


(defvar isa-thy-minor-sml-mode-map nil)

(defun isa-thy-install-sml-mode ()
  (progn
    (require 'sml-mode)
    (setq isa-thy-minor-sml-mode-map (copy-keymap sml-mode-map))
    (define-key isa-thy-minor-sml-mode-map "\C-c\C-c" 'isa-thy-mode)
    (define-key isa-thy-minor-sml-mode-map "\t" 'isa-thy-do-sml-indent)))

(defun isa-thy-minor-sml-mode ()
  "Invoke sml-mode as if a minor mode inside a theory file.
This has no effect if isa-thy-use-sml-mode is nil."
  (interactive)
  (if isa-thy-use-sml-mode
      (progn
	(if (not (boundp 'sml-mode)) 
	    (isa-thy-install-sml-mode))
	(kill-all-local-variables)
	(sml-mode)					; Switch to sml-mode
	(setq major-mode 'isa-thy-mode)
	(setq mode-name "Theory Sml")			; looks like it's a minor-mode.
	(use-local-map isa-thy-minor-sml-mode-map)	; special map has \C-c\C-c binding.
	(force-mode-line-update)
	(message "Use C-c C-c to exit SML mode."))))

(defun isa-thy-do-sml-indent ()
  "Run sml-indent-line in an Isabelle theory file, provided inside ML section.
If not, will turn off simulated minor mode and run isa-thy-indent-line."
  (interactive)
  (if (string= (isa-thy-current-section) "ML")		; NB: Assumes that TAB key was 
      (sml-indent-line)					; bound to sml-indent-line.
    (isa-thy-mode)					; (at least, it is now!).
    (isa-thy-indent-line)))
    

(defconst isa-thy-sections
  ;; NB: preceding white space in templates deleted by indentation alg.
  ;;     top must come first.
  '(("top" . 
"S  =  T +\n
classes\n
default\n
types\n
arities\n
consts\n
translations\n
rules\n
end\n
ML\n")
    ("classes" . "<\t")
    ("default")
    ("types") 
    ("arities" . "::")
    ("consts"  . ":: \"\"\t\t\t()")
    ("translations" . "\"\"\t==\t\"\"")
    ("rules"   . "\"\"")
    ("end")
    ("ML"))
  "Names of theory file sections and their templates")

(defun isa-thy-insert-template ()
  "Insert a syntax template according to the current section"
  (interactive)
  (if (eq major-mode 'isa-thy-mode)
      (let ((start (point)) end)
	(let* ((sect (isa-thy-current-section))
	       (tmpl (cdr-safe (assoc sect isa-thy-sections))))
	  (if tmpl
	      (progn
		(save-excursion 
		  (insert tmpl)
		  (setq end (point)))
		(indent-region end start nil)))))
    (error "Not in Theory mode.")))

(defconst isa-thy-headings-regexp
  (substring 
   (apply 'concat
	   (mapcar '(lambda (pair)
		      (concat "\\|^\\s-*" (car pair) "[ \t]*"))
		   (cdr isa-thy-sections)))
   2))

(defun isa-thy-goto-next-section (&optional count noerror)
  "Goto the next (or COUNT'th next) section of a theory file.
Negative value for count means previous sections.
If NOERROR is non-nil, failed search will not be signalled."
  (interactive "p")
  (condition-case nil
      ;; string matching would probably be good enough
      (cond ((and count (< count 0))
	     (re-search-backward isa-thy-headings-regexp nil nil (- count))
	     t)
	    (t
	     (re-search-forward isa-thy-headings-regexp nil nil count)
	     t))
      (search-failed (if noerror nil
		       (error "No more headings")))))

(defun isa-thy-goto-previous-section (&optional count noerror)
  "Goto the previous section (or COUNT'th previous) of a theory file.
Negative value for count means following sections.
If NOERROR is non-nil, failed search will not be signalled."
  (interactive)
  (isa-thy-goto-next-section (if count (- count) -1) noerror))

(defun isa-thy-current-section ()
  "Return the current section of the theory file, as a string.
\"top\" indicates no section."
  (save-excursion
    (end-of-line)			
    (let* ((gotsect (isa-thy-goto-previous-section 1 t))
	   (start   (if gotsect
			(progn
			  (skip-chars-forward " \t")
			  (point)))))
      (if (not start)
	  "top"
	(skip-chars-forward "a-zA-z")
	(buffer-substring start (point))))))

	

(defun isa-thy-kill-line ()
  "As kill-line, except in a string will kill continuation backslashes also."
  (interactive)
  (if (not (isa-thy-comment-or-string-start))
      (kill-line nil)
    (let* ((pos    (point))
	   (eolpos (if (looking-at "[ \t]*$")		
		       ;; end of line bar whitespace
		       (save-excursion
			 (skip-chars-backward " \t")
			 (backward-char)
			 (if (looking-at "\\\\")
			     ;; backslash before point
			     (point)
			   pos))
		     (if (looking-at "[ \t]*\\\\[ \t]*$")
			 pos))))
      (if (not eolpos)
	  (kill-line nil)
	(goto-char eolpos)
	(kill-region eolpos
		     (progn
		       (forward-line 1)
		       (or (save-excursion
			     (skip-chars-forward " \t")
			     (if (looking-at "\\\\")
				 (1+ (point))))
			   (point))))
	(just-one-space)) ; seems nice
      )))

  
;;; Could do with isa-thy-correct-string function,
;;; which does roughly the same as indent-region.
;;; Then we could have an electric " that did this!

;;; Could perhaps have used fill-prefix to deal with backslash indenting,
;;; rather than lengthy code below?

(defun isa-thy-indent-line ()
  "Indent the current line in an Isabelle theory file.
If in the ML section, this switches into a simulated minor sml-mode."
  (let ((sect (isa-thy-current-section)))
    (cond
     ((and isa-thy-use-sml-mode (string= sect "ML"))
      (progn				               ; In "ML" section,
	(isa-thy-minor-sml-mode)	               ; switch to sml mode.
	(sml-indent-line)))

     (t   (let* ((indents   (isa-thy-calculate-indentation sect))
		 (first     (if (consp indents) (car indents) indents))
		 (second    (cdr-safe indents)))

	    (if (/= (current-indentation) first)	; indent before any backslash
		(save-excursion
		  (let ((beg (progn 
			       (beginning-of-line) 
			       (point))))
		    (skip-chars-forward "\t ")
		    (delete-region beg (point))
		    (indent-to first))))
	    (if second					; indent after backslash
		(save-excursion
		  (beginning-of-line)
		  (skip-chars-forward "\t ")
		  (forward-char)			; (skip backslash)
		  (let ((beg (point)))
		    (skip-chars-forward "\t ")
		    (delete-region beg (point))
		    (indent-to second))))

	    (if second					; move to correct column
		(if (< (current-column)
		       (save-excursion
			 (beginning-of-line)		; .. calculate post-backslash
			 (skip-chars-forward "\t ")	; indentation ..
			 (forward-char)
			 (skip-chars-forward "\t ") 	; <could make this bit more eff.>
			 (current-column)))
		    (progn
		      (beginning-of-line)
		      (skip-chars-forward "\t ")
		      (forward-char)
		      (skip-chars-forward "\t ")))
	      (if (< (current-column) 
		     (current-indentation))		
		  (skip-chars-forward "\t "))))))
	  ))

;; Better Emacs major modes achieve a kind of "transparency" to
;; the user, where special indentation,etc. happens under your feet, but
;; in a useful way that you soon get accustomed to.  Worse modes
;; cause frustration and repetitive re-editing of automatic indentation.
;; I hope I've achieved something in the first category...

(defun isa-thy-calculate-indentation (sect)
  "Calculate the indentation for the current line.
SECT should be the string name of the current section."
  (save-excursion
    (beginning-of-line)
    (or (isa-thy-long-comment-string-indentation)
	(if (looking-at "\\s-*$")
	    ;; Empty lines use indentation for section.
	    (isa-thy-indentation-for sect)
	  (progn
	    (skip-chars-forward "\t ")
	    (cond
	     ;; A comment?
	     ((looking-at "(\\*")         
	      (isa-thy-indentation-for sect))
	     ;; A section name?
	     ((looking-at isa-thy-headings-regexp)
	      isa-thy-heading-indent)
	     ;; Anything else, use indentation for section
	     (t (isa-thy-indentation-for sect))))))))

(defun isa-thy-long-comment-string-indentation ()
  "Calculate the indentation if inside multi-line comment or string."
  (let* ((comstr (isa-thy-comment-or-string-start))
	 (bolpos (save-excursion
		   (beginning-of-line)
		   (point)))
	 (multi  (and comstr 
		      (< comstr bolpos))))
    (if (not multi)
	nil		
      (save-excursion
	(beginning-of-line)
	(cond

	 ;; Inside a comment?
	 ((char-equal (char-after comstr) ?\( )
	  (forward-line -1)
	  (if (looking-at "[\t ]*(\\*")
	      (+ 3 (current-indentation))
	    (current-indentation)))
	 
	 ;; Otherwise, a string.
	 ;; Enforce correct backslashing on continuing
	 ;; line and return cons of backslash indentation
	 ;; and string contents indentation for continued
	 ;; line.
	 (t
	  (if (looking-at "[\t ]*\\\\") 
	      nil
	    ;; Missing backslash -
	    ;; add one in first column.
	    (insert "\\"))
	  (save-excursion
	    (forward-line -1)
	    (end-of-line)
	    (let ((eol (point)))
	      (skip-chars-backward "\t ")
	      (backward-char)
	      (if (looking-at "\\\\")
		  (progn
		    ;; delete space after backslash
		    (delete-region (1+ (point)) eol)
		    (let ((preb (point)))
		      (skip-chars-backward "\t ")
		      ;; delete space before backslash
		      (delete-region (point) preb))
		    (isa-thy-indent-right-backslash))
		;; Missing backslash - add one in correct place.
		(forward-char)
		(insert "\\")
		(backward-char)
		(isa-thy-indent-right-backslash))))
	  ;; Now process continued line
	  (let* ((quote-col (save-excursion (goto-char comstr) 
					    (current-column)))
		 (pre-back  (or isa-thy-rigid-backslash-left
				quote-col))
		 (post-back
		  (if isa-thy-indent-strings
		      (isa-thy-string-indentation comstr)
		    ;; just to right of matching " 
		    (+ quote-col 2))))
	    (cons pre-back post-back))))))))

(defun isa-thy-indent-right-backslash ()
  ;; Looking at backslash
  (cond ((stringp isa-thy-rigid-backslash-right)
	 (insert isa-thy-rigid-backslash-right))
	(isa-thy-rigid-backslash-right
	 (indent-to-column isa-thy-rigid-backslash-right))))


(defun isa-thy-string-indentation (start)
  ;; Guess indentation for text inside a string
  (let* ((startcol  (save-excursion (goto-char start) (current-column)))
	 (pps-state (parse-partial-sexp start (point)))
	 (par-depth (car pps-state)))
	 (cond (;; If not in nested expression, startcol+1
		(zerop par-depth)
		(1+ startcol))
;              Old version used same as previous line:
;	       (save-excursion
;		  (forward-line -1)
;		  (or (isa-thy-current-string-indentation)
;		      (1+ startcol))))
	       (;; If in a nested expression, use position of opening bracket
		(natnump par-depth)
		(save-excursion
		  (goto-char (nth 1 pps-state))
		  (+ (current-column)
		     (cond ((looking-at "\\[|") 3)
			   (t 1)))))
	       (;; Give error for too many closing parens
		t
		(error "Mismatched parentheses")))))

(defun isa-thy-current-string-indentation ()
  "post-backslash indentation of current line"
  (save-excursion
    (beginning-of-line)
    (skip-chars-forward " \t")
    (cond ((looking-at "\\\\")
	   (forward-char)
	   (skip-chars-forward " \t")
	   (current-column)))))

(defun isa-thy-indentation-for (sect)
  "Return the indentation for section SECT"
  (let ((fact (assoc sect isa-thy-indent-factors)))
    (if fact
      (* (cdr fact) isa-thy-indent-level)
      isa-thy-indent-level)))

(defun isa-thy-comment-or-string-start ()
  (isa-thy-comment-or-string-start-from
   (save-excursion
     (if (isa-thy-goto-previous-section 1 t)
	 (forward-word 1)
       (goto-char (point-min)))
     ;; Beginning of section or buffer
     (point))))

(defun isa-thy-comment-or-string-start-from (min)			
  "Find if point is in a comment or string.
Returns the position of the comment or string start or nil.
Starts parsing from min.

Doesn't understand nested comments, but then neither does the
theory file parser."
  (if (<= (point) min)
      nil
    (let ((pos   (point))
	  incom instring)
      (goto-char min)
      (while (< (point) pos)
	(cond (incom					; ignore strings inside comments
	       (if (looking-at "\\*)")
		   (setq incom nil)))
	      (instring
	       (if (looking-at "\"")
		   (setq instring nil)))
	      ((looking-at "(\\*")
	       (setq incom (point)))
	      ((looking-at "\"")
	       (setq instring (point))))
	(forward-char))
      (or incom instring))))






(provide 'isa-thy-mode)

;;; end of isa-thy-mode.el
