;;; isa-screen.el - Screen/frame manipulation functions.
;;;      
;;; Versions of functions provided in screen.el in Lucid Emacs
;;; distribution for application specific screen management.
;;; Eventually this file should die, and we should rely on functions
;;; in Lucid or FSF distribution.
;;;
;;; $Id: isa-screen.el,v 1.4 1994/02/19 16:12:28 da Exp $
;;;

(require 'isa-load)

(defvar get-screen-for-buffer-default-screen-name nil
  "The default screen to select; see doc of `get-screen-for-buffer'.")

(defun get-screen-name-for-buffer (buffer)
  (let ((mode (save-excursion (set-buffer buffer) major-mode)))
    (or (get mode 'screen-name)
	get-screen-for-buffer-default-screen-name)))

;;; New versions of functions in screen.el from jwz 5.2.94

(defun get-screen-for-buffer-noselect (buffer
				       &optional not-this-window-p on-screen)
    "Return a screen in which to display BUFFER.
This is a subroutine of `get-screen-for-buffer' (which see.)"
  (if (or on-screen (eq (selected-window) (minibuffer-window)))
      ;; don't switch screens if a screen was specified, or to list
      ;; completions from the minibuffer, etc.
      nil
    ;; else
  (let ((name (get-screen-name-for-buffer buffer)))
    (if (null name)
	(selected-screen)
      (let ((limit (get name 'instance-limit))
	    (defaults (get name 'screen-defaults))
	    (matching-screens '())
	    (screens (screen-list))
	    screen already-visible)
	;; Sort the list so that iconic screens will be found last.  They
	;; will be used too, but mapped screens take prescedence.  And
	;; fully visible screens come before occluded screens.
	(setq screens
	      (sort screens
		    (function (lambda (s1 s2)
				(cond ((screen-totally-visible-p s2)
				       nil)
				      ((not (screen-visible-p s2))
				       (screen-visible-p s1))
				      ((not (screen-totally-visible-p s2))
				       (and (screen-visible-p s1)
					    (screen-totally-visible-p s1)
					    )))))))
	;; but the selected screen should come first, even if it's occluded,
	;; to minimize thrashing.
	(setq screens (cons (selected-screen)
			    (delq (selected-screen) screens)))

	(setq name (symbol-name name))
	(while screens
	  (setq screen (car screens))
	  (if (equal name (screen-name screen))
	      (if (get-buffer-window buffer screen)
		  (setq already-visible screen
			screens nil)
		(setq matching-screens (cons screen matching-screens))))
	  (setq screens (cdr screens)))
	(cond (already-visible
	       already-visible)
	      ((or (null matching-screens)
		   (eq limit 0) ; means create with reckless abandon
		   (and limit (< (length matching-screens) limit)))
	       (let* ((sc (funcall screen-creation-func
				   (cons (cons 'name name)
					 (append defaults
						 screen-default-alist))))
		      (w (screen-root-window sc)))
		 ;;
		 ;; Make the one buffer being displayed in this newly created
		 ;; screen be the buffer of interest, instead of something
		 ;; random, so that it won't be shown in two-window mode.
		 ;; Avoid calling switch-to-buffer here, since that's something
		 ;; people might want to call this routine from.
		 ;;
		 ;; (If the root window doesn't have a buffer, then that means
		 ;; there is more than one window on the screen, which can only
		 ;; happen if the user has done something funny on the screen-
		 ;; creation-hook.  If that's the case, leave it alone.)
		 ;;
		 (if (window-buffer w)
		     (set-window-buffer w buffer))

		 ;; delay for screen to appear and enter screen list
		 (sit-for 0)

		 sc))
	      (t
	       ;; do not switch any of the window/buffer associations in an
	       ;; existing screen; this function only picks a screen; the
	       ;; determination of which windows on it get reused is up to
	       ;; display-buffer itself.
;;	       (or (window-dedicated-p (selected-window))
;;		   (switch-to-buffer buffer))
	       (car matching-screens))))))))

;; The pre-display-buffer-function is called for effect, so this needs to
;; actually select the screen it wants.  Fdisplay_buffer() takes notice of
;; changes to the selected screen.
(defun get-screen-for-buffer (buffer &optional not-this-window-p on-screen)
  "Select and return a screen in which to display BUFFER.
Normally, the buffer will simply be displayed in the current screen.
But if the symbol naming the major-mode of the buffer has a 'screen-name
property (which should be a symbol), then the buffer will be displayed in
a screen of that name.  If there is no screen of that name, then one is
created.  

If the major-mode doesn't have a 'screen-name property, then the screen
named by `get-screen-for-buffer-default-screen-name' will be used.  If
that is nil (the default) then the currently selected screen will used.

If the screen-name symbol has an 'instance-limit property (an integer)
then each time a buffer of the mode in question is displayed, a new screen
with that name will be created, until there are `instance-limit' of them.
If instance-limit is 0, then a new screen will be created each time.

If a buffer is already displayed in a screen, then `instance-limit' is 
ignored, and that screen is used.

If the screen-name symbol has a 'screen-defaults property, then that is
prepended to the `screen-default-alist' when creating a screen for the
first time.

This function may be used as the value of `pre-display-buffer-function', 
to cause the display-buffer function and its callers to exhibit the above
behavior."
  (let ((screen (get-screen-for-buffer-noselect
		 buffer not-this-window-p on-screen)))
    (if (null screen)
	nil
      (select-screen screen)
      (make-screen-visible screen) ; uniconify, raise, etc
      screen)))



(provide 'isa-screen)

;;; End of isa-screen.el


