;;; isa-mode.el - Isabelle interaction mode.
;;; 
;;;
;;; Author:  David Aspinall <da@dcs.ed.ac.uk>
;;;
;;; $Id: isa-mode.el,v 1.13 1994/02/23 19:02:21 da Exp $
;;;


;;; DESIRABLE CHANGES:
;;;   - output filtering to use latest comint, much simplification
;;;     here.
;;;   - functions to alter isa-denoted-subgoal, and message it.


(require 'isa-load)
(require 'isa-proc)
(require 'isa-ml-comp)
(require 'isa-menus)
(require 'isa-display)			; 
(require 'isa-thy-mode)			; for isa-thy-mode-syntax-table
					; (should be elsewhere?)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; isa-mode  
;;
;;;
;;; local variables
;;;
;;; isa-input-sentinel          <see isa-proc>
;;; isa-weed-output-function	.
;;; isa-output-buf		.
;;; isa-last-trace-start	.
;;; isa-trace-count		.
;;; isa-denoted-subgoal		.
;;; isa-logic-name              <see below>
;;; isa-associated-buffers
;;;
;;; <plus various comint variables, kill-buffer-hook>
;;;
;;; hooks
;;;
;;; isa-mode-hook
;;; isa-mode-startup-hook
;;;

;; These variables are only used buffer locally and so do not
;; need global definitions.  (The byte compiler will moan,tho').

;(defvar isa-associated-buffers nil
;  "List of buffers associated with current Isabelle buffer.")
;
;(defvar isa-logic-name ""
;  "Name of logic for current Isabelle buffer.")
;
;(defvar isa-output-buf nil
;  "Hidden output buffer associated with current Isabelle buffer.")




;;;
;;; Mode for Isabelle interaction buffers: isa-mode
;;;

;;; NB: isa-mode-map defined in isa-menus.el

(defun isa-mode ()
  "Major mode for interaction with Isabelle.

These commands activate/display further buffers:
\\<isa-mode-map>
 \\[listener]\t\t- Listener buffer 
 \\[proofstate]\t\t- Proof State buffer
 \\[ruletable]\t\t- a Rule Table for a given theory
 \\[isa-ruletable-for-logic]\t\t- a Rule Table for this logic

Listener and ProofState are activated automatically if
buffers or screens with the default names exist.

This mode is built on top of comint-mode, and most of the comint-mode
key bindings are available.  There are many additional functions for
inserting common Isabelle commands.

Key bindings:
\\{isa-mode-map}
Entry to this mode runs isa-mode-hook."
  ;; comint customisation...
  (comint-mode)						; Based on comint
  (setq comint-prompt-regexp isa-prompt-pattern)	
  (setq comint-input-sentinel 'isa-run-input-sentinel)
  (make-local-variable 'comint-file-completion-suffix)
  (setq comint-file-completion-suffix "\"")             ; " terminates files.
  (setq comint-input-autoexpand nil)	                ; no history expansion.
  (make-local-variable 'comint-dynamic-complete-command-command)
  (setq comint-dynamic-complete-command-command 'isa-complete-command)
  (make-local-variable comint-after-partial-filename-command)
  (setq comint-after-partial-filename-command 'isa-after-filename)


  (setq major-mode 'isa-mode)				; 
  (setq mode-name "Isabelle")				; 
  (use-local-map isa-mode-map)
  (set-syntax-table isa-thy-mode-syntax-table)		; shares with isa-thy-mode
  (make-local-variable 'isa-weed-output-function)	; real output filter
  (setq isa-weed-output-function 
	'isa-weed-output-default)                       ; default function.
  (make-local-variable 'isa-output-buf)			; output accumulator buffer
  (make-local-variable 'isa-input-sentinel)		
  (make-local-variable 'isa-denoted-subgoal)
  (setq isa-denoted-subgoal 1)
  (make-local-variable 'isa-associated-buffers)
  (setq isa-associated-buffers nil)	                ; no linked buffers yet.
  (make-local-variable 'isa-logic-name)			
  (make-local-variable 'kill-buffer-hook)
  (add-hook 'kill-buffer-hook 
	    'isa-kill-associated-buffers)		; Killing Isabelle buffer
							; kills associated ones too.
  (isa-mode-process-init)
  (set-buffer-menubar isa-default-menubar)		; Base menu for Isabelle Mode
  (isa-add-main-menu)					; .. with "Isabelle" menu
  (isa-add-interaction-menus)				;   .. and others.
  (run-hooks 'isa-mode-hook))				; User hooks.



(defun isa-kill-associated-buffers ()
  "Kill buffers associated with an Isabelle buffer.
Intended as a value for kill-buffer-hook"
  (let ((bufs-to-go (copy-sequence isa-associated-buffers)))
    (mapcar
     '(lambda (b)
	(if (isa-buffer-active b)
	    (kill-buffer b)))
     bufs-to-go))
  (if isa-multiple-screen-mode
      (isa-kill-associated-screens)
    (delete-other-windows)))

(defun isa-kill-associated-screens ()
  "If there are redundant Isabelle related screens, remove them."
  ;; bit daft at the moment: better to map over *screens*
  ;; and use real screen name symbol.  Change var. appropriately.
  (mapcar
   '(lambda (mode)
      (let* ((sn   (get mode 'screen-name))
	     (scs  (isa-find-screens sn))
	     (bfs  (if scs (isa-find-buffers-in-mode mode)))
	     (fbf (car-safe bfs))
	     (rbf (cdr-safe bfs)))
	(cond ((and bfs (not (and (null rbf) (eq fbf (current-buffer)))))
	       (display-buffer fbf nil (car scs))
	       (setq scs (cdr scs))))
	(if scs
	    (condition-case nil
		(mapcar 'delete-screen scs)
	      (error nil)))))
   isa-associated-screen-names))


;;;
;;; Start-up an Isabelle Interaction Buffer.
;;;

(defun isabelle (logic)
  "Start or switch to an Isabelle session in buffer *LOGIC*
LOGIC should be one of the built-in Isabelle logics,
(amongst isa-builtin-object-logic-names), a logic in
the current directory, or a path name to a logic.

There is no limit on the number of Isabelle sessions that you may
activate: to have another session with the same logic, simply rename
the Isabelle buffer.

See the mode documentation for isa-mode for the commands
available in Isabelle buffers."
  (interactive 
   (list (completing-read "Name of logic: " isa-builtin-object-logic-names)))
  (let* ((logic-short-name  (file-name-nondirectory logic))
	 (base-name          logic-short-name)       ; did say "Isabelle-LOGIC" 
	 (buff-name         (isa-name base-name))
         (buffer            (get-buffer-create buff-name))
	 (coml              (isa-command-list logic)))
    (if (comint-check-proc buffer)			; If running, select
        (progn					        ; it and associated
          (isa-select-buffer buffer t)			;  buffers; then exit.
	  (isa-mode-process-init)			; (set process redirection)
	  (mapcar 'isa-display-if-active isa-associated-buffers))
      (progn	    
	(message "Starting Isabelle...")		;
        (set-buffer buffer)
	(erase-buffer)					; Clear buffer
	(sit-for 0)					; Update display
	(apply 'make-comint base-name (car coml)	; Start comint process
	                  (cons nil (cdr coml)))	; running Isabelle
	(isa-mode)					; Switch to Isabelle mode
	(isa-select-buffer buffer)			; Select buffer.
	(setq isa-logic-name logic-short-name)		; 

	(while (zerop (+ (buffer-size)
		       (save-excursion			;  Wait for process to get 
		        (set-buffer isa-output-buf)	; going...
		         (buffer-size))))		;
	    (sit-for 1)					; 
	    (accept-process-output))			; 
       
        (run-hooks 'isa-mode-startup-hook)
        (isa-synchronise-wd)
	(message "Starting Isabelle...done.")))))


(defun isa-startup-function-for (sym)
  "Used to make functions for isa-mode-startup-hook."
    (if (or (memq sym isa-startup-defaults)
	    (isa-find-screens sym)
	    (get-buffer (isa-buffer-name-for sym)))
	(funcall sym)))

(defun isa-find-associated (sym)
  (let* ((modesym (intern (concat (symbol-name sym) "-mode"))))
    (isa-find-buffers-in-mode modesym 
			      isa-associated-buffers)))
  
(defun isa-create-new-associated (sym &optional nocheck)
  "Find or create a new associated buffer for SYM."
  (if (or nocheck
	  (null (isa-find-associated sym)))
      (let ((newbuf (if nocheck
			(generate-new-buffer (isa-buffer-name-for sym))
		      (get-buffer-create (isa-buffer-name-for sym))))
	    (curbuf (current-buffer)))
	(setq isa-associated-buffers
	      (cons newbuf isa-associated-buffers))
	(set-buffer newbuf)
	(kill-all-local-variables)
	(make-local-variable 'isa-buffer) 
	(setq isa-buffer curbuf)			; linked isabelle buffer
	(make-local-variable 'kill-buffer-hook)
	(add-hook 'kill-buffer-hook 'isa-remove-associated-buffer)
	(set-buffer curbuf)
	newbuf)
    (car (isa-find-associated sym))))

(defun isa-remove-associated-buffer ()
  "Remove current buffer from isa-associated-buffers"
  (if (isa-buffer-active isa-buffer)
      (let ((rbuf (current-buffer)))
	(save-excursion
	  (set-buffer isa-buffer)
	  (setq isa-associated-buffers
		(delq rbuf isa-associated-buffers))))))


(defun isa-command-list (logic)
  "Return the command,arg list to start Isabelle for LOGIC.
If LOGIC matches one of isa-builtin-object-logic-names, then a builtin
logic will be selected using isa-builtin-logic-path.  Otherwise,
the logic is to be found either in the current directory or in
the directory specified by isa-user-logic-path.
To force a user logic with the same name as a built-in one, you
must give a path in the logic name."
  (let* ((builtin      (assoc logic isa-builtin-object-logic-names))
	 (exists-here  (file-exists-p logic))
	 (filename     
	  (cond (builtin     (concat isa-builtin-logic-path logic))
		(exists-here logic)
		(t           (concat isa-user-logic-path logic))))
	 (exists       (file-exists-p filename))
	 (fullname     (expand-file-name filename)))
    (if exists
	(funcall isa-run-command-function fullname)
      (error "Can't find logic %s" fullname))))


;;; ============ Completion ============

(defun isa-complete-command ()
  "Complete the ML value after point, use isa-completion-list."
  (interactive)
  (if (isa-in-string)
      nil  ; do nothing inside strings if filename expansion failed.
    (let ((stub (buffer-substring
		 (point)
		 (save-excursion
		   (skip-chars-backward "[a-zA-Z_.'0-9]")
		   (point)))))
      (if (string= "" stub)
	  nil
	(comint-dynamic-simple-complete stub isa-completion-list)))))

(defun isa-after-filename ()
  "Non-nil if point is in a string that looks like a filename so far."
  (let ((p (point)))
    (save-excursion
      (skip-chars-backward "^\"" (save-excursion
					(comint-bol nil) (point)))

      (backward-char 1)
      (if (looking-at "\"[~/A-Za-z0-9+@:_.$#,={}-]+")
	  (>= (match-end 0) p)))))

(defun isa-in-string ()
  "Non-nil if point is between (comint-bol) and a non-even number of \"'s."
  (let ((p (point)) (instring nil))
    (save-excursion
      (comint-bol nil)
      (while (< (point) p)
	(skip-chars-forward "^\"" p)
	(if (eq (char-after (point)) ?\")
	    (progn
	      (setq instring (not instring))
	      (forward-char 1)))))
    instring))



(provide 'isa-mode)

;;; End of isa-mode.el

