;;; isa-menus.el - Menus for Isabelle mode.
;;;
;;; Author:  David Aspinall <da@dcs.ed.ac.uk>
;;;
;;; $Id: isa-menus.el,v 1.12 1994/03/04 11:24:34 da Exp $
;;;

(require 'isa-load)
(require 'isa-easymenu)
(require 'isa-rules)			; for object logic names
(require 'isa-cmd)			; for most menu functions


;;; ============ Isabelle mode map ============

;; appears here because of FSF menus-in-keymap system.

(defconst isa-mode-map
  (let
      ((map (copy-keymap comint-mode-map))
       (do-define-key 
	'(lambda (table-entry)
	   (if (or (null table-entry) 
		   (null (car table-entry)))
	       nil
	     (if (null (nth 1 table-entry))
		 (define-key map (car table-entry) (nth 3 table-entry))
	       (define-key map (car table-entry) (nth 1 table-entry)))))))
; changed to copy-keymap for v18/FSF compat.
;   (set-keymap-parent map comint-mode-map)
    (set-keymap-name map 'isa-mode-map)
    (isa-define-popup-key map 'button3           'isa-popup-tactic-menu)
    (isa-define-popup-key map '(shift button3)   'isa-popup-goal-menu)
    (isa-define-popup-key map '(control button3) 'isa-popup-option-menu)
    (isa-define-popup-key map '(meta button3)    'isa-popup-prover-menu)
    (define-key map '(control down) 'isa-inc-denoted-subgoal)
    (define-key map '(control up) 'isa-dec-denoted-subgoal)
    (define-key map "\M-\t"     'comint-dynamic-complete)
    (define-key map "\t"        'comint-dynamic-complete)
    (define-key map "\M-?"      'comint-dynamic-list-completions)
    (define-key map "\C-c\C-c"  'isa-interrupt)
    (mapcar '(lambda (pair)
	       (mapcar do-define-key (cdr pair))) isa-commands-table)
    map))

;;; ============ Interaction Mode Menus ============

(defun isa-generate-interaction-menus ()
  "Make menus from isa-commands-table (see isa-cmd.el)."
  (mapcar '(lambda (pair)
	     (cons (car pair)
		   (mapcar 'isa-make-menu-entry (cdr pair))))
	  isa-commands-table))
 
(defun isa-make-menu-entry (table-entry)
  (if table-entry
      (let* ((command (or (nth 1 table-entry)
			  (nth 3 table-entry)))
	     (name    (nth 2 table-entry)))
	(vconcat (list name command t)))
    "----"))

(defun isa-apply-easy-menu-define (sym menu)
  (eval
   (` (easy-menu-define (, sym) 
			(list isa-mode-map)
			"Menu used in Isabelle mode."
			'(, menu))))
   (if (not (boundp sym))
       (set sym nil)))

(defconst isa-interaction-menus
  (mapcar
   '(lambda (m)
      (let ((menu-name (intern
			(concat "isa-" (car m) "-menu"))))
	(isa-apply-easy-menu-define menu-name m)
	menu-name))
   (isa-generate-interaction-menus)))

(defun isa-apply-easy-menu-add (menu)
  (easy-menu-add menu))

(defun isa-add-interaction-menus () 
  "Add Isabelle interaction-mode menus to the menubar."
  (mapcar 'isa-apply-easy-menu-add isa-interaction-menus))

(setq isa-interaction-menus (isa-generate-interaction-menus))


;;; ============== Main "Isabelle" Menu. ==============

(defconst isa-startup-menu-items
  (let ((vc '(lambda (nm)	
	       (vector (car nm) (list 'isabelle (car nm)) t))))
    (append
     (mapcar vc isa-builtin-object-logic-names)
     '(["Named logic..."  isabelle t]))))

(defconst isa-help-menu-items
  '(["Introduction to Isabelle"  (isa-view-man "intro")  t]
    ["Reference Manual"          (isa-view-man "ref")    t]
    ["Logics Manual"             (isa-view-man "logics") t]))


(defconst isa-edit-menu-items
  '(["Theory file template"   isa-thy-insert-template t]
    "----"
    ["Batchify proof"         isa-batchify t]
    ["Unbatchify proof"       isa-unbatchify t]))

(defconst isa-main-menu-items
  (list
   (cons "Edit"     isa-edit-menu-items)
;   "----"
   (cons "Session"  isa-startup-menu-items)
;   "----"
   (cons "Manuals"  isa-help-menu-items)))

(easy-menu-define isa-main-menu
		  (list isa-mode-map)
		  "Isabelle main menu"
		  (cons "Isabelle" isa-main-menu-items))


(defun isa-add-main-menu ()
  (easy-menu-add isa-main-menu))



;;; ========== Main menu functions ==========

(defun isa-view-man (name)
  "View isabelle manual NAME."
  (interactive "sWhich manual? (choose intro,ref,logics) ")
  (start-process 
   "isa-view-man" nil 
   isa-view-man-command (concat isa-manual-path name)))



;;; ========== Menu Pop-up functions ==========

(defun isa-popup-tactic-menu ()
  (interactive)
  (popup-menu (assoc "Tactic" isa-interaction-menus)))

(defun isa-popup-goal-menu ()
  (interactive)
  (popup-menu (assoc "Goal" isa-interaction-menus)))

(defun isa-popup-option-menu ()
  (interactive)
  (popup-menu (assoc "Option" isa-interaction-menus)))

(defun isa-popup-prover-menu ()
  (interactive)
  (popup-menu (assoc "Prover" isa-interaction-menus)))



;;; ###autoload
(defun isa-menus ()	
  "Add main Isabelle menu to current menubar."
  (interactive)
  (easy-menu-define isa-main-menu
		    (list (current-local-map))
		    "Isabelle main menu"
		    (cons "Isabelle" isa-main-menu-items))
  (easy-menu-add isa-main-menu))


(provide 'isa-menus)

;;; End of isa-menus.el
