(* -------------------------------------------------------------*)
(*		Syntax.ml					*)
(*								*)
(*		Isabelle syntax for a subset of SML		*)
(*		SML is essentially the pure functional		*)
(*		subset of the Core Language of Standard ML	*)
(* ------------------------------------------------------------ *)


signature SML_SYNTAX =
sig
  val ntypes: (string list * (string list * string)) list
  val sext : Syntax.sext
  val thy : theory
  val show_full_syntax : bool ref
end;

functor SML_SyntaxFun () : SML_SYNTAX = 
struct
local open Syntax 
in

val special_constants =
            ["Scon"] 		(* special constants      *)

val identifiers =
            ["Ide",		(* the general class      *)
	     "Var", 		(* value variables        *)
	     "Con", 		(* value constructors     *)
	     "ExCon", 		(* exception constructors *)
	     "TyVar",		(* type variables         *)
	     "TyCon",		(* type constructors      *)
             "Lab",             (* record labels          *)
             "StrId"];          (* structure identifiers  *)

val phrase_classes =
             ["ExpRow",         (* expression rows       *)
              "Exp",            (* expressions           *)
	      "Match",          (* matches               *)
	      "Mrule",          (* match rules           *)
              "Dec",            (* declarations          *)
              "ValBind",        (* value bindings        *)
              "TypBind",        (* type bindings         *)
              "DatBind",        (* datatype bindings     *)
              "ConBind",        (* constructor bindings  *)
              "ExBind",         (* exception bindings    *)
              "PatRow",         (* pattern rows          *)
              "Pat",            (* patterns              *)
              "Ty",             (* type expressions      *)
              "TyRow",          (* ty-expression rows    *)
              "Program"];       (* programs              *)

val auxiliary_classes = ["Nat"];

val seq_classes = 
             ["ExpSeq",		(* expression sequences *)
              "TyVarSeq",	(* type variable sequences *)
              "TyVarArgs",	(* type variable arguments *)
              "TySeq",          (* type sequences        *)
              "TyArgs"]          (* type arguments        *)

val types = special_constants @
            identifiers @ 
            phrase_classes @ 
            auxiliary_classes @ 
            seq_classes ;

(* -------------------------------------------------------------------- *)
(*	    The Bare Language Syntax for the Core			*)
(*	taken directly from the Definition of SML. pp 8-9		*)
(*	Note that it omits the following:				*)
(*		open declarations					*)
(*		fixity directives, and					*)
(*		"op" qualifiers						*)
(* -------------------------------------------------------------------- *)

val bare_mixfix = 

 [  (* type variables *)

     Delimfix ("''_",  "Ide => TyVar", "TyVar"),

    (* atexp *)

     Delimfix ("(scon _)",  "Nat => Exp", "Scon"),
     Delimfix ("(var _)", "Var => Exp", "Var"),
     Delimfix ("(con _)", "Con => Exp", "Con"),
     Delimfix ("(excon _)", "ExCon => Exp", "ExCon"),
     Delimfix ("{}", "Exp", "Empty_Record"),
     Delimfix ("{_}", "ExpRow => Exp", "Record"),
     Delimfix ("(4let _ )/ (3in/ _) end", "[ Dec,  Exp] =>   Exp", "Let"),
     Delimfix ("('(- _ -'))", "Exp => Exp", "Bracket"), (* chicane ! *)

    (*  exprow *)

     Delimfix ("_ = _", "[Lab, Exp] => ExpRow", "ExpRow1"),
     Delimfix ("_ = _ , _", "[Lab, Exp, ExpRow] => ExpRow", "ExpRow2"),

    (*  exp *)

     Delimfix ("(1_ `/ _)","[Exp, Exp] => Exp", "Apply"),  
     Delimfix ("(1_  _/  _)", "[Exp, Var, Exp] => Exp", "Infix_Apply"),  
     Delimfix ("(1_ : _)", "[Exp, Ty] => Exp", "Typed"), 
     Delimfix ("(1_ handle _)", "[Exp, Match] => Exp", "Handle"), 
     Delimfix ("(1raise _)", "Exp => Exp", "Raise"), 
     Delimfix ("(1fn _)", "Match => Exp", "Fn"), 
 
    (*  match *)

     Delimfix ("(_ =>/ _)"," [ Pat, Exp] =>  Match", "Mrule"),  
     Delimfix ("(_ =>/ _ |/ _)", "[Pat, Exp, Match] =>  Match", "Match"),  

    (* dec *)

     Delimfix ("(1val _)", "ValBind => Dec", "Val"),
     Delimfix ("(1type _ )", "TypBind => Dec", "Type"),
     Delimfix ("(1datatype _ )", "DatBind => Dec", "DataType"),
     Delimfix ("(1abstype _ with _ end)", "[DatBind, Dec] => Dec", "AbsType"),
     Delimfix ("(1exception _ )", "ExBind => Dec", "Exception"),
     Delimfix ("(6local _)/ (6in/ _ end)", "[Dec, Dec] => Dec", "Local"),
       (* NB:  'open' declaration omitted for now  *)
     Delimfix ("nothing ", "Dec", "Nothing"),
     Delimfix ("(_/ _)", "[Dec, Dec] => Dec", "Comp1"),
     Delimfix ("(_;/ _)", "[Dec, Dec] => Dec", "Comp2"),
       (* NB:   fixity directives omitted for now  *)


    (* valbind *)

     Delimfix ("(1_ =/ _)","[Pat,  Exp] => ValBind", "ValBind1"),
     Delimfix ("(1_ =/ _ and/ _)", 
          "[Pat, Exp, ValBind] => ValBind", "ValBind2"),
     Delimfix ("(1rec _ )", "ValBind => ValBind", "Rec"),


    (* typbind *)

     Delimfix ("_ _ is _", "[TyVarSeq, TyCon, Ty] => TypBind", "TypBind1"), 
     Delimfix ("_ _ is _ and _",
        "[TyVarSeq, TyCon, Ty, TypBind] => TypBind", "TypBind2"), 

    (* datbind *)

     Delimfix ("_ _ = _", "[TyVarSeq, TyCon, ConBind] => DatBind", "DatBind1"), 
     Delimfix ("_ _ = _ and _",
        "[TyVarSeq, TyCon, ConBind, DatBind] => DatBind", "DatBind2"), 

    (* conbind *)

     Delimfix ("con _", "Con => ConBind", "ConBind1"),
     Delimfix ("con _ of _", "[Con, Ty] => ConBind", "ConBind2"),
     Delimfix ("con _ | _", "[Con, ConBind] => ConBind", "ConBind3"),
     Delimfix ("con _ of _ | _", "[Con, Ty, ConBind] => ConBind", "ConBind4"),

    (* exbind *)

     Delimfix ("excon _", "ExCon => ExBind", "ExBind1"),
     Delimfix ("excon _ of _", "[ExCon, Ty] => ExBind", "ExBind2"),
     Delimfix ("excon _ and _", "[ExCon, ExBind] => ExBind", "ExBind3"),
     Delimfix ("excon _ of _ and _", 
             "[ExCon, Ty, ExBind] => ExBind", "ExBind4"),
     Delimfix ("excon _ is _", "[ExCon, ExCon] => ExBind", "ExBind5"),
     Delimfix ("excon _ is _ and _",
           "[ExCon, ExCon, ExBind] => ExBind", "ExBind6"),

    (* atpat *)

     Delimfix ("'_", "Pat", "Wild_Pat"),
     Delimfix ("scon _", "Nat => Pat", "Scon_Pat"),
     Delimfix ("var _", "Var => Pat", "Var_Pat"),
     Delimfix ("con _", "Con => Pat", "Con_Pat"),
     Delimfix ("excon _", "ExCon =>  Pat", "ExCon_Pat"),
     Delimfix ("{}", "Pat", "Empty_Record_Pat"),
     Delimfix ("{_}", "PatRow => Pat", "Record_Pat"),
     Delimfix ("('(- _ -'))", "Pat => Pat", "Bracket_Pat"), 

    (* patrow *)

     Delimfix ("...", "PatRow", "Wild_PatRow"),
     Delimfix ("_ = _", "[Lab, Pat] => PatRow", "PatRow1"),
     Delimfix ("_ = _, _", "[Lab, Pat, PatRow] => PatRow", "PatRow2"),

    (* pat *)

     Delimfix ("con _`_", "[Con, Pat] => Pat", "Con_Val_Pat"),
     Delimfix ("excon _`_", "[ExCon, Pat] => Pat", "ExCon_Val_Pat"),
     Delimfix ("_ con _ _", "[Pat, Con, Pat] => Pat", "Infix_Con_Val_Pat"),
     Delimfix ("_ excon _ _", 
        "[Pat, ExCon, Pat] => Pat", "Infix_ExCon_Val_Pat"),
     Delimfix ("_ : _", "[Pat, Ty] => Pat", "Typed_Pat"),
     Delimfix ("var _ as _", "[Var, Pat] => Pat", "Layered_Pat"),
     Delimfix ("var _ : _ as _", "[Var, Ty, Pat] => Pat", "Typed_Layered_Pat"),

    (* ty *)

     Delimfix ("_", "TyVar =>  Ty", "TyVar_Ty"),
     Delimfix ("{}", "Ty", "Empty_Record_Ty"),
     Delimfix ("{_}", "TyRow =>  Ty", "Record_Ty"),
     Delimfix ("_ _", "[TySeq, TyCon]  =>  Ty", "TyCon"),
     Delimfix ("_ -> _", "[Ty, Ty] => Ty", "Fn_Ty"), 
     Delimfix ("'(- _ -')", "Ty => Ty", "Bracket_Ty"), 


    (* tyrow *)

     Delimfix ("_ = _"," [ Lab,  Ty] =>  TyRow ", "TyRow1"),
     Delimfix ("_ = _ , _"," [ Lab,  Ty,  TyRow] =>  TyRow ", "TyRow2"),

    (* program *)
      
     Delimfix ("_ ;","  Dec =>  Program", "Program1"),
     Delimfix ("_ ; _"," [ Dec,  Program] =>  Program", "Program2")];


(* -------------------------------------------------------------------- *)
(*		 Derived Forms	 					*)
(*	   (see parse and print translations below)			*)
(* -------------------------------------------------------------------- *)
 
val derived_mixfix =

 [  (* exp *)

     Delimfix ("(1'(-_,/ _-'))"," [ Exp,  Exp] => Exp", " Pr"),
     Delimfix ("(1if _/ then _/ else _)", "[Exp, Exp, Exp] =>   Exp", " If"),
     Delimfix ("(1case _/ of _)", "[ Exp, Match] => Exp", " Case"),
     Delimfix ("(1_ orelse _)",  "[Exp,  Exp] => Exp", " Orelse"),
     Delimfix ("(1_ andalso _)", "[Exp, Exp] => Exp", " Andalso"),
     Delimfix ("(1_ +/ _)"," [ Exp,  Exp] =>   Exp", " Plus"),
     Delimfix ("(1_ -/ _)"," [ Exp,  Exp] =>   Exp", " Minus"),
     Delimfix ("(1_ */ _)"," [ Exp,  Exp] =>   Exp", " Times"),
     Delimfix ("(1_ </ _)"," [ Exp,  Exp] =>   Exp", " Less"),
     Delimfix ("(1zero _)","  Exp =>   Exp", " Zero"),
     Delimfix ("(1not _)","  Exp =>   Exp", " Knot"),
     Delimfix ("[]","  Exp", " Nil"),
     Delimfix ("(1_ ::/ _)"," [ Exp,  Exp] =>   Exp", " Cons"),
     Delimfix ("[_]","  ExpSeq =>  Exp", " List"),
     Delimfix ("(1head _)","  Exp =>   Exp", " Head"),
     Delimfix ("(1tail _)","  Exp =>   Exp", " Tail"),
     Delimfix ("(1null _)","  Exp =>   Exp", " Null"),
     Delimfix ("<>", "TyVarSeq", " tyvarempty"),
     Delimfix ("<_>", "TyVarArgs =>  TyVarSeq", " TyVarList"),

   (* pat *)

     Delimfix ("'(-_,_-')"," [ Pat,  Pat] =>  Pat", " Pr_Pat"),

    (* ty *)

     Delimfix ("_ * _", "[ Ty, Ty] => Ty", " Pr_Type"),
     Delimfix ("Int", "Ty", " Int"),
     Delimfix ("Bool", "Ty", " Bool"),
     Delimfix ("Exn", "Ty", " Exn"),
     Delimfix ("{}", "Ty", " Unit"),
     Delimfix ("<>", "TySeq", " tyempty"),
     Delimfix ("<_>", "TyArgs =>  TySeq", " TyList"),

    (* program *)

     Delimfix ("_;","  Exp =>  Program", " Exp_Program")];


(* -------------------------------------------------------------------- *)
(*		Predefined identifiers and Constructors			*)
(* -------------------------------------------------------------------- *)

val predefined_mixfix =
  
    [Delimfix ("plus","  Var", "plus"),             
     Delimfix ("minus","  Var", "minus"),             
     Delimfix ("times","  Var", "times"),             
     Delimfix ("less","  Var", "less"),             
     Delimfix ("zero","  Var", "zero"),             
     Delimfix ("not","  Var", "not"),      
     Delimfix ("head","  Var", "head"),      
     Delimfix ("tail","  Var", "tail"),      
     Delimfix ("null","  Var", "null"),      

     Delimfix ("true","  Con", "true_con"),             
     Delimfix ("false","  Con", "false_con"),             
     Delimfix ("nil","  Con", "nil_con"),      
     Delimfix ("cons","  Con", "cons_con"),

     Delimfix ("INT","  TyCon", "int_con"),  (* don't confuse with int etc *)
     Delimfix ("BOOL","  TyCon", "bool_con"), 
     Delimfix ("UNIT","  TyCon", "unit_con"), 
     Delimfix ("EXN","  TyCon", "exn_con"), 
     Delimfix ("List","  TyCon", "list_con") ];      

(* -------------------------------------------------------------------- *)
(* 		Natural Numbers 					*) 
(* -------------------------------------------------------------------- *)

val nat_mixfix =

    [Delimfix ("_ #+ _"," [ Nat,  Nat] =>  Nat", "nat_plus"), 
     Delimfix ("_ #- _"," [ Nat,  Nat] =>  Nat", "nat_minus"), 
     Delimfix ("_ #* _"," [ Nat,  Nat] =>  Nat", "nat_times"), 
     Delimfix ("_ #< _"," [ Nat,  Nat] =>  o", "nat_less"), 
     Delimfix ("0","  Nat", "nought"), 
     Delimfix ("1","  Nat", "one"), 
     Delimfix ("suc _","  Nat =>  Nat", "suc")]; 

(* -------------------------------------------------------------------- *)
(* 	Sequences are put last so that they are not 			*) 					(*	confused with other syntactic forms involving 			*)
(*		comma separated items 					*)
(* -------------------------------------------------------------------- *)

val seq_mixfix =

    [Delimfix ("_", "Exp => ExpSeq ", " ExpSeq1"),
     Delimfix ("_,_", "[Exp, ExpSeq] =>  ExpSeq ", " ExpSeq2"),
     Delimfix ("tyvarempty", "TyVarSeq", "TyVarSeq_Empty"), 
     Delimfix ("_ tyvarcons _", "[TyVar,  TyVarSeq] =>  TyVarSeq", "TyVarSeq"),
     Delimfix ("_", "TyVar => TyVarArgs", " TyVarArgs1"), 
     Delimfix ("_, _", "[TyVar, TyVarArgs] => TyVarArgs", " TyVarArgs2"), 
     Delimfix ("tyempty", "TySeq", "TySeq_Empty"), 
     Delimfix ("_ tycons _", "[Ty,  TySeq] =>  TySeq", "TySeq"),
     Delimfix ("_", "Ty => TyArgs", " TyArgs1"), 
     Delimfix ("_, _", "[Ty, TyArgs] => TyArgs", " TyArgs2") 
];

(* -------------------------------------------------------------------- *)
(*		 The Complete Syntax 					*)
(* -------------------------------------------------------------------- *)


val mixfix = bare_mixfix @ 
             derived_mixfix @ 
             predefined_mixfix @ 
             nat_mixfix @
             seq_mixfix;

(* -------------------------------------------------------------------- *)
(*	 Parse  Translations for Derived Forms 				*)
(* -------------------------------------------------------------------- *)


fun unoptr q [t] = Const("Apply", dummyT) $ 
		   (Const ("Var", dummyT) $ Const (q, dummyT)) $ t;

fun Prtr [s, t] = 
    Const ("Record",dummyT) $ (Const ("ExpRow2",dummyT) $
      Free ("n1",dummyT) $ s  
        $ (Const ("ExpRow1",dummyT) $ Free ("n2",dummyT) $  t));;

fun binoptr q [s,t] = 
   Const("Apply", dummyT) $  
     (Const ("Var", dummyT) $ Const (q, dummyT)) $ Prtr [s,t];
 
val Zerotr   = unoptr "zero";
val Knottr   = unoptr "not";
val Nulltr   = unoptr "null";
val Headtr   = unoptr "head";
val Tailtr   = unoptr "tail";
val Plustr   = binoptr "plus";
val Minustr  = binoptr "minus";
val Timestr  = binoptr "times";
val Lesstr   = binoptr "less";

(* lists require some machinery *)

fun Niltr [] = Const("Con", dummyT) $ Const("nil_con", dummyT);

fun  Constr [s, t] = 
   Const("Apply", dummyT) $ (Const ("Con", dummyT) $ 
    Const ("cons_con", dummyT)) $ Prtr [s,t];

fun Listtr [Const(" ExpSeq1", _) $ s] = Constr [s, Niltr []] |
    Listtr [Const(" ExpSeq2", _) $ s $ t] = Constr [s, Listtr [t]] ;

fun Casetr [t1,t2] =  Const ("Apply", dummyT) $ 
     (Const("Fn", dummyT) $ t2) $ 
       (Const("Bracket", dummyT) $ t1); (* force argument to be atomic *)

fun Iftr [t1,t2,t3] = Casetr [t1,  Const("Match", dummyT) $ 
     (Const("Con_Pat", dummyT) $ Const ("true_con", dummyT)) $
      t2 $ (Const("Mrule", dummyT) $
	   (Const("Con_Pat", dummyT) $ Const("false_con", dummyT)) $ t3)];

fun Orelsetr [t1,t2] = Iftr [t1, Const("true_con", dummyT), t2];

fun Andalsotr [t1,t2] = Iftr [t1,t2, Const("false_con", dummyT)];

fun Exp_Programtr [t] = Const("Program1", dummyT) $ 
   (Const("Val", dummyT) $ (Const ("ValBind1", dummyT) $ 
     (Const("Var_Pat", dummyT) $ Free ("it", dummyT)) $ t));

fun Pr_Pattr [s, t] = 
    Const ("Record_Pat",dummyT) $ (Const ("PatRow2",dummyT) $
      Free ("n1",dummyT) $ s  
        $ (Const ("PatRow1",dummyT) $ Free ("n2",dummyT) $  t));

fun Pr_Typetr [s, t] = 
    Const ("Record_Ty",dummyT) $ (Const ("TyRow2",dummyT) $
      Free ("n1",dummyT) $ s  
        $ (Const ("TyRow1",dummyT) $ Free ("n2",dummyT) $  t));

(* type constructions *)

fun Inttr [] = Const ("TyCon", dummyT) $ 
     Const("TySeq_Empty", dummyT) $ Const("int_con", dummyT);

fun Booltr [] = Const ("TyCon", dummyT) $  
     Const("TySeq_Empty", dummyT) $ Const("bool_con", dummyT);

fun Unittr [] = Const ("TyCon", dummyT) $  
     Const("TySeq_Empty", dummyT) $ Const("unit_con", dummyT);

fun Exntr [] = Const ("TyCon", dummyT) $  
     Const("TySeq_Empty", dummyT) $ Const("exn_con", dummyT);

fun tyemptytr [] = Const ("TySeq_Empty", dummyT);

fun tyvaremptytr [] = Const ("TyVarSeq_Empty", dummyT);

fun TyListtr [Const(" TyArgs1", _) $ s] 
 = Const("TySeq", dummyT) $ s $ tyemptytr [] |
    TyListtr [Const(" TyArgs2", _) $ s $ t] 
 = Const("TySeq", dummyT) $ s $ TyListtr [t] ;

fun TyVarListtr [Const(" TyVarArgs1", _) $ s] 
 = Const("TyVarSeq", dummyT) $ s $ tyvaremptytr [] |
    TyVarListtr [Const(" TyVarArgs2", _) $ s $ t] 
 = Const("TyVarSeq", dummyT) $ s $ TyVarListtr [t] ;

(* -------------------------------------------------------------------- *)
(*	 Print  Translations for Derived Forms 				*)
(* -------------------------------------------------------------------- *)

fun stringtr "zero"  = " Zero" |
    stringtr "not"   = " Knot" |
    stringtr "null"  = " Null" |
    stringtr "head"  = " Head" |
    stringtr "tail"  = " Tail" |
    stringtr "plus"  = " Plus" |
    stringtr "minus" = " Minus" |
    stringtr "times" = " Times" |
    stringtr "less" = " Less" ;

fun Recordtr' [Const ("ExpRow2",_) $
      Free ("n1",_) $ s  $ (Const ("ExpRow1",_) $ Free ("n2",_) $  t)]
     =  Const(" Pr", dummyT) $ s $ t;

fun Record_Pattr' [Const ("PatRow2",_) $
      Free ("n1",_) $ s  $ (Const ("PatRow1",_) $ Free ("n2",_) $  t)]
     =  Const(" Pr_Pat", dummyT) $ s $ t;

fun Record_Tytr' [Const ("TyRow2",_) $
      Free ("n1",_) $ s  $ (Const ("TyRow1",_) $ Free ("n2",_) $  t)]
     =  Const(" Pr_Type", dummyT) $ s $ t;


fun Contr' [ Const("nil_con", _)] = Const(" Nil", dummyT) |
    Contr' [s] = s ;

local fun  
  f (Const("Apply", _) $ (Const ("Con", _) $ Const ("cons_con", _)) $
    (Const ("Record",_) $ (Const ("ExpRow2",_) $
     Free ("n1",_) $ s  $
        (Const ("ExpRow1",_) $ Free ("n2",_) $
	  (Const ("Con", _) $ Const("nil_con", _))))))
   =  Const(" ExpSeq1", dummyT) $ s |
   f (Const("Apply", _) $ (Const ("Con", _) $ Const ("cons_con", _)) $
    (Const ("Record",_) $ (Const ("ExpRow2",_) $
     Free ("n1",_) $ s  $
        (Const ("ExpRow1",_) $ Free ("n2",dummyT) $ t)))) =
      Const(" ExpSeq2", dummyT) $ s $ f t
in 
fun  Applytr'  [Const ("Con", _) $ Const ("cons_con", _) ,
                Const ("Record",_) $ (Const ("ExpRow2",_) $
                  Free ("n1",_) $ s  $ (Const ("ExpRow1",_) $ Free ("n2",_) $ t))]
     = (Const (" List", dummyT) $ 
      f  (Const("Apply", dummyT) $ 
      (Const ("Con", dummyT) $ Const ("cons_con", dummyT)) $
    (Const ("Record",dummyT) $ (Const ("ExpRow2",dummyT) $
     Free ("n1",dummyT) $ s  $
        (Const ("ExpRow1",dummyT) $ Free ("n2",dummyT) $ t))))
        handle _ => Const (" Cons", dummyT) $ s $ t) |
    Applytr' [Const("Fn", _) $ (Const("Match", _) $ 
                (Const("Con_Pat", _) $ Const ("true_con", _)) $
                 Const("true_con", _) $ (Const("Mrule", _) $
	            (Const("Con_Pat", _) $ Const("false_con", _)) $ t2)) , 
              Const("Bracket", dummyT) $ t1]
	   = Const(" Orelse", dummyT) $ t1 $ t2 |
    Applytr' [Const("Fn", _) $ (Const("Match", _) $ 
                 (Const("Con_Pat", _) $ Const ("true_con", _)) $
                     t2 $ (Const("Mrule", _) $
	                 (Const("Con_Pat", _) $ Const("false_con", _)) $ 
	                      Const("false_con", _))) ,  
	      Const("Bracket", dummyT) $ t1]
	   = Const(" Andalso", dummyT) $ t1 $ t2 |
    Applytr' [Const("Fn", _) $ (Const("Match", _) $ 
                (Const("Con_Pat", _) $ Const ("true_con", _)) $
                     t2 $ (Const("Mrule", _) $
	     (Const("Con_Pat", _) $ Const("false_con", _)) $ t3)) , 
	     Const("Bracket", dummyT) $ t1]
	   = Const (" If", dummyT) $ t1 $ t2 $ t3 |
    Applytr' [Const("Fn", _) $ t2 , 
	      Const("Bracket", dummyT) $ t1] =
               Const(" Case", dummyT) $ t1 $ t2 |
    Applytr' [Const ("Var", _)  $ Const (str, _) ,
              Const ("Record",_) $ (Const ("ExpRow2",_) $ Free ("n1",_) $ s  $ 
                   (Const ("ExpRow1",_) $ Free ("n2",_) $  t))]
       = Const (stringtr str, dummyT) $ s $ t |
    Applytr'  [Const ("Var", _) $ Const (str, _), t] = 
         Const (stringtr str, dummyT) $ t
end;

fun TySeq_Emptytr' [] = Const(" tyempty", dummyT);

fun TyVarSeq_Emptytr' [] = Const(" tyvarempty", dummyT);

fun TyContr' [Const("TySeq_Empty", _), Const("int_con", _)] = 
      Const(" Int", dummyT) |
    TyContr' [Const("TySeq_Empty", _), Const("bool_con", _)] =          
      Const(" Bool", dummyT) |
    TyContr' [Const("TySeq_Empty", _), Const("unit_con", _)] =          
      Const(" Unit", dummyT) |
    TyContr' [Const("TySeq_Empty", _), Const("exn_con", _)] =          
      Const(" Exn", dummyT);

local fun f (Const("TySeq", _) $ s $ Const("TySeq_Empty", _)) 
  = Const(" TyArgs1", dummyT) $ s |
          f (Const("TySeq", _) $ s $ t)
  =  Const(" TyArgs2", dummyT) $ s $ (f t)
in 
    fun TySeqtr' [s, t] = 
    Const(" TyList", dummyT) $ f (Const("TySeq", dummyT) $ s $ t)
end;

local fun f (Const("TyVarSeq", _) $ s $ Const("TyVarSeq_Empty", _)) 
  = Const(" TyVarArgs1", dummyT) $ s |
          f (Const("TyVarSeq", _) $ s $ t)
  =  Const(" TyVarArgs2", dummyT) $ s $ (f t)
in 
    fun TyVarSeqtr' [s, t] = 
    Const(" TyVarList", dummyT) $ f (Const("TyVarSeq", dummyT) $ s $ t)
end;

val show_full_syntax = ref false;

fun tr' [x] = 
  if (!show_full_syntax) then raise Match else x;

(* -------------------------------------------------------------------- *)
(*	 Isabelle Syntax for SML 					*)
(* -------------------------------------------------------------------- *)

val sext = Sext {mixfix=mixfix,
	   parse_translation=[(" Plus", Plustr), 
                              (" Minus", Minustr),
                              (" Times", Timestr) , 
                              (" Less", Lesstr) , 
                              (" Pr", Prtr) , 
                              (" Case", Casetr) , 
                              (" Orelse", Orelsetr) , 
                              (" Andalso", Andalsotr) , 
                              (" If", Iftr) , 
                              (" Exp_Program", Exp_Programtr) , 
                              (" Knot", Knottr), 
                              (" Zero", Zerotr), 
                              (" Null", Nulltr), 
                              (" Head", Headtr), 
                              (" Tail", Tailtr), 
                              (" Pr_Pat", Pr_Pattr) , 
                              (" Pr_Type", Pr_Typetr) , 
                              (" Nil", Niltr) , 
                              (" Cons", Constr),
                              (" List", Listtr),
                              (" Int", Inttr),
                              (" Bool", Booltr),
                              (" Exn", Exntr),
                              (" tyempty", tyemptytr),
                              (" tyvarempty", tyvaremptytr),
                              (" TyList", TyListtr) ,
                              (" TyVarList", TyVarListtr) ],
	   print_translation=[("Con", Contr'),
			      ("Apply", Applytr'),
                              ("Record", Recordtr'),
                              ("Record_Pat", Record_Pattr'),
                              ("Record_Ty", Record_Tytr'),
			      ("TySeq_Empty", TySeq_Emptytr'),
			      ("TyVarSeq_Empty", TyVarSeq_Emptytr'),
                              ("TyCon", TyContr'),
                              ("TySeq", TySeqtr'),
                              ("TyVarSeq", TyVarSeqtr'),
                              ("Scon", tr'),
                              ("Var", tr'),
                              ("ExCon", tr'),
                              ("Scon_Pat", tr'),
                              ("Var_Pat", tr'),
                              ("Con_Pat", tr'),
                              ("ExCon_Pat", tr') ]};

val ntypes =   
       [(types, ([], "term"))]; 

val thy = extend_theory Int_Rule.thy "SYNTAX"
     ([], [],  [], ntypes, [], Some(sext)) [];

end;
end;
 

