(* -------------------------------------------------------------------- *)
(*									*)
(*	Prover.ml:	theorem prover for evaluation			*)
(*					 				*)
(* -------------------------------------------------------------------- *)

print_depth 3;

signature PROVER =
sig
  structure DRule: DRULE
  val mk_Dec_Seq : term * term * term -> term
  val mk_combine : term * term * term -> term
  val mk_Combine : term * term * term -> term
  val dest_Seq : term -> term * term * term * term
  val dest_combine : term -> term * term * term
  val res_tac : string  -> int -> tactic
  val get_seq_env : unit -> term
  val get_combine_env : unit -> term
  val rew_tac :  (string * string) list -> tactic
  val lookup_tac : tactic
  val record_lookup_tac : tactic
  val delete_tac : tactic
  val combine_tac : tactic
  val unfold_tac : tactic
  val apply_tac : tactic
  val FOL_tac : tactic  
  val cong_tac : tactic  
  val simp_tac : tactic  
  val asm_simp_tac : tactic  
  val reduce_tac : tactic 
  val eval_step_tac : tactic
  val evaluate_tac : tactic
  val eval_tac : tactic
  val exp_eval : string -> string -> unit
  val dec_eval : string -> string -> unit
  val prog_eval : string -> string -> unit
  val thy : theory
end;

functor ProverFun (structure DRule: DRULE) : PROVER = 
struct
structure DRule = DRule;
local open DRule in 

(* -------------------------------------------------------------------- *)
(*	Discriminators for terms - used to make				*)
(*	sure that only safe resolution rules are used			*)
(* -------------------------------------------------------------------- *)

fun is con (Const (t, _)) = (con = t) |  
    is con (t $ t') = (is con t) |
    is con _ = false;

fun get_name  (Const (con, _)) = con |  
    get_name (t $ t') = (get_name t) |
    get_name _ = "undef";

fun is_schvar (Var(_,_))  = true |  is_schvar _ = false;

fun is_free (Free(_,_))  = true |  is_free _ = false;

fun is_tyvar (Const ("tyvar", _) $ a) = true |
    is_tyvar _ = false;

fun dest_prop (Const ("Trueprop",_) $ a) = a;

fun mk_prop a =  Const ("Trueprop", dummyT) $ a;

fun dest_Not (Const("Not", _) $ v1) =  v1;

(* -------------------------------------------------------------------- *)
(*	Destructors for terms.  					*)
(* -------------------------------------------------------------------- *)

fun dest_prop (Const ("Trueprop",_) $ a) = a;

fun dest_Seq (Const("Exp_Seq", _) $ E $ exp $ v)
              = (E,exp,v, v) |  (* a padded tuple *)
    dest_Seq (Const("AtExp_Seq", _) $ E $ atexp $ v)
              = (E,atexp,v, v) |  
    dest_Seq  (Const("ExpRow_Seq", _) $ E $ exprow $ v)
              = (E,exprow,v,v) |
    dest_Seq  (Const("Match_Seq", _) $ E $ v $ match $ v')
              = (E,match,v,v') |  (* note order of arguments *)
    dest_Seq  (Const("Pat_Seq", _) $ E $ v $ pat $ E')
              = (E,pat,v,E') |
    dest_Seq  (Const("Dec_Seq", _) $ E1 $ dec $ E2)
              = (E1,dec,E2, E2) |
    dest_Seq  (Const("ValBind_Seq", _) $ E1 $ valbind $ E2)
              = (E1,valbind,E2, E2) |
    dest_Seq  (Const("PatRow_Seq", _) $ E $ v $ patrow $ f)
              = (E,patrow,v,f) |
    dest_Seq  (Const("Program_Seq", _) $ E $ prog $ E')
              = (E,prog,E',E');

fun dest_lookup (Const("lookup", _) $ ide $ E $ v)  = (ide,E,v);

fun dest_delete (Const("delete", _) $ x $ E1 $ E2)  = (x,E1,E2);

fun dest_combine 
     (Const("combine", _) $ E1 $ E2 $ E3) = (E1, E2, E3)|
    dest_combine 
     (Const("Combine", _) $ E1 $ E2 $ E3) = (E1, E2, E3);

fun dest_unfold  (Const("unfold", _) $ E1 $ E2 $ E3)  = (E1,E2,E3);

fun dest_record_lookup (Const("record_lookup", _) $ lab $ v $ v') = (lab,v,v');

fun dest_apply (Const("apply", _) $ v1 $ v2 $ v3) = (v1,v2,v3);

fun dest_Not (Const("Not", _) $ v1) =  v1;

fun dest_Var (Const ("Var",_) $ Free(v,_)) = v;

fun dest_Const (Const ("Const",_) $ Const(c,_)) = c ;

fun dest_Apply (Const ("Apply",_) $ e1 $ e2) = (e1,e2) ;

fun dest_AtExp (Const("AtExp", _) $ e) = e;

fun mk_Dec_Seq (E, prog, E') = 
      Const("Dec_Seq", dummyT) $ E $ prog $ E';

fun mk_combine (E1,E2,E3) = Const("combine", dummyT) $ E1 $ E2 $ E3;

fun mk_Combine (E1,E2,E3) = Const("Combine", dummyT) $ E1 $ E2 $ E3;

(* -------------------------------------------------------------------- *)
(*	Special Purpose Resolution tactics 		  		*)
(*	NB: they depend on the theory, namely thy			*)
(* -------------------------------------------------------------------- *)

fun res_tac s k = resolve_tac (intr s) k;

fun rew_tac rulelist =
 rewrite_tac (map (get_axiom thy) (map fst rulelist));

fun get_seq_env () =  
  let val (_,_,VE,_) = dest_Seq (dest_prop (concl_of (topthm ())))
   in VE end;

fun get_combine_env () =  
  let val (_,_,VE) = dest_combine (dest_prop (concl_of (topthm ())))
   in VE end;

(* -------------------------------------------------------------------- *)
(*	reduction tactics 				  		*)
(* -------------------------------------------------------------------- *)

val cong_tac = FIRSTGOAL (resolve_tac congs);

val ss = FOL_ss addrews VE0 addcongs congs;

val simp_tac = SIMP_TAC ss;

val asm_simp_tac = ASM_SIMP_TAC ss;

val lookup_tac = FIRSTGOAL
 (SUBGOAL (fn (prem, i) => 
  let val t = Logic.strip_assums_concl prem;
      val (x,E,v) = dest_lookup (dest_prop t)
  in case E of 
         Var (_,_)  => no_tac |      (* indeterminate goals *)
         _ $ Var (_,_)  => no_tac |      
         Const ("VarEnv0",_)  => 
              ASM_SIMP_TAC ss i |
         _ => DETERM (resolve_tac lookup i)
   end
   handle _ => no_tac));
 
val delete_tac = FIRSTGOAL  
  (SUBGOAL (fn (prem,i) =>
   let val t = Logic.strip_assums_concl prem;
       val (x,E,_) = dest_delete (dest_prop t)
   in case E of 
         Var (_,_)  => no_tac |      (* indeterminate goals *)
         _ $ Var (_,_)  => no_tac |      
      _ => DETERM (resolve_tac delete i)
   end
   handle _ => no_tac));

val combine_tac = FIRSTGOAL  
  (SUBGOAL (fn (prem,i) =>
   let val t = Logic.strip_assums_concl prem;
       val (E1,E2,_) = dest_combine (dest_prop t)
   in if (is_schvar E1 orelse is_schvar E2)  (* indeterminate goals *) then 
         no_tac
      else
         DETERM (resolve_tac (combine @ Combine) i)
   end
   handle _ => no_tac));


val unfold_tac = FIRSTGOAL  
  (SUBGOAL (fn (prem,i) =>
  let val t = Logic.strip_assums_concl prem;
      val (_,E,_) = dest_unfold (dest_prop t)
  in case E of 
         Var (_,_)  => no_tac |      (* indeterminate goals *)
         _ $ Var (_,_)  => no_tac |      
         _ => DETERM (resolve_tac unfold i)
  end
  handle _ => no_tac));

val record_lookup_tac = FIRSTGOAL
 (SUBGOAL (fn (prem, i) => 
  let val t = Logic.strip_assums_concl prem;
      val (lab,v,v') = dest_record_lookup (dest_prop t)
  in case v of 
         Var (_,_)  => no_tac |      (* indeterminate goals *)
         _ $ Var (_,_)  => no_tac |      
         _ => DETERM (resolve_tac record_lookup i)
   end
   handle _ => no_tac));

val apply_tac = FIRSTGOAL 
 (SUBGOAL (fn (prem,i) =>
  let val t = Logic.strip_assums_concl prem;
      val (v,_,_) = dest_apply (dest_prop t);
      val (_ $ Const(c, _)) = v 
 in case c of
    _ => resolve_tac apply i   (* !!!!! *)
 end
 handle _ => no_tac));

val FOL_tac = REPEAT1 (FIRSTGOAL (fast_tac FOL_cs));

val cong_tac = FIRSTGOAL (resolve_tac congs);

val simp_tac = FIRSTGOAL (SIMP_TAC FOL_ss);

val asm_simp_tac = FIRSTGOAL (ASM_SIMP_TAC FOL_ss);

val reduce_tac  = 
  REPEAT1
  (lookup_tac ORELSE 
   delete_tac ORELSE 
   combine_tac ORELSE 
   unfold_tac ORELSE
   record_lookup_tac ORELSE
   apply_tac ORELSE 
   cong_tac ORELSE
   FOL_tac);


(* -------------------------------------------------------------------- *)
(*	step tactic:	applies a language rule once, and safely	*)
(* -------------------------------------------------------------------- *)


val eval_step_tac = FIRSTGOAL
 (SUBGOAL (fn (prem,i) => 
  let val t = Logic.strip_assums_concl prem;
     val (_,phrase,_,_) = dest_Seq (dest_prop t) in
   if (is_schvar phrase) then no_tac (* indeterminate goal *) 
   else if (is "Apply" phrase) then (* Apply needs special handling ... *)
       let val (e1,e2) = dest_Apply phrase;
	   val (_ $ Const (ide, _)) = e1
       in
    	   resolve_tac [Apply_Basval] i 
       end
       handle _ =>  DETERM (resolve_tac [Apply_Match, Apply_Con] i)
   else res_tac (get_name phrase) i  (* ... most constructs don't *)
  end
  handle _ => no_tac));

(* -------------------------------------------------------------------- *)
(*	evaluate_tac:	repeatedly applies evaluation rules,		*)
(* -------------------------------------------------------------------- *)

val evaluate_tac  = REPEAT1 (eval_step_tac ORELSE reduce_tac);
              
(* -------------------------------------------------------------------- *)
(*	 eval_tac:	general-purpose tactic for proving goals by	*)
(*			depth-first search				*)
(* -------------------------------------------------------------------- *)

val eval_tac = DEPTH_FIRST (has_fewer_prems 1) (evaluate_tac);

fun exp_eval E s = 
(print "\n---------------------------\
         \------------------------------------------\n\
         \Trying to evaluate ....\n";
         goal thy (E^" |- "^s^" ~> ?v"); by eval_tac);

fun dec_eval E s = 
(print "\n---------------------------\
         \------------------------------------------\n\
         \Trying to evaluate ....\n";
         goal thy (E^" ||- "^s^" ~> ?E"); by eval_tac);

fun prog_eval E s = 
(print "\n---------------------------\
         \------------------------------------------\n\
         \Trying to evaluate ....\n";
         goal thy (E^" ||- "^s^" ~> ?E [Program]"); by eval_tac);

val thy = thy;

end;
end;


