(* -------------------------------------------------------------*)
(*		Obj.ml:  Semantic Objects for Evaluation	*)
(* ------------------------------------------------------------ *)




signature OBJ =
sig
  structure SML_Syntax: SML_SYNTAX 
  val ntypes: (string list * (string list * string)) list
  val sext : Syntax.sext
  val thy : theory
  val show_envs : bool ref
  val show_full_values : bool ref
end;

functor ObjFun (SML_Syntax: SML_SYNTAX) : OBJ = 
struct
structure SML_Syntax = SML_Syntax;
local open Syntax  
in

val simple_objects =  (* note - not all of these are used yet! *)
            [ "Addr",		(*  addresses *)
              "ExName", 	(* exception names *)
              "BasVal", 	(* basic values *)
              "SVal", 		(* special values *)
	      "Fail",		(* failure - singleton set *)
	      "Id"];		(* general term for var, con and excon *)

val compound_objects =
            [ "Val", 		(* values *)
	      "Val'", 		(* value/fail/packet *)
	      "Record",		(* record values *)
              "ExVal",		(* exception values *)
              "Pack",		(* packets *)
              "Closure",	(* function closures *)
              "Mem",		(* memories *)
              "ExNameSet",	(* exception name set *)
              "State",		(* states *)
              "Env",		(* environments *)
              "Env'",		(* environment/packet *)
              "StrEnv",		(* structure environments *)
              "VarEnv",		(* variable environments *)
              "VarEnv'",	(* variable env/fail *)
              "ExConEnv"];	(* exception constructor environments *)

val auxiliary_objects =
	     ["ValRow", 	(* value rows *)
	      "ValRow'", 	(* value row/packet *)
	      "IdValSeq",	(* binding sequence *)
	      "ValSeq"];	(* value sequence *)

val types = simple_objects @ compound_objects @ auxiliary_objects;

(* -------------------------------------------------------------------- *)
(*		 Evaluation - Semantic Objects	 			*)
(* -------------------------------------------------------------------- *)


val eval_mixfix =

 [  (* values  *)

     Delimfix ("var _", "Var => Id", "VarId"),
     Delimfix ("con _", "Con => Id", "ConId"),
     Delimfix ("excon _", "ExCon => Id", "ExConId"),

     Delimfix ("val _", "Var   => Val", "val"),  
     Delimfix ("num _", "Nat   => Val", "num"),             
     Delimfix ("<con _> ", "Con   => Val", "con"),             
     Delimfix ("<con _,_>", "[Con, Val]   => Val", "con_val"),             
     Delimfix ("exname _ ", "ExName   => ExVal", "exname"),             
     Delimfix ("exname _,_", "[ExName, Val]   => ExVal", "exname_val"),    
     Delimfix ("<_>", "ExVal   => Val", "exval"),             
     Delimfix ("[_]", "ExVal   => Pack", "packet"),             
     Delimfix ("{} inVal", "Val", "empty_record"),
     Delimfix ("{_} inVal", "ValRow   => Val", "record"),
     Delimfix ("_ = _", "[ Lab, Val]   => ValRow ", "ValRow1"),
     Delimfix ("_ = _ , _", "[ Lab, Val, ValRow]   => ValRow ", "ValRow2"),
     Delimfix ("[]", "Val", " nil"),
     Delimfix ("(1_ cons/ _)", "[Val, Val]   =>  Val", " cons"),
     Delimfix ("[_]", "ValSeq   => Val", " list"),
     Delimfix ("closure '(_,_,_')", "[Match, Env, VarEnv]   => Val", "closure"),
     Delimfix ("_ = / _ : value", "[Val, Val]   =>   o", "value_equals"),
     Delimfix ("FAIL", "Fail", "FAIL"),  (* for pattern matching *)
     Delimfix ("_ in1Val''", "Val   => Val'", "Val'1"),
     Delimfix ("_ in2Val''", "Fail   => Val'", "Val'2"),
     Delimfix ("_ in3Val''", "Pack   => Val'", "Val'3"),
     Delimfix ("_ in1ValRow''", "ValRow   => ValRow'", "ValRow'1"),
     Delimfix ("_ in2ValRow''", "Fail   => ValRow'", "ValRow'2"),
     Delimfix ("tr _", "o => Val", "tr"),  

   (* environments  *)

     Delimfix ("{| |}", "VarEnv", "VarEnv_empty"),             
     Delimfix ("{| |}", "StrEnv", "StrEnv_empty"),             
     Delimfix ("{| |}", "ExConEnv", "ExConEnv_empty"),             
     Delimfix ("{| _ |}", "IdValSeq => VarEnv", "VarEnv"),             
     Delimfix ("'(_,_')",  "[Id, Val]   => IdValSeq ", "IdValSeq1"),
     Delimfix ("'(_,_'),_",  
        "[Id, Val, IdValSeq]   => IdValSeq ", "IdValSeq2"),
     Delimfix ("VE0", "VarEnv", "VarEnv0"),             
     Delimfix ("_ in1VarEnv''", "VarEnv   => VarEnv'", "VarEnv'1"),
     Delimfix ("_ in2VarEnv''", "Fail   => VarEnv'", "VarEnv'2"),
     Delimfix ("...", "Env", " Env"),
     Delimfix ("record'_lookup '(_,_,_')", 
	 (* chicane - these must appear before value sequences and pairs ! *)
	 "[ Lab, Val, Val]   =>   o", "record_lookup"),
     Delimfix ("lookup '(_,_,_')", 
	 "[Id, VarEnv, Val]   =>   o", "lookup"),
     Delimfix ("delete '(_,_,_')", 
	 "[Id, VarEnv, VarEnv]   =>   o", "delete"),
     Delimfix ("combine '(_,_,_')", 
	 "[VarEnv, VarEnv, VarEnv]   =>   o", "combine"),
     Delimfix ("unfold '(_,_,_')", 
	 "[VarEnv, VarEnv, VarEnv]   =>   o", "unfold"),
     Delimfix ("<_,_,_>", "[StrEnv, VarEnv, VarEnv] => Env", "Env"),
                    (* NB : VarEnv includes ExConEnv as subtype *)
     Delimfix ("Combine '(_,_,_')", 
	 "[Env, Env, Env]   =>   o", "Combine"),
     Delimfix ("_ in1Env''", "Env   => Env'", "Env'1"),
     Delimfix ("_ in2Env''", "Pack   => Env'", "Env'2"),
 
    
   (* semantic function for basic values *)

     Delimfix ("apply '<_,_,_'>", 
        "[Val, Val, Val]   =>   o", "apply"),             

   (* sequents *)

     Delimfix ("(3_ |-/ _ ~> /_)",		
	   "[Env,Exp,Val']   =>   o", "Exp_Seq"),
     Delimfix ("(3_ |- / _ ~> /_    [ExpRow])",		
	   "[Env,ExpRow,ValRow']   =>   o", "ExpRow_Seq"),
     Delimfix ("(3_ , _ |-/ _ ~> /_  [Match])",		
	   "[Env,Val,Match,Val']   =>   o", "Match_Seq"),
     Delimfix ("(3_ , _ ||-/ _ ~> /_  [Pat])",		
	   "[Env,Val,Pat,VarEnv']   =>   o", "Pat_Seq"),
     Delimfix ("(3_ , _ ||-/ _ ~> /_  [PatRow])",		
	   "[Env,Val,PatRow,VarEnv']   =>   o", "PatRow_Seq"),
     Delimfix ("(3_ ||-/ _ ~> /_)",
	   "[Env,Dec,Env']   =>     o", "Dec_Seq"),
     Delimfix ("(3_ ||-/ _ ~> /_  [ValBind])",
	   "[Env,ValBind,VarEnv']   =>     o", "ValBind_Seq"),
     Delimfix ("(3_ ||-/ _ ~> /_  [ExBind])",
	   "[Env,ExBind,VarEnv']   =>     o", "ExBind_Seq"),
     Delimfix ("(3_ ||-/ _ ~> /_  [Program])",
	   "[Env, Program,Env]   =>     o", "Program_Seq"),

   (* equivalence *)

     Delimfix ("_ ~~ _",  "[Exp, Exp] => o", "Equiv")];

(* -------------------------------------------------------------------- *)
(*		 Derived Forms and Chain Productions			*)
(* -------------------------------------------------------------------- *)

val derived_mixfix =
   [Delimfix ("'(-_,_-')", "[Val,Val]   => Val", " pr")];

(* -------------------------------------------------------------------- *)
(*	Val sequences and pairs are put last so that other 		*)
(*	syntactic forms involving comma separated items 		*)
(*	are not confused with values!					*)
(* -------------------------------------------------------------------- *)

val valseq_mixfix =

   [ Delimfix ("('(_,/_'))",  "[Id, Val]   => IdValSeq ", "IdValSeq1"),
     Delimfix ("('(_,/_'),/_)",  
        "[Id, Val, IdValSeq]   => IdValSeq ", "IdValSeq2"),
     Delimfix ("_",  "Val   => ValSeq ", " ValSeq1"),
     Delimfix ("_,_",  "[Val, ValSeq]   => ValSeq ", " ValSeq2")];                

(* -------------------------------------------------------------------- *)
(*		 The Complete Syntax 					*)
(* -------------------------------------------------------------------- *)

val mixfix = eval_mixfix @ derived_mixfix @ valseq_mixfix;
 
(* -------------------------------------------------------------------- *)
(*		 Parse Translations for Derived Forms 			*)
(* -------------------------------------------------------------------- *)

fun niltr [] = Const("con",  dummyT) $ Const("nil_con",  dummyT);

fun prtr [s, t] = 
    Const ("record", dummyT) $ (Const ("ValRow2", dummyT) $
      Free ("n1", dummyT) $ s  
        $ (Const ("ValRow1", dummyT) $ Free ("n2", dummyT) $  t));;

fun  constr [s, t] = 
   Const("con_val", dummyT) $ Const ("cons_con", dummyT) $ prtr [s,t];

fun listtr [Const(" ValSeq1", _) $ s] = constr [s, niltr []] |
    listtr [Const(" ValSeq2", _) $ s $ t] = constr [s, listtr [t]] ;

fun valemptytr [] = Const ("ValSeq_Empty", dummyT);

(* -------------------------------------------------------------------- *)
(*	 	 Print Translations 			 		*)
(* -------------------------------------------------------------------- *)

fun recordtr' [Const ("ValRow2",_) $
      Free ("n1",_) $ s  $ (Const ("ValRow1",_) $ Free ("n2",_) $  t)]
     =  Const(" pr", dummyT) $ s $ t |
    recordtr' [s] = s;

fun contr' [Const("nil_con", _)] = Const (" nil",  dummyT) |
    contr' [s] = s;

(* NB : constr' is not used directly - it appears in con_valtr' *)

fun  constr' (Const("con_val", _) $ Const ("cons_con", _) $
    (Const ("record",_) $ (Const ("ValRow2",_) $
     Free ("n1",_) $ s  $
        (Const ("ValRow1",_) $ Free ("n2",_) $ 
     (Const("con",  dummyT) $ Const("nil_con", _))))))
   =  Const(" ValSeq1",  dummyT) $ s |
   constr' (Const("con_val", _) $ Const ("cons_con", _) $
    (Const ("record",_) $ (Const ("ValRow2",_) $
     Free ("n1",_) $ s  $
        (Const ("ValRow1",_) $ Free ("n2", dummyT) $ t)))) =
      Const(" ValSeq2",  dummyT) $ s $ constr' t; 

fun con_valtr' 
      [Const ("cons_con",_), 
       Const ("record",_) $ (Const ("ValRow2",_) $ 
             Free ("n1",_) $ s $ (Const ("ValRow1",_) $ 
         Free ("n2",_) $ Free (x,_)))] =
     Const("cons",  dummyT) $ s $ Free (x,  dummyT) |
    con_valtr' [Const ("cons_con", _) , arg]
       =  Const (" list",  dummyT) $ 
        constr' (Const("con_val",  dummyT) $ 
                 Const ("cons_con",  dummyT) $ arg);


val show_envs = ref false;
val show_full_values = ref false;

fun Envtr' [SE,VE,EE] =  
  if (!show_envs) then raise Match else Const (" Env", dummyT); 

fun tr' [x] = 
  if (!show_full_values) then raise Match else x;


(* -------------------------------------------------------------------- *)
(*	 Isabelle Syntax for SML 					*)
(* -------------------------------------------------------------------- *)

val sext = Sext {mixfix=mixfix,
	   parse_translation=[(" nil", niltr) , 
                              (" cons", constr),
                              (" list", listtr),
                              (" pr", prtr)],
	   print_translation=[("con", contr'), 
		       	      ("con_val", con_valtr'),
                              ("Env", Envtr'),
                              ("VarId", tr'),
                              ("ConId", tr'),
                              ("ExConId", tr'),
                              ("val", tr'),
                              ("num", tr'),
                              ("excon", tr'),
                              ("exname", tr'),
                              ("empty_record", tr'),
                              ("Val'1", tr'),
                              ("Val'2", tr'),
                              ("Val'3", tr'),
                              ("Env'1", tr'),
                              ("Env'2", tr'),
                              ("VarEnv'1", tr'),
                              ("VarEnv'2", tr'),
                              ("ValRow'1", tr'),
                              ("ValRow'2", tr'),
                              ("record", recordtr') ]};

val sorts = [];

val ntypes = [(types, ([], "term"))];

val thy = extend_theory SML_Syntax.thy "OBJ"
     ([], [], [], ntypes, [], Some(sext)) [];

end;
end;
 

