(* -------------------------------------------------------------------- *)
(*									*)
(*	Prover.ml:							*)
(*	Theorem prover for Elaboration					*)
(*					 				*)
(* -------------------------------------------------------------------- *)

print_depth 3;

signature PROVER =
sig
  structure DRule: DRULE
  val mk_Dec_Seq : term * term * term -> term
  val mk_combine : term * term * term -> term
  val dest_Seq : term -> term * term * term * term
  val dest_combine : term -> term * term * term
  val res_tac : string -> int -> tactic 
  val elab_env : unit -> term
  val simp_tac : tactic  
  val asm_simp_tac : tactic  
  val lookup_tac : tactic
  val delete_tac : tactic
  val combine_tac : tactic
  val subst_tac : tactic
  val is_free_tac : tactic
  val is_Free_tac : tactic
  val is_bound_tac : tactic
  val Close_tac : tactic
  val As_tac : tactic
  val Abs_tac : tactic
  val ABS_tac : tactic
  val instance_tac : tactic 
  val trivial_instance_tac : tactic 
  val reduce_tac : tactic 
  val instance_unify_tac : tactic 
  val simplify_tac : tactic 
  val elab_step_tac : tactic
  val elaborate_tac : tactic
  val infer : thm -> thm
  val elab_tac : tactic
  val exp_elab : string -> string -> unit
  val dec_elab : string -> string -> unit
  val prog_elab : string -> string -> unit
  val thy : theory
end;

functor ProverFun (structure DRule: DRULE) : PROVER = 
struct
structure DRule = DRule;
local open DRule in 

(* -------------------------------------------------------------------- *)
(*	Discriminators for terms - used to make				*)
(*	sure that only safe resolution rules are used			*)
(* -------------------------------------------------------------------- *)

fun is con (Const (t, _)) = (con = t) |  
    is con (t $ t') = (is con t) |
    is con _ = false;

fun get_name  (Const (con, _)) = con |  
    get_name (t $ t') = (get_name t) |
    get_name _ = "undef";

fun is_schvar (Var(_,_))  = true |  is_schvar _ = false;

fun is_TyVar (Const ("TyVar", _) $ a) = true |
    is_TyVar _ = false;

fun is_Ide (Var (_, Type("Ide", []))) = true |
    is_Ide _ = false;

fun dest_prop (Const ("Trueprop",_) $ a) = a;

fun mk_prop a =  Const ("Trueprop", dummyT) $ a;

fun dest_Not (Const("Not", _) $ v1) =  v1;

local fun prep_a n = Free ("a"^(string_of_int n), Type("Ide", []));
      fun list_from_to n m =
         if (n = m) then  [n]
            else
         n::(list_from_to (n+1) m);
in  fun newvars 0 = []
   | newvars 1 = [Free ("a", Type("Ide", []))]
   | newvars n = map prep_a (list_from_to 1 n)
end;

(* -------------------------------------------------------------------- *)
(*	Destructors for terms.  					*)
(* -------------------------------------------------------------------- *)


fun dest_Seq (Const("Exp_Seq", _) $ C $ exp $ t)
              = (C,exp,t, t) |  (* a padded tuple *)
    dest_Seq  (Const("ExpRow_Seq", _) $ C $ exprow $ typerow)
              = (C,exprow,typerow,typerow) |
    dest_Seq  (Const("Match_Seq", _) $ C $ match $ t)
              = (C,match,t,t) |  
    dest_Seq  (Const("Pat_Seq", _) $ C $ pat $ VE $ t)
              = (C,pat,VE,t) |
    dest_Seq  (Const("PatRow_Seq", _) $ C $ patrow $ VE $ t)
              = (C,patrow,VE,t) |
    dest_Seq  (Const("Ty_Seq", _) $ C $ ty $ t)
              = (C,ty,t,t) |
    dest_Seq  (Const("TySeq_Seq", _) $ C $ ty $ t)
              = (C,ty,t,t) |
    dest_Seq  (Const("TyRow_Seq", _) $ C $ tyrow $ typerow)
              = (C,tyrow,typerow,typerow) |
    dest_Seq  (Const("Dec_Seq", _) $ C $ dec $ E)
              = (C,dec,E, E) |
    dest_Seq  (Const("ValBind_Seq", _) $ C $ valbind $ VE)
              = (C,valbind,VE,VE) |
    dest_Seq  (Const("TypBind_Seq", _) $ C $ typbind $ TE)
              = (C,typbind,TE,TE) |
    dest_Seq  (Const("DatBind_Seq", _) $ C $ datbind $ VE $ TE)
              = (C,datbind,VE,TE) |
    dest_Seq  (Const("ConBind_Seq", _) $ C $ t $ conbind $ VE)
              = (C,conbind,t,VE) |
    dest_Seq  (Const("ExBind_Seq", _) $ C $ exbind $ VE)
              = (C,exbind,VE,VE) |
    dest_Seq  (Const("Program_Seq", _) $ C $ program $ C')
              = (C,program,C',C');

fun dest_lookup (Const("lookup", _) $ id $ VE $ t) = (id, VE, t) |
    dest_lookup (Const("lookup_tycon", _) $ tycon $ TE $ tystr) = 
            (tycon,TE,tystr);

fun dest_delete (Const("VE_delete", _) $ id $ E1 $ E2)  = (id, E1, E2) |
    dest_delete (Const("TE_delete", _) $ id $ E1 $ E2)  = (id, E1, E2);

fun dest_combine 
     (Const("VE_combine", _) $ E1 $ E2 $ E3) = (E1, E2, E3)|
    dest_combine 
     (Const("TE_combine", _) $ E1 $ E2 $ E3) = (E1, E2, E3)|
    dest_combine 
     (Const("Combine", _) $ E1 $ E2 $ E3) = (E1, E2, E3)|
    dest_combine 
     (Const("COMBINE", _) $ E1 $ E2 $ E3) = (E1, E2, E3);

fun dest_fn_type (Const ("fn_type", _) $ a $ b) = (a, b);

fun dest_TypeSeq (Const ("TypeSeq", _) $ a $ b) = (a, b);

fun dest_env (Const ("VarEnv", _) $ E) = E |
    dest_env (Const ("TyEnv", _) $ E) = E;

fun dest_ConsType (Const ("ConsType", _) $ t $ tyname) = (t,tyname);

fun dest_subst (Const("subst", _) $ a $ b $ c $ d) = (a,b,c,d) |
    dest_subst (Const("subst_seq", _) $ a $ b $ c $ d) = (a,b,c,d);

fun dest_seq (Const("IdTypeSeq1", _) $ id $ t) = (id,t, t) |
    dest_seq (Const("IdTypeSeq2", _) $ id $ t $ seq) = (id,t,seq) |
    dest_seq (Const("TyConTyStrSeq1", _) $ tycon $ tystr) = 
          (tycon,tystr, tystr) |
    dest_seq (Const("TyConTyStrSeq2", _) $ tycon $ tystr $ seq) =  
          (tycon,tystr,seq);

fun dest_is_free
    (Const("is_free", _) $ a $ b) = (a,b);

fun dest_is_Free
   (Const("is_Free", _) $ a $ b) = (a,b);

fun dest_Close
    (Const("Close", _) $ a $ b $ c) = (a,b,c);

fun dest_As (Const("As", _) $ a $ b) = (a,b);

fun dest_Abs (Const("Abs", _) $ TE $ TE') = (TE, TE');

fun dest_ABS (Const("ABS", _) $ TE $ E $ E') = (TE, E, E');

fun dest_instance  (Const ("instance", _) $ a $ b) = (a, b);

fun dest_close (Const ("close", _) $ a $ b) = (a, b);

fun dest_close1 (Const ("close", _) $ a $ b) = a;

fun mk_Dec_Seq (C, dec, E) = Const("Dec_Seq", dummyT) $ C $ dec $ E;

fun mk_combine (E1, E2, E3) = Const("combine", dummyT) $ E1 $ E2 $ E3;

(* -------------------------------------------------------------------- *)
(*	Special Purpose Resolution tactics 		  		*)
(*	NB: they depend on the theory, namely thy		*)
(* -------------------------------------------------------------------- *)

fun res_tac s k = resolve_tac (intr s) k;

fun elab_env () =  
  let val (_,_,VE,_) = dest_Seq (dest_prop (concl_of (topthm ())))
   in VE end;

(* -------------------------------------------------------------------- *)
(*	reduction and simplication tactics 		  		*)
(* -------------------------------------------------------------------- *)

val cong_tac = FIRSTGOAL (resolve_tac congs);

val ss = FOL_ss addrews (VE0 @ TE0) addcongs congs;

val simp_tac = FIRSTGOAL (SIMP_TAC ss);

val asm_simp_tac = FIRSTGOAL (ASM_SIMP_TAC ss);

val lookup_tac = FIRSTGOAL
 (SUBGOAL (fn (prem, i) => 
  let val t = Logic.strip_assums_concl prem;
      val (x,E,v) = dest_lookup (dest_prop t)
  in case E of 
         Var (_,_)  => no_tac |      (* indeterminate goal *)
         Const ("VarEnv0",_)  =>    (* types of predefined functions *)
              ASM_SIMP_TAC ss i |
         Const ("TyEnv0",_)  =>    (* types of predefined functions *)
              ASM_SIMP_TAC ss i |
         _ => DETERM (resolve_tac (lookup @ lookup_tycon) i)
   end
   handle _ => no_tac));

val delete_tac = FIRSTGOAL  
  (SUBGOAL (fn (prem,i) =>
   let val t = Logic.strip_assums_concl prem;
       val (x,E,_) = dest_delete (dest_prop t);
       val (id, _, _) = dest_seq (dest_env E)
   in if (is_schvar id orelse is_schvar x)  (* indeterminate goals *) then 
         no_tac
      else 
         DETERM (resolve_tac (VE_delete @ TE_delete) i)
   end
   handle _ => no_tac));

val combine_tac = FIRSTGOAL  
  (SUBGOAL (fn (prem,i) =>
   let val t = Logic.strip_assums_concl prem;
       val (_,E,v) = dest_combine (dest_prop t)
   in case E of 
         Var (_,_)  => no_tac |      (* indeterminate goal *)
         _ => DETERM (resolve_tac 
                 (VE_combine @ TE_combine @ Combine @ COMBINE) i)
   end
   handle _ => no_tac));
               

val subst_tac = FIRSTGOAL  
  (SUBGOAL (fn (prem,i) =>
   let val t = Logic.strip_assums_concl prem;
       val (ty1, _, _, _) = dest_subst (dest_prop t)
   in  if (is_schvar ty1) then no_tac 
   else resolve_tac (subst @ subst_seq) i
   end
   handle _ => no_tac));

val is_free_tac = FIRSTGOAL  
  (SUBGOAL (fn (prem,i) =>
   let val t = Logic.strip_assums_concl prem;
       val tt = dest_prop t
   in  
     if (is "Not" tt) then
       let val (_, ty) = dest_is_free (dest_Not tt)
       in  if (is_schvar ty) then no_tac 
           else resolve_tac is_not_free i
       end
     else
       let val (_, ty) = dest_is_free tt
       in  if (is_schvar ty) then no_tac 
           else resolve_tac is_free i
       end
   end
   handle _ => no_tac));

val is_Free_tac = FIRSTGOAL  
  (SUBGOAL (fn (prem,i) =>
   let val t = Logic.strip_assums_concl prem;
       val tt = dest_prop t
   in  
     if (is "Not" tt) then
       let val (_, ty) = dest_is_Free (dest_Not tt)
       in  if (is_schvar ty) then no_tac 
           else resolve_tac is_not_Free i
       end
     else
       let val (_, ty) = dest_is_Free tt
       in  if (is_schvar ty) then no_tac 
           else resolve_tac is_Free i
       end
   end
   handle _ => no_tac));

val is_bound_tac  = 
   FIRSTGOAL (resolve_tac is_bound); 

val Close_tac = FIRSTGOAL  
  (SUBGOAL (fn (prem,i) =>
   let val t = Logic.strip_assums_concl prem;
       val (_, env, _) = dest_Close (dest_prop t)
   in  if (is_schvar env) then no_tac 
   else resolve_tac Close i
   end
   handle _ => no_tac));

val As_tac = FIRSTGOAL  
  (SUBGOAL (fn (prem,i) =>
   let val t = Logic.strip_assums_concl prem;
       val (seq, _) = dest_As (dest_prop t)
   in  if (is_schvar seq) then no_tac 
   else resolve_tac As i
   end
   handle _ => no_tac));

val Abs_tac = FIRSTGOAL  
  (SUBGOAL (fn (prem,i) =>
   let val t = Logic.strip_assums_concl prem;
       val (TE, TE') = dest_Abs (dest_prop t)
   in  if (is_schvar TE) then no_tac 
   else resolve_tac Abs' i
   end
   handle _ => no_tac));

val ABS_tac = FIRSTGOAL  
  (SUBGOAL (fn (prem,i) =>
   let val t = Logic.strip_assums_concl prem;
       val (TE, TE', _) = dest_ABS (dest_prop t)
   in  if (is_schvar TE orelse is_schvar TE') then no_tac 
   else resolve_tac ABS i
   end
   handle _ => no_tac));

val instance_tac = FIRSTGOAL
  (SUBGOAL (fn (prem,i) =>
   let val t = Logic.strip_assums_concl prem;
       val (ty1, ty2) = dest_instance (dest_prop t)
   in  if ((is_schvar ty1) andalso (is_schvar ty2)) then no_tac
   else if (is "close" ty1) then resolve_tac instance i
   else no_tac
   end
   handle _ => no_tac));

val trivial_instance_tac = FIRSTGOAL  
  (SUBGOAL (fn (prem,i) =>
   let val t = dest_prop (Logic.strip_assums_concl prem);
       val (ty1, ty2) = dest_instance t
   in  if ((is_schvar ty1) andalso (is_schvar ty2)) then no_tac
       else resolve_tac trivial_instance i 
   end
   handle _ => no_tac));


val reduce_tac  = 
  REPEAT1
  (lookup_tac ORELSE 
   delete_tac ORELSE 
   combine_tac ORELSE 
   trivial_instance_tac ORELSE
   Close_tac ORELSE
   As_tac ORELSE 
   Abs_tac ORELSE ABS_tac);

(* -------------------------------------------------------------------- *)
(*	step tactics:	applies a language rule once, and safely	*)
(* -------------------------------------------------------------------- *)

val elab_step_tac = FIRSTGOAL  (* no surprises here *)
 (SUBGOAL (fn (prem,i) => 
  let val t = Logic.strip_assums_concl prem;
     val (_,phrase,_,_) = dest_Seq (dest_prop t) in
   if (is_schvar phrase) then no_tac (* indeterminate goal *) 
   else res_tac (get_name phrase) i  
   end
  handle _ => no_tac));
              
(* -------------------------------------------------------------------- *)
(*	 Functions for Polymorphic Type Inference			*)
(* -------------------------------------------------------------------- *)

fun mk_pair_list [] [] = [] |
    mk_pair_list (a::b) (c::d) = (a,c)::mk_pair_list b d;


fun term_list_to_term [t] = t | 
    term_list_to_term (t::y) = t $ (term_list_to_term y);

(* -------------------------------------------------------------------- *)
(*	 infer:		takes a given theorem, and replaces each	*)
(*			scheme (as yet uninstantiated) variable 	*)
(*			occurring in a type variable by			*)
(*		        a corresponding type variable			*)
(* -------------------------------------------------------------------- *)

fun EACH_GOAL t  = 
  let fun tac (i, n, state) = 
  if i <= n
  then
    (SELECT_GOAL t i) APPEND (tac (i+1, n, state))
  else
           no_tac
  in Tactic(fn state => 
     if (length (prems_of state) <= 1) then
         (tapply(t, state)) 
     else
         (tapply(tac(1, length(prems_of state), state), state)) )end;


fun infer th =
   let val t =  term_list_to_term ((concl_of th) :: (prems_of th)); 
       val vars = filter is_Ide (Logic.add_term_vars (t,[]));
       val cvars = map (Sign.cterm_of (sign_of thy)) vars;
       val cx    = map (Sign.cterm_of (sign_of thy)) 
                         (newvars (length vars))
   in
       instantiate ([], mk_pair_list cvars cx) th
   end;


(* -------------------------------------------------------------------- *)
(*	 instance_unify_tac:						*)
(*		checks closure instance subgoals, to see whether 	*)
(*		a scheme variable will need to be instantiated to	*)
(*		a list, fn, pair or basic type.   			*)
(* -------------------------------------------------------------------- *)

exception INSTANCE;

fun unify (l, r) =
if is_schvar r then
   (if (is "TyVar_Type" l) orelse (is_schvar l) then
     raise INSTANCE
   else (l, r))
else if is "ConsType" r then
   let val (l1, l2) = dest_ConsType l;
       val (r1, r2) = dest_ConsType r
   in
   unify (l1, r1)
   handle _ => unify (l2, r2)
   end
else if is "fn_type" r then
   let val (l1, l2) = dest_fn_type l;
       val (r1, r2) = dest_fn_type r
   in
   unify (l1, r1)
   handle _ => unify (l2, r2)
   end
else if is "TypeSeq" r then
   let val (l1, l2) = dest_TypeSeq l;
       val (r1, r2) = dest_TypeSeq r
   in
   unify (l1, r1)
   handle _ => unify (l2, r2)
   end
else raise INSTANCE;


fun find_unify t =
let val (l, r) = dest_instance (dest_prop t)
in
if is "close" l then
  unify ((dest_close1 l), r)
else
  raise INSTANCE
end;

val ugly_int = ref 0;
fun newint () = (ugly_int := (!ugly_int) + 1; !ugly_int);

fun scheme_version (Const("ConsType", _) $ _ $ tyname) = 
   Const ("ConsType",Type ("fun",[Type ("Type",[]),
	Type ("fun",[Type ("TyName",[]),Type ("Type",[])])]))
          $ (Var (("ugly", newint()), Type ("Type",[]))) 
          $ tyname 
| scheme_version (Const("fn_type", _) $ _ $ _ ) = 
   (Const ("fn_type",
   Type ("fun",[Type ("Type",[]),
   Type ("fun",[Type ("Type",[]),Type ("Type",[])])])) 
          $ (Var (("ugly", newint()), Type ("Type",[])))
          $ (Var (("ugly", newint()), Type ("Type",[]))))
| scheme_version l = l;

val instance_unify_tac = 
FIRSTGOAL  
  (SUBGOAL (fn (prem,i) =>
   let val t = Logic.strip_assums_concl prem;
       val (l, r) = find_unify t
   in
   let
     val cvar = (Sign.cterm_of (sign_of thy)) r;
     val cx = (Sign.cterm_of (sign_of thy)) (scheme_version l)
   in
      PRIMITIVE (instantiate ([], [(cvar, cx)]))
   end
   end
   handle _ => no_tac));

val simplify_tac =
  (EACH_GOAL
  (DETERM
   (REPEAT1
    (is_bound_tac ORELSE
     is_Free_tac ORELSE
     is_free_tac ORELSE
     subst_tac ORELSE
     trivial_instance_tac ORELSE
     instance_tac)))) ORELSE
     (REPEAT1 (FIRSTGOAL (resolve_tac trivial_instance)));


(* -------------------------------------------------------------------- *)
(*	elaborate_tac:	repeatedly applies elaboration rules,		*)
(* -------------------------------------------------------------------- *)

val elaborate_tac  = REPEAT1 (reduce_tac ORELSE elab_step_tac);

(* -------------------------------------------------------------------- *)
(*	 elab_tac:	general-purpose tactic for proving goals by	*)
(*			depth-first search				*)
(* -------------------------------------------------------------------- *)

val elab_tac       = DEPTH_FIRST (has_fewer_prems 1) 
                       (elaborate_tac THEN TRY(REPEAT1 instance_unify_tac) 
			THEN TRY (REPEAT1 simplify_tac)) THEN
                     TRY (PRIMITIVE infer);


fun exp_elab E s = 
(print "\n---------------------------\
         \----------------------------------------------\n\
         \Trying to elaborate ....\n";
         goal thy (E^" |- "^s^" ~~> ?t"); by elab_tac);

fun dec_elab C s = 
(print "\n---------------------------\
         \----------------------------------------------\n\
         \Trying to elaborate ....\n";
         goal thy (C^" ||- "^s^" ~~> ?E"); by elab_tac);

fun prog_elab E s = 
(print "\n---------------------------\
         \----------------------------------------------\n\
         \Trying to elaborate ....\n";
         goal thy (E^" ||- "^s^" ~~> ?C [Program]"); by elab_tac);

val thy = thy;

end;
end;




