(* -------------------------------------------------------------*)
(*		Semantic Objects for Elaboration		*)
(* ------------------------------------------------------------ *)

signature OBJ =
sig
  structure SML_Syntax: SML_SYNTAX 
  val ntypes: (string list * (string list * string)) list
  val sext : Syntax.sext
  val thy : theory
  val show_closures : bool ref
  val show_envs : bool ref
  val show_contexts : bool ref
  val show_full_types : bool ref
end;

functor ObjFun (SML_Syntax: SML_SYNTAX) : OBJ = 
struct
structure SML_Syntax = SML_Syntax;
local open Syntax  
in

val simple_objects =  (* note - not all of these are used yet! *)
            [ "TyName",		(*  type names *)
              "StrName", 	(* structure names *)
	      "Id"];		(* general term for var, con and excon *)

val compound_objects =
            ["Type", 		(* type values *)
             "RecType",		(* record types *)
             "FunType",		(* function types *)
             "ConsType",	(* constructor types *)
             "TypeFcn",		(* type functions *)
             "TypeScheme", 	(* type schemes *)
             "Str",		(* structures *)
             "TyStr",		
             "VarEnv",		(* variable environments *)
				(* - includes constructor and *)
				(* exception constructor environments *)
             "StrEnv",		(* structure environments *)
             "TyEnv",		(* type environments *)
             "Env",		(*  environments  *)
             "TyNameSet",	(*  type name set *)
             "TyVarSet",	(*  type variable set *)
             "Context"];	(*  contexts *)

val auxiliary_objects =
	    ["TypeRow", 	(* type rows *)
	     "IdTypeSeq",	(* var/type binding sequences *)
	     "TyConTyStrSeq",	(* just don't ask *)
	     "TypeSeq",		(* type sequences *)
	     "TypeArgs"];	(* type arguments *)

val types = simple_objects @ compound_objects @ auxiliary_objects;

(* -------------------------------------------------------------------- *)
(*		 Elaboration - Semantic Objects	 			*)
(* -------------------------------------------------------------------- *)

val elab_mixfix =

  [ (* types as values *)

     Delimfix ("var _", "Var => Id", "VarId"),
     Delimfix ("con _", "Con => Id", "ConId"),
     Delimfix ("excon _", "ExCon => Id", "ExConId"),

     Delimfix ("_", "TyVar => Type", "TyVar_Type"),
     Delimfix ("{} inType", "Type", "empty_record_type"),
     Delimfix ("{_} inType", "TypeRow => Type", "record_type"),
     Delimfix ("_ = _", "[Lab, Type] => TypeRow ", "TypeRow1"),
     Delimfix ("_ = _ , _", "[Lab, Type, TypeRow] => TypeRow ", "TypeRow2"),
     Delimfix ("_ ->> _", "[Type, Type] => Type", "fn_type"),  (* NB!! *)
     Delimfix ("_ _", "[TypeSeq, TyName] => Type", "ConsType"),
     Delimfix ("INT", "TyName", "int_name"),      
     Delimfix ("BOOL", "TyName", "bool_name"),      
     Delimfix ("UNIT", "TyName", "unit_name"),      
     Delimfix ("EXN", "TyName", "exn_name"),      
     Delimfix ("list", "TyName", "list"),      
     Delimfix ("_ = /_ : tyvar", "[TyVar, TyVar] => o", "tyvar_equals"),

   (* type functions, schemes, structures etc *)

     Delimfix ("lam _._","[TyVarSeq, Type] => TypeFcn", "TypeFcn"),  
     Delimfix ("'(_,_')", "[TypeFcn, VarEnv] => TyStr", "TyStr"), 
                  (* NB : VarEnv includes ConEnv as subtype *)            

   (* type environments *)
      

     Delimfix ("{| |}", "VarEnv", "VarEnv_empty"),             
                  (* NB : VarEnv includes ConEnv, ExConEnv as subtypes *)
     Delimfix ("{| |}", "StrEnv", "StrEnv_empty"),             
     Delimfix ("{| |}", "TyEnv", "TyEnv_empty"),             
     Delimfix ("{| |}", "TyVarSet", "TyVarSet_empty"),             
     Delimfix ("{| |}", "TyNameSet", "TyNameSet_empty"),             
     Delimfix ("{| _ |}", "IdTypeSeq => VarEnv", "VarEnv"),             
     Delimfix ("VE0", "VarEnv", "VarEnv0"),            
     Delimfix ("TE0", "TyEnv", "TyEnv0"),            
     Delimfix ("lookup '(_,_,_')", "[Id, VarEnv, Type] => o", "lookup"), 
     Delimfix ("lookup'_tycon '(_,_,_')", "[TyCon, TyEnv, TyStr] => o", 
                         "lookup_tycon"), 
     Delimfix ("VE'_delete '(_,_,_')", 
           "[Id, VarEnv, VarEnv] => o", "VE_delete"),
     Delimfix ("TE'_delete '(_,_,_')", 
           "[TyCon, TyEnv, TyEnv] => o", "TE_delete"),
     Delimfix ("VE'_combine '(_,_,_')", "[VarEnv, VarEnv, VarEnv] => o", 
                         "VE_combine"), 
     Delimfix ("TE'_combine '(_,_,_')", "[TyEnv, TyEnv, TyEnv] => o", 
                         "TE_combine"),  
     Delimfix ("Combine '(_,_,_')", "[Env, Env, Env] => o", "Combine"), 
     Delimfix ("COMBINE '(_,_,_')", "[Context, Context, Context] => o", 
                         "COMBINE"), 
     Delimfix ("{| _ |}", "TyConTyStrSeq => TyEnv", "TyEnv"),             
     Delimfix ("subst '(_,_,_,_')", 
          "[Type, Type, TyVar, Type] => o", "subst"),
     Delimfix ("subst'_seq '(_, _, _, _')", 
          "[Type, TypeSeq, TyVarSeq, Type] => o", "subst_seq"),
     Delimfix ("is'_free '(_,_')", "[TyVar, Type] => o", "is_free"),
     Delimfix ("is'_Free '(_,_')", "[TyVar, VarEnv] => o", "is_Free"),
     Delimfix ("is'_bound '(_,_')", "[TyVar, Type] => o", "is_bound"),
     Delimfix ("Close '(_,_,_')", 
             "[Context, VarEnv, VarEnv] => o", "Close"),
     Delimfix ("<_,_,_,_>", "[StrEnv,TyEnv,VarEnv,VarEnv] => Env", "Env"),
                    (* NB : VarEnv includes ExConEnv as subtype *)
     Delimfix ("<_,_,_>", "[TyNameSet, TyVarSet,Env] => Context", "Context"),
     Delimfix ("...", "Env", " Env"),
     Delimfix ("...", "Context", " Context"),
     Delimfix ("_ As _", "[TyVarSeq, TypeSeq] => o", "As"),             
     Delimfix ("ABS '(_,_')", "[TyEnv, TyEnv] => o", "Abs"),             
     Delimfix ("ABS '(_,_,_')", "[TyEnv, Env, Env] => o", "ABS"),             
   
   (* type instances - for polymorphism *)

     Delimfix ("_ > _", "[Type, Type] => o", "instance"), 
     Delimfix ("close '(_,_')", "[Type, VarEnv] => Type", "close"), 
     Delimfix ("close '(_, ...')", "[Type, VarEnv] => Type", " close"), 

   (* sequents for type inference *)

     Delimfix ("(3_ /|- _ /~~> _)",		
	   "[Context,Exp,Type] => o", "Exp_Seq"),
     Delimfix ("(3_ /|- _ /~~> _ [ExpRow])",		
	   "[Context,ExpRow,TypeRow] => o", "ExpRow_Seq"),
     Delimfix ("(3_ /|- _ /~~> _ [Match])",		
	   "[Context,Match,Type] => o", "Match_Seq"),
     Delimfix ("(3_ /||- _ /~~> '(_,_') [Pat])",		
	   "[Context,Pat,VarEnv,Type] => o", "Pat_Seq"),
     Delimfix ("(3_ /||- _ /~~> '(_,_')  [PatRow])",		
	   "[Context,PatRow,VarEnv,Type] => o", "PatRow_Seq"),
     Delimfix ("(3_ /|- _ /~~> _    [Ty])",		
	   "[Context,Ty,Type] => o", "Ty_Seq"),
     Delimfix ("(3_ /|- _ /~~> _    [TySeq])",		
	   "[Context,TySeq,TypeSeq] => o", "TySeq_Seq"),
     Delimfix ("(3_ /|- _ /~~> _    [TyRow])",		
	   "[Context,TyRow,TypeRow] => o", "TyRow_Seq"),
     Delimfix ("(3_ /||- _ /~~> _)",
	   "[Context,Dec,Env] =>   o", "Dec_Seq"),
     Delimfix ("(3_ /||- _ /~~> _  [ValBind])",
	   "[Context,ValBind,VarEnv] =>   o", "ValBind_Seq"),
     Delimfix ("(3_ /||- _ /~~> _  [TypBind])",
	   "[Context,TypBind,TyEnv] =>   o", "TypBind_Seq"),
     Delimfix ("(3_ /||- _ /~~> _ , _ [DatBind])",
	   "[Context,DatBind,VarEnv,TyEnv] =>   o", "DatBind_Seq"),
     Delimfix ("(3_ , _ /||- _ /~~> _  [ConBind])",
	   "[Context,Type,ConBind,VarEnv] =>   o", "ConBind_Seq"),
                    (* NB : VarEnv includes ConEnv as subtype *)
    Delimfix ("(3_ /||- _ /~~> _  [ExBind])",
	   "[Context,ExBind,VarEnv] =>   o", "ExBind_Seq"),
                    (* NB : VarEnv includes ExConEnv as subtype *)
    Delimfix ("(3_ /||- _ /~~> _  [Program])",
	   "[Context,Program,Context] =>   o", "Program_Seq")];
     
(* -------------------------------------------------------------------- *)
(*		 Derived Forms and Chain Productions			*)
(* -------------------------------------------------------------------- *)

val derived_mixfix =
    [Delimfix ("'(- _ -')", "Type => Type", ""),
     Delimfix ("int", "Type", " int"),      
     Delimfix ("bool", "Type", " bool"),
     Delimfix ("{}", "Type", " unit"),
     Delimfix ("exn", "Type", " exn"),
     Delimfix ("<_>", "TypeArgs => TypeSeq", " TypeList"),
     Delimfix ("<>", "TypeSeq", " typeempty"),
      
     Delimfix ("_ * _", "[Type, Type] => Type", " pr_type")]; 


(* -------------------------------------------------------------------- *)
(*		  Sequences 						*)
(* -------------------------------------------------------------------- *)

val seq_mixfix =

     [Delimfix ("('(_ :/ _'))",  "[Id, Type] => IdTypeSeq" , "IdTypeSeq1"),
      Delimfix ("('(_ :/ _'), /_)", 
         "[Id, Type, IdTypeSeq] => IdTypeSeq ", "IdTypeSeq2"),
      Delimfix ("('(_ ,/ _'))",  
         "[TyCon, TyStr] => TyConTyStrSeq", "TyConTyStrSeq1"),
      Delimfix ("('(_ ,/ _'), /_)",  
         "[TyCon, TyStr, TyConTyStrSeq] => TyConTyStrSeq", "TyConTyStrSeq2"),
      Delimfix ("typeempty", "TypeSeq", "TypeSeq_Empty"), 
      Delimfix ("_ typecons _", "[Type, TypeSeq] => TypeSeq", "TypeSeq"),
      Delimfix ("_", "Type => TypeArgs", " TypeArgs1"), 
      Delimfix ("_, _", "[Type, TypeArgs] => TypeArgs", " TypeArgs2")];

(* -------------------------------------------------------------------- *)
(*		 The Complete Syntax 					*)
(* -------------------------------------------------------------------- *)

val mixfix = elab_mixfix @ derived_mixfix @ seq_mixfix;
 
(* -------------------------------------------------------------------- *)
(*		 Parse Translations for Derived Forms 			*)
(* -------------------------------------------------------------------- *)

fun pr_typetr [s, t] = 
    Const ("record_type",dummyT) $ (Const ("TypeRow2",dummyT) $
      Free ("n1",dummyT) $ s  
        $ (Const ("TypeRow1",dummyT) $ Free ("n2",dummyT) $  t));;

fun inttr [] = Const ("ConsType", dummyT) $ 
 Const("TypeSeq_Empty", dummyT) $ Const("int_name", dummyT);

fun booltr [] = Const ("ConsType", dummyT) $ 
  Const("TypeSeq_Empty", dummyT) $ Const("bool_name", dummyT);

fun unittr [] = Const ("ConsType", dummyT) $ 
  Const("TypeSeq_Empty", dummyT) $ Const("unit_name", dummyT);

fun exntr [] = Const ("ConsType", dummyT) $ 
  Const("TypeSeq_Empty", dummyT) $ Const("exn_name", dummyT);

fun typeemptytr [] = Const ("TypeSeq_Empty", dummyT);

fun TypeListtr [Const(" TypeArgs1", _) $ s] 
 = Const("TypeSeq", dummyT) $ s $ typeemptytr [] |
    TypeListtr [Const(" TypeArgs2", _) $ s $ t] 
 = Const("TypeSeq", dummyT) $ s $ TypeListtr [t] ;

(* -------------------------------------------------------------------- *)
(*	 	 Print Translations 			 		*)
(* -------------------------------------------------------------------- *)

fun record_typetr' [Const ("TypeRow2",_) $
      Free ("n1",_) $ s  $ (Const ("TypeRow1",_) $ Free ("n2",_) $  t)]
     =  Const(" pr_type", dummyT) $ s $ t |
    record_typetr' [s] = s;

fun ConsTypetr' [Const("TypeSeq_Empty", _), Const("int_name", _)] = 
      Const(" int", dummyT) |
    ConsTypetr' [Const("TypeSeq_Empty", _), Const("bool_name", _)] =          
      Const(" bool", dummyT) |
    ConsTypetr' [Const("TypeSeq_Empty", _), Const("unit_name", _)] =          
      Const(" unit", dummyT) |
    ConsTypetr' [Const("TypeSeq_Empty", _), Const("exn_name", _)] =          
      Const(" exn", dummyT);

fun TypeSeq_Emptytr' [] = Const(" typeempty", dummyT);

local fun f (Const("TypeSeq", _) $ s $  Const("TypeSeq_Empty", _)) 
  = Const(" TypeArgs1", dummyT) $ s |
          f (Const("TypeSeq", _) $ s $ t)
  =  Const(" TypeArgs2", dummyT) $ s $ (f t)
in 
    fun TypeSeqtr' [s, t] = 
    Const(" TypeList", dummyT) $ f (Const("TypeSeq", dummyT) $ s $ t)
end;

val show_closures = ref false;
val show_envs = ref false;
val show_contexts = ref false;
val show_full_types = ref false;

fun closetr' [t,E] =  
  if (!show_closures) then raise Match else Const (" close", dummyT) $ t; 

fun Envtr' [SE,TE,VE,EE] =  
  if (!show_envs) then raise Match else Const (" Env", dummyT); 

fun Contexttr' [T,U,E] =  
  if (!show_contexts) then raise Match else Const (" Context", dummyT); 

fun tr' [x] = 
  if (!show_full_types) then raise Match else x;

(* -------------------------------------------------------------------- *)
(*	 Isabelle Syntax for SML 					*)
(* -------------------------------------------------------------------- *)

val sext =
  Sext {mixfix=mixfix,
	   parse_translation=[ (" pr_type", pr_typetr), 
                               (" int", inttr), 
                               (" bool", booltr), 
                               (" unit", unittr), 
                               (" exn", exntr), 
                               (" typeempty", typeemptytr),
                               (" TypeList", TypeListtr)],
	   print_translation=[ ("record_type", record_typetr'),
                               ("ConsType", ConsTypetr'),
                               ("TypeSeq_Empty", TypeSeq_Emptytr'),
                               ("TypeSeq", TypeSeqtr'),
                               ("close", closetr'),
                               ("Env", Envtr'),
                               ("Context", Contexttr'),
                               ("VarId", tr'),
                               ("ConId", tr'),
                               ("ExConId", tr'),
                               ("empty_record_type", tr')] };

val ntypes = [(types, ([], "term"))];

val thy = extend_theory SML_Syntax.thy "OBJ"
     ([], [], [], ntypes, [], Some(sext)) [];

end;
end;
 

