(* -------------------------------------------------------------*)
(*		EvalRule.ml: 					*)
(*	Evaluation Inference Rules for the Theory FUNC		*)
(* ------------------------------------------------------------ *)


signature EVALRULE =
  sig
  structure FUNC_Syntax : FUNC_SYNTAX and
       EvalObj : EVALOBJ and Int_Rule: INT_RULE
  val sign: Sign.sg
  val thy: theory
  val show : string -> thm
  val iff_subst : (string * string) list 
  val congs : (string * string) list 
  val lookup : (string * string) list 
  val delete : (string * string) list
  val combine : (string * string) list
  val Combine : (string * string) list
  val unfold : (string * string) list 
  val record_lookup : (string * string) list
  val basval : (string * string) list
  val arith : (string * string) list
  val apply_plus : (string * string) list
  val apply_minus : (string * string) list
  val apply_times  : (string * string) list 
  val apply_less  : (string * string) list 
  val apply_not : (string * string) list
  val apply_zero : (string * string) list
  val apply_head : (string * string) list
  val apply_tail : (string * string) list
  val apply_null : (string * string) list
  val apply_cons : (string * string) list 
  val atexp_evaluation : (string * string) list
  val exprow_evaluation : (string * string) list
  val exp_evaluation : (string * string) list
  val match_evaluation : (string * string) list
  val mrule_evaluation : (string * string) list
  val dec_evaluation : (string * string) list
  val valbind_evaluation : (string * string) list
  val atpat_evaluation : (string * string) list
  val patrow_evaluation : (string * string) list
  val pat_evaluation : (string * string) list
  val program_evaluation : (string * string) list
 (*INSERT-RULESIG -- file produced by make-rulenames*)
end;

functor EvalRuleFun 
 (structure FUNC_Syntax : FUNC_SYNTAX and EvalObj : EVALOBJ 
      and Int_Rule: INT_RULE) : EVALRULE =  
struct
structure FUNC_Syntax = FUNC_Syntax
structure EvalObj = EvalObj
structure Int_Rule = Int_Rule
local open Syntax
in

(* -------------------------------------------------------------*)
(*		Inference Rules for Semantic Domains		*)
(* ------------------------------------------------------------ *)

val iff_subst =
    [("iff_subst",  "[| a <-> b; P(b) |] ==> P(a)")];

val congs =
    [("tr_cong",  "a <-> b ==>  tr a = tr b")];

val lookup = 
    [("lookup1_rule", 	
         " lookup (x, {| (x,v) |}, v)"),
     ("lookup2_rule", 	
         " lookup (x, {| (x,v), s |}, v)"),
     ("lookup3_rule", 	
         " lookup (x, {| s |}, v) ==> \
     \     lookup (x, {| (y,w), s |}, v)") ];

val delete =
    [("delete1_rule",
        "delete (x, {| |}, {| |})"),
     ("delete2_rule",
        "delete (x, {| (x,v) |}, {| |})"),
     ("delete3_rule",
        "delete (x, {| (y,v) |}, {| (y,v) |})"),
     ("delete4_rule",
        "delete (x, {| (x,v), s |}, {| s |})"),
     ("delete5_rule",
        "delete (x, {| s |}, {| s' |}) ==> \
     \   delete (x, {| (y,v), s |}, {| (y,v), s' |})")];


val combine =
    [("combine1_rule",
        "combine (VE, {| |}, VE)"),
     ("combine2_rule",
        "combine ({| |}, VE, VE)"),
     ("combine3_rule",
        "combine ({| |}, {| (x,v) |}, {| (x,v) |})"),
     ("combine4_rule",
        "combine ({| (x,v) |}, {| |}, {| (x,v) |})"),
     ("combine5_rule",
        "combine ({| (x,v) |}, {| (x,v') |}, {| (x,v') |})"),
     ("combine6_rule",
        "combine ({| (x,v) |}, {| (y,v') |}, {| (y,v'), (x,v) |})"),
     ("combine7_rule",
        "delete (x, {| s |}, {| s' |}) ==> \
        \ combine ({| s |}, {| (x,v) |}, {| (x,v),s' |})"),
     ("combine8_rule",
        "[| combine (VE, {| s |}, VE'); \
         \ combine (VE', {| (x,v) |}, VE'') |] ==> \
         \ combine (VE, {| (x,v), s |}, VE'')") ];

val Combine =
   [("Combine1_rule",
      "combine (VE,VE',VE'') ==> \
       \ Combine (<{| |}, VE, {| |}>, <{| |}, VE', {| |}>,  \
       \          <{| |}, VE'', {| |}>)")];

val unfold = 
    [("unfold1_rule",
        "unfold (VE, {| |},{| |})"),
     ("unfold2_rule",
        "unfold (VE, {| (f, <match,E,VE'>) |}, {| (f, <match,E,VE>) |}) "),
     ("unfold3_rule",
        "unfold (VE,{| s |}, {| s' |} ) ==> \
       \  unfold (VE, {| (f, <match,E,VE'>), s |}, \
       \             {| (f, <match,E,VE>), s' |}) ")];


val record_lookup = 
    [("record_lookup1_rule", 	
         " record_lookup (lab, {lab = v}, v)"),
     ("record_lookup2_rule", 	
         " record_lookup (lab, {lab = v, valrow}, v)"),
     ("record_lookup3_rule", 	
         " record_lookup (lab, {valrow}, v) ==> \
        \  record_lookup (lab, {lab' = w, valrow}, v)") ];

val basval =
    [ ("plus_val",
       "lookup (plus, E_init, val plus)"),
     ("minus_val",
       "lookup (minus, E_init, val minus)"),
     ("times_val",
       "lookup (times, E_init, val times)"),
     ("less_val",
       "lookup (less, E_init, val less)"),
     ("zero_val",
       "lookup (zero, E_init, val zero)"),
     ("not_val",
       "lookup (not, E_init, val not)"),
     ("head_val",
       "lookup (head, E_init, val head)"),
     ("tail_val",
       "lookup (tail, E_init, val tail)"),
     ("null_val",
       "lookup (null, E_init, val null)") ];

val arith =
    [("minus1",    "m #- m == 0"),
     ("minus2",	   "m  #- 0 == m"),
     ("minus3",	   "(succ m)  #- (succ n) == m  #- n"),
     ("mult1",	   "m  #* (succ 0) == m"),
     ("mult2",	   "(succ 0)  #* m == m")];

val apply_plus =
    [("plus_rule",
        "v = num m  #+ n ==> apply <val plus, (-num m, num n-), v>")];

val apply_minus =
    [ ("minus_rule",
        "v = num m  #- n ==> apply <val minus, (-num m, num n-), v>")];

val apply_times = 
    [ ("times_rule",
        "v = num m  #* n ==> apply <val times, (-num m, num n-), v>")];

val apply_less =
    [ ("less_rule",
        "v = tr (m #< n) ==> apply <val less, (-num m, num n-), v>")];

val apply_zero =
     [("zero1_rule",
        "v = tr True ==> apply <val zero, num 0,  v> "),
      ("zero2_rule",
        "v = tr False ==> apply <val zero, num succ n,  v> ")];

val apply_not =
    [ ("not1_rule",
        "v = tr False ==> apply <val not, tr True,  v> "),
      ("not2_rule",
        "v = tr True ==> apply <val not, tr False,  v> ")];

val apply_head =
    [ ("head_rule",
        "apply <val head, v :: L, v>")];

val apply_tail =
    [ ("tail_rule",
        "apply <val tail, v :: L, L>") ];

val apply_null =
   [  ("null1_rule",
        "v = tr False ==> apply <val null, w :: L, v>"),
      ("null2_rule",
        "v = tr True ==> apply <val null, [], v>") ];

val apply_cons =
    [ ("cons_rule",
        "apply <cons in Val, (-v, L-), v::L>") ]; 


(* -------------------------------------------------------------------- *)
(*									*)
(*		Operational Semantics for Phrase Evaluation		*)
(*		In the Definition, rules 103 to 159 cover		*)
(*		the dynamic semantics for the Core of SML		*)
(*									*)
(* -------------------------------------------------------------------- *)

val atexp_evaluation =
   [ ("Scon",					(* Rule 103  *)
	  "E |- scon n ~> num n  "), 
     ("Var1",					(* Rule 104a *)
	  "lookup (x,VE,v) <-> <SE, VE, EE> |-  var x ~> v  "),
     ("Var2",		(* predefined identifiers  -- Rule 104b *)
	  "lookup (x, E_init, v) <-> E |-  var x ~> v  "),
     ("True",					(* Rule 105a *)
	  "E |- con  true ~> tr True  "),
     ("False",					(* Rule 105b *)
	  "E |- con  false ~> tr False  "),
     ("Con",					(* Rule 105 *)
	  "E |- con  c ~> c in Val  "),
					  	 (* Rule 106 omitted *)
     ("Empty_Record",				(* Rule 107a *)
	  "E |- {} ~> {}  "),
     ("Record",					(* Rule 107b *)
	  "E |- exprow ~> valrow [ExpRow] <-> \
         \ E |- {exprow} ~> {valrow}  "),
     ("Let",					(* Rule 108  *)
        "(EX E' E''. E ||- dec ~> E' & Combine (E,E',E'') & \
          \ E'' |- exp ~> v) <-> \
          \ E |- let dec in exp end ~> v  "),
     ("Bracket",					(* Rule 109 *)
	"E |- exp ~> v <-> E |- (- exp -) ~> v ")];

val exprow_evaluation =
    [("ExpRow1",					(* Rule 110a *)
	  "E |- exp ~> v <-> E |- lab = exp ~> lab = v [ExpRow]"),
     ("ExpRow2",					(* Rule 110b *)
	  "(E |- exp ~> v & E |- exprow ~> valrow [ExpRow]) <-> \
          \ E |- lab = exp, exprow  ~> lab = v, valrow  [ExpRow]")];

val exp_evaluation =
					 (* Rule 111 is not needed   *)
    [("Con_Val",					(* Rule 112  *)
	  "(E |- exp ~> c in Val & E |- atexp ~> v) \
          \    <-> E |- exp`atexp ~> <c,v>"),
    					    (* Rules 113-115 omitted *)
     ("Apply1",					(* Rule 116  *)
	  "(EX b v'. E |- exp ~> b & E |- atexp ~> v' & \
       \   apply <b, v', v''>) <-> E |- exp`atexp ~> v''"),
     ("Apply2",					(* Rule 117a *)
          "(EX match E' VE v VE' E''.  \
        \  E |- exp ~> <match,E',VE> & E |- atexp ~> v & \
        \    unfold (VE,VE,VE') & \
        \  Combine (E', <{| |}, VE', {| |}>, E'')  &\
        \    E'', v |- match ~> v' in Val' [Match] ) <-> \
	\    E |- exp`atexp ~> v'"),
					 (* Rules 118 - 122 omitted *)
     ("Fn",				        (* Rule 123 *)
         "E |- fn match ~> <match,E,{| |}>")];

val match_evaluation =
    [("Match1",					(* Rule 124  *)
        "E, v |- pat => exp ~> v' in Val' [Match] <-> \
       \ E, v |- pat => exp | match ~> v' in Val' [Match]"),
					      (* Rule 125 not needed *)
     ("Match2",					(* Rule 126a *)
        "(E, v |- pat => exp ~> FAIL [Match] &   \
        \   E, v |- match ~> v' in Val' [Match]) <-> \
        \   E, v |- pat => exp | match ~> v' in Val' [Match]"),
     ("Match3",					(* Rule 126b *)
        "(E, v |- pat => exp ~> FAIL [Match] &   \
        \   E, v |- match ~> FAIL [Match] ) <-> \
        \   E, v |- pat => exp | match ~> FAIL [Match]")];

val mrule_evaluation =
    [("Mrule1",					(* Rule 127  *)
        "(EX VE E''. E, v ||- pat ~> VE in VarEnv'  [Pat] & \
        \  Combine (E, <{| |}, VE, {| |}>, E'') &  \
        \   E'' |- exp ~> v') <-> E,v |- pat => exp ~> v' in Val' [Match]"),
     ("Mrule2",					(* Rule 128  *)
        " E, v ||- pat ~> FAIL [Pat] <-> \ 
        \ E, v |- pat => exp ~> FAIL [Match]")];

val dec_evaluation =
    [("Val",					(* Rule 129 *)
	"E ||- valbind ~> VE [ValBind] <-> \		
	\ E ||- val valbind ~> <{| |}, VE, {| |}>"),
						 (* Rule 130 omitted *)
     ("Local",					(* Rule 131 *)
	" (EX E1 E'. E ||- dec1 ~> E1 & Combine (E, E1, E') & \
	\   E' ||- dec2 ~> E2)  <-> \
	\   E ||- local dec1 in dec2 end ~> E2"),
						 (* Rule 132 omitted *)
     ("Nothing",					(* Rule 133 *)
	"E ||- nothing ~> <{| |}, {| |}, {| |}>"),
     ("Comp1",					(* Rule 134a *)
	"(EX E1 E' E2. E ||- dec1 ~> E1 & Combine (E,E1,E') & \
	\   E' ||- dec2 ~> E2 & Combine (E1,E2,E3)) <-> \
	\   E ||- dec1  dec2 ~> E3"),
     ("Comp2",					(* Rule 134b *)
	"(EX E1 E' E2. E ||- dec1 ~> E1 & Combine (E,E1,E') & \
	\   E' ||- dec2 ~> E2 & Combine (E1,E2,E3) ) <-> \
	\   E ||- dec1 ; dec2 ~> E3")];

val valbind_evaluation =
    [("ValBind1",					(* Rule 135a *)
	"(EX v. E |- exp ~> v & E, v ||- pat ~> VE in VarEnv'  [Pat] ) <->  \
	\   E ||- pat = exp ~> VE [ValBind]"),
     ("ValBind2",					(* Rule 135b *)
	"(EX v VE VE'. E |- exp ~> v & E, v ||- pat ~> VE in VarEnv'  [Pat] & \
	\   E ||- valbind ~> VE' [ValBind] & combine(VE,VE',VE'') ) <->  \
	\   E ||- pat = exp and valbind ~> VE''  [ValBind]"),
						 (* Rule 136 omitted *)
     ("Rec",					(* Rule 137 *)
	"(EX VE. E ||- valbind ~> VE [ValBind] & unfold (VE,VE,VE')) \
        \   <-> E ||- rec valbind ~> VE' [ValBind]")];
				           (* Rules 138-139 omitted *)

val atpat_evaluation =
    [("Wild_Pat",					(* Rule 140  *)
         "E, v ||- _ ~>  {| |} in VarEnv' [Pat]"),
     ("Scon_Pat1",					(* -- Rule 141  *)
         "E, num n ||- scon n ~> {| |} in VarEnv' [Pat]"), 
     ("Scon_Pat2",					(* -- Rule 142  *)
         "E, v ||- scon n ~> FAIL [Pat]"),
     ("Var_Pat",					(* Rule 143  *)
         "E, v ||- var x ~>  {| (x,v) |} in VarEnv' [Pat]"),
     ("True_Pat",					(* -- Rule 144a  *)
         "E, tr True ||- con true ~> {| |} in VarEnv' [Pat]"),
     ("False_Pat",					(* -- Rule 144b  *)
         "E, tr False ||- con false ~> {| |} in VarEnv' [Pat]"),
     ("Con_Pat1",					(* -- Rule 144  *)
         "E, c in Val ||- con c ~> {| |} in VarEnv' [Pat]"),
     ("Con_Pat2",					(* -- Rule 145  *)
         "E, v ||- con c ~> FAIL [Pat]"),
				              (* Rules 146-147 omitted *)
     ("Empty_Record_Pat",				(* Rule 148a *)
         "E, {} ||- {} ~> {| |} in VarEnv' [Pat]"),
     ("Record_Pat1",				(* -- Rule 148b *)
         "E, r ||- patrow ~> VE in VarEnv' [PatRow]  \
        \  <-> E, r ||- {patrow} ~> VE in VarEnv' [Pat]"),
     ("Record_Pat2",				(* -- Rule 148c *)
         "E, r ||- patrow ~> FAIL [PatRow] \
        \  <-> E, r ||- {patrow} ~> FAIL [Pat]")];
					        (* Rule 149 not needed *)
val patrow_evaluation =
    [("Wild_PatRow",				(* Rule 150 *)
        "E, r ||- ... ~> {| |} in VarEnv' [PatRow]"),
     ("PatRow11",					(* Rule 151a *)
        "(EX v. record_lookup (lab, r, v) & E, v ||- pat ~> FAIL [Pat] ) <-> \
       \  E, r ||- lab = pat ~> FAIL [PatRow] "),
     ("PatRow21",					(* Rule 151b *)
        "(EX v. record_lookup (lab, r, v) & E, v ||- pat ~> FAIL [Pat] ) <-> \
       \ E, r ||- lab = pat, patrow ~> FAIL [PatRow] "),
     ("PatRow12",					(* Rule 152a *)
        "(EX v. record_lookup (lab, r, v) & \
         \ E, v ||- pat ~> VE in VarEnv' [Pat] ) \ 
         \ <-> E, r ||- lab = pat ~>  VE in VarEnv' [PatRow] "),
     ("PatRow22",					(* Rule 152b *)
        "(EX v VE VE'. record_lookup (lab, r, v) &   \
       \   E, v ||- pat ~> VE in VarEnv' [Pat] & \
       \   E, r ||- patrow ~> VE' in VarEnv' [PatRow] & \
       \     combine (VE, VE', VE'') ) <-> \
       \    E, r ||- lab = pat, patrow ~> VE'' in VarEnv' [PatRow] "),
     ("PatRow23",					(* Rule 152c *)
        "(EX v VE. record_lookup (lab, r, v) &  \
       \    E, v ||- pat ~> VE in VarEnv' [Pat] & \
       \    E, r ||- patrow ~> FAIL [PatRow] ) <-> \
       \    E, r ||- lab = pat, patrow ~> FAIL [PatRow] ")];

val pat_evaluation =
    [("Con_Val_Pat1",				(* Rule 154a *)
	"E, v' ||- atpat ~> VE in VarEnv' [Pat] <-> \
	\ E, <c,v'> ||- c`atpat ~> VE in VarEnv' [Pat]"),
     ("Con_Val_Pat2",				(* Rule 154b *)
	"E, v' ||- atpat ~> FAIL [Pat] <-> \
        \ E, <c,v'> ||- c`atpat ~> FAIL [Pat]") ];

val program_evaluation =
    [("Program1", 					(* Rule 196a *)
	"(EX E1. E ||- dec ~> E1 & Combine (E,E1,E') ) <-> \
      \     E ||- dec ; ~> E' [Program] "),
     ("Program2", 					(* Rule 196b *)
	"(EX E1. E ||- dec ~> E1 & Combine (E,E1,E') & \
	\   E' ||- program ~> E'' [Program] ) <->  \
	\   E ||- dec ; program ~> E'' [Program]")];
				              (* Rules 153-159 omitted *)


val sext = FUNC_Syntax.merge (FUNC_Syntax.sext, EvalObj.sext);

val sorts = FUNC_Syntax.sorts @ EvalObj.sorts;

val ntypes = FUNC_Syntax.ntypes @ EvalObj.ntypes;

val rules = iff_subst @ congs @
            lookup @
	    delete @
	    combine @
	    Combine @
	    unfold @ 
            record_lookup @
            basval @
            arith @
            apply_plus  @ apply_minus @ apply_times @ apply_less @
            apply_zero @ apply_not @ 
            apply_head @ apply_tail @ apply_null @ apply_cons   @ 
	    atexp_evaluation @
	    exprow_evaluation @
	    exp_evaluation @ match_evaluation @ mrule_evaluation @
            atpat_evaluation @ 
	    patrow_evaluation @
            pat_evaluation @  
	    dec_evaluation @ 
            valbind_evaluation @
	    program_evaluation; 

val thy = extend_theory Int_Rule.thy "FUNC"
     (sorts, [], ntypes, [], Some(sext)) rules;

val sign = sign_of thy;

fun show s = (print_depth 0; prth (get_axiom thy s));

val ax = get_axiom thy;

(*INSERT-RULENAMES -- file produced by make-rulenames*)

(*  temporary *)
(*

val thy = extend_theory thy "FUNC"
     ([], [], [], [],None) congs;

*)

end;
end;
