;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File: symbols.lsp
;;; System: HIPER
;;; Programmer: Jim Christian
;;; Date: April, 1989
;;; Copyright (c) 1989 by Jim Christian.  All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Symbol table management.

;; Function symbols are represented as integers beginning with
;; *max-all-vars*.  Sometimes it is useful to iterate
;; from 0 to some limit, so we have to convert the index
;; variable to a valid function symbol ID.
(defmacro to-fsym (int) `(the fixnum (+ *max-all-vars* ,int)))

;; The symbol field of a flatterm is an integer. From 0 to
;; *max-all-vars*, this represents an variable.  From *max-all-vars* to
;; *max-all-vars* + *max-fsyms*, this represents a function symbol.  So
;; we cushion the arrays associated with symbols with dummy slots, so
;; that accesses can be done directly, without subtraction.

(defvar *fsym-names*			;Print names
  #+kcl (make-array (+ *max-fsyms* *max-all-vars*) :static t)
  #-kcl (make-array (+ *max-fsyms* *max-all-vars*))
  )
(defvar *fsym-arities*			;Arities
  #+kcl (make-array (+ *max-fsyms* *max-all-vars*) :element-type 'fixnum :static t)
  #-kcl (make-array (+ *max-fsyms* *max-all-vars*) :element-type 'fixnum)
  )
(defvar *fsym-lex-weights*		;Lex weights for KB ordering
  #+kcl (make-array (+ *max-fsyms* *max-all-vars*) :element-type 'fixnum :static t)
  #-kcl (make-array (+ *max-fsyms* *max-all-vars*) :element-type 'fixnum)
  )
(defvar *fsym-weights*			;Symbol weights for KB ordering
  #+kcl (make-array (+ *max-fsyms* *max-all-vars*) :element-type 'fixnum :static t)
  #-kcl (make-array (+ *max-fsyms* *max-all-vars*) :element-type 'fixnum)
  )
(defvar *fsym-slot-nums*		;Offsets into discrimination net nodes
  #+kcl (make-array (+ *max-fsyms* *max-all-vars*) :element-type 'fixnum :static t)
  #-kcl (make-array (+ *max-fsyms* *max-all-vars*) :element-type 'fixnum)
  )
(defvar *fsym-mutations*		;Permutations sorted by function symbol
  #+kcl (make-array (+ *max-fsyms* *max-all-vars*) :static t)
  #-kcl (make-array (+ *max-fsyms* *max-all-vars*))
  )

;; Access operations for symbol arrays
(defmacro fsym-arity (i)
  `(the fixnum (aref (the (array fixnum) *fsym-arities*) (the fixnum ,i))))
(defmacro fsym-name (i)
  `(aref (the (array t) *fsym-names*) (the fixnum ,i)))
(defmacro fsym-weight (i)
  `(the fixnum (aref (the (array fixnum) *fsym-weights*) (the fixnum ,i))))
(defmacro fsym-lex-weight (i)
  `(the fixnum(aref (the (array fixnum) *fsym-lex-weights*) (the fixnum ,i))))
(defmacro fsym-slot-num (i)
  `(the fixnum (aref (the (array fixnum) *fsym-slot-nums*) (the fixnum ,i))))
(defmacro fsym-mutations (i)
  `(aref (the (array t) *fsym-mutations*) (the fixnum ,i)))

;; Fetch the integer ID of a function symbol, given its print-name
(proclaim '(function get-fsym-id (t) t))
(defun get-fsym-id (name)
  (sloop for i from (to-fsym 0)
	 below (to-fsym *num-fsyms*)
	 declare (fixnum i)
	 when (eq (aref (the (array t) *fsym-names*) i) name)
	 return i))

;; Initialize the symbol manager.
(proclaim '(function clear-fsyms () t))
(defun clear-fsyms ()
  (sloop for i from (to-fsym 0) below (to-fsym *num-fsyms*)
	 do (progn
	      (setf (aref *fsym-names* i) nil)
	      (setf (aref *fsym-arities* i) 0)
	      (setf (aref *fsym-weights* i) 0)
	      (setf (aref *fsym-lex-weights* i) 0)
	      (setf (aref *fsym-slot-nums* i) 0)
	      (setf (aref *fsym-mutations* i) nil)))
  (setf *num-fsyms* 0)
  (setf *new-fsym-counter* 0)
  )

;; Declare a new function symbol
(defun declare-fsym (sym arity &optional (weight 1) &aux id)
  (format t "~%Declaring symbol ~s/~s with weight ~s" sym arity weight)
  (unless (setf id (get-fsym-id sym))
	  (setf id (to-fsym *num-fsyms*))
	  (incf *num-fsyms*)
	  (setf (fsym-name id) sym))
  (setf (fsym-arity id) arity)
  (setf (fsym-weight id) weight)
  (setf (fsym-mutations id) nil)
  id
  )

;; Check that the declared arity of a symbol matches the
;; parsed arity.
(proclaim '(function check-arity (t t) t))
(defun check-arity (sym arity)
  (declare (type fixnum arity))
  (let ((id (get-fsym-id sym)))
    (when (not id)
	  (error "Undeclared symbol ~s" sym))
    (when (not (= (fsym-arity id) arity))
	  (error "Symbol ~s already declared with arity ~s"
		 sym
		 (fsym-arity id)))))


