;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File: reduce.lsp
;;; System: HIPER
;;; Programmer: Jim Christian
;;; Date: April, 1989
;;; Copyright (c) 1989 by Jim Christian.  All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Reduction.  

;; Reduction is terminated if an equation becomes trivial.
;; This flag is used for communication among routines.
(defvar *equal-sides* nil)  

;; Top-level hook.
(defmacro e-reduce (lhs rhs)
  `(if *equations*
       (e-reduce-by-all ,lhs ,rhs)
     (reduce-by-all ,lhs ,rhs)))


;; Reduce an equation by all rules in the system.
(proclaim '(function reduce-eqn (t) t))
(defun reduce-eqn (e &aux altered)
 ;(format t "~%Reducing ") (print-eqn e)(terpri)
  (setf *equal-sides* nil)
  (setf altered (e-reduce (eqn-lhs e) (eqn-rhs e)))
  (unless *equal-sides*
     (setf altered
	   (or (e-reduce (eqn-rhs e) (eqn-lhs e)) altered)))
  (if altered
      *equal-sides*
    (e-equal (eqn-lhs e) (eqn-rhs e))))

;; Reduce a term by all rules in the system.  The term is traversed
;; from right-to-left.  (Note that this isn't necessarily strictly 
;; bottom-up, but close enough for our purposes.)
(proclaim '(function reduce-by-all (t t) t))
(defun reduce-by-all (ft1 ft2
			  &aux redex altered lf-trail last-changed
			  new-redex)
   (setf redex (ft-end ft1))
   ;; Remember earliest quitting point.  If we come around to
   ;; it again without having made any reductions, then
   ;; the term is irreducible.
   (setf last-changed (ft-end ft1))
   (tagbody
    TOP
    (repeat (eq redex last-changed)
	    :return altered
	    (when (and (not (var? (ft-symbol redex)))
		       (or (and
			    (setf lf-trail (net-find-lhs redex *lhs-net* *single*))
			    (setf new-redex (ft-rewrite redex lf-trail)))
			   (and *pseudo-rules* 
				(setf new-redex (rewrite-with-pseudo-rules redex)))))
		  (setf redex new-redex)
		  (when (equal-terms ft1 ft2) ;Quit if reduced to trivial eqn
			(setf *equal-sides* t)
			(return-from reduce-by-all t))
		  (setf last-changed redex) ;Mark earliest stopping point
		  (setf altered t)
		  (go TOP))
	    (setf redex (ft-prev redex))
	    (unless redex (setf redex (ft-end ft1))))))

;; Reduce a term by all rules in the system, using E-matching.
;; The early-failure strategy of reduce-by-all can't be used because
;; ft1 might get permuted during the index lookup, and the last-changed
;; point would be invalid as a quitting point.
(proclaim '(function e-reduce-by-all (t t) t))
(defun e-reduce-by-all (ft1 ft2
			  &aux altered lf-trail (changed t))
  (while changed
    :return altered
    (setf changed nil)
    (do-ft-backwards ft1 :safely
       (when (and (not (var? (ft-sym)))
		  (or (and
		       (setf lf-trail (e-net-find-lhs (ft) *lhs-net* *single*))
		       (ft-rewrite (ft) lf-trail))
		      (and *pseudo-rules* 
			   (e-rewrite-with-pseudo-rules (ft)))))
	     (when (e-equal ft1 ft2)
		   (setf *equal-sides* t)
		   (return-from e-reduce-by-all t))
	     (setf altered t)
	     (setf changed t)))))

;; Find all equations reducible by the given rule, and queue them up
;; for a full reduction.  
(proclaim '(function find-reducible-rules (t) t))
(defun find-reducible-rules (rule
			     &aux matches r restart-fsyms nets is-prule lf)
  (setf nets (cons1 *lhs-net* (cons1 *subterm-net*
				     (cons1 *pseudo-rules-net* nil))))
  (setf is-prule (if (eq (eqn-type rule) '*pseudo-rule*) t nil))
  (dolist (net nets)
	  (setf matches (find-matches rule net t))
	  (while matches
	    (setf lf (pop1 matches))
	    (setf r (leaf-eqn lf))
	    (check-abort-pairs r)
	    (delete-eqn r)
	    (print-reducible r)
	    (queue-in-front r *pairs*)
	    (delete-from-nets r)
	    (when (and (not *disable-fsym-restart*)
		       (reducible-fsym-def-p rule r))
		  (pushnew (eqn-fsym-def-p r) restart-fsyms))
	    ))
  (free-list nets)
  restart-fsyms
  )
  
		



