;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File: permuters.lsp
;;; System: HIPER
;;; Programmer: Jim Christian
;;; Date: April, 1989
;;; Copyright (c) 1989 by Jim Christian.  All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Permutation of terms

;; Idea: For each fsym which has a permuter associated with it,
;; store a list of permutations (in the array *fsym-mutations*)
;; A permutation is an array of
;; fixnums denoting the permutation.  The fixnums are used as 
;; indices into a base array, which contains the original subterms
;; of the term being permuted. Hence, we don't need to do any
;; matching or copying; terms can be permuted in place.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; High-level permutation functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Generate the first permutation
(proclaim '(function e-permute (t) t))
(defun e-permute (term &aux perms base)
  (if (not (setf perms (fsym-mutations (ft-symbol term))))
      nil
    (progn
      (setf base (new-permutation))
      (init-permute term perms base)
      (permute term (pop perms) base)
      (cons1 term (cons1 perms (cons1 base nil))))))

;; Generate the next permutation, using the state returned by e-permute
(proclaim '(function re-permute (t) t))
(defun re-permute (state)
  (unless (second state)
	  (quit-permute (first state) (third state))
	  (free-permutation (third state))
	  (free-list state)
	  (return-from re-permute nil))
  (permute (first state) (pop (second state)) (third state))
  state)

;; Undo any permutation, using the state returned by e-permute or re-permute
(proclaim '(function stop-permute (t) t))
(defun stop-permute (state)
  (quit-permute (first state) (third state))
  (free-permutation (third state))
  (free-list state)
  nil)

;; Given a permutative equation, generate the permutations for it, and
;; save them in the *fsym-mutations* array.  After this routine
;; is called, e-unification, etc. will be activated.
;; Note that gen-permutations can be called on several equations
;; with the same function symbol; only the permutations which
;; haven't already been generated will be added to the mutation list
;; for the symbol.  For example, the equations f(x,y,z)=f(y,x,z)
;; and f(x,y,z)=f(x,z,y) can both be in effect for a ternary symbol f.
(proclaim '(function gen-permutations (t) t))
(defun gen-permutations (eqn &aux new-perms perms newp)
  (normalize eqn)
  (setf perms (fsym-mutations (ft-symbol (eqn-rhs eqn))))
  (setf newp (make-permutation eqn))
  (when (unique-permuter? newp perms)
	(push1 newp perms)
	(let ((changed t))
	  (while changed
	    (setf changed nil)
	    (setf new-perms
		  (nconc
		    (sloop for p in perms collect (next-permutation p newp))
		    (sloop for p in perms collect (next-permutation newp p))))
	    (sloop for p in new-perms
		   do (if (unique-permuter? p perms)
			  (progn (setf changed t)
				 (push1 p perms))
			(free-permutation p))))))
  (setf (fsym-mutations (ft-symbol (eqn-rhs eqn))) perms)
  )
 
;; T2 is a permutation of t1 if t2 contains the same symbols (and the
;; same number of occurrences of each symbol) as t1.
(proclaim '(function permutation? (t t) t))
(defun permutation? (t1 t2 &aux syms)
  (macrolet ((count-key (key list)
			(let ((x (gensym)))
			  `(let* ((,x (assoc ,key ,list)))
			     (if ,x
				 (incf (cdr ,x))
			       (push1 (cons1 ,key 1) ,list)))))
	     
	     (decf-key (key list)
		       (let ((x (gensym)))
			 `(let ((,x (assoc ,key ,list)))
			    (if (and ,x (> (cdr ,x) 0))
				(decf (cdr ,x))
			      nil)))))
	    
 (do-ft t1 (count-key (ft-sym) syms))
 (prog1 (and (do-ft t2 (unless (decf-key (ft-sym) syms) (abort)))
	     (sloop for s in syms always (zerop (cdr s))))
   (until (null syms) (free-cons (pop1 syms))))))
 


;; t1 is a simple linear permutation of t2 if t1 = f(x1,...,xn), 
;; t2 = f(y1,...,yn), the xi and yj are vars, and there is a 
;; permutation on the xi so that t1 = t2.
(proclaim '(function simple-linear-permuter? (t) t))
(with-stacks ((*trail* :trail))
(defun simple-linear-permuter? (e &aux (t1 (eqn-lhs e)) (t2 (eqn-rhs e)))
  (unless (and (ft-next t1) (ft-next t2))
	  (return-from simple-linear-permuter? nil))
  (with-local-trail *trail*
    (when
     (do-ft t1
       :skip-first
       (when (or (not (var? (ft-sym))) (bound-var? (ft-sym)))
	     (abort))
       (trail (ft-sym))
       (set-binding (ft-sym) 1))
     (do-ft t2
       :skip-first
       (when (or (not (var? (ft-sym))) 
		 (not (eql (var-binding (ft-sym)) 1)))
	     (abort))
       (set-binding (ft-sym) 2))))))

;;;;;;;;;;;;;;;;;;;;;;
;; Low level functions
;;;;;;;;;;;;;;;;;;;;;;

;; Set up for a permutation, but don't actually do any permutations 
;; This takes a flatterm (ft), the list stored in (fsym-mutations
;; (ft-symbol ft)), and a base permutation array (which can be gotten by
;; (new-permutation).  The base array is initialized to hold all the
;; top-level arguments of ft.
(proclaim '(function init-permute (t t t) t))
(defun init-permute (ft perms base &aux (t1 (ft-next ft))
			(numvars (aref (the (array t) (first perms))
				     *max-permuter-vars*)))
  (declare (type (array t) base))
  (setf (aref base *max-permuter-vars*) numvars)
  (setf t1 (ft-next ft))
  (floop :for i :below numvars
	 :do (progn (setf (aref base i) t1)
		    (setf t1 (ft-next (ft-end t1))))))

;; Generate the next permutation.  Ft is the flatterm passed to
;; init-permute, p is a member of the fsym-mutations of ft, and
;; base is the array initialized by init-permute.
(proclaim '(function permute (t t t) nil))
(defun permute (ft p base &aux last old-end new-end)
  (declare (type (array t) p base))
  (setf old-end (ft-end ft))
  (setf last (ft-next old-end))
  (setf (ft-end ft) ft)
  (floop :for i :below (aref p *max-permuter-vars*)
	 :do (ft-append ft (aref base (aref p i))))
  (setf new-end (ft-end ft))
  (setf (ft-next new-end) last)
  (when last (setf (ft-prev last) new-end))
  (setf ft (ft-prev ft))
  (while ft
    (when (eq (ft-end ft) old-end)
	  (setf (ft-end ft) new-end))
    (setf ft (ft-prev ft))))
	      
;; Restore ft to its original state.
(proclaim '(function quit-permute (t t) nil))
(defun quit-permute (ft base &aux last old-end new-end)
  (declare (type (array t) base))
  (setf old-end (ft-end ft))
  (setf last (ft-next old-end))
  (setf (ft-end ft) ft)
  (floop :for i :below (aref base *max-permuter-vars*)
	 :do (ft-append ft (aref base i)))
  (setf new-end (ft-end ft))
  (setf (ft-next new-end) last)
  (when last (setf (ft-prev last) new-end))
  (setf ft (ft-prev ft))
  (while ft
    (when (eq (ft-end ft) old-end)
	  (setf (ft-end ft) new-end))
    (setf ft (ft-prev ft))))

;; Build a permutation.
;; Assumes eqn has been normalized.
(proclaim '(function make-permutation (t) t))
(defun make-permutation (eqn &aux ft last (i 0) (p (new-permutation)))
  (declare (type (array t) p))
  (setf ft (eqn-rhs eqn))
  (setf last (ft-next (ft-end ft)))
  (setf ft (ft-next ft))
  (while (not (eq ft last))
    (setf (aref p i) (ft-symbol ft))
    (incf i)
    (setf ft (ft-next ft)))
  (setf (aref p *max-permuter-vars*) i)
  p)

;; Equality of permutations
(proclaim '(function same-permutation? (t t) t))
(defun same-permutation? (p1 p2)
  (declare (type (array t) p1 p2))
  (sloop nodeclare t for i below (aref p1 *max-permuter-vars*)
	 declare (fixnum i)
	 always (= (aref p1 i) (aref p2 i))))

;; Generate the next permutation from the current one.
(proclaim '(function next-permutation (t t) t))
(defun next-permutation (base p &aux (newp (new-permutation))
			      (numvars (aref base *max-permuter-vars*)))
  (declare (type (array t) base p newp))
  (setf (aref newp *max-permuter-vars*) numvars)
  (floop :for i :below numvars
	 :do (setf (aref newp i) (aref base (aref p i))))
  newp)

;; Check for identity permutation 
(proclaim '(function trivial-permutation? (t) t))
(defun trivial-permutation? (p)
  (declare (type (array t) p))
  (sloop nodeclare t for i below (aref p *max-permuter-vars*)
	 declare (fixnum i)
	 always (= (aref p i) i)))

;; See if permutation has already been generated
(proclaim '(function unique-permuter? (t t) t))
(defun unique-permuter? (p list)
  (and (not (trivial-permutation? p))
       (not (sloop for v in list thereis (same-permutation? v p)))))











