;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File: pairs.lsp
;;; System: HIPER
;;; Programmer: Jim Christian
;;; Date: April, 1989
;;; Copyright (c) 1989 by Jim Christian.  All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Generation of critical pairs.

;; For copying an equation when superposing it with itself.
(defvar *dummy-eqn* (new-eqn))


;; Top level.  Keep overlapping equations until at least one
;; critical pair is generated.
(proclaim '(function get-critical-pairs () nil))
(defun get-critical-pairs (&aux got-one)
  (until (or got-one (not (find-parents)))
	 (unless (or
		  (and *set-of-support*
		       (not (eqn-supported *P1*))
		       (not (eqn-supported *P2*)))
		  (and *passive-failures*
		       (or (eq (eqn-type *P1*) 'failure)
			   (eq (eqn-type *P2*) 'failure))))
		 (setf got-one (get-pairs)))))

;; Find candidates for critical pair generation.  Note that
;; we don't use discrimination nets for this.  I believe it
;; would randomize the strategy for selecting pairs, thus
;; resulting in more pairs being generated than with this 
;; prioritized scheme.
(proclaim '(function find-parents () t))
(defun find-parents ()
  (when (or (not *P1*) (eq (eqn-next *P1*) (queue-tail *marked-rules*)))
	(setf *P1* (queue-head *marked-rules*))
	(when *P2* (eqn-ys-to-xs *P2*))
	(when (setf *P2* (dequeue-eqn *rules*))
	      (enqueue-eqn *P2* *marked-rules*)
	      (eqn-xs-to-ys *P2*)))
  (when *P2*
	(setf *P1* (eqn-next *P1*))
	(when (eq *P1* *P2*)
	      (when (eqn-lhs *dummy-eqn*) (kill-ft (eqn-lhs *dummy-eqn*)))
	      (when (eqn-rhs *dummy-eqn*) (kill-ft (eqn-rhs *dummy-eqn*)))
	      (setf *P2* *dummy-eqn*)
	      (copy-eqn-slots *P1* *P2*)
	      (eqn-ys-to-xs *P1*))
	t))

;; Compute critical pairs between *P1* and *P2*
(proclaim '(function get-pairs () t))
(defun get-pairs (&aux tr result p)
  (cond
   ;; Compute restricted pairs if *P1* = *P2*
   ((= (the fixnum (eqn-id *P1*)) (the fixnum (eqn-id *P2*)))
    (do-ft (eqn-lhs *P1*)
      :skip-first
      (unless (var? (ft-sym))
	      (setf tr (e-unify (ft) (eqn-lhs *P2*)))
	      (while tr
		(when (setf p (new-pair (ft) *P1* *P2* tr))
		      (setf result t)
		      (enqueue-eqn p *pairs*))
		(setf tr (re-unify (ft) (eqn-lhs *P2*)))))))
   ;; Otherwise compute all pairs
   (t
    (do-ft (eqn-lhs *P1*)
      (unless (var? (ft-sym))
	      (setf tr (e-unify (ft) (eqn-lhs *P2*)))
	      (while tr
		(when (setf p (new-pair (ft) *P1* *P2* tr))
		      (setf result t)
		      (enqueue-eqn p *pairs*))
		(setf tr (re-unify (ft) (eqn-lhs *P2*))))))
    (do-ft (eqn-lhs *P2*)
      :skip-first
      (unless (var? (ft-sym))
	      (setf tr (e-unify (ft) (eqn-lhs *P1*)))
	      (while tr
		(when (setf p (new-pair (ft) *P2* *P1* tr))
		      (setf result t)
		      (enqueue-eqn p *pairs*))
		(setf tr (re-unify (ft) (eqn-lhs *P1*))))))))
   result)

;; Build a new critical pair and queue it for reduction.
;; When using a pseudo rule, only generate a new pair if
;; the rule yields a reduction with respect to the unifying
;; substitution.
(proclaim '(function new-pair (t t t t) t))
(defun new-pair (where p1 p2 trail &aux e new-lhs new-rhs old-lhs
		       (prule1 (eq (eqn-type p1) '*pseudo-rule*))
		       (prule2 (eq (eqn-type p2) '*pseudo-rule*)))
  
  (setf new-lhs (dupl-insert (eqn-lhs p1) (eqn-rhs p2) where))
  (setf new-rhs (dupl-ft-after-unify (eqn-rhs p1)))
  (when (or prule1 prule2)
	(setf old-lhs (dupl-ft-after-unify (eqn-lhs p1)))
	(when (or (and prule1 (not (e-greater old-lhs new-rhs)))
		  (and prule2 (not (e-greater old-lhs new-lhs))))
	      (kill-ft old-lhs)
	      (kill-ft new-lhs)
	      (kill-ft new-rhs)
	      (unless *equations* (restore-vars trail))
	      (return-from new-pair nil))
	(kill-ft old-lhs))
  (setf e (new-eqn))
  (setf (eqn-type e) '*new-pair*)
  (setf (eqn-id e) nil)
  (setf (eqn-rhs e) new-rhs)
  (setf (eqn-lhs e) new-lhs)
  (unless *equations* (restore-vars trail))
  (setf (eqn-parents e) (cons1 (eqn-id p1) (eqn-id p2)))
  (if (or (eqn-supported p1) (eqn-supported p2))
      (setf (eqn-supported e) t)
    (setf (eqn-supported e) nil))
  (normalize e)
  (incf-pair-counter)
  ;(print-new-pair e)
  e
  )

;; If *p1* or *p2* becomes reducible, then we can stop 
;; using it for critical pairs.
(proclaim '(function check-abort-pairs (t) nil))
(defun check-abort-pairs (e)
  (when (eq e *P1*)
	(setf *P1* (eqn-prev *P1*)))
  (when (eq e *P2*)
	(eqn-ys-to-xs *P2*)
	(setf *P2* nil)
	(setf *P1* nil)))

;; Initialize. 
(proclaim '(function init-pairs () nil))
(defun init-pairs ()
  (setf *P1* (eqn-prev (queue-tail *marked-rules*)))
  (setf *P2* nil))


