;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File: init.lsp
;;; System: HIPER
;;; Programmer: Jim Christian
;;; Date: April, 1989
;;; Copyright (c) 1989 by Jim Christian.  All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;; Load-time initialization.
(eval-when (load eval)
  (setf *pairs* (get-queue :pfunc1 #'pfunc-front
			   :pfunc2 #'pfunc-front))
  (setf *rules* (get-queue :indexed t))
  (setf *marked-rules* (get-queue :indexed t))
  (setf *lhs-net* (new-net))
  (setf *subterm-net* (new-net))
  (setf *failures-net* (new-net))
  (setf *pseudo-rules-net* (new-net))
  (setf *restart-q* (get-queue)))

;; Top level of HIPER.
(proclaim '(function complete (t) t))
(defun complete (file)
  (init-completion)
  (let ((eqs (load-file file)))
    (init-ordering)
    (init-nets)
    (init-pairs)
    (completion eqs)))


;; The main completion loop.
(proclaim '(function completion (t) t))
(defun completion (eqs &aux E fsym-list v1 v2)
  (setf (loop-counter) 0)
  ;; Queue the initial equations for reduction.
  (dolist (E eqs) (enqueue-eqn E *pairs*))
  ;; Remember input equations in case of a restart.
  (dolist (E eqs) (enqueue-eqn (dupl-eqn E) *restart-q*))
  
  (setf *start-time* (get-internal-run-time))
  
  (prog ()
  (until (queue-empty-p *pairs*)
      (until (queue-empty-p *pairs*)
	  (setf E (dequeue-eqn *pairs*))
	  (incf-loop-counter)
	  (cond

	   ;; Delete if reduces to trivial equation
	   ((reduce-eqn E)
	    (print-deleted E)
	    (kill-eqn E))

	   ;; Make into rewrite rule if orientable
	   ((e-orient E)
	    (unless (eqn-id E) (setf (eqn-id E) (incf-eqn-counter)))
	    (print-oriented E)
	    ;; Check for an equation of the form (false)=(true), 
	    ;; which implies that a proof has been found.
	    (when (and (eq (fsym-name (ft-symbol (eqn-lhs E))) 'false)
		       (eq (fsym-name (ft-symbol (eqn-rhs E))) 'true))
		  (go EXIT2))
	    (when *pause-for-user* (pause-for-user))
	    (setf (eqn-type E) '*rewrite-rule*)
	    (setf fsym-list (find-reducible-rules E))
	    (when fsym-list (new-fsym-restart fsym-list))
	    (enqueue-eqn E *rules*)
	    (index-rule E *lhs-net* *subterm-net*))
	   
	   ;; Delete E if it is subsumed by other failures or pseudo-rules
	   ((or (find-subsuming-eq E *failures-net*)
		(and *pseudo-rules* (find-subsuming-eq E *pseudo-rules-net*)))
	    (print-deleted E)
	    (kill-eqn E))

	   ;; Extend unification if E is a permuter
	   ((and *enable-permuters* (simple-linear-permuter? e))
	    (unless (eqn-id E) (setf (eqn-id E) (incf-eqn-counter)))
	    (print-equation e)
	    (push1 e *equations*)
	    (gen-permutations e)
	    (unify-restart))

	   ;; Try to introduce a new function symbol
	   ((new-fsym E)
	    (rebuild-nets))

	   ;; Try to add a new pseudo rule.
	   ((and (not *disable-pseudo-rules*)
		 (progn (setf v1 (vars-contained (eqn-lhs E) (eqn-rhs E)))
			(setf v2 (vars-contained (eqn-rhs E) (eqn-lhs E)))
			(or v1 v2)))
	    (setf (eqn-type E) '*pseudo-rule*)
	    (setf *pseudo-rules* t)
	    (unless (eqn-id E) (setf (eqn-id E) (incf-eqn-counter)))
	    (print-pseudo-rule E)
	    (when v1 (setf fsym-list (find-reducible-rules E)))
	    (when v2
		  (rotatef (eqn-lhs E) (eqn-rhs E))
		  (setf fsym-list (nconc fsym-list (find-reducible-rules E))))
	    (index-pseudo-rule E v1 v2))

	   ;; Defer E as a failure.
	   (t
	    (setf (eqn-type E) '*failure*)
	    (when fsym-list (new-fsym-restart fsym-list))
	    (unless (eqn-id E) (setf (eqn-id E) (incf-eqn-counter)))
	    (index-failure E)
	    (print-failure E))))
      
      (get-critical-pairs)
      )
	  
  EXIT1  ;Normal exit
  (setf *end-time* (get-internal-run-time))
  (format t "~%~% *** Completion terminated ***~%~%")
  (print-rules)
  (print-stats *end-time*)
  (return)

  EXIT2 ;Exit after a theorem has been proved
  (setf *end-time* (get-internal-run-time))
  (format t "~%~% *** Theorem Proved ***~%~%")
  (print-stats *end-time*)
  (return)
  ))


;; Print results
(proclaim '(function print-rules () t))
(defun print-rules ()
  (format t "~%~%Rules/Failures:")
  (do-queue e *marked-rules*
     (terpri)
     (case (eqn-type e)
	   (*failure* (princ "F"))
	   (*pseudo-rule* (princ "P"))
	   (t (princ " ")))
     (if (and *support-set* (eqn-supported e)) (princ "$") (princ " "))
     (print-eqn e))
  (do-queue e *rules*
     (terpri)
     (case (eqn-type e)
	   (*failure* (princ "F"))
	   (*pseudo-rule* (princ "P"))
	   (t (princ " ")))
     (if (and *support-set* (eqn-supported e)) (princ "$") (princ " "))
     (print-eqn e))
  (format t "~%~%Equations:")
  (dolist (e *equations*)
	  (terpri) (princ " ") (print-eqn e))
  (format t "~%~%Unprocessed pairs:")
  (do-queue e *pairs*
	  (terpri) (princ " ") (print-eqn e))
  (when (eq *which-order* 'rpo)
	(format t "~%~%Precedence:  ")
	(print-precedence *precedence*))
  t
  )

;; Initialize.
(proclaim '(function init-completion () nil))
(defun init-completion ()
  (init-vars)
  (do-queue e *rules*
	    (delete-eqn e) (delete-from-nets e) (kill-eqn e))
  (do-queue e *marked-rules*
	    (delete-eqn e) (delete-from-nets e) (kill-eqn e))
  (setf *lhs-net* (new-net))
  (setf *subterm-net* (new-net))
  (setf *failures-net* (new-net))
  (do-queue e *restart-q* (delete-eqn e) (kill-eqn e))
  (do-queue e *pairs* (delete-eqn e) (kill-eqn e))
  (setf *precedence* '(nil))
  (setf *equations* nil)
  (clear-fsyms)
  (setf (pair-counter) 0)
  (setf *pseudo-rules* nil)
  )

(proclaim '(function pause-for-user () nil))
(defun pause-for-user ()
  (format t "~%Hit a carriage return to continue...")
  (read-line))

(proclaim '(function dupl-failure (t) t))
(defun dupl-failure (e &aux copy)
  (setf copy (dupl-eqn e))
  (setf (eqn-type copy) '*failure*)
  (setf (eqn-parents copy) (copy-list (eqn-parents e)))
  (setf (eqn-backpointers copy) nil)
  (setf (eqn-next copy) nil)
  (setf (eqn-prev copy) nil)
  (setf (eqn-queue copy) nil)
  copy)


;; Add a failure to the indexing nets.
(proclaim '(function index-failure (t) t))
(defun index-failure (e)
  (index-unique-failure e)
  (enqueue-eqn e *rules*)
  (unless *passive-failures*
	  (let ((ecopy (dupl-failure e)))
	    (rotatef (eqn-lhs ecopy) (eqn-rhs ecopy))
	    (if (find-subsuming-eq ecopy *failures-net*)
		(kill-eqn ecopy)
	      (progn (index-unique-failure ecopy)
		     (enqueue-eqn ecopy *rules*)))))
  t)

;; Add a pseudo-rule to the indexing nets, along
;; with a reverse-direction copy if appropriate.
(proclaim '(function index-pseudo-rule (t t t) nil))
(defun index-pseudo-rule (e v1 v2 &aux ecopy)
  (cond
   ((and v1 v2)
    (index-rule e *pseudo-rules-net* *subterm-net*)
    (enqueue-eqn e *rules*)
    (setf ecopy (dupl-eqn e))
    (setf (eqn-type ecopy) '*pseudo-rule*)
    (setf (eqn-parents ecopy) (copy-list (eqn-parents e)))
    (setf (eqn-id ecopy) (incf-eqn-counter))
    (rotatef (eqn-lhs ecopy) (eqn-rhs ecopy))
    (index-rule ecopy *pseudo-rules-net* *subterm-net*)
    (enqueue-eqn ecopy *rules*))
   (v1
    (index-rule e *pseudo-rules-net* *subterm-net*)
    (enqueue-eqn e *rules*))
   (v2
    (rotatef (eqn-lhs e) (eqn-rhs e))
    (index-rule e *pseudo-rules-net* *subterm-net*)
    (enqueue-eqn e *rules*))))


;; After unification is extended, must re-queue all rules for reduction,
;; since they might now be reducible or non-orientable modulo the
;; new unification algorithm.  Also, new critical pairs might be
;; possible.
(proclaim '(function unify-restart () nil))
(defun unify-restart ()
  (format t "~%~%*** Restarting ...")
  (do-queue-backwards e *rules*
      (delete-eqn e)
      (delete-from-nets e)
      (enqueue-eqn e *pairs*))
  (do-queue-backwards e *marked-rules*
      (delete-eqn e)
      (delete-from-nets e)
      (enqueue-eqn e *pairs*))
  (when *p2* (eqn-ys-to-xs *p2*))
  (init-pairs)
  (format t " Done~%"))

;; Run-time statistics.
;; Useful for finding out stats in a break loop during a long or
;; non-terminating run.
(defun print-stats (&optional end-time)
  (unless end-time (setf end-time (get-internal-run-time)))
  (format t "~%~%Run time: ~F seconds"
	  (/ (- end-time *start-time*)
	     internal-time-units-per-second))
  (format t "~%  ~S Equations retained " (eqn-counter))
  (format t "~%  ~S Pairs generated " (pair-counter))
  (format t "~%  ~S Equations processed " (loop-counter))
  )
  









