;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File: equal.lsp
;;; System: HIPER
;;; Programmer: Jim Christian
;;; Date: April, 1989
;;; Copyright (c) 1989 by Jim Christian.  All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Term equality.

;; Ordinary equality.
(defmacro equal-terms (ft1 ft2)
  (let ((t1 (gentemp "T1")) (t2 (gentemp "T2")) (last (gentemp "LAST")))
    `(let* ((,t1 ,ft1) (,t2 ,ft2) (,last (ft-next (ft-end ,t1))))
       (while (not (eq ,t1 ,last))
	 :return t
	 (unless (same-symbol? (ft-symbol ,t1) (ft-symbol ,t2))
		 (break-from-loop nil))
	 (setf ,t1 (ft-next ,t1))
	 (setf ,t2 (ft-next ,t2))))))

;; Functional version of equality, for use as a :test parameter
;; or mapping function.
(proclaim '(function e-equal-func (t t) t))
(defun e-equal-func (ft1 ft2)
  (if *equations*
      (e-equal-terms ft1 ft2)
    (equal-terms ft1 ft2)))

;; E-equality, choosing the more efficient ordinary equality
;; if there are no permuters in the system yet.
(defmacro e-equal (ft1 ft2)
  `(if *equations*
       (e-equal-terms ,ft1 ,ft2)
     (equal-terms ,ft1 ,ft2)))

;; E-equality.  Using a backtracking algorithm, permute t1 until
;; it looks like t2.
(proclaim '(function e-equal-terms (t t) t))
(with-stacks ((*choicestack* :stack))
(defun e-equal-terms (t1 t2 
			 &aux 
			 (UC 0)
			 (chstack (get-c-array *choicestack*))
			 muts (sym1 0) (sym2 0) last)
  (declare (type fixnum sym1 sym2 UC))
  (declare (type (array t) chstack))

  (macrolet
   (
    (choice-t1 () `(c-aref chstack UC))
    (choice-t2 () `(c-aref chstack (the fixnum (+ UC 1))))
    (choice-muts () `(c-aref chstack (the fixnum (+ UC 2))))
    (choice-base () `(c-aref chstack (the fixnum (+ UC 3))))
    (set-choice-t1 (x) `(c-aset chstack UC ,x))
    (set-choice-t2 (x) `(c-aset chstack (the fixnum (+ UC 1)) ,x))
    (set-choice-muts (x) `(c-aset chstack (the fixnum (+ UC 2)) ,x))
    (set-choice-base (x) `(c-aset chstack (the fixnum (+ UC 3)) ,x))
    (next-choice () `(incf UC 4))
    (prev-choice () `(decf UC 4))
    (choice-stack-empty? () `(zerop UC))
    )
  
  (prog
   ()

   (setf last (ft-next (ft-end t1)))
   LOOP
   (while (not (eq t1 last))
     (setf sym1 (ft-symbol t1))
     (setf sym2 (ft-symbol t2))
     (unless (same-symbol? sym1 sym2)
	     (go BACKTRACK))
     
     ;; Build new choicepoint if appropriate
     (when (and (not (var? sym1))
		(setf muts (fsym-mutations sym1)))
	   (set-choice-t1 t1)
	   (set-choice-t2 t2)
	   (set-choice-muts muts)
	   (let ((b (new-permutation)))
	     (set-choice-base b)
	     (init-permute t1 muts b))
	   (next-choice)
	   )
     
       (setf t1 (ft-next t1))
       (setf t2 (ft-next t2))
       )
  
     ;; Save locals and succeed
   (until (zerop UC)
	  (prev-choice)
	  (quit-permute (choice-t1) (choice-base))
	  (free-permutation (choice-base)))
   (return-from e-equal-terms t)

   BACKTRACK
   ;; Fail if choice stack empty
   (when (choice-stack-empty?)
	 (return-from e-equal-terms nil))
   ;; Restore state
   (prev-choice)
   (setf t1 (choice-t1))
   (setf t2 (choice-t2))
   ;; Permute
   (unless (setf muts (choice-muts))
	   (quit-permute t1 (choice-base))
	   (free-permutation (choice-base))
	   (go BACKTRACK))
   (permute t1 (car muts) (choice-base))
   (set-choice-muts (cdr muts))
   (next-choice) ;Save choicepoint
   (setf t1 (ft-next t1))
   (setf t2 (ft-next t2))
   (go LOOP)
   ))))



