;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File: init.lsp
;;; System: HIPER
;;; Programmer: Jim Christian
;;; Date: April, 1989
;;; Copyright (c) 1989 by Jim Christian.  All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;; Automatic lexicographic recursive path ordering.  The 
;;; precedence relation is automatically extended in the 
;;; minimal way to accomodate new equations.
;;;   This procedure is based on Hassan Ait-Kaci's automatic
;;; RPO procedure, with some minor simplifications and improvements.
;;; The method simply mirrors the inductive
;;; definition of the RPO ordering in "rpo.lsp", at each 
;;; point trying the minimal precedence extension necessary for the 
;;; computation to proceed.  


;; Top level.  Try to orient e, first without extending
;; the precedence relation.  If that fails, then try
;; to extend the precedence relation.  If both orientations
;; are possible, then ask the user to pick one.
(proclaim '(function rpo-orient (t) t))
(defun rpo-orient (e)
  (when (rpo-greater (eqn-lhs e) (eqn-rhs e) *precedence*)
	(return-from rpo-orient t))
  (when (rpo-greater (eqn-rhs e) (eqn-lhs e) *precedence*)
	(rotatef (eqn-lhs e) (eqn-rhs e))
	(return-from rpo-orient t))
  (let (
	(o1 (when (vars-contained (eqn-lhs e) (eqn-rhs e))
		  (auto-rpo-greater (eqn-lhs e) (eqn-rhs e) *precedence*)))
	(o2 (when (vars-contained (eqn-rhs e) (eqn-lhs e))
		  (auto-rpo-greater (eqn-rhs e) (eqn-lhs e) *precedence*)))
	)
    (cond
     ((and o1 o2)
      (user-choose-orientation e o1 o2))
     ((and o1
	   *user-verify-status*
	   (sloop for x in o1 thereis (and (eq (car x) 'status)
					   (not (member x *precedence*)))))
      (when (user-verify-status e o1)
	    (setf *precedence* o1)
	    (return-from rpo-orient t)))
     ((and o2
	   *user-verify-status*
	   (sloop for x in o2 thereis (and (eq (car x) 'status)
					   (not (member x *precedence*)))))
      (rotatef (eqn-lhs e) (eqn-rhs e))
      (when (user-verify-status e o2)
	    (setf *precedence* o2)
	    (return-from rpo-orient t)))
     (o1
      (setf *precedence* o1)
      (return-from rpo-orient t))
     (o2
      (setf *precedence* o2)
      (rotatef (eqn-lhs e) (eqn-rhs e))
      (return-from rpo-orient t))
     (t
      (return-from rpo-orient nil)))))

(proclaim '(function user-choose-orientation (t t t) t))
(defun user-choose-orientation (e p1 p2 &aux answer)
  (format t "~%~%*** ")
  (print-eqn e)
  (format t " can be oriented both ways~%")
  (format t "  L->R if ")
  (print-precedence (set-difference p1 *precedence*)) (terpri)
  (format t "  R->L if ")
  (print-precedence (set-difference p2 *precedence*)) (terpri)
  (do ()(nil)
      (format t "~%Choose an orientation or postponement [lr rl p]~%")
      (setf answer (read))
      (case answer
	    (p
	     (return-from user-choose-orientation nil))
	    (lr
	     (setf *precedence* p1)
	     (return-from user-choose-orientation t))
	    (rl
	     (setf *precedence* p2)
	     (rotatef (eqn-lhs e) (eqn-rhs e))
	     (return-from user-choose-orientation t)))))

(proclaim '(function user-verify-status (t t) t))
(defun user-verify-status (e prec)
  (format t "~%~%*** ")
  (print-eqn e)
  (format t "~%  can be oriented left-to-right if ")
  (print-precedence (set-difference prec *precedence*))
  (format t ".~% Is this okay?")
  (y-or-n-p " "))


(proclaim '(function auto-multiargs-greater (t t t) t))
(defun auto-multiargs-greater (t1 t2 prec &aux ml1 ml2 prec2)
  (setf prec2 prec)
  (do-args t1 (push1 (ft) ml1))
  (do-args t2 (push1 (ft) ml2))
  (let* (
	 tmp
	 (m2-m1
	  (progn
	    (sloop for x in ml2
		   when (> (count x ml2 :test #'e-equal-func)
			   (count x ml1 :test #'e-equal-func))
		   do (push1 x tmp))
	    tmp))
	 (m1-m2
	  (progn
	    (setf tmp nil)
	    (sloop for x in ml1
		   when (> (count x ml1 :test #'e-equal-func)
			   (count x ml2 :test #'e-equal-func))
		   do (push1 x tmp))
	    tmp))
	 (result
	  (and m1-m2
	       (or
		(sloop for x in m2-m1
		       always (sloop for y in m1-m2
				     thereis (rpo-greater y x prec)))
		(sloop for x in m2-m1
		       always
		       (progn
			 (setf prec prec2)
			 (sloop
			  for y in m1-m2
			  thereis
			  (setf prec2 (auto-rpo-greater y x prec))))))))
	 )
    (free-list m2-m1)
    (free-list m1-m2)
    (free-list ml1)
    (free-list ml2)
    (when result prec2)))

;; Returns nil if failure, a precedence if success.
;; Precedence should not be nil; use (list nil) if necessary.
(proclaim '(function auto-rpo-greater (t t t) t))
(defun auto-rpo-greater (t1 t2 precedence
			    &aux prec
			    (sym1 (ft-symbol t1))
			    (sym2 (ft-symbol t2)))
  
  (cond
   ((var? sym1) nil)
   ((var? sym2)
    (when (occurs? sym2 t1)
	  precedence))
   (t
      (cond
       ((and (op-equal sym1 sym2 precedence)
	     (op-status sym1 precedence))
	(auto-lex-rpo t1 t2 precedence))
       ((op-equal sym1 sym2 precedence)
	(setf prec (auto-multiargs-greater t1 t2 precedence))
	(when prec (return-from auto-rpo-greater prec))
	(unless (or (< (fsym-arity sym1) 2) (fsym-mutations sym1))
		(setf prec (auto-lex-rpo t1 t2 (cons1 `(status ,sym1 lr)
						      precedence)))
		(if prec
		    prec
		  (auto-lex-rpo t1 t2 (cons1 `(status ,sym1 rl)
					     precedence)))))
       ((op-greater sym1 sym2 precedence)
	(when (all-args t2
		(setf precedence
		      (auto-rpo-greater t1 (ft) precedence)))
	      precedence))
       ((op-greater sym2 sym1 precedence)
	(when
	 (some-arg t1
	     (setf prec precedence)
	     (or (equiv-terms (ft) t2 precedence)
		 (setf prec
		       (auto-rpo-greater (ft) t2 precedence))))
	 prec))
       ((some-arg t1
		(or (equiv-terms (ft) t2 precedence)
		    (rpo-greater (ft) t2 precedence)))
	precedence)
       ((some-arg t1
	  (setf prec precedence)
	  (or (equiv-terms (ft) t2 precedence)
	      (setf prec
		    (auto-rpo-greater (ft) t2 precedence))))
	prec)
       ((not (op-greater sym2 sym1 precedence))
	(setf precedence (cons1 `(> ,sym1 ,sym2) precedence))
	(when (all-args t2
		 (setf precedence
		       (auto-rpo-greater t1 (ft) precedence)))
	      precedence))
      ))))

(proclaim '(function auto-lex-rpo (t t t) t))
(defun auto-lex-rpo (t1 t2 precedence &aux list1 list2
					   prec result)
  (do-args t1 (push1 (ft) list1))
  (do-args t2 (push1 (ft) list2))
  (unless (eq (op-status (ft-symbol t1) precedence) 'rl)
	  (setf list1 (rev1 list1))
	  (setf list2 (rev1 list2)))
  (while (equiv-terms (car list1) (car list2) precedence)
    (pop1 list1)
    (pop1 list2))
  (setf result
	(cond
	 ((null list1) nil)
	 ((null list2) precedence)
	 ((setf prec
		(auto-rpo-greater (pop1 list1) (pop1 list2) precedence))
	  (when
	   (sloop for x in list2
		  always (setf prec (auto-rpo-greater t1 x prec)))
	   prec))
	 ((sloop for x in list1
		 thereis (or (equiv-terms x t2 precedence)
			     (rpo-greater x t2 precedence)))
	  precedence)
	 ((sloop for x in list1 thereis
		 (progn
		   (setf prec precedence)
		   (or (equiv-terms x t2 precedence)
		       (setf prec (auto-rpo-greater x t2 precedence)))))
	  prec)
	 ))
  (free-list list1)
  (free-list list2)
  result)

	
  
