;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File: arrays.lsp
;;; System: HIPER
;;; Programmer: Jim Christian
;;; Date: April, 1989
;;; Copyright (c) 1989 by Jim Christian.  All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Stuff for fast handling of arrays in Kyoto Common Lisp.
;;; Ultimately, we want to allocate stacks directly on the C call stack,
;;; and access them directly using array[index] instead of
;;; array->v.v_self[index].
;;; For now, we'll ignore local allocation.

;;; Two kinds of single-dimension arrays are supported: Those of type
;;; (array t) and those of type (array fixnum).


;; Hook to the KCL compiler, to tell it what C code to emit when
;; compiling particular functions with safety 0.
#+kcl (defmacro make-inline (name args val side-effect new-obj code)
	`(eval-when (load compile eval)
		    (push '(,args ,val ,side-effect ,new-obj ,code)
			  (get ',name 'compiler::inline-unsafe))))

(defmacro declare-c-array (x)
  `(declare (object ,x) (type (array t) ,x)))
(defmacro declare-fixnum-array (x)
  `(declare (object ,x) (type (array fixnum) ,x)))


;; Get hold of the body of an array whose elements can 
;; be of any lisp type
#+kcl (proclaim '(function get-c-array ((array t)) (array t)))
#+kcl (defun get-c-array (a) a)
#+kcl (make-inline get-c-array ((array t)) (array t) nil nil
		   "((object)((#0)->v.v_self))")
#-kcl (defmacro get-c-array (a) `,a)


;; Get hold of the body of a fixnum array
#+kcl (proclaim '(function get-fixnum-array ((array fixnum)) (array fixnum)))
#+kcl (defun get-fixnum-array (a) a)
#+kcl (make-inline get-fixnum-array ((array fixnum)) (array fixnum) nil nil
		   "((object)((#0)->fixa.fixa_self))")
#-kcl (defmacro get-fixnum-array (a) `,a)



;; Reference the body of an object array
#+kcl (proclaim '(function c-aref ((array t) fixnum) t))
#+kcl (defun c-aref (a n)
	(aref (the (array t) a) (the fixnum n)))
#+kcl (make-inline c-aref ((array t) fixnum) t nil nil
		   "(((object *)#0)[#1])")
#-kcl (defmacro c-aref (a n)
	`(aref (the (array t) ,a) (the fixnum ,n)))



;; Reference the body of a fixnum array
#+kcl (proclaim '(function f-aref ((array fixnum) fixnum) fixnum))
#+kcl (defun f-aref (a n)
	(aref (the (array fixnum) a) (the fixnum n)))
#+kcl (make-inline f-aref ((array fixnum) fixnum) fixnum nil nil
		   "(((fixnum *)#0)[#1])")
#-kcl (defmacro f-aref (a n)
	`(aref (the (array fixnum) ,a) (the fixnum ,n)))



;; Set the element of an object array
#+kcl (proclaim '(function c-aset ((array t) fixnum t) t))
#+kcl (defun c-aset (a n v)
	(setf (aref (the (array t) a) (the fixnum n)) v))
#+kcl (make-inline c-aset ((array t) fixnum t) t t nil
		   "((object *)(#0))[#1] = (object)(#2)")
#-kcl (defmacro c-aset (a n v)
	`(setf (aref (the (array t) ,a) (the fixnum ,n)) ,v))


;; Set the element of a fixnum array.
#+kcl (proclaim '(function f-aset ((array fixnum) fixnum fixnum) fixnum))
#+kcl (defun f-aset (a n v)
	(setf (aref (the (array fixnum) a) (the fixnum n)) (the fixnum v)))
#+kcl (make-inline f-aset ((array fixnum) fixnum fixnum) fixnum t nil
		   "((fixnum *)(#0))[#1] = (#2)")
#-kcl (defmacro f-aset (a n v)
	`(setf (aref (the (array fixnum) ,a) (the fixnum ,n)) (the fixnum ,v)))


;; Push an object on to an object array
(defmacro c-stack-push (x stk i)
  `(progn (c-aset ,stk ,i ,x)
	  (incf ,i)))

;; Pop an object array
(defmacro c-stack-pop (stk i)
  `(progn (decf ,i)
	  (c-aref ,stk ,i)))

;; Push a fixnum onto a fixnum array
(defmacro f-stack-push (x stk i)
  `(progn (f-aset ,stk ,i ,x)
	  (incf ,i)))

;; Pop a fixnum array
(defmacro f-stack-pop (stk i)
  `(progn (decf ,i)
	  (f-aref ,stk ,i)))



#|
;; Usage example (for fixnum arrays; other arrays are handled
;; in the same way, using get-c-array, c-aref, c-aset,
;; and declare-c-array.
;;###Apparently this example must be compiled in another
;; file in order for the AKCL compiler to generate the
;; correct inline code.
;; (time (test1)) --> 7.983 seconds on Sun 4/280
;; (time (test2)) --> 12.283 seconds
;; ==> Direct array access can be a substantial win.

(defvar *fstack*)
(eval-when (load compile eval)
   (setf *fstack*
	 ;; Must use :static allocation, else a gc could
         ;; relocate the array body and leave the local
	 ;; handle dangling.
	 (make-array '(100) :element-type 'fixnum :static t)))

(proclaim '(function test1 () fixnum))
(defun test1 (&aux (fstack (get-fixnum-array *fstack*))
		   (k 0))
  (declare-fixnum-array fstack)
  (declare (type fixnum k))
  (sloop for j from 1 to 100000
	 do (progn
	      (setf k 0)
	      (sloop for x from 1 to 100
		     declare (fixnum x)
		     do (block ()
			       (incf k (f-aref fstack x))
			       (f-aset fstack x (f-aref fstack 0)))
		     )))
  k)

(proclaim '(function test2 () fixnum))
(defun test2 (&aux (fstack *fstack*)
		   (k 0))
  (declare (type (array fixnum) fstack) (object fstack))
  (declare (type fixnum k))
  (sloop for j from 1 to 100000
	 do (progn
	      (setf k 0)
	      (sloop for x from 1 to 100
		     declare (fixnum x)
		     do (block ()
			       (incf k (aref fstack x))
			       (setf (aref fstack x) (aref fstack 0)))
		     )))
  k)

|#

