
#include "ft.h"



/*************************** unify ***********************************/


/*
   To understand the code below, the representation of terms must be
   recalled:

   The routines deal with pointers to the three kinds of terms: variables,
   parameters, and substitution terms.
   
   A variable consists of two longs: the restriction value plus 1, followed
   by a pointer to the value of the variable, or NULL if the variable is
   unbound. A variable with restriction value n must not be bound to any
   term containing any prm(k) for k>=n.

   A parameter consists of two longs: the number 0 followed by the number
   k for which the parameter is prm(k).

   A substitution term consists of three longs: the number -1, followed
   by a pointer to a base term, followed by a pointer to a substitution
   table. Thus the three types of term are distinguished by the sign of
   the initial long.

   A base term is either a syntactic variable or a functional term.
   A syntactic variable is a longword containing a value <=BMAX.
   A functional term is a series of consecutive longs beginning with
   a function symbol - a value > BMAX - followed by the arity, followed
   by pointers to the arguments, which are also base terms. Atomic
   base formulas are represented similarly, with a beginning predicate
   symbol (a value > FMAX) instead of function symbol.
   
   The substitution table is used to obtain the value substituted for
   a syntactic variable. It is a table of longs of length N, where N is
   the total number of syntactic variables in the input formula. The
   k:th value in the table is a pointer to the term substituted for the
   syntactic variable k, or NULL if nothing has yet been substituted for
   that variable. What is substituted for syntactic variables is always
   either a variable (in the above sense) or a parameter.

   The function getvalvar(s) returns a pointer to the current value of
   the variable s (i.e. the function follows the chain of references
   until it reaches an unbound variable, a parameter, or a substitution
   term). getvalsub(i,subt) returns a pointer to the value of the term
   substituted for the syntactic variable i according to the substitution
   table subt.

   The syntax checking of input formulas ensures that no function symbol
   or predicate symbol occurs with different arities. Also, all bound
   formulas are made unique, so that substitutions of terms for free
   occurrences of x in a formula can be made by substituting for all
   occurrences of x. */




/* unify(sb,ss,tb,ts,&q) attempts to unify the substitution terms or atomic 
   formulas sb,ss and tb,ts and returns 0 for failure, 1 for success. 
   In addition, the variables bound in the course of the unification are
   appended to the list q. The calling function must initialize q and also
   unbind q if the unification fails. unify() is only used in checking for
   atomic axioms. It is never applied to falsum. The lists and terms created in
   the course of the unification are stored on stack 3. */



unify(sbase,ssub,tbase,tsub,q)
     basefmla sbase,tbase;
     subtab ssub,tsub;
     list *q;
{
  long i,j,args;
  term s,t,getvalsub();
  variable vmin,vmax;
  int k;
#ifdef DEBUG
  if (trace) mess("\nunify");
#endif
  i=sbase[0];     /* each of i and j is either a function symbol or a */
  j=tbase[0];     /* predicate symbol or a syntactic variable         */
  /* first the case where the first term is composite */
  if (i>BMAX)
    {
      if (j>BMAX)   /* both terms composite */
	{
	  if (i!=j)
	    return 0;
	  for(args=sbase[1];args>0;args--)
	    {
	      if
	((k=unify((baseterm)sbase[1+args],ssub,(baseterm)tbase[1+args],tsub,q))<=0)
		  return k;
	    }
	  return 1;
	}
      /* only the first term composite */
      t=getvalsub(j,tsub);
      if (var(t))
	return checkp(t,sbase,ssub,q);
      if (param(t))
	return 0;
      /* so t is a substitution term */
      return unify((baseterm)t[1],(subtab)t[2],sbase,ssub,q);
    }
  /* next the case when only the second is composite */
  if (j>BMAX)
    {
      s=getvalsub(i,ssub);
      if (var(s))
	return checkp(s,tbase,tsub,q);
      if (param(s))
	return 0;
      return unify((baseterm)s[1],(subtab)s[2],tbase,tsub,q);
    }
  /* so neither is composite */
  s=getvalsub(i,ssub);
  t=getvalsub(j,tsub);
  /* the case where s is a variable */
  if (var(s))
    {
      if (var(t))  /* two-variable case */
	{
	  if (s==t)
	    return 1;
	  if (s[0]<t[0])
	    {
	      vmin=t;
	      vmax=s;
	    }
	  else
	    {
	      vmin=s;
	      vmax=t;
	    }
	  /* bind the less restricted variable to the more restricted */
	  sp3[-1]=(long)vmin;     /* append vmin to the list *q */
	  sp3[0]=(long)*q;
	  *q=sp3-1;
	  sp3-=2;
	  vmin[1]=(long)vmax;
	  return 1;
	}
      /* s variable, t parameter */
      if (param(t))
	{
	  if (t[1]>=s[0]-1)
	    return 0;
	  /* bind s to t */
	  sp3[-1]=(long)s;
	  sp3[0]=(long)*q;
	  *q=sp3-1;
	  sp3-=2;
	  s[1]=(long)t;
	  return 1;
	}
      /* thus s variable and t composite */
      return checkp(s,(baseterm)t[1],(subtab)t[2],q);
    }
  /* the case where s is a parameter */
  if (param(s))
    {
      if (var(t))
        {
	  if (s[1]>=t[0]-1)
	    return 0;
	  /* bind t to s */
	  sp3[-1]=(long)t;
	  sp3[0]=(long)*q;
	  *q=sp3-1;
	  sp3-=2;
	  t[1]=(long)s;
	  return 1;
        }
      if (param(t) && s[1]==t[1])
	return 1;
      return 0;
    }
  /* so s is composite */
  if (param(t))
    return 0;
  if (var(t))
    return checkp(t,(baseterm)s[1],(subtab)s[2],q);
  /* both are composite */
  return unify((baseterm)t[1],(subtab)t[2],(baseterm)s[1],(subtab)s[2],q);
}




checkp(s,tbase,tsub,q)  /* bind variable s to subterm t if this is ok */
     variable s;
     baseterm tbase;
     subtab tsub;
     list *q;
{
  int i;
  if (i=admissible(s,s[0]-1,tbase,tsub,q))  /* including variable adjustment */
    {
      if (i<0)
	return i;
      sp3[-4]=(long)s;   /* add s to list */
      sp3[-3]=(long)*q;
      *q=sp3-4;
      sp3[-2]=(-1L);     /* make subterm */
      sp3[-1]=(long)tbase;
      sp3[0]=(long)tsub;
      s[1]=(long)(sp3-2);  /* bind s */
      sp3-=5;
      return 1;
    }
  return 0;
}



admissible(s,res,tbase,tsub,q)
     variable s;
     long res;
     baseterm tbase;
     subtab tsub;
     list *q;
{
  long args,i;
  term t,getvalsub();
  int k;
  if (cflag)
    {
      cflag=0;
      return ABORT;
    }
  if ((i=tbase[0])>BMAX)
      {
        for(args=tbase[1];args>0;args--)
	  {
	    if ((k=admissible(s,res,(baseterm)tbase[1+args],tsub,q))<=0)
	      return k;
	  }
        return 1;
      }
  t=getvalsub(i,tsub);
  if (param(t))
    {
      if (t[1]>=res) return 0;
      return 1;
    }
  if (var(t))
    {
      if (s==t)  /* occur check here */
	return 0;
      if (t[0]-1>res) /* this variable is too permissive */
        {
	  sp3[-3]=(long)t;
	  sp3[-2]=(long)*q;
	  *q=sp3-3;
	  sp3[-1]=res+1;  /* make new variable */
	  sp3[0]=NULL;
	  t[1]=(long)(sp3-1);
	  sp3-=4;
        }
      return 1;
    }
  return admissible(s,res,(baseterm)t[1],(subtab)t[2],q);
}




/****************************** copy *************************************/


/* copyto4(s,&q) copies the value of the term s to stack 4 and returns the
   address of the copy. To make it possible to copy several terms while
   retaining overlapping variables, variables in s are bound to their
   copies on stack 4 and appended to the list q, initialized by the
   calling function. Copies of variables are tagged, to indicate where
   copying is to stop.The calling function must undo bindings and remove tags.
   Note that the representation of terms implies that there is no
   simple way of achieving the same effect by constructing a new term.
   copyto4() leaves a lot of garbage on the main stack, to be removed
   by the calling function. Note that a top level call to copyto4 must
   be applied to a term s not already on stack 4. */




term copyto4(s,q)
     term s;
     list *q;
{
  baseterm sbase;
  subtab ssub;
  long i;
  term getvalvar();
  register long *rsp4=sp4;
  if (var(s)&&s[1])
    s=getvalvar(s);
  if (param(s))         /* copy parameter */
    {
      rsp4[0]=0L;
      rsp4[1]=s[1];
      sp4+=2;
      return rsp4;
    }
  if (var(s))           /* copy variable */
    {
      if (s[0]>BARVAL)
	return s; /* no copying of tagged variables */
      rsp4[0]=s[0]+BARVAL;  /* make new variable on stack 4 */
      rsp4[1]=NULL;
      sp4+=2;
      sp[0]=(long)s;  /* append s to *q */
      sp[1]=(long)*q;
      *q=sp;
      sp+=2;
      s[1]=(long)rsp4;  /* bind s to the copy */
      return rsp4;
    }
  sbase=(baseterm)s[1];  /* copy substitution term */
  ssub=(subtab)s[2];
  rsp4[0]=(-1L);        /* identifying tag for substitution term */
  rsp4[1]=(long)sbase;  /* same base term */
  sp4+=3+sublength;     /* make room for substitution table */
  if (sublength==0)     /* can this happen? */
    {
      rsp4[2]=NULL;
      return rsp4;
    }
  rsp4[2]=(long)(rsp4+3); /* pointer to substitution table */
  for(i=0;i<sublength;i++) /* substitution table */
    {
      if (tbvarocc(i,sbase)) /* only copy if i occurs in sbase */
	{
	  rsp4[3+i]=(long)copyto4((term)ssub[i],q);
	}
    }
  return rsp4;
}


tbvarocc(i,s)
     long i;      /* syntactic variable */
     baseterm s;
{
  long j,args;
  if ((j=s[0])<=BMAX)
    return i==j;
  for (args=s[1];args>0;args--)
    if (tbvarocc(i,(baseterm)s[1+args]))
      return 1;
  return 0;
}



/* copysol(s,p,&q,&r) copies the value of the term s to the save stack
of the frame p, returning 1 or overflow status. As in copyto4(),
variables in s are bound to their copies and appended to the list q,
and copies of variables are tagged by adding BARVAL to the first
field: thus the calling function must unbind the variables in q and
remove the tag from each variable to which a variable in the list q is
bound. Also, the calling function must clean up the main stack.
Finally, the address of the copy is appended to the list r, also
constructed on the save stack.

 Note the safety margin of 100 longs: there isn't any checking in
copys, so there is the possibility of getting a memory crash by
copying a term that requires more than 100 longs at a point where
there isn't room on the save stack. */


copysol(s,p,q,r)
     term s;
     frame p;
     list *q,*r;
{
  term copys(),g;
  long *rs=(long *)SPOINT(p);
#ifdef DEBUG
  if (trace) mess("\ncopysol");
#endif
  if (rs>(long *)SAVE(p)+szsave-100)  /* 100 longs margin */
    return -1;
  SPOINT(p)+=2*sizeof(0L);
  g=copys(s,p,q);
  rs[0]=(long)g;
  rs[1]=(long)*r;
  *r=rs;
  return 1;
}

  

/* copys(s,p,&q) copies s to the save stack of the frame p, returning
   the address of the copy. Variables in s are bound to their copies
   and appended to the list q. */


term copys(s,p,q)
     term s;
     frame p;
     list *q;
{
  baseterm sbase;
  subtab ssub;
  long i;
  term getvalvar();
  long *rs;
  if (var(s)&&s[1])             /* dereference variable */
    s=getvalvar(s);
  rs=(long *)SPOINT(p);
  if (param(s))           /* copy parameter */
    {
      rs[0]=0L;
      rs[1]=s[1];
      SPOINT(p)=(long)(rs+2);
      return rs;
    }
  if (var(s))             /* copy unbound variable */
    {
      if (s[0]>BARVAL)
	return s;
      rs[0]=s[0]+BARVAL;
      rs[1]=NULL;
      SPOINT(p)=(long)(rs+2);
      sp[0]=(long)s;   /* append s to q */
      sp[1]=(long)*q;
      *q=sp;
      sp+=2;
      s[1]=(long)rs;   /* bind s to copy */
      return rs;
    }
  sbase=(baseterm)s[1];    /* copy substitution term */
  ssub=(subtab)s[2];
  rs[0]=(-1L);
  rs[1]=(long)sbase;
  SPOINT(p)=(long)(rs+3+sublength);
  if (sublength==0)
    {
      rs[2]=NULL;
      return rs;
    }
  rs[2]=(long)(rs+3);
  for(i=0;i<sublength;i++)
    {
      if (tbvarocc(i,sbase))
	rs[3+i]=(long)copys((term)ssub[i],p,q);
    }
  return rs;
}


  

/******************************* instance **********************************/


/* doinst3(s,t,&q) returns true if the value of s is an instance of the value
   of t. Bindings of variables in the value of t are used. These variables
   are appended to the list q. Since we must not bind variables deriving from
   s, we tag these variables by adding BARVAL to their restriction value.
   The calling function must initialize q and tag the variables in s before
   calling doinst3, and on return unbind the variables in q, remove tags, and
   clean up the main stack. */

doinst3(s,t,q)
     term s,t;
     list *q;
{
  baseterm tbase;
  subtab tsub;
  long i;
  term getvalvar(),getvalsub();
#ifdef DEBUG
  if (trace) mess("\ndoinst3");
#endif
  if (var(s)&&s[1])
    s=getvalvar(s);
  if (var(t)&&t[1])
    t=getvalvar(t);
  if (var(t))          /* is s an instance of the unbound variable t? */
    {
      if (s==t)  /* the same variable */
	return 1;
      if (t[0]>BARVAL)  /* no binding of tagged variables */
	return 0;
      if (s[0]<0)   /* s is a substitution term */
	  return instcheck(t,(baseterm)s[1],(subtab)s[2],q);
      if (param(s)&&s[1]>=t[0]-1) /* s is an inadmissible parameter */
	return 0;
      if (var(s)&&t[0]<((i=s[0])>BARVAL?i-BARVAL:i)) /* too permissive var */
	return 0;
      /* so s is an admissible parameter or a sufficiently restricted var */
      sp[0]=(long)t;   /* append variable t to list *q */
      sp[1]=(long)*q;
      *q=sp;
      sp+=2;
      t[1]=(long)s;   /* binding */
      return 1;
    }
  if (param(t))     /* t is a parameter, so s must be the same parameter */
    return (param(s)&&s[1]==t[1]);
  tbase=(baseterm)t[1];
  tsub=(subtab)t[2];
  if ((i=tbase[0])<=BMAX)
    return doinst3(s,getvalsub(i,tsub),q);
  /* so t is a composite substitution term */
  if (var(s)||param(s))
    return 0;  /* since t is composite */
  return inst5((baseterm)s[1],(subtab)s[2],tbase,tsub,q);
}



inst5(sbase,ssub,tbase,tsub,q)
     baseterm sbase,tbase;
     subtab ssub,tsub;
     list *q;
{
  long i,j,args;
  term s,t;
  term getvalsub();
#ifdef DEBUG
  if (trace) mess("\ninst5");
#endif
  i=sbase[0];
  j=tbase[0];
  if (i>BMAX)   /* s composite */
    {
      if (j>BMAX)    /* both composite */
	{
	  if (i!=j)
	    return 0;
	  for(args=sbase[1];args>0;args--)
	    {
	      if
		(inst5((baseterm)sbase[1+args],ssub,
		       (baseterm)tbase[1+args],tsub,q)==0)
		  return 0;
	    }
	  return 1;
	}
      /* just s composite */
      t=getvalsub(j,tsub);
      if (var(t))
	return instcheck(t,sbase,ssub,q);
      if (param(t))
	return 0;
      return inst5(sbase,ssub,(baseterm)t[1],(subtab)t[2],q);
    }
  if (j>BMAX) /* just t composite */
    {
      s=getvalsub(i,ssub);
      if (var(s)||param(s))
	return 0;
      sbase=(baseterm)s[1];
      ssub=(subtab)s[2];
      return inst5(sbase,ssub,tbase,tsub,q);
    }
  /* neither composite */
  s=getvalsub(i,ssub);
  t=getvalsub(j,tsub);
  return doinst3(s,t,q);
}


instcheck(t,sbase,ssub,q)
     baseterm sbase;
     subtab ssub;
     variable t;
     list *q;
{
  register long *rsp;
  if (t[0]>BARVAL)
    return 0;
  if (instadmissible(t,t[0]-1,sbase,ssub))
    {
      rsp=sp;
      rsp[0]=(-1L);
      rsp[1]=(long)sbase;
      rsp[2]=(long)ssub;
      t[1]=(long)rsp;      /* binding of t to the subterm */
      rsp[3]=(long)t;
      rsp[4]=(long)*q;
      *q=rsp+3;
      sp+=5;
      return 1;
    }
  return 0;
}


instadmissible(t,res,sbase,ssub)
     variable t;
     long res;
     baseterm sbase;
     subtab ssub;
{
  long args,i;
  term s,getvalsub();
  if((i=sbase[0])>BMAX)
    {
      for(args=sbase[1];args>0;args--)
        {
	  if (instadmissible(t,res,(baseterm)sbase[1+args],ssub)==0)
	    return 0;
        }
      return 1;
    }
  s=getvalsub(i,ssub);
  if (param(s))
    {
      if (s[1]>=res)
	return 0;
      return 1;
   }
  if (var(s))
    {
      if (s==t)
	return 0;
      if (t[0]<((i=s[0])>BARVAL?(i-BARVAL):i))
	return 0;
      return 1;
    }
  return instadmissible(t,res,(baseterm)s[1],(subtab)s[2]);
}



tagvars(s)
     term s;
{
  term getvalvar();
  if (var(s)&&s[1])
    s=getvalvar(s);
  if var(s)
    {
      if (s[0]<BARVAL)
	s[0]+=BARVAL;
      return;
    }
  if param(s)
    return;
  tagvars2((baseterm)s[1],(subtab)s[2]);
}


tagvars2(sbase,ssub)
     baseterm sbase;
     subterm ssub;
{
  long i,args;
  term getvalsub();
  if ((i=sbase[0])<=BMAX)
    {
      tagvars(getvalsub(i,ssub));
      return;
    }
  for(args=sbase[1];args>0;args--)
    tagvars2((term)sbase[1+args],ssub);
}

  
untagvars(s)
     term s;
{
  term getvalvar();
  if (var(s)&&s[1])
    s=getvalvar(s);
  if var(s)
    {
      if (s[0]>BARVAL) s[0]-=BARVAL;
      return;
    }
  if param(s)
    return;
  untagvars2((baseterm)s[1],(subtab)s[2]);
}

untagvars2(sbase,ssub)
     baseterm sbase;
     subterm ssub;
{
  long i,args;
  term getvalvar();
  if ((i=sbase[0])<=BMAX)
    {
      untagvars(getvalsub(i,ssub));
      return;
    }
  for(args=sbase[1];args>0;args--)
    untagvars2((term)sbase[1+args],ssub);
}

  

/**************************** various ********************************/


term getvalsub(i,subt)
     long i;
     subtab subt;
{
  register term j;
  term getvalvar();
  if (var(j=(term)subt[i])&&j[0])
    return getvalvar(j);
  return j;
}


term getvalvar(s)
     variable s;
{
  register term j;
  while((j=(term)s[1])&&var(j))
    s=j;
  if (j)
    return j;
  return s;
}



unbind(q)
     list q;  /* list of variables to unbind */
{
  register list r;
  r=q;
  while(r)
    {
      ((variable)r[0])[1]=NULL;
      r=(list)r[1];
    }
}





